unit FadeLabel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, StdCtrls, JoeLib;

type
  TFadeLabel = class(TCustomLabel)
  private
    { Private declarations }
    fFadeTemplate: TBitmap;
    fTopTemplate: TBitmap;
    fEdgeTemplate: TBitmap;
    fSolidTemplate: TBitmap;
    fDisplayBmp: TBitmap;
    fFadeBmp: TBitmap;
    fTopBmp: TBitmap;
    fEdgeBmp: TBitmap;
    fSolidBmp: TBitmap;
    fFadeColor: TColor;
    fBackColor: TColor;
    fOnColorChange: TNotifyEvent;
    procedure fSetFadeColor(color: TColor);
    procedure fSetBackcolor(color: TColor);
  protected
    { Protected declarations }
    procedure MakeFade;
    procedure SysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure ParentColorChanged(var Message: TMessage); message CM_PARENTCOLORCHANGED;
    function  GetLabelText: String; override;
  public
    { Public declarations }
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
    procedure Paint; override;
  published
    { Published declarations }
    property Color: TColor read fFadeColor write fSetFadeColor;
    property BackColor: TColor read fBackColor write fSetBackColor;
    property OnSysColorChange: TNotifyEvent read fOnColorChange write fOnColorChange;
    property Align;
    property Alignment;
    property AutoSize;
    property Caption;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowAccelChar;
    property ShowHint;
    property Layout;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
  end;

procedure Register;

implementation

{$R FadeLabel.RES}

//---------------------------------------------------------------------------------------------
constructor TFadeLabel.Create(aOwner: TComponent);
begin
   inherited;
   fFadeTemplate := TBitmap.Create;
   fTopTemplate := TBitmap.Create;
   fEdgeTemplate := TBitmap.Create;
   fSolidTemplate := TBitmap.Create;

   fFadeBmp := TBitmap.Create;
   fTopBmp := TBitmap.Create;
   fEdgeBmp := TBitmap.Create;
   fSolidBmp := TBitmap.Create;

   fDisplayBmp := TBitmap.Create;

   fFadeTemplate.LoadFromResourceName(HInstance, 'FADELABELBODYFADE');
   fSolidTemplate.LoadFromResourceName(HInstance, 'FADELABELBODYSOLID');
   fTopTemplate.LoadFromResourceName(HInstance, 'FADELABELTOP');
   fEdgeTemplate.LoadFromResourceName(HInstance, 'FADELABELEDGE');

   fFadeBmp.Assign(fFadeTemplate);
   fSolidBmp.Assign(fSolidTemplate);
   fTopBmp.Assign(fTopTemplate);
   fEdgeBmp.Assign(fEdgeTemplate);

   fBackColor := clBtnFace;
   fFadeColor := clWhite;
   MakeFade;

   Transparent := True;
   WordWrap := False;
end;

//---------------------------------------------------------------------------------------------
destructor TFadeLabel.Destroy;
begin
   if assigned(fFadeTemplate) then
      fFadeTemplate.Free;
   if assigned(fTopTemplate) then
      fTopTemplate.Free;
   if assigned(fDisplayBmp) then
      fDisplayBmp.Free;
   if assigned(fEdgeTemplate) then
      fEdgeTemplate.Free;
   if assigned(fSolidTemplate) then
      fSolidTemplate.Free;
   if assigned(fFadeBmp) then
      fFadeBmp.Free;
   if assigned(fTopBmp) then
      fTopBmp.Free;
   if assigned(fEdgeBmp) then
      fEdgeBmp.Free;
   if assigned(fSolidBmp) then
      fSolidBmp.Free;
   inherited;
end;

//---------------------------------------------------------------------------------------------
procedure TFadeLabel.SysColorChange(var Message: TMessage);
begin
   if assigned(fOnColorChange) then
      fOnColorChange(Self);
end;

//---------------------------------------------------------------------------------------------
procedure TFadeLabel.ParentColorChanged(var Message: TMessage);
var
   tmpColor: TColor;
begin
   if ParentColor then begin
      if (Message.wParam <> 0) then
         tmpColor := TColor(Message.lParam)
      else
         tmpColor := Parent.Brush.Color;

      if (tmpColor <> fBackColor) then begin
         fBackColor := tmpColor;
         MakeFade;
         Paint;
      end; {if}
  end;
end;

//---------------------------------------------------------------------------------------------
function TFadeLabel.GetLabelText: String;
begin
   result := ' ' + Caption + '          ';
end;

//---------------------------------------------------------------------------------------------
procedure TFadeLabel.Paint;
(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
   We have four bitmap pieces:

     fFadeBmp   - the fade part of the body
     fTopBmp    - the fade top line
     fEdgeBmp   - the non-fade left edge
     fSolidBmp  - the non-fade part of body

   These bmp's should have been pre-tinted.  They get tinted upon creation and
   when the Color and BackColor properties are changed.

   Here, we just assemble them into the given client area.

(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)
var
   cRect, tmpRect: TRect;
begin
   if csLoading in ComponentState then exit;

   cRect := ClientRect;
   with Canvas do begin
        SetStretchBltMode(Handle, HALFTONE);
        SetBrushOrgEx(Handle, 0, 0, nil);

        // TOP EDGE
        tmpRect := cRect;
        tmpRect.Bottom := tmpRect.Top + fTopBmp.Height;
        StretchDraw(tmpRect, fTopBmp);

        // LEFT EDGE
        tmpRect := cRect;
        inc(tmpRect.Top);
        tmpRect.Right := tmpRect.Left + fEdgeBmp.Width;
        StretchDraw(tmpRect, fEdgeBmp);

        // BODY SOLID
        tmpRect := cRect;
        inc(tmpRect.Top);
        dec(tmpRect.Right, ((tmpRect.Right - tmpRect.Left) div 2));
        inc(tmpRect.Left);
        StretchDraw(tmpRect, fSolidBmp);

        // BODY FADE
        tmpRect := cRect;
        inc(tmpRect.Top);
        inc(tmpRect.Left, ((tmpRect.Right - tmpRect.Left) div 2));
        StretchDraw(tmpRect, fFadeBmp);
   end; {with}

   inherited;
end;

//---------------------------------------------------------------------------------------------
procedure TFadeLabel.MakeFade;
(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
   Goes thru and translates all the weird yellow/blue combos into a tinted alpha-
   blended bitmap.  :)
(* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *)
begin
   TintBlueAlphaBitmap(fFadeTemplate, fFadeBmp, fFadeColor, fBackColor);
   TintBlueAlphaBitmap(fTopTemplate, fTopBmp, fFadeColor, fBackColor);
   TintBlueAlphaBitmap(fEdgeTemplate, fEdgeBmp, fFadeColor, fBackColor);
   TintBlueAlphaBitmap(fSolidTemplate, fSolidBmp, fFadeColor, fBackColor);
end;

//---------------------------------------------------------------------------------------------
procedure TFadeLabel.fSetFadeColor(color: TColor);
begin
   if (color = fFadeColor) then exit;
   fFadeColor := color;
   MakeFade;
   Paint;
end;

//---------------------------------------------------------------------------------------------
procedure TFadeLabel.fSetBackColor(color: TColor);
begin
   if (color = fBackColor) then exit;
   ParentColor := False;
   fBackColor := color;
   MakeFade;
   Paint;
end;

//---------------------------------------------------------------------------------------------
procedure Register;
begin
  RegisterComponents('Other', [TFadeLabel]);
end;

end.
