unit HideDirectoryListBox;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FileCtrl;

type
  TFileAttr = (ftReadOnly, ftHidden, ftSystem, ftArchive);
  TFileType = set of TFileAttr;
  TExtDirectoryListBox = class(TDirectoryListBox)
  private
    { Dclarations prives }
    FPreserveCase: Boolean;
    FCaseSensitive: Boolean;
    procedure SetFileType(NewFileType: TFileType);
  protected
    { Dclarations protges }
    FFileType : TFileType;
    SPBMP     : TBitmap;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    function  ReadDirectoryNames(const ParentDirectory: string;
      DirectoryList: TStringList): Integer;
    procedure BuildList; override;
    procedure ReadBitmaps; override;
    function GetItemPath(Index: Integer): string;
    procedure DblClick; override;
  public
    { Dclarations publiques }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure OpenCurrent;
  published
    { Dclarations publies }
    property FileType: TFileType read FFileType write SetFileType default [];
  end;

procedure Register;

implementation

{$R hidedirectorylistbox}

constructor TExtDirectoryListBox.create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFileType := [];
end;

destructor TExtDirectoryListBox.Destroy;
begin
 inherited destroy;
 spbmp.free;
end;

procedure TExtDirectoryListBox.ReadBitmaps;
begin
 inherited readbitmaps;
  SPBMP := TBitmap.Create;
  SPBMP.LoadFromResourceName(HInstance, 'HIDEFOLDER');
end;

function DirLevel(const PathName: string): Integer;  { counts '\' in path }
var
  P: PChar;
begin
  Result := 0;
  P := AnsiStrScan(PChar(PathName), '\');
  while P <> nil do
  begin
    Inc(Result);
    Inc(P);
    P := AnsiStrScan(P, '\');
  end;
end;

function SlashSep(const Path, S: String): String;
begin
  if AnsiLastChar(Path)^ <> '\' then
    Result := Path + '\' + S
  else
    Result := Path + S;
end;

function TExtDirectoryListBox.ReadDirectoryNames(const ParentDirectory: string;
  DirectoryList: TStringList): Integer;
type
  Tattributs  = set of 16..55 ;
var
  Status: Integer;
  SearchRec: TSearchRec;
  ensemble : Tattributs;
begin
  Result := 0;
  ensemble:=[];
  ensemble:=ensemble+[16];
  if ftReadOnly in filetype then ensemble:=ensemble+[17];
  if ftHidden in filetype then
  begin
   ensemble:=ensemble+[18];
   if 17 in ensemble then ensemble:=ensemble+[19];
  end;
  if ftSystem in filetype then
  begin
   ensemble:=ensemble+[20];
   if 17 in ensemble then ensemble:=ensemble+[21];
   if 18 in ensemble then ensemble:=ensemble+[22];
   if 19 in ensemble then ensemble:=ensemble+[23];
  end;
  if ftArchive in filetype then
  begin
   ensemble:=ensemble+[48];
   if 17 in ensemble then ensemble:=ensemble+[49];
   if 18 in ensemble then ensemble:=ensemble+[50];
   if 19 in ensemble then ensemble:=ensemble+[51];
   if 20 in ensemble then ensemble:=ensemble+[52];
   if 21 in ensemble then ensemble:=ensemble+[53];
   if 22 in ensemble then ensemble:=ensemble+[54];
   if 23 in ensemble then ensemble:=ensemble+[55];
  end;
  Status := FindFirst(SlashSep(ParentDirectory, '*.*'), faAnyfile, SearchRec);
  try
    while Status = 0 do
    begin {16=dir 17=dir+ro 18=dir+h 19=dir+h+ro 20=dir+sys
           21=dir+sys+r 23=dir+h+r+sys 22=dir+sys+h}
      if (SearchRec.Attr in ensemble) then
      begin
        if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then
        begin
          if SearchRec.Attr in [17..23] then
          DirectoryList.Addobject(SearchRec.Name,spbmp) else
          DirectoryList.Add(SearchRec.Name);
          Inc(Result);
        end;
      end;
      Status := FindNext(SearchRec);
    end;
  finally
    FindClose(SearchRec);
  end;
end;

procedure TExtDirectoryListBox.BuildList;
var
  TempPath: string;
  DirName: string;
  IndentLevel, BackSlashPos, i, VolFlags: Integer;
  Siblings: TStringList;
  NewSelect: Integer;
  Root: String;
begin
  try
    Items.BeginUpdate;
    Items.Clear;
    IndentLevel := 0;
    Root := ExtractFileDrive(Directory)+'\';
    GetVolumeInformation(PChar(Root), nil, 0, nil, i, VolFlags, nil, 0);
    FPreserveCase := VolFlags and (FS_CASE_IS_PRESERVED or FS_CASE_SENSITIVE) <> 0;
    FCaseSensitive := (VolFlags and FS_CASE_SENSITIVE) <> 0;
    if (Length(Root) >= 2) and (Root[2] = '\') then
    begin
      Items.AddObject(Root, OpenedBMP);
      Inc(IndentLevel);
      TempPath := Copy(Directory, Length(Root)+1, Length(Directory));
    end
    else
      TempPath := Directory;
    if (Length(TempPath) > 0) then
    begin
      if AnsiLastChar(TempPath)^ <> '\' then
      begin
        BackSlashPos := AnsiPos('\', TempPath);
        while BackSlashPos <> 0 do
        begin
          DirName := Copy(TempPath, 1, BackSlashPos - 1);
          if IndentLevel = 0 then DirName := DirName + '\';
          Delete(TempPath, 1, BackSlashPos);
          Items.AddObject(DirName, OpenedBMP);
          Inc(IndentLevel);
          BackSlashPos := AnsiPos('\', TempPath);
        end;
      end;
      Items.AddObject(TempPath, CurrentBMP);
    end;
    NewSelect := Items.Count - 1;
    Siblings := TStringList.Create;
    try
      Siblings.Sorted := True;
        { read all the dir names into Siblings }
      ReadDirectoryNames(Directory, Siblings);
      for i := 0 to Siblings.Count - 1 do
       if siblings.Objects[i]= nil then
        Items.AddObject(Siblings[i], ClosedBMP)else
        Items.AddObject(Siblings.Strings[i], SPBMP);
    finally
      Siblings.Free;
    end;
  finally
    Items.EndUpdate;
  end;
  if HandleAllocated then
    ItemIndex := NewSelect;
end;

procedure TExtDirectoryListBox.SetFileType(NewFileType: TFileType);
begin
  if NewFileType <> FFileType then
  begin
    FFileType := NewFileType;
    buildlist;
  end;
end;

function TExtDirectoryListBox.GetItemPath (Index: Integer): string;
var
  CurDir: string;
  i, j: Integer;
  Bitmap: TBitmap;
begin
  Result := '';
  if Index < Items.Count then
  begin
    CurDir := Directory;
    Bitmap := TBitmap(Items.Objects[Index]);
    if Index = 0 then
      Result := ExtractFileDrive(CurDir)+'\'
    else if ((Bitmap = ClosedBMP) or (Bitmap = SPBMP))then
      Result := SlashSep(CurDir,Items[Index])
    else if Bitmap = CurrentBMP then
      Result := CurDir
    else
    begin
      i   := 0;
      j   := 0;
      Delete(CurDir, 1, Length(ExtractFileDrive(CurDir)));
      while j <> (Index + 1) do
      begin
        Inc(i);
        if i > Length (CurDir) then
          break;
        if CurDir[i] in LeadBytes then
          Inc(i)
        else if CurDir[i] = '\' then
          Inc(j);
      end;
      Result := ExtractFileDrive(Directory) + Copy(CurDir, 1, i - 1);
    end;
  end;
end;

procedure TExtDirectoryListBox.DrawItem(Index: Integer; Rect: TRect;
  State: TOwnerDrawState);
var
  Bitmap: TBitmap;
  bmpWidth: Integer;
  dirOffset: Integer;
begin
  with Canvas do
  begin
    FillRect(Rect);
    bmpWidth  := 16;
    dirOffset := Index * 4 + 2;    {add 2 for spacing}

    Bitmap := TBitmap(Items.Objects[Index]);
    if Bitmap <> nil then
    begin
      if (Bitmap = ClosedBMP) or (Bitmap = SPBMP)then
        dirOffset := (DirLevel (Directory) + 1) * 4 + 2;

      bmpWidth := Bitmap.Width;
      BrushCopy(Bounds(Rect.Left + dirOffset,
               (Rect.Top + Rect.Bottom - Bitmap.Height) div 2,
               Bitmap.Width, Bitmap.Height),
               Bitmap, Bounds(0, 0, Bitmap.Width, Bitmap.Height),
               Bitmap.Canvas.Pixels[0, Bitmap.Height - 1]);
    end;
    TextOut(Rect.Left + bmpWidth + dirOffset + 4, Rect.Top, DisplayCase(Items[Index]))
  end;
end;

procedure TExtDirectoryListBox.OpenCurrent;
begin
  Directory := GetItemPath(ItemIndex);
end;

procedure TExtDirectoryListBox.DblClick;
begin
  inherited DblClick;
  OpenCurrent;
end;

procedure Register;
begin
  RegisterComponents('Exemples', [TExtDirectoryListBox]);
end;

end.
