unit ExpertManager;

{I put this together to test the possibility of loading and unloading experts at runtime.

Delphi provides a method for experts to register themselves but no way to unregister
themselves, so it is only possible to safely unload experts that Delphi does not
need to know about.  Hence, only esAddin Experts can be managed successfully,
providing that they clean up properly when they are freed.


This is an esAddIn expert itself
  It manages a collection of sub experts that can be dynamically installed and
  uninstalled in the Delphi IDE without having to restart Delphi.

  The managed experts are listed in the registry and can be managed via a form
  linked to a menu item under the Delphi Tools menu. 

  This is freeware.

  Written by Richard Brooks (c) 1999.

  See the accompanying readme file for full details

  THIS SOFTWARE IS PROVIDED SOLELY AS-IS AND WITHOUT WARRANTY INCLUDING WITHOUT
  LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
  PARTICULAR PURPOSE.  THE USER AGREES TO INDEMNIFY AND HOLD THE AUTHOR HARMLESS
  FROM AND AGAINST ANY AND ALL CLAIMS ARISING FROM OR RELATED TO USE OF THE
  PRODUCT. UNDER NO CIRCUMSTANCES SHALL THE AUTHOR'S LIABILITY EXCEED THE
  SOFTWARE LICENSE FEE.
}

interface

uses
   Exptintf, toolintf, classes, windows;

type
   TExpertCollectionItem = class; {forward declaration}

   TExpertManager = class(TIExpert)
   private
      FExperts : TCollection;
      FExpertRegisterProc : TExpertRegisterProc;
      {menu item}
      FConfigurationItem  : TIMenuItemIntf;
      procedure InsertConfigurationMenuItem;
      {show the about/configuration screen}
      procedure OnConfigClick(Sender: TIMenuItemIntF);
   public
      constructor Create(AExpertRegisterProc : TExpertRegisterProc); virtual;
      destructor Destroy; override;
      procedure LoadExperts;
      //this sets all the subexperts loaded properties to FALSE
      procedure UnLoadExperts;

      function AddExpert(AFileName : AnsiString)    : boolean;
      function RemoveExpert(AFileName : AnsiString) : boolean;
      {Toolservices is stored globally by the vcl}
      property ExpertRegisterProc : TExpertRegisterProc read FExpertRegisterProc;
      property Experts : TCollection read FExperts;// write SetExperts;

      {TIExpert abstract functions}
      function GetName: String; override;
      function GetAuthor: string; override;
      function GetComment: string; override;
      function GetPage: string; override;
      function GetGlyph: HICON; override;
      function GetStyle: TExpertStyle; override;
      function GetState: TExpertState; override;
      function GetIDString: String; override;
      function GetMenuText: string; override;
      procedure Execute; override;
   end;

   TLoadedState = (exNotLoaded, exLoading, exLoaded);

   TExpertCollectionItem = class(TCollectionItem)
   private
      FInstanceHandle : HINST;
      FFileName       : AnsiString;
      FExpertInitProc : TExpertInitProc;
      FExpertExitProc : TExpertTerminateProc;
      FExpert         : TIExpert;
      FLoaded         : TLoadedState;
      procedure SetLoaded(AValue : boolean);
      function GetLoaded : boolean;
      {}
      function LoadExpert   : boolean;
      function UnLoadExpert : boolean;
   protected
      function GetDisplayName: string; override;
   public
      destructor Destroy; override;

      property InstanceHandle : HINST                read FInstanceHandle write FInstanceHandle;
      property FileName       : AnsiString           read FFileName       write FFileName;
      property ExpertInitProc : TExpertInitProc      read FExpertInitProc write FExpertInitProc;
      property ExpertExitProc : TExpertTerminateProc read FExpertExitProc write FExpertExitProc;
      property Expert         : TIExpert             read FExpert         write FExpert;
      property Loaded         : Boolean              read GetLoaded       write SetLoaded;
   end;

   function OwnRegisterProc(Expert: TIExpert): Boolean;

implementation

uses
   SysUtils, Registry, ExpertMgrFrm, Dialogs, forms;

const
   ManagerRegistryKey = 'Software\RJB\Expert Manager';

var
   Manager : TExpertManager;


function OwnRegisterProc(Expert: TIExpert): Boolean;
var
   i_count : integer;
begin
   Result := FALSE;
   if Manager <> nil then
      if assigned(Manager.ExpertRegisterProc) then
      begin
         //don't bother registering the sub experts with delphi as we own them and will
         // free them when the manager is freed
         //They are happy as they have already received the toolservices interface
//       Result := Manager.ExpertRegisterProc(Expert)

         Result := TRUE;

         //find the expert which is loading
         for i_count := 0 to Manager.Experts.Count-1 do
         begin
            if (Manager.Experts.Items[i_count] <> nil) and
               (Manager.Experts.Items[i_count] is TExpertCollectionItem) and
               (TExpertCollectionItem(Manager.Experts.Items[i_count]).Floaded = exLoading) then
            begin
               TExpertCollectionItem(Manager.Experts.Items[i_count]).Expert := Expert;
            end;
         end;
      end;
end;

constructor TExpertManager.Create(AExpertRegisterProc : TExpertRegisterProc);
begin
   inherited Create;
   FExpertRegisterProc := AExpertRegisterProc;
   FExperts := TCollection.Create(TExpertCollectionItem);
   if Manager = nil then
      Manager := self
   else
      Exception.Create('Creating Duplicate Instance of Expert Manager');
   //create the config menu item on the tools menu
   InsertConfigurationMenuItem;
   //load all the subexperts using the registry settings
   LoadExperts;
end;

destructor TExpertManager.Destroy;
begin
   FExperts.Free;
   if FConfigurationItem <> nil then
   begin
      FConfigurationItem.free;
      FConfigurationItem := nil;
   end;
   inherited;
end;

//create the config menu item on the tools menu
procedure TExpertManager.InsertConfigurationMenuItem;
var
   MainMenu   : TIMainMenuIntf;
   ToolsMenu,
   ConfigToolsItem : TIMenuItemIntf;
begin
   MainMenu := ToolServices.GetMainMenu;
   if MainMenu <> nil then
   try
      ToolsMenu := MainMenu.FindMenuItem('ToolsMenu');
      if ToolsMenu <> nil then
      try
         ConfigToolsItem := MainMenu.FindMenuItem('ToolsToolsItem');
         if ConfigToolsItem <> nil then
         try
            FConfigurationItem := ToolsMenu.InsertItem(ConfigToolsItem.GetIndex, 'Manage Experts...',
               'ConfigurationItem', '', 0, 0, 0, [mfVisible, mfEnabled], OnConfigClick);
         finally
            ConfigToolsItem.Free;
         end;
      finally
         ToolsMenu.Free;
      end;
   finally
      MainMenu.Free;
   end;
end;

//show a dialog under tools menu to load and unload the experts
procedure TExpertManager.OnConfigClick(Sender: TIMenuItemIntF);
var
   frm_ExpertMgrFrm : TExpertMgrFrm;
begin
   frm_ExpertMgrFrm := TExpertMgrFrm.Create(Application);
   try
      frm_ExpertMgrFrm.ExpertManager := self;
      frm_ExpertMgrFrm.ShowModal;
   finally
      frm_ExpertMgrFrm.Free;
   end;
end;

//creates all the subexperts and loads them into delphi according to the registry
// settings
procedure TExpertManager.LoadExperts;
var
   ExpertDlls           : TStringlist;
   ExpertCollectionItem : TExpertCollectionItem;
   i_count : integer;
begin
   ExpertDlls := TStringlist.Create;
   try
      with TRegistry.Create do
         try
            OpenKey(ManagerRegistryKey, True);
            //iterate through all the registry values
            GetValueNames(ExpertDlls);
            for i_count := 0 to ExpertDlls.Count-1 do
            begin
               if (ExpertDlls[i_count] <> '') and FileExists(ExpertDlls[i_count]) then
               begin
                  ExpertCollectionItem := TExpertCollectionItem(FExperts.Add);
                  ExpertCollectionItem.FileName    := ExpertDlls[i_count];
                  ExpertCollectionItem.Loaded      := (ReadInteger(ExpertDlls[i_count]) > 0);
               end;
            end;
         finally
            free; {registry}
         end;
   finally
      ExpertDlls.Free;
   end;
end;

//this sets all the subexperts loaded properties to FALSE
// - this is not actually used at the moment
procedure TExpertManager.UnLoadExperts;
var
   i_count : integer;
begin
   for i_count := FExperts.Count-1 downto 0 do
      (FExperts.items[i_count] as TExpertCollectionItem).Loaded := FALSE;
end;

//public method (used by the config form) to add an expert to the subexpert collection
function TExpertManager.AddExpert(AFileName : AnsiString): boolean;
var
   ExpertCollectionItem : TExpertCollectionItem;
begin
   Result := FALSE;
   with TRegistry.Create do
      try
         OpenKey(ManagerRegistryKey, True);
         if ValueExists(AFileName) then
         begin
            MessageDlg('The specified file is already installed in the expert manager', mtInformation, [mbOK], 0);
            exit;
         end;
         if FileExists(AFileName) then
         begin
            WriteInteger(AFileName, 0);
            ExpertCollectionItem := TExpertCollectionItem(FExperts.Add);
            ExpertCollectionItem.FileName := AFileName;
            ExpertCollectionItem.Loaded   := FALSE;
            Result := TRUE;
         end
         else
            MessageDlg('The specified file does not exist', mtInformation, [mbOK], 0);
      finally
         free; {registry}
      end;
end;

//public method (used by the config form) to remove an expert from the subexpert collection
function TExpertManager.RemoveExpert(AFileName : AnsiString): boolean;
var
   i_count              : integer;
   ExpertCollectionItem : TExpertCollectionItem;
begin
   Result := FALSE;
   for i_count := 0 to FExperts.Count-1 do
      if (FExperts.Items[i_count] as TExpertCollectionItem).FileName = AFileName then
      begin
         ExpertCollectionItem := TExpertCollectionItem(FExperts.Items[i_count]);
         ExpertCollectionItem.Loaded := FALSE;
         ExpertCollectionItem.free; {this removes it from the collection
                                     no need to update the count}
         with TRegistry.Create do
            try
               OpenKey(ManagerRegistryKey, True);
               DeleteValue(AFileName);
            finally
               free; {registry}
            end;
         Result := TRUE;
         break;{from the for loop}
      end;
end;

//******************************************************************************
//****************************TExpertCollectionItem*****************************
//******************************************************************************

destructor TExpertCollectionItem.Destroy;
begin
   {make sure the expert is unloaded}
   Loaded := FALSE;
   inherited;
end;

//use the expert's dll filename as the descriptive display name
// this has to be overriden due to the default behaviour of a CollectionItem
function TExpertCollectionItem.GetDisplayName: string;
begin
   Result := ExtractFileName(FFileName);
end;

//loaded property set routine, loads or removes the associated expert
procedure TExpertCollectionItem.SetLoaded(AValue : Boolean);
begin
   if (AValue and (FLoaded <> exLoaded))
      or ((not AValue) and (FLoaded = exLoaded)) then
   begin
      if FLoaded = exLoaded then
         {unload it}
         UnloadExpert
      else
         {so load it }
         LoadExpert;
   end;
end;

function TExpertCollectionItem.GetLoaded : boolean;
begin
   Result := (FLoaded = exLoaded);
end;

//load the associated expert into the delphi ide
function TExpertCollectionItem.LoadExpert : boolean;
var
   InstanceHandle       : HInst;
   ExpertInitProc       : TExpertInitProc;
   ExpertExitProc       : TExpertTerminateProc;
begin
   Result := FALSE;
   InstanceHandle := LoadLibrary(pchar(FFileName));
   if InstanceHandle = 0 then
      raise Exception.Create('Could not load Expert from DLL : ' + FFileName)
   else
   begin
      @ExpertInitProc := GetProcAddress(InstanceHandle, ExpertEntryPoint);
      if @ExpertInitProc <> nil then
      begin
         FInstanceHandle := InstanceHandle;
         FExpertInitProc := @ExpertInitProc;
         {execute the expert}
         ExpertExitProc := nil;
         FLoaded := exLoading;
         try
            FExpertInitProc(ToolServices, OwnRegisterProc, ExpertExitProc);
            {set the terminate procedure}
            if assigned(ExpertExitProc) then
               FExpertExitProc := @ExpertExitProc
            else
               FExpertExitProc := nil;
            FLoaded := exLoaded;
            Result  := TRUE;
         except
            if FLoaded <> exLoaded then
            begin
               FLoaded := exNotLoaded;
            end;
            raise;
         end;
      end
      else begin
         // ignore DLLs that do not have the expert entry point
         FreeLibrary(InstanceHandle);
         MessageDlg(FFileName + ' is not a Delphi Expert', mtError, [mbOK], 0);
      end;
   end;
end;

//removes the associated expert from the delphi ide
function TExpertCollectionItem.UnLoadExpert : boolean;
begin
   Result := FALSE;
   if FLoaded = exLoaded then
   begin
      if @FExpertExitProc <> nil then
      begin
         FExpertExitProc;
         FExpertExitProc := nil;
      end;
      if FExpert <> nil then
      begin
         FExpert.free;
         FExpert := nil;
      end;
      FreeLibrary(FInstanceHandle);
      FInstanceHandle := 0;

      FLoaded := exNotLoaded;
   end;
end;


{overriden abstract TIExpert methods}
function TExpertManager.GetName: String;
begin
   Result := 'ExpertManager';
end;

function TExpertManager.GetStyle: TExpertStyle;
begin
   Result := esAddIn;
end;

function TExpertManager.GetIDString: String;
begin
   Result := 'RichardBrooks.ExpertManager';
end;

function TExpertManager.GetState: TExpertState;
begin
   Result := [];
end;

function TExpertManager.GetMenuText: string;
begin
   Result := '';
end;

procedure TExpertManager.Execute;
begin
   //nothing
end;

function TExpertManager.GetAuthor: string;
begin
   Result := '';
end;

function TExpertManager.GetComment: string;
begin
   Result := '';
end;

function TExpertManager.GetPage: string;
begin
   Result := '';
end;

function TExpertManager.GetGlyph: HICON;
begin
   Result := 0;
end;


initialization
   Manager := nil;

finalization
   Manager := nil;
end.
