{
TExRxDBGrid Version 1.5 FREEWARE

Copyright () 1999 by GJL Software
Updated 18th Nov 1999 (18/11/1999)

Email:         ExRxDBGrid@gjl-software.co.uk
Home Page:     http://www.gjl-software.co.uk

Description:   ExRxDBGrid is an Enhanced Data-Aware Grid Component that can
               display Graphic Fields as images, Memo Fields as text and
               Boolean Fields as Checkboxes instead of the normal
               (GRAPHIC), (MEMO) or True/False that appear.

               It has the following new Properties:
               EditColor, InplaceEdit, DefaultRowHeight, RowSizingAllowed,
               DisplayImages, DisplayMemo, DisplayBoolean, Col, Row, TopRow,
               Canvas, GridAutoSize, FullSizeMemo, ScrollBars, Cells3D
               (when False, the Grid has a Flat look and feel - also the 3D
               Cells can be any color) and CellHints (when True and Columns are
               not wide enough for the DisplayText, moving the mouse over a cell
               shows the text in a hint window).

               It has the following new Methods:
               MoveToRow and CellRect method (which is extremely
               useful if you want to drop other controls on the grid
               i.e. pop a dropdown list over a cell when the user enters it).

               It has the following new Events:
               OnColumnResize, OnMouseDown, OnMouseUp, OnRightClick
               and OnApplyCellAttribute (useful when you want to change the
               colour of an individual cell, based on the value in that cell
               or some other particular condition).

               It has the following new Features:
               OptWidth (making the Grid automatically size itself so that no
               white space is left at the right) and Enhanced Scrollbar Tracking.

               It also has popup viewers/editors for Graphic/Memo Fields.
               To use these and the Checkbox, simply click on the Fields
               or use the Spacebar.

*PLEASE NOTE*  Before making any bug reports please first verify you are
               using the latest version by checking my home page. And if
               you do report a bug, please, if applicable, include a code
               sample.

               Any suggested enhancements would also be appreciated.

Notes:         I cannot support modified versions of this code. So if you
               encounter a possible bug while using a modified version,
               always first revert back to the original code before making
               an attempt to contact me.
}

unit ExRxDBGrid;

interface

uses
    ShellAPI, Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
    Grids, DBGrids, DB, DBTables, BDE, StdCtrls, ClipBrd, RXDBCtrl;

type
    TCellAttribute = procedure(Sender: TObject; Field: TField; Canvas: TCanvas; State: TGridDrawState) of object;

    TExRxDBGrid = class(TRxDBGrid)
    private
        FNewDefaultRowHeight: Integer;
        FRowSizingAllowed: Boolean;
        FOnApplyCellAttribute: TCellAttribute;
        FOnMouseDown: TMouseEvent;
        FOnMouseUp: TMouseEvent;
        FOnColEnter: TNotifyEvent;
        FRowHeight: Integer;
        FDisplayImages: boolean;
        FDisplayMemo: boolean;
        FDisplayBool: boolean;
        FGridAutoSize: boolean;
        FOnRightClick: TNotifyEvent;
        FFullSizeMemo: boolean;
        FCells3D: boolean;
        FCellHints: boolean;
        FHintWnd: THintWindow;
        function AcquireFocus: Boolean;
        function GetTitleOffset: Byte;
        procedure SetRowSizingAllowed(Value: Boolean);
        procedure SetDisplayImages(Value: boolean);
        procedure SetDisplayMemo(Value: boolean);
        procedure SetDisplayBool(Value: boolean);
        procedure SetGridAutoSize(Value: boolean);
        procedure SetFullSizeMemo(Value: boolean);
        procedure WMRButtonDown(var Message: TWMRButtonDown); message WM_RBUTTONDOWN;
        procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
        procedure SetCells3D(Value: boolean);
        procedure SetCellHints(Value: boolean);
    protected
        FOnColResize: TNotifyEvent;
        FEditColor: TColor;
        FInplaceEdit: TInplaceEdit;
        function CreateEditor: TInplaceEdit; override;
        function GetDefaultRowHeight: Integer;
        function GetOptWidth: Integer;
        function CalcHintRect(MaxWidth: Integer; const AHint: string; HintWnd: THintWindow): TRect;
        procedure DoHint(X, Y: Integer);
        procedure SetDefaultRowHeight(Value: Integer);
        procedure LayoutChanged; override;
        procedure RowHeightsChanged; override;
        procedure CellClick(Column: TColumn); override;
        procedure DrawColumnCell(const Rect: TRect; DataCol: Integer; Column: TColumn; State: TGridDrawState); override;
        procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
        procedure ColWidthsChanged; override;
        procedure SetCheckBoxSize;
        procedure ApplyCellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState);
        procedure CellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState); dynamic;
        procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
        procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
        procedure ColEnter; override;
        procedure WMSize(var Message: TWMSize); message WM_SIZE;
        procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
        procedure KeyDown(var Key: Word; Shift: TShiftState); override;
        procedure KeyPress(var Key: Char); override;
    public
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        property Canvas;
        property Col;
        property Row;
        property TopRow;
        property SelectedRows;
        property Datalink;
        property ColCount;
        property RowCount;
        property ColWidths;
        property GridLineWidth;
        property TitleOffset: Byte read GetTitleOffset;
        property OnColumnResize: TNotifyEvent read FOnColResize write FOnColResize;
        property OptWidth: Integer read GetOptWidth;
        function CellRect(ACol, ARow: Longint): TRect;
        procedure MoveToRow(NewRow: Integer);
        procedure WMHScroll(var Msg: TWMHScroll); message wm_HScroll;
        procedure WMVScroll(var Msg: TWMVScroll); message wm_VScroll;
        procedure CMMouseEnter(var Msg: TMessage); message cm_MouseEnter;
        procedure CMMouseLeave(var Msg: TMessage); message cm_MouseLeave;
        procedure WMMouseMove(var Msg: TWMMouseMove); message wm_MouseMove;
    published
        property ScrollBars;
        property DefaultDrawing default False;
        property EditColor: TColor read FEditColor write FEditColor;
        property InplaceEdit: TInplaceEdit read FInplaceEdit;
        property DefaultRowHeight: Integer read GetDefaultRowHeight write SetDefaultRowHeight;
        property RowSizingAllowed: Boolean read FRowSizingAllowed write SetRowSizingAllowed default False;
        property OnApplyCellAttribute: TCellAttribute read FOnApplyCellAttribute write FOnApplyCellAttribute;
        property OnMouseDown read FOnMouseDown write FOnMouseDown;
        property OnMouseUp read FOnMouseUp write FOnMouseUp;
        property OnColEnter: TNotifyEvent read FOnColEnter write FOnColEnter;
        property DisplayImages: boolean read FDisplayImages write SetDisplayImages default True;
        property DisplayMemo: boolean read FDisplayMemo write SetDisplayMemo default True;
        property DisplayBoolean: boolean read FDisplayBool write SetDisplayBool default True;
        property GridAutoSize: boolean read FGridAutoSize write SetGridAutoSize default False;
        property FullSizeMemo: boolean read FFullSizeMemo write SetFullSizeMemo default False;
        property OnRightClick: TNotifyEvent read FOnRightClick write FOnRightClick;
        property Cells3D: boolean read FCells3D write SetCells3D default False;
        property CellHints: boolean read FCellHints write SetCellHints default False;
    end;

    TInplaceEditCracker = class(TInplaceEdit)
    public
        property Color;
    end;

    TExRxDBGridMod = class(TCustomGrid)
    public
        property Options;
    end;

procedure Register;

implementation

uses ExDBGridCtls;

var
    GCheckWidth, GCheckHeight: Integer;
    gk: Word;

constructor TExRxDBGrid.Create(AOwner: TComponent);
begin
    inherited Create(AOwner);
    DataSource := Datalink.DataSource;
    DefaultDrawing := False;
    ControlStyle := ControlStyle + [csAcceptsControls, csDisplayDragImage, csFramed];
    SetCheckBoxSize;
    EditColor := clWindow;
    FRowheight := 17;
    FRowSizingAllowed := False;
    FDisplayImages := True;
    FDisplayMemo := True;
    FDisplayBool := True;
    GridAutoSize := False;
    FullSizeMemo := False;
    Cells3D := False;
    CellHints := False;
end;

{ BEGIN PRIVATE }

function TExRxDBGrid.AcquireFocus: Boolean;
begin
    Result := True;
    if FAcquireFocus and CanFocus and not (csDesigning in ComponentState) then
        begin
            SetFocus;
            Result := Focused or (InplaceEditor <> nil) and InplaceEditor.Focused;
        end;
end;

function TExRxDBGrid.GetTitleOffset: Byte;
begin
    Result := 0;
    if dgTitles in Options then
        Inc(Result);
end;

procedure TExRxDBGrid.SetRowSizingAllowed(Value: Boolean);
begin
    if Value <> FRowSizingAllowed then
        begin
            FRowSizingAllowed := Value;
            if FRowSizingAllowed then
                TExRxDBGridMod(Self).Options := TExRxDBGridMod(Self).Options + [goRowSizing, goColSizing, goThumbTracking, goTabs, goColMoving, goRowMoving, goDrawFocusSelected]
            else
                TExRxDBGridMod(Self).Options := TExRxDBGridMod(Self).Options + [goColSizing, goThumbTracking, goTabs, goColMoving, goRowMoving, goDrawFocusSelected] - [goRowSizing];
        end
end;

procedure TExRxDBGrid.SetDisplayImages(value: boolean);
begin
    if FDisplayImages <> value then
        begin
            FDisplayImages := value;
            invalidate;
        end;
end;

procedure TExRxDBGrid.SetDisplayMemo(value: boolean);
begin
    if FDisplayMemo <> value then
        begin
            FDisplayMemo := value;

            if FDisplayMemo = False then
                begin
                    if FFullSizeMemo = True then
                        begin
                            SetFullSizeMemo(False);
                            FFullSizeMemo := True;
                        end;
                end
            else
                begin
                    if FFullSizeMemo = True then
                        begin
                            SetFullSizeMemo(True);
                        end;
                end;

            invalidate;
        end;
end;

procedure TExRxDBGrid.SetDisplayBool(value: boolean);
begin
    if FDisplayBool <> value then
        begin
            FDisplayBool := value;
            invalidate;
        end;
end;

procedure TExRxDBGrid.SetGridAutoSize(value: boolean);
var
    RowNo: Integer;
    ColNo: Integer;
    WidthNew: Integer;
    WidthMax: Integer;
    TmpDS: TDataSource;
begin
    FGridAutoSize := value;

    if not (csDesigning in ComponentState) then
        begin
            if Columns.State <> csCustomized then
                begin
                    if FGridAutoSize = True then
                        begin
                            if (Datalink <> nil) and Datalink.Active then
                                begin
                                    Datalink.DataSet.DisableControls;

                                    Columns.RestoreDefaults;

                                    for ColNo := 0 to (ColCount - 2) do
                                        begin
                                            WidthMax := 0;
                                            Datalink.DataSet.First;

                                            for RowNo := 0 to (RowCount - 1) do
                                                begin
                                                    if Fields[ColNo].Value <> Null then
                                                        begin
                                                            if FFullSizeMemo = False then
                                                                begin
                                                                    WidthNew := Canvas.TextWidth(Fields[ColNo].DisplayText);
                                                                end
                                                            else
                                                                begin
                                                                    if (Fields[ColNo].DataType = ftMemo) or (Fields[ColNo].DataType = ftFmtMemo) then
                                                                        begin
                                                                            WidthNew := Canvas.TextWidth(Fields[ColNo].Value);
                                                                        end
                                                                    else
                                                                        begin
                                                                            WidthNew := Canvas.TextWidth(Fields[ColNo].DisplayText);
                                                                        end;
                                                                end;
                                                        end;

                                                    if WidthNew > WidthMax then
                                                        WidthMax := WidthNew;

                                                    Datalink.DataSet.Next;
                                                end;

                                            try
                                                Columns[ColNo].Width := (WidthMax + (GridLineWidth + 5))
                                            except
                                            end;
                                        end;

                                    Datalink.DataSet.First;

                                    Datalink.DataSet.EnableControls;
                                end;
                        end
                    else
                        begin
                            if (Datalink <> nil) and Datalink.Active then
                                begin
                                    Datalink.DataSet.DisableControls;

                                    TmpDS := TExRxDBGrid(Self).DataSource;
                                    TExRxDBGrid(Self).DataSource := nil;
                                    TExRxDBGrid(Self).DataSource := TmpDS;

                                    Columns.RestoreDefaults;

                                    Datalink.DataSet.EnableControls;
                                end;
                        end;
                end;
        end;
end;

procedure TExRxDBGrid.SetFullSizeMemo(value: boolean);
begin
    FFullSizeMemo := value;

    if not (csDesigning in ComponentState) then
        begin
            if Columns.State <> csCustomized then
                begin
                    if (Datalink <> nil) and Datalink.Active then
                        begin
                            if FFullSizeMemo = True then
                                FDisplayMemo := True;

                            if FGridAutoSize = False then
                                begin
                                    SetGridAutoSize(True);

                                    SetGridAutoSize(False);
                                end
                            else
                                begin
                                    SetGridAutoSize(FGridAutoSize);
                                end;
                        end;
                end;
        end;
end;

procedure TExRxDBGrid.WMRButtonDown(var Message: TWMRButtonDown);
begin
    try
        SetFocus;
        inherited;

        if Assigned(FOnRightClick) then
            FOnRightClick(Self);
    except
    end;
end;

procedure TExRxDBGrid.WMLButtonDown(var Message: TWMLButtonDown);
begin
    try
        SetFocus;
        inherited;
    except
    end;
end;

procedure TExRxDBGrid.SetCells3D(Value: Boolean);
begin
    if FCells3D <> Value then
        begin
            FCells3D := Value;
            invalidate;
        end;
end;

procedure TExRxDBGrid.SetCellHints(Value: Boolean);
begin
    if FCellHints <> Value then
        begin
            FCellHints := Value;
            invalidate;
        end;
end;

{ END PRIVATE }

{ BEGIN PROTECTED }

function TExRxDBGrid.CreateEditor: TInplaceEdit;
begin
    Result := inherited CreateEditor;

    TInplaceEditCracker(Result).Color := FEditColor;
    FInplaceEdit := Result;
end;

function TExRxDBGrid.GetDefaultRowHeight: Integer;
begin
    Result := inherited DefaultRowHeight;
end;

function TExRxDBGrid.GetOptWidth: Integer;
var
    Loop: Integer;
begin
    Result := GetSystemMetrics(sm_CXVScroll);

    if BorderStyle = bsSingle then
        Inc(Result, 2 * GetSystemMetrics(sm_CXBorder));

    for Loop := 0 to Pred(ColCount) do
        Inc(Result, Succ(ColWidths[Loop]));

    if Parent is TForm then
        begin
            if Result > TForm(Parent).ClientWidth then
                Result := TForm(Parent).ClientWidth;
        end
    else
        if Result > Parent.Width then
            Result := Parent.Width;
end;

function TExRxDBGrid.CalcHintRect(MaxWidth: Integer;
    const AHint: string; HintWnd: THintWindow): TRect;
begin
    Result := HintWnd.CalcHintRect(Screen.Width, AHint, nil)
end;

procedure TExRxDBGrid.DoHint(X, Y: Integer);
const
    TextOffset = 2;
var
    Col, Row, LogCol, LogRow: Longint;
    R, OldR: TRect;
    Pt: TPoint;
    GPt: TGridCoord;
    OldActive: Integer;
    Text: string;
begin
    GPt := MouseCoord(X, Y);
    Col := GPt.X;
    Row := GPt.Y;
    LogCol := Col;
    LogRow := Row;

    if dgTitles in Options then
        Dec(LogRow);

    if dgIndicator in Options then
        Dec(LogCol);

    Text := '';

    if (LogCol >= 0) and (LogRow >= 0) then
        begin
            OldActive := DataLink.ActiveRecord;
            try
                Datalink.ActiveRecord := LogRow;

                if not (Columns[LogCol].Field is TMemoField) then
                    Text := Columns[LogCol].Field.DisplayText
                else
                    begin
                        Text := Columns[LogCol].Field.AsString;
                    end;
            finally
                Datalink.ActiveRecord := OldActive
            end
        end;

    Canvas.Font := Font;
    if (Text <> '') and not EditorMode and
        (Canvas.TextWidth(Text) + TextOffset > ColWidths[Col]) and
        not (csDesigning in ComponentState) then
        begin
            if not Assigned(FHintWnd) then
                begin
                    FHintWnd := HintWindowClass.Create(Self);
                    FHintWnd.Color := Application.HintColor;
                end;
            Hint := Text;
            R := CalcHintRect(Screen.Width, Hint, FHintWnd);

            Pt := ClientToScreen(CellRect(Col, Row).TopLeft);
            Dec(Pt.X);
            Dec(Pt.Y);
            OffsetRect(R, Pt.X, Pt.Y);
            if R.Right > Screen.Width then
                OffsetRect(R, Screen.Width - R.Right, 0);
            if R.Bottom > Screen.Height then
                OffsetRect(R, Screen.Height - R.Bottom, 0);

            GetWindowRect(FHintWnd.Handle, OldR);
            if not IsWindowVisible(FHintWnd.Handle) or
                not ((R.Left = OldR.Left) and (R.Top = OldR.Top)) then
                FHintWnd.ActivateHint(R, Hint)
        end
    else
        if Assigned(FHintWnd) then
            FHintWnd.ReleaseHandle
end;

procedure TExRxDBGrid.SetDefaultRowHeight(Value: Integer);
begin
    if Value = 0 then
        Value := inherited DefaultRowHeight;

    inherited DefaultRowHeight := Value;

    FNewDefaultRowHeight := Value;
    if dgTitles in Options then
        begin
            Canvas.Font := TitleFont;
            RowHeights[0] := Canvas.TextHeight('Wg') + 4;
        end;
end;

procedure TExRxDBGrid.LayoutChanged;
begin
    inherited;

    SetDefaultRowHeight(FNewDefaultRowHeight);
end;

procedure TExRxDBGrid.RowHeightsChanged;
var
    i, ThisHasChanged, Def: Integer;
begin
    ThisHasChanged := -1;
    Def := DefaultRowHeight;

    for i := Ord(dgTitles in Options) to RowCount do
        if RowHeights[i] <> Def then
            begin
                ThisHasChanged := i;
                Break;
            end;

    if ThisHasChanged <> -1 then
        begin
            SetDefaultRowHeight(RowHeights[ThisHasChanged]);
            RecreateWnd;
        end;

    inherited;
end;

procedure TExRxDBGrid.CellClick(Column: TColumn);
begin
    if not (csDesigning in ComponentState) then
        begin
            if not ReadOnly and Datalink.Active and not Datalink.Readonly then
                begin
                    BeginUpdate;

                    if (FDisplayBool) then
                        begin
                            if (Datalink <> nil) and Datalink.Active and Assigned(Column.Field) and (Column.Field.DataType = ftBoolean) and (CanEditModify = True) then
                                begin
                                    try
                                        if Column.Field.AsBoolean = True then
                                            Column.Field.AsBoolean := False
                                        else
                                            if Column.Field.AsBoolean = False then
                                                Column.Field.AsBoolean := True;
                                    except
                                        Column.Field.Value := NULL;
                                    end;
                                end;
                        end;

                    if (Datalink <> nil) and Datalink.Active and Assigned(Column.Field) and
                        ((Column.Field.DataType = ftGraphic) or (Column.Field.DataType = ftTypedBinary)) then
                        begin
                            Application.CreateForm(TExDBGridCtlsForm, ExDBGridCtlsForm);
                            ExDBGridCtlsForm.BlobCaption := Format('Field: %s', [Column.FieldName]);
                            ExDBGridCtlsForm.BlobField := Column.Field;
                            ExDBGridCtlsForm.DataSource := DataSource;
                            ExDBGridCtlsForm.ShowModal;
                            ExDBGridCtlsForm.Release;
                        end;

                    if (Datalink <> nil) and Datalink.Active and Assigned(Column.Field) and ((Column.Field.DataType = ftMemo) or (Column.Field.DataType = ftFmtMemo)) then
                        begin
                            Application.CreateForm(TExDBGridCtlsForm, ExDBGridCtlsForm);
                            ExDBGridCtlsForm.BlobCaption := Format('Field: %s', [Column.FieldName]);
                            ExDBGridCtlsForm.BlobField := Column.Field;
                            ExDBGridCtlsForm.DataSource := DataSource;
                            ExDBGridCtlsForm.ShowModal;
                            ExDBGridCtlsForm.Release;
                        end;

                    inherited CellClick(Column);

                    EndUpdate;
                end;
        end;
end;

procedure TExRxDBGrid.DrawColumnCell(const Rect: TRect; DataCol: Integer; Column: TColumn; State: TGridDrawState);
var
    Bmp: TBitmap;
    style: UINT;
    TmpRect: TRect;
    FactorX, FactorY: Double;
    TmpWidth, TmpHeight: Integer;
    CWidth, CHeight: Integer;
    BWidth, BHeight: Integer;
    vAlignment: TAlignment;
    I: Integer;
begin
    BeginUpdate;

    if (Column <> nil) and (Column.Field <> nil) and not (gdFixed in State) then
        ApplyCellAttribute(Column.Field, Canvas, State);

    with Canvas do
        begin
            Lock;

            TmpRect := Rect;

            FillRect(Rect);

            if (Column.Field.DataType = ftGraphic) or
                (Column.Field.DataType = ftTypedBinary) then
                begin
                    if (FDisplayImages) then
                        begin
                            try
                                Bmp := TBitmap.Create;
                                Bmp.Assign(Column.Field);
                                begin
                                    CWidth := Rect.Right - Rect.Left + 1 - 1;
                                    CHeight := Rect.Bottom - Rect.Top + 1 - 1;
                                    BWidth := Bmp.Width;
                                    BHeight := Bmp.Height;
                                    try
                                        if BWidth = 0 then
                                            FactorX := 0
                                        else
                                            FactorX := CWidth / BWidth;
                                        if BHeight = 0 then
                                            FactorY := 0
                                        else
                                            FactorY := CHeight / BHeight;
                                    except
                                        FactorX := 1;
                                        FactorY := 1;
                                    end;
                                    if FactorY < FactorX then
                                        begin
                                            TmpWidth := Round(BWidth * FactorY);
                                            TmpHeight := CHeight;
                                        end
                                    else
                                        begin
                                            TmpHeight := Round(BHeight * FactorX);
                                            TmpWidth := CWidth;
                                        end;
                                    SetRect(TmpRect, 0, 0, TmpWidth, TmpHeight);
                                    OffsetRect(TmpRect, Rect.Left + ((CWidth - TmpWidth) div 2), Rect.Top + ((CHeight - TmpHeight) div 2));
                                    StretchDraw(TmpRect, Bmp);
                                end
                            finally
                                Bmp.Free;
                            end;
                        end
                    else
                        begin
                            SetTextAlign(Canvas.Handle, TA_CENTER);
                            I := (Rect.Right + Rect.Left) div 2;
                            TextRect(Rect, I, Rect.Top + 2, Column.Field.DisplayText);

                            SetTextAlign(Canvas.Handle, TA_LEFT);
                        end;
                end
            else
                if Column.Field is TMemoField then
                    begin
                        if (FDisplayMemo) then
                            begin
                                try
                                    TmpRect := Rect;
                                    Inc(TmpRect.Top, 2);
                                    Inc(TmpRect.Left, 2);
                                    DrawText(Canvas.Handle, PChar(Column.Field.AsString), Length(Column.Field.AsString), TmpRect, DT_WORDBREAK or DT_NOPREFIX);
                                finally
                                end;
                            end
                        else
                            begin
                                SetTextAlign(Canvas.Handle, TA_CENTER);
                                I := (Rect.Right + Rect.Left) div 2;
                                TextRect(Rect, I, Rect.Top + 2, Column.Field.DisplayText);

                                SetTextAlign(Canvas.Handle, TA_LEFT);
                            end;
                    end
                else
                    if Column.Field is TBooleanField then
                        begin
                            if (FDisplayBool) then
                                begin
                                    TmpRect := Rect;
                                    TmpRect.Left := Rect.Left + (Rect.Right - Rect.Left - GCheckWidth) div 2;
                                    TmpRect.Right := TmpRect.Left + GCheckWidth;
                                    TmpRect.Top := Rect.Top + (Rect.Bottom - Rect.Top - GCheckWidth) div 2;
                                    TmpRect.Bottom := TmpRect.Top + GCheckHeight;

                                    if Column.Field.AsBoolean = True then
                                        style := DFCS_CHECKED
                                    else
                                        if Column.Field.AsBoolean = False then
                                            style := DFCS_BUTTONCHECK;

                                    DrawFrameControl(Canvas.Handle, TmpRect, DFC_BUTTON, style);
                                end
                            else
                                begin
                                    vAlignment := Column.Field.Alignment;

                                    case vAlignment of
                                        taRightJustify:
                                            begin
                                                SetTextAlign(Canvas.Handle, TA_RIGHT);
                                                I := Rect.Right - 2;
                                            end;

                                        taLeftJustify:
                                            begin
                                                SetTextAlign(Canvas.Handle, TA_LEFT);
                                                I := Rect.Left + 2;
                                            end;

                                        taCenter:
                                            begin
                                                SetTextAlign(Canvas.Handle, TA_CENTER);
                                                I := (Rect.Right + Rect.Left) div 2;
                                            end;
                                    end;
                                    TextRect(Rect, I, Rect.Top + 2, Column.Field.DisplayText);

                                    SetTextAlign(Canvas.Handle, TA_LEFT);
                                end;
                        end
                    else
                        begin
                            vAlignment := Column.Field.Alignment;

                            case vAlignment of
                                taRightJustify:
                                    begin
                                        SetTextAlign(Canvas.Handle, TA_RIGHT);
                                        I := Rect.Right - 2;
                                    end;

                                taLeftJustify:
                                    begin
                                        SetTextAlign(Canvas.Handle, TA_LEFT);
                                        I := Rect.Left + 2;
                                    end;

                                taCenter:
                                    begin
                                        SetTextAlign(Canvas.Handle, TA_CENTER);
                                        I := (Rect.Right + Rect.Left) div 2;
                                    end;
                            end;
                            TextRect(Rect, I, Rect.Top + 2, Column.Field.DisplayText);

                            SetTextAlign(Canvas.Handle, TA_LEFT);
                        end;
            Unlock;
        end;

    inherited DrawColumnCell(Rect, DataCol, Column, State);

    EndUpdate;
end;

procedure TExRxDBGrid.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
begin
    inherited;

    if (FCells3D = True) and ([dgRowLines, dgColLines] * Options = [dgRowLines, dgColLines]) then
        begin
            with ARect do
                begin
                    if (Color = clWhite) or (Color = clWindow) then
                        Color := TExRxDBGrid(Self).FixedColor;

                    Canvas.Pen.Color := clHighLightText;
                    Canvas.PolyLine([Point(Left, Bottom - 1), Point(Left, Top), Point(Right, Top)]);
                end;
        end
    else
        begin
            with ARect do
                begin
                    if (Color = clBtnFace) or (Color = clSilver) then
                        Color := clWindow;

                    if ARow = 0 then
                        begin
                            if dgTitles in Options then
                                begin
                                    Canvas.Pen.Color := TExRxDBGrid(Self).FixedColor;
                                    Canvas.PolyLine([Point(Left + 1, Bottom - 1), Point(Right - 1, Bottom - 1), Point(Right - 1, Top)]);

                                    Canvas.Pen.Color := clBtnShadow;
                                    Canvas.PolyLine([Point(Left - 1, Bottom), Point(Right, Bottom), Point(Right, Top - 1)]);
                                end
                            else
                                begin
                                    Canvas.Pen.Color := TExRxDBGrid(Self).FixedColor;
                                    Canvas.PolyLine([Point(Left, Bottom), Point(Right, Bottom), Point(Right, Top - 1)]);
                                end;

                            if dgIndicator in Options then
                                begin
                                    if ACol = 0 then
                                        begin
                                            Canvas.Pen.Color := TExRxDBGrid(Self).FixedColor;
                                            Canvas.PolyLine([Point(Left + 1, Bottom - 1), Point(Right - 1, Bottom - 1), Point(Right - 1, Top)]);

                                            Canvas.Pen.Color := clBtnShadow;
                                            Canvas.PolyLine([Point(Left - 1, Bottom), Point(Right, Bottom), Point(Right, Top - 1)]);
                                        end;
                                end;
                        end
                    else
                        if ARow > 0 then
                            begin
                                if dgIndicator in Options then
                                    begin
                                        if ACol = 0 then
                                            begin
                                                Canvas.Pen.Color := TExRxDBGrid(Self).FixedColor;
                                                Canvas.PolyLine([Point(Left + 1, Bottom - 1), Point(Right - 1, Bottom - 1), Point(Right - 1, Top)]);

                                                Canvas.Pen.Color := clBtnShadow;
                                                Canvas.PolyLine([Point(Left - 1, Bottom), Point(Right, Bottom), Point(Right, Top - 1)]);
                                            end
                                        else
                                            begin
                                                Canvas.Pen.Color := TExRxDBGrid(Self).FixedColor;
                                                Canvas.PolyLine([Point(Left, Bottom), Point(Right, Bottom), Point(Right, Top - 1)]);
                                            end;
                                    end
                                else
                                    begin
                                        Canvas.Pen.Color := TExRxDBGrid(Self).FixedColor;
                                        Canvas.PolyLine([Point(Left, Bottom), Point(Right, Bottom), Point(Right, Top - 1)]);
                                    end;
                            end;
                end;
        end;
end;

procedure TExRxDBGrid.ColWidthsChanged;
begin
    inherited;
    try
        if Assigned(FOnColResize) then
            FOnColResize(TExRxDBGrid(Self));
    except
    end;
end;

procedure TExRxDBGrid.SetCheckBoxSize;
begin
    with TBitmap.Create do
        try
            Handle := LoadBitmap(0, PChar(32759));
            GCheckWidth := Width div 4;
            GCheckHeight := Height div 3;
        finally
            Free;
        end;
end;

procedure TExRxDBGrid.ApplyCellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState);
begin
    if Assigned(FOnApplyCellAttribute) then
        FOnApplyCellAttribute(Self, Field, Canvas, State);
end;

procedure TExRxDBGrid.CellAttribute(Field: TField; Canvas: TCanvas; State: TGridDrawState);
begin
    ApplyCellAttribute(Field, Canvas, State);
end;

procedure TExRxDBGrid.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
    if not AcquireFocus then
        Exit;

    if Assigned(FOnMouseDown) then
        FOnMouseDown(Self, Button, Shift, X, Y);

    inherited MouseDown(Button, Shift, X, Y);
end;

procedure TExRxDBGrid.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
    if Assigned(FOnMouseUp) then
        FOnMouseUp(Self, Button, Shift, X, Y);

    inherited MouseUp(Button, Shift, X, Y);
end;

procedure TExRxDBGrid.ColEnter;
begin
    if not (csDesigning in ComponentState) then
        begin
            if not ReadOnly and Datalink.Active and not Datalink.Readonly then
                begin
                    with Columns[SelectedIndex] do
                        if (Field.DataType = ftBoolean) or
                            (Field.DataType = ftGraphic) or
                            (Field.DataType = ftMemo) or
                            (Field.DataType = ftFmtMemo) or
                            (Field.DataType = ftTypedBinary) then
                            begin
                                if (goEditing in TExRxDBGridMod(Self).Options) then
                                    begin
                                        BeginUpdate;
                                        TExRxDBGridMod(Self).Options := TExRxDBGridMod(Self).Options - [goEditing];
                                        EndUpdate;
                                    end;
                            end
                        else
                            begin
                                if not (goEditing in TExRxDBGridMod(Self).Options) then
                                    begin
                                        BeginUpdate;
                                        TExRxDBGridMod(Self).Options := TExRxDBGridMod(Self).Options + [goEditing];
                                        EndUpdate;
                                    end;
                            end;

                    if TExRxDBGrid(Self).SelectedField.Lookup = True then
                        begin
                            TExRxDBGrid(Self).EditorMode := True;
                            keybd_event(VK_MENU, MapVirtualKey(VK_MENU, 0), 0, 0);
                            keybd_event(VK_DOWN, MapVirtualKey(VK_DOWN, 0), 0, 0);
                            keybd_event(VK_DOWN, 0, KEYEVENTF_KEYUP, 0);
                            keybd_event(VK_MENU, 0, KEYEVENTF_KEYUP, 0);
                        end;

                    if Assigned(FOnColEnter) then
                        FOnColEnter(Self);
                end;
        end;
end;

procedure TExRxDBGrid.WMSize(var Message: TWMSize);
begin
    try
        inherited;
        if (Datalink <> nil) and Datalink.Active then
            begin
                //SetFullSizeMemo(FFullSizeMemo);
                if Columns.State <> csCustomized then
                    SetGridAutoSize(FGridAutoSize);
            end;

        invalidate;
    except
    end;
end;

procedure TExRxDBGrid.CMFontChanged(var Message: TMessage);
var
    h: integer;
begin
    inherited;

    Canvas.Font.Assign(font);
    h := Canvas.TextHeight('Wg');

    if FRowHeight < h then
        SetDefaultRowHeight(h);

    if (csdesigning in componentstate) then
        invalidate;
end;

procedure TExRxDBGrid.KeyDown(var Key: Word; Shift: TShiftState);
const
    vk_c = $43;
    vk_v = $56;
begin
    inherited KeyDown(Key, Shift);

    if not (csDesigning in ComponentState) then
        begin
            if not ReadOnly and Datalink.Active and not Datalink.Readonly then
                begin
                    BeginUpdate;

                    if Shift = [ssCtrl] then
                        begin
                            if key = vk_v then
                                Shift := [ssShift];
                            if (key = vk_c) or (key = vk_v) then
                                begin
                                    gk := Key;
                                    key := 0;
                                end;
                        end;

                    if Key = 32 then
                        begin
                            if (FDisplayBool) then
                                if (Columns[SelectedIndex].Field.DataType = ftBoolean) then
                                    begin
                                        if not Columns[SelectedIndex].Field.ReadOnly then
                                            CellClick(Columns[SelectedIndex]);
                                    end;

                            if (Columns[SelectedIndex].Field.DataType = ftGraphic) or
                                (Columns[SelectedIndex].Field.DataType = ftMemo) or
                                (Columns[SelectedIndex].Field.DataType = ftTypedBinary) or
                                (Columns[SelectedIndex].Field.DataType = ftFmtMemo) then
                                begin
                                    CellClick(Columns[SelectedIndex]);
                                end;
                        end;

                    EndUpdate;
                end;
        end;
end;

procedure TExRxDBGrid.KeyPress(var Key: Char);
const
    vk_c = $43;
    vk_v = $56;
begin
    inherited KeyPress(Key);

    if not (csDesigning in ComponentState) then
        begin
            if not ReadOnly and Datalink.Active and not Datalink.Readonly then
                begin
                    BeginUpdate;

                    if gk <> 0 then
                        begin
                            Key := chr(0);
                            if gk = vk_c then
                                ClipBoard.AsText := Columns[SelectedIndex].Field.AsString;
                            if gk = vk_v then
                                begin
                                    if (Datalink.DataSet.State = dsEdit) or (Datalink.DataSet.State = dsInsert) then
                                        Columns[SelectedIndex].Field.AsString := ClipBoard.AsText
                                    else
                                        MessageBeep(0);
                                end;
                            gk := 0;
                        end;

                    EndUpdate;
                end;
        end;
end;

{ END PROTECTED }

{ BEGIN PUBLIC }

function TExRxDBGrid.CellRect(ACol, ARow: Longint): TRect;
begin
    Result := inherited CellRect(ACol, ARow);
end;

procedure TExRxDBGrid.MoveToRow(NewRow: Integer);
var
    Mark: TBookmarkStr;
begin
    Mark := DataLink.DataSet.Bookmark;
    DataLink.ActiveRecord := NewRow;
    DbiSetToBookmark(TDBDataSet(DataLink.DataSet).Handle, Pointer(Mark));
    DataLink.DataSet.Resync([rmExact]);
end;

procedure TExRxDBGrid.WMHScroll(var Msg: TWMHScroll);
begin
    BeginUpdate;
    inherited;
    if Msg.ScrollCode = sb_ThumbTrack then
        Perform(wm_HScroll, MakeLong(sb_ThumbPosition, Msg.Pos), Msg.ScrollBar);
    EndUpdate;
end;

procedure TExRxDBGrid.WMVScroll(var Msg: TWMVScroll);
begin
    BeginUpdate;
    inherited;
    if Msg.ScrollCode = sb_ThumbTrack then
        Perform(wm_VScroll, MakeLong(sb_ThumbPosition, Msg.Pos), Msg.ScrollBar);
    EndUpdate;
end;

procedure TExRxDBGrid.CMMouseEnter(var Msg: TMessage);
var
    Pt: TPoint;
begin
    if FCellHints = True then
        begin
            GetCursorPos(Pt);
            Pt := ScreenToClient(Pt);
            DoHint(Pt.X, Pt.Y);
        end;
end;

procedure TExRxDBGrid.CMMouseLeave(var Msg: TMessage);
begin
    inherited;
    if FCellHints = True then
        if Assigned(FHintWnd) then
            begin
                FHintWnd.Destroy;
                FHintWnd := nil;
            end;
end;

procedure TExRxDBGrid.WMMouseMove(var Msg: TWMMouseMove);
begin
    inherited;
    if FCellHints = True then
        DoHint(Msg.XPos, Msg.YPos)
end;

{ END PUBLIC }

destructor TExRxDBGrid.Destroy;
begin
    inherited Destroy;
end;

procedure Register;
begin
    RegisterComponents('GJL Software', [TExRxDBGrid]);
end;

end.

