unit DsRadio;

// DsRadio Release 1.1 for Delphi 3 and up
// Copyright (c) 1997, 1998, 1999 by Djoko Susilo
// djokos@cabi.net.id
// ------------- Q14 Project -----------------------
// Files needed to install:
// DsRadio.pas  (this file)
// DsRadio.dcr  (resource for palette icon)
// -------------------------------------------------
// update 20-Jun-99
//   add properties Color_High, Color_Low, Alignment
// update 12-Okt-99---------------------------------
//   check Caption length when alignment is taRight
//--------------------------------------------------

interface

uses
  SysUtils, Windows, Classes, Graphics, Messages, Controls, StdCtrls, Forms;

type
  TAlignment = (taLeft, taRight);
  TButtonShape = (btRound, btSquare, btDiamond);
  TTextStyle = (txNone, txRaised, txLowered);
  TDsRadio = class(TCustomControl)
  private
    FAlignment: TAlignment;
    FButtonShape: TButtonShape;
    FCheckColor: TColor;
    FChecked: Boolean;
    FColor_High: TColor;
    FColor_Low: TColor;
    FDown: Boolean;
    FFocused: Boolean;
    FGroupIndex: Byte;
    FTextStyle: TTextStyle;
    procedure TurnRadOthersOff;
    procedure CMEnabledChanged(var Message: TMessage);
              message CM_ENABLEDCHANGED;
    procedure CMTextChanged(var Message: TMessage);
              message CM_TEXTCHANGED;
    procedure CMDialogChar(var Message: TCMDialogChar);
              message CM_DIALOGCHAR;
    procedure WMSetFocus(var Message: TWMSetFocus);
              message WM_SETFOCUS;
  protected
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure KeyDown(var Key: Word; Shift: TShiftSTate); override;
    procedure KeyUp(var Key: Word; Shift: TShiftSTate); override;
    procedure MouseDown(Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer); override;
    procedure Paint; override;
    procedure SetAlignment(Value: TAlignment);
    procedure SetButtonShape(Value: TButtonShape);
    procedure SetCheckColor(Value: TColor);
    procedure SetChecked(Value: Boolean);
    procedure SetColor_High(Value: TColor);
    procedure SetColor_Low(Value: TColor);
    procedure SetDown(Value: Boolean);
    procedure SetTextStyle(Value: TTextStyle);

  public
    constructor Create(AOwner: TComponent); override;
  published
    property Alignment: TAlignment
             read FAlignment write SetAlignment default taLeft;
    property ButtonShape: TButtonShape
             read FButtonShape write SetButtonShape default btDiamond;
    property Caption;
    property Color;
    property CheckColor: TColor
             read FCheckColor write SetCheckColor default clBlack;
    property Checked: Boolean
             read FChecked write SetChecked default False;
    property Color_High: TColor
             read FColor_High write SetColor_High default $00DDE6EA;
    property Color_Low: TColor
             read FColor_Low write SetColor_Low default $00688DA2;
    property Down: Boolean
             read FDown write SetDown default False;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property GroupIndex: Byte
             read FGroupIndex write FGroupIndex default 0;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property TextStyle: TTextStyle
             read FTextStyle write SetTextStyle default txRaised;
    property Visible;
    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

constructor TDsRadio.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAlignment:= taLeft;
  FButtonShape:=btDiamond;
  FColor_High:=$00DDE6EA;
  FColor_Low:=$00688DA2;
  FTextStyle:=txRaised;
  Height:=17;
  ParentColor:=True;
  ParentFont:=True;
  Width:=81;
end;

procedure TDsRadio.Paint;
var XL, YT, XR, YB, XM, YM: Integer;
    TextX, TextY, n: Integer;
    R, TR: TRect;
    ColorUp, ColorDown: TColor;
    Flags: word;
    P: array[0..3] of TPoint;
begin
  { button width = 13 }
  if FAlignment=taLeft then begin XL:=0; XR:=XL+12; end
  else begin XR:=Width-1; XL:=XR-12; end;

  YT:=(Height-12) div 2-1;  YB:=YT+12;
  XM:=XL+6;                 YM:=YT+6;

  P[0].X:=XL+3;  P[0].Y:=YM;
  P[1].X:=XM;    P[1].Y:=YT+3;
  P[2].X:=XR-3;  P[2].Y:=YM;
  P[3].X:=XM;    P[3].Y:=YB-3;

  with Canvas do
  begin
    Brush.Color:=Color;
    if Down then  // posisi tombol ditekan
    begin
      Pen.Color:=CheckColor; //clBlack;
      case ButtonShape of
        btDiamond : begin
                      MoveTo(XM, YT);
                      LineTo(XL, YM); LineTo(XM, YB);
                      LineTo(XR, YM); LineTo(XM, YT);
                    end;
        btSquare  : RoundRect(XL, YT, XR, YB, 2, 2);
        else        Ellipse(XL, YT, XR, YB);
      end;
    end
    else   // posisi tombol tidak ditekan
    begin
      if Checked then
        begin ColorUp:=FColor_Low; ColorDown:=FColor_High; end
      else begin ColorUp:=FColor_High; ColorDown:=FColor_Low; end;

      case ButtonShape of
        btDiamond : begin
                      Pen.Color:=ColorUp;
                      MoveTo(XM, YT); LineTo(XL, YM); LineTo(XM, YB);
                      Pen.Color:=ColorDown;
                      LineTo(XR, YM); LineTo(XM, YT);
                    end;
        btSquare  : begin
                      Pen.Color:=ColorUp;
                      MoveTo(XL, YB-1); LineTo(XL, YT); LineTo(XR-1, YT);
                      Pen.Color:=ColorDown;
                      LineTo(XR-1, YB-1); LineTo(XL+1, YB-1);
                    end;
        else        begin  {round}
                      Pen.Color:=ColorUp;
                      Arc(XL, YT, XR, YB, XR-2, YT+2, XL+2, YB-2);
                      Pen.Color:=ColorDown;
                      Arc(XL, YT, XR, YB, XL+2, YB-2, XR-2, YT+2);
                    end;
      end;
    end;

    // menggambar titik di tengah tombol
    if Checked then
    begin
      if Enabled then Pen.Color:=CheckColor
      else Pen.Color:=FColor_Low;
      Brush.Color:=Pen.Color; // fill color
      case ButtonShape of
        btDiamond : Polygon(P);
        btSquare  : Rectangle(XL+3, YT+3, XR-3, YB-3);
        else        Ellipse(XL+3, YT+3, XR-3, YB-3);
      end;
    end;

    // pengaturan teks dengan efek penulisannya
    Flags:=DT_LEFT or DT_SINGLELINE;
    TR:=ClientRect;
    R:=ClientRect;
    Font:=Self.Font;
    if Alignment=taLeft then
    begin
      TextX:=XR+5;
    end
    else
    begin
      R.Right:=Width-18;  // batas penulisan Caption
      TextX:=1;
    end;
    TextY:=(Height-TextHeight(Caption)) div 2-1;
    Brush.Style:=bsClear;

    case TextStyle of
      txRaised  : begin
                    Font.Color:=FColor_High;
                    OffsetRect(R, TextX-1, TextY-1);
                    DrawText(Handle, PChar(Caption), Length(Caption), R, Flags);
                  end;
      txLowered : begin
                    Font.Color:=FColor_High;
                    OffsetRect(R, TextX+1, TextY+1);
                    DrawText(Handle, PChar(Caption), Length(Caption), R, Flags);
                  end;
    end;

    if Enabled then Font.Color:=Self.Font.Color
    else Font.Color:=FColor_Low;

    if Alignment=taRight then TR.Right:=Width-18;  // batas penulisan Caption
    OffsetRect(TR, TextX, TextY);
    DrawText(Handle, PChar(Caption), Length(Caption), TR, Flags);

    if pos('&', Caption)>0 then n:=TextWidth('&') else n:=0;
    Brush.Color:=Color;

    if Alignment=taLeft then
      R:=Bounds(TextX-1, TextY, TextWidth(Caption)-n+3, TextHeight(Caption)+1)
    else
    begin
      if TextWidth(Caption)-n+3<Width-18 then
        R:=Bounds(TextX-1, TextY, TextWidth(Caption)-n+3, TextHeight(Caption)+1)
      else
        R:=Bounds(TextX-1, TextY, Width-17, TextHeight(Caption)+1);
    end;
    FrameRect(R);
    if FFocused then DrawFocusRect(R);
  end;
end;

procedure TDsRadio.SetTextStyle(Value: TTextStyle);
begin
  if Value<>FTextStyle then
  begin
    FTextStyle:=Value;
    Invalidate;
  end;
end;

procedure TDsRadio.SetAlignment(Value: TAlignment);
begin
  if Value<>FAlignment then
  begin
    FAlignment:=Value;
    Invalidate;
  end;
end;

procedure TDsRadio.SetButtonShape(Value: TButtonShape);
begin
  if Value<>FButtonShape then
  begin
    FButtonShape:=Value;
    Invalidate;
  end;
end;

procedure TDsRadio.SetDown(Value: Boolean);
begin
  if Value<>FDown then
  begin
    FDown:=Value;
    Invalidate;
  end;
end;

procedure TDsRadio.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  invalidate;
end;

procedure TDsRadio.WMSetFocus(var Message: TWMSetFocus);
begin // fix double focus rect drawing bug in Ctl3D when switching notebook pages
  if Ctl3D and not NewStyleControls then UpdateWindow(Handle);
  inherited;
end;

procedure TDsRadio.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
  Realign;
end;

procedure TDsRadio.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, Caption) and CanFocus then
    begin
      SetFocus;
      if Focused then Checked:=True;
      Result:=1;
    end
    else inherited;
end;

procedure TDsRadio.TurnRadOthersOff;
var i:Integer;
    RadOther: TDsRadio;
begin
  if Parent<>nil then
    for i:=0 to Parent.ControlCount-1 do
      if Parent.Controls[i] is TDsRadio then
      begin
        RadOther:=TDsRadio(Parent.Controls[i]);
        if (RadOther<>Self) and
           (RadOther.GroupIndex=GroupIndex) then
           RadOther.SetChecked(False);
      end;
end;

procedure TDsRadio.SetChecked(Value: Boolean);
begin
  if Value<>FChecked then
  begin
    TabStop:=Value;
    FChecked:=Value;
    if Value then
    begin
      TurnRadOthersOff;
      inherited Changed;
    end;
    Repaint;
  end;
end;

procedure TDsRadio.SetCheckColor(Value: TColor);
begin
  if Value<>FCheckColor then
  begin
    FCheckColor:=Value;
    Invalidate; //Paint;
  end;
end;

procedure TDsRadio.SetColor_High(Value: TColor);
begin
  if Value<>FColor_High then
  begin
    FColor_High:=Value;
    Invalidate;
  end;
end;

procedure TDsRadio.SetColor_Low(Value: TColor);
begin
  if Value<>FColor_Low then
  begin
    FColor_Low:=Value;
    Invalidate;
  end;
end;

procedure TDsRadio.DoEnter;
begin
  inherited DoEnter;
  FFocused:=True;
  Checked:=True;
  Paint;
end;

procedure TDsRadio.DoExit;
begin
  inherited DoExit;
  FFocused:=False;
  Paint;
end;

procedure TDsRadio.MouseDown(Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  SetFocus;
  FFocused:=True;
  inherited MouseDown(Button, Shift, X, Y);
  MouseCapture:=True;
  Down:=True;
end;

procedure TDsRadio.MouseUp(Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  MouseCapture:=False;
  Down:=False;
  if (X>=0) and (X<=Width) and (Y>=0) and (Y<=Height)
    and not Checked then Checked:=True;
  inherited MouseUp(Button, Shift, X, Y);
end;

procedure TDsRadio.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  if MouseCapture then
    Down:=(X>=0) and (X<=Width) and (Y>=0) and (Y<=Height);
  inherited MouseMove(Shift, X, Y);
end;

procedure TDsRadio.KeyDown(var Key: Word; Shift: TShiftSTate);
begin
  if Key=vk_Space then Down:=True;
    inherited KeyDown(Key, Shift);
end;

procedure TDsRadio.KeyUp(var Key: Word; Shift: TShiftSTate);
begin
  if Key=vk_Space then
  begin
    Down:=False;
    if not Checked then Checked:=True;
  end;
end;

procedure Register;
begin
  RegisterComponents('My Compo', [TDsRadio]);
end;

end.
