unit DsCheck;

// DsCheck Release 1.1 for Delphi 3 and up
// Copyright (c) 1997~1999 by Djoko Susilo
// djokos@cabi.net.id
// ------------- Q14 Project -------------
// Files needed to install:
// DsCheck.pas  (this file)
// DsCheck.dcr  (resource for icon)
//----------------------------------------
// update 1-Jun-99:
//   add properties Color_High, Color_Low
//                  CheckStyle
//----------------------------------------

interface

uses
  SysUtils, Windows, StdCtrls, Classes, Messages, Graphics, Controls, Forms;

type
  TAlignment = (taLeft, taRight);
  TCheckStyle = (ckRaised, ckFlat);
  TTextStyle = (txNone, txRaised, txLowered);
  TDsCheck = class(TCustomControl)
  private
    FAlignment: TAlignment;
    FCheckColor: TColor;
    FCheckStyle: TCheckStyle;
    FColor_High: TColor;
    FColor_Low: TColor;
    FDown: Boolean;
    FFocused: Boolean;
    FState: TCheckBoxState;
    FTextStyle: TTextStyle;

    procedure CMEnabledChanged(var Message: TMessage);
              message CM_ENABLEDCHANGED;
    procedure CMTextChanged(var Message: TMessage);
              message CM_TEXTCHANGED;
    procedure CMDialogChar(var Message: TCMDialogChar);
              message CM_DIALOGCHAR;
    procedure WMSetFocus(var Message: TWMSetFocus);
              message WM_SETFOCUS;
  protected
    procedure DoEnter; override;
    procedure DoExit; override;
    function  GetChecked: Boolean;
    procedure KeyDown(var Key: Word;
              Shift: TShiftSTate); override;
    procedure KeyUp(var Key: Word;
              Shift: TShiftSTate); override;
    procedure MouseDown(Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState;
              X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton;
              Shift: TShiftState; X, Y: Integer); override;
    procedure Paint; override;
    procedure SetAlignment(Value: TAlignment);
    procedure SetCheckColor(Value: TColor);
    procedure SetCheckStyle(Value: TCheckStyle);
    procedure SetChecked(Value: Boolean);
    procedure SetColor_High(Value: TColor);
    procedure SetColor_Low(Value: TColor);
    procedure SetDown(Value: Boolean);
    procedure SetState(Value: TCheckBoxState);
    procedure SetTextStyle(Value: TTextStyle);
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Alignment: TAlignment
             read FAlignment write SetAlignment default taLeft;
    property Caption;
    property CheckColor: TColor
             read FCheckColor write SetCheckColor default clBlack;
    property Checked: Boolean
             read GetChecked write SetChecked default False;
    property CheckStyle: TCheckStyle
             read FCheckStyle write SetCheckStyle default ckRaised;
    property Color;
    property Color_High: TColor
             read FColor_High write SetColor_High default $00E0E8E8;
    property Color_Low: TColor
             read FColor_Low write SetColor_Low default $00688DA2;
    property Down: Boolean
             read FDown write SetDown default False;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property State: TCheckBoxState
             read FState write SetState default cbUnchecked;
    property TabOrder;
    property TabStop;
    property TextStyle: TTextStyle
             read FTextStyle write SetTextStyle default txRaised;
    property Visible;
    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

constructor TDsCheck.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Alignment:=taLeft;
  FCheckStyle:=ckRaised;
  FColor_High:=$00E0E8E8;
  FColor_Low:=$00688DA2;
  FFocused:=True;
  FTextStyle:=txRaised;
  Height:=17;
  ParentColor:=True;
  Width:=81;
end;

procedure TDsCheck.SetColor_High(Value: TColor);
begin
  if Value<>FColor_High then
  begin
    FColor_High:=Value;
    Invalidate;
  end;
end;

procedure TDsCheck.SetColor_Low(Value: TColor);
begin
  if Value<>FColor_Low then
  begin
    FColor_Low:=Value;
    Invalidate;
  end;
end;

procedure TDsCheck.Paint;
var XL, YT, XR, YB: Integer;
    TextX, TextY, n: Integer;
    ColorUp, ColorDown: TColor;
    R, TR: TRect;
    Flags: word;

    procedure DrawCheck(Clr: TColor);
    var X, Y: Integer;
    begin
      X:=XL+2; Y:=YT+5;
      with canvas do
      begin
        Pen.Width:=1;
        Pen.Color:=clBtnHighlight; //FColor_High;
        MoveTo(X+2, Y-1);
        LineTo(X+2, Y+4);
        LineTo(X+9, Y-3);

        if Enabled then Pen.Color:=Clr
        else Pen.Color:=clBtnShadow; //FColor_Low;
        MoveTo(X, Y);
        LineTo(X, Y+4);
        LineTo(X+8, Y-4);
        MoveTo(X+1, Y-1);
        LineTo(X+1, Y+4);
        LineTo(X+9, Y-4);
      end;
    end;

begin
  { button width = 12 }
  case FAlignment of
    taLeft: begin XL:=1; XR:=XL+11; end
    else    begin XR:=Width-1; XL:=XR-11; end;
  end;

  YT:=(Height-11) div 2-1;  YB:=YT+11;
  with Canvas do
  begin
    if FDown then
    begin
      ColorUp:=FColor_Low;
      ColorDown:=Color_High; //clBtnHighlight;
    end
    else
    begin
      ColorUp:=Color_High; //clBtnHighlight;
      ColorDown:=FColor_Low;
    end;

    Pen.Color:=ColorUp;
    MoveTo(XL, YB);
    LineTo(XL, YT);
    LineTo(XR, YT);
    Pen.Color:=ColorDown;
    LineTo(XR, YB);
    LineTo(XL, YB);

    if (CheckStyle=ckFlat) and (not FDown) then
    begin
      Pen.Color:=ColorDown;
      MoveTo(XL-1, YB);
      LineTo(XL-1, YT-1);
      LineTo(XR+1, YT-1);
      Pen.Color:=ColorUp;
      LineTo(XR+1, YB+1);
      LineTo(XL-1, YB+1);
    end;

    // menulis teks dengan efek penulisan 3D
    Flags:=DT_LEFT or DT_SINGLELINE;
    Font:=Self.Font;
    R:=ClientRect;

    if FAlignment=taLeft then TextX:=XR+5
    else begin TextX:=1; R.Right:=Width-15; end;

    TR:=R; //ClientRect;

    case CheckStyle of
      ckRaised : TextY:=(Height-TextHeight(Caption)) div 2;
      else       TextY:=(Height-TextHeight(Caption)) div 2-1;
    end;

    Brush.Style:=bsClear;

    case TextStyle of
      txRaised:
        begin
          Font.Color:=Color_High;
          OffsetRect(R, TextX-1, TextY-1);
          DrawText(Handle, PChar(Caption), Length(Caption), R, Flags);
        end;
      txLowered:
        begin
          Font.Color:=Color_High;
          OffsetRect(R, TextX+1, TextY+1);
          DrawText(Handle, PChar(Caption), Length(Caption), R, Flags);
        end;
    end;

    if Enabled then Font.Color:=Self.Font.Color
    else Font.Color:=FColor_Low;

    OffsetRect(TR, TextX, TextY);
    DrawText(Handle, PChar(Caption), Length(Caption), TR, Flags);

    if State=cbChecked then DrawCheck(FCheckColor)
    else if State=cbGrayed then
    begin
      if Down then
      begin
        Pen.Color:=clBtnFace;
        Brush.Color:=clBtnFace;
        Rectangle(XL+2, YT+2, XR-1, YB-1);
      end;
      DrawCheck(FColor_Low);
    end;

    if pos('&', Caption)>0 then n:=TextWidth('&') else n:=0;
    Brush.Color:=Color;

    if FAlignment=taLeft then
      R:=Bounds(TextX-1, TextY, TextWidth(Caption)-n+3, TextHeight(Caption)+1)
    else
      begin
        if TextWidth(Caption)-n+3<Width-15 then
          R:=Bounds(TextX-1, TextY, TextWidth(Caption)-n+3, TextHeight(Caption)+1)
      else
          R:=Bounds(TextX-1, TextY, Width-15, TextHeight(Caption)+1);
      end;
    FrameRect(R);
    if Focused then DrawFocusRect(R);
  end;
end;

procedure TDsCheck.SetAlignment(Value: TAlignment);
begin
  if Value<>FAlignment then
  begin
    FAlignment:=Value;
    Invalidate;
  end;
end;

procedure TDsCheck.SetCheckStyle(Value: TCheckStyle);
begin
  if Value<>FCheckStyle then
  begin
    FCheckStyle:=Value;
    Invalidate;
  end;
end;

procedure TDsCheck.SetDown(Value: Boolean);
begin
  if Value<>FDown then
  begin
    FDown:=Value;
    Invalidate;
  end;
end;

procedure TDsCheck.SetState(Value: TCheckBoxState);
begin
  if Value<>FState then
  begin
    FState:=Value;
    Invalidate;
    Click;
  end;
end;

procedure TDsCheck.SetTextStyle(Value: TTextStyle);
begin
  if Value<>FTextStyle then
  begin
    FTextStyle:=Value;
    Invalidate;
  end;
end;

function TDsCheck.GetChecked: Boolean;
begin
  Result:=State=cbChecked;
end;

procedure TDsCheck.SetChecked(Value: Boolean);
begin
  if Value then State:=cbChecked
  else State:=cbUnchecked;
end;

procedure TDsCheck.SetCheckColor(Value: TColor);
begin
  if Value<>FCheckColor then
  begin
    FCheckColor:=Value;
    Paint;
  end;
end;

procedure TDsCheck.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  invalidate;
end;

procedure TDsCheck.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TDsCheck.WMSetFocus(var Message: TWMSetFocus);
begin // fix double focus rect drawing bug in Ctl3D when switching notebook pages
  if Ctl3D and not NewStyleControls then UpdateWindow(Handle);
  inherited;
end;

procedure TDsCheck.DoEnter;
begin
  inherited DoEnter;
  FFocused:=True;
  Paint;
end;

procedure TDsCheck.DoExit;
begin
  inherited DoExit;
  FFocused:=False;
  Paint;
end;

procedure TDsCheck.MouseDown(Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  SetFocus;
  FFocused:=True;
  inherited MouseDown(Button, Shift, X, Y);
  MouseCapture:=True;
  Down:=True;
end;

procedure TDsCheck.MouseUp(Button: TMouseButton;
          Shift: TShiftState; X, Y: Integer);
begin
  MouseCapture:=False;
  Down:=False;
  if (X>=0) and (X<=Width) and (Y>=0) and (Y<=Height) then
    Checked:=not Checked;
  inherited MouseUp(Button, Shift, X, Y);
end;

procedure TDsCheck.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  if MouseCapture then
    Down:=(X>=0) and (X<=Width) and (Y>=0) and (Y<=Height);
  inherited MouseMove(Shift, X, Y);
end;

procedure TDsCheck.KeyDown(var Key: Word; Shift: TShiftSTate);
begin
  if Key=vk_Space then Down:=True;
  inherited KeyDown(Key, Shift);
end;

procedure TDsCheck.KeyUp(var Key: Word; Shift: TShiftSTate);
begin
  if Key=vk_Space then
  begin
    Down:=False;
    Checked:=not Checked;
  end;
end;

procedure TDsCheck.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, Caption) and CanFocus then
    begin
      SetFocus;
      if Focused then
      if Checked then Checked:=False else Checked:=True;
      Result:=1;
    end
    else inherited;
end;

procedure Register;
begin
  RegisterComponents('My Compo', [TDsCheck]);
end;

end.
