unit DBSumLst;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, DB;

type
  TGroupOperation = (goSum,goCount);

  TDBSum = class(TCollectionItem)
  protected
   FGroupOperation:TGroupOperation;
   FFieldName:String;
   Value:Currency;
  public
   SumValue:Currency;
   procedure Assign(Source: TPersistent); override;

  published
   property GroupOperation:TGroupOperation read FGroupOperation write FGroupOperation;
   property FieldName:String read FFieldName write FFieldName;
  end;

  TDBSumCollection = class(TCollection)
  protected
   FOwner:TPersistent;
   function GetOwner:TPersistent; override;
   function GetItem(Index: Integer): TDBSum;
   procedure SetItem(Index: Integer; Value: TDBSum);
  public
    property Items[Index: Integer]: TDBSum read GetItem write SetItem; default;
  end;


  TDBSumList = class(TComponent)
  private
    { Private declarations }
  protected
    FSumCollection:TDBSumCollection;
    FDataSet:TDataSet;
    FSumListChanged:TNotifyEvent;

    Filtered:Boolean;
    Changing:Boolean;

    OldAfterEdit :TDataSetNotifyEvent;
    OldAfterInsert :TDataSetNotifyEvent;
    OldAfterOpen :TDataSetNotifyEvent;
    OldAfterPost :TDataSetNotifyEvent;
    OldAfterScroll :TDataSetNotifyEvent;
    OldBeforeDelete :TDataSetNotifyEvent;
    OldAfterClose :TDataSetNotifyEvent;

    procedure DataSetAfterEdit(DataSet: TDataSet);
    procedure DataSetAfterInsert(DataSet: TDataSet);
    procedure DataSetAfterOpen(DataSet: TDataSet);
    procedure DataSetAfterPost(DataSet: TDataSet);
    procedure DataSetAfterScroll(DataSet: TDataSet);
    procedure DataSetBeforeDelete(DataSet: TDataSet);
    procedure DataSetAfterClose(DataSet: TDataSet);

    procedure SetDataSet(Value:TDataSet);
    procedure Loaded; override;
    procedure SetSumCollection(const Value: TDBSumCollection);
    { Protected declarations }
  public

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure RecalcAll;
    { Public declarations }
  published
    property SumCollection:TDBSumCollection read FSumCollection write SetSumCollection;

    property DataSet: TDataSet read FDataSet write SetDataSet;
    property SumListChanged: TNotifyEvent read FSumListChanged write FSumListChanged;
    { Published declarations }
  end;

procedure Register;

implementation

//
//  TDBSumList
//

constructor TDBSumList.Create(AOwner: TComponent);
begin
  inherited;
  FSumCollection := TDBSumCollection.Create(TDBSum);
  FSumCollection.FOwner := Self;
end;

destructor TDBSumList.Destroy;
begin
  inherited;
  FSumCollection.Free;
end;

procedure TDBSumList.SetDataSet(Value:TDataSet);
begin
  if (csDesigning in ComponentState) or (csLoading in ComponentState) then
    FDataSet := Value
  else
    Raise Exception.Create(' Assigning DataSet at runtime not available ');
end;

procedure TDBSumList.Loaded;
begin
  inherited;
  if not (csDesigning in ComponentState) and Assigned(FDataSet) then begin
    OldAfterEdit := FDataSet.AfterEdit;
    OldAfterInsert := FDataSet.AfterInsert;
    OldAfterOpen := FDataSet.AfterOpen;
    OldAfterPost := FDataSet.AfterPost;
    OldAfterScroll := FDataSet.AfterScroll;
    OldBeforeDelete := FDataSet.BeforeDelete;
    OldAfterClose := FDataSet.AfterClose;

    FDataSet.AfterEdit := DataSetAfterEdit;
    FDataSet.AfterInsert := DataSetAfterInsert;
    FDataSet.AfterOpen := DataSetAfterOpen;
    FDataSet.AfterPost := DataSetAfterPost;
    FDataSet.AfterScroll := DataSetAfterScroll;
    FDataSet.BeforeDelete := DataSetBeforeDelete;
    FDataSet.AfterClose := DataSetAfterClose;

    if (FDataSet.Active = True) then RecalcAll;
    if Assigned(SumListChanged) then SumListChanged(Self);

  end;
end;

procedure TDBSumList.RecalcAll;
var i: Integer;
    item:TDBSum;
begin

  FDataSet.DisableControls;
  for i := 0 to FSumCollection.Count - 1 do
    TDBSum(FSumCollection.Items[i]).SumValue := 0;

  Changing := True;

  FDataSet.First;
  while FDataSet.Eof = False do begin
    for i := 0 to FSumCollection.Count - 1 do begin
      item := TDBSum(FSumCollection.Items[i]);
      if (item.GroupOperation = goCount) or (item.FieldName <> '') then begin
        case Item.GroupOperation of
          goSum:
          begin
            if (FDataSet.FieldByName(Item.FieldName).IsNull = False) then
              Item.SumValue := Item.SumValue + FDataSet.FieldByName(Item.FieldName).AsFloat;
          end;
          goCount: Item.SumValue := Item.SumValue + 1;
        end;
      end;
    end;
    FDataSet.Next;
  end;
  FDataSet.First;

  FDataSet.EnableControls;
//  Form1.Edit1.Text := FormatFloat('#,##0.0',cur);
//  SumValue := Cur;
  Filtered := FDataSet.Filtered;
  Changing := False;

end;

procedure TDBSumList.DataSetAfterEdit(DataSet: TDataSet);
var i: Integer;
    item:TDBSum;
begin
  if (Assigned(OldAfterEdit)) then
   OldAfterEdit(DataSet);

  for i := 0 to FSumCollection.Count - 1 do begin
    item := TDBSum(FSumCollection.Items[i]);
    if (item.GroupOperation = goCount) or (item.FieldName <> '') then begin
      case Item.GroupOperation of
        goSum:
          if (FDataSet.FieldByName(Item.FieldName).IsNull = False) then
            Item.Value := FDataSet.FieldByName(Item.FieldName).AsFloat
          else
            Item.Value := 0;
        goCount: Item.Value := 0;
      end;
    end;
  end;

end;

procedure TDBSumList.DataSetAfterInsert(DataSet: TDataSet);
var i: Integer;
    item:TDBSum;
begin
  if (Assigned(OldAfterInsert)) then
   OldAfterInsert(DataSet);

  for i := 0 to FSumCollection.Count - 1 do begin
    item := TDBSum(FSumCollection.Items[i]);
    if (item.GroupOperation = goCount) or (item.FieldName <> '') then begin
      case Item.GroupOperation of
        goSum: Item.Value := 0;
        goCount: Item.Value := 1;
      end;
    end;
  end;

end;

procedure TDBSumList.DataSetAfterOpen(DataSet: TDataSet);
begin
  if (Assigned(OldAfterOpen)) then
   OldAfterOpen(DataSet);

  RecalcAll;
  if Assigned(SumListChanged) then SumListChanged(Self);

end;

procedure TDBSumList.DataSetAfterPost(DataSet: TDataSet);
var i: Integer;
    item:TDBSum;
begin
  if (Assigned(OldAfterPost)) then
   OldAfterPost(DataSet);

  for i := 0 to FSumCollection.Count - 1 do begin
    item := TDBSum(FSumCollection.Items[i]);
    if (item.GroupOperation = goCount) or (item.FieldName <> '') then begin
      case Item.GroupOperation of
        goSum:
          if (FDataSet.FieldByName(Item.FieldName).IsNull = False) then
            Item.SumValue := Item.SumValue - Item.Value + FDataSet.FieldByName(Item.FieldName).AsFloat
          else
            Item.SumValue := Item.SumValue - Item.Value;
        goCount:
          Item.SumValue := Item.SumValue + Item.Value;
      end;
    end;
  end;

  if Assigned(SumListChanged) then SumListChanged(Self);

end;

procedure TDBSumList.DataSetAfterScroll(DataSet: TDataSet);
begin
  if (Assigned(OldAfterScroll)) then
   OldAfterScroll(DataSet);

  if (Filtered <> DataSet.Filtered) and (Changing = False) then  begin
    RecalcAll;
    if Assigned(SumListChanged) then SumListChanged(Self);
  end;
end;

procedure TDBSumList.DataSetBeforeDelete(DataSet: TDataSet);
var i: Integer;
    item:TDBSum;
begin
  if (Assigned(OldBeforeDelete)) then
   OldBeforeDelete(DataSet);

  for i := 0 to FSumCollection.Count - 1 do begin
    item := TDBSum(FSumCollection.Items[i]);
    if (item.GroupOperation = goCount) or (item.FieldName <> '') then begin
      case Item.GroupOperation of
        goSum: Item.SumValue := Item.SumValue - FDataSet.FieldByName(Item.FieldName).AsFloat;
        goCount: Item.SumValue := Item.SumValue - 1;
      end;
    end;
  end;

  if Assigned(SumListChanged) then SumListChanged(Self);

end;

procedure TDBSumList.DataSetAfterClose(DataSet: TDataSet);
var i: Integer;
    item:TDBSum;
begin
  if (Assigned(OldAfterClose)) then
   OldAfterClose(DataSet);

  for i := 0 to FSumCollection.Count - 1 do begin
    item := TDBSum(FSumCollection.Items[i]);
    item.SumValue := 0;
    item.Value := 0;
  end;

  if Assigned(SumListChanged) then SumListChanged(Self);

  Changing := False;
end;

procedure TDBSumList.SetSumCollection(const Value: TDBSumCollection);
begin
   FSumCollection.Assign(Value);
end;


//
//  TDBSum
//

procedure TDBSum.Assign(Source: TPersistent);
begin
  if Source is TCheckConstraint then
  begin
   GroupOperation := TDBSum(Source).GroupOperation;
   FieldName := TDBSum(Source).FieldName;
   Value := TDBSum(Source).Value;
   SumValue := TDBSum(Source).SumValue;
  end;
  inherited Assign(Source);
end;

//
//  TDBSumCollection
//

function TDBSumCollection.GetOwner:TPersistent;
begin
  Result := FOwner;
end;

function TDBSumCollection.GetItem(Index: Integer): TDBSum;
begin
 Result := TDBSum(inherited GetItem(Index));
end;

procedure TDBSumCollection.SetItem(Index: Integer; Value: TDBSum);
begin
 inherited SetItem(Index, Value);
end;



procedure Register;
begin
  RegisterComponents('Data Controls', [TDBSumList]);
end;

end.

