unit dbf;

{
Very simple DBF table.
Allow the use of database aware components without the BDE.
It allow you to use DBF tables in your small delphi project with no need of any
big dll.

This component is based on

	Example 1 "Read Only Custom Dataset" in Delphi 3
	by Tony BenBrahim in the The Delphi Treasure Chest...
	Wich seems not to be maintained any more.
	I didn't find any example 2.

And also from
	bindata.pas by Steven R. Kamradt
	wich I found after but was much more advanced

And also from
	XBase file format description
	by Eric BachMann
}

interface

uses
	SysUtils, Classes, Db;

type
	EBinaryDataSetError = class (Exception);

	PRecInfo = ^TRecInfo;
	TRecInfo = record
		Bookmark: Longint;
		BookmarkFlag: TBookmarkFlag;
	end;

	rFileHdr = record
		VerDBF      : byte;	     // 0
		Year        : byte;	     // 1
		Month       : byte;	     // 2
		Day         : byte;      // 3
		RecCount : longword;  // 4-7
		HdrSize: word;      // 8-9
		RecordSize  : word;      // 10-11
	end;

	rAfterHdrV3 = record
		Dummy   : array[12..31] of byte;
	end;

	rAfterHdrV4 = record
		Dummy   : array[12..67] of byte;
	end;

	rFieldHdrV3 = record
		FieldName   : array[0..10] of char;
		FieldType   : char; // 11
		Dummy				: array[12..15] of byte;
		FieldSize   : byte; // 16
		FieldPrecision	: byte; //17
		dummy2			: array[18..31] of byte;
	end;

	rFieldHdrV4 = record
		FieldName   : array[0..10] of char;
		Dummy0				: array[11..31] of byte;
		FieldType   : char; // 32
		FieldSize   : byte; // 33
		FieldPrecision	: byte; //34
		dummy2			: array[35..47] of byte;
	end;

	rMultipleIndexHdr = record
		MdxHdr   : byte;	     // 0
		Year        : byte;	     // 1
		Month       : byte;	     // 2
		Day         : byte;      // 3
		FileName    : array[0..15] of char; // 4..19 of byte
		BlockSize		: word; // 20 21
		BlockAdder  : word; // 22 23
		IndexFlag   : byte; // 24
		NoTag       : byte; // 25
		TagSize     : byte; // 26
		Dummy1      : byte; // 27
		TagUsed     : word; // 28..29
		Dummy2      : word; // 30..31
		NbPage      : longword; // 32..35
		FreePage    : longword; // 36..39
		BlockFree   : longword; // 40..43
		UpdYear     : byte; // 44
		UpdMonth    : byte; // 45
		UpdDay      : byte; // 46
	end;

	rMdxTag = record
		pageno			: longword; // 0..3
		tagname			: array [0..11] of char; // 4..14
		keyformat		: byte; // 15
		forwardTag1	: char; // 16
		forwardTag2 : byte; // 17
		backwardTag : byte; // 18
		dummy   		: byte; // 19
		keytype 		: byte; // 20
	end;
	PMdxTag = ^rMdxTag;

	rMdxTagHdr = record
		RootPage				: longint;// 0..3
		FilePages				: longint;// 4..7
		KeyFormat				: byte;   // 8
		KeyType					: char;   // 9
		dummy		      	: word;   // 10..11
		IndexKeyLength 	: word; 	// 12..13
		MaxNbKeys 			: word; 	// 14..15
		SecondKeyType 	: word; 	// 16..17
		IndexKeyItemLen	: word; 	// 18..19
		dummy2				 	: array [20..22] of byte;
		UniqueFlag    	: byte; 	// 23
	end;
	PMdxTagHdr = ^rMdxTagHdr;

	PDataFieldHdrV3 = ^rFieldHdrV3;
	PDataFieldHdrV4 = ^rFieldHdrV4;

	pRecordHdr = ^tRecordHdr;
	tRecordHdr = record
		DeletedFlag : char;
	end;



	rMemoHdr = record
		NextBlock:Longint;
		Dummy : array [4..7] of byte;
		DBFFile : array [0..7] of Byte; //8..15
		Version : Byte; //16
		Dummy2 : array [17..19] of byte;
		BlockLen:	Word;
	end;

	TDbf = class(TDataSet)
	public
		_TempIndex:TList;
	protected
		_MyFieldInfos: TList;
		_Indexes:TList;
		_SizeOfFileHdr:integer;
		_SizeOfFieldHdr:integer;

		_FieldCount : Integer;
		//FirstDeleted : Integer;
		//LastDeleted : Integer;
		//DeletedCount : Integer;
		StartData : Integer;
		//DataSize: integer;
		_dataStream: TStream; // the physical table
		_indexStream: TStream; // the physical table
		_memoStream: TStream; // the physical table
		FTableName: string; // table path and file name
		_IndexHdr : rMultipleIndexHdr;

		// record data
		FRecordBufferSize : integer;  // data + housekeeping (TRecInfo)
		FRecordInfoOffset : integer;  // offset of RecInfo in record buffer
		FCurrentRecord : integer; // current record (0 to _DataHdr.RecCount - 1)
		// status
		FIsTableOpen: Boolean;
		// field offsets in record
		//Enhancements
		fReadOnly : Boolean;
		_FilterBuffer:pchar;
	protected
		// TDataSet virtual abstract method
		procedure _MyFieldInfosClear;
		function AllocRecordBuffer: PChar; override;
		procedure FreeRecordBuffer(var Buffer: PChar); override;
		procedure GetBookmarkData(Buffer: PChar;
			Data: Pointer); override;
		function GetBookmarkFlag(Buffer: PChar):
			TBookmarkFlag; override;
		function GetFieldData(Field: TField;
			Buffer: Pointer): Boolean; override;
		function GetRecord(Buffer: PChar; GetMode: TGetMode;
			DoCheck: Boolean): TGetResult; override;
		function GetRecordSize: Word; override;
		procedure InternalAddRecord(Buffer: Pointer;
			Append: Boolean); override;
		procedure InternalClose; override;
		procedure InternalDelete; override;
		procedure InternalFirst; override;
		procedure InternalGotoBookmark(
			Bookmark: Pointer); override;
		procedure InternalHandleException; override;
		procedure InternalInitFieldDefs; override;
		procedure InternalInitRecord(Buffer: PChar); override;
		procedure InternalLast; override;
		procedure InternalOpen; override;
		procedure InternalPost; override;
		procedure InternalSetToRecord(Buffer: PChar); override;
		function IsCursorOpen: Boolean; override;
		procedure SetBookmarkFlag(Buffer: PChar;
			Value: TBookmarkFlag); override;
		procedure SetBookmarkData(Buffer: PChar;
			Data: Pointer); override;
		procedure SetFieldData(Field: TField;
			Buffer: Pointer); override;

		// TDataSet virtual method (optional)
		function GetRecordCount: Integer; override;
		procedure SetRecNo(Value: Integer); override;
		function GetRecNo: Integer; override;
		function GetCanModify: Boolean; override;
		Procedure WriteHdr;

		procedure CloseBlob(Field: TField); override;
		function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;

	private
		Procedure _ReadRecord(Buffer:PChar;IntRecNum:Integer);
		Procedure _WriteRecord(Buffer:PChar;IntRecNum:Integer);
		Procedure _AppendRecord(Buffer:PChar);
		Function  _CompareRecords(SortFields:Array of String;Rec1,Rec2:Integer):Integer;
		procedure _CreateFieldDefsFromFile;
		procedure _OpenFiles(CanCreate:boolean);
		procedure _CloseFiles;
		function  _FilterRecord(Buffer: PChar): Boolean;
		procedure _SwapRecords(Rec1,REc2:Integer);
	public
		_DataHdr : rFileHdr;
		_MemoHdr : rMemoHdr;
		constructor Create(AOwner:tComponent); override;
		destructor Destroy; override;
		procedure CreateTable;
		Procedure PackTable;
//		Procedure SortTable(SortFields : Array of String);
		Procedure FSortTable(Column:integer);
		Procedure Test(Column:integer);
		Procedure UnsortTable;
		function GetDBFVersion:integer;
	published
		property FileName: string	read FTableName write FTableName;
		property ReadOnly : Boolean read fReadOnly write fReadonly default False;
		property DBFVersion : integer read GetDBFVersion;
		// redeclared data set properties
    property Active;
    property Filtered;
		property BeforeOpen;
    property AfterOpen;
    property BeforeClose;
		property AfterClose;
		property BeforeInsert;
		property AfterInsert;
    property BeforeEdit;
		property AfterEdit;
		property BeforePost;
		property AfterPost;
		property BeforeCancel;
    property AfterCancel;
    property BeforeDelete;
		property AfterDelete;
		property BeforeScroll;
		property AfterScroll;
		property OnCalcFields;
		property OnDeleteError;
		property OnEditError;
		property OnFilterRecord;
		property OnNewRecord;
		property OnPostError;
	end;

	TDbfIndex = class
	protected
		MdxTag : rMdxTag;
		MdxTagHdr : rMdxTagHdr;
		_dbf: TDbf;
		_buffsize: integer;
		_pagesize: integer;
	public
		constructor Create;
		procedure Init(Owner:TDbf);
		destructor Destroy; override;
		procedure Test;
		procedure Lire(Page:integer);
	end;
	
	TMyBlobStream = class(TMemoryStream)
	public
		Mode: TBlobStreamMode;
		Dbf: TDBF;
		Field:TField;
		constructor Create(ModeVal:TBlobStreamMode;DbfVal: TDbf; FieldVal:TField);
		destructor destroy;  override;
	end;

	TMyFieldInfo = class
	public
		Size:Integer;
		Offset:Integer;
		Blob:TMyBlobStream;
		BlobSize:Integer;
		destructor destroy;  override;
	end;


procedure Register;

implementation

uses
	TypInfo, Dialogs, Windows, Forms, Controls, IniFiles;

Const
	dfhVersionNumber = 14;
	BofCrack = -1; // before the first record (crack)

TYPE
	PBufArray = PCHAR;



// ****************************************************************************
// Low Level Routines for accessing an internal record

function TDbf.GetDBFVersion:integer;
begin
	Result:= _DataHdr.VerDBF and 7;
end;

// ____________________________________________________________________________
// TDbf._ReadRecord

Procedure TDbf._ReadRecord(Buffer:PChar;IntRecNum:Integer);
	{-Read a record based on the internal record number (absolute)}
begin
	_dataStream.Position := StartData + (_DataHdr.RecordSize * IntRecNum);
	_dataStream.ReadBuffer (Buffer^, _DataHdr.RecordSize);
end;

// ____________________________________________________________________________
// TDbf._WriteRecord
Procedure TDbf._WriteRecord(Buffer:PChar;IntRecNum:Integer);
	{-Write a record based on the internal record number (absolute)}
begin
	_dataStream.Position := StartData + (_DataHdr.RecordSize * IntRecNum);
	_dataStream.WriteBuffer (Buffer^, _DataHdr.RecordSize);
end;

// ____________________________________________________________________________
// TDbf._AppendRecord
Procedure TDbf._AppendRecord(Buffer:PChar);
begin
	InternalLast;
	pRecordHdr(ActiveBuffer)^.DeletedFlag := ' ';
	Inc (_DataHdr.RecCount);
	_WriteRecord(Buffer, _DataHdr.RecCount-1)
end;

/////////////////////////////////////////////////
////// Part I:
////// Initialization, opening, and closing
/////////////////////////////////////////////////

// ____________________________________________________________________________
// TDbf.InternalOpen
// I: open the table/file
procedure TDbf.InternalOpen;
var
	lExpectedRecordCount : longint;
begin
	_OpenFiles(false);
	// initialize the field definitions
	// (another virtual abstract method of TDataSet)
	InternalInitFieldDefs;

	if (_DataHdr.RecordSize <> _DataHdr.RecordSize) then
			Raise eBinaryDataSetError.Create('Internal open: File record size mismatch.');

	// if there are no persistent field objects,
	// create the fields dynamically
	if DefaultFields then
		CreateFields;
	// connect the TField objects with the actual fields
	BindFields (True);


	//FieldCount:=(_DataHdr.HdrSize - sizeof(rFileHdr) div sizeof(rFieldHdr));
	// sets cracks and record position
	FCurrentRecord := BofCrack;

	FRecordInfoOffset := _DataHdr.RecordSize;
	FRecordBufferSize := _DataHdr.RecordSize + sizeof (TRecInfo);

	// set the bookmark size
	BookmarkSize := sizeOf (Integer);

	// everything OK: table is now open
	FIsTableOpen := True;

// ShowMessage ('InternalOpen: RecCount: ' + IntToStr (_DataHdr.RecCount));
end;

// ____________________________________________________________________________
// TDbf.InternalInitFieldDefs
// I: define the fields
procedure TDbf.InternalInitFieldDefs;
begin
	FieldDefs.Clear;
	with FieldDefs do
	begin
		if _dataStream=nil then begin
			_OpenFiles(false);
			_CreateFieldDefsFromFile;
			_CloseFiles;
		end else begin
			_CreateFieldDefsFromFile;
		end;
	end;
end;


procedure TDbf._CreateFieldDefsFromFile;
var
	Il : Integer;
	lFieldHdrV3 : rFieldHdrV3;
	lFieldHdrV4 : rFieldHdrV4;
	TmpFieldOffset : Integer;
	IX : Integer;
	fn:string;
	ft:TFieldType;
	fs,nfs,fd:Integer;
	MyFieldInfo:TMyFieldInfo;
	function ToFieldType(dbasetype:char;var fs,fd:Integer):TFieldType;
	begin
		case dbasetype of
		'C' :
			begin
				// fs:=fs;
				Result:=ftString;
			end;
		'L' :
			begin
				fs:=0;
				Result:=ftBoolean;
			end;
		'N' :
			begin
				if fd=0 then begin
					if fs <= 4 then begin
						Result:=ftSmallInt;
					end else if fs <= 9 then begin
						Result:=ftInteger;
					end else begin
						Result:=ftLargeInt;
					end;
				end else begin
					Result:=ftFloat;
				end;
				fs:=0;
			end;
		'D' :
			begin
				fs:=0;
				Result:=ftDate;
			end;
		'M' :
			begin
				fs:=0;
				Result:=ftMemo;
			end;
		else
			begin
				// fs:=fs;
				Result:=ftString;
			end;
		end; //case
	end;
begin
	_MyFieldInfosClear;
	_dataStream.Seek(_SizeOfFileHdr,soFromBeginning);
	Assert(_FieldCount>0,'Invalid number of fields.');

	TmpFieldOffset := 1;
	for Il:=0 to _FieldCount-1 do
	begin
		_dataStream.Seek(	_SizeOfFileHdr + Il *_SizeOfFieldHdr,0);
		if (DBFVersion)>=4 then begin
			_dataStream.Read(lFieldHdrV4,SizeOf(lFieldHdrV4));
			fn:=PCHAR(@lFieldHdrV4.FieldName[0]);
			fs:=lFieldHdrV4.FieldSize;
			fd:=lFieldHdrV4.FieldPrecision;
			nfs:=fs;
			ft:=ToFieldType(lFieldHdrV4.FieldType,nfs,fd);
			TFieldDef.Create(FieldDefs, fn,ft,nfs,false,Il+1);

			MyFieldInfo:=TMyFieldInfo.Create;
			MyFieldInfo.Offset:=TmpFieldOffset;
			MyFieldInfo.Size:=fs;
			_MyFieldInfos.Add(MyFieldInfo);
			Inc(tmpFieldOffset,fs);
		end else begin
			_dataStream.Read(lFieldHdrV3,SizeOf(lFieldHdrV3));
			fn:=PCHAR(@lFieldHdrV3.FieldName[0]);
			fs:=lFieldHdrV3.FieldSize;
			fd:=lFieldHdrV3.FieldPrecision;
			nfs:=fs;
			ft:=ToFieldType(lFieldHdrV3.FieldType,nfs,fd);
			TFieldDef.Create(FieldDefs, fn,ft,nfs,false,Il+1);
			MyFieldInfo:=TMyFieldInfo.Create;
			MyFieldInfo.Offset:=TmpFieldOffset;
			MyFieldInfo.Size:=fs;
			_MyFieldInfos.Add(MyFieldInfo);
			Inc(tmpFieldOffset,fs);
		end;
	end;
	_DataHdr.RecordSize := tmpFieldOffset;
	// skip one char (end of Hdr).
	StartData := _dataStream.Position + 1;
end;

procedure TDbf._OpenFiles(CanCreate:boolean);
var
	lDataFile:string;
	lMemoFile:string;
	i:integer;
	Buff:Array[0..31] of byte;
	Day,Month,Year:integer;
	lExpectedRecordCount : longint;
	fileopenmode : integer;
	lIndexName,lMemoName : string;

{
	rMdxTag = record
		pageno			: longword; // 0..3
		tagname			: array [0..11] of byte; // 4..14
		keyformat		: byte; // 15
		forwardTag1	: byte; // 16
		forwardTag2 : byte; // 17
		backwardTag : byte; // 18
		dummy   		: byte; // 19
		keytype 		: byte; // 20
	end;

	rMdxTagHdr = record
		RootPage				: longint;// 0..3
		FilePages				: longint;// 4..7
		KeyFormat				: byte;   // 8
		KeyType					: byte;   // 9
		dummy		      	: word;   // 10..11
		IndexKeyLength 	: word; 	// 12..13
		MaxNbKeys 			: word; 	// 14..15
		SecondKeyType 	: word; 	// 16..17
		IndexKeyItemLen	: word; 	// 18..19
		dummy2				 	: array [20..22] of byte;
		UniqueFlag    	: byte; 	// 23
	end;
}
	BlockNo : integer;
	Idx:TDbfIndex;
	ff:string;
	lAfterHdrV3 : rAfterHdrV3;
	lAfterHdrV4 : rAfterHdrV4;
begin

	if not FileExists (FTableName) then begin
		ff:=ExtractFilePath(Application.Exename)+ExtractFileName(FTableName);
		if FileExists(ff) then FTableName:=ff;
	end;

	// check if the file exists
	if CanCreate then begin
		_dataStream := tFileStream.Create( fTableName, fmCreate);
		FillChar(_DataHdr,sizeof(_DataHdr),0);
		exit;
	end;
	if not FileExists (FTableName) then begin
		raise eBinaryDataSetError.Create ('Open: Table file not found');
	end else begin
		// create a Stream for the file
		if fReadOnly or (csDesigning in ComponentState) then
			fileopenmode := fmOpenRead + fmShareDenyNone
		else
			fileopenmode := fmOpenReadWrite + fmShareDenyWrite;
		_dataStream := tFileStream.Create( fTableName, fileopenmode);
		_dataStream.ReadBuffer(_DataHdr,sizeof(_DataHdr));
	end;

	// get the number of records and check size

	if (DBFVersion) >=4 then begin
		_SizeOfFileHdr := SizeOf(rAfterHdrV4) + SizeOf(rFileHdr);
		_SizeOfFieldHdr := SizeOf(rFieldHdrV4);
		_dataStream.ReadBuffer(lAfterHdrV4,sizeof(rAfterHdrV4));
	end else begin
		_SizeOfFileHdr := SizeOf(rAfterHdrV3) + SizeOf(rFileHdr);
		_SizeOfFieldHdr := SizeOf(rFieldHdrV3);
		_dataStream.ReadBuffer(lAfterHdrV3,sizeof(rAfterHdrV3));
	end;

	if _DataHdr.RecordSize = 0 then lExpectedRecordCount:=0
	else lExpectedRecordCount:= (_dataStream.Size - _DataHdr.HdrSize) div _DataHdr.RecordSize;

	if _DataHdr.RecCount <> lExpectedRecordCount then
		ShowMessage('Invalid Record Count,'+^M+
																		 'RecordCount in Hdr : '+IntToStr(_DataHdr.RecCount)+^M+
																		 'expected : '+IntToStr(lExpectedRecordCount));

	_DataHdr.RecCount := lExpectedRecordCount;

	_FieldCount:= (_DataHdr.HdrSize - _SizeOfFileHdr) div _SizeOfFieldHdr;

	if (_FieldCount <= 0) or (_FieldCount > 255) then
			Raise eBinaryDataSetError.Create('Invalid Field count : ' + IntToStr(_FieldCount) + ' (must be between 1 and 255)');

	lIndexName:=ChangeFileExt(fTableName,'.mdx');
	if FileExists (lIndexName) then begin
		_indexStream := tFileStream.Create( lIndexName, fileopenmode);
		//
// TagSize = M
// BlockAdder = N
		_indexStream.ReadBuffer(_IndexHdr,sizeof(rMultipleIndexHdr));
		for i:= 0 to _IndexHdr.TagUsed-1 do begin
			Idx:=TDbfIndex.Create;
			_indexStream.Seek(544 + i * _IndexHdr.TagSize ,0);
			_indexStream.ReadBuffer(Idx.MdxTag,sizeof(rMdxTag));
			_Indexes.Add(Idx);

			BlockNo := Idx.MdxTag.pageno * (_IndexHdr.BlockAdder div _IndexHdr.BlockSize);
			_indexStream.Seek(BlockNo  ,0);
			_indexStream.ReadBuffer(Idx.MdxTagHdr,sizeof(rMdxTagHdr));
			Idx.Init(self);
		end;
	end;

	lMemoName:=ChangeFileExt(fTableName,'.dbt');
	if FileExists (lMemoName) then begin
		_memoStream := tFileStream.Create( lMemoName, fileopenmode);
		_memoStream.Seek(0,0);
		_memoStream.read(_MemoHdr,SizeOf(rMemoHdr));
		if _MemoHdr.Version=0 then _MemoHdr.BlockLen := $200;
	end;
(*	i:=0;
	try
		lDataFile:=_GetFileName(FileName);
		if lDataFile<>'' then begin
			_DataStream := TFileStream.Create(lDataFile, fmOpenReadWrite	+ fmShareDenyNone);
		end;
	except
	end;
	if _DataStream<>nil then begin
		_DataStream.Seek(0,0);
		_DataStream.Read(Buff,32);
		Year:=Buff[1];
		Month:=Buff[2];
		Day:=Buff[3];
		_RecordCount:=(Buff[7] shl 24)+(Buff[6] shl 16)+(Buff[5] shl 8)+Buff[4];
		_HdrSize:=(Buff[9] shl 8)+Buff[8];
		_FileRecordSize:=(Buff[11] shl 8)+Buff[10];
		lExpectedRecordCount:= (_DataStream.Size - _HdrSize) div _FileRecordSize;
		_NbField:=(_HdrSize div 32)-1;
		if (_RecordCount = lExpectedRecordCount) then begin
			lMemoFile:= _GetFileName(ChangeFileExt(FileName,'.dbt'));
			if lMemoFile<>''  then begin
				try
					_MemoStream := TFileStream.Create(lMemoFile, fmOpenReadWrite	+ fmShareDenyNone);
				except
				end;
			end;
		end;
	end;
	if _MemoStream<>nil then begin
		_MemoStream.Seek(0,0);
		_MemoStream.Read(Buff[0],32);
		_MemoBlockLength:=(Buff[21] shl 8)+Buff[20];
	end;
*)
end;

procedure TDbf._CloseFiles;
var
	Buff:Array[0..31] of byte;
	i:integer;
begin
	if _dataStream=nil then exit;

	_dataStream.Free;
	_dataStream:=nil;

	_indexStream.Free;
	_indexStream:=nil;

	_memoStream.Free;
	_memoStream:=nil;

	for i:=0 to _Indexes.Count-1 do begin
		TDBFIndex(_Indexes[i]).Free;
	end;
	_Indexes.Clear;

	// free the internal list field offsets
	_MyFieldInfosClear;
end;
// ____________________________________________________________________________
// TDbf.InternalClose
// I: close the table/file
procedure TDbf.InternalClose;
begin
	// if required, save updated Hdr
//	if (_DataHdr.RecCount <> _DataHdr.RecCount) or
//		(_DataHdr.RecordSize = 0) then
	if fReadOnly or (csDesigning in ComponentState) then
	begin
		// nothing...
	end	else begin
			WriteHdr;
	end;

	// disconnet field objects
	BindFields (False);
	// destroy field object (if not persistent)
	if DefaultFields then
		DestroyFields;

	// close the file
	FIsTableOpen := False;
	_CloseFiles;
end;

// ____________________________________________________________________________
// TDbf.IsCursorOpen
// I: is table open
function TDbf.IsCursorOpen: Boolean;
begin
  Result := FIsTableOpen;
end;

// ____________________________________________________________________________
// TDbf.WriteHdr
procedure TDbf.WriteHdr;
var
	lAfterHdrV3 : rAfterHdrV3;
	lAfterHdrV4 : rAfterHdrV4;
	y,m,d:Word;
begin
	Assert(_dataStream<>nil,'_dataStream=Nil');
	_dataStream.Seek(0,soFromBeginning);

	DecodeDate(Now,y,m,d);
	if (y<2000) or (y>2100) then begin
		y:=2000;
		m:=1;
		d:=1;
	end else begin
		y:=y-1900;
	end;
	_DataHdr.Year := y;
	_DataHdr.Month := m;
	_DataHdr.Day := d;
	_dataStream.WriteBuffer(_DataHdr,sizeof(rFileHdr));

	if (DBFVersion) >=4 then begin
		FillChar(lAfterHdrV4,sizeof(lAfterHdrV4),0);
		_dataStream.WriteBuffer(lAfterHdrV4,sizeof(rAfterHdrV4));
	end else begin
		FillChar(lAfterHdrV3,sizeof(lAfterHdrV3),0);
		_dataStream.WriteBuffer(lAfterHdrV3,sizeof(rAfterHdrV3));
	end;
end;

// ____________________________________________________________________________
// TDbf.Create
constructor TDbf.Create(AOwner:tComponent);
begin
	inherited create(aOwner);
	_MyFieldInfos:=TList.Create;
	_TempIndex:=TList.Create;
	_Indexes:=TList.Create;
end;

destructor TDbf.Destroy;
begin
	inherited Destroy;
	_CloseFiles; // just in case, it should be already done in 'inherited destroy'
	_MyFieldInfosClear;
	_MyFieldInfos.free;
	_TempIndex.free;
	_Indexes.free;
end;

// ____________________________________________________________________________
// TDbf.CreateTable
// I: Create a new table/file
procedure TDbf.CreateTable;
var
	Ix : Integer;
	lFieldHdrV3 : rFieldHdrV3;
	lAfterHdrV3 :	rAfterHdrV3;
	Offs : Integer;
	DataSize:integer;
	lterminator:Byte;
begin
	CheckInactive;
	//  InternalInitFieldDefs;
	if FieldDefs.Count = 0 then
	begin
		for Ix := 0 to FieldCount - 1 do
		begin
			with Fields[Ix] do
			begin
				if FieldKind = fkData then
					FieldDefs.Add(FieldName,DataType,Size,Required);
			end;
		end;
	end;
	_DataHdr.VerDBF := 3;
	_OpenFiles(true);

		_SizeOfFileHdr := SizeOf(rAfterHdrV3) + SizeOf(rFileHdr);
		_SizeOfFieldHdr := SizeOf(rFieldHdrV3);

//	_DataHdr.VerDBF := 3;
//	_DataHdr.RecCount := 0;
//	_SizeOfFileHdr := SizeOf(rAfterHdrV3) + SizeOf(rFileHdr);
//	_DataHdr.RecordSize := 0;

	try
		FillChar(_DataHdr,SizeOf(_DataHdr),0);
		_DataHdr.VerDBF := 3;
		_dataStream.Write(_DataHdr,SizeOf(_DataHdr));
		_dataStream.Write(lAfterHdrV3,SizeOf(lAfterHdrV3));

		for Ix:=0 to FieldDefs.Count-1 do
		begin
			with FieldDefs.Items[Ix] do
			begin
					FillChar(lFieldHdrV3,SizeOf(lFieldHdrV3),#0);
					lFieldHdrV3.FieldType:='C'; //DataType;
					StrCopy(lFieldHdrV3.FieldName,PChar(FieldDefs.Items[Ix].Name));
					lFieldHdrV3.FieldSize:=Size;
					lFieldHdrV3.FieldPrecision:=Precision;
				 case DataType of
					 ftString:
						 DataSize := Size;
					ftBoolean, ftSmallInt, ftWord:
						 DataSize := 2;
					 ftInteger, ftDate, ftTime:
						DataSize := 4;
					 ftFloat, ftCurrency, ftDateTime:
						 //DataSize:= Fields[Ix].DataSize; //8;
						DataSize := 8;
				else
					begin
						DataSize := 0;
						raise EBinaryDataSetError.Create (
						 'InitFieldsDefs: Unsupported field type');
					end;
				end;
				Inc(Offs,DataSize);
				//INC(FieldCount);
				_dataStream.Write(lFieldHdrV3,SizeOf(lFieldHdrV3));
			end;
		end;
		// end of header
		lterminator := $0d;
		_dataStream.Write(lterminator,1);

		_DataHdr.RecordSize := Offs;
		_DataHdr.HdrSize := _dataStream.Position;
		// end of file
		lterminator := $1a;
		_dataStream.Write(lterminator,1);
		// rewrite the updated header
		WriteHdr;
	finally
		// close the file
		_CloseFiles;
	end;
end;

// ____________________________________________________________________________
// TDbf.PackTable
//Enhancement: Remove all deleted items from the table.
Procedure TDbf.PackTable;
var
	NewStream, OldStream : tStream;
	PC : PChar;
	Ix : Integer;
	lFieldHdrV3 : rFieldHdrV3;
	lFieldHdrV4 : rFieldHdrV4;

	NewFileHdr : rFileHdr;
	NewAfterHdrV3 : rAfterHdrV3;
	NewAfterHdrV4 : rAfterHdrV4;
	DataBuffer : Pointer;
begin
	OldStream := Nil;
	NewStream := Nil;
	CheckInactive;

	if fTableName = '' then
		raise EBinaryDataSetError.Create('Table name not specified.');
  if not FileExists (FTableName) then
    raise EBinaryDataSetError.Create('Table '+fTableName+' does not exist.');
  PC := @fTablename[1];
  CopyFile(PChar(PC),PChar(PC+',old'+#0),False);
  // create the new file
  if FieldDefs.Count = 0 then
  begin
    for Ix := 0 to _FieldCount - 1 do
    begin
			with Fields[Ix] do
			begin
        if FieldKind = fkData then
					FieldDefs.Add(FieldName,DataType,Size,Required);
      end;
    end;
  end;
  try
    NewStream := TFileStream.Create (FTableName+',new',
      fmCreate or fmShareExclusive);
    OldStream := tFileStream.Create (fTableName+',old',
			fmOpenRead or fmShareExclusive);
		OldStream.ReadBuffer(NewFileHdr,SizeOf(rFileHdr));
		if (NewFileHdr.VerDBF and 7)<=3 then begin
			OldStream.ReadBuffer(NewAfterHdrV3,SizeOf(rAfterHdrV3));
		end else begin
			OldStream.ReadBuffer(NewAfterHdrV4,SizeOf(rAfterHdrV4));
		end;
		NewStream.WriteBuffer(NewFileHdr,SizeOf(rFileHdr));
		if (NewFileHdr.VerDBF and 7)<=3 then begin
			NewStream.WriteBuffer(NewAfterHdrV3,SizeOf(rAfterHdrV3));
		end else begin
			NewStream.WriteBuffer(NewAfterHdrV4,SizeOf(rAfterHdrV4));
		end;

		for IX := 0 to _FieldCount-1 do
			begin
				if (DBFVersion)>=4 then begin
					OldStream.Read(lFieldHdrV4,SizeOf(lFieldHdrV4));
					NewStream.Write(lFieldHdrV4,SizeOf(lFieldHdrV4));
				end else begin
					OldStream.Read(lFieldHdrV3,SizeOf(lFieldHdrV3));
					NewStream.Write(lFieldHdrV3,SizeOf(lFieldHdrV3));
				end;
      end;
		GetMem(DataBuffer,NewFileHdr.RecordSize);
		REPEAT
      IX := OldStream.Read(DataBuffer^,NewFileHdr.RecordSize);
			if (IX = NewFileHdr.REcordSize) and (pRecordHdr(DataBuffer)^.DeletedFlag = ' ') then
				NewStream.WRite(DataBuffer^,NewFileHdr.RecordSize);
    Until IX <> NewFileHdr.RecordSize;
    FreeMem(DataBuffer,NewFileHdr.RecordSize);
	finally
		// close the file
		NewStream.Free;
		OldStream.Free;
  end;
  CopyFile(PChar(PC+',new'+#0),PChar(PC),False);
  DeleteFile(Pchar(PC+',new'+#0));
  DeleteFile(Pchar(PC+',old'+#0));
end;

// ____________________________________________________________________________

// ____________________________________________________________________________
// TDbf._CompareRecords
// Compare two records.  Returns -1 if REC1 < REC2, 0 if REC1 = REC2, or
// 1 if REC1 > REC2.
Function TDbf._CompareRecords(SortFields:Array of String;Rec1,Rec2:Integer):Integer; FAR;
{-Compare the records Rec1, Rec2 and return -1 if Rec1 < Rec2, 0 if Rec1 = Rec2,
	1 if Rec1 > Rec2 }
VAR
	IX : Integer;

	Function CompareHelper(KeyId:String;Rec1,Rec2:Integer):Integer;
	VAR
		SKey1, SKey2 : String;
		IKey1, IKey2 : Integer;
		fKey1, fKey2 : Double;
		dKey1, dKey2 : tDateTime;
		CompareType : tFieldType;
		KeyField : tField;
	begin
		KeyField := FieldByName(KeyID);
		CompareType := KeyField.DataType;
		Case CompareType of
			ftFloat,
			ftCurrency,
			ftBCD :
				begin
					_ReadRecord(ActiveBuffer,Rec1-1);
					fKey1 := KeyField.AsFloat;
					_ReadRecord(ActiveBuffer,Rec2-1);
					fKey2 := KeyField.AsFloat;
					if fKey1 < fKey2 then
						Result := -1
					else
						if fKey1 > fKey2 then
							Result := 1
						else
							Result := 0;
				end;
			ftSmallInt,
			ftInteger,
			ftWord :
				begin
					_ReadRecord(ActiveBuffer,Rec1-1);
					IKey1 := KeyField.AsInteger;
					_ReadRecord(ActiveBuffer,Rec2-1);
					IKey2 := KeyField.AsInteger;
					if IKey1 < IKey2 then
						Result := -1
					else
						if IKey1 > IKey2 then
							Result := 1
						else
							Result := 0;
				end;
			ftDate,
			ftTime,
			ftDateTime :
				begin
					_ReadRecord(ActiveBuffer,Rec1-1);
					dKey1 := KeyField.AsDateTime;
					_ReadRecord(ActiveBuffer,Rec2-1);
					dKey2 := KeyField.AsDateTime;
					if dKey1 < dKey2 then
						Result := -1
					else
						if dKey1 > dKey2 then
							Result := 1
						else
							Result := 0;
				end;
			else
				begin
					_ReadRecord(ActiveBuffer,Rec1-1);
					SKey1 := KeyField.AsString;
					_ReadRecord(ActiveBuffer,Rec2-1);
					SKey2 := KeyField.AsString;
					if SKey1 < SKey2 then
						Result := -1
					else
						if SKey1 > SKey2 then
							Result := 1
						else
							Result := 0;
				end;
		end;
	end;

begin
	IX := 0;
	REPEAT // Loop through all available sortfields until not equal or no more sort fiels.
		Result := CompareHelper(SortFields[IX],Rec1,Rec2);
		Inc(IX);
	UNTIL (Result <> 0) or (IX > High(SortFields));
end;


Procedure TDbf._SwapRecords(Rec1,REc2:Integer);
VAR
	Buffer1, Buffer2 : PChar;
	Bookmark1, BOokmark2 : TBookmarkFlag;
begin
	Rec1 := Rec1 - 1;
	Rec2 := Rec2 - 1;
	if Rec1 < 0 then Exit;
	if Rec2 < 0 then Exit;
	Buffer1 := AllocRecordBuffer;
	Buffer2 := AllocRecordBuffer;
	_ReadRecord(Buffer1,Rec1);
	_ReadRecord(Buffer2,Rec2);
	Bookmark1 := GetBookmarkFlag(Buffer1);
	Bookmark2 := GetBookmarkFlag(Buffer2);
	SetBookmarkFlag(Buffer1,Bookmark2);
	SetBookmarkFlag(Buffer2,Bookmark1);
	_WriteRecord(Buffer1,Rec2);
	_WriteRecord(Buffer2,Rec1);
	StrDispose(Buffer1);
	StrDispose(Buffer2);
end;

Procedure TDbf.Test(Column:integer);
begin
	if Column<_Indexes.Count then TDbfIndex(_Indexes[Column]).Test;
end;

Procedure TDbf.FSortTable(Column:integer);
	{ This is a test of a new crap sorting routine.
		works only in ram.
		No real swap.
	}
var
	sl:TStringList;
	i:integer;
	t:longint;
begin
	CheckActive;
	if fReadOnly or (csDesigning in ComponentState) then
		raise eBinaryDataSetError.Create ('Dataset must be opened for read/write to perform sort.');

	t:=getTickCount;
	try
		sl:=TStringList.Create;
		sl.Capacity:=_DataHdr.RecCount;
		for i:=0 to _DataHdr.RecCount-1 do begin
			_ReadRecord(ActiveBuffer,i);
			sl.AddObject(Fields[Column].AsString,Pointer(i));
		end;
		sl.sorted:=true;

		_TempIndex.Clear;
		_TempIndex.Capacity:=sl.count;
		for i:=0 to sl.Count-1 do begin
			_TempIndex.Add(sl.Objects[i]);
		end;

	finally
		sl.free;
	end;
//	ShowMessage(intToStr(t-GetTickCount));
	First;
	Refresh;
end;

// ____________________________________________________________________________
// TDbf.UnsortTable
// Used to help test the sort routine.  Attempts to generate a random
// dispersment of the records in the dataset.
Procedure TDbf.UnsortTable;
Var
  IX : Integer;
begin
  First;
  Randomize;
	for IX := 0 to _DataHdr.RecCount-1 do
		begin
			_SwapRecords(IX,Random(_DataHdr.RecCount));
    end;
  First;
end;

////////////////////////////////////////
////// Part II:
////// Bookmarks management and movement
////////////////////////////////////////

// ____________________________________________________________________________
// TDbf.InternalGotoBookmark
// II: set the requested bookmark as current record
procedure TDbf.InternalGotoBookmark (Bookmark: Pointer);
var
  ReqBookmark: Integer;
begin
  ReqBookmark := PInteger (Bookmark)^;
//  ShowMessage ('InternalGotoBookmark: ' +
//    IntToStr (ReqBookmark));
  if (ReqBookmark >= 0) and (ReqBookmark < _DataHdr.RecCount) then
    FCurrentRecord := ReqBookmark
  else
    raise eBinaryDataSetError.Create ('Bookmark ' +
      IntToStr (ReqBookmark) + ' not found');
end;

// ____________________________________________________________________________
// TDbf.InternalSetToRecord
// II: same as above (but passes a buffer)
procedure TDbf.InternalSetToRecord (Buffer: PChar);
var
  ReqBookmark: Integer;
begin
//  ShowMessage ('InternalSetToRecord');
  ReqBookmark := PRecInfo(Buffer + FRecordInfoOffset).Bookmark;
  InternalGotoBookmark (@ReqBookmark);
end;

// ____________________________________________________________________________
// TDbf.GetBookmarkFlag
// II: retrieve bookmarks flags from buffer
function TDbf.GetBookmarkFlag (
  Buffer: PChar): TBookmarkFlag;
begin
//  ShowMessage ('GetBookmarkFlag');
  Result := PRecInfo(Buffer + FRecordInfoOffset).BookmarkFlag;
end;

// ____________________________________________________________________________
// TDbf.SetBookmarkFlag
// II: change the bookmark flags in the buffer
procedure TDbf.SetBookmarkFlag (Buffer: PChar;
	Value: TBookmarkFlag);
begin
//  ShowMessage ('SetBookmarkFlag');
  PRecInfo(Buffer + FRecordInfoOffset).BookmarkFlag := Value;
end;

// ____________________________________________________________________________
// TDbf.GetBookmarkData
// II: read the bookmark data from record buffer
procedure TDbf.GetBookmarkData (
  Buffer: PChar; Data: Pointer);
begin
//  ShowMessage ('GetBookmarkData: ' + IntToStr (PRecInfo(Buffer + FRecordInfoOffset).Bookmark));
  PInteger(Data)^ :=
    PRecInfo(Buffer + FRecordInfoOffset).Bookmark;
end;

// ____________________________________________________________________________
// TDbf.SetBookmarkData
// II: set the bookmark data in the buffer
procedure TDbf.SetBookmarkData (
	Buffer: PChar; Data: Pointer);
begin
//  ShowMessage ('SetBookmarkData: ' + IntToStr (PInteger(Data)^));
	PRecInfo(Buffer + FRecordInfoOffset).Bookmark :=
		PInteger(Data)^;
end;

// ____________________________________________________________________________
// TDbf.InternalFirst
// II: Go to a special position before the first record
procedure TDbf.InternalFirst;
begin
	FCurrentRecord := BofCrack;
end;

// ____________________________________________________________________________
// TDbf.InternalLast
// II: Go to a special position after the last record
procedure TDbf.InternalLast;
begin
	FCurrentRecord := _DataHdr.RecCount; // 1 too much
end;

// ____________________________________________________________________________
// TDbf.GetRecordCount
// II (optional): Record count
function TDbf.GetRecordCount: Longint;
begin
  CheckActive;
  Result := _DataHdr.RecCount;
end;

// ____________________________________________________________________________
// TDbf.GetRecNo
// II (optional): Get the number of the current record
function TDbf.GetRecNo: Longint;
begin
	UpdateCursorPos;
	if FCurrentRecord < 0 then
		Result := 1
	else
		Result := FCurrentRecord + 1;
end;

function TDbf.GetCanModify: Boolean;
begin
	result:=not FReadOnly;
end;
// ____________________________________________________________________________
// TDbf.SetRecNo
// II (optional): Move to the given record number
procedure TDbf.SetRecNo(Value: Integer);
begin
  CheckBrowseMode;
  if (Value > 1) and (Value <= _DataHdr.RecCount) then
  begin
    FCurrentRecord := Value - 1;
    Resync([]);
  end;
end;

//////////////////////////////////////////
////// Part III:
////// Record buffers and field management
//////////////////////////////////////////

// ____________________________________________________________________________
// TDbf.GetRecordSize
/// III: Determine the size of each record buffer in memory
function TDbf.GetRecordSize: Word;
begin
  Result := _DataHdr.RecordSize; // data only
end;

// ____________________________________________________________________________
// TDbf.AllocRecordBuffer
/// III: Allocate a buffer for the record
function TDbf.AllocRecordBuffer: PChar;
begin
	Result := StrAlloc(FRecordBufferSize);
end;

// ____________________________________________________________________________
// TDbf.InternalInitRecord
// III: Initialize the record (set to zero)
procedure TDbf.InternalInitRecord(Buffer: PChar);
begin
	FillChar(Buffer^, FRecordBufferSize, 0);
	Buffer^ := ' ';
end;

// ____________________________________________________________________________
// TDbf.FreeRecordBuffer
// III: Free the buffer
procedure TDbf.FreeRecordBuffer (var Buffer: PChar);
begin
  StrDispose(Buffer);
end;

// ____________________________________________________________________________
// TDbf.GetRecord
// III: Retrieve data for current, previous, or next record
// (eventually moving to it) and return the status
function TDbf.GetRecord(Buffer: PChar;
  GetMode: TGetMode; DoCheck: Boolean): TGetResult;
var
  Acceptable : Boolean;
begin
	result := grOk;
  Acceptable := False;
	if _DataHdr.RecCount < 1 then
    Result := grEOF
	else
  repeat
		begin
			case GetMode of
				gmCurrent :
					begin
						// ShowMessage ('GetRecord Current');
						if (FCurrentRecord >= _DataHdr.RecCount) or
								(FCurrentRecord < 0) then
							Result := grError;
					end;
				gmNext :
					begin
						if (fCurrentRecord < (_DataHdr.RecCount)-1) then
							Inc (FCurrentRecord)
						else
							Result := grEOF;
					end;
				gmPrior :
					begin
						if (fCurrentRecord > 0) then
							Dec(fCurrentRecord)
						else
							Result := grBOF;
					end;
			end;
			// fill record data area of buffer
			if Result = grOK then
			begin
				if fCurrentRecord<_TempIndex.Count then begin
					_ReadRecord(Buffer,Integer(_TempIndex.Items[fCurrentRecord]))
				end else begin
					_ReadRecord(Buffer,fCurrentRecord)
				end;
				ClearCalcFields(Buffer);
				GetCalcFields(Buffer);
				with PRecInfo(Buffer + FRecordInfoOffset)^ do
				begin
					BookmarkFlag := bfCurrent;
					Bookmark := FCurrentRecord;
				end;
			end
			else
				if (Result = grError) and DoCheck then
					raise eBinaryDataSetError.Create ('GetRecord: Invalid record');
			Acceptable := (pRecordHdr(Buffer)^.DeletedFlag = ' ') and _FilterRecord(Buffer);
			if (GetMode=gmCurrent) and Not Acceptable then Result := grError;
		end;
	until (Result <> grOK) or Acceptable;
end;

function TDbf._FilterRecord(Buffer: PChar): Boolean;
var
	SaveState: TDatasetState;
begin
	Result:=True;
	if not Filtered or not Assigned(OnFilterRecord) then Exit;
	_FilterBuffer:=buffer;
	SaveState:=SetTempState(dsFilter);
//	_FilterBuffer:=Buffer;
	OnFilterRecord(self,Result);
	RestoreState(SaveState);
end;

// ____________________________________________________________________________
// TDbf.InternalPost
// III: Write the current data to the file
procedure TDbf.InternalPost;
begin
	CheckActive;
  if State = dsEdit then
  begin
    // replace data with new data
		_WriteRecord(ActiveBuffer, fCurrentRecord);
  end
  else
  begin
		// always append
		_AppendRecord(ActiveBuffer);
	end;
end;

// ____________________________________________________________________________
// TDbf.InternalAddRecord
// III: Add the current data to the file
procedure TDbf.InternalAddRecord(
  Buffer: Pointer; Append: Boolean);
begin
  // always append
	_AppendRecord(ActiveBuffer);
end;

// ____________________________________________________________________________
// TDbf.InternalDelete
// III: Delete the current record
procedure TDbf.InternalDelete;
begin
  CheckActive;
	pRecordHdr(ActiveBuffer)^.DeletedFlag := '*'; //_DataHdr.LastDeleted;
	_WriteRecord(ActiveBuffer,fCurrentRecord);
	Resync([]);
end;

// ____________________________________________________________________________
// TDbf.GetFieldData
// III: Move data from record buffer to field
function TDbf.GetFieldData (
	Field: TField; Buffer: Pointer): Boolean;
var
	FieldOffset: Integer;
	FieldSize: Integer;
	Ptr: Pointer;
	s:string;
	d:TDateTime;
	ld,lm,ly:Integer;
	MyFieldInfo:TMyFieldInfo;
begin
	Result := False;
	if State=dsFilter then begin
		Ptr:=_FilterBuffer;
	end else begin
		if IsEmpty or (Field.FieldNo <= 0) then exit;
		Ptr := ActiveBuffer;
	end;
	MyFieldInfo:=TMyFieldInfo(_MyFieldInfos[Field.FieldNo - 1]);
	FieldOffset := MyFieldInfo.Offset;
	FieldSize := MyFieldInfo.Size;
	if Assigned(Buffer) then begin
		SetString(s,PChar(Ptr) + FieldOffset, FieldSize);
		if Field.DataType = ftBoolean then begin
			if (s[1]='T') then Word(Buffer^) := 1
			else Word(Buffer^) := 0;
		end	else if Field.DataType in [ftInteger,ftSmallInt,ftLargeInt] then begin
			ld:=StrToIntDef(s,0);
			case Field.DataType of
			ftInteger : PInteger(Buffer)^ := ld;
			ftSmallInt : PSmallInt(Buffer)^ := ld;
			ftLargeint : PLargeInteger(Buffer)^ := ld;
			end;
		end	else if Field.DataType = ftFloat then begin
			PDouble(Buffer)^ := StrToFloat(s);
		end	else if Field.DataType = ftDate then begin
			ld:=StrToIntDef(Copy(s,7,2),1);
			lm:=StrToIntDef(Copy(s,5,2),1);
			ly:=StrToIntDef(Copy(s,1,4),0);
			if ld=0 then ld:=1;
			if lm=0 then lm:=1;
			if (ly<1900) or (ly>2100) then ly:=1900;
			try
				d:=EncodeDate(ly,lm,ld);
			except
			end;
			Integer(Buffer^) := DateTimeToTimeStamp(d).Date;
		end	else if Field.DataType = ftString then begin
			s:=trim(s);
			if Length(s)>0 then Move (char(s[1]), Buffer^, Length(s));
			PChar(Buffer)[Length(s)]:=#0;
		end;
	end;
	Result := True;
end;

// ____________________________________________________________________________
// TDbf.SetFieldData
// III: Move data from field to record buffer
procedure TDbf.SetFieldData(Field: TField; Buffer: Pointer);
var
	FieldOffset: Integer;
	FieldSize: Integer;
	Ptr: PCHAR;
	s:string;
	li:Longint;
	fl:Double;
	da:TDateTime;
	ly,lm,ld:Word;
	ts:TTimeStamp;
	MyFieldInfo:TMyFieldInfo;
begin
	if Field.FieldNo >= 0 then
	begin
		MyFieldInfo:=TMyFieldInfo(_MyFieldInfos[Field.FieldNo - 1]);
		FieldOffset := MyFieldInfo.Offset;
		FieldSize := MyFieldInfo.Size;

		if Assigned (Buffer) then begin
			Ptr := PChar(ActiveBuffer + FieldOffset);
			if Field.DataType = ftBoolean then begin
				if Word(Buffer^) = 1 then Char(Ptr^):='T'
				else Char(Ptr^):='F';
			end	else if Field.DataType in [ftInteger,ftSmallInt,ftLargeInt] then begin
				case Field.DataType of
				ftInteger : li := PInteger(Buffer)^;
				ftSmallInt : li := PSmallInt(Buffer)^;
				ftLargeint : li := PLargeInteger(Buffer)^;
				end;
				s:=IntToStr(li);
				Move (PChar(s)^,Ptr^, FieldSize);
			end	else if Field.DataType = ftFloat then begin
				fl := Double(Buffer^);
				s:=FloatToStr(fl);
				Move (PChar(s)^,Ptr^, FieldSize);
			end	else if Field.DataType = ftDate then begin
				ts.Time:=0;
				ts.Date:=Integer(Buffer^);
				da:=TimeStampToDateTime(ts);
				DecodeDate(da,ly,lm,ld);
				FmtStr(s,'%4d%2d%2d',[ly,lm,ld]);
				Move (PChar(s)^,Ptr^, FieldSize);
			end	else if Field.DataType = ftString then begin
				SetString(s,PChar(Buffer), FieldSize);
				s:=trim(s);
				Move (PChar(s)^, Ptr^, FieldSize);
			end;
		end;
		DataEvent (deFieldChange, Longint(Field));
	end;
end;

procedure TDbf._MyFieldInfosClear;
var i:Integer;
begin
	for i:=0 to _MyFieldInfos.Count-1 do begin
		TMyFieldInfo(_MyFieldInfos.Items[i]).Free;
	end;
	_MyFieldInfos.Clear;
end;


// ____________________________________________________________________________
// TDbf.InternalHandleException
// default exception handling
procedure TDbf.InternalHandleException;
begin
	// standard exception handling
	Application.HandleException(Self);
end;

procedure TDbf.CloseBlob(Field: TField);
var
	pBlob:TMyBlobStream;
	MyFieldInfo:TMyFieldInfo;
begin
	if Field.DataType = ftBlob then begin
		MyFieldInfo:=TMyFieldInfo(_MyFieldInfos[Field.FieldNo - 1]);
		pBlob:=MyFieldInfo.Blob;
	end;
end;

function TDbf.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
var
	pBlob:TMyBlobStream;
	MyFieldInfo:TMyFieldInfo;
	FieldOffset :integer;
	FieldSize :integer;
	s:string;
	memoi:Integer;
	Buff:array[0..511] of char;
	lastc:char;
	finish:boolean;
	i:integer;
begin
	MyFieldInfo:=TMyFieldInfo(_MyFieldInfos[Field.FieldNo - 1]);
	MyFieldInfo.Blob:=TMyBlobStream.Create(Mode,Self,Field);
	if (Mode=bmRead) and (_memoStream<>nil) then begin
//		NextBlock:Longint;
		//if IsEmpty or (Field.FieldNo <= 0) then exit;
		MyFieldInfo:=TMyFieldInfo(_MyFieldInfos[Field.FieldNo - 1]);
		FieldOffset := MyFieldInfo.Offset;
		FieldSize := MyFieldInfo.Size;
		SetString(s,PChar(ActiveBuffer) + FieldOffset, FieldSize);
		memoi:=StrToIntDef(s,0);
		lastc:=#0;
		if memoi>0 then begin
			_memoStream.Seek(_MemoHdr.BlockLen * memoi,0);
			finish:=False;
			repeat
				_memoStream.read(Buff,SizeOf(Buff));
				for i:=0 to 510 do begin
					if (Buff[i]=#$1A) and
						((Buff[i+1]=#$1A) or ((i=0) and (lastc=#$1A)))
						then begin
						if i>0 then MyFieldInfo.Blob.Write(buff,i);
						finish:=True;
						break;
					end;
				end;
				if finish then Break;
				MyFieldInfo.Blob.Write(buff,512);
				lastc:=Buff[511];
			until finish;
		end;
		MyFieldInfo.BlobSize:=MyFieldInfo.Blob.Size;
	end;
(*
	rMemoUsed = record
		magic:Longint;
		fieldLength:Longint;
	end;

	rMemoUnused = record
		nextfree:Longint;
		nextused:Longint;
	end;
*)
	MyFieldInfo.Blob.Seek(0,0);
	Result:=MyFieldInfo.Blob;
end;

destructor TMyFieldInfo.destroy;
begin
	inherited;
end;

constructor TMyBlobStream.Create(ModeVal:TBlobStreamMode;DbfVal: TDbf;FieldVal:TField);
begin
	Mode:=ModeVal;
	Dbf:=DbfVal;
	Field:=FieldVal;
end;

destructor TMyBlobStream.destroy;
var
	Buff:array[0..511] of char;
	i:Integer;
	MyFieldInfo:TMyFieldInfo;
	FieldOffset,FieldSize:Integer;
	s:string;
	memoi:Integer;
	c:Byte;
	Append:Boolean;
begin
	if (Mode=bmWrite) then begin
//		NextBlock:Longint;
		if Dbf.IsEmpty or (Field.FieldNo <= 0) then exit;
		MyFieldInfo:=TMyFieldInfo(Dbf._MyFieldInfos[Field.FieldNo - 1]);
		FieldOffset := MyFieldInfo.Offset;
		FieldSize := MyFieldInfo.Size;
		memoi:=Dbf._MemoHdr.NextBlock;

		if ((Size+2) div Dbf._MemoHdr.BlockLen) <= (MyFieldInfo.Size div Dbf._MemoHdr.BlockLen) then begin
			Append:=False;
			SetString(s,PChar(Dbf.ActiveBuffer) + FieldOffset, FieldSize);
			memoi:=StrToIntDef(s,0);
		end else begin
			Append:=True;
			s:=IntToStr(memoi);
			while Length(s)<FieldSize do s:=' ' + s;
			Move (PChar(s)^, PChar(Dbf.ActiveBuffer + FieldOffset)^, FieldSize);
		end;
		Dbf._memoStream.Seek(Dbf._MemoHdr.BlockLen * memoi,0);
		Seek(0,0);
		repeat
			i:=Read(buff,512);
			if i=0 then break;
			Inc(Dbf._MemoHdr.NextBlock);
			Dbf._memoStream.Write(Buff,i);
		until i<512;
		c:=$1A;
		Dbf._memoStream.Write(c,1);
		Dbf._memoStream.Write(c,1);
		if Append then begin
			Dbf._memoStream.Seek(0,0);
			Dbf._memoStream.Write(Dbf._MemoHdr,SizeOf(Dbf._MemoHdr))
  	end;
	end;
	inherited;
end;

// ____________________________________________________________________________
// ____________________________________________________________________________
// ____________________________________________________________________________

constructor TDbfIndex.Create;
begin
end;

destructor TDbfIndex.destroy;
begin
end;

procedure TDbfIndex.Init(Owner:TDbf);
begin

	_dbf:=Owner;
	// seems stupid... not easy to understand some field's purpose.
	_buffsize:=_dbf._IndexHdr.BlockAdder;
	_pagesize:=_buffsize div _dbf._IndexHdr.BlockSize;
end;

procedure TDbfIndex.Test;
begin
	_dbf._TempIndex.clear;
	Lire(MdxTagHdr.RootPage);
	_dbf.Refresh;
end;

procedure TDbfIndex.Lire(Page:integer);
type

	rIdxNode = record
		DataRecordNumber : longword;
		KeyData : array[0..507] of char; // run-time defined real length...
	end;
	pIdxNode = ^rIdxNode;

	rIdxPage = record
		NoEntry : longword;
		LowLevel : longword;
		IdxNodes : rIdxNode; // run-time defined real length...
	end;
	pIdxPage = ^rIdxPage;

var
	i:integer;
	BlockNo:integer;
	IdxPage:pIdxPage;
	IdxNode:pIdxNode;
	_buff:pchar;
begin


	_buff:=StrAlloc(_buffsize+1);
	try
		while Page>0 do begin
			BlockNo := Page * _pagesize;
			_dbf._indexStream.Seek(BlockNo  ,0);
			_dbf._indexStream.ReadBuffer(_buff^,_buffsize);
			IdxPage:=pIdxPage(_buff);
			if IdxPage.NoEntry>0 then begin
				for i:=0 to IdxPage.NoEntry-1 do begin
					//
					IdxNode:=pIdxNode(PIdxNode(Integer(@IdxPage.IdxNodes.DataRecordNumber) + i * (MdxTagHdr.IndexKeyLength+4)));
					//if IdxNode.DataRecordNumber>0 then Lire(IdxNode.DataRecordNumber);
					// _dbf._ReadRecord(_dbf.ActiveBuffer,IdxNode.RecNo);

					_dbf._TempIndex.Add(Pointer(IdxNode.DataRecordNumber - 1));
					//	if IdxNode.<>0 then begin
					//	Lire(IdxNode.NextPg);
				end;
			end; // if
			Page := IdxPage.LowLevel;
		end; // while
	finally
		strdispose(_buff);
	end;
end;



procedure Register;
begin
	RegisterComponents('DBAccess', [TDbf]);
end;

end.

(*
// ____________________________________________________________________________
// TDbf.SortTable
// Enhancement: Sort the table by the fields passed.
Procedure TDbf.SortTable(SortFields : Array of String);

	{ This is the main sorting routine. It is passed the number of elements and the
		two callback routines. The first routine is the function that will perform
		the comparison between two elements. The second routine is the procedure that
		will swap two elements if necessary } // Source: UNDU #8
	// TDbf._SwapRecords
	// Enhancement: Quick swap of two records.  Used primarily for sorting.

	procedure QSort(uNElem: Integer);
	{ uNElem - number of elements to sort }

		procedure qSortHelp(pivotP: Integer; nElem: word);
		label
			TailRecursion,
			qBreak;
		var
			leftP, rightP, pivotEnd, pivotTemp, leftTemp: word;
      lNum: Integer;
      retval: integer;
		begin
			TailRecursion:
        if (nElem <= 2) then
					begin
            if (nElem = 2) then

							begin
                rightP := pivotP +1;
                if (_CompareRecords(SortFields,pivotP, rightP) > 0) then
									_SwapRecords(pivotP, rightP);
							end;
            exit;
					end;
        rightP := (nElem -1) + pivotP;
        leftP :=  (nElem shr 1) + pivotP;
        { sort pivot, left, and right elements for "median of 3" }
				if (_CompareRecords(SortFields,leftP, rightP) > 0) then _SwapRecords(leftP, rightP);
				if (_CompareRecords(SortFields,leftP, pivotP) > 0) then _SwapRecords(leftP, pivotP)

				else if (_CompareRecords(SortFields,pivotP, rightP) > 0) then _SwapRecords(pivotP, rightP);
				if (nElem = 3) then
          begin
						_SwapRecords(pivotP, leftP);
						exit;
          end;
        { now for the classic Horae algorithm }
        pivotEnd := pivotP + 1;
        leftP := pivotEnd;
        repeat
          retval := _CompareRecords(SortFields,leftP, pivotP);
					while (retval <= 0) do
            begin
              if (retval = 0) then
                begin
                  _SwapRecords(LeftP, PivotEnd);
									Inc(PivotEnd);
								end;
              if (leftP < rightP) then
								Inc(leftP)
							else
                goto qBreak;
							retval := _CompareRecords(SortFields,leftP, pivotP);
						end; {while}
          while (leftP < rightP) do
						begin

							retval := _CompareRecords(SortFields,pivotP, rightP);
              if (retval < 0) then
                Dec(rightP)
              else
                begin
                  _SwapRecords(leftP, rightP);
                  if (retval <> 0) then
                    begin
											Inc(leftP);
                      Dec(rightP);
                    end;
                  break;
                end;
            end; {while}

				until (leftP >= rightP);
			qBreak:
				if (_CompareRecords(SortFields,leftP, pivotP) <= 0) then Inc(leftP);
        leftTemp := leftP -1;
        pivotTemp := pivotP;
				while ((pivotTemp < pivotEnd) and (leftTemp >= pivotEnd)) do
					begin
						_SwapRecords(pivotTemp, leftTemp);
            Inc(pivotTemp);
            Dec(leftTemp);
          end; {while}
        lNum := (leftP - pivotEnd);
        nElem := ((nElem + pivotP) -leftP);

        if (nElem < lNum) then
          begin
            qSortHelp(leftP, nElem);
						nElem := lNum;
					end
				else
					begin
						qSortHelp(pivotP, lNum);
						pivotP := leftP;
					end;
				goto TailRecursion;
			end; {qSortHelp }

	begin
		if (uNElem < 2) then  exit; { nothing to sort }
		qSortHelp(1, uNElem);
	end; { QSort }

var
	t:longint;
begin
	t:=getTickCount;
	CheckActive;
	if fReadOnly or (csDesigning in ComponentState) then
		raise eBinaryDataSetError.Create ('Dataset must be opened for read/write to perform sort.');
//  if _DataHdr.DeletedCount > 0 then
//    begin
//      Close;
//      PackTable;
//      Open;
//    end;
	QSort(_DataHdr.RecCount);
	First;
	ShowMessage(intToStr(t-GetTickCount));
end;

Internal buffer layout:
+------------------------+------------------------+---------------------------+
|     RECORD DATA        |     My Information     |     Calculated Fields     |
| SizeOf(TARecord) bytes |  SizeOf(TMyInfo) bytes |    CalcFieldSize bytes    |
+------------------------+------------------------+---------------------------+
											 ^                        ^
									StartMyInfo            StartCalculated

function TDbf._GetFileName(f:string):string;
var
	ff:string;
begin
	if FileExists(f) then result:=f
	else begin
		else result:='';
	end;
end;

*)


