//=============================================================================
// Unit    : CompDataSetExport
//
// Purpose : Save a dataset to a file.
//
//           Methods:
//           --------
//           Execute        - Save the DataSet to a file. Returns True if
//                            successful.
//                            This method can be called at Design time from the
//                            component menu.
//                            String values are quoted as necessary.
//                            The FieldDelimiter Character is taken from the
//                            regional settings (Control Panel).
//
//           Properties:
//           -----------
//           DataSource     - The data to export.
//           FileName       - The file to export to. If left empty, the user
//                            will be asked for one.
//           Caption        - The caption of the progress dialog (if shown).
//           Options
//             eoAutoOpen       : Open the DataSet as necessary.
//             eoShowProgress   : Show a progress dialog.
//             eoCanCancel      : Shows a Cancel button on the progress dialog.
//           Header
//             ehNone           : No header.
//             ehFieldName      : Use the field names as header.
//             ehDisplayName    : Use DisplayName as header.
//
//         Events:
//         -------
//           OnExecute      - Called before execute. Use this for short
//                            operations, such as setting parameters.
//           OnInitialize   - Called before execute, after showing the
//                            progress dialog. Use this for timely operations,
//                            e.g. for opening of the dataset.
//
//-----------------------------------------------------------------------------
// Rev  Date         Author(s)        Remarks
//-----------------------------------------------------------------------------
// 1.0  20-Aug-1998  M.Smits/TRC      Initial Creation
//
//=============================================================================

unit DBExport;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Db, DsgnIntf;

resourcestring
  FILTER_TYPES   = 'Comma delimited file (*.csv)|*.csv|Text file (*.txt)|*.txt|Any file (*.*)|*.*';
  DEFAULT_FILTER = 'csv';

type
  //===========================================================================
  // TDBExportOptions
  //===========================================================================
  TDBExportOption  = (eoAutoOpen, eoCanCancel, eoShowProgress);
  TDBExportOptions = set of TDBExportOption;
  TDBExportHeader  = (ehNone, ehFieldName, ehDisplayName);
const
  DefaultOptions   = [eoAutoOpen, eoShowProgress];
type
  //===========================================================================
  // TDBExport
  //===========================================================================
  TDBExport = class(TComponent)
  private // --------------------------------------------- Private declarations
    FDataSource     : TDataSource;
    FFieldDelimiter : Char;
    FFileName       : TFileName;
    FCaption        : TCaption;
    FHeader         : TDBExportHeader;
    FOptions        : TDBExportOptions;
    FOnExecute      : TNotifyEvent;
    FOnInitialize   : TNotifyEvent;
    FExecuting      : boolean;

  protected // ----------------------------------------- Protected declarations
    procedure SetDataSource (Value : TDataSource);
    procedure Notification (AComponent : TComponent; Operation : TOperation); override;

    function StoredCaption   : boolean;
    function GetFileName     : string;
    function GetHeader       : string;
    function GetRecord       : string;

    function FilteredData (const Data : string) : string;
    function GetSeparator : Char;

  public // ----------------------------------------------- Public declarations
    constructor Create (AOwner : TComponent); override;

    function Execute        : boolean;
    property FieldDelimiter : Char             read FFieldDelimiter;

  published // ----------------------------------------- Published declarations
    property DataSource     : TDataSource      read FDataSource     write SetDataSource;
    property FileName       : TFileName        read FFileName       write FFileName;
    property Caption        : TCaption         read FCaption        write FCaption stored StoredCaption;
    property Header         : TDBExportHeader  read FHeader         write FHeader  default ehNone;
    property Options        : TDBExportOptions read FOptions        write FOptions default DefaultOptions;

    property OnExecute      : TNotifyEvent     read FOnExecute      write FOnExecute;
    property OnInitialize   : TNotifyEvent     read FOnInitialize   write FOnInitialize;
  end;

  //===========================================================================
  // TDBExportEditor
  //===========================================================================
  TDBExportEditor = class (TComponentEditor)
    function GetVerbCount : integer;              override;
    function GetVerb (Index : integer) : string;  override;
    procedure ExecuteVerb (Index : integer);      override;
  end;

  //===========================================================================
  // TDEFileNameProperty
  //===========================================================================
  TDEFileNameProperty = class(TStringProperty)
  public
    function GetAttributes : TPropertyAttributes; override;
    procedure Edit;                               override;
  end;

procedure Register;

implementation

{$R *.DCR}

uses FProgress;

resourcestring
  DEF_CAPTION = 'Export data';

//=============================================================================
// PROTECTED METHODS
//=============================================================================

//-----------------------------------------------------------------------------
// Procedure : SetDataSource
//
// Purpose   : SetDataSource.
//-----------------------------------------------------------------------------
procedure TDBExport.SetDataSource;
begin
  if Value <> FDataSource then begin
    FDataSource := Value;
    if Assigned (FDataSource) then
      FDataSource.FreeNotification (Self);
  end;
end;

//-----------------------------------------------------------------------------
// Function  : GetFileName
//
// Purpose   : GetFileName, display dialog if FileName is empty.
// Output    : FileName.
//-----------------------------------------------------------------------------
function TDBExport.GetFileName;
var
  FSaveDlg  : TSaveDialog;
begin
  FSaveDlg  := TSaveDialog.Create (nil);
  try
    FSaveDlg.DefaultExt := DEFAULT_FILTER;
    FSaveDlg.FileName   := FFileName;
    FSaveDlg.Filter     := FILTER_TYPES;
    if FSaveDlg.Execute then
      Result := FSaveDlg.FileName
    else
      Abort;
  finally
    FSaveDlg.Free;
  end;
end;

//-----------------------------------------------------------------------------
// Procedure : Notification
//
// Purpose   : Notification.
//-----------------------------------------------------------------------------
procedure TDBExport.Notification;
begin
  inherited Notification (AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDataSource) then
    FDataSource := nil;
end;

//-----------------------------------------------------------------------------
// Function  : FilteredData
//
// Purpose   : Quote data if it contains the delimiter.
// Input     : string.
// Output    : string.
//-----------------------------------------------------------------------------
function TDBExport.FilteredData;
begin
  if Assigned (StrScan (PChar (Data), FFieldDelimiter)) then
    Result := AnsiQuotedStr (Data, '"')
  else
    Result := Data;
end;

//-----------------------------------------------------------------------------
// Function  : GetHeader
//
// Purpose   : Get the Header.
// Output    : string.
//-----------------------------------------------------------------------------
function TDBExport.GetHeader;
var
  FieldIdx : integer;
begin
  Result := '';
  with FDataSource.DataSet do
    for FieldIdx := 0 to FieldCount - 1 do
      if Fields[FieldIdx].Visible and not Fields[FieldIdx].IsBlob then
        case FHeader of
          ehFieldName   : Result := Result + Fields[FieldIdx].FieldName   + FFieldDelimiter;
          ehDisplayName : Result := Result + FilteredData (Fields[FieldIdx].DisplayName) + FFieldDelimiter;
        end;
  SetLength (Result, Length (Result) - 1);
end;

//-----------------------------------------------------------------------------
// Function  : GetRecord
//
// Purpose   : Get the current record.
// Output    : string.
//-----------------------------------------------------------------------------
function TDBExport.GetRecord;
var
  FieldIdx : integer;
begin
  Result := '';
  with FDataSource.DataSet do
    for FieldIdx := 0 to FieldCount - 1 do
      if Fields[FieldIdx].Visible and not Fields[FieldIdx].IsBlob then
        if (Fields[FieldIdx].DataType = ftString) then
          Result := Result + FilteredData (Fields[FieldIdx].AsString) + FFieldDelimiter
        else
          Result := Result + Fields[FieldIdx].AsString + FFieldDelimiter;
  SetLength (Result, Length (Result) - 1);
end;

//-----------------------------------------------------------------------------
// Function  : StoredCaption
//
// Purpose   : Determine if Caption has to be Stored.
//-----------------------------------------------------------------------------
function TDBExport.StoredCaption;
begin
  Result := FCaption <> DEF_CAPTION;
end;

//-----------------------------------------------------------------------------
// Function  : GetSeparator
//
// Purpose   : Obtain List Separator from Windows.
// Output    : Separator.
//-----------------------------------------------------------------------------
function TDBExport.GetSeparator;
var
  Size : integer;
  Sep  : PChar;
begin
  // Allocate seperator size.
  Size := GetLocaleInfo (LOCALE_USER_DEFAULT, LOCALE_SLIST, nil, 0);
  Sep  := StrAlloc (Size + 1);
  try
    GetLocaleInfo (LOCALE_USER_DEFAULT, LOCALE_SLIST, Sep, Size);
    Result := Sep^;
  finally
    StrDispose (Sep);
  end;
end;

//=============================================================================
// PUBLIC METHODS
//=============================================================================

//-----------------------------------------------------------------------------
// Function  : Execute
//
// Purpose   : Export data.
//-----------------------------------------------------------------------------
function TDBExport.Execute;
var
  FContents : TStrings;
  FOpened   : boolean;
  Progress  : TProgressForm;
begin
  if FExecuting then Abort;
  Result     := False;
  Progress   := nil;
  FExecuting := True;
  try
    if Assigned (FOnExecute) then
      FOnExecute (Self);
    if Assigned (FDataSource) and Assigned (FDataSource.DataSet) then
    begin
      if Trim (FFileName) = '' then FFileName := GetFileName;
      if eoShowProgress in FOptions then Progress  := TProgressForm.Create (nil);
      try
        if eoShowProgress in FOptions then begin
          Progress.StatusMessage := 'Initializing...';
          Progress.CanCancel     := eoCanCancel in FOptions;
          Progress.Caption       := FCaption;
          Progress.Show;
          Progress.Refresh;
        end;
        FDataSource.DataSet.DisableControls;
        if Assigned (FOnInitialize) then
          FOnInitialize (Self);
        try
          FOpened := (eoAutoOpen in FOptions) and (not FDataSource.DataSet.Active);
          FContents := TStringList.Create;
          try
            if FOpened then FDataSource.DataSet.Open;
            try
              if (FHeader <> ehNone) then FContents.Add (GetHeader);
              if (eoShowProgress in FOptions) then begin
                Progress.MaxValue      := FDataSource.DataSet.RecordCount;
                Progress.StatusMessage := 'Exporting...';
              end;
              FDataSource.DataSet.First;
              while not FDataSource.DataSet.EOF do begin
                FContents.Add (GetRecord);
                FDataSource.DataSet.Next;
                if eoShowProgress in FOptions then begin
                  Progress.ProgressBy (1);
                  if (eoCanCancel in FOptions) then Application.ProcessMessages;
                  if Progress.ModalResult <> mrNone then Abort;
                end;
              end;
            finally
              if FOpened then FDataSource.DataSet.Close;
            end; // DataSet.Open
            if eoShowProgress in FOptions then Progress.StatusMessage := 'Saving...';
            FContents.SaveToFile (FFileName);
            Result := True;
          finally
            FContents.Free;
          end; // FContents.Create
        finally
          FDataSource.DataSet.EnableControls;
        end; // DataSet.DisableControls
      finally
        if eoShowProgress in FOptions then Progress.Release;
      end; // TProgressForm.Create
    end;
  finally
    FExecuting := False;
  end; // Executing
end;

//=============================================================================
// CREATE/DESTROY METHODS
//=============================================================================

//-----------------------------------------------------------------------------
// Constructor : Create
//
// Purpose     : Initialize instance.
//-----------------------------------------------------------------------------
constructor TDBExport.Create;
begin
  inherited Create (AOwner);
  FFieldDelimiter := GetSeparator;
  FOptions        := DefaultOptions;
  FCaption        := DEF_CAPTION;
  FHeader         := ehNone;
end;

//=============================================================================
// TDBExportEditor
//=============================================================================

//-----------------------------------------------------------------------------
// Function  : GetVerbCount
//-----------------------------------------------------------------------------
function TDBExportEditor.GetVerbCount;
begin
  Result := 1;
end;

//-----------------------------------------------------------------------------
// Function  : GetVerb
//-----------------------------------------------------------------------------
function TDBExportEditor.GetVerb;
begin
  case (index) of
    0 : Result := '&Export dataset';
  end; //case
end;

//-----------------------------------------------------------------------------
// Procedure : ExecuteVerb
//-----------------------------------------------------------------------------
procedure TDBExportEditor.ExecuteVerb;
begin
  case (index) of
    0 : (Component as TDBExport).Execute;
  end; //case
end;

//=============================================================================
// TDEFilenameProperty
//=============================================================================

procedure TDEFilenameProperty.Edit;
begin
  with (GetComponent(0) as TDBExport) do
  try
    FileName := GetFileName;
  except
    on EAbort do ;
  end;
end;

function TDEFilenameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paRevertable];
end;

//=============================================================================
// REGISTER
//=============================================================================

procedure Register;
begin
  RegisterComponents('Class', [TDBExport]);
  RegisterComponentEditor (TDBExport, TDBExportEditor);
  RegisterPropertyEditor (TypeInfo (string), TDBExport, 'FileName', TDEFilenameProperty);
end;

end.
