{
This file Copyright 2000 (c) CDF, Inc.
Written By: Edward Flick (Directrix1@yahoo.com)
Use at your own risk!
}

unit ExpressionEval;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, db;

const
  whitespace=[' ','+'];

type
  EUnknownFieldFunction = class(Exception);

type
  RegCustFunction = function(ins: array of String): String;
  ObjCustFunction = function(ins: array of String): String of Object;
  TCustFunction = class
    public
      { Public declarations }
      internalName: String;
      reg: boolean;
      regFunc: RegCustFunction;
      objFunc: ObjCustFunction;
    end;

type
  TCompiledChunkType = (cctLiteral, cctField, cctRegFunc, cctObjFunc);
  PCompiledExpr = ^TCompiledExpr;
  TCompiledExpr = record
    CompiledChunkType: TCompiledChunkType;
    Literal: String;
    Field: TField;
    RegFunc: RegCustFunction;
    ObjFunc: ObjCustFunction;
    ParamsHigh: Integer;
    Params: array of PCompiledExpr;
    Next: PCompiledExpr;
    end;

type
  TExpressionEval = class(TComponent)
  protected
    { Protected declarations }
    FCompExprs: array of PCompiledExpr;
    FSource: TDataset;
    procedure freeCompExpr(ce: PCompiledExpr);
    function compileFrom(expr: String;var idx: integer): PCompiledExpr;
    function processQuote(expr: String;var idx:integer;quotechar: char): PCompiledExpr;
    function processChar(expr: String;var idx: integer): PCompiledExpr;
    function processOther(expr: String;var idx: integer): PCompiledExpr;
    procedure skipWhiteSpace(expr: String;var idx:integer);
    //Predefined Custom functions
    function CUpper(ins: array of String): String;
    function CSubstr(ins: array of String): String;
    function CLen(ins: array of String): String;
    function CLeft(ins: array of String): String;
    function CRight(ins: array of String): String;
    function CPadL(ins: array of String): String;
    function CPadR(ins: array of String): String;
    function CAllTrim(ins: array of String): String;
    function CAt(ins: array of String): String;
  public
    { Public declarations }
    FCustFunctions: array of TCustFunction;
    function compile(expr: String): PCompiledExpr;
    function evaluateCompiled(ce: PCompiledExpr): String;
    function evaluate(expr: String): String;
    procedure freeCompiledExpressions;
    procedure registerFunction(FuncName: String; theFunc: RegCustFunction); overload; virtual;
    procedure registerFunction(FuncName: String; theFunc: ObjCustFunction); reintroduce; overload;
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property Source: TDataset read FSource write FSource;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Data Access', [TExpressionEval]);
end;

function TExpressionEval.processQuote(expr: String;var idx:integer;quotechar: char): PCompiledExpr;
var
  done: boolean;
  temps: String;
begin
try
  temps:='';
  inc(idx);
  done:=false;
  repeat
  if expr[idx]=quotechar then //A quotechar has been encountered
    begin                    //in the literal
    inc(idx);
    if idx<=length(expr) then
      if expr[idx]=quotechar then
        begin
        temps:=temps+quotechar; //At double quotechars denotes an embedded
        inc(idx);    //quotechar in the literal
        end
      else
        done:=true //At Single quotechar denotes end of literal
    else
      done:=true;  //End of line so don't check for double quote
    end
  else
    begin
    temps:=temps+expr[idx]; //Just another char in the literal
    inc(idx);
    end;
  until (done);
  New(result);
  result.Next:=nil;
  result.CompiledChunkType:=cctLiteral;
  result.Literal:=temps;
except
  raise;
  end;
end;

function TExpressionEval.processChar(expr: String;var idx: integer): PCompiledExpr;
var
  done: boolean;
  temps: string;
begin
try
  temps:='';
  inc(idx);
  done:=false;
  while (not done) do
    begin
    if idx>length(expr) then
      done:=true
    else
      if (ord(expr[idx])>=ord('0')) and (ord(expr[idx])<=ord('9')) then
        begin
        temps:=temps+expr[idx];
        inc(idx);
        end
      else
        done:=true;
    end;
  temps:=char(strtoint(temps));
  New(result);
  result.Next:=nil;
  result.CompiledChunkType:=cctLiteral;
  result.Literal:=temps;
except
  raise;
  end;
end;

function TExpressionEval.processOther(expr: String;var idx: integer): PCompiledExpr;
var
  done, bigkey: boolean;
  j,k: integer;
  key: string;
begin
result:=nil;
try
  New(result);
  result.Next:=nil;
  result.params:=nil;
  key:='';
  done:=false;
  bigkey:=expr[idx]='[';
  if bigkey then
    inc(idx);
  while (not done) do
    begin
    if (idx>length(expr)) then
      done:=true
    else
      if ((not (expr[idx] in whitespace)) and
         (expr[idx]<>')') and
         (expr[idx]<>',') and
         (expr[idx]<>'(')) or bigkey then
        begin
        if (expr[idx]=']') and bigkey then
          begin
          done:=true;
          inc(idx);
          end
        else
          begin
          key:=key+expr[idx];
          inc(idx);
          end;
        end
      else
        done:=true;
    end;
  skipWhiteSpace(expr,idx);
  if idx<=length(expr) then
    begin
    if expr[idx]='(' then
      begin
        k:=-1;
        for j:=0 to length(FCustFunctions)-1 do   //Check custom methods registry
          if (FCustFunctions[j].internalName=UpperCase(key)) then
            k:=j;
        if k=-1 then
          raise Exception(''''+key+''' is not a registered function.');
        if FCustFunctions[k].reg then  //Check if it is a regular or object function
          begin                        //and allocate chunk accordingly
          result.CompiledChunkType:=cctRegFunc;
          result.RegFunc:=FCustFunctions[k].regFunc;
          end
        else
          begin
          result.CompiledChunkType:=cctObjFunc;
          result.ObjFunc:=FCustFunctions[k].objFunc;
          end;
        skipWhiteSpace(expr,idx);
        try
          repeat
            inc(idx);
            setLength(result.params,length(result.params)+1);
            result.params[length(result.params)-1]:=nil;
            result.params[length(result.params)-1]:=compileFrom(expr,idx);
          until expr[idx]=')';
          result.ParamsHigh:=Length(result.params)-1;
          inc(idx);
        except
          for j:=0 to length(result.params) - 2 do
            try
              freeCompExpr(result.params[j]);
            except
              end;
          raise;
          end;
      end
    else
      begin
      if FSource.FindField(key)<>nil then
        begin
        result.CompiledChunkType:=cctField;
        result.Field:=FSource.FieldByName(key);   //Try fieldval method
        end
      else
        raise EUnknownFieldFunction.create(''''+key+''' is not a valid field or function.');
      end;
    end
  else
    if FSource.FindField(key)<>nil then
        begin
        result.CompiledChunkType:=cctField;
        result.Field:=FSource.FieldByName(key);   //Try fieldval method
        end
    else
      raise EUnknownFieldFunction.create(''''+key+''' is not a valid field or function.');
except
  Dispose(result);
  raise;
  end;
end;

procedure TExpressionEval.skipWhiteSpace(expr: String;var idx:integer);
var
  done: boolean;
begin
done:=false;
while (not done) do        //Skip placeholders
  begin
  if idx>length(expr) then
    done:=true
  else
    if expr[idx] in whitespace then
      inc(idx)
    else
      done:=true;
  end;
end;

//Evaluate expr using the Source dataset, as a fieldval source
function TExpressionEval.compileFrom(expr: String;var idx: integer): PCompiledExpr;
begin
  result:=nil;
  skipWhiteSpace(expr,idx);
  if ((idx<=length(expr)) and (not (expr[idx] in [',',')']))) then
    begin
    if expr[idx]='''' then //An apostrophe has been encountered
      begin                    //in the string
      result:=processQuote(expr,idx,'''');
      end
    else
      begin
      if expr[idx]='#' then //Ordinal Value code
        begin
        result:=processChar(expr,idx);
        end
      else
        begin                  //Get Other
        result:=processOther(expr,idx);
        end;
      end;
    result.Next:=compileFrom(expr,idx);
    end;
end;

function TExpressionEval.evaluateCompiled(ce: PCompiledExpr): String;
var
  cur: PCompiledExpr;
  j: integer;
  params: array of string;
begin
  cur:=ce;
  result:='';
  while cur<>nil do
    begin
    case cur.CompiledChunkType of
      cctLiteral: result:=result+cur.Literal;
      cctField: result:=result+cur.Field.AsString;
      cctRegFunc: begin
                  setLength(params,cur.ParamsHigh+1);
                  for j:=0 to cur.ParamsHigh do
                    params[j]:=evaluateCompiled(cur.Params[j]);
                  result:=result+cur.RegFunc(params);
                  end;
      cctObjFunc: begin
                  setLength(params,cur.ParamsHigh+1);
                  for j:=0 to cur.ParamsHigh do
                    params[j]:=evaluateCompiled(cur.Params[j]);
                  result:=result+cur.ObjFunc(params);
                  end
      end;
    cur:=cur.Next;
    end;
end;

function TExpressionEval.compile(expr: String): PCompiledExpr;
var
  index: integer;
begin
index:=1;
try
  result:=compileFrom(expr,index);
  setLength(FCompExprs,Length(FCompExprs)+1);
  FCompExprs[Length(FCompExprs)-1]:=result;
except
  on e: EUnknownFieldFunction do
    begin
    raise EUnknownFieldFunction.create('Error while compiling expression: '+chr(13)+chr(10)+
    expr+chr(13)+chr(10)+
    '@char('+inttostr(index)+'), With Error:'+chr(13)+chr(10)+
    e.message);         //Very handy exception routine, I believe
    end;
  on e: Exception do
    begin
    raise Exception.create('Error while compiling expression: '+chr(13)+chr(10)+
    expr+chr(13)+chr(10)+
    '@char('+inttostr(index)+'), With Error:'+chr(13)+chr(10)+
    e.message);         //Very handy exception routine, I believe
    end;
  end;
end;

function TExpressionEval.evaluate(expr: String): String;
var
  tce: PCompiledExpr;
begin
  tce:=compile(expr);
  result:=evaluateCompiled(tce);
  freeCompExpr(tce);
  FCompExprs:=Copy(FCompExprs,0,Length(FCompExprs)-1);
end;

procedure TExpressionEval.registerFunction(FuncName: String; theFunc: RegCustFunction);
var
  temp: TCustFunction;
  j: integer;
begin
temp:=nil;
if length(FCustFunctions)>0 then
  for j:=0 to length(FCustFunctions) -1 do
    if UpperCase(FuncName)=FCustFunctions[j].internalName then
      temp:=FCustFunctions[j];
if temp=nil then
  begin
  temp:=TCustFunction.Create();
  setLength(FCustFunctions, length(FCustFunctions)+1);
  FCustFunctions[length(FCustFunctions)-1]:=temp;
  end;
temp.internalName:=UpperCase(FuncName);
temp.regFunc:=theFunc;
temp.reg:=true;
end;

procedure TExpressionEval.registerFunction(FuncName: String; theFunc: ObjCustFunction);
var
  temp: TCustFunction;
  j: integer;
begin
temp:=nil;
if length(FCustFunctions)>0 then
  for j:=0 to length(FCustFunctions) -1 do
    if UpperCase(FuncName)=FCustFunctions[j].internalName then
      temp:=FCustFunctions[j];
if temp=nil then
  begin
  temp:=TCustFunction.Create();
  setLength(FCustFunctions, length(FCustFunctions)+1);
  FCustFunctions[length(FCustFunctions)-1]:=temp;
  end;
temp.internalName:=UpperCase(FuncName);
temp.objFunc:=theFunc;
temp.reg:=false;
end;

procedure TExpressionEval.freeCompExpr(ce: PCompiledExpr);
var
  j: integer;
begin
if ce<>nil then
  begin
  if ce.CompiledChunkType in [cctRegFunc, cctObjFunc] then
    for j:=0 to length(ce.Params) - 1 do
      freeCompExpr(ce.Params[j]);
  try dispose(ce); except end;
  end;
end;

procedure TExpressionEval.freeCompiledExpressions;
var
  j: integer;
begin
if length(FCompExprs)>0 then
  for j:=0 to length(FCompExprs)-1 do
    freeCompExpr(FCompExprs[j]);
FCompExprs:=nil;
end;

constructor TExpressionEval.Create(Owner: TComponent);
begin
inherited;
FCustFunctions:=nil;
FCompExprs:=nil;
registerFunction('UPPER',CUpper);
registerFunction('SUBSTR',CSubstr);
registerFunction('LEN',CLen);
registerFunction('LEFT',CLeft);
registerFunction('RIGHT',CRight);
registerFunction('PADL',CPadL);
registerFunction('PADR',CPadR);
registerFunction('ALLTRIM',CAllTrim);
registerFunction('AT',CAt);
end;

destructor TExpressionEval.Destroy;
var
  j: integer;
begin
if length(FCustFunctions)>0 then
  for j:=0 to length(FCustFunctions)-1 do
    FCustFunctions[j].Free;
freeCompiledExpressions;
inherited;
end;

// Default functions
function TExpressionEval.CUpper(ins: array of String): String;
begin
  result:=UpperCase(ins[0]);
end;

function TExpressionEval.CSubstr(ins: array of String): String;
begin
  if length(ins)<3 then
    result:=copy(ins[0],strtoint(ins[1]),2000000000)
  else
    result:=copy(ins[0],strtoint(ins[1]),strtoint(ins[2]));
end;

function TExpressionEval.CLen(ins: array of String): String;
begin
  result:=inttostr(length(ins[0]));
end;

function TExpressionEval.CLeft(ins: array of String): String;
begin
  result:=copy(ins[0],1,strtoint(ins[1]));
end;

function TExpressionEval.CRight(ins: array of String): String;
begin
  result:=copy(ins[0],length(ins[0])-strtoint(ins[1])+1,strtoint(ins[1]));
end;

function TExpressionEval.CPadL(ins: array of String): String;
var
  dc: char;
begin
  if length(ins)<3 then
    dc:=' '
  else
    dc:=ins[3][1];
  if length(ins[0])>=strtoint(ins[1]) then
    result:=ins[0]
  else
    result:=StringOfChar(dc,strtoint(ins[1])-length(ins[0]))+ins[0];
end;

function TExpressionEval.CPadR(ins: array of String): String;
var
  dc: char;
begin
  if length(ins)<3 then
    dc:=' '
  else
    dc:=ins[3][1];
  if length(ins[0])>=strtoint(ins[1]) then
    result:=ins[0]
  else
    result:=ins[0]+StringOfChar(dc,strtoint(ins[1])-length(ins[0]));
end;

function TExpressionEval.CAllTrim(ins: array of String): String;
begin
  result:=trim(ins[0]);
end;

function TExpressionEval.CAt(ins: array of String): String;
begin
  result:=inttostr(Pos(ins[0],ins[1]));
end;

end.
