unit msWordMerge;
// TmsWordMerge encapsulates a number of functions to make Mail merging with Word and Delphi Projects easy.

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DB, DBTables, comObj;

{$I msWordConstants.inc}

type

  TfmsMergeOptions = (fmsNoMerge,fmsPreview,fmsPrint,fmsNew);
  TmsWordMerge = class(TComponent)

  private
    { Private declarations }
    //fQuery: TQuery;
    fMergeOption: TfmsMergeOptions;
    fDataSource: TDataSource;
    fmsWordDoc : TfileName;
    fCloseWord : Boolean;
    fpromptsave : Boolean;
    fDataFile : String;

    Procedure mergedoc(docname,datafile: String; mergemode: TfmsMergeOptions );

  protected
    { Protected declarations }
  public
    { Public declarations }
    // this function gets the Word Object. If word is not running, it's started.
    Function GetmsWordObject{}:OLEVariant;
    Function WordDocumentPath:String;
    Function WordTemplatePath:String;
    Function Execute: Boolean;
    Procedure WriteWordFields;
    Function SelectWordDocumentFile{}:Boolean;
    Function selectWordTemplateFile{}:Boolean;
    Function selectMergeDataFile{}:Boolean;
    Procedure CloseWordObject;
    Constructor Create(Aowner:TComponent); Override;
    Destructor Destroy; Override;
  published
    { Published declarations }
    Property DataSource: TDataSource Read fDataSource Write fDataSource;
    Property WordMergeDocument : TfileName Read fmsWordDOc    Write fmsWordDOc;
    Property OutputFile : String Read fdatafile    Write fdatafile ;
    Property MergeOption : TFmsMergeOptions Read fMergeOption Write fMergeOption default fmsPreview;
    Property CloseWord   : Boolean  Read fCloseWord Write fCloseWord;
    Property PromptSave   : Boolean  Read fPromptSave Write fPromptSave;
  end;

 //procedure Register;



implementation


Function tmsWordMErge.GetmsWordObject{}:OLEVariant;

Begin

   try
    try
      Result := GetActiveOLEObject('Word.Application');
    except
      Result := CreateOLEObject('Word.Application');
    end;
    except on EOleSysError  do
      Result := Unassigned
    end;



end;

Procedure TmsWordMerge.CloseWordObject;
var WordObj: OLEVariant;
begin
   //  Close Word
   wordObj := Unassigned ;

    try
      // but only if it's running!
      wordObj := GetActiveOLEObject('Word.Application');

      if fPromptSave then
       WordObj.quit (wdPromptToSaveChanges)
       else
       WordObj.quit(wdDoNotSaveChanges);

    except  on EOleSysError do
      wordObj := Unassigned ;
    end;


end;


Function TmsWordMerge.SelectWordDocumentFile{}:Boolean;
var d : TOpenDialog ;

begin
     result := false;
     d := nil;
     try
        d := TOpenDialog.create(nil);
        d.InitialDir := WordDocumentPath ;
        d.FileName := '*.doc';
        d.DefaultExt := 'doc';
        d.Filter := 'Word Document File|*.doc';
        d.title := 'Choose Word Document';
        result := d.execute  ;
           if result then
           fmsWordDoc  := d.filename;


     finally
       d.Free;
     end;

end;

Function TmsWordMerge.selectWordTemplateFile{}: Boolean;
         // Lets the user Select a Word Template for creating a new document
         // returns true if a valid selection was made

var d : TOpenDialog ;
begin
     d := nil;
     result := false;
     try
        d := TOpenDialog.create(nil);
        d.InitialDir := WordTemplatePath ;
        d.FileName := '*.dot';
        d.DefaultExt := 'dot';
        d.Filter := 'Word Template File|*.dot';
        d.title := 'Choose template';
        result := d.execute;
        if result then
           fmsWordDoc  := d.filename;

     finally
       d.Free;
     end;

end;


Function TmsWordMerge.selectMergeDataFile{}:Boolean ;
         // Lets the user specify the name for the file containing the exported data from Tdatasource;
         // returns true if a valid selection was made

var d : TSaveDialog ;
begin
     d := nil;
     result := false;
     try
        d := TSaveDialog.create(nil);
        d.InitialDir := WordDocumentPath ;
        d.FileName := fDataFile ;
        d.DefaultExt := 'txt';
        d.Filter := 'Merge Data Text File|*.txt';
        d.title := 'Merge Data File';
        result := d.execute;
        if result then
           fdatafile := d.filename;

     finally
       d.Free;
     end;

end;



Function ReplaceChar(oldChar,newChar, argStr:String): String;
var
  p : Integer;
  s, s1 : String;
Begin
  s := argStr;
  p := POS(oldChar,s);
  while p <> 0 do
  begin
    s1 := copy(s,1,p-1) + newChar + copy(s,p + 1,length(s ) - p + 1);
    s := s1;
    p := Pos(oldChar,s);
  end; // while
  ReplaceChar := s;
end;

Procedure TmsWordMerge.WriteWordFields;
var
  s : TStringList;
  item : String;
  i : Integer;
begin
  s := TstringList.Create;
  try
    item := '';

    // add field names to first row
    with fdataSource.DataSet do
    begin
      for i := 0 to FieldCount - 1 do
      begin
        if not (Fields[i].isblob) then
        begin
          if item = '' then
          begin
            item := #34 + Fields[i].Fieldname  +#34;
          end
          else
          begin
            item := item + ',' + #34 +  Fields[i].fieldname + #34;
          end; // if
        end ; // if not blob
      end; // next

      item := replaceChar('.','',item);
      s.Add(replaceChar(' ','_',item));

      // write the field values
      first;
      while not(eof) do
      begin
        item := '';
        for i := 0 to fieldcount - 1 do
        begin
          if not (Fields[i].isblob) then
          begin
            if item = '' then
              item := #34 + Fields[i].Text +#34
            else
              item := item + ',' + #34 +  Fields[i].Text  + #34;
          end; // if not blob
        end;  //     for i
        s.add(item);
        next;
      end; // next while
    end ; // with dataset

    s.SaveToFile(fdatafile);


  finally
    s.free;
  end;
end;

Function TmsWordMerge.WordDocumentPath:String;
         // function to return WORD's Document Directory Name

var msWordObj: OLEVariant;
begin
  try
    msWordObj := GetMSWordObject ;
  try
    result:= msWordObj.Options.DefaultFilePath[wdDocumentsPath];
    except
     on EOleException do
      Result := ''
     end;

  finally
   msWordObj := unassigned;
  end;

end;

Function TmsWordMerge.WordTemplatePath:String;
var msWordObj: OLEVariant;
begin
 try
    msWordObj := GetMSWordObject ;
    try
     result:= msWordObj.Options.DefaultFilePath[wdUserTemplatesPath];
    except
    on EOleException do
      Result := '';
    end;
  finally
   msWordObj := unassigned;
  end;

end;


Procedure TmsWordMerge.mergedoc(docname,datafile: String; mergeMode : TfmsMergeOptions );
var
    mergeDoc,mergedDoc: String;
     msWordObj: OLEVariant;
begin
 // first, check the datafile and document file exist
  if not(fileExists(docname) ) or not(fileExists(datafile))  then
     begin
          MessageDlg('Either the document file "'
          + docname + '" or the data file "' +
          datafile + '" is missing, corrupt or invalid. Cannot proceed!',mtError,[mbOK],0);
          abort;
     end;


 try
    msWordObj := GetMSWordObject ;

   try
    msWordObj.Visible := true;
    msWordObj.Activate;

    // decide whether to create a new document or open en existing one
    if (mergeMode = fmsNew) then
        msWordObj.Documents.Add(docName,false)
    else
       msWOrdOBJ.Documents.Open(docname);

     application.processmessages;
     // attach data source
    msWordObj.ActiveDocument.MailMerge.OpenDataSource(datafile,,True);

    // now merge
    if (mergeMode = fmsNew) then
      Begin
       // it's a new document so display for editing
       msWordObj.Visible := true;
       msWordObj.Activate;
      end;

    if (mergeMode = fmsPrint) then
       msWordObj.ActiveDocument.MailMerge.Destination := WdSendToPrinter;


    if (mergeMode = fmsPreview) then
      begin
       msWordObj.ActiveDocument.MailMerge.Destination := wdSendToNewDocument;
       msWordObj.Visible := true;
       msWordObj.Activate;
      end;

    // save the document name
    mergeDoc := msWordObj.ActiveDocument.name;

    // execute the merge
    if (mergeMode = fmsPreview) or (mergeMode = fmsPrint) then
      begin
      // merge the document
      msWordObj.ActiveDocument.MailMerge.Execute;
      // let it happen
      Application.ProcessMessages ;

      // save the new documents name
      mergedDoc :=  msWordObj.ActiveDocument.name;

      // now close the mail merge document
      // select the mail merge document
       msWordObj.Documents.open (docName);
       Application.ProcessMessages ;
       msWordObj.ActiveDocument.close ( wdDoNotSaveChanges);
       Application.ProcessMessages ;

      // if the mergeoption was print
      // then close the merged document at well
      if (mergeMode = fmsPrint) then
         msWordObj.ActiveDocument.close  ( wdDoNotSaveChanges);
         Application.ProcessMessages ;
      end;

   except on EOleException do
   end;
  finally
    //Free;
    msWordObj := Unassigned ;

 end;
end;

Function TmsWordMerge.Execute: Boolean;
Begin
  //  check that the word document file exists
  result := false;

  if DataSource = nil then
  begin
    MessageDlg('DataSource not specified!', mtError, [mbOk], 0);
    abort;
  end;
  // is the Datasource valid?
  if DataSource.DataSet = nil then
  begin
    MessageDlg('DataSource needs to specify a dataset!', mtError, [mbOk], 0);
    abort;
  end;

  // if the user wants a new document then the WordMergeDocument must be
  // Word Document Template
  if (fMergeOption = fmsNew)  then
     begin
        // check the file exists
       if not(fileExists(fmsWordDoc )) then
         begin
         MessageDlg('You must specify a valid Template Document to create a new document!',mtError,[mbOK],0);
         Abort;
         end;
      // now make sure it's a document template file!
       if Pos('.dot',fmsWordDoc ) = 0 then
            begin
            MessageDlg( fmsWordDoc  + ' is not a Template Document! Try again.',mtError,[mbOK],0);
            Abort;
            end;
  end; // if new doc



  screen.Cursor := crHourglass;
  try
    // check the data file
    if fdataFile = ''  then
    begin
      MessageDlg('Merge data file not specified!', mtError, [mbOk], 0);
      abort;
    end;

     WriteWordFields;

    // if a doc file was specified then do try the merge (perhaps a separate flag
    //  would be good??)
    if fmsWordDoc <> '' then
    begin
      if not FileExists(fmsWordDoc) then
      begin
        MessageDlg('Cannot find the nominated word document to merge!', mtError, [mbOK], 0);
        abort;
      end
      else
      begin
        mergedoc(fmsWordDOc,fdatafile, fMergeOption )
      end;
    end;

    result := true;
  finally
    screen.Cursor := crDefault;
  end;
end;

Constructor TmsWordMerge.Create(Aowner:TComponent);

begin
     inherited Create(Aowner);

     fMergeOption := fmsPreview ;
     fDataFile := WordDocumentPath +  '\MergeData.txt';
     fCloseWord := false;
     fpromptsave := false;
     
end;

destructor TmsWordMerge.Destroy;

begin

   //  Close Word

   if fCloseWord then
      CloseWordObject;

   // free the variant

     inherited Destroy;

end;


end.
