{*****************************************************************************
 *
 *  DAComponentGrid.pas - Component Grid
 *
 *  Copyright (c) 2000 Diego Amicabile
 *
 *  Author:     Diego Amicabile
 *  E-mail:     diegoami@yahoo.it
 *  Homepage:   http://www.geocities.com/diegoami
 *
 *  This component is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation;
 *
 *  This component is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this component; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 *****************************************************************************}


unit DAComponentGrid;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

type
  TControlClass = class of TControl;
  EOutOfListException = class(Exception);
  EComponentNotFoundException = class(Exception);


  TOnPaintingControl = procedure(Sender : TObject; X, Y : integer; Control : TControl) of object;
  TOnCreatingControl = procedure(Sender : TObject; X, Y : integer; var Control : TControl) of object;

  TDAComponentGrid = class(TPanel)
  private
    FClassType : TControlClass;
    FXCompNumber : integer;
    FYCompNumber : Integer;
    FXRatio : double;
    FYRatio : double;
    FOnPaintingControl : TOnPaintingControl;
    FOnCreatingControl : TOnCreatingControl;
    procedure setChildClassType(const Value: TControlClass);
    procedure setXCompNumber(const Value: integer);
    procedure setYCompNumber(const Value: integer);
    procedure setComponentAt(X,Y : integer; const Value : TControl);
    function getComponentAt(X,Y : integer) : TControl;
    procedure setRatio(ind : integer; const Value : double);
  protected
    ComponentList : TList;
    procedure UpdateClasses;
    procedure FreeClasses;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Resize; override;
    property ComponentAt[X : integer; Y : integer] : TControl read GetComponentAt write SetComponentAt;
    procedure GetCoordsForControl(Control : TControl; var X, Y : integer);
    property ChildClassType : TControlClass read  FClassType write setChildClassType;
  published
    property XCompNumber : integer read FXCompNumber write setXCompNumber default 1;
    property YCompNumber : integer read FYCompNumber write setYCompNumber default 1;
    property XRatio : double index 1 read FXRatio write setRatio;
    property YRatio : double index 2 read FYRatio write setRatio;
    property OnPaintingControl : TOnPaintingControl read FOnPaintingControl write FOnPaintingControl;
    property OnCreatingControl : TOnCreatingControl read FOnCreatingControl write FOnCreatingControl;
  end;

procedure Register;

implementation
{$R DACOMPONENTGRID.DCR}
procedure Register;
begin
  RegisterComponents('Diego Amicabile', [TDAComponentGrid]);
end;

{ TDAComponentGrid }

constructor TDAComponentGrid.Create(AOwner: TComponent);
begin
  inherited;
  FXRatio := 0.5;
  FYRatio := 0.5;
  ComponentList := TList.Create;
  UpdateClasses;
end;

destructor TDAComponentGrid.Destroy;
begin
  FreeClasses;
  ComponentList.Free;
  inherited
end;

procedure  TDAComponentGrid.Resize;
var xgrid, ygrid : integer;
    i,j  : integer;
    CurrControl : TControl;
begin
  inherited;
  if ComponentList.Count = 0 then exit;
  xgrid := Width div (4*FXCompNumber);
  ygrid := Height div (4*FYCompNumber);
  for j := 1 to FYCompNumber do begin
    for i := 1 to FXCompNumber do begin
      CurrControl := TControl(ComponentList.Items[(J-1)*FXCompNumber+(I-1)]);
      if CurrControl = nil then continue;
        CurrControl.Left := Round( 4*XGrid*(1-XRatio)/2+(i-1)*XGrid*4);
        CurrControl.Width := Round(XGrid*4*XRatio);
        CurrControl.Height := Round(YGrid*4*YRatio);
        CurrControl.Top := Round(4*YGrid*(1-YRatio)/2+(j-1)*YGrid*4);
        if Assigned(FOnPaintingControl) then
          FOnPaintingControl(Self,i,j,CurrControl);
    end;
  end;
  Invalidate;
end;


function TDAComponentGrid.getComponentAt(X, Y: integer): TControl;
begin
  result := TControl(ComponentList.Items[(Y-1)*FXCompNumber+X-1]);
end;

procedure TDAComponentGrid.setChildClassType(const Value: TControlClass);
begin
  FreeClasses;
  FClassType := Value;
  UpdateClasses;
end;

procedure TDAComponentGrid.setComponentAt(X, Y: integer;
  const Value: TControl);
begin
  if (X > FXCompNumber) or (Y > FYCompNumber) or (X < 1) or (Y < 1) then begin
    raise EOutOfListException.Create('Component out of bounds');
    exit
  end;
  ComponentList.Items[(Y-1)*FXCompNumber+X-1] := Value;
end;

procedure TDAComponentGrid.setXCompNumber(const Value: integer);
begin
  FXCompNumber := Value;
  if FXCompNumber < 1 then FXCompNumber := 1;
  FreeClasses;
  UpdateClasses;
end;

procedure TDAComponentGrid.setYCompNumber(const Value: integer);
begin
  FYCompNumber := Value;
  if FYCompNumber < 1 then FYCompNumber := 1;
  FreeClasses;
  UpdateClasses;
end;

procedure TDAComponentGrid.FreeClasses;
var i : integer;
begin
  if FClassType = nil then exit;
  for i := 0 to ComponentList.Count-1 do
  begin
    TControl(ComponentList.Items[i]).Free;
  end;
  ComponentList.Clear;
end;

procedure TDAComponentGrid.UpdateClasses;
var i,j : integer;
  NewObj : TControl;
begin
  if FClassType = nil then exit;
  for j := 1 to FYCompNumber do
  begin
    for i := 1 to FXCompNumber  do
    begin
      NewObj := FClassType.Create(Self);
      NewObj.Parent := Self;
      ComponentList.Add(NewObj);
      if Assigned(FOnCreatingControl) then
        FOnCreatingControl(Self,i,j,NewObj);
    end;
  end;
  Resize;
end;

procedure TDAComponentGrid.setRatio( ind : integer; const Value: double);
var tempval : double;
begin
  TempVal := Value;
  if TEmpVal > 1 then
    TEmpval := 1;
  if Tempval < 0 then
    tempval := 0.01;
  if ind = 1 then
    FXRatio := Tempval
  else
    FYRatio := Tempval;
  Resize;

end;

procedure TDAComponentGrid.GetCoordsForControl(Control: TControl; var X,
  Y: integer);
var i,j : integer;
begin
  for j := 1 to FYCompNumber do
    for i := 1 to FXCompNumber do
      if ComponentAt[i,j] = Control then begin
        X := I;
        Y := J;
        exit
      end;
  raise EComponentNotFoundException.Create('Component not found in grid ');
end;

end.
