unit CRCfind;

interface
uses
  Windows, Messages, Dialogs, SysUtils, Classes;
//-------------------------------------------------
//-------------------------------------------------
const CM_CRC=WM_USER+149;

const
  crcUnknownError=-6;
  crcCodeInvalid =-5;
  crcProcessAbort=-4;
  crcNoFileAccess=-3;
  crcTwiceFound  =-2;
  crcMarkNotFound=-1;

type
  TCRCOperation=(crcWrite,crcRead);

type
  TCRCdata=record
    Mark :array[1..14]of Char;
    Code :Word;
  end;

type
  TCRCFinder = class(TThread)
  private
    { Private declarations }
    FFileName :string;
    FCRCdata  :TCRCdata;
    FWindow   :hWnd;
    FOperation:TCRCOperation;
    FCRCstart :Cardinal;
    FCRCcode  :Word;
    function FindCRC:Integer;
    function CheckCRC:Integer;
    function WriteCRC:Integer;
    function UpdateCRC(ANewByte:Byte;ACRCsum:Word):Word;
  protected
    { Protected declarations }
    procedure Execute; override;
  public
    { Public declarations }
    constructor CreateAndOperate(FileName:string;
                                 CRCdata:TCRCdata;
                                 Window:hWnd;
                                 Operation:TCRCOperation);virtual;
  end;

implementation
//-------------------------------------------------
//-------------------------------------------------
const
  CRCtable:array[0..255] of Word=(
    $0000,$1021,$2042,$3063,$4084,$50a5,$60c6,$70e7,
    $8108,$9129,$a14a,$b16b,$c18c,$d1ad,$e1ce,$f1ef,
    $1231,$0210,$3273,$2252,$52b5,$4294,$72f7,$62d6,
    $9339,$8318,$b37b,$a35a,$d3bd,$c39c,$f3ff,$e3de,
    $2462,$3443,$0420,$1401,$64e6,$74c7,$44a4,$5485,
    $a56a,$b54b,$8528,$9509,$e5ee,$f5cf,$c5ac,$d58d,
    $3653,$2672,$1611,$0630,$76d7,$66f6,$5695,$46b4,
    $b75b,$a77a,$9719,$8738,$f7df,$e7fe,$d79d,$c7bc,
    $48c4,$58e5,$6886,$78a7,$0840,$1861,$2802,$3823,
    $c9cc,$d9ed,$e98e,$f9af,$8948,$9969,$a90a,$b92b,
    $5af5,$4ad4,$7ab7,$6a96,$1a71,$0a50,$3a33,$2a12,
    $dbfd,$cbdc,$fbbf,$eb9e,$9b79,$8b58,$bb3b,$ab1a,
    $6ca6,$7c87,$4ce4,$5cc5,$2c22,$3c03,$0c60,$1c41,
    $edae,$fd8f,$cdec,$ddcd,$ad2a,$bd0b,$8d68,$9d49,
    $7e97,$6eb6,$5ed5,$4ef4,$3e13,$2e32,$1e51,$0e70,
    $ff9f,$efbe,$dfdd,$cffc,$bf1b,$af3a,$9f59,$8f78,
    $9188,$81a9,$b1ca,$a1eb,$d10c,$c12d,$f14e,$e16f,
    $1080,$00a1,$30c2,$20e3,$5004,$4025,$7046,$6067,
    $83b9,$9398,$a3fb,$b3da,$c33d,$d31c,$e37f,$f35e,
    $02b1,$1290,$22f3,$32d2,$4235,$5214,$6277,$7256,
    $b5ea,$a5cb,$95a8,$8589,$f56e,$e54f,$d52c,$c50d,
    $34e2,$24c3,$14a0,$0481,$7466,$6447,$5424,$4405,
    $a7db,$b7fa,$8799,$97b8,$e75f,$f77e,$c71d,$d73c,
    $26d3,$36f2,$0691,$16b0,$6657,$7676,$4615,$5634,
    $d94c,$c96d,$f90e,$e92f,$99c8,$89e9,$b98a,$a9ab,
    $5844,$4865,$7806,$6827,$18c0,$08e1,$3882,$28a3,
    $cb7d,$db5c,$eb3f,$fb1e,$8bf9,$9bd8,$abbb,$bb9a,
    $4a75,$5a54,$6a37,$7a16,$0af1,$1ad0,$2ab3,$3a92,
    $fd2e,$ed0f,$dd6c,$cd4d,$bdaa,$ad8b,$9de8,$8dc9,
    $7c26,$6c07,$5c64,$4c45,$3ca2,$2c83,$1ce0,$0cc1,
    $ef1f,$ff3e,$cf5d,$df7c,$af9b,$bfba,$8fd9,$9ff8,
    $6e17,$7e36,$4e55,$5e74,$2e93,$3eb2,$0ed1,$1ef0
  );
//-------------------------------------------------
constructor TCRCFinder.CreateAndOperate;
begin
  inherited Create(False);
  FFileName :=FileName;
  FCRCdata  :=CRCdata;
  FWindow   :=Window;
  FOperation:=Operation;
end; { constructor TCRCfinder.CreateAndOperate }
//-------------------------------------------------
procedure TCRCFinder.Execute;
begin
  FreeOnTerminate:=True;
  if FOperation=crcWrite then
    WriteCRC()
  else
    CheckCRC();
end;
//-------------------------------------------------
function TCRCFinder.FindCRC:Integer;
var
  CRCzone:^TCRCdata;
  hF,hFileMap,FileSize:Integer;
  i,TempResult:Integer;
  Found:Boolean;
  MapStart:Pointer;
const
  CRCsize=sizeof(FCRCdata);
  MarkSize=sizeof(FCRCdata.Mark);
begin
  TempResult:=crcMarkNotFound;
  Found:=False;
  FCRCcode:=0;
  FCRCstart:=0;
  hFileMap:=0;
  MapStart:=nil;
  hF:=INVALID_HANDLE_VALUE;
  try
    Result:=crcUnknownError;
    { Try to open the file }
    hF:=CreateFile(PChar(FFileName),
                   GENERIC_READ,
                   FILE_SHARE_READ,
                   nil,
                   OPEN_EXISTING,
                   FILE_ATTRIBUTE_COMPRESSED or FILE_ATTRIBUTE_HIDDEN or
                   FILE_ATTRIBUTE_READONLY or FILE_ATTRIBUTE_SYSTEM	,
                   0);
    If hF=INVALID_HANDLE_VALUE then
    begin
      Result:=crcNoFileAccess;
      Exit ;
    end;
    { get file size to calculate the range of addresses }
    FileSize:=GetFileSize(hF,nil);
    { map the file into memory }
    hFileMap:=CreateFileMapping(hF,
                                nil,
                                PAGE_READONLY,
                                0,0,nil);
    if (hFileMap=0) or (FileSize=0) then
    begin
      Result:=crcNoFileAccess;
      Exit;
    end ;
    { get starting address of the mapping }
    MapStart:=MapViewOfFile(hFileMap,
                            FILE_MAP_READ,
                            0,0,0);
    if MapStart=nil then
    begin
      Result:=crcUnknownError;
      Exit;
    end;
    { walk to the end of the mapping minus some last bytes }
    CRCzone:=MapStart;
    for i:=0 to FileSize-CRCsize do
    begin
      Inc(Cardinal(CRCzone));
      if CompareMem(@(CRCzone^.Mark),@FCRCdata.Mark,MarkSize)then
      begin
        if not Found then   // we've found CRCmark
        begin
          Found:=True;
          { all right so far; save CRCdata.Code }
          TempResult:=CRCzone^.Code;
          { remember position of the CRC code }
          FCRCstart:=Cardinal(CRCzone)-Cardinal(MapStart)+MarkSize;
          { skip CRCsize-1 bytes }
          CRCzone:=Pointer(Cardinal(CRCzone)+CRCsize-1);
        end
        else  begin         // an error-twice found
          TempResult:=crcTwiceFound;
          Break;
        end
      end
      else begin // CompareMem=False -> update CRC
        FCRCcode:=UpdateCRC(Byte(Pointer(CRCzone)^),FCRCcode);
      end;
      { abort if the process terminates }
      if Terminated then
      begin
        TempResult:=crcProcessAbort;
        Break;
      end;
    end;  // finish the walk
    Result:=TempResult;
  finally
    if MapStart<>nil then UnMapViewOfFile(MapStart);
    if hFileMap<>0 then CloseHandle(hFileMap);
    if hF<>0 then CloseHandle(hF);
  end;
end; { function TCRCFinder.FindCRC }
//-------------------------------------------------
function TCRCFinder.CheckCRC:Integer;
var
  TempResult:Integer;
begin
  Result:=crcUnknownError;
  try
    TempResult:=FindCRC;
    if TempResult>=0 then
    { check CRC codes }
    if TempResult<>FCRCcode then
      TempResult:=crcCodeInvalid ;
    Result:=TempResult;
  finally
    PostMessage(FWindow,CM_CRC,36,Result);
  end;
end; { function TCRCFinder.CheckCRC }
//-------------------------------------------------
function TCRCFinder.WriteCRC:Integer;
var
  hF,PutBytes:Integer;
begin
  Result:=FindCRC;
  if Result>=0 then
  begin
    { write CRCcode }
    Result:=crcUnknownError;
    hF:=INVALID_HANDLE_VALUE;
    try
      { Try to open the file }
      hF:=CreateFile(PChar(FFileName),
                     GENERIC_WRITE or GENERIC_READ,
                     FILE_SHARE_READ,
                     nil,
                     OPEN_EXISTING,
                     FILE_ATTRIBUTE_COMPRESSED or FILE_ATTRIBUTE_HIDDEN or
                     FILE_ATTRIBUTE_READONLY or FILE_ATTRIBUTE_SYSTEM	,
                     0);
      If hF=INVALID_HANDLE_VALUE then
      begin
        Result:=crcNoFileAccess;
        Exit ;
      end;
      SetFilePointer(hF,FCRCstart,nil,FILE_BEGIN);
      WriteFile(hF,FCRCcode,sizeof(FCRCdata.code),PutBytes,nil);
      if PutBytes<>sizeof(FCRCdata.code) then
        Result:=crcNoFileAccess
      else
        Result:=FCRCcode;
    finally
      if hF<>0 then CloseHandle(hF);
      SendMessage(FWindow,CM_CRC,36,Result);
    end
  end
  else
    SendMessage(FWindow,CM_CRC,36,Result);
end; { function TCRCFinder.WriteCRC }
//-------------------------------------------------
function TCRCFinder.UpdateCRC(ANewByte:Byte;ACRCsum:Word):Word;
begin
  Result:=CRCtable[(ACRCsum and $00FF)xor ANewByte]xor(ACRCsum shr 8);
end; { function UpdateCRC }
//-------------------------------------------------
end.
