{written by Carl Dippel, Carl@netusa.net, donated to the public domain}

unit StartProcessBitBtn;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  buttons;

type

  TProcessReturnEvent = procedure(Owner: TComponent; Normal: boolean) of Object;
  TCancelClickEvent = procedure(Owner: TComponent) of Object;
  TStartFailEvent = procedure(Owner: TComponent; Error: integer) of Object;
  TCancelFailEvent = procedure(Owner: TComponent) of Object;

  TStartProcessBitBtn = class(TBitBtn)
  private
    ProcessInfo: TProcessInformation;
    Running: Boolean;
    Normal: Boolean;
    FOnProcessReturn: TProcessReturnEvent;
    FOnCancelClick: TCancelClickEvent;
    FOnStartFail: TStartFailEvent;
    FOnCancelFail: TCancelFailEvent;
    FWaitForReturn:Boolean;
    FCommandLine: string;
    FWorkingDir: string;
    FStartCaption: string;
    FStartGlyph: TBitmap;
    FCancelCaption: string;
    FCancelGlyph: TBitmap;
    FCancelWarning: string;
    FDisplayCancelWarning: Boolean;
    procedure SetCancelGlyph(const Value: TBitmap);
  protected
    procedure ProcessReturn; dynamic;
    procedure CancelClick; dynamic;
    procedure StartFail(Error:Integer); dynamic;
    procedure CancelFail; dynamic;
  public
    constructor Create( Owner: TComponent ); override;
    destructor Destroy; override;
    procedure Click; override;
  published
    property OnProcessReturn: TProcessReturnEvent read FOnProcessReturn write FOnProcessReturn;
    property OnCancelClick: TCancelClickEvent read FOnCancelClick write FOnCancelClick;
    property OnStartFail: TStartFailEvent read FOnStartFail write FOnStartFail;
    property OnCancelFail: TCancelFailEvent read FOnCancelFail write FOnCancelFail;
    property WaitForReturn: Boolean read FWaitForReturn write FWaitForReturn;
    property CommandLine: string read FCommandLine write FCommandLine;
    property WorkingDir: string read FWorkingDir write FWorkingDir;
    property CancelCaption: string read FCancelCaption write FCancelCaption;
    property CancelGlyph: TBitmap read FCancelGlyph write SetCancelGlyph;
    property CancelWarning: string read FCancelWarning write FCancelWarning;
    property DisplayCancelWarning:Boolean read FDisplayCancelWarning write FDisplayCancelWarning;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Freeware', [TStartProcessBitBtn]);
end;

{ TStartProcessBitBtn }

constructor TStartProcessBitBtn.Create(Owner: TComponent);
begin
  inherited Create(Owner);
  FCancelGlyph:=TBitMap.Create;
  FStartGlyph:=TBitMap.Create;
  Running:=false;
  FWaitForReturn:=true;
  FCommandLine:='COMMAND.COM';
  FWorkingDir:='.';
  FCancelCaption:='Cancel';
  FDisplayCancelWarning:=true;
  FCancelWarning:='Exiting from the process is best.'+#13+#10+
                  'Select "Yes" only if you can''t'+#13+#10+
                  'exit normally.  Are you sure?';
end;

destructor TStartProcessBitBtn.Destroy;
begin
  FStartGlyph.Free;
  FCancelGlyph.Free;
  inherited Destroy
end;

procedure TStartProcessBitBtn.SetCancelGlyph(const Value: TBitmap);
begin
  FCancelGlyph.Assign(Value);
end;

procedure TStartProcessBitBtn.Click;
var
  StartupInfo: TStartupInfo;
  rc:dword;
begin
  if Caption=CancelCaption then
    begin
      CancelClick;
      exit
    end;

  inherited Click;

  // execute process
  GetStartupInfo(StartupInfo);
  if not CreateProcess(nil, pChar(FCommandLine), nil, nil, false, 0, nil,
         pChar(FWorkingDir),StartupInfo, ProcessInfo) then StartFail(GetLastError);

  if not FWaitForReturn then exit;

  // wait for process to complete
  rc:=WAIT_TIMEOUT;
  FStartCaption:=Caption;
  FStartGlyph.Assign(Glyph);
  Caption:=FCancelCaption;
  Glyph.Assign(FCancelGlyph);
  Running:=true;

  while (rc=WAIT_TIMEOUT) and (Running) do
    begin
      Application.ProcessMessages;
      rc:=WaitForSingleObject(ProcessInfo.hProcess,500);
    end;
  Caption:=FStartCaption;
  Glyph.Assign(FStartGlyph);
  Running:=false;
  ProcessReturn;
end;

procedure TStartProcessBitBtn.CancelClick;
begin
  if Assigned(FOnCancelClick) then FOnCancelClick(Self);
  if FDisplayCancelWarning then
    begin
      beep;
      if MessageDlg(FCancelWarning,mtConfirmation,[mbYes,mbNo],0)= mrNo then exit
    end;
  Normal:=false;
  if not TerminateProcess(ProcessInfo.hProcess,1) then CancelFail;
end;

procedure TStartProcessBitBtn.ProcessReturn;
begin
  if Assigned(FOnProcessReturn) then FOnProcessReturn(Self,Normal);
end;

procedure TStartProcessBitBtn.StartFail(Error: Integer);
begin
  if Assigned(FOnStartFail)
  then FOnStartFail(Self,Error)
  else MessageDlg('Start failed'+#13+#10+'(Error = '+
                  IntToStr(Error)+')',mtError,[mbOK],0);
end;

procedure TStartProcessBitBtn.CancelFail;
begin
  if Assigned(FOnCancelFail)
  then FOnCancelFail(Self)
  else MessageDlg('Cancel failed',mtError,[mbOK],0);
end;

end.
