//*****************************************************************************
//
//    System          : BtrUtils
//    Category        : Main class for incapsulate Btrive API
//    Source          : none
//
//    Copyright (c), Leonov Alexander
//
//    File version $Revision:1.4$ at $Date:28.01.2004 19:32:30$
//
//    Author : Leonov Alexander
//    web    : http://dev.zibn.net/     mirror at:  http://home.ural.ru/~tigra/
//    e-mail : dev@zibn.net
//
//*****************************************************************************

unit lBtrieve;

interface

uses lBtrType, Classes, SysUtils;

Type
  TBtrErrorSet = set of byte;

Type
 TBtrieve = class
  protected
    FOpenMode : integer;
    FName  : string;
    FOwner : string;
    FStatus : integer;
    FPosBlock : TPosBlock;
    FActive : boolean;

    FRecSize : word;
    FVarRec : boolean;
    FKeyCount : integer;
    FRecordCount : longint;
    FFileVersion : integer;

    FClientID : TBtrClientID;
    FUseClientID : boolean;

    FKeyNumber : integer;
    FKey : TKeyBuff;
    FKeyLen : integer;

    FMultiLock : boolean;
    FLockType : integer;
    FTransactionType : integer;
    FUseChunkOperation : boolean;

    FData : pointer;

    FIgnoreException : boolean;

    FInternalPos : longint;

    function BtrieveCall (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                          var keybuf; KeyLen : byte; KeyNum : byte) : SmallInt;

    procedure CheckInactive;
    procedure CheckActive;
    procedure CheckSuccess;

    procedure GetVarRec (AVarRec : TStream);
    procedure UpdateChunk (AOffset : Longint; ARec : TStream; ATruncate : boolean);

    function GetPosition : longint;
    procedure SetPosition (APosition : longint);

    function GetRecord (OpType : integer; AVarRec : TStream; AErrorIgnore : TBtrErrorSet) : boolean;
    procedure ProcessRecord (OpType : integer; AVarRec : TStream);

    function GetKeyBuffer : pointer;
    function GetRecordCount : longint;
    procedure SetActive (AActive : boolean);
    function GetInternalLockType : integer;

    property InternalLockType : integer read GetInternalLockType;
  public
    constructor Create; virtual;
    constructor CreateClientID (AClientID : longint); virtual;
    constructor CreateByPID (APID : pointer); virtual;

    destructor Destroy; override;

    procedure Open; virtual;
    procedure Close; virtual;

    procedure BeginTransaction;
    procedure EndTransaction;
    procedure AbortTransaction;

    procedure CreateTable (var ADataBuf; ADataBufLen : TDataLen);
    procedure Reset;
    procedure SetOwner (AMode : byte);
    procedure ClearOwner;

    procedure BtrieveCallDirect (Operation : word; var DataBuf; var DataLen : TDataLen;
                                  var keybuf; KeyLen : byte; KeyNum : byte);

    procedure LoadFileStat;

    procedure DeleteRecord;

    function FirstRecord (AVarRec : TStream) : boolean;
    function NextRecord (AVarRec : TStream) : boolean;
    function PrevRecord (AVarRec : TStream) : boolean;
    function LastRecord (AVarRec : TStream) : boolean;

    function FindEqualRecord (AVarRec : TStream) : boolean;
    function FindGreaterRecord (AVarRec : TStream) : boolean;
    function FindGreaterOrEqualRecord (AVarRec : TStream) : boolean;
    function FindLessRecord (AVarRec : TStream) : boolean;
    function FindLessOrEqualRecord (AVarRec : TStream) : boolean;

    function StepFirstRecord (AVarRec : TStream) : boolean;
    function StepNextRecord (AVarRec : TStream) : boolean;
    function StepPrevRecord (AVarRec : TStream) : boolean;
    function StepLastRecord (AVarRec : TStream) : boolean;

    procedure UpdateRecord (AVarRec : TStream);
    procedure IncrementalUpdateRecord;

    procedure InsertRecord (AVarRec : TStream);

    procedure GetDirect (APos : LongInt; AVarRec : TStream);

    procedure Unlock;
    procedure UnlockAll;

    procedure StartContinuousOperation (AFileNames : string);
    procedure EndContinuousOperation (AFileNames : string);
    procedure EndAllContinuousOperation;

    function GetClientVersion (var VerData : TVersionDescriptor) : integer;
    function GetServerVersion (var VerData : TFullVersion) : integer;
    function GetServerVersionByName (ServerName : string; var VerData : TFullVersion) : integer;

    property Name  : string read FName write FName;
    property Owner : string read FOwner write FOwner;

    property Status : integer read FStatus;
    property Active : boolean read FActive write SetActive;
    property UseClientID : boolean read FUseClientID write FUseClientID;
    property ClientID : TBtrClientID read FClientID;

    property RecSize : word read FRecSize;
    property VarRec : boolean read FVarRec;
    property KeyCount : integer read FKeyCount;
    property FileVersion : integer read FFileVersion;

    property KeyNumber : integer read FKeyNumber write FKeyNumber;
    property Key : pointer read GetKeyBuffer;
    property KeyLen : integer read FKeyLen;

    property LockType : integer read FLockType write FLockType;
    property TransactionType : integer read FTransactionType write FTransactionType;

    property Data : pointer read FData;

    property Position : longint read GetPosition write SetPosition;
    property RecordCount : longint read GetRecordCount;

    property OpenMode : integer read FOpenMode write FOpenMode;

    property UseChunkOperation : boolean read FUseChunkOperation write FUseChunkOperation;
    property IgnoreException : boolean read FIgnoreException write FIgnoreException;
 end;

implementation

uses lBtrAPI, lBtrCnst, lBtrErr, lBtrVar, lBtrOS;

constructor TBtrieve.Create;
var
  ID : longint;
begin
  inherited;

  Name := '';
  Owner := '';
  OpenMode := BTR_OPEN_NORMAL;
  FActive := False;
  FStatus := BTR_SUCCESS;
  FMultiLock := False;
  FLockType := BTR_NO_LOCK_BIAS;
  FTransactionType := BTR_EXCLUSIVE_BIAS;
  FRecSize := 0;
  FKeyCount := 0;
  FVarRec := False;
  FData := Nil;
  FKeyLen := 255;
  FUseChunkOperation := UseChunkOperationDef;
  FUseClientID := UseClientIDDef;
  FIgnoreException := IgnoreExceptionDef;
  FInternalPos := 0;

  ID := lGetCurrentThreadID;

  FillChar (FClientID, sizeof (FClientID), #0);
  FClientID.applicationID [0] := 'L';
  FClientID.applicationID [1] := 'A';
  FClientID.threadID := word (ID); // use only 2 low byte
end;

constructor TBtrieve.CreateClientID;
begin
  Create;

  FClientID.FullAppID := AClientID;
  FUseClientID := True;
end;

constructor TBtrieve.CreateByPID;
begin
  Create;

  Move (APID^, FClientID, sizeof (FClientID));
  FUseClientID := True;
end;

destructor TBtrieve.Destroy;
begin
  if FActive then
  begin
    try
      Close;
    except
    end;
  end;

  if FData <> Nil then
    ReallocMem (FData, 0);

  inherited;
end;

function TBtrieve.GetKeyBuffer;
begin
  Result := @FKey;
end;

function TBtrieve.GetInternalLockType;
begin
  Result := LockType;
  if (LockType = BTR_MULTIPLE_WAIT_LOCK_BIAS) or (LockType = BTR_MULTIPLE_NOWAIT_LOCK_BIAS) then
    FMultiLock := True;
  LockType := BTR_NO_LOCK_BIAS;
end;

function TBtrieve.GetRecordCount;
begin
  LoadFileStat;
  Result := FRecordCount;
end;

function TBtrieve.BtrieveCall;
begin
 if FUseClientID
   then Result := BtrCallID (Operation, PosBlock, DataBuf, DataLen, keybuf, KeyLen, KeyNum, FClientID)
   else Result := BtrCall (Operation, PosBlock, DataBuf, DataLen, keybuf, KeyLen, KeyNum);
end;

procedure TBtrieve.BtrieveCallDirect;
begin
  FStatus := BtrieveCall (Operation, FPosBlock, DataBuf, DataLen, keybuf, KeyLen, KeyNum);
  CheckSuccess;
end;

procedure TBtrieve.CheckActive;
begin
  if not FActive then
    raise EBtrieveException.Create (FName, BTR_FILE_NOT_OPEN);
end;

procedure TBtrieve.CheckInactive;
begin
  if FActive then
    raise EBtrieveException.Create (FName, BTR_FILE_IN_USE);
end;

procedure TBtrieve.CheckSuccess;
begin
  if Status <> BTR_SUCCESS then
    raise EBtrieveException.Create (FName, Status);
end;

procedure TBtrieve.Reset;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  FillChar (FKey, sizeof (TKeyBuff), #0);
  DataLen := 0;
  FStatus := BtrieveCall (BTR_RESET, FPosBlock, Dummy, DataLen, FKey, KeyLen, 0);
  CheckSuccess;

  FActive := False;
  ReallocMem (FData, 0);
end;

procedure TBtrieve.SetOwner;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
  s : string [8];
begin
  CheckActive;
  FillChar (FKey, sizeof (TKeyBuff), #0);
  FillChar (Dummy, sizeof (Dummy), #0);
  DataLen := 9;
  s := Copy (Owner, 1, 8);
  while Length (s) < 8 do
    s := s+' ';
  Move (s [1], FKey, 8);
  Move (s [1], Dummy, 8);
  FStatus := BtrieveCall (BTR_SET_OWNER, FPosBlock, Dummy, DataLen, FKey, KeyLen, AMode);
  CheckSuccess;
end;

procedure TBtrieve.ClearOwner;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  CheckActive;
  DataLen := 0;
  FillChar (Dummy, sizeof (Dummy), #0);
  FStatus := BtrieveCall (BTR_CLEAR_OWNER, FPosBlock, Dummy, DataLen, FKey, 0, 0);
  CheckSuccess;
end;

procedure TBtrieve.Open;
var
  DataLen : TDataLen;
  AOwner : array [0..255] of Char;
  AName : array [0..255] of Char;
begin
  if Active then Exit;

  FillChar (AOwner, SizeOf (AOwner), #0);
  FillChar (AName, SizeOf (AName), #0);

  StrPLCopy (AOwner, FOwner, sizeof (AOwner)-1);
  StrPLCopy (AName, FName, sizeof (AName)-1);
  DataLen := StrLen (AOwner) + 1;

  FStatus := BtrieveCall (BTR_OPEN, FPosBlock, AOwner, DataLen,
                          AName, 255,
                          Byte (OpenMode+BTR_OPEN_SINGLE_ENGINE_FILE_SHARING));
  CheckSuccess;
  FActive := True;

  LoadFileStat;

  ReallocMem (FData, FRecSize);
end;

procedure TBtrieve.Close;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  if not FActive then Exit;

  DataLen := 0;
  FillChar (Dummy, sizeof (Dummy), #0);
  FStatus := BtrieveCall (BTR_CLOSE, FPosBlock, Dummy, DataLen, FKey, 0, 0);
  CheckSuccess;
  FActive := False;
  ReallocMem (FData, 0);
end;

procedure TBtrieve.LoadFileStat;
var
  FileStat : TBtrieveAlternateFileStat;
  DataLen : TDataLen;
  LocalKey : TKeyBuff;

begin
  CheckActive;
  FillChar (FileStat, sizeof (FileStat), #0);
  FillChar (LocalKey, sizeof (LocalKey), #0);
  DataLen := sizeof (FileStat);
  FStatus := BtrieveCall (BTR_STAT, FPosBlock, FileStat, DataLen, LocalKey, sizeof (LocalKey), byte (-1));
  CheckSuccess;
  FRecSize := FileStat.RecordLen;
  FVarRec := (FileStat.FileFlag and 1) <> 0;
  FRecordCount := FileStat.NumOfRec;
  FKeyCount := FileStat.NumOfIdx;
  FFileVersion := FileStat.FileVer;
end;

procedure TBtrieve.BeginTransaction;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  DataLen := 0;
  FillChar (Dummy, sizeof (Dummy), #0);
  FStatus := BtrieveCall (BTR_BEGIN_TRANSACTION + TransactionType + InternalLockType, FPosBlock, Dummy, DataLen, FKey, 0, 0);
  CheckSuccess;
end;

procedure TBtrieve.EndTransaction;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  DataLen := 0;
  FillChar (Dummy, sizeof (Dummy), #0);
  FStatus := BtrieveCall (BTR_END_TRANSACTION, FPosBlock, Dummy, DataLen, FKey, 0, 0);
  CheckSuccess;
end;

procedure TBtrieve.AbortTransaction;
var
  DataLen : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  DataLen := 0;
  FillChar (Dummy, sizeof (Dummy), #0);
  FStatus := BtrieveCall (BTR_ABORT_TRANSACTION, FPosBlock, Dummy, DataLen, FKey, 0, 0);
  CheckSuccess;
end;

procedure TBtrieve.Unlock;
var
  DataLen : TDataLen;
  Pos : longint;
begin
  CheckActive;
  DataLen := sizeof (Pos);
  Pos := Position;
  if FMultiLock then
    FStatus := BtrieveCall (BTR_UNLOCK, FPosBlock, Pos, DataLen, FKey, 0, byte (-1))
  else
    FStatus := BtrieveCall (BTR_UNLOCK, FPosBlock, Pos, DataLen, FKey, 0, 0);
//  FMultiLock := False;
  CheckSuccess;
end;

procedure TBtrieve.UnlockAll;
var
  DataLen : TDataLen;
  Pos : longint;
begin
  CheckActive;
  DataLen := sizeof (Pos);
  Pos := Position;
  if FMultiLock then
    FStatus := BtrieveCall (BTR_UNLOCK, FPosBlock, Pos, DataLen, FKey, 0, byte (-2))
  else
    FStatus := BtrieveCall (BTR_UNLOCK, FPosBlock, Pos, DataLen, FKey, 0, 0);

  FMultiLock := False;
  CheckSuccess;
end;

procedure TBtrieve.CreateTable;
var
  DataLen : TDataLen;
  AName : array [0..255] of Char;
begin
  CheckInactive;
  FillChar (AName, SizeOf (AName), #0);
  StrPLCopy (AName, FName, sizeof (AName)-1);

  DataLen := ADataBufLen;
  FStatus := BtrieveCall (BTR_CREATE, FPosBlock, ADataBuf, DataLen, AName, 255, FKeyNumber);
  CheckSuccess;
end;

function TBtrieve.GetRecord;
var
  DataLen : TDataLen;
  pData : pointer;
  tmpBuff : longint;
begin
  Result := True;
  CheckActive;

  if UseChunkOperation then
  begin
    if (OpType = BTR_GET_DIRECT) and (FRecSize < sizeof (FInternalPos)) then
    begin
      pData := @tmpBuff;
      DataLen := sizeof (FInternalPos);
    end
    else
    begin
      pData := FData;
      DataLen := FRecSize;
    end;
    if OpType = BTR_GET_DIRECT then
      Move (FInternalPos, pData^, sizeof (FInternalPos));

    FStatus := BtrieveCall (OpType + InternalLockType, FPosBlock, pData^, DataLen, FKey, FKeyLen, FKeyNumber);
    if IgnoreException then
      if FStatus in AErrorIgnore then
      begin
        FStatus := BTR_SUCCESS;
        Result := False;
        Exit;
      end;
    if VarRec then
      if (FStatus = BTR_WRONGBUFFERSIZE) or (DataLen > FRecSize) then
      begin
        FStatus := BTR_SUCCESS;
        GetVarRec (AVarRec);
      end;
    CheckSuccess;
    if pData <> FData then
      Move (pData^, FData^, FRecSize);
  end
  else
  begin
    if FRecSize > BlockOperationSize then
    begin
      FStatus := BTR_WRONGBUFFERSIZE;
      CheckSuccess;
    end;

    if (OpType = BTR_GET_DIRECT) and (BlockOperationSize < sizeof (FInternalPos)) then
    begin
      FStatus := BTR_WRONGBUFFERSIZE;
      CheckSuccess;
    end;
    
    GetMem (pData, BlockOperationSize);
    try
      FillChar (pData^, BlockOperationSize, 0);
//      Move (FData^, pData^, FRecSize);
      if OpType = BTR_GET_DIRECT then
        Move (FInternalPos, pData^, sizeof (FInternalPos));
      DataLen := BlockOperationSize;
      FStatus := BtrieveCall (OpType+InternalLockType, FPosBlock, pData^, DataLen, FKey, FKeyLen, FKeyNumber);
      if IgnoreException then
        if FStatus in AErrorIgnore then
        begin
          FStatus := BTR_SUCCESS;
          Result := False;
          Exit;
        end;
      CheckSuccess;
      Move (pData^, FData^, FRecSize);
      if VarRec then
        if Assigned (AVarRec) then
          AVarRec.Write ((PChar (pData)+FRecSize)^, DataLen - FRecSize);
    finally
      FreeMem (pData, BlockOperationSize);
    end;
  end;
end;

procedure TBtrieve.GetDirect;
begin
  FInternalPos := APos;
  GetRecord (BTR_GET_DIRECT, AVarRec, []);
end;

function TBtrieve.FirstRecord;
begin
  Result := GetRecord (BTR_GET_FIRST, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.NextRecord;
begin
  Result := GetRecord (BTR_GET_NEXT, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.PrevRecord;
begin
  Result := GetRecord (BTR_GET_PREVIOUS, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.LastRecord;
begin
  Result := GetRecord (BTR_GET_LAST, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.FindEqualRecord;
begin
  Result := GetRecord (BTR_GET_EQUAL, AVarRec, [BTR_KEYNOTFOUND]);
end;

function TBtrieve.FindGreaterRecord;
begin
  Result := GetRecord (BTR_GET_GREATER, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.FindGreaterOrEqualRecord;
begin
  Result := GetRecord (BTR_GET_GREATER_OR_EQUAL, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.FindLessRecord;
begin
  Result := GetRecord (BTR_GET_LESS_THAN, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.FindLessOrEqualRecord;
begin
  Result := GetRecord (BTR_GET_LESS_THAN_OR_EQUAL, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.StepFirstRecord;
begin
  Result := GetRecord (BTR_STEP_FIRST, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.StepNextRecord;
begin
  Result := GetRecord (BTR_STEP_NEXT, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.StepPrevRecord;
begin
  Result := GetRecord (BTR_STEP_PREVIOUS, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.StepLastRecord;
begin
  Result := GetRecord (BTR_STEP_LAST, AVarRec, [BTR_ENDOFFILE]);
end;

function TBtrieve.GetPosition;
var
  DataLen    : TDataLen;
begin
  CheckActive;
  DataLen := sizeof (Result);
  FillChar (Result, sizeof (Result), #0);
  FStatus := BtrieveCall (BTR_GET_POSITION, FPosBlock, Result, DataLen, FKey, KeyLen, KeyNumber);
  CheckSuccess;
end;

procedure TBtrieve.DeleteRecord;
var
  DataLen    : TDataLen;
  Dummy : array [0..1023] of byte;
begin
  CheckActive;
  DataLen := 0;
  FillChar (Dummy, sizeof (Dummy), #0);
  FStatus := BtrieveCall (BTR_DELETE, FPosBlock, Dummy, DataLen, FKey, KeyLen, KeyNumber);
  CheckSuccess;
end;

procedure TBtrieve.ProcessRecord (OpType : integer; AVarRec : TStream);
var
  ms : TMemoryStream;
  DataLen : TDataLen;
  pData : pointer;
  len : integer;
begin
  CheckActive;
  if UseChunkOperation and VarRec then
  begin
    if OpType = BTR_UPDATE then
    begin
      ms := TMemoryStream.Create;
      try
        ms.Write (Data^, RecSize);
        ms.Seek (0, soFromBeginning);
        UpdateChunk (0, ms, True);
      finally
        ms.Free;
      end;
    end
    else
    begin
      DataLen := FRecSize;
      FStatus := BtrieveCall (OpType, FPosBlock, FData^, DataLen, FKey, FKeyLen, FKeyNumber);
      CheckSuccess;
    end;
    if VarRec then
      UpdateChunk (RecSize, AVarRec, True);
  end
  else
  begin
    if FRecSize > BlockOperationSize then
    begin
      FStatus := BTR_WRONGBUFFERSIZE;
      CheckSuccess;
    end;
    GetMem (pData, BlockOperationSize);
    try
      Move (FData^, pData^, FRecSize);
      len := 0;
      if VarRec then
        if Assigned (AVarRec) then
        begin
          len := AVarRec.Size - AVarRec.Position;
          if len > (BlockOperationSize-FRecSize) then
          begin
            FStatus := BTR_WRONGBUFFERSIZE;
            CheckSuccess;
//            len := BlockOperationSize-FRecSize;
          end;
          AVarRec.Read ((PChar (pData)+FRecSize)^, len);
        end;
      DataLen := len + FRecSize;
      FStatus := BtrieveCall (OpType, FPosBlock, pData^, DataLen, FKey, FKeyLen, FKeyNumber);
      CheckSuccess;
    finally
      FreeMem (pData, BlockOperationSize);
    end;
  end;
end;

procedure TBtrieve.InsertRecord (AVarRec : TStream);
begin
  ProcessRecord (BTR_INSERT, AVarRec);
end;

procedure TBtrieve.UpdateRecord (AVarRec : TStream);
begin
  ProcessRecord (BTR_UPDATE, AVarRec);
end;

procedure TBtrieve.GetVarRec;
Type
  TDataBuffer = packed record
   Position : LongInt;
   SubFunction : LongInt;
   NumChunks : LongInt;
    ChunkOffset : LongInt;
    ChunkLen    : LongInt;
    UsersData   : LongInt;
  end;

var
  DataLen    : TDataLen;
  Data : pointer;
  DataMemLen : LongInt;
  Offset : integer;

begin
  CheckActive;
  if AVarRec = nil then Exit;

  DataMemLen := ChunkBlockOperationSize;
  if DataMemLen < sizeof (TDataBuffer) then
    DataMemLen := sizeof (TDataBuffer);

  GetMem (Data, DataMemLen);
  try
    Offset := FRecSize;

    repeat
      TDataBuffer (Data^).Position := Position;
      TDataBuffer (Data^).SubFunction := LongInt ($80000000);
      TDataBuffer (Data^).NumChunks := 1;
      TDataBuffer (Data^).ChunkLen    := ChunkBlockOperationSize;
      TDataBuffer (Data^).UsersData   := Integer (Data);
      TDataBuffer (Data^).ChunkOffset := Offset;

      DataLen := DataMemLen;
      FStatus := BtrieveCall (BTR_GET_DIRECT, FPosBlock, Data^, DataLen, FKey, 0, $FE);
      AVarRec.Write (Data^, DataLen);
      inc (Offset, DataLen);
    until (FStatus <> BTR_SUCCESS) or (DataLen <> ChunkBlockOperationSize);
    if FStatus = BTR_CHUNK_OFFSET_TOO_BIG then
      FStatus := BTR_SUCCESS;
  finally
    FreeMem (Data, DataMemLen);
  end;
  CheckSuccess;
end;

procedure TBtrieve.UpdateChunk;
Type
  TUpdateChunk = packed record
   SubFunction : LongInt;
   NumChunks : LongInt;
    ChunkOffset : LongInt;
    ChunkLen    : LongInt;
    UsersData   : LongInt;

   Data0 : packed record end;
  end;

var
  Buffer : pointer;
  DataLen    : TDataLen;
  WriteDataOff : integer;
  WriteDataLen : integer;
  OffsetInFile : integer;

  DBTrunk : packed record
   Subfunction : LongInt;
   ChunkOffset : LongInt;
  end;

begin
  CheckActive;
  if ARec = nil then Exit;

  GetMem (Buffer, sizeof (TUpdateChunk) + ChunkBlockOperationSize);
  try
    WriteDataOff := ARec.Position;
    OffsetInFile := AOffset;
    WriteDataLen := ChunkBlockOperationSize;
    repeat
      if (WriteDataLen+WriteDataOff) > ARec.Size
        then WriteDataLen := ARec.Size-WriteDataOff;

//      DataLen := sizeof (TUpdateChunk) + ChunkBlockOperationSize;
      DataLen := sizeof (TUpdateChunk) + WriteDataLen;
      ARec.Read (TUpdateChunk (Buffer^).Data0, WriteDataLen);
      TUpdateChunk (Buffer^).SubFunction := LongInt ($80000000);
      TUpdateChunk (Buffer^).NumChunks := 1;
      TUpdateChunk (Buffer^).ChunkOffset := OffsetInFile;
      TUpdateChunk (Buffer^).ChunkLen    := WriteDataLen;
//      TUpdateChunk (Buffer^).UsersData   := LongInt (Buffer);

      inc (WriteDataOff, WriteDataLen);
      inc (OffsetInFile, WriteDataLen);
      FStatus := BtrieveCall (BTR_UPDATE_CHUNK, FPosBlock, Buffer^, DataLen, FKey, FKeyLen, FKeyNumber);
      CheckSuccess;
    until WriteDataOff = ARec.Size;

    if ATruncate then
    begin
      DataLen := sizeof (DBTrunk);
      with DBTrunk do
      begin
        SubFunction := LongInt ($80000004);
        ChunkOffset := OffsetInFile;
      end;
      FStatus := BtrieveCall (BTR_UPDATE_CHUNK, FPosBlock, DBTrunk, DataLen, FKey, FKeyLen, FKeyNumber);
      if FStatus = BTR_CHUNK_OFFSET_TOO_BIG then
        FStatus := BTR_SUCCESS;
    end;
  finally
    FreeMem (Buffer, sizeof (TUpdateChunk) + ChunkBlockOperationSize);
  end;
  CheckSuccess;
end;

procedure TBtrieve.IncrementalUpdateRecord;
var
  ms : TMemoryStream;
begin
  ms := TMemoryStream.Create;
  try
    ms.Write (Data^, RecSize);
    ms.Seek (0, soFromBeginning);
    UpdateChunk (0, ms, False);
  finally
    ms.Free;
  end;
end;

procedure TBtrieve.SetActive;
begin
  if FActive = AActive then Exit;
  if AActive then 
    Open
  else 
    Close;
end;

procedure TBtrieve.SetPosition;
begin
  GetDirect (APosition, nil);
end;

procedure TBtrieve.StartContinuousOperation (AFileNames : string);
var
  PosBlock : TPosBlock;
  Data : pointer;
  Key : TKeyBuff;
  KeyLen : integer;
  Len : TDataLen;
  DataLen : TDataLen;
begin
  FillChar (PosBlock, sizeof (PosBlock), 0);
  Len := length (AFileNames)+1;
  GetMem (Data, Len);
  try
    KeyLen := sizeof (Key);
    FillChar (Key, sizeof (Key), 0);
    DataLen := Len;
    FillChar (Data^, Len, 0);
    if AFileNames <> '' then
      Move (AFileNames [1], Data^, Len-1);
    FStatus := BtrieveCall (BTR_CONTINUOUS_OPERATION, PosBlock, Data^, DataLen, Key, KeyLen, 0);
    CheckSuccess;
  finally
    FreeMem (Data, Len);
  end;
end;

procedure TBtrieve.EndContinuousOperation (AFileNames : string);
var
  PosBlock : TPosBlock;
  Data : pointer;
  Key : TKeyBuff;
  KeyLen : integer;
  Len : TDataLen;
  DataLen : TDataLen;
begin
  FillChar (PosBlock, sizeof (PosBlock), 0);
  Len := length (AFileNames)+1;
  GetMem (Data, Len);
  try
    KeyLen := sizeof (Key);
    FillChar (Key, sizeof (Key), 0);
    DataLen := Len;
    FillChar (Data^, Len, 0);
    if AFileNames <> '' then
      Move (AFileNames [1], Data^, Len-1);
    FStatus := BtrieveCall (BTR_CONTINUOUS_OPERATION, PosBlock, Data^, DataLen, Key, KeyLen, 2);
    CheckSuccess;
  finally
    FreeMem (Data, Len);
  end;
end;

procedure TBtrieve.EndAllContinuousOperation;
var
  PosBlock : TPosBlock;
  Data : integer;
  Key : TKeyBuff;
  KeyLen : integer;
  DataLen : TDataLen;
begin
  FillChar (PosBlock, sizeof (PosBlock), 0);
  DataLen := 0;
  FillChar (Data, sizeof (Data), 0);
  KeyLen := sizeof (Key);
  FillChar (Key, sizeof (Key), 0);
  FStatus := BtrieveCall (BTR_CONTINUOUS_OPERATION, PosBlock, Data, DataLen, Key, KeyLen, 1);
  CheckSuccess;
end;

function TBtrieve.GetClientVersion (var VerData : TVersionDescriptor) : integer;
var
  PosBlock : TPosBlock;
  Key : TKeyBuff;
  KeyLen : integer;
  DataLen : TDataLen;
begin
  DataLen := sizeof (VerData);
  FillChar (PosBlock, sizeof (PosBlock), 0);
  FillChar (Key, sizeof (Key), 0);
  KeyLen := sizeof (Key);
  FStatus := BtrieveCall (BTR_VERSION, PosBlock, VerData, DataLen, Key, KeyLen, 0);
  CheckSuccess;

  Result := DataLen div sizeof (TVersionDescriptor);
end;

function TBtrieve.GetServerVersion (var VerData : TFullVersion) : integer;
var
  Key : TKeyBuff;
  KeyLen : integer;
  DataLen : TDataLen;
begin
  DataLen := sizeof (VerData);
  FillChar (Key, sizeof (Key), 0);
  KeyLen := sizeof (Key);
  FStatus := BtrieveCall (BTR_VERSION, FPosBlock, VerData, DataLen, Key, KeyLen, 0);
  CheckSuccess;

  Result := DataLen div sizeof (TVersionDescriptor);
end;

function TBtrieve.GetServerVersionByName (ServerName : string; var VerData : TFullVersion) : integer;
var
  Key : TKeyBuff;
  KeyLen : integer;
  DataLen : TDataLen;
  PosBlock : TPosBlock;
begin
  DataLen := sizeof (VerData);
  FillChar (PosBlock, sizeof (PosBlock), 0);

  FillChar (Key, sizeof (Key), 0);
  if ServerName <> '' then
    Move (ServerName [1], Key, Length (ServerName));
  KeyLen := sizeof (Key);

  FStatus := BtrieveCall (BTR_VERSION, PosBlock, VerData, DataLen, Key, KeyLen, 0);
  CheckSuccess;

  Result := DataLen div sizeof (TVersionDescriptor);
end;

end.

