//*****************************************************************************
//
//    System          : BtrUtils
//    Category        : Btrive API declaration
//    Source          : Btrieve API programmers guide
//
//    Copyright (c), Leonov Alexander
//
//    File version $Revision:1.3$ at $Date:28.01.2004 19:32:30$
//
//    Author : Leonov Alexander
//    web    : http://dev.zibn.net/     mirror at:  http://home.ural.ru/~tigra/
//    e-mail : dev@zibn.net
//
//*****************************************************************************

unit lBtrAPI;

{$INCLUDE lBtrDef.inc}

interface

 Uses lBtrType, lBtrCnst;

 Type
   TLoadBtrieveResult = (loadBtrieveSuccess, loadBtrieveAlreadyLoad, loadBtrieveFailedLoad);

 function IsBtrieveLoad : boolean;
 function TryToLoadBtrieve : TLoadBtrieveResult;
 procedure UnloadBtrieve;

 function BtrCall (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                   var keybuf; KeyLen : byte; KeyNum : byte) : SmallInt;

 function BtrCallID (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                     var keybuf; KeyLen : byte; KeyNum : byte; var ClientID) : SmallInt;

implementation

uses SyncObjs, SysUtils {$IFDEF OS_WIN}, Windows {$ENDIF};

Const
 {$IFDEF OS_WIN}
   BtrieveDLL = 'WBTRV32.DLL';
   BtrCallName = 'BTRCALL';
   BtrCallIDName = 'BTRCALLID';
 {$ENDIF}
 {$IFDEF OS_UNIX}
   BtrieveDLL = 'libbtrvif.so';
   BtrCallName = 'BTRV';
   BtrCallIDName = 'BTRVID';
 {$ENDIF}

var
  loadCS : TCriticalSection;
  bBtrieveLoad : boolean;

  libBtrieve : hModule;
  procBtrv : pointer;
  procBtrvID : pointer;

function IsBtrieveLoad : boolean;
begin
  Result := bBtrieveLoad;
end;

function TryToLoadBtrieve : TLoadBtrieveResult;
begin
  Result := loadBtrieveAlreadyLoad;
  if bBtrieveLoad then Exit;

  loadCS.Enter;
  try
    if not bBtrieveLoad then
    begin
      Result := loadBtrieveFailedLoad;
      libBtrieve := SafeLoadLibrary (BtrieveDLL);
      if libBtrieve = 0 then Exit;

      procBtrv := GetProcAddress (libBtrieve, BtrCallName);
      procBtrvID := GetProcAddress (libBtrieve, BtrCallIDName);

      if (@procBtrv = nil) or (@procBtrvID = nil) then
      begin
        FreeLibrary(libBtrieve);
        libBtrieve := 0;

        procBtrv := nil;
        procBtrvID := nil;
      end
      else
      begin
        bBtrieveLoad := True;
        Result := loadBtrieveSuccess;
      end;
    end;
  finally
    loadCS.Leave;
  end;
end;

procedure UnloadBtrieve;
begin
  if not bBtrieveLoad then Exit;

  loadCS.Enter;
  try
    if bBtrieveLoad then
    begin
      FreeLibrary(libBtrieve);
      libBtrieve := 0;

      procBtrv := nil;
      procBtrvID := nil;
      bBtrieveLoad := False;
    end;
  finally
    loadCS.Leave;
  end;
end;

{$IFDEF OS_WIN}
Type
  TBTRVProc = function (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                  var keybuf; KeyLen : byte; KeyNum : byte) : SmallInt; stdcall;
  TBTRVIDProc = function (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                  var keybuf; KeyLen : byte; KeyNum : byte; var ClientID) : SmallInt; stdcall;
{$ENDIF}
{$IFDEF OS_UNIX}
Type
  TBTRVProc = function (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                  var keybuf; KeyNum : byte) : SmallInt; cdecl;
  TBTRVIDProc = function (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                  var keybuf; KeyNum : byte; var ClientID) : SmallInt; cdecl;
{$ENDIF}

function BtrCall (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                  var keybuf; KeyLen : byte; KeyNum : byte) : SmallInt;
begin
  if TryToLoadBtrieve in [loadBtrieveSuccess, loadBtrieveAlreadyLoad] then
    Result := TBTRVProc (procBtrv) (Operation, PosBlock, DataBuf, DataLen, keybuf,
      {$IFDEF OS_WIN} KeyLen,{$ENDIF} KeyNum)
  else
    Result := BTR_RECORD_MGR_INACTIVE;
end;

function BtrCallID (Operation : word; var PosBlock; var DataBuf; var DataLen : TDataLen;
                    var keybuf; KeyLen : byte; KeyNum : byte; var ClientID) : SmallInt;
begin
  if TryToLoadBtrieve in [loadBtrieveSuccess, loadBtrieveAlreadyLoad] then
    Result := TBTRVIDProc (procBtrvID) (Operation, PosBlock, DataBuf, DataLen, keybuf,
      {$IFDEF OS_WIN} KeyLen,{$ENDIF} KeyNum, ClientID)
  else
    Result := BTR_RECORD_MGR_INACTIVE;
end;

initialization
  loadCS := TCriticalSection.Create;
  bBtrieveLoad := False;
finalization
  loadCS.Free;
end.

