{ ==============================================================================
  TBoCustomEdit
  --> TBoEdit
  --> TBoCustomDBEdit
      --> TBoDBEdit

  Author :   Boffi

 ============================================================================== }

unit BoEdit;

interface

{$include BoGlobal.inc}
{$R BoEdit.Res}

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Mask, Buttons, comctrls, extCtrls, menus, db, DBCtrls,
  BoButton;

const
  BoDefaultWidth = 20;

type
  TBoCustomEdit = class;
  TBoEditButton = class;
  TBoEditButtonClass = class of TBOEditButton;
  TBoEditButtonPosition = (ebp_left, ebp_right);
  TBoEditButtonType = (ebt_normal, ebt_checked);
  TBoEditInputType = (eit_normal, eit_integer, eit_float);
  TBoEditButtons = class;
  TBoButtonClickEvent = procedure ( sender : TObject; aButton : TBoEditButton ) of Object;
  TBoHidePopupEvent = procedure ( sender : TObject; UseNewValue : Boolean ) of Object;

//==============================================================================
  TBoPopupBase = class( TCustomControl )
  private
    fBoOnHidePopup: TBoHidePopupEvent;
    procedure   WMMouseActivate(var Message: TMessage); message WM_MOUSEACTIVATE;
  protected
    BoParentControl : TWinControl;
    procedure   CreateParams(var Params: TCreateParams); override;
    function    BoGetText  : String;  virtual;
    function    BoGetId    : Variant; virtual;
  public
    constructor Create(AOwner: TComponent); override;
    procedure   BoHidePopup(UseNewValue : Boolean);
    procedure   BoShowPopup(Where: TPoint);
    property    BoOnHidePopup : TBoHidePopupEvent read fBoOnHidePopup write fBoOnHidePopup;
  end;

//==============================================================================
  TBOEditButton = Class(TCollectionItem)
  private
    FBoButtons : TBOEditButtons;
    FBoCustomEdit : TBoCustomEdit;
    FBoOnclick : TNotifyEvent;
    FBoWidth : Integer;
    FBoPosition : TBoEditButtonPosition;
    FBoIdName : String;
    FBoShortCut: TShortCut;
    FBoDefault: Boolean;
    FBoEnabled: Boolean;
    FBoButtonType: TBoEditButtonType;
    FBoVisible: Boolean;
    FBoAutoHide: Boolean;
    FBoEditFocused: Boolean;
    procedure   SetBoWidth(const Value: Integer);
    procedure   SetBoPosition(const Value: TBoEditButtonPosition);
    procedure   SetBoIdName(const Value: String);
    procedure   SetBoCaption(const Value: String);
    procedure   OnClickHandler(sender : TObject);
    procedure   SetBoGlyph(const Value: TBitmap);
    procedure   SetBoNumGlyphs(const Value: Integer);
    procedure   SetBoResBmpName(const Value: String);
    procedure   SetBoShortCut(const Value: TShortCut);
    procedure   SetBoDefault(const Value: Boolean);
    procedure   SetBoEnabled(const Value: Boolean);
    procedure   SetBoHint(const Value: String);
    procedure   SetBoGroupIndex(const Value: Integer);
    function    GetBoCaption: String;
    function    GetBoGlyph: TBitmap;
    function    GetBoNumGlyphs: Integer;
    function    GetBoGroupIndex: Integer;
    function    GetBoResBmpName: String;
    function    GetBoHint: String;
    function    BoGlyphStored : boolean;
    function    GetBoDown: Boolean;
    procedure   SetBoDown(const Value: Boolean);
    function    GetBoAllowAllUp: Boolean;
    procedure   SetBoAllowAllUp(const Value: Boolean);
    procedure   SetBoButtonType(const Value: TBoEditButtonType);
    function    GetBoChecked: Boolean;
    procedure   SetBoChecked(const Value: Boolean);
    procedure   SetBoVisible(const Value: Boolean);
    procedure   SetBoAutoHide(const Value: Boolean);
    function    GetBoButtonCurrentlyVisible: Boolean;
    procedure   SetBoEditFocused(const Value: Boolean);
  protected
    fBoSpeedButton : TBoSpeedbutton;
  public
    constructor Create(Collection: TCollection); override;
    destructor  Destroy; override;
    function    GetdisplayName: String; override;
    procedure   BOEnableDisableButton( blnEnabled : Boolean );
    procedure   BoSetBounds(aLeft, aTop, aWidth, aHeight : Integer; aParent : TWinControl);
    procedure   BoDoClick;
    property    BoButtonCurrentlyVisible : Boolean read GetBoButtonCurrentlyVisible;
    property    BoEditFocused : Boolean read FBoEditFocused write SetBoEditFocused;
  published
    Property    BoGroupIndex : Integer read GetBoGroupIndex write SetBoGroupIndex default 0;
    property    BoIdName : String read FBoIdName write SetBoIdName;
    property    BoOnClick : TNotifyEvent read fBoOnclick write fBoOnClick;
    property    BoWidth : Integer read FBoWidth write SetBoWidth default BoDefaultWidth;
    property    BoPosition : TBoEditButtonPosition read FBoPosition write SetBoPosition default ebp_right;
    property    BoCaption : String read GetBoCaption write SetBoCaption;
    property    BoGlyph : TBitmap read GetBoGlyph write SetBoGlyph stored BoGlyphStored;
    property    BoNumGlyphs : Integer read GetBoNumGlyphs write SetBoNumGlyphs default 2;
    property    BoResBmpName : String read GetBoResBmpName write SetBoResBmpName;
    property    BoShortCut : TShortCut read FBoShortCut write SetBoShortCut default 0;
    property    BoDefault : Boolean read FBoDefault write SetBoDefault default false;
    property    BoEnabled : Boolean read FBoEnabled write SetBoEnabled default true;
    property    BoHint : String read GetBoHint write SetBoHint;
    property    BoDown : Boolean read GetBoDown write SetBoDown default false;
    property    BoAllowAllUp: Boolean read GetBoAllowAllUp write SetBoAllowAllUp default false;
    property    BoButtonType : TBoEditButtonType read FBoButtonType write SetBoButtonType default ebt_normal;
    property    BoChecked : Boolean read GetBoChecked write SetBoChecked default false;
    property    BoVisible : Boolean read FBoVisible write SetBoVisible default true;
    property    BoAutoHide : Boolean read FBoAutoHide write SetBoAutoHide default false;
  end;

//==============================================================================
  TBOEditButtons = Class(TCollection)
  private
    FBoCustomEdit : TBoCustomEdit;
    function    GetBOEditButton(Index: Integer): TBOEditButton;
    procedure   SetBOEditButton(Index: Integer; const Value: TBOEditButton);
  protected
    function    GetOwner: TPersistent; override;
    procedure   Update(Item: TCollectionItem); override;
  public
    constructor Create(BoCustomEdit: TBoCustomEdit; BOEditButtonClass: TBOEditButtonClass);
    destructor  Destroy; override;
    function    BoGetWidthForPosition( aPosition : TBoEditButtonPosition;
                                       CheckVisible : Boolean ) : integer;
    function    Add: TBOEditButton;
    procedure   BOEnableDisableButtons( blnEnabled : Boolean );
    property    Items[Index: Integer]: TBOEditButton read GetBOEditButton write SetBOEditButton; default;
  published
  end;

//==============================================================================
  TBoCustomEdit = class(TCustomMaskEdit)
  private
    FBoButtons: TBOEditButtons;
    FBOFixedButtons: TBOEditButtons;
    FBoFixedButtonsBeforeCustom: boolean;
    FBoLeftPanel : TPanel;
    FBoRightPanel : TPanel;
    FBOFlat: Boolean;
    FBoButtonOnClick: TBoButtonClickEvent;
    FBoAlignment: TAlignment;
    FCanvas: TControlCanvas;
    FBoOnPopupClosed: TBoHidePopupEvent;
    FBoEditInputType: TBoEditInputType;
    procedure   SeTBOEditButtons( const Value: TBOEditButtons);
    procedure   WMSize(var message: TWMSize); message WM_SIZE;
    procedure   SetBoFixedEditButtonCollection(const Value: TBOEditButtons);
    procedure   SetBoFixedButtonsBeforeCustom(const Value: boolean);
    procedure   SetBOFlat(const Value: Boolean);
    function    GetCtl3d: Boolean;
    procedure   SetCtl3d(const Value: Boolean);
    function    GetBoButtonByIdname(IdName: String): TBOEditButton;
    function    GetBoChecked: Boolean;
    procedure   SetBoChecked(const Value: Boolean);
    procedure   CMCancelMode(var Message: TCMCancelMode); message CM_CANCELMODE;
    procedure   HidePopupWindowEvent( sender : TObject; UseNewValue : Boolean );
    procedure   InternalKeyDown(var Key: Word; Shift: TShiftState);
    procedure   SetBoAlignment(const Value: TAlignment);
    procedure   CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure   CMExit(var Message: TCMExit); message CM_EXIT;
    procedure   WMPaint(var Message: TWMPaint); message WM_PAINT;
    function    GetTextMargins: TPoint;
    procedure   TakeCareOfTheMinusSign;
    function    GetBoAsFloat: Extended;
    function    GetBoAsInteger: Integer;
    procedure   SetBoAsFloat(const Value: Extended);
    procedure   SetBoAsInteger(const Value: Integer);
  protected
    FBoFocused      : Boolean;
    FBoSearchButton : TBoEditButton;
    FBoPopup        : TBoPopupBase;
    procedure   DoEnter; override;
    procedure   ResizeControls;
    procedure   CreateWnd; override;
    procedure   CreateParams(var Params: TCreateParams); override;
    procedure   DoExit; override;
    procedure   BoSetRectNP;
    procedure   DblClick; override;
    function    BoGetDisplayText : String; virtual;
    procedure   KeyPress(var Key : char); override;
    procedure   KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure   CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    function    BoGetLeftButtonsWidth : Integer;
    function    BoGetRightButtonsWidth : Integer;
    procedure   BoUpdateButtonsFocusedStatus; virtual;
    procedure   BoCreatePopupWindow; virtual;
    procedure   BoShowPopupWindow; virtual;
    function    BoCanSetValueFromPopup : boolean; virtual;
    procedure   BoSetFocused(Value: Boolean); Virtual;
    property    BoFixedButtons: TBOEditButtons
                read   FBOFixedButtons
                write  SetBoFixedEditButtonCollection;
    property    BoFixedButtonsBeforeCustom : boolean read FBoFixedButtonsBeforeCustom
                write SetBoFixedButtonsBeforeCustom default False;
    property    BoAlignment : TAlignment read FBoAlignment write SetBoAlignment default taLeftJustify;
  public
    constructor Create(aOwner : TComponent); override;
    destructor  Destroy; override;
    function    BoIsPopupVisible : Boolean;
    property    BoButtonByIdname[IdName: String]: TBOEditButton read GetBoButtonByIdname; default;
    property    BOButtons : TBOEditButtons read FBoButtons write SeTBOEditButtons;
    property    BOFlat : Boolean read FBOFlat write SetBOFlat default false;
    property    Ctl3d : Boolean read GetCtl3d write SetCtl3d;
    property    BoChecked : Boolean read GetBoChecked write SetBoChecked stored false;
    property    BoButtonOnClick : TBoButtonClickEvent read FBoButtonOnClick write FBoButtonOnClick;
    property    BoFocused : Boolean read FBoFocused;
    property    BoOnPopupClosed : TBoHidePopupEvent read FBoOnPopupClosed write FBoOnPopupClosed;
    property    BoEditInputType : TBoEditInputType read FBoEditInputType write FBoEditInputType;
    property    BoAsInteger : Integer read GetBoAsInteger write SetBoAsInteger;
    property    BoAsFloat : Extended read GetBoAsFloat write SetBoAsFloat;    
  published
  end;

//------------------------------------------------------------------------------
  TBoEdit = class( TBoCustomEdit )
  published

    property  BoFlat;
    property  BoAlignment;
    property  BoChecked;
    property  BoButtons;
    property  BoButtonOnClick;
    property  BoEditInputType;    

    {$IFDEF Bo_Delphi4} property Anchors; {$ENDIF}
    property AutoSelect Default True;
    property AutoSize;
    {$IFDEF Bo_Delphi4} property BiDiMode;{$ENDIF}
    property BorderStyle;
    property CharCase;
    property Color;
    {$IFDEF Bo_Delphi4} property Constraints; {$ENDIF}
    property Ctl3D;
    property DragCursor;
    {$IFDEF Bo_Delphi4} property DragKind; {$ENDIF}
    property DragMode;
    property Enabled;
    property EditMask;
    property Font;
    property ImeMode;
    property ImeName;
    property MaxLength;
    {$IFDEF Bo_Delphi4} property ParentBiDiMode; {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    {$IFDEF Bo_Delphi4} property OnEndDock; {$ENDIF}
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    {$IFDEF Bo_Delphi4} property OnStartDock; {$ENDIF}
    property OnStartDrag;
    
  end;

//------------------------------------------------------------------------------
  TBoSpecialEdit = class(TBoEdit)
  private
    procedure SearchButtonClick( sender : TObject );
  protected
    procedure BoDoSearchButtonClick; virtual;
  public
    BoMainButton : TBoEditButton;  
    constructor Create(aOwner : TComponent); override;
  published
    property  BoFixedButtonsBeforeCustom;
  end;

//------------------------------------------------------------------------------
  TBoCustomDBEdit = class(TBoCustomEdit)
  private
    FDataLink: TFieldDataLink;
    FBoEditmask: String;
    procedure DataChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    function  GetDataField: string;
    function  GetDataSource: TDataSource;
    function  GetField: TField;
    function  GetReadOnly: Boolean;
    procedure ResetMaxLength;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetReadOnly(Value: Boolean);
    procedure UpdateData(Sender: TObject);
    procedure WMCut(var Message: TMessage); message WM_CUT;
    procedure WMPaste(var Message: TMessage); message WM_PASTE;
    procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
    procedure CMGetDataLink(var Message: TMessage); message CM_GETDATALINK;
  protected
    procedure Change; override;
    function  EditCanModify: Boolean; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    function  BoGetDisplayText : String; override;    
    procedure BoSetFocused(Value: Boolean); override;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure Reset; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    {$IFDEF Bo_Delphi4}
    function ExecuteAction(Action: TBasicAction): Boolean; override;
    function UpdateAction(Action: TBasicAction): Boolean; override;
    function UseRightToLeftAlignment: Boolean; override;
    {$ENDIF}
    property Field: TField read GetField;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property BoEditmask : String read FBoEditmask write FBoEditmask;
  published
  end;

//------------------------------------------------------------------------------
  TBoDBEdit = class(TBoCustomDBEdit)
  private
  protected
  public
  published

    property  BoButtons;
    property  BoFlat;
    property  BoChecked;
    property  BoButtonOnClick;

    property  DataField;
    property  DataSource;
    property  ReadOnly;
    property  BoEditmask;    

    {$IFDEF Bo_Delphi4}   property Anchors;  {$ENDIF}
    property AutoSelect;
    property AutoSize;
    {$IFDEF Bo_Delphi4}   property BiDiMode; {$ENDIF}
    property BorderStyle;
    property CharCase;
    property Color;
    {$IFDEF Bo_Delphi4}   property Constraints;  {$ENDIF}
    property Ctl3D;
    property DragCursor;
    {$IFDEF Bo_Delphi4}   property DragKind;  {$ENDIF}
    property DragMode;
    property Enabled;
    property Font;
    property ImeMode;
    property ImeName;
    property MaxLength;
    {$IFDEF Bo_Delphi4}   property ParentBiDiMode; {$ENDIF}
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    {$IFDEF Bo_Delphi4}   property OnEndDock;   {$ENDIF}
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    {$IFDEF Bo_Delphi4}  property OnStartDock;  {$ENDIF}
    property OnStartDrag;
  end;

implementation

{$IFDEF BoProtect}
var
  DelphiActive : Boolean;

function CheckDelphiActive(Handle: HWND; Info: Pointer): BOOL; stdcall;
var
  CharStr : PChar;
  StrClass: String;
  StrName: String;
const
  Size = 4096;
begin
  Result := True;
  GetMem(CharStr, Size);
  GetClassName(Handle, CharStr, Size);
  StrClass := Trim(StrPas(CharStr));
  If CompareText(StrClass, 'TAppBuilder') = 0 then
  begin
    GetWindowText(Handle, CharStr, Size);
    StrName := Copy(Trim(StrPas(CharStr)), 1, 6);
    If CompareText(StrName, 'Delphi') = 0 then begin
      DelphiActive := True;
      Result := False;
    end;
  end;
  FreeMem(CharStr);
end;
{$ENDIF}

{ TBOEditButton }

constructor TBOEditButton.Create(Collection: TCollection);
begin
  inherited Create(Collection);

  FBoButtons := TBOEditButtons(Collection);
  fBoCustomEdit := TBOEditButtons(Collection).FBoCustomEdit;

  fBoButtonType := ebt_normal;
  FBoVisible := True;
  FBoAutoHide := False;

  //--> Create speedbutton
  fBoSpeedButton := TBoSpeedButton.create(fBoCustomEdit);
  With fBoSpeedButton do begin
    Width   := BoDefaultWidth;
    OnClick := OnClickHandler;
    cursor  := crHandPoint;
    BoRepeatClick := True;
    NumGlyphs := 2;
    flat := FBoCustomEdit.BoFlat;
  end;

  FBoDefault := False;
  FBoEnabled := True;
  fBoWidth := BoDefaultWidth;
  fBoPosition := ebp_right;

end;

destructor TBOEditButton.Destroy;
begin
  fBoSpeedButton.Free; fBoSpeedButton := nil;
  inherited Destroy;
end;

function TBOEditButton.GetdisplayName: String;
begin
  Result := BoIdName;
  if Trim(Result) = '' then
    result := '<no name>';
  if BoPosition = ebp_left then
    result := result + ' --> Left '
  else
    result := result + ' --> Right ';
end;

procedure TBOEditButton.OnClickHandler(sender: TObject);
begin
  if assigned(BoOnClick) then
    BoOnClick(FBoCustomEdit);
  if assigned( FBoCustomEdit.BoButtonOnClick ) then
    FBoCustomEdit.BoButtonOnClick( FBoCustomEdit, Self );
end;

procedure TBOEditButton.SetBoGlyph(const Value: TBitmap);
begin
  fBoSpeedButton.Glyph.assign(value);
  BoResBmpName := '';
end;

procedure TBOEditButton.SetBoCaption(const Value: String);
begin
  if (trim(fBoIdName) = '') or (BoCaption = fBoIdName) then
    BoIdName := Value;
  fBoSpeedButton.caption := value;
  changed(false);
end;

procedure TBOEditButton.SetBoIdName(const Value: String);
begin
  FBoIdName := Value;
  changed(false);
end;

procedure TBOEditButton.SetBoPosition(const Value: TBoEditButtonPosition);
begin
  FBoPosition := Value;
  changed(false);
end;

procedure TBOEditButton.SetBoWidth(const Value: Integer);
begin
  FBoWidth := Value;
  changed(false);
end;

procedure TBOEditButton.SetBoNumGlyphs(const Value: Integer);
begin
  fBoSpeedButton.NumGlyphs := value;
end;

procedure TBOEditButton.SetBoResBmpName(const Value: String);
begin
  fBoSpeedButton.BoResBmpName := value;
end;

procedure TBOEditButton.SetBoShortCut(const Value: TShortCut);
begin
  FBoShortCut := Value;
end;

procedure TBOEditButton.SetBoDefault(const Value: Boolean);
begin
  FBoDefault := Value;
end;

procedure TBOEditButton.SetBoEnabled(const Value: Boolean);
begin
  FBoEnabled := Value;
  fBoSpeedButton.enabled := value;
end;

procedure TBOEditButton.SetBoHint(const Value: String);
begin
  fBoSpeedButton.hint := value;
end;

procedure TBOEditButton.BoDoClick;
begin
  fBoSpeedButton.Click;
end;

procedure TBOEditButton.SetBoGroupIndex(const Value: Integer);
begin
  fBoSpeedButton.groupindex := value;
end;

function TBOEditButton.GetBoCaption: String;
begin
  result := fBoSpeedButton.caption;
end;

function TBOEditButton.GetBoGlyph: TBitmap;
begin
  result := fBoSpeedButton.Glyph;
end;

function TBOEditButton.GetBoNumGlyphs: Integer;
begin
  result := fBoSpeedButton.NumGlyphs;
end;

function TBOEditButton.GetBoGroupIndex: Integer;
begin
  result := fBoSpeedButton.groupindex;
end;

function TBOEditButton.GetBoResBmpName: String;
begin
  result := fBoSpeedButton.BoResBmpName;
end;

function TBOEditButton.GetBoHint: String;
begin
  result := fBoSpeedButton.hint;
end;

procedure TBOEditButton.BOEnableDisableButton(blnEnabled: Boolean);
begin
  if BoEnabled then
    fBoSpeedButton.Enabled := BlnEnabled
  else
    fBoSpeedButton.Enabled := false;
end;

procedure TBOEditButton.BoSetBounds(aLeft, aTop, aWidth, aHeight : Integer; aParent : TWinControl);
begin
  fBoSpeedButton.SetBounds(aLeft,aTop,aWidth,aHeight);
  fBoSpeedButton.Parent := aParent;
  fBoSpeedButton.Flat := FBoCustomEdit.BoFlat;
end;

function TBOEditButton.BoGlyphStored: boolean;
begin
  result := Trim(BoResBmpName) = '';
end;

function TBOEditButton.GetBoDown: Boolean;
begin
  result := fBoSpeedbutton.Down;
end;

procedure TBOEditButton.SetBoDown(const Value: Boolean);
begin
  fBoSpeedButton.Down := Value;
end;

function TBOEditButton.GetBoAllowAllUp: Boolean;
begin
  result := fBoSpeedButton.AllowAllUp;
end;

procedure TBOEditButton.SetBoAllowAllUp(const Value: Boolean);
begin
  fBoSpeedButton.AllowAllUp := value;
end;

procedure TBOEditButton.SetBoButtonType(const Value: TBoEditButtonType);
begin
  FBoButtonType := Value;
  With fBoSpeedButton do
  begin
    BoValues.Clear;
    BoResBmpNames.Clear;
    if value = ebt_checked then
      begin
        BoValues.add('0');
        BoValues.add('1');
        BoResBmpNames.add('bobmp_unchecked');
        BoResBmpNames.add('bobmp_checked');
        BoValue := '0';
        BoWidth := 18;
      end
    else
      begin
        BoValue := '';
        BoResBmpName := '';
        BoGlyph.assign(nil);
      end
  end;
end;

function TBOEditButton.GetBoChecked: Boolean;
begin
  if BoButtonType = ebt_checked then
    result := (fBoSpeedButton.BoValue = '1')
  else
    result := false;
end;

procedure TBOEditButton.SetBoChecked(const Value: Boolean);
begin
  if BoButtonType = ebt_checked then
    if Value then
      fBoSpeedButton.BoValue := '1'
    else
      fBoSpeedButton.BoValue := '0';
end;

procedure TBOEditButton.SetBoVisible(const Value: Boolean);
begin
  FBoVisible := Value;
  changed(false);
end;

procedure TBOEditButton.SetBoAutoHide(const Value: Boolean);
begin
  FBoAutoHide := Value;
  changed(false);
end;

function TBOEditButton.GetBoButtonCurrentlyVisible: Boolean;
begin
  if csDesigning in FBoCustomEdit.componentstate then
    result := true
  else
    if BoAutoHide then
      result := BoVisible and BoEditFocused
    else
      result := BoVisible;
end;

procedure TBOEditButton.SetBoEditFocused(const Value: Boolean);
begin
  FBoEditFocused := Value;
//  changed(false);  
end;

{ TBOEditButtons }

function TBOEditButtons.Add: TBOEditButton;
begin
  Result := TBOEditButton(inherited Add);
end;

procedure TBOEditButtons.BOEnableDisableButtons(blnEnabled: Boolean);
var
  i : integer;
begin
  for i := 0 to Pred(self.count) do
    items[i].BOEnableDisableButton(blnEnabled);
end;

constructor TBOEditButtons.create(BoCustomEdit: TBoCustomEdit;
  BOEditButtonClass: TBOEditButtonClass);
begin
  inherited Create(BOEditButtonClass);
  fBoCustomEdit := BoCustomEdit;
end;

destructor TBOEditButtons.Destroy;
var
  IntCount, IntIndex : Integer;
begin
  IntCount := Self.Count;
  for IntIndex := 0 to IntCount - 1 do
    Self.Items[0].Free;
  inherited Destroy;
end;

function TBOEditButtons.GetBOEditButton(
  Index: Integer): TBOEditButton;
begin
  Result := TBOEditButton(inherited Items[Index]);
end;

function TBOEditButtons.GetOwner: TPersistent;
begin
  Result := FBoCustomEdit;
end;

function TBOEditButtons.BoGetWidthForPosition( aPosition    : TBoEditButtonPosition;
                                               CheckVisible : Boolean ) : integer;
var
  i : integer;
begin
  result := 0;
  for i := 0 to Pred(count) do
  begin
    if (items[i].BoPosition = aPosition) then
      if (items[i].BoButtonCurrentlyVisible) or (not CheckVisible) then
        result := result + items[i].BoWidth;
  end;
end;

procedure TBOEditButtons.SetBOEditButton(Index: Integer;
  const Value: TBOEditButton);
begin
  Items[Index].Assign(Value);
end;

procedure TBOEditButtons.Update(Item: TCollectionItem);
begin
  if assigned(FBOCustomEdit) then
    FBOCustomEdit.RecreateWnd;
end;

{ TBoCustomEdit }

function TBoCustomEdit.BoCanSetValueFromPopup : Boolean;
begin
  result := True;
  //virtual
end;

procedure TBoCustomEdit.BoCreatePopupWindow;
begin
  //virtual
end;

function TBoCustomEdit.BoGetDisplayText: String;
begin
  result := EditText;
end;

function TBoCustomEdit.BoGetLeftButtonsWidth: Integer;
begin
  result := FBOFixedButtons.BoGetWidthForPosition( ebp_left, True ) +
            FBoButtons.BoGetWidthForPosition( ebp_left, True ) ;
end;

function TBoCustomEdit.BoGetRightButtonsWidth: Integer;
begin
  result := FBOFixedButtons.BoGetWidthForPosition( ebp_right, True ) +
            FBoButtons.BoGetWidthForPosition( ebp_right, True ) ;
end;

function TBoCustomEdit.BoIsPopupVisible: Boolean;
begin
  if Assigned(FBoPopup) then
    result := FBoPopup.Visible
  else
    result := false;
end;

procedure TBoCustomEdit.BoSetFocused(Value: Boolean);
begin
  if FBoFocused <> Value then
  begin
    FBoFocused := Value;

    if (FBoAlignment <> taLeftJustify) {and not IsMasked} then
      Invalidate;

    //--> Update focused status
    BoUpdateButtonsFocusedStatus;

  end;

end;

procedure TBoCustomEdit.BoSetRectNP;
var
  xRect      : TRect;
  LeftWidth  : integer;
  RightWidth : integer;
begin
  LeftWidth  := BoGetLeftButtonsWidth;
  RightWidth := BoGetRightButtonsWidth;
  SetRect(xRect, LeftWidth, 0, ClientWidth - RightWidth , ClientHeight + 1);
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@xRect));
end;

procedure TBoCustomEdit.BoShowPopupWindow;
var
  p : TPoint;
begin
  BoCreatePopupWindow;
  if not assigned(FBoPopup) then
   exit;
  FBoPopup.BoOnHidePopup := HidePopupWindowEvent;
  P := Parent.ClientToScreen(Point(Left, Top));
  p.Y := P.Y + Height;
  if p.Y + FBoPopup.Height > Screen.Height then
    p.Y := P.Y - FBoPopup.Height;
  Dec(P.X, FBoPopup.Width - Width);
  if P.X < 0 then
    Inc(P.X, FBoPopup.Width - Width);
  if P.X < 0 then
    P.X := 0
   else
     if P.X + FBoPopup.Width > Screen.Width then
       P.X := Screen.Width - FBoPopup.Width;
  FBoPopup.BoShowPopup(p);
  repaint;
end;

procedure TBoCustomEdit.BoUpdateButtonsFocusedStatus;
var
  i : integer;
begin
  try
    for i := 0 to self.FBoButtons.Count - 1 do
      fBoButtons[i].BoEditFocused := FBoFocused;
    for i := 0 to self.FBoFixedButtons.Count - 1 do
      fBoFixedButtons[i].BoEditFocused := FBoFocused;
  finally
    BoSetRectNP;
    ResizeControls;
  end;
end;

procedure TBoCustomEdit.CMCancelMode(var Message: TCMCancelMode);
begin
  if (Assigned(FBoPopup)) and
     (Message.Sender <> Self) and
     (Message.Sender <> FBoPopup) and
     (Message.Sender <> FBoSearchButton.fBoSpeedButton) and
     (not FBoPopup.ContainsControl(Message.Sender)) {and}
     {(assigned(message.sender))} then
     FBoPopup.BoHidePopup(false);
end;

procedure TBoCustomEdit.CMEnabledChanged(var Message: TMessage);
begin
  FBoButtons.BOEnableDisableButtons(Enabled);
  FBOFixedButtons.BOEnableDisableButtons(Enabled);
  inherited;
end;

procedure TBoCustomEdit.CMEnter(var Message: TCMEnter);
begin
  BoSetFocused(True);
  inherited;
end;

procedure TBoCustomEdit.TakeCareOfTheMinusSign;
var
  idx : Integer;
  s   : String;
begin
  s := Text;
  //--> Put - sign in front of the number
  if (FBoFocused) and (FBoEditInputType = eit_float) or (FBoEditInputType = eit_integer) then
  begin
    idx := Pos('-',s);
    if (Idx  > 0) then
    begin
      repeat
        Delete(s,Idx,1);
        idx := Pos('-',s);
      until (Idx = 0);
      Text := '-' + s;
    end;
  end;
end;

procedure TBoCustomEdit.CMExit(var Message: TCMExit);
begin
  TakeCareOfTheMinusSign;   
  BoSetFocused(False);
  inherited;  
end;

constructor TBoCustomEdit.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  FBoAlignment := taLeftJustify;
  FBoFlat := false;
  FBoFixedButtonsBeforeCustom := false;
  FBoButtons := TBOEditButtons.Create(Self, TBOEditButton);
  FBOFixedButtons := TBOEditButtons.Create(Self, TBOEditButton);
end;

procedure TBoCustomEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.Style := Params.Style or WS_CLIPCHILDREN or ES_MULTILINE;
  { bocheck  //or ES_MULTILINE
    This seems not to be needed anymore with the current windows edit control
    Tested on : Winnt 4.0 service pack 4
    With older windows versions there was a problem with the WS_ClipChildren param
    for non-multiline editboxes.

    Setting the ES_Multiline param causes a VK_UP key not to work properly
  }

  {$IFDEF BoProtect}
  DelphiActive := false;
  EnumWindows(@CheckDelphiActive, 0);
  if not DelphiActive then halt(1);
  {$ENDIF}

end;

procedure TBoCustomEdit.CreateWnd;
begin
  inherited CreateWnd;
  BoSetRectNP;
end;

procedure TBoCustomEdit.DblClick;
//var
//  Key : Word;
begin
  inherited;
//  Key := Vk_Down;
//  InternalKeyDown(key,[ssAlt]);
end;

destructor TBoCustomEdit.Destroy;
begin
  FCanvas.Free;
  FBoButtons.FBoCustomEdit := nil;
  FBoButtons.Free;
  FBOFixedButtons.FBoCustomEdit := nil;
  FBOFixedButtons.Free;
  inherited Destroy;
end;

procedure TBoCustomEdit.DoEnter;
begin
  inherited;
  if AutoSelect then
    SelectAll;
end;

procedure TBoCustomEdit.DoExit;
begin
  inherited DoExit;
  if assigned(FBoPopup) then FBoPopup.BoHidePopup(false);
end;

function TBoCustomEdit.GetBoButtonByIdname(IdName: String): TBOEditButton;
var
  i : integer;
begin
  result := nil;
  for i := 0 to pred(BoButtons.Count) do
    if CompareText( BoButtons[i].BoIdName, IdName ) = 0 then
    begin
      result := BoButtons[i];
      break;
    end;
  if Not assigned(result) then
    for i := 0 to pred(BoFixedButtons.Count) do
      if CompareText( BoFixedButtons[i].BoIdName, IdName ) = 0 then
      begin
        result := BoFixedButtons[i];
        break;
      end;
  if Not assigned(result) then
    raise Exception.Create('Button ' + IdName + ' does not exists');
end;

function TBoCustomEdit.GetBoChecked: Boolean;
var
  i : integer;
begin
  result := true;
  for i := 0 to pred(BoButtons.Count) do
    if BoButtons[i].BoButtonType = ebt_checked then
      if not BoButtons[i].BoChecked then
      begin
        result := false;
        exit;
      end;
  for i := 0 to pred(BoFixedButtons.Count) do
    if BoFixedButtons[i].BoButtonType = ebt_checked then
      if not BoFixedButtons[i].BoChecked then
      begin
        result := false;
        exit;
      end;
end;

function TBoCustomEdit.GetCtl3d: Boolean;
begin
  result := inherited Ctl3d;
end;

function TBoCustomEdit.GetTextMargins: TPoint;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  if NewStyleControls then
  begin
    if BorderStyle = bsNone then I := 0 else
      if Ctl3D then I := 1 else I := 2;
    Result.X := SendMessage(Handle, EM_GETMARGINS, 0, 0) and $0000FFFF + I;
    Result.Y := I;
  end else
  begin
    if BorderStyle = bsNone then I := 0 else
    begin
      DC := GetDC(0);
      GetTextMetrics(DC, SysMetrics);
      SaveFont := SelectObject(DC, Font.Handle);
      GetTextMetrics(DC, Metrics);
      SelectObject(DC, SaveFont);
      ReleaseDC(0, DC);
      I := SysMetrics.tmHeight;
      if I > Metrics.tmHeight then I := Metrics.tmHeight;
      I := I div 4;
    end;
    Result.X := I;
    Result.Y := I;
  end;
end;

procedure TBoCustomEdit.HidePopupWindowEvent(sender: TObject; UseNewValue: Boolean);
begin
  if UseNewValue and assigned(fBoPopup) then
  begin
    if BoCanSetValueFromPopup then
      self.text := fBoPopup.BoGetText
    else
      beep;
    self.selectall;
  end;
  if Assigned(FBoOnPopupClosed) then
    FBoOnPopupClosed(Self,UseNewValue);
end;

procedure TBoCustomEdit.InternalKeyDown(var Key: Word;
  Shift: TShiftState);
var
  FirstButton : TBoEditButton;
  ThisShortCut : TShortCut;
  BlnKeyHandled : Boolean;

  function KeyHandled( aCollection : TBOEditButtons ) : boolean;
  var i : integer;
  begin
    result := false;
    for i := 0 to pred(aCollection.Count) do
    begin
      if (aCollection[i].boEnabled) then
      begin
        //--> Save pointer to first enabled button
        if Not assigned(FirstButton) then
          FirstButton := aCollection[i];
        //--> Test shortcut or default button
        if (aCollection[i].BoShortCut = ThisShortCut) or
           ((aCollection[i].BoDefault) and (key = vk_down) and (Shift = [ssAlt])) then
        begin
          aCollection[i].BoDoClick;
          result := true;
          break;
        end;
      end;
    end;
  end;
begin
  ThisShortCut := ShortCut(Key,Shift);
  FirstButton := nil;

  //--> Check if a button needs this key
  if FBoFixedButtonsBeforeCustom then
    begin
      BlnKeyHandled := KeyHandled( BOFixedButtons );
      if Not BlnKeyHandled then
        BlnKeyHandled := KeyHandled( BOButtons );
    end
  else
    begin
      BlnKeyHandled := KeyHandled( BOButtons );
      if Not BlnKeyHandled then
        BlnKeyHandled := KeyHandled( BOFixedButtons );
    end;

  //--> When no default button found, use first button
  if (key = vk_down) and (Shift = [ssAlt]) then
    if assigned( FirstButton ) then
      if Not BlnKeyHandled then
        FirstButton.BoDoClick;

end;

procedure TBoCustomEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if Key = 0 then
  begin
    inherited KeyDown( Key, Shift );
    exit;
  end;
  InternalKeyDown(key,Shift);
  inherited KeyDown( Key, Shift );
end;

procedure TBoCustomEdit.KeyPress(var Key: char);
begin

  //--> Input type is integer
  if FBoEditInputType = eit_integer then
  begin
    if (not (Key in ['0'..'9','-',#8,#13,#35,#36,#37,#39])) or
       (Key = #32) or
       ((Key = '-') and (Pos('-',Text) > 0)) then
       Key := #15
   end;

  //--> Input type is float
  if FBoEditInputType = eit_float then
  begin
    if (not (Key in ['0'..'9',',','.','-',#8,#13,#35,#36,#37,#39])) or
       (Key = #32) or
       ((Key = '-') and (Pos('-',Text) > 0)) then
       Key := #15;
    if (Key = ',') or (Key = '.') then
       if (Pos(',',Text) > 0) or (Pos('.',Text) > 0) then
          Key := #15
        else
          Key := DecimalSeparator;
  end;

  //--> Fix for multiline edit
  if (Key = Char(VK_RETURN)) or (Key = Char(VK_ESCAPE)) then
  begin
    // remove this key, this control is a multi-line editbox
    // but I'll use it as a single line editbox
    GetParentForm(Self).Perform(CM_DIALOGKEY, Byte(Key), 0);
    if Key = Char(VK_RETURN) then Key := #0;
  end;

  inherited KeyPress(Key);
end;

procedure TBoCustomEdit.ResizeControls;
var
  LeftWidth,
  RightWidth  : integer;
  LeftOffset  : integer;
  RightOffset : integer;

  procedure SetPositionEditButton( xEditButton : TBOEditButton );
  begin
    if xEditButton.BoPosition = ebp_left then
      begin
        xEditButton.BoSetBounds( LeftOffset, 0,
                                 xEditButton.BoWidth, clientHeight,
                                 fBoLeftPanel );
        LeftOffset := LeftOffset + xEditButton.BoWidth;
      end
    else
      begin
        xEditButton.BoSetBounds( RightOffset, 0,
                                 xEditButton.BoWidth, clientHeight,
                                 fBoRightPanel );
        RightOffset := RightOffset + xEditButton.BoWidth;
      end;
  end;

  Procedure SetPositionFixed;
  var i : integer;
  begin
    for i := 0 to Pred(FBOFixedButtons.Count) do
      SetPositionEditButton( FBOFixedButtons[i] );
  end;

  Procedure SetPosition;
  var i : integer;
  begin
    for i := 0 to Pred(FBoButtons.Count) do
      SetPositionEditButton( FBoButtons[i] );
  end;

  procedure CreatePanel( Var   aPanel : TPanel;
                         Const aLeft, aWidth : Integer );
  begin
    if Not Assigned(aPanel) then
      aPanel := TPanel.Create(self);
    aPanel.BevelInner := bvNone;
    aPanel.BevelOuter := bvNone;
    aPanel.BorderWidth := 0;
    if (Not Ctl3d) then
      aPanel.BorderStyle := bsSingle
    else
      aPanel.BorderStyle := bsNone;
    if BoFlat then
      aPanel.BevelOuter := bvRaised;
    aPanel.SetBounds(aLeft,0,aWidth,clientHeight);
    aPanel.Parent := self;
  end;

  procedure FreePanel( Var    aPanel : TPanel;
                       Const  aWidth : Integer );
  begin
    if aWidth = 0 then
    begin
      aPanel.Free;
      aPanel := Nil;
    end;
  end;

begin

  //--> Calculate widths of left/right panel
  LeftWidth := FBoButtons.BoGetWidthForPosition( ebp_left, True ) +
               FBOFixedButtons.BoGetWidthForPosition( ebp_left, True );
  RightWidth := FBoButtons.BoGetWidthForPosition( ebp_right, true ) +
                FBOFixedButtons.BoGetWidthForPosition( ebp_right, True );

  //--> Create parent panels
  //    This panel is needed because a TGraphic control painting has problems
  //    in the derived TBoDBEdit component
  //    A TWincontrol is needed to fix this problem
  CreatePanel( FBOLeftPanel,  0,                        LeftWidth  );
  CreatePanel( FBORightPanel, ClientWidth - RightWidth, RightWidth );

  //--> Put buttons inside the panels
  LeftOffset  := 0;
  RightOffset := 0;
  if assigned(FBOLeftPanel) or assigned(FBORightPanel) then
  begin
    if FBoFixedButtonsBeforeCustom then
      begin
        SetPositionFixed;
        SetPosition;
      end
    else
      begin
        SetPosition;
        SetPositionFixed;
      end;
  end;

  //--> Free the parent panels when not used
  FreePanel( FBOLeftPanel,  FBoButtons.BoGetWidthForPosition( ebp_left, False ) +
                            FBOFixedButtons.BoGetWidthForPosition( ebp_left, False )  );
  FreePanel( FBORightPanel, FBoButtons.BoGetWidthForPosition( ebp_right, False ) +
                            FBOFixedButtons.BoGetWidthForPosition( ebp_right, False ) );
end;

procedure TBoCustomEdit.SetBoAlignment(const Value: TAlignment);
begin
  FBoAlignment := Value;
  Invalidate;
end;

procedure TBoCustomEdit.SetBoChecked(const Value: Boolean);
var
  i : integer;
begin
  for i := 0 to pred(BoButtons.Count) do
    if BoButtons[i].BoButtonType = ebt_checked then
      BoButtons[i].BoChecked := Value;
  for i := 0 to pred(BoFixedButtons.Count) do
    if BoFixedButtons[i].BoButtonType = ebt_checked then
      BoFixedButtons[i].BoChecked := Value;
end;

procedure TBoCustomEdit.SeTBOEditButtons(const Value: TBOEditButtons);
begin
  BoButtons.Assign(Value);
end;

procedure TBoCustomEdit.SetBoFixedButtonsBeforeCustom(const Value: boolean);
begin
  FBoFixedButtonsBeforeCustom := Value;
  RecreateWnd;
end;

procedure TBoCustomEdit.SetBoFixedEditButtonCollection( const Value: TBOEditButtons);
begin
  FBOFixedButtons.assign(Value);
end;

procedure TBoCustomEdit.SetBOFlat(const Value: Boolean);
begin
  FBOFlat := Value;
  recreateWnd;
end;

procedure TBoCustomEdit.SetCtl3d(const Value: Boolean);
begin
  inherited Ctl3d := value;
  recreateWnd;
end;

procedure TBoCustomEdit.WMPaint(var Message: TWMPaint);
const
  AlignStyle : array[Boolean, TAlignment] of DWORD =
   ((WS_EX_LEFT, WS_EX_RIGHT, WS_EX_LEFT),
    (WS_EX_RIGHT, WS_EX_LEFT, WS_EX_LEFT));
var
  Left: Integer;
  Margins: TPoint;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
  AAlignment: TAlignment;
  {$IFDEF Bo_Delphi4}
  ExStyle: DWORD;
  {$ENDIF}  
begin
  AAlignment := FBoAlignment;

  {$IFDEF Bo_Delphi4}
  if UseRightToLeftAlignment then ChangeBiDiModeAlignment(AAlignment);
  if ((AAlignment = taLeftJustify) or FBoFocused) and
    not (csPaintCopy in ControlState) then
  begin
    if SysLocale.MiddleEast and HandleAllocated and (IsRightToLeft) then
    begin { This keeps the right aligned text, right aligned }
      ExStyle := DWORD(GetWindowLong(Handle, GWL_EXSTYLE)) and (not WS_EX_RIGHT) and
        (not WS_EX_RTLREADING) and (not WS_EX_LEFTSCROLLBAR);
      if UseRightToLeftReading then ExStyle := ExStyle or WS_EX_RTLREADING;
      if UseRightToLeftScrollbar then ExStyle := ExStyle or WS_EX_LEFTSCROLLBAR;
      ExStyle := ExStyle or
        AlignStyle[UseRightToLeftAlignment, AAlignment];
      if DWORD(GetWindowLong(Handle, GWL_EXSTYLE)) <> ExStyle then
        SetWindowLong(Handle, GWL_EXSTYLE, ExStyle);
    end;
    inherited;
    Exit;
  end;
  {$ENDIF}

  {$IFDEF Bo_Delphi3}
  if ((AAlignment = taLeftJustify) or FBoFocused) and
    not (csPaintCopy in ControlState) then
  begin
    inherited;
    Exit;
  end;
  {$ENDIF}

{ Since edit controls do not handle justification unless multi-line (and
  then only poorly) we will draw right and center justify manually unless
  the edit has the focus. }
  if FCanvas = nil then
  begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      R := ClientRect;
      if not (NewStyleControls and Ctl3D) and (BorderStyle = bsSingle) then
      begin
        Brush.Color := clWindowFrame;
        FrameRect(R);
        InflateRect(R, -1, -1);
      end;
      Brush.Color := Color;
      if not Enabled then
        Font.Color := clGrayText;

      s := BoGetDisplayText;

      if PasswordChar <> #0 then FillChar(S[1], Length(S), PasswordChar);
      Margins := GetTextMargins;

      //bocheck:dit moet nog beter....
      case AAlignment of
        taLeftJustify: Left := Margins.X;
        taRightJustify: Left := ClientWidth - TextWidth(S) - Margins.X - 1 -
                                BoGetRightButtonsWidth;
      else
        left := Margins.x + BoGetLeftButtonsWidth + (ClientWidth - boGetRightButtonsWidth - BoGetleftButtonsWidth - TextWidth(S)) div 2        
      end;

      {$IFDEF Bo_Delphi4}
      if SysLocale.MiddleEast then UpdateTextFlags;
      {$ENDIF}
      TextRect(R, Left, Margins.Y, S);
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then EndPaint(Handle, PS);
  end;
end;

procedure TBoCustomEdit.WMSize(var message: TWMSize);
begin
  inherited;
  BoSetRectNP;
  ResizeControls;
end;

function TBoCustomEdit.GetBoAsFloat: Extended;
var
  i : Integer;
  s : String;
  FCode : Integer;
begin
  TakeCareOfTheMinusSign;
  s := Text;
  for i := 1 to length(s) do
    if s[i] = DecimalSeparator then s[i] := '.';  //VAL can only handle '.'
  Val( s, Result, FCode );
end;

function TBoCustomEdit.GetBoAsInteger: Integer;
var
  FCode : Integer;
begin
  TakeCareOfTheMinusSign;
  Val( Text, Result, FCode );
end;

procedure TBoCustomEdit.SetBoAsFloat(const Value: Extended);
begin
  BoEditInputType := Eit_Float;
  text := FloatToStr(Value);
end;

procedure TBoCustomEdit.SetBoAsInteger(const Value: Integer);
begin
  BoEditInputType := Eit_Integer;
  text := IntToStr(Value);
end;

{ TBoSpecialEdit }

procedure TBoSpecialEdit.BoDoSearchButtonClick;
begin
end;

constructor TBoSpecialEdit.Create(aOwner: TComponent);
begin
  inherited create(aOwner);
  BoMainButton := BoFixedButtons.add;
  with BoMainButton do
  begin
    BoDefault := true;
    BoOnClick := SearchButtonClick;
    BoResBmpName := 'BoBmp_Search';
    BoShortCut := ShortCut( vk_f3, []);
    BoHint := 'search';
  end;
end;

procedure TBoSpecialEdit.SearchButtonClick(sender: TObject);
begin
  BoDoSearchButtonClick;
end;

{ TBoCustomDBEdit }

procedure TBoCustomDBEdit.ResetMaxLength;
var
  F: TField;
begin
  if (MaxLength > 0) and Assigned(DataSource) and Assigned(DataSource.DataSet) then
  begin
    F := DataSource.DataSet.FindField(DataField);
    if Assigned(F) and (F.DataType = ftString) and (F.Size = MaxLength) then
      MaxLength := 0;
  end;
end;

constructor TBoCustomDBEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited ReadOnly := True;
  ControlStyle := ControlStyle + [csReplicatable];
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
end;

destructor TBoCustomDBEdit.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TBoCustomDBEdit.Loaded;
begin
  inherited Loaded;
  ResetMaxLength;
  if (csDesigning in ComponentState) then DataChange(Self);
end;

procedure TBoCustomDBEdit.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

{$IFDEF Bo_Delphi4}
function TBoCustomDBEdit.UseRightToLeftAlignment: Boolean;
begin
  Result := DBUseRightToLeftAlignment(Self, Field);
end;
{$ENDIF}

procedure TBoCustomDBEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
    FDataLink.Edit;
end;

procedure TBoCustomDBEdit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
    not FDataLink.Field.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      FDataLink.Edit;
    #27:
      begin
        FDataLink.Reset;
        SelectAll;
        Key := #0;
      end;
  end;
end;

function TBoCustomDBEdit.EditCanModify: Boolean;
begin
  Result := FDataLink.Edit;
end;

procedure TBoCustomDBEdit.Reset;
begin
  FDataLink.Reset;
  SelectAll;
end;

procedure TBoCustomDBEdit.Change;
begin
  FDataLink.Modified;
  inherited Change;
end;

function TBoCustomDBEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TBoCustomDBEdit.SetDataSource(Value: TDataSource);
begin
  if not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then
    FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TBoCustomDBEdit.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TBoCustomDBEdit.SetDataField(const Value: string);
begin
  if not (csDesigning in ComponentState) then
    ResetMaxLength;
  FDataLink.FieldName := Value;
end;

function TBoCustomDBEdit.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TBoCustomDBEdit.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TBoCustomDBEdit.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TBoCustomDBEdit.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
  begin
    if BoAlignment <> FDataLink.Field.Alignment then
    begin
      EditText := '';  {forces update}
      BoAlignment := FDataLink.Field.Alignment;
    end;
    if Trim(BoEditmask) = '' then
      EditMask := FDataLink.Field.EditMask
    else
      EditMask := BoEditMask;
    if not (csDesigning in ComponentState) then
    begin
      if (FDataLink.Field.DataType = ftString) and (MaxLength = 0) then
        MaxLength := FDataLink.Field.Size;
    end;
    if BoFocused and FDataLink.CanModify then
      Text := FDataLink.Field.Text
    else
    begin
      if Trim(BoEditmask) = '' then
        EditText := FDataLink.Field.DisplayText
      else
        EditText := FDataLink.Field.Text;
      if FDataLink.Editing {and FDataLink.Modified //bocheck} then
        Modified := True;
    end;
  end else
  begin
    BoAlignment := taLeftJustify;
    if Trim(BoEditmask) = '' then
      EditMask := ''
    else
      EditMask := BoEditMask;
    if csDesigning in ComponentState then
      EditText := Name else
      EditText := '';
  end;
end;

procedure TBoCustomDBEdit.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not FDataLink.Editing;
end;

procedure TBoCustomDBEdit.UpdateData(Sender: TObject);
begin
  ValidateEdit;
  FDataLink.Field.Text := Text;
end;

procedure TBoCustomDBEdit.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TBoCustomDBEdit.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure TBoCustomDBEdit.CMEnter(var Message: TCMEnter);
begin
  inherited;
  if SysLocale.FarEast and FDataLink.CanModify then
    inherited ReadOnly := False;
end;

procedure TBoCustomDBEdit.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  BoSetFocused(False);
  CheckCursor;
  DoExit;
end;

procedure TBoCustomDBEdit.CMGetDataLink(var Message: TMessage);
begin
  Message.Result := Integer(FDataLink);
end;

{$IFDEF Bo_Delphi4}
function TBoCustomDBEdit.ExecuteAction(Action: TBasicAction): Boolean;
begin
  Result := inherited ExecuteAction(Action) or (FDataLink <> nil) and
    FDataLink.ExecuteAction(Action);
end;
{$ENDIF}

{$IFDEF Bo_Delphi4}
function TBoCustomDBEdit.UpdateAction(Action: TBasicAction): Boolean;
begin
  Result := inherited UpdateAction(Action) or (FDataLink <> nil) and
    FDataLink.UpdateAction(Action);
end;
{$ENDIF}

procedure TBoCustomDBEdit.BoSetFocused(Value: Boolean);
begin
  inherited;
  FDataLink.Reset;
end;

function TBoCustomDBEdit.BoGetDisplayText: String;
begin
  result := inherited BoGetDisplayText;
  if (csPaintCopy in ControlState) and (FDataLink.Field <> nil) then
  begin
    if Trim(BoEditmask) = '' then
      result := FDataLink.Field.DisplayText    
    else
      result := FDataLink.Field.Text;
    case CharCase of
      ecUpperCase: result := AnsiUpperCase(result);
      ecLowerCase: result := AnsiLowerCase(result);
    end;
  end;
end;

{ TBoPopupBase }

constructor TBoPopupBase.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  BoParentControl := TWinControl(aOwner);
  ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable,
    csAcceptsControls];
  Ctl3D := False;
  ParentCtl3D := False;
  Visible := False;
  Parent := TWinControl(AOwner);
  Height := 200;
  Width  := 400;
end;

procedure TBoPopupBase.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do begin
    Style := WS_POPUP or WS_BORDER or WS_CLIPCHILDREN;
    ExStyle := WS_EX_TOOLWINDOW;
    WindowClass.Style := WindowClass.Style or CS_SAVEBITS;
  end;
end;

procedure TBoPopupBase.BoHidePopup(UseNewValue : Boolean);
begin
  SetWindowPos(Handle, 0, 0, 0, 0, 0, SWP_NOZORDER or
    SWP_NOMOVE or SWP_NOSIZE or SWP_NOACTIVATE or SWP_HIDEWINDOW);
  Visible := False;
  if Assigned(fBoOnHidePopup) then
    fBoOnHidePopup(self,UseNewValue);
end;

procedure TBoPopupBase.BoShowPopup(Where: TPoint);
begin
  SetWindowPos(Handle, HWND_TOP, Where.X, Where.Y, 0, 0,
    SWP_NOACTIVATE or SWP_SHOWWINDOW or SWP_NOSIZE);
  Visible := True;
end;

procedure TBoPopupBase.WMMouseActivate(var Message: TMessage);
begin
  Message.Result := MA_NOACTIVATE;
end;

function TBoPopupBase.BoGetId: Variant;
begin
  result := Null;
end;    

function TBoPopupBase.BoGetText: String;
begin
  result := '';
end;

end.


