unit RegInfo;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Registry;

type
  TRegInfo = class(TComponent)
  private
    { Private declarations }
    FReadOnly: Boolean;
    FRegistry: TRegistry;
    FKey: string;

  protected
    { Protected declarations }
    procedure RaiseReadOnly;
    procedure WriteBinaryData(const Name: string; var Buffer; BufSize: Integer);
    procedure WriteBool(const Name: string; Value: Boolean);
    procedure WriteCurrency(const Name: string; Value: Currency);
    procedure WriteDate(const Name: string; Value: TDateTime);
    procedure WriteDateTime(const Name: string; Value: TDateTime);
    procedure WriteFloat(const Name: string; Value: Double);
    procedure WriteInteger(const Name: string; Value: Integer);
    procedure WriteString(const Name, Value: string);
    procedure WriteTime(const Name: string; Value: TDateTime);

    function ReadBinaryData(const Name: string; var Buffer;
        BufSize: Integer): Integer;
    function ReadBool(const Name: string): Boolean;
    function ReadCurrency(const Name: string): Currency;
    function ReadDate(const Name: string): TDateTime;
    function ReadDateTime(const Name: string): TDateTime;
    function ReadFloat(const Name: string): Double;
    function ReadInteger(const Name: string): Integer;
    function ReadString(const Name: string): string;
    function ReadTime(const Name: string): TDateTime;

    // A estas, se aaden las siguientes:
    procedure WriteShortint(const Name: string; Value: Shortint);
    procedure WriteSmallint(const Name: string; Value: Smallint);
    procedure WriteLongint(const Name: string; Value: Longint);
    procedure WriteByte(const Name: string; Value: Byte);
    procedure WriteWord(const Name: string; Value: Word);
    procedure WriteSingle(const Name: string; Value: Single);
    procedure WriteSingleDateTime(const Name: string; Value: TDateTime);

    function ReadShortint(const Name: string): Shortint;
    function ReadSmallint(const Name: string): Smallint;
    function ReadLongint(const Name: string): Longint;
    function ReadByte(const Name: string): Byte;
    function ReadWord(const Name: string): Word;
    function ReadSingle(const Name: string): Single;
    function ReadSingleDateTime(const Name: string): TDateTime;

    property ReadOnly: Boolean read FReadOnly write FReadOnly default True;
    property Reg: TRegistry read FRegistry write FRegistry;
    property Key: string read FKey write FKey;

    property ValueBool[const Name: string]: Boolean
        read ReadBool write WriteBool;
    property ValueCurrency[const Name: string]: Currency
        read ReadCurrency write WriteCurrency;
    property ValueDate[const Name: string]: TDateTime
        read ReadDate write WriteDate;
    property ValueDateTime[const Name: string]: TDateTime
        read ReadDateTime write WriteDateTime;
    property ValueFloat[const Name: string]: Double
        read ReadFloat write WriteFloat;
    property ValueInteger[const Name: string]: Integer
        read ReadInteger write WriteInteger;
    property ValueString[const Name: string]: string
        read ReadString write WriteString;
    property ValueTime[const Name: string]: TDateTime
        read ReadTime write WriteTime;

    property ValueSmallint[const Name: string]: Smallint
        read ReadSmallint write WriteSmallint;
    property ValueSingle[const Name: string]: Single
        read ReadSingle write WriteSingle;
    property ValueSingleDateTime[const Name: string]: TDateTime
        read ReadSingleDateTime write WriteSingleDateTime;

    procedure RenameValue(const OldName, NewName: string);
    function GetDataType(const Name: string): TRegDataType;
    function GetDataSize(const Name: string): Integer;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
  end;

implementation

uses
  Consts;

constructor TRegInfo.Create;
begin
  inherited;
  FRegistry := TRegistry.Create;
  FReadOnly := True;
end;

destructor TRegInfo.Destroy;
begin
  FRegistry.Free;
  inherited;
end;

procedure TRegInfo.RaiseReadOnly;
begin
  if FReadOnly then
    raise Exception.Create(SReadOnlyProperty);
end;

procedure TRegInfo.WriteString;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteString(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteBool;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteBool(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteCurrency;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteCurrency(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteDate;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteDate(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteDateTime;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteDateTime(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteFloat;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteFloat(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteInteger;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteInteger(Name, Value);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteTime;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, True);
  Reg.WriteTime(Name, Value);
  Reg.CloseKey;
end; // procedure


function TRegInfo.ReadString;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadString(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadBool;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadBool(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadCurrency;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadCurrency(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadDate;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadDate(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadDateTime;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadDateTime(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadFloat;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadFloat(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadInteger;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadInteger(Name);
  Reg.CloseKey;
end;

function TRegInfo.ReadTime;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadTime(Name);
  Reg.CloseKey;
end;

procedure TRegInfo.RenameValue(const OldName, NewName: string);
begin
  Reg.OpenKey(Key, not ReadOnly);
  Reg.RenameValue(OldName, NewName);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteBinaryData;
begin
  RaiseReadOnly;
  Reg.OpenKey(Key, not ReadOnly);
  Reg.WriteBinaryData(Name, Buffer, BufSize);
  Reg.CloseKey;
end;

procedure TRegInfo.WriteSingle;
begin
  RaiseReadOnly;
  WriteBinaryData(Name, Value, sizeof(Value));
end;

procedure TRegInfo.WriteShortint;
begin
  RaiseReadOnly;
  WriteBinaryData(Name, Value, sizeof(Value));
end; // procedure

procedure TRegInfo.WriteSmallint;
begin
  RaiseReadOnly;
  WriteBinaryData(Name, Value, sizeof(Value));
end; // procedure

procedure TRegInfo.WriteLongint;
begin
  RaiseReadOnly;
  WriteBinaryData(Name, Value, sizeof(Value));
end; // procedure

procedure TRegInfo.WriteByte;
begin
  RaiseReadOnly;
  WriteBinaryData(Name, Value, sizeof(Value));
end; // procedure

procedure TRegInfo.WriteWord;
begin
  RaiseReadOnly;
  WriteBinaryData(Name, Value, sizeof(Value));
end; // procedure


procedure TRegInfo.WriteSingleDateTime;
var
  x: Single;
begin
  RaiseReadOnly;
  x := Value;
  WriteBinaryData(Name, x, sizeof(x));
end;

function TRegInfo.ReadBinaryData;
begin
  Reg.OpenKey(Key, not ReadOnly);
  Result := Reg.ReadBinaryData(Name, Buffer, BufSize);
  Reg.CloseKey;
end;

function TRegInfo.ReadShortint;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end; // function

function TRegInfo.ReadSmallint;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end; // function

function TRegInfo.ReadLongint;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end; // function

function TRegInfo.ReadByte;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end; // function

function TRegInfo.ReadWord;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end; // function

function TRegInfo.ReadSingle;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end;

function TRegInfo.ReadSingleDateTime;
begin
  ReadBinaryData(Name, Result, sizeof(Result));
end;

function TRegInfo.GetDataSize;
begin
  Reg.OpenKey(Name, not ReadOnly);
  Result := Reg.GetDataSize(Name);
  Reg.CloseKey;
end; // function

function TRegInfo.GetDataType;
begin
  Reg.OpenKey(Name, not ReadOnly);
  Result := Reg.GetDataType(Name);
  Reg.CloseKey;
end; // function

end.
