unit CuDTEdit;
{
  This unit is a modified version of the TDateTimePicker implementation.

  Esta unidiad es una version modificada de la implementacion del
  TDateTimePicker
}
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, CommCtrl;

type

  TCustomDateTimeEdit = class;

  TDateTimeEditColors = class(TPersistent)
  private
    Owner: TCustomDateTimeEdit;
    FBackColor: TColor;
    FTextColor: TColor;
    FTitleBackColor: TColor;
    FTitleTextColor: TColor;
    FMonthBackColor: TColor;
    FTrailingTextColor: TColor;
    procedure SetColor(Index: Integer; Value: TColor);
    procedure SetAllColors;
  public
    constructor Create(AOwner: TCustomDateTimeEdit);
    procedure Assign(Source: TPersistent); override;
  published
    property BackColor: TColor index 0 read FBackColor write SetColor default clWindow;
    property TextColor: TColor index 1 read FTextColor write SetColor default clWindowText;
    property TitleBackColor: TColor index 2 read FTitleBackColor write SetColor default clActiveCaption;
    property TitleTextColor: TColor index 3 read FTitleTextColor write SetColor default clWhite;
    property MonthBackColor: TColor index 4 read FMonthBackColor write SetColor default clWhite;
    property TrailingTextColor: TColor index 5read FTrailingTextColor
      write SetColor default clInactiveCaptionText;
  end;

  TCustomDateTimeEdit = class(TWinControl)
  private
    FCalAlignment: TDTCalAlignment;
    FCalColors: TDateTimeEditColors;
    FChecked: Boolean;
    FDateTime: TDateTime;
    FDateFormat: TDTDateFormat;
    FDateMode: TDTDateMode;
    FKind: TDateTimeKind;
    FParseInput: Boolean;
    FMaxDate: TDate;
    FMinDate: TDate;
    FShowCheckbox: Boolean;
    FOnUserInput: TDTParseInputEvent;
    FOnCloseUp: TNotifyEvent;
    FOnChange: TNotifyEvent;
    FOnDropDown: TNotifyEvent;
    procedure AdjustHeight;
    function GetDate: TDate;
    function GetTime: TTime;
    procedure SetCalAlignment(Value: TDTCalAlignment);
    procedure SetCalColors(Value: TDateTimeEditColors);
    procedure SetChecked(Value: Boolean);
    procedure SetDate(Value: TDate);
    procedure SetDateMode(Value: TDTDateMode);
    procedure SetDateFormat(Value: TDTDateFormat);
    procedure SetDateTime(Value: TDateTime);
    procedure SetKind(Value: TDateTimeKind);
    procedure SetParseInput(Value: Boolean);
    procedure SetMaxDate(Value: TDate);
    procedure SetMinDate(Value: TDate);
    procedure SetRange(MinVal, MaxVal: TDateTime);
    procedure SetShowCheckbox(Value: Boolean);
    procedure SetTime(Value: TTime);
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;

  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure Change; dynamic;
    procedure DropDown; dynamic;
    procedure CloseUp; dynamic;
    procedure UserInput(const UserString: string; var DateAndTime: TDateTime;
      var AllowChange: Boolean); dynamic;

    property CalAlignment: TDTCalAlignment read FCalAlignment write SetCalAlignment;
    property CalColors: TDateTimeEditColors read FCalColors write SetCalColors;
    // The Date, Time, ShowCheckbox, and Checked properties must be in this order:
    property Date: TDate read GetDate write SetDate;
    property Time: TTime read GetTime write SetTime;
    property ShowCheckbox: Boolean read FShowCheckbox write SetShowCheckbox default False;
    property Checked: Boolean read FChecked write SetChecked default True;
    property DateTime: TDateTime read FDateTime write SetDateTime;
    property Color stored True default clWindow;
    property DateFormat: TDTDateFormat read FDateFormat write SetDateFormat;
    property DateMode: TDTDateMode read FDateMode write SetDateMode;
    property Kind: TDateTimeKind read FKind write SetKind;
    property MaxDate: TDate read FMaxDate write SetMaxDate;
    property MinDate: TDate read FMinDate write SetMinDate;
    property ParseInput: Boolean read FParseInput write SetParseInput;
    property ParentColor default False;
    property TabStop default True;
    property OnCloseUp: TNotifyEvent read FOnCloseUp write FOnCloseUp;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnDropDown: TNotifyEvent read FOnDropDown write FOnDropDown;
    property OnUserInput: TDTParseInputEvent read FOnUserInput write FOnUserInput;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published

  end;

implementation
uses
  Consts, ComStrs;

{ TDateTimeEditColors }

const
  ColorIndex: array[0..5] of Integer = (MCSC_BACKGROUND, MCSC_TEXT,
    MCSC_TITLEBK, MCSC_TITLETEXT, MCSC_MONTHBK, MCSC_TRAILINGTEXT);

constructor TDateTimeEditColors.Create(AOwner: TCustomDateTimeEdit);
begin
  Owner := AOwner;
  FBackColor := clWindow;
  FTextColor := clWindowText;
  FTitleBackColor := clActiveCaption;
  FTitleTextColor := clWhite;
  FMonthBackColor := clWhite;
  FTrailingTextColor := clInactiveCaptionText;
end;

procedure TDateTimeEditColors.Assign(Source: TPersistent);
var
  SourceName: string;
begin
  if Source = nil then SourceName := 'nil'
  else SourceName := Source.ClassName;
  if (Source = nil) or not (Source is TDateTimeEditColors) then
    raise EConvertError.CreateFmt(SAssignError, [SourceName, ClassName]);
  FBackColor := TDateTimeEditColors(Source).BackColor;
  FTextColor := TDateTimeEditColors(Source).TextColor;
  FTitleBackColor := TDateTimeEditColors(Source).TitleBackColor;
  FTitleTextColor := TDateTimeEditColors(Source).TitleTextColor;
  FMonthBackColor := TDateTimeEditColors(Source).MonthBackColor;
  FTrailingTextColor := TDateTimeEditColors(Source).TrailingTextColor;
end;

procedure TDateTimeEditColors.SetColor(Index: Integer; Value: TColor);
begin
  DateTime_SetMonthCalColor(Owner.Handle, ColorIndex[Index], ColorToRGB(Value));
  case Index of
    0: FBackColor := Value;
    1: FTextColor := Value;
    2: FTitleBackColor := Value;
    3: FTitleTextColor := Value;
    4: FMonthBackColor := Value;
    5: FTrailingTextColor := Value;
  end;
end;

procedure TDateTimeEditColors.SetAllColors;
begin
  SetColor(0, FBackColor);
  SetColor(1, FTextColor);
  SetColor(2, FTitleBackColor);
  SetColor(3, FTitleTextColor);
  SetColor(4, FMonthBackColor);
  SetColor(5, FTrailingTextColor);
end;

{ TCustomDateTimeEdit }
constructor TCustomDateTimeEdit.Create(AOwner: TComponent);
begin
  CheckCommonControl(ICC_DATE_CLASSES);
  FCalColors := TDateTimeEditColors.Create(Self);
  FDateTime := Now;
  FShowCheckbox := False;
  FChecked := True;
  inherited Create(AOwner);
  ControlStyle := [csOpaque, csClickEvents, csDoubleClicks, csFixedHeight,
    csReflector];
  Color := clWindow;
  ParentColor := False;
  TabStop := True;
  Width := 186;
  AdjustHeight;
end;

destructor TCustomDateTimeEdit.Destroy;
begin
  FCalColors.Free;
  inherited Destroy;
end;

procedure TCustomDateTimeEdit.CreateParams(var Params: TCreateParams);
const
  Formats: array[TDTDateFormat] of Integer = (DTS_SHORTDATEFORMAT,
    DTS_LONGDATEFORMAT);
begin
  inherited CreateParams(Params);
  CreateSubClass(Params, DATETIMEPICK_CLASS);
  with Params do
  begin
    Style := Style or Formats[FDateFormat];
    if FDateMode = dmUpDown then Style := Style or DTS_UPDOWN;
    if FKind = dtkTime then Style := Style or DTS_TIMEFORMAT;
    if FCalAlignment = dtaRight then Style := Style or DTS_RIGHTALIGN;
    if FParseInput then Style := Style or DTS_APPCANPARSE;
    if FShowCheckbox then Style := Style or DTS_SHOWNONE;
    WindowClass.Style := WindowClass.Style and not (CS_HREDRAW or CS_VREDRAW) or
      CS_DBLCLKS;
  end;
end;

procedure TCustomDateTimeEdit.CreateWnd;
begin
  inherited CreateWnd;
  SetDateTime(FDateTime);
  FCalColors.SetAllColors;
  SetChecked(FChecked);
end;

procedure TCustomDateTimeEdit.CMColorChanged(var Message: TMessage);
begin
  inherited;
  InvalidateRect(Handle, nil, True);
end;

procedure TCustomDateTimeEdit.CMFontChanged(var Message: TMessage);
begin
  inherited;
  AdjustHeight;
  InvalidateRect(Handle, nil, True);
end;

procedure TCustomDateTimeEdit.CloseUp;
begin
  if Assigned(FOnCloseUp) then FOnCloseUp(Self);
end;

procedure TCustomDateTimeEdit.DropDown;
begin
  if Assigned(FOnDropDown) then FOnDropDown(Self);
end;

procedure TCustomDateTimeEdit.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TCustomDateTimeEdit.UserInput(const UserString: string;
  var DateAndTime: TDateTime; var AllowChange: Boolean);
begin
//  if Assigned(FOnUserInput) then
    FOnUserInput(Self, UserString, DateAndTime, AllowChange);
end;

procedure TCustomDateTimeEdit.CNNotify(var Message: TWMNotify);

  function IsBlankSysTime(ST: TSystemTime): Boolean;
  begin
    with ST do
      Result := (wYear = 0) and (wMonth = 0) and (wDayOfWeek = 0) and
        (wDay = 0) and (wHour = 0) and (wMinute = 0) and (wSecond = 0) and
        (wMilliseconds = 0);
  end;

var
  DT: TDateTime;
  AllowChange: Boolean;
begin
  with Message, Message.NMHdr^ do
  begin
    Result := 0;
    case code of
      DTN_CLOSEUP:
        begin
          SetDateTime(FDateTime);
          CloseUp;
        end;
      DTN_DATETIMECHANGE:
        begin
          if FShowCheckbox and IsBlankSysTime(PNMDateTimeChange(NMHdr)^.st) then
            FChecked := False
          else begin
            DT := SystemTimeToDateTime(PNMDateTimeChange(NMHdr)^.st);
            if Kind = dtkDate then SetDate(DT)
            else SetTime(DT);
            if FShowCheckbox then FChecked := True;
          end;
          Change;
        end;
      DTN_DROPDOWN:
        DropDown;
      DTN_USERSTRING:
        begin
          AllowChange := Assigned(FOnUserInput);
          with PNMDateTimeString(NMHdr)^ do
          begin
            if AllowChange then
            begin
              DT := 0.0;
              UserInput(pszUserString, DT, AllowChange);
              DateTimeToSystemTime(DT, st);
            end;
            dwFlags := Ord(not AllowChange);
          end;
        end;
    else
      inherited;
    end;
  end;
end;

procedure TCustomDateTimeEdit.AdjustHeight;
var
  DC: HDC;
  SaveFont: HFont;
  SysMetrics, Metrics: TTextMetric;
begin
  DC := GetDC(0);
  try
    GetTextMetrics(DC, SysMetrics);
    SaveFont := SelectObject(DC, Font.Handle);
    GetTextMetrics(DC, Metrics);
    SelectObject(DC, SaveFont);
  finally
    ReleaseDC(0, DC);
  end;
  Height := Metrics.tmHeight + (GetSystemMetrics(SM_CYBORDER) * 8);
end;

function TCustomDateTimeEdit.GetDate: TDate;
begin
  Result := TDate(FDateTime);
end;

function TCustomDateTimeEdit.GetTime: TTime;
begin
  Result := TTime(FDateTime);
end;

procedure TCustomDateTimeEdit.SetCalAlignment(Value: TDTCalAlignment);
begin
  if FCalAlignment <> Value then
  begin
    FCalAlignment := Value;
    if not (csDesigning in ComponentState) then RecreateWnd;
  end;
end;

procedure TCustomDateTimeEdit.SetCalColors(Value: TDateTimeEditColors);
begin
  if FCalColors <> Value then FCalColors.Assign(Value);
end;

procedure TCustomDateTimeEdit.SetChecked(Value: Boolean);
var
  ST: TSystemTime;
begin
  FChecked := Value;
  if FShowCheckbox then
  begin
    if Value then SetDateTime(FDateTime)
    else DateTime_SetSystemTime(Handle, GDT_NONE, ST);
    Invalidate;
  end;
end;

procedure TCustomDateTimeEdit.SetDate(Value: TDate);
begin
  if Trunc(FDateTime) <> Trunc(Value) then
  begin
    Value := Trunc(Value) + Frac(FDateTime);
    if Value = 0.0 then
    begin
      if not FShowCheckbox then raise EDateTimeError.Create(SNeedAllowNone);
      FChecked := False;
      Invalidate;
    end
    else begin
      try
        if (FMaxDate <> 0.0) and (Value > FMaxDate) then
          raise EDateTimeError.CreateFmt(SDateTimeMax, [DateToStr(FMaxDate)]);
        if (FMinDate <> 0.0) and (Value < FMinDate) then
          raise EDateTimeError.CreateFmt(SDateTimeMin, [DateToStr(FMinDate)]);
        SetDateTime(Value);
      except
        SetDateTime(FDateTime);
        raise;
      end;
    end;
  end;
end;

procedure TCustomDateTimeEdit.SetDateTime(Value: TDateTime);
var
  ST: TSystemTime;
begin
  DateTimeToSystemTime(Value, ST);
  if DateTime_SetSystemTime(Handle, GDT_VALID, ST) then
    FDateTime := Value;
end;

procedure TCustomDateTimeEdit.SetMaxDate(Value: TDate);
begin
  if Value < FMinDate then
    raise EDateTimeError.CreateFmt(SDateTimeMin, [DateToStr(FMinDate)]);
  if FMaxDate <> Value then
  begin
    SetRange(FMinDate, Value);
    FMaxDate := Value;
  end;
end;

procedure TCustomDateTimeEdit.SetMinDate(Value: TDate);
begin
  if Value > FMaxDate then
    raise EDateTimeError.CreateFmt(SDateTimeMin, [DateToStr(FMaxDate)]);
  if FMinDate <> Value then
  begin
    SetRange(Value, FMaxDate);
    FMinDate := Value;
  end;
end;

procedure TCustomDateTimeEdit.SetRange(MinVal, MaxVal: TDateTime);
var
  STA: packed array[1..2] of TSystemTime;
  Flags: DWORD;
begin
  Flags := 0;
  if Double(MinVal) <> 0.0 then
  begin
    Flags := Flags or GDTR_MIN;
    DateTimeToSystemTime(MinVal, STA[1]);
  end;
  if Double(MaxVal) <> 0.0 then
  begin
    Flags := Flags or GDTR_MIN;
    DateTimeToSystemTime(MaxVal, STA[2]);
  end;
  if Flags <> 0 then DateTime_SetRange(Handle, Flags, @STA[1]);
end;

procedure TCustomDateTimeEdit.SetDateFormat(Value: TDTDateFormat);
begin
  if FDateFormat <> Value then
  begin
    FDateFormat := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDateTimeEdit.SetDateMode(Value: TDTDateMode);
begin
  if FDateMode <> Value then
  begin
    FDateMode := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDateTimeEdit.SetKind(Value: TDateTimeKind);
begin
  if FKind <> Value then
  begin
    FKind := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDateTimeEdit.SetParseInput(Value: Boolean);
begin
  if FParseInput <> Value then
  begin
    FParseInput := Value;
    if not (csDesigning in ComponentState) then RecreateWnd;
  end;
end;

procedure TCustomDateTimeEdit.SetShowCheckbox(Value: Boolean);
begin
  if FShowCheckbox <> Value then
  begin
    FShowCheckbox := Value;
    RecreateWnd;
  end;
end;

procedure TCustomDateTimeEdit.SetTime(Value: TTime);
begin
  if Frac(FDateTime) <> Frac(Value) then
  begin
    Value := Trunc(FDateTime) + Frac(Value);
    if Value = 0.0 then
    begin
      if not FShowCheckbox then raise EDateTimeError.Create(SNeedAllowNone);
      FChecked := False;
      Invalidate;
    end
    else
      SetDateTime(Value);
  end;
end;

end.
