(*////////////////////////////////////////////////////////////////////////////
//   Part of AlexSoft VCL/DLL Library.                                      //
//   All rights reserved. (c) Copyright 1998.                               //
//   Created by: Alex Rabichooc                                             //
//**************************************************************************//
//  Users of this unit must accept this disclaimer of warranty:             //
//    "This unit is supplied as is. The author disclaims all warranties,    //
//    expressed or implied, including, without limitation, the warranties   //
//    of merchantability and of fitness for any purpose.                    //
//    The author assumes no liability for damages, direct or                //
//    consequential, which may result from the use of this unit."           //
//                                                                          //
//  This Unit is donated to the public as public domain.                    //
//                                                                          //
//  This Unit can be freely used and distributed in commercial and          //
//  private environments provided this notice is not modified in any way.   //
//                                                                          //
//  If you do find this Unit handy and you feel guilty for using such a     //
//  great product without paying someone - sorry :-)                        //
//                                                                          //
//  Please forward any comments or suggestions to Alex Rabichooc at:        //
//                                                                          //
//  a_rabichooc@yahoo.com or alex@carmez.mldnet.com                         //
/////////////////////////////////////////////////////////////////////////////*)

unit ImgEdt;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, ExtCtrls, Menus, ComCtrls, ClipBrd,
  ToolWin, Db, DBTables, DBCtrls, RichEdit, DBxNav{$IFNDEF VER100}, ImgList{$ENDIF};

type
  TImageEditor = class(TForm)
    MainMenu: TMainMenu;
    FileOpenItem: TMenuItem;
    FileSaveItem: TMenuItem;
    FileExitItem: TMenuItem;
    EditUndoItem: TMenuItem;
    EditCutItem: TMenuItem;
    EditCopyItem: TMenuItem;
    EditPasteItem: TMenuItem;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    StatusBar: TStatusBar;
    ToolBar: TToolBar;
    OpenButton: TToolButton;
    SaveButton: TToolButton;
    ToolButton5: TToolButton;
    UndoButton: TToolButton;
    CutButton: TToolButton;
    CopyButton: TToolButton;
    PasteButton: TToolButton;
    ToolbarImages: TImageList;
    DataSource: TDataSource;
    Editor: TDBImage;
    N3: TMenuItem;
    EmptyButton: TToolButton;
    procedure FormCreate(Sender: TObject);
    procedure ShowHint(Sender: TObject);
    procedure FileOpen(Sender: TObject);
    procedure FileSave(Sender: TObject);
    procedure FileSaveAs(Sender: TObject);
    procedure FileExit(Sender: TObject);
    procedure EditUndo(Sender: TObject);
    procedure EditCut(Sender: TObject);
    procedure EditCopy(Sender: TObject);
    procedure EditPaste(Sender: TObject);
    procedure HelpContents(Sender: TObject);
    procedure HelpSearch(Sender: TObject);
    procedure HelpHowToUse(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure DataSourceDataChange(Sender: TObject; Field: TField);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure EmptyButtonClick(Sender: TObject);
  private
    FFieldName: string;
    FClipboardOwner: HWnd;
    FOnIdle: TIdleEvent;
    FOnHint: TNotifyEvent;
    procedure OnIdle(Sender: TObject; var Done: Boolean);
    procedure ClipboardChanged;
    procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DROPFILES;
    procedure WMChangeCBChain(var Msg: TWMChangeCBChain); message WM_CHANGECBCHAIN;
    procedure WMDrawClipboard(var Msg: TWMDrawClipboard); message WM_DRAWCLIPBOARD;
    procedure PerformFileOpen(const AFileName: string);
    procedure SetModified(Value: Boolean);
    procedure ReadStatusKeys;
  public
    constructor CreateWithField(AOwner: TComponent; AField: TField); virtual;
  end;

var
  ImageEditor: TImageEditor;

implementation

uses ShellAPI, Printers, DbXCnsts;

{$R *.DFM}

constructor TImageEditor.CreateWithField(AOwner: TComponent; AField: TField);
begin
   Create(AOwner);
   with Editor do
   if AField <> nil then
   begin
      DataSource.DataSet := AField.DataSet;
      DataField := AField.FieldName;
   end;
end;

procedure TImageEditor.OnIdle(Sender: TObject; var Done: Boolean);
begin
   if Assigned(FOnIdle) then
      FOnIdle(Sender, Done);
   ReadStatusKeys;
end;

procedure TImageEditor.ReadStatusKeys;
var PanelID : Byte;
    Text    : String;
    Key: Word;
begin
  Key := 0;
  for PanelID := 0 to 2 do
  begin
     case PanelID of
       0: begin
             Text := 'NUM';
             Key := vk_NumLock;
          end;
       1: begin
             Text := 'CAPS';
             Key := vk_Capital;
          end;
       2: begin
             Text := 'SCR';
             Key := vk_Scroll;
          end;
     end;
     if GetKeyState(Key) MOD 2 = 0 then
            Text := '';
     StatusBar.Panels.Items[PanelID].Text:=Text;
  end;
end;

{ Event Handlers }

procedure TImageEditor.FormCreate(Sender: TObject);
var ParentForm: TCustomForm;
begin
  if (Owner is TWinControl) then
  begin
     ParentForm := GetParentForm(Owner as TWinControl);
     if ParentForm <> nil then
       Font := ParentForm.Font;
  end;
  FOnHint := Application.OnHint;
  FOnIdle := Application.OnIdle;
  Application.OnHint := ShowHint;
  Application.OnIdle := OnIdle;
  OpenDialog.InitialDir := ExtractFilePath(ParamStr(0));
  SaveDialog.InitialDir := OpenDialog.InitialDir;
  FClipboardOwner := SetClipboardViewer(Handle);
  DataSourceDataChange(Self, Editor.Field);
end;

procedure TImageEditor.ShowHint(Sender: TObject);
begin
  if Length(Application.Hint) > 0 then
  begin
    StatusBar.SimplePanel := True;
    StatusBar.SimpleText := Application.Hint;
  end
  else StatusBar.SimplePanel := False;
end;

procedure TImageEditor.PerformFileOpen(const AFileName: string);
begin
  if (Editor.Field <> nil) and Editor.Field.DataSet.CanModify then
     Editor.Field.DataSet.Edit
    else
     exit;
  Editor.Picture.LoadFromFile(AFileName);
end;

procedure TImageEditor.FileOpen(Sender: TObject);
begin
  if (Editor.Field = nil) or not Editor.Field.DataSet.CanModify then
     exit;
  if OpenDialog.Execute then
  begin
    PerformFileOpen(OpenDialog.FileName);
    Editor.ReadOnly := (ofReadOnly in OpenDialog.Options) or
                                Editor.Field.ReadOnly;
  end;
end;

procedure TImageEditor.FileSave(Sender: TObject);
begin
  if SaveDialog.Execute then
  begin
     if FileExists(SaveDialog.FileName) then
       if MessageDlg(Format(SOverWriteConfirm,
          [SaveDialog.FileName]), mtConfirmation, mbYesNoCancel, 0) <> idYes then
       Exit;
     Editor.Picture.SaveToFile(SaveDialog.FileName);
  end;
end;

procedure TImageEditor.FileSaveAs(Sender: TObject);
begin
  FileSave(Sender);
end;

procedure TImageEditor.FileExit(Sender: TObject);
begin
  Close;
end;

procedure TImageEditor.EditUndo(Sender: TObject);
begin
  if Editor.Field <> nil then
      Editor.Field.DataSet.Cancel;
end;

procedure TImageEditor.EditCut(Sender: TObject);
begin
  Editor.CutToClipboard;
end;

procedure TImageEditor.EditCopy(Sender: TObject);
begin
  Editor.CopyToClipboard;
end;

procedure TImageEditor.EditPaste(Sender: TObject);
begin
  if (Editor.Field <> nil) and Editor.Field.DataSet.CanModify then
     Editor.Field.DataSet.Edit
    else
     exit;
  Editor.PasteFromClipboard;
end;

procedure TImageEditor.HelpContents(Sender: TObject);
begin
  Application.HelpCommand(HELP_CONTENTS, 0);
end;

procedure TImageEditor.HelpSearch(Sender: TObject);
const
  EmptyString: PChar = '';
begin
  Application.HelpCommand(HELP_PARTIALKEY, Longint(EmptyString));
end;

procedure TImageEditor.HelpHowToUse(Sender: TObject);
begin
  Application.HelpCommand(HELP_HELPONHELP, 0);
end;

{ Ruler Indent Dragging }
procedure TImageEditor.FormShow(Sender: TObject);
begin
  DragAcceptFiles(Handle, True);
  SetModified(False);
  ClipboardChanged;

  // check if we should load a file from the command line
  if (ParamCount > 0) and FileExists(ParamStr(1)) then
    PerformFileOpen(ParamStr(1));
  FFieldName := '';
  if Editor.Field <> nil then
    FFieldName := Editor.Field.DisplayLabel;
   if Length(FFieldName) > 0 then
     Caption := Format('%s - %s', [FFieldName, Application.Title]);
end;

procedure TImageEditor.WMDropFiles(var Msg: TWMDropFiles);
var
  CFileName: array[0..MAX_PATH] of Char;
begin
  try
    if DragQueryFile(Msg.Drop, 0, CFileName, MAX_PATH) > 0 then
    begin
      PerformFileOpen(CFileName);
      Msg.Result := 0;
    end;
  finally
    DragFinish(Msg.Drop);
  end;
end;

procedure TImageEditor.SetModified(Value: Boolean);
begin
  if Value then StatusBar.Panels[3].Text := SChanged
  else StatusBar.Panels[3].Text := '';
end;

procedure TImageEditor.WMChangeCBChain(var Msg: TWMChangeCBChain);
begin
  if Msg.Remove = FClipboardOwner then FClipboardOwner := Msg.Next
  else SendMessage(FClipboardOwner, WM_CHANGECBCHAIN, Msg.Remove, Msg.Next);
  Msg.Result := 0;
end;

procedure TImageEditor.ClipboardChanged;
begin
  PasteButton.Enabled := OpenButton.Enabled;
  EditPasteItem.Enabled := PasteButton.Enabled;
end;

procedure TImageEditor.WMDrawClipboard(var Msg: TWMDrawClipboard);
begin
  SendMessage(FClipboardOwner, WM_DRAWCLIPBOARD, 0, 0);
  Msg.Result := 0;
  ClipboardChanged;
end;

procedure TImageEditor.FormDestroy(Sender: TObject);
begin
  // remove ourselves from the viewer chain
  ChangeClipboardChain(Handle, FClipboardOwner);
  Application.OnIdle := FOnIdle;
  Application.OnHint := FOnHint;
end;

procedure TImageEditor.DataSourceDataChange(Sender: TObject; Field: TField);
begin
   UndoButton.Enabled := (DataSource.DataSet <> nil) and
                                     (DataSource.DataSet.State <> dsBrowse);
   EditUndoItem.Enabled := UndoButton.Enabled;
   SetModified(EditUndoItem.Enabled);
   OpenButton.Enabled := (DataSource.DataSet <> nil) and
                          DataSource.DataSet.CanModify and not Editor.ReadOnly;
   FileOpenItem.Enabled := OpenButton.Enabled;
   CutButton.Enabled := OpenButton.Enabled;
   EditCutItem.Enabled := OpenButton.Enabled;
   EmptyButton.Enabled := OpenButton.Enabled;
end;

procedure TImageEditor.FormKeyPress(Sender: TObject; var Key: Char);
begin
   if Key = Char(VK_ESCAPE) then
   begin
      if UndoButton.Enabled then
         UndoButton.Click;
      Key := #0;
   end;
end;

procedure TImageEditor.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
   if UndoButton.Enabled then DataSource.DataSet.Post;
end;

procedure TImageEditor.EmptyButtonClick(Sender: TObject);
begin
  if (Editor.Field <> nil) and Editor.Field.DataSet.CanModify then
     Editor.Field.DataSet.Edit
    else
     exit;
  Editor.Field.Value := NULL;
end;

end.
