(*////////////////////////////////////////////////////////////////////////////
//   Part of AlexSoft VCL/DLL Library.                                      //
//   All rights reserved. (c) Copyright 1998.                               //
//   Created by: Alex Rabichooc                                             //
//**************************************************************************//
//  Users of this unit must accept this disclaimer of warranty:             //
//    "This unit is supplied as is. The author disclaims all warranties,    //
//    expressed or implied, including, without limitation, the warranties   //
//    of merchantability and of fitness for any purpose.                    //
//    The author assumes no liability for damages, direct or                //
//    consequential, which may result from the use of this unit."           //
//                                                                          //
//  This Unit is donated to the public as public domain.                    //
//                                                                          //
//  This Unit can be freely used and distributed in commercial and          //
//  private environments provided this notice is not modified in any way.   //
//                                                                          //
//  If you do find this Unit handy and you feel guilty for using such a     //
//  great product without paying someone - sorry :-)                        //
//                                                                          //
//  Please forward any comments or suggestions to Alex Rabichooc at:        //
//                                                                          //
//  a_rabichooc@yahoo.com or alex@carmez.mldnet.com                         //
/////////////////////////////////////////////////////////////////////////////*)

unit MemoEdit;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, ExtCtrls, Menus, ComCtrls, ClipBrd,
  ToolWin, Db, DBTables, DBCtrls, RichEdit, DBxNav{$IFNDEF VER100}, ImgList{$ENDIF};

type
  TMemoEditor = class(TForm)
    MainMenu: TMainMenu;
    FileOpenItem: TMenuItem;
    FileSaveItem: TMenuItem;
    FileSaveAsItem: TMenuItem;
    FilePrintItem: TMenuItem;
    FileExitItem: TMenuItem;
    EditUndoItem: TMenuItem;
    EditCutItem: TMenuItem;
    EditCopyItem: TMenuItem;
    EditPasteItem: TMenuItem;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    PrintDialog: TPrintDialog;
    Ruler: TPanel;
    FontDialog1: TFontDialog;
    FirstInd: TLabel;
    LeftInd: TLabel;
    RulerLine: TBevel;
    RightInd: TLabel;
    N5: TMenuItem;
    miEditFont: TMenuItem;
    StatusBar: TStatusBar;
    ToolBar: TToolBar;
    OpenButton: TToolButton;
    SaveButton: TToolButton;
    PrintButton: TToolButton;
    ToolButton5: TToolButton;
    UndoButton: TToolButton;
    CutButton: TToolButton;
    CopyButton: TToolButton;
    PasteButton: TToolButton;
    ToolButton10: TToolButton;
    FontName: TComboBox;
    FontSize: TEdit;
    ToolButton11: TToolButton;
    UpDown1: TUpDown;
    BoldButton: TToolButton;
    ItalicButton: TToolButton;
    UnderlineButton: TToolButton;
    ToolButton16: TToolButton;
    LeftAlign: TToolButton;
    CenterAlign: TToolButton;
    RightAlign: TToolButton;
    ToolButton20: TToolButton;
    BulletsButton: TToolButton;
    ToolbarImages: TImageList;
    Editor: TDBRichEdit;
    DataSource: TDataSource;
    FindDialog1: TFindDialog;
    N3: TMenuItem;
    EditFindText: TMenuItem;

    procedure SelectionChange(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure ShowHint(Sender: TObject);
    procedure FileOpen(Sender: TObject);
    procedure FileSave(Sender: TObject);
    procedure FileSaveAs(Sender: TObject);
    procedure FilePrint(Sender: TObject);
    procedure FileExit(Sender: TObject);
    procedure EditUndo(Sender: TObject);
    procedure EditCut(Sender: TObject);
    procedure EditCopy(Sender: TObject);
    procedure EditPaste(Sender: TObject);
    procedure HelpContents(Sender: TObject);
    procedure HelpSearch(Sender: TObject);
    procedure HelpHowToUse(Sender: TObject);
    procedure SelectFont(Sender: TObject);
    procedure RulerResize(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure BoldButtonClick(Sender: TObject);
    procedure ItalicButtonClick(Sender: TObject);
    procedure FontSizeChange(Sender: TObject);
    procedure AlignButtonClick(Sender: TObject);
    procedure FontNameChange(Sender: TObject);
    procedure UnderlineButtonClick(Sender: TObject);
    procedure BulletsButtonClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure RulerItemMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure RulerItemMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FirstIndMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure LeftIndMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure RightIndMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormShow(Sender: TObject);
    procedure RichEditChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure EditorKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure EditFindTextClick(Sender: TObject);
    procedure FindDialog1Find(Sender: TObject);
    procedure DataSourceDataChange(Sender: TObject; Field: TField);
  private
    FFieldName: string;
    FUpdating: Boolean;
    FDragOfs: Integer;
    FDragging: Boolean;
    FClipboardOwner: HWnd;
    FOnIdle: TIdleEvent;
    FOnHint: TNotifyEvent;
    procedure OnIdle(Sender: TObject; var Done: Boolean);
    function CurrText: TTextAttributes;
    procedure GetFontNames;
    procedure CheckFileSave;
    procedure SetupRuler;
    procedure SetEditRect;
    procedure UpdateCursorPos;
    procedure ClipboardChanged;
    procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DROPFILES;
    procedure WMChangeCBChain(var Msg: TWMChangeCBChain); message WM_CHANGECBCHAIN;
    procedure WMDrawClipboard(var Msg: TWMDrawClipboard); message WM_DRAWCLIPBOARD;
    procedure PerformFileOpen(const AFileName: string);
    procedure SetModified(Value: Boolean);
    procedure ReadStatusKeys;
  public
    constructor CreateWithField(AOwner: TComponent; AField: TField); virtual;
  end;

var
  MemoEditor: TMemoEditor;

implementation

uses ShellAPI, DbXCnsts;

const
  RulerAdj = 4/3;
  GutterWid = 6;

{$R *.DFM}

constructor TMemoEditor.CreateWithField(AOwner: TComponent; AField: TField);
begin
   Create(AOwner);
   with Editor do
   if AField <> nil then
   begin
      DataSource.DataSet := AField.DataSet;
      DataField := AField.FieldName;
   end;
end;

procedure TMemoEditor.OnIdle(Sender: TObject; var Done: Boolean);
begin
   if Assigned(FOnIdle) then
      FOnIdle(Sender, Done);
   ReadStatusKeys;
end;

procedure TMemoEditor.ReadStatusKeys;
var PanelID : Byte;
    Text    : String;
    Key: Word;
begin
  Key := 0;
  for PanelID := 2 to 4 do
  begin
     case PanelID of
       2: begin
             Text := 'NUM';
             Key := vk_NumLock;
          end;
       3: begin
             Text := 'CAPS';
             Key := vk_Capital;
          end;
       4: begin
             Text := 'SCR';
             Key := vk_Scroll;
          end;
     end;
     if GetKeyState(Key) MOD 2 = 0 then
            Text := '';
     StatusBar.Panels.Items[PanelID].Text:=Text;
  end;
end;

procedure TMemoEditor.SelectionChange(Sender: TObject);
begin
  with Editor.Paragraph do
  try
    FUpdating := True;
    FirstInd.Left := Trunc(FirstIndent*RulerAdj)-4+GutterWid;
    LeftInd.Left := Trunc((LeftIndent+FirstIndent)*RulerAdj)-4+GutterWid;
    RightInd.Left := Ruler.ClientWidth-6-Trunc((RightIndent+GutterWid)*RulerAdj);
    BoldButton.Down := fsBold in Editor.SelAttributes.Style;
    ItalicButton.Down := fsItalic in Editor.SelAttributes.Style;
    UnderlineButton.Down := fsUnderline in Editor.SelAttributes.Style;
    BulletsButton.Down := Boolean(Numbering);
    FontSize.Text := IntToStr(Editor.SelAttributes.Size);
    FontName.Text := Editor.SelAttributes.Name;
    case Ord(Alignment) of
      0: LeftAlign.Down := True;
      1: RightAlign.Down := True;
      2: CenterAlign.Down := True;
    end;
    UpdateCursorPos;
  finally
    FUpdating := False;
  end;
end;

function TMemoEditor.CurrText: TTextAttributes;
begin
  if Editor.SelLength > 0 then Result := Editor.SelAttributes
  else Result := Editor.DefAttributes;
end;

function EnumFontsProc(var LogFont: TLogFont; var TextMetric: TTextMetric;
  FontType: Integer; Data: Pointer): Integer; stdcall;
begin
  TStrings(Data).Add(LogFont.lfFaceName);
  Result := 1;
end;

procedure TMemoEditor.GetFontNames;
var
  DC: HDC;
begin
  DC := GetDC(0);
  EnumFonts(DC, nil, @EnumFontsProc, Pointer(FontName.Items));
  ReleaseDC(0, DC);
  FontName.Sorted := True;
end;

procedure TMemoEditor.CheckFileSave;
var
  SaveResp: Integer;
begin
  if not Editor.Modified then Exit;
  SaveResp := MessageDlg(SSaveConfirm,
    mtConfirmation, mbYesNoCancel, 0);
  case SaveResp of
    idYes: FileSave(Self);
    idNo: {Nothing};
    idCancel: Abort;
  end;
end;

procedure TMemoEditor.SetupRuler;
var
  I: Integer;
  S: String;
begin
  SetLength(S, 201);
  I := 1;
  while I < 200 do
  begin
    S[I] := #9;
    S[I+1] := '|';
    Inc(I, 2);
  end;
  Ruler.Caption := S;
end;

procedure TMemoEditor.SetEditRect;
var
  R: TRect;
begin
  with Editor do
  begin
    R := Rect(GutterWid, 0, ClientWidth-GutterWid, ClientHeight);
    SendMessage(Handle, EM_SETRECT, 0, Longint(@R));
  end;
end;

{ Event Handlers }

procedure TMemoEditor.FormCreate(Sender: TObject);
var ParentForm: TCustomForm;
begin
  if (Owner is TWinControl) then
  begin
     ParentForm := GetParentForm(Owner as TWinControl);
     if ParentForm <> nil then
       Font := ParentForm.Font;
  end;
  FOnHint := Application.OnHint;
  FOnIdle := Application.OnIdle;
  Application.OnHint := ShowHint;
  Application.OnIdle := OnIdle;
  OpenDialog.InitialDir := ExtractFilePath(ParamStr(0));
  SaveDialog.InitialDir := OpenDialog.InitialDir;
  GetFontNames;
  SetupRuler;
  SelectionChange(Self);
  FClipboardOwner := SetClipboardViewer(Handle);
  StatusBar.Panels[5].Text := SInsert;
end;

procedure TMemoEditor.ShowHint(Sender: TObject);
begin
  if Length(Application.Hint) > 0 then
  begin
    StatusBar.SimplePanel := True;
    StatusBar.SimpleText := Application.Hint;
  end
  else StatusBar.SimplePanel := False;
end;

procedure TMemoEditor.PerformFileOpen(const AFileName: string);
var FRichEdit: TRichEdit;
begin
  if (Editor.Field <> nil) then
     Editor.Field.DataSet.Edit;
  FRichEdit := TRichEdit.Create(Self);
  try
    FRichEdit.Top := Editor.Top;
    FRichEdit.Width := Editor.Width;
    FRichEdit.Height := Editor.Height;
    FRichEdit.Visible := False;
    FRichEdit.Parent := Editor.Parent;
    FRichEdit.Lines.LoadFromFile(AFileName);
    FRichEdit.SelectAll;
    FRichEdit.CopyToClipboard;
    try
      FUpdating := True;
      Editor.Visible := False;
      Editor.SelectAll;
      Editor.PasteFromClipboard;
      ClipBoard.Clear;
      Editor.SelStart := 0;
      Editor.SelLength := 0;
    finally
      Editor.Visible := True;
      FUpdating := False;
    end;
  finally
    FRichEdit.Free;
  end;
  Editor.SetFocus;
end;

procedure TMemoEditor.FileOpen(Sender: TObject);
begin
  CheckFileSave;
  if OpenDialog.Execute then
  begin
    PerformFileOpen(OpenDialog.FileName);
    Editor.ReadOnly := ofReadOnly in OpenDialog.Options;
  end;
end;

procedure TMemoEditor.FileSave(Sender: TObject);
begin
  if (Editor.Field <> nil) and (Editor.Modified) and
                        (Editor.Field.DataSet.CanModify) then
  with Editor.Field.DataSet do
  begin
     Post;
     Edit;
  end;
  SetModified(False);
end;

procedure TMemoEditor.FileSaveAs(Sender: TObject);
var FRichEdit: TRichEdit;
begin
  if SaveDialog.Execute then
  begin
    if FileExists(SaveDialog.FileName) then
      if MessageDlg(Format(SOverWriteConfirm,
        [SaveDialog.FileName]), mtConfirmation, mbYesNoCancel, 0) <> idYes then
       Exit;

    FRichEdit := TRichEdit.Create(Self);
    try
      FRichEdit.Top := Editor.Top;
      FRichEdit.Width := Editor.Width;
      FRichEdit.Height := Editor.Height;
      FRichEdit.Visible := False;
      FRichEdit.Parent := Editor.Parent;
      try
        FUpdating := True;
        Editor.Visible := False;
        Editor.SelectAll;
        Editor.CopyToClipboard;
        FRichEdit.PasteFromClipBoard;
        ClipBoard.Clear;
        FRichEdit.Lines.SaveToFile(SaveDialog.FileName);
        Editor.SelStart := 0;
        Editor.SelLength := 0;
      finally
        Editor.Visible := True;
        FUpdating := False;
      end;
    finally
      FRichEdit.Free;
    end;
  end;
  Editor.SetFocus;
end;

procedure TMemoEditor.FilePrint(Sender: TObject);
begin
  if PrintDialog.Execute then
    Editor.Print(FFieldName);
end;

procedure TMemoEditor.FileExit(Sender: TObject);
begin
  Close;
end;

procedure TMemoEditor.EditUndo(Sender: TObject);
begin
  with Editor do
    if HandleAllocated then SendMessage(Handle, EM_UNDO, 0, 0);
end;

procedure TMemoEditor.EditCut(Sender: TObject);
begin
  Editor.CutToClipboard;
end;

procedure TMemoEditor.EditCopy(Sender: TObject);
begin
  Editor.CopyToClipboard;
end;

procedure TMemoEditor.EditPaste(Sender: TObject);
begin
  Editor.PasteFromClipboard;
end;

procedure TMemoEditor.HelpContents(Sender: TObject);
begin
  Application.HelpCommand(HELP_CONTENTS, 0);
end;

procedure TMemoEditor.HelpSearch(Sender: TObject);
const
  EmptyString: PChar = '';
begin
  Application.HelpCommand(HELP_PARTIALKEY, Longint(EmptyString));
end;

procedure TMemoEditor.HelpHowToUse(Sender: TObject);
begin
  Application.HelpCommand(HELP_HELPONHELP, 0);
end;

procedure TMemoEditor.SelectFont(Sender: TObject);
begin
  FontDialog1.Font.Assign(Editor.SelAttributes);
  if FontDialog1.Execute then
    CurrText.Assign(FontDialog1.Font);
  Editor.SetFocus;
end;

procedure TMemoEditor.RulerResize(Sender: TObject);
begin
  RulerLine.Width := Ruler.ClientWidth - (RulerLine.Left*2);
end;

procedure TMemoEditor.FormResize(Sender: TObject);
begin
  SetEditRect;
  SelectionChange(Sender);
end;

procedure TMemoEditor.FormPaint(Sender: TObject);
begin
  SetEditRect;
end;

procedure TMemoEditor.BoldButtonClick(Sender: TObject);
begin
  if FUpdating then Exit;
  if BoldButton.Down then
    CurrText.Style := CurrText.Style + [fsBold]
  else
    CurrText.Style := CurrText.Style - [fsBold];
end;

procedure TMemoEditor.ItalicButtonClick(Sender: TObject);
begin
  if FUpdating then Exit;
  if ItalicButton.Down then
    CurrText.Style := CurrText.Style + [fsItalic]
  else
    CurrText.Style := CurrText.Style - [fsItalic];
end;

procedure TMemoEditor.FontSizeChange(Sender: TObject);
begin
  if FUpdating then Exit;
  CurrText.Size := StrToInt(FontSize.Text);
end;

procedure TMemoEditor.AlignButtonClick(Sender: TObject);
begin
  if FUpdating then Exit;
  Editor.Paragraph.Alignment := TAlignment(TControl(Sender).Tag);
end;

procedure TMemoEditor.FontNameChange(Sender: TObject);
begin
  if FUpdating then Exit;
  CurrText.Name := FontName.Items[FontName.ItemIndex];
end;

procedure TMemoEditor.UnderlineButtonClick(Sender: TObject);
begin
  if FUpdating then Exit;
  if UnderlineButton.Down then
    CurrText.Style := CurrText.Style + [fsUnderline]
  else
    CurrText.Style := CurrText.Style - [fsUnderline];
end;

procedure TMemoEditor.BulletsButtonClick(Sender: TObject);
begin
  if FUpdating then Exit;
  Editor.Paragraph.Numbering := TNumberingStyle(BulletsButton.Down);
end;

procedure TMemoEditor.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  try
    CheckFileSave;
  except
    CanClose := False;
  end;
end;

{ Ruler Indent Dragging }

procedure TMemoEditor.RulerItemMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  FDragOfs := (TLabel(Sender).Width div 2);
  TLabel(Sender).Left := TLabel(Sender).Left+X-FDragOfs;
  FDragging := True;
end;

procedure TMemoEditor.RulerItemMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
begin
  if FDragging then
    TLabel(Sender).Left :=  TLabel(Sender).Left+X-FDragOfs
end;

procedure TMemoEditor.FirstIndMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  FDragging := False;
  Editor.Paragraph.FirstIndent := Trunc((FirstInd.Left+FDragOfs-GutterWid) / RulerAdj);
  LeftIndMouseUp(Sender, Button, Shift, X, Y);
end;

procedure TMemoEditor.LeftIndMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  FDragging := False;
  Editor.Paragraph.LeftIndent := Trunc((LeftInd.Left+FDragOfs-GutterWid) / RulerAdj)-Editor.Paragraph.FirstIndent;
  SelectionChange(Sender);
end;

procedure TMemoEditor.RightIndMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  FDragging := False;
  Editor.Paragraph.RightIndent := Trunc((Ruler.ClientWidth-RightInd.Left+FDragOfs-2) / RulerAdj)-2*GutterWid;
  SelectionChange(Sender);
end;

procedure TMemoEditor.UpdateCursorPos;
var
  CharPos: TPoint;
begin
  CharPos.Y := SendMessage(Editor.Handle, EM_EXLINEFROMCHAR, 0,
    Editor.SelStart);
  CharPos.X := (Editor.SelStart -
    SendMessage(Editor.Handle, EM_LINEINDEX, CharPos.Y, 0));
  Inc(CharPos.Y);
  Inc(CharPos.X);
  StatusBar.Panels[0].Text := Format('%4d:%-3d', [CharPos.Y, CharPos.X]);

  // update the status of the cut and copy command
  CopyButton.Enabled := Editor.SelLength > 0;
  EditCopyItem.Enabled := CopyButton.Enabled;
  CutButton.Enabled := CopyButton.Enabled;
  EditCutItem.Enabled := CopyButton.Enabled;
end;

procedure TMemoEditor.FormShow(Sender: TObject);
begin
  UpdateCursorPos;
  DragAcceptFiles(Handle, True);
  if (Editor.Field <> nil) and (Editor.Modified) and
                        (Editor.Field.DataSet.CanModify) then
  with Editor.Field.DataSet do
  begin
     if State = dsBrowse then
       if IsEmpty then
        Insert
         else
        Edit;
  end;
  RichEditChange(nil);
  SetModified(False);
  Editor.SetFocus;
  ClipboardChanged;

  // check if we should load a file from the command line
  if (ParamCount > 0) and FileExists(ParamStr(1)) then
    PerformFileOpen(ParamStr(1));
end;

procedure TMemoEditor.WMDropFiles(var Msg: TWMDropFiles);
var
  CFileName: array[0..MAX_PATH] of Char;
begin
  try
    if DragQueryFile(Msg.Drop, 0, CFileName, MAX_PATH) > 0 then
    begin
      CheckFileSave;
      PerformFileOpen(CFileName);
      Msg.Result := 0;
    end;
  finally
    DragFinish(Msg.Drop);
  end;
end;

procedure TMemoEditor.RichEditChange(Sender: TObject);
begin
  SetModified(Editor.Modified);
  UndoButton.Enabled := SendMessage(Editor.Handle, EM_CANUNDO, 0, 0) <> 0;
  EditUndoItem.Enabled := UndoButton.Enabled;
end;

procedure TMemoEditor.SetModified(Value: Boolean);
begin
  Editor.Modified := Value;
  if Value then StatusBar.Panels[1].Text := SChanged
  else StatusBar.Panels[1].Text := '';
end;

procedure TMemoEditor.WMChangeCBChain(var Msg: TWMChangeCBChain);
begin
  if Msg.Remove = FClipboardOwner then FClipboardOwner := Msg.Next
  else SendMessage(FClipboardOwner, WM_CHANGECBCHAIN, Msg.Remove, Msg.Next);
  Msg.Result := 0;
end;

procedure TMemoEditor.ClipboardChanged;
var
  I: Integer;
  Format: Word;
  E: Boolean;
begin
  // check to see if we can paste what's on the clipboard
  E := False;
  for I := 0 to Clipboard.FormatCount - 1 do
  begin
    Format := Clipboard.Formats[I];
    if SendMessage(Editor.Handle, EM_CANPASTE, Format, 0) <> 0 then
    begin
      E := True;
      Break;
    end;
  end;
  PasteButton.Enabled := E;
  EditPasteItem.Enabled := E;
end;

procedure TMemoEditor.WMDrawClipboard(var Msg: TWMDrawClipboard);
begin
  SendMessage(FClipboardOwner, WM_DRAWCLIPBOARD, 0, 0);
  Msg.Result := 0;
  ClipboardChanged;
end;

procedure TMemoEditor.FormDestroy(Sender: TObject);
begin
  // remove ourselves from the viewer chain
  ChangeClipboardChain(Handle, FClipboardOwner);
  Application.OnIdle := FOnIdle;
  Application.OnHint := FOnHint;
  if (Editor.Field <> nil) and (Editor.Field.DataSet.State <> dsBrowse) then
     Editor.Field.DataSet.Cancel;
end;

procedure TMemoEditor.EditorKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
   if Key = vk_Insert then
      if StatusBar.Panels[5].Text = SInsert then
         StatusBar.Panels[5].Text := SOverWrite
        else
         StatusBar.Panels[5].Text := SInsert;
end;

procedure TMemoEditor.EditFindTextClick(Sender: TObject);
begin
  FindDialog1.Execute;
end;

procedure TMemoEditor.FindDialog1Find(Sender: TObject);
var
  FoundAt: LongInt;
  StartPos, ToEnd: integer;
  SearchType: TSearchTypes;
begin
  SearchType := [];
  if frMatchCase in FindDialog1.Options then
    Include(SearchType, stMatchCase);
  if frWholeWord in FindDialog1.Options then
    Include(SearchType, stWholeWord);
  with Editor do
  begin
     if SelLength <> 0 then
        StartPos := SelStart+SelLength
       else
        StartPos := 0;
     ToEnd := Length(Text) - StartPos;
     FoundAt := FindText(FindDialog1.FindText, StartPos, ToEnd, SearchType);
     if FoundAt <> -1 then
     begin
       SetFocus;
       SelStart := FoundAt;
       SelLength := Length(FindDialog1.FindText);
     end
       else
         MessageDlg(SValueNotFound, mtInformation, [mbOk], 0);
  end;
end;

procedure TMemoEditor.DataSourceDataChange(Sender: TObject; Field: TField);
begin
   FFieldName := '';
   if Editor.Field <> nil then
      FFieldName := Editor.Field.DisplayLabel;
   if Length(FFieldName) > 0 then
     Caption := Format('%s - %s', [FFieldName, Application.Title]);
end;

end.
