(*////////////////////////////////////////////////////////////////////////////
//   Part of AlexSoft VCL/DLL Library.                                      //
//   All rights reserved. (c) Copyright 1998.                               //
//   Created by: Alex Rabichooc                                             //
//**************************************************************************//
//  Users of this unit must accept this disclaimer of warranty:             //
//    "This unit is supplied as is. The author disclaims all warranties,    //
//    expressed or implied, including, without limitation, the warranties   //
//    of merchantability and of fitness for any purpose.                    //
//    The author assumes no liability for damages, direct or                //
//    consequential, which may result from the use of this unit."           //
//                                                                          //
//  This Unit is donated to the public as public domain.                    //
//                                                                          //
//  This Unit can be freely used and distributed in commercial and          //
//  private environments provided this notice is not modified in any way.   //
//                                                                          //
//  If you do find this Unit handy and you feel guilty for using such a     //
//  great product without paying someone - sorry :-)                        //
//                                                                          //
//  Please forward any comments or suggestions to Alex Rabichooc at:        //
//                                                                          //
//  a_rabichooc@yahoo.com or alex@carmez.mldnet.com                         //
/////////////////////////////////////////////////////////////////////////////*)

{---------------------------------------------------------------------------
  TRaDBNavigator - Extended TDBNavigator component
     properties
       DataSourceAuto: boolean;
         Determines whether Component will changes DataSource property to
         the Active data-aware Control DataSource property
       DefaultAction: boolean;
         If you set the OnClick property, DefaultAction determines,
         whether Components will execute an action by default for this event.
         You also can call an action by default manually: DoDefaultClick.
       SearchComponent: TRaDBSearch;
         If you set SearchComponent then for pressing to a search button
         Execute method of the SearchComponent will be called.
----------------------------------------------------------------------------}
unit DBxNav;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, dbCtrls, db, Buttons, DbSearch;

type
  TxNavButton = class;
  TxNavDataLink = class;

  TxNavGlyph = (ngEnabled, ngDisabled);
  TxNavigateBtn = (nbFirst, nbPrior, nbNext, nbLast,
                  nbInsert, nbDelete, nbEdit, nbPost,
                  nbCancel, nbRefresh, nbSearch);
  TButtonSet = set of TxNavigateBtn;
  TxNavButtonStyle = set of (nsAllowTimer, nsFocusRect);

  ENavClick = procedure (Sender: TObject; Button: TxNavigateBtn) of object;

  TRaDBNavigator = class (TCustomPanel)
  private
    FSearchComponent: TRaDBSearch;
    FDataSource: TDataSource;
    FDataSourceAuto: Boolean;
    FDataLink: TxNavDataLink;
    FVisibleButtons: TButtonSet;
    FHints: TStrings;
    ButtonWidth: Integer;
    MinBtnSize: TPoint;
    FOnNavClick: ENavClick;
    FBeforeAction: ENavClick;
    FocusedButton: TxNavigateBtn;
    FConfirmDelete: Boolean;
    FFlat: Boolean;
    FDefaultAction: Boolean;
    procedure ClickHandler(Sender: TObject);
    function CurrentField: TField;
    function GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    procedure InitButtons;
    procedure InitHints;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure HintsChanged(Sender: TObject);
    procedure SetHints(Value: TStrings);
    procedure SetFlat(Value: Boolean);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure CMFocusChanged(var Message: TCMFocusChanged); message CM_FOCUSCHANGED;
    procedure SetDataSourceAuto(Value: Boolean);
    procedure ChangeDataSource;
  protected
    Buttons: array[TxNavigateBtn] of TxNavButton;
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure DoDefaultClick(Index: TxNavigateBtn);
    procedure BtnClick(Index: TxNavigateBtn);
  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DataSourceAuto: boolean read FDataSourceAuto write SetDataSourceAuto;
    property DefaultAction: boolean read FDefaultAction write FDefaultAction
                                                           default True;
    property VisibleButtons: TButtonSet read FVisibleButtons write SetVisible
      default [nbFirst, nbPrior, nbNext, nbLast, nbInsert, nbDelete,
        nbEdit, nbPost, nbCancel, nbRefresh, nbSearch];
    property SearchComponent: TRaDBSearch read FSearchComponent
                                          write FSearchComponent;
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Flat: Boolean read FFlat write SetFlat default False;
    property Ctl3D;
    property Hints: TStrings read FHints write SetHints;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property BeforeAction: ENavClick read FBeforeAction write FBeforeAction;
    property OnClick: ENavClick read FOnNavClick write FOnNavClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    property OnStartDrag;
  end;

  TxNavButton = class(TSpeedButton)
  private
    FIndex: TxNavigateBtn;
    FNavStyle: TxNavButtonStyle;
    FRepeatTimer: TTimer;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    destructor Destroy; override;
    property NavStyle: TxNavButtonStyle read FNavStyle write FNavStyle;
    property Index : TxNavigateBtn read FIndex write FIndex;
  end;

  TxNavDataLink = class(TDataLink)
  private
    FNavigator: TRaDBNavigator;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(ANav: TRaDBNavigator);
    destructor Destroy; override;
  end;

implementation

{$R DBXNAV}

uses dbConsts, dbXCnsts, dbGrids, dbCgrids, dbEdFld, dbXgrid, dbTools, dbBoxGrd;

{TRaDBNavigator}
var
  BtnTypeName: array[TxNavigateBtn] of PChar = ('FIRST', 'PRIOR', 'NEXT',
    'LAST', 'INSERT', 'DELETE', 'EDIT', 'POST', 'CANCEL', 'REFRESH', 'SEARCH');
  BtnHintId: array[TxNavigateBtn] of String = (SFirstRecord, SPriorRecord,
    SNextRecord, SLastRecord, SInsertRecord, SDeleteRecord, SEditRecord,
    SPostEdit, SCancelEdit, SRefreshRecord, SSearchRecord);

constructor TRaDBNavigator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] + [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FDataLink := TxNavDataLink.Create(Self);
  FVisibleButtons := [nbFirst, nbPrior, nbNext, nbLast, nbInsert,
    nbDelete, nbEdit, nbPost, nbCancel, nbRefresh, nbSearch];
  FHints := TStringList.Create;
  TStringList(FHints).OnChange := HintsChanged;
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 241;
  Height := 25;
  ButtonWidth := 0;
  FocusedButton := nbFirst;
  FConfirmDelete := True;
  FullRepaint := False;
  FDefaultAction := True;
end;

destructor TRaDBNavigator.Destroy;
begin
  FDataLink.Free;
  FHints.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TRaDBNavigator.InitButtons;
var
  I: TxNavigateBtn;
  Btn: TxNavButton;
  X: Integer;
  ResName: string;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TxNavButton.Create (Self);
    Btn.Flat := Flat;
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    FmtStr(ResName, 'dbxn_%s', [BtnTypeName[I]]);
    Btn.Glyph.LoadFromResourceName(HInstance, ResName);
    Btn.NumGlyphs := 2;
    Btn.Enabled := False;
    Btn.Enabled := True;
    Btn.OnClick := ClickHandler;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
  InitHints;
  Buttons[nbPrior].NavStyle := Buttons[nbPrior].NavStyle + [nsAllowTimer];
  Buttons[nbNext].NavStyle  := Buttons[nbNext].NavStyle + [nsAllowTimer];
end;

procedure TRaDBNavigator.InitHints;
var
  I: Integer;
  J: TxNavigateBtn;
begin
  for J := Low(Buttons) to High(Buttons) do
    Buttons[J].Hint := BtnHintId[J];
  J := Low(Buttons);
  for I := 0 to (FHints.Count - 1) do
  begin
    if FHints.Strings[I] <> '' then Buttons[J].Hint := FHints.Strings[I];
    if J = High(Buttons) then Exit;
    Inc(J);
  end;
end;

procedure TRaDBNavigator.HintsChanged(Sender: TObject);
begin
  InitHints;
end;

procedure TRaDBNavigator.SetFlat(Value: Boolean);
var
  I: TxNavigateBtn;
begin
  if FFlat <> Value then
  begin
    FFlat := Value;
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Flat := Value;
  end;
end;

procedure TRaDBNavigator.SetHints(Value: TStrings);
begin
  FHints.Assign(Value);
end;

procedure TRaDBNavigator.GetChildren(Proc: TGetChildProc; Root: TComponent);
begin
end;

procedure TRaDBNavigator.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
  if (Operation = opRemove) and (FDataSource <> nil) and
    (AComponent = FDataSource) then FDataSource := nil;
  if (Operation = opRemove) and (FSearchComponent <> nil) and
     (AComponent = FSearchComponent) then FSearchComponent := nil;
end;

procedure TRaDBNavigator.SetVisible(Value: TButtonSet);
var
  I: TxNavigateBtn;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TRaDBNavigator.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TxNavigateBtn;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * MinBtnSize.X;
  if W < MinW then W := MinW;
  if H < MinBtnSize.Y then H := MinBtnSize.Y;

  ButtonWidth := W div Count;
  Temp := Count * ButtonWidth;
  if Align = alNone then W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec(Temp, Remain);
        if Temp < 0 then
        begin
          Inc(Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds(X, 0, ButtonWidth + Space, Height);
      Inc(X, ButtonWidth + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TRaDBNavigator.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  if not HandleAllocated then AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TRaDBNavigator.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;
  { check for minimum size }
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TRaDBNavigator.ClickHandler(Sender: TObject);
begin
  BtnClick (TxNavButton (Sender).Index);
end;

procedure TRaDBNavigator.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TxNavigateBtn;
begin
  OldFocus := FocusedButton;
  FocusedButton := TxNavButton (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TRaDBNavigator.BtnClick(Index: TxNavigateBtn);
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then
  begin
    if not (csDesigning in ComponentState) and Assigned(FOnNavClick) then
    begin
       if FDefaultAction then
         DoDefaultClick(Index);
       FOnNavClick(Self, Index);
    end
      else
       DoDefaultClick(Index);
  end;
end;

procedure TRaDBNavigator.DoDefaultClick(Index: TxNavigateBtn);
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then
  begin
    if not (csDesigning in ComponentState) and Assigned(FBeforeAction) then
      FBeforeAction(Self, Index);
    with DataSource.DataSet do
    begin
      case Index of
        nbPrior: Prior;
        nbNext: Next;
        nbFirst: First;
        nbLast: Last;
        nbInsert: Insert;
        nbEdit: Edit;
        nbCancel: Cancel;
        nbPost: Post;
        nbRefresh: Refresh;
        nbDelete:
          if not FConfirmDelete or
            (MessageDlg(SDeleteRecordQuestion, mtConfirmation,
            mbOKCancel, 0) <> idCancel) then Delete;
        nbSearch: begin
                     if FDataLink.DataSet.Modified then
                        FDataLink.DataSet.CheckBrowseMode;
                     if FSearchComponent <> nil then
                        FSearchComponent.Execute(CurrentField)
                       else
                        DBTools.DBSearch(FDataLink.DataSet, nil,
                                         CurrentField, Owner);
                  end;
      end;
    end;
  end;
end;

procedure TRaDBNavigator.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TRaDBNavigator.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TRaDBNavigator.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TxNavigateBtn;
  OldFocus: TxNavigateBtn;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TRaDBNavigator.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TRaDBNavigator.DataChanged;
var
  UpEnable, DnEnable: Boolean;
begin
  UpEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.BOF;
  DnEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.EOF;
  Buttons[nbFirst].Enabled := UpEnable;
  Buttons[nbPrior].Enabled := UpEnable;
  Buttons[nbNext].Enabled := DnEnable;
  Buttons[nbLast].Enabled := DnEnable;
  Buttons[nbDelete].Enabled := Enabled and FDataLink.Active and
    FDataLink.DataSet.CanModify and
    not (FDataLink.DataSet.BOF and FDataLink.DataSet.EOF);
  Buttons[nbSearch].Enabled := Enabled and FDataLink.Active and
                               not FDataLink.DataSet.IsEmpty;
end;

procedure TRaDBNavigator.EditingChanged;
var
  CanModify: Boolean;
begin
  CanModify := Enabled and FDataLink.Active and FDataLink.DataSet.CanModify;
  Buttons[nbInsert].Enabled := CanModify;
  Buttons[nbEdit].Enabled := CanModify and not FDataLink.Editing;
  Buttons[nbPost].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbCancel].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbRefresh].Enabled := Enabled and FDataLink.Active;
  Buttons[nbSearch].Enabled := Enabled and FDataLink.Active and
                               not FDataLink.DataSet.IsEmpty;
end;

procedure TRaDBNavigator.ActiveChanged;
var
  I: TxNavigateBtn;
begin
  if not (Enabled and FDataLink.Active) then
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Enabled := False
  else
  begin
    DataChanged;
    EditingChanged;
  end;
end;

procedure TRaDBNavigator.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

function TRaDBNavigator.CurrentField: TField;
var ParentForm: TCustomForm;
    AControl: TControl;
begin
   Result := nil;
   ParentForm := GetParentForm(Self);
   if (ParentForm <> nil) then
      AControl := ParentForm.ActiveControl
     else
      AControl := nil;

   if (AControl <> nil) then
   begin
      if (AControl.Parent is TRaDBEdit) then
         Result := (AControl.Parent as TRaDBEdit).Field
        else
      if (AControl is TDBGrid) then
         Result := (AControl as TDBGrid).SelectedField
        else
      if (AControl is TRaDBBox) then
      begin
         if (AControl as TRaDBBox).ControlCount > 1 then
            Result := ((AControl as TRaDBBox).Controls[0] as TRaDBEdit).Field;
      end
        else
      if (AControl is TDBEdit) then
         Result := (AControl as TDBEdit).Field
        else
      if (AControl is TDBMemo) then
         Result := (AControl as TDBMemo).Field
        else
      if (AControl is TDBImage) then
         Result := (AControl as TDBImage).Field
        else
      if (AControl is TDBListBox) then
         Result := (AControl as TDBListBox).Field
        else
      if (AControl is TDBComboBox) then
         Result := (AControl as TDBComboBox).Field
        else
      if (AControl is TDBCheckBox) then
         Result := (AControl as TDBCheckBox).Field
        else
      if (AControl is TDBComboBox) then
         Result := (AControl as TDBComboBox).Field
        else
      if (AControl is TDBRadioGroup) then
         Result := (AControl as TDBRadioGroup).Field
        else
      if (AControl is TDBLookupListBox) then
         Result := (AControl as TDBLookupListBox).Field
        else
      if (AControl is TDBLookupComboBox) then
         Result := (AControl as TDBLookupComboBox).Field
        else
      if (AControl is TxDBLookupComboBox) then
         Result := (AControl as TxDBLookupComboBox).Field
        else
      if (AControl is TDBRichEdit) then
         Result := (AControl as TDBRichEdit).Field
   end;
end;

procedure TRaDBNavigator.ChangeDataSource;
var ParentForm: TCustomForm;
    AControl: TControl;
    ADataSource: TDataSource;
begin
   ADataSource := FDataSource;
   ParentForm := GetParentForm(Self);
   if (ParentForm <> nil) then
      AControl := ParentForm.ActiveControl
     else
      AControl := nil;

   if (AControl <> nil) and FDataSourceAuto then
   begin
      if (AControl.Parent is TRaDBEdit) then
         ADataSource := (AControl.Parent as TRaDBEdit).DataSource
        else
      if (AControl is TDBGrid) then
         ADataSource := (AControl as TDBGrid).DataSource
        else
      if (AControl is TRaDBBox) then
         ADataSource := (AControl as TRaDBBox).DataSource
        else
      if (AControl is TDBEdit) then
         ADataSource := (AControl as TDBEdit).DataSource
        else
      if (AControl is TDBMemo) then
         ADataSource := (AControl as TDBMemo).DataSource
        else
      if (AControl is TDBImage) then
         ADataSource := (AControl as TDBImage).DataSource
        else
      if (AControl is TDBListBox) then
         ADataSource := (AControl as TDBListBox).DataSource
        else
      if (AControl is TDBComboBox) then
         ADataSource := (AControl as TDBComboBox).DataSource
        else
      if (AControl is TDBCheckBox) then
         ADataSource := (AControl as TDBCheckBox).DataSource
        else
      if (AControl is TDBComboBox) then
         ADataSource := (AControl as TDBComboBox).DataSource
        else
      if (AControl is TDBRadioGroup) then
         ADataSource := (AControl as TDBRadioGroup).DataSource
        else
      if (AControl is TDBLookupListBox) then
         ADataSource := (AControl as TDBLookupListBox).DataSource
        else
      if (AControl is TDBLookupComboBox) then
         ADataSource := (AControl as TDBLookupComboBox).DataSource
        else
      if (AControl is TxDBLookupComboBox) then
         ADataSource := (AControl as TxDBLookupComboBox).DataSource
        else
      if (AControl is TDBRichEdit) then
         ADataSource := (AControl as TDBRichEdit).DataSource
        else
      if (AControl is TDBCtrlGrid) then
         ADataSource := (AControl as TDBCtrlGrid).DataSource;
   end;

   if (ADataSource <> FDataLink.DataSource) then
   begin
      FDataLink.DataSource := ADataSource;
      if not (csLoading in ComponentState) then
        ActiveChanged;
      if ADataSource <> nil then ADataSource.FreeNotification(Self);
   end;
end;

procedure TRaDBNavigator.CMFocusChanged(var Message: TCMFocusChanged);
begin
   Inherited;
   ChangeDataSource;
end;

procedure TRaDBNavigator.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  FDataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TRaDBNavigator.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TRaDBNavigator.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitHints;
  ActiveChanged;
end;

procedure TRaDBNavigator.SetDataSourceAuto(Value: Boolean);
begin
   FDataSourceAuto := Value;
   ChangeDataSource;
end;
{TxNavButton}

destructor TxNavButton.Destroy;
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy;
end;

procedure TxNavButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  if nsAllowTimer in FNavStyle then
  begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);

    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True;
  end;
end;

procedure TxNavButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
                                  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled  := False;
end;

procedure TxNavButton.TimerExpired(Sender: TObject);
begin
  FRepeatTimer.Interval := RepeatPause;
  if (FState = bsDown) and MouseCapture then
  begin
    try
      Click;
    except
      FRepeatTimer.Enabled := False;
      raise;
    end;
  end;
end;

procedure TxNavButton.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TRaDBNavigator (Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

{ TxNavDataLink }
constructor TxNavDataLink.Create(ANav: TRaDBNavigator);
begin
  inherited Create;
  FNavigator := ANav;
end;

destructor TxNavDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy;
end;

procedure TxNavDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged;
end;

procedure TxNavDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged;
end;

procedure TxNavDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged;
end;

end.
