(*////////////////////////////////////////////////////////////////////////////
//   Part of AlexSoft VCL/DLL Library.                                      //
//   All rights reserved. (c) Copyright 1998.                               //
//   Created by: Alex Rabichooc                                             //
//**************************************************************************//
//  Users of this unit must accept this disclaimer of warranty:             //
//    "This unit is supplied as is. The author disclaims all warranties,    //
//    expressed or implied, including, without limitation, the warranties   //
//    of merchantability and of fitness for any purpose.                    //
//    The author assumes no liability for damages, direct or                //
//    consequential, which may result from the use of this unit."           //
//                                                                          //
//  This Unit is donated to the public as public domain.                    //
//                                                                          //
//  This Unit can be freely used and distributed in commercial and          //
//  private environments provided this notice is not modified in any way.   //
//                                                                          //
//  If you do find this Unit handy and you feel guilty for using such a     //
//  great product without paying someone - sorry :-)                        //
//                                                                          //
//  Please forward any comments or suggestions to Alex Rabichooc at:        //
//                                                                          //
//  a_rabichooc@yahoo.com or alex@carmez.mldnet.com                         //
/////////////////////////////////////////////////////////////////////////////*)

{---------------------------------------------------------------------------
  TRaDBPanel - Data-aware component like TDBCtrlGrid or/and TDBGrid
               Supports Master-Detail editing.
     properties
        DataSource: TDataSource;
           Defines DataSource of Component.
        ReadOnly: Boolean;
           If this property is True then user can't modify the DataSet.
        property PanelStyle: TRaDBPanelStyle;
           psGrid - Panel is Grid
           psBox - Panel is Box
           psAuto - Panel is Grid, but if the DataSource has master DataSource
                     then Panel is both Box and Grid (Master-detail).
        property CloseOnEscape: Boolean;
           If this property is true, the user can close the Form pressing Esc.
----------------------------------------------------------------------------}

unit DBPanel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, Grids, DBGrids, DBEdFld, db, dbCtrls, dbBoxGrd, dbXGrid, StdUtils,
  Dsgnintf;

type
  TDBPanelBox = class;

  TRaDBPanelFields = class;

  TFieldsProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

  TRaDBPanelField = class(TCollectionItem)
  private
    FDBEditField: TRaDBEdit;
    function GetCollection: TRaDBPanelFields;
    function GetCursor: TCursor;
    procedure SetCursor(Value: TCursor);
    function GetHelpContext: THelpContext;
    procedure SetHelpContext(Value: THelpContext);
    function GetTag: LongInt;
    procedure SetTag(Value: LongInt);
    function GetLeft: Integer;
    procedure SetLeft(Value: Integer);
    function GetTop: Integer;
    procedure SetTop(Value: Integer);
    function GetWidth: Integer;
    procedure SetWidth(Value: Integer);
    function GetHeight: Integer;
    procedure SetHeight(Value: Integer);
    function GetHint: String;
    procedure SetHint(Value: String);
    function GetDataField: string;
    procedure SetDataField(const Value: string);
    function GetShowBlob: Boolean;
    procedure SetShowBlob(Value: Boolean);
    function GetDistanceX: Integer;
    procedure SetDistanceX(Value: Integer);
    function GetDistanceY: Integer;
    procedure SetDistanceY(Value: Integer);
    function GetMaxLength: Integer;
    procedure SetMaxLength(Value: Integer);
    function GetFont: TFont;
    procedure SetFont(Value: TFont);
    function GetLabelFont: TFont;
    procedure SetLabelFont(Value: TFont);
    function GetCaption: String;
    procedure SetCaption(Value: String);
    function GetEditKind: TEditKind;
    procedure SetEditKind(Value: TEditKind);
    function GetButtonHint: String;
    procedure SetButtonHint(Value: String);
    function GetEnabled: boolean;
    procedure SetEnabled(Value: boolean);
    function GetOnClick: TNotifyEvent;
    procedure SetOnClick(Value: TNotifyEvent);
    function GetOnButtonClick: TNotifyEvent;
    procedure SetOnButtonClick(Value: TNotifyEvent);
    function GetOnDblClick: TNotifyEvent;
    procedure SetOnDblClick(Value: TNotifyEvent);
    function GetOnDragDrop: TDragDropEvent;
    procedure SetOnDragDrop(Value: TDragDropEvent);
    function GetOnDragOver: TDragOverEvent;
    procedure SetOnDragOver(Value: TDragOverEvent);
    function GetOnEndDrag: TEndDragEvent;
    procedure SetOnEndDrag(Value: TEndDragEvent);
    function GetOnMouseDown: TMouseEvent;
    procedure SetOnMouseDown(Value: TMouseEvent);
    function GetOnMouseMove: TMouseMoveEvent;
    procedure SetOnMouseMove(Value: TMouseMoveEvent);
    function GetOnMouseUp: TMouseEvent;
    procedure SetOnMouseUp(Value: TMouseEvent);
    function GetOnStartDrag: TStartDragEvent;
    procedure SetOnStartDrag(Value: TStartDragEvent);
    function GetOnEnter: TNotifyEvent;
    procedure SetOnEnter(Value: TNotifyEvent);
    function GetOnExit: TNotifyEvent;
    procedure SetOnExit(Value: TNotifyEvent);
    function GetOnKeyDown: TKeyEvent;
    procedure SetOnKeyDown(Value: TKeyEvent);
    function GetOnKeyPress: TKeyPressEvent;
    procedure SetOnKeyPress(Value: TKeyPressEvent);
    function GetOnKeyUp: TKeyEvent;
    procedure SetOnKeyUp(Value: TKeyEvent);
    function GetTabStop: boolean;
    procedure SetTabStop(Value: boolean);
    function  GetOnReplaceField: TFieldNotifyEvent;
    procedure SetOnReplaceField(const Value: TFieldNotifyEvent);
  protected
    function GetDisplayName: string; override;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    property DBEditField: TRaDBEdit read FDBEditField;
  published
    property Cursor: TCursor read GetCursor write SetCursor;
    property HelpContext: THelpContext read GetHelpContext write SetHelpContext;
    property Tag: LongInt read GetTag write SetTag;
    property Hint: String read GetHint write SetHint;
    property ShowBlob: Boolean read GetShowBlob write SetShowBlob;
    property DataField: string read GetDataField write SetDataField;
    property MaxLength: Integer read GetMaxLength write SetMaxLength;
    property Font: TFont read GetFont write SetFont;
    property LabelFont: TFont read GetLabelFont write SetLabelFont;
    property Caption: String read GetCaption write SetCaption;
    property EditKind: TEditKind read GetEditKind write SetEditKind;
    property ButtonHint: String read GetButtonHint write SetButtonHint;
    property Enabled: boolean read GetEnabled write SetEnabled;
    property DistanceX: Integer read GetDistanceX write SetDistanceX;
    property DistanceY: Integer read GetDistanceY write SetDistanceY;
    property Left: Integer read GetLeft write SetLeft;
    property Top: Integer read GetTop write SetTop;
    property Width: Integer read GetWidth write SetWidth;
    property Height: Integer read GetHeight write SetHeight;
    property OnClick: TNotifyEvent read GetOnClick write SetOnClick;
    property OnButtonClick: TNotifyEvent read GetOnButtonClick
                                         write SetOnButtonClick;
    property OnDblClick: TNotifyEvent read GetOnDblClick
                                      write SetOnDblClick;
    property OnDragDrop: TDragDropEvent read GetOnDragDrop
                                        write SetOnDragDrop;
    property OnDragOver: TDragOverEvent read GetOnDragOver
                                      write SetOnDragOver;
    property OnEndDrag: TEndDragEvent read GetOnEndDrag
                                       write SetOnEndDrag;
    property OnMouseDown: TMouseEvent read GetOnMouseDown
                                          write SetOnMouseDown;
    property OnMouseMove: TMouseMoveEvent read GetOnMouseMove
                                          write SetOnMouseMove;
    property OnMouseUp: TMouseEvent read GetOnMouseUp
                                    write SetOnMouseUp;
    property OnStartDrag: TStartDragEvent read GetOnStartDrag
                                          write SetOnStartDrag;
    property OnEnter: TNotifyEvent read GetOnEnter write SetOnEnter;
    property OnExit: TNotifyEvent read GetOnExit write SetOnExit;
    property OnKeyDown: TKeyEvent read GetOnKeyDown write SetOnKeyDown;
    property OnKeyPress: TKeyPressEvent read GetOnKeyPress write SetOnKeyPress;
    property OnKeyUp: TKeyEvent read GetOnKeyUp write SetOnKeyUp;
    property TabStop: boolean read GetTabStop write SetTabStop;
    property OnReplaceField: TFieldNotifyEvent read GetOnreplaceField
                                               write SetOnReplaceField;
  end;

  TRaDBPanelFieldClass = class of TRaDBPanelField;

  TRaDBPanelFields = class(TCollection)
  private
    FDBPanelBox: TDBPanelBox;
    function GetField(Index: Integer): TRaDBPanelField;
    procedure SetField(Index: Integer; Value: TRaDBPanelField);
  protected
    function GetOwner: TPersistent; override;
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(ADBPanelBox: TDBPanelBox;
                                         DBPanelFieldClass: TRaDBPanelFieldClass);
    function Add: TRaDBPanelField;
    property DBPanelBox: TDBPanelBox read FDBPanelBox;
    property Fields[Index: Integer]: TRaDBPanelField read GetField
                                                 write SetField; default;
  end;

  TRaDBPanelStyle = (psAuto, psBox, psGrid);

  TRaDBPanel = class;

  TRaDBPanelDataLink = class(TDataLink)
  private
    FDBPanel: TRaDBPanel;
  protected
    procedure ActiveChanged; override;
  end;


  TDBPanelBox = class(TRaDBBox)
  private
     FFields: TRaDBPanelFields;
     FDBPanel: TRaDBPanel;
     FStoreFields: Boolean;
  protected
     function FindNext(StartControl: TWinControl; GoForward: Boolean;
                                  var WrapFlag: Integer): TWinControl; override;
     property CreateMode;
  public
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;
     procedure RefreshFields; override;
     property StoreFields: Boolean read FStoreFields;
  published
     property Fields: TRaDBPanelFields read FFields write FFields stored FStoreFields;
  end;

  TDBPanelGrid = class(TRaDBGrid)
  private
     FDBPanel: TRaDBPanel;
  public
     constructor Create(AOwner: TComponent); override;
  end;

  TRaDBPanel = class(TPanel)
  private
     FStoreFields: Boolean;
     FBoxLink: TRaDBPanelDataLink;
     FGridLink: TRaDBPanelDataLink;
     FDBPanelBox: TDBPanelBox;
     FSplitter: TSplitter;
     FDBPanelGrid: TDBPanelGrid;
     FPanelStyle: TRaDBPanelStyle;
     FBoxHeight: Integer;
     FReadOnly: Boolean;
     FStoreColumns: Boolean;
     FStyleChanging: Boolean;
     FCloseOnEscape: Boolean;
     procedure CNCloseDBForm(var Message: TMessage); Message CN_CLOSEDBFORM;
     procedure CMChildKey(var Message: TCMChildKey); message CM_CHILDKEY;
     procedure CNPanelActivated(var Message: TMessage); message CN_PANELACTIVATED;
     function GetStoreFields: Boolean;
     procedure SetStoreFields(Value: Boolean);
     function GetActiveControl: TWinControl;
     function GetActualHeight: Integer;
     function GetActualWidth: Integer;
     function GetColumns: TDBGridColumns;
     function GetFields: TRaDBPanelFields;
     function GetDataSource: TDataSource;
     function GetOrientation: TOrientation;
     procedure SetOrientation(Value: TOrientation);
     procedure SetColumns(Value: TDBGridColumns);
     procedure SetFields(Value: TRaDBPanelFields);
     procedure SetPanelStyle(Value: TRaDBPanelStyle);
     procedure SetDataSource(Value: TDataSource);
     procedure SetReadOnly(Value: Boolean);
     procedure SetBoxHeight(Value: Integer);
     function GetOnGridEnter: TNotifyEvent;
     function GetOnGridExit: TNotifyEvent;
     function GetOnGridColEnter: TNotifyEvent;
     function GetOnGridColExit: TNotifyEvent;
     function GetOnGridReplaceField: TFieldNotifyEvent;
     function GetOnGridEditButtonClick: TNotifyEvent;
     function GetOnGridCellClick: TDBGridClickEvent;
     function GetOnGridColumnMoved: TMovedEvent;
     function GetOnGridDrawColumnCell: TDrawColumnCellEvent;
     function GetOnGridDrawDataCell: TDrawDataCellEvent;
     procedure SetOnGridEnter(Value: TNotifyEvent);
     procedure SetOnGridExit(Value: TNotifyEvent);
     procedure SetOnGridColEnter(Value: TNotifyEvent);
     procedure SetOnGridColExit(Value: TNotifyEvent);
     procedure SetOnGridReplaceField(Value: TFieldNotifyEvent);
     procedure SetOnGridEditButtonClick(Value: TNotifyEvent);
     procedure SetOnGridCellClick(Value: TDBGridClickEvent);
     procedure SetOnGridColumnMoved(Value: TMovedEvent);
     procedure SetOnGridDrawColumnCell(Value: TDrawColumnCellEvent);
     procedure SetOnGridDrawDataCell(Value: TDrawDataCellEvent);
     procedure CNStyleChanged(var Message: TMessage); message CN_STYLECHANGED;
  protected
     procedure Notification(AComponent: TComponent;
                                          Operation: TOperation); override;
     procedure SetVisibleAttributes;
  public
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;
     property Box: TDBPanelBox read FDBPanelBox;
     property Grid: TDBPanelGrid read FDBPanelGrid;
     property ActiveControl: TWinControl read GetActiveControl;
     property ActualHeight: Integer read GetActualHeight;
     property ActualWidth: Integer read GetActualWidth;
  published
     property ReadOnly: Boolean read FReadOnly write SetReadOnly;
     property Caption stored False;
     property DataSource: TDataSource read GetDataSource write SetDataSource;
     property PanelStyle: TRaDBPanelStyle read FPanelStyle write SetPanelStyle;
     property Orientation: TOrientation read GetOrientation write SetOrientation;
     property BoxHeight: Integer read FBoxHeight write SetBoxHeight;
     property StoreFields: Boolean read GetStoreFields write SetStoreFields;
     property GridColumns: TDBGridColumns read GetColumns
                                      write SetColumns stored FStoreColumns;
     property BoxFields: TRaDBPanelFields read GetFields
                                      write SetFields stored FStoreFields;
     property CloseOnEscape: Boolean read FCloseOnEscape write FCloseOnEscape
                                                         default True;
     property OnGridEnter: TNotifyEvent read GetOnGridEnter write SetOnGridEnter;
     property OnGridExit: TNotifyEvent read GetOnGridExit write SetOnGridExit;
     property OnGridColEnter: TNotifyEvent read GetOnGridColEnter
                                           write SetOnGridColEnter;
     property OnGridColExit: TNotifyEvent read GetOnGridColExit
                                          write SetOnGridColExit;
     property OnGridEditButtonClick: TNotifyEvent read GetOnGridEditButtonClick
                                                  write SetOnGridEditButtonClick;
     property OnGridReplaceField: TFieldNotifyEvent read GetOnGridReplaceField
                                                    write SetOnGridReplaceField;
     property OnGridCellClick: TDBGridClickEvent read GetOnGridCellClick
                                                  write SetOnGridCellClick;
     property OnGridColumnMoved: TMovedEvent read GetOnGridColumnMoved
                                                  write SetOnGridColumnMoved;
     property OnGridDrawDataCell: TDrawDataCellEvent read GetOnGridDrawDataCell
                                                  write SetOnGridDrawDataCell;
     property OnGridDrawColumnCell: TDrawColumnCellEvent read GetOnGridDrawColumnCell
                                                  write SetOnGridDrawColumnCell;
  end;

implementation

uses dbTables, dbTools;

{TFieldsProperty}
function TFieldsProperty.GetAttributes: TPropertyAttributes;
begin
 Result := [paValueList];
end;

procedure TFieldsProperty.GetValues(Proc: TGetStrProc);
var i: Integer;
    AField: TRaDBPanelField;
begin
   AField := GetComponent(0) as TRaDBPanelField;
   if AField <> nil then
   with AField.FDBEditField do
   begin
    if (DataSource <> nil) and (DataSource.DataSet <> nil) then
      for i := 0 to DataSource.DataSet.FieldCount-1 do
        Proc(DataSource.DataSet.Fields[i].FieldName);
  end;
end;

{TRaDBPanelField}
constructor TRaDBPanelField.Create(Collection: TCollection);
var i: Integer;
    DBPanelBox: TDBPanelBox;
begin
   Inherited Create(Collection);
   DBPanelBox := GetCollection.FDBPanelBox;
   if DBPanelBox = nil then exit;
   FDBEditField := TRaDBEdit.Create(DBPanelBox);
   FDBEditField.Changing := False;
   FDBEditField.Parent := DBPanelBox;
   with FDBEditField do
   begin
      DataSource := DBPanelBox.DataSource;
      if csDesigning in ComponentState then
         if Parent <> nil then
         begin
            i := 1;
            while Parent.FindComponent(Format('DBPanelField%d', [i])) <> nil do
              inc(i);
            Name := Format('DBPanelField%d', [i]);
         end;
      LabelFont := DBPanelBox.LabelFont;
      ShowBlob := DBPanelBox.ShowBlobs;
   end;
end;

destructor TRaDBPanelField.Destroy;
begin
   FDBEditField.Free;
   FDBEditField := nil;
   Inherited Destroy;
end;

function TRaDBPanelField.GetCollection: TRaDBPanelFields;
begin
   Result := Collection as TRaDBPanelFields;
end;

function TRaDBPanelField.GetDisplayName: string;
begin
   Result := FDBEditField.Caption;
end;

function TRaDBPanelField.GetCursor: TCursor;
begin
   Result := FDBEditField.Cursor;
end;

procedure TRaDBPanelField.SetCursor(Value: TCursor);
begin
   GetCollection.Update(Self);
   FDBEditField.Cursor := Value;
end;

function TRaDBPanelField.GetHelpContext: THelpContext;
begin
   Result := FDBEditField.HelpContext;
end;

procedure TRaDBPanelField.SetHelpContext(Value: THelpContext);
begin
   GetCollection.Update(Self);
   FDBEditField.HelpContext := Value;
end;

function TRaDBPanelField.GetTag: LongInt;
begin
   Result := FDBEditField.Tag;
end;

procedure TRaDBPanelField.SetTag(Value: LongInt);
begin
   GetCollection.Update(Self);
   FDBEditField.Tag := Value;
end;

function TRaDBPanelField.GetLeft: Integer;
begin
   Result := FDBEditField.Left;
end;

procedure TRaDBPanelField.SetLeft(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.Left := Value;
end;

function TRaDBPanelField.GetTop: Integer;
begin
   Result := FDBEditField.Top;
end;

procedure TRaDBPanelField.SetTop(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.Top := Value;
end;

function TRaDBPanelField.GetWidth: Integer;
begin
   Result := FDBEditField.Width;
end;

procedure TRaDBPanelField.SetWidth(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.Width := Value;
end;

function TRaDBPanelField.GetHeight: Integer;
begin
   Result := FDBEditField.Height;
end;

procedure TRaDBPanelField.SetHeight(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.Height := Value;
end;

function TRaDBPanelField.GetHint: String;
begin
   Result := FDBEditField.Hint;
end;

procedure TRaDBPanelField.SetHint(Value: String);
begin
   GetCollection.Update(Self);
   FDBEditField.Hint := Value;
end;

function TRaDBPanelField.GetDataField: string;
begin
   Result := FDBEditField.DataField;
end;

procedure TRaDBPanelField.SetDataField(const Value: string);
begin
   GetCollection.Update(Self);
   FDBEditField.DataField := Value;
end;

function TRaDBPanelField.GetShowBlob: Boolean;
begin
   Result := FDBEditField.ShowBlob;
end;

procedure TRaDBPanelField.SetShowBlob(Value: Boolean);
begin
   GetCollection.Update(Self);
   FDBEditField.ShowBlob := Value;
end;

function TRaDBPanelField.GetDistanceX: Integer;
begin
   Result := FDBEditField.DistanceX;
end;

procedure TRaDBPanelField.SetDistanceX(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.DistanceX := Value;
end;

function TRaDBPanelField.GetDistanceY: Integer;
begin
   Result := FDBEditField.DistanceY;
end;

procedure TRaDBPanelField.SetDistanceY(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.DistanceY := Value;
end;

function TRaDBPanelField.GetMaxLength: Integer;
begin
   Result := FDBEditField.MaxLength;
end;

procedure TRaDBPanelField.SetMaxLength(Value: Integer);
begin
   GetCollection.Update(Self);
   FDBEditField.MaxLength := Value;
end;

function TRaDBPanelField.GetFont: TFont;
begin
   Result := FDBEditField.Font;
end;

procedure TRaDBPanelField.SetFont(Value: TFont);
begin
   GetCollection.Update(Self);
   FDBEditField.Font := Value;
end;

function TRaDBPanelField.GetLabelFont: TFont;
begin
   Result := FDBEditField.LabelFont;
end;

procedure TRaDBPanelField.SetLabelFont(Value: TFont);
begin
   GetCollection.Update(Self);
   FDBEditField.LabelFont := Value;
end;

function TRaDBPanelField.GetCaption: String;
begin
   Result := FDBEditField.Caption;
end;

procedure TRaDBPanelField.SetCaption(Value: String);
begin
   GetCollection.Update(Self);
   FDBEditField.Caption := Value;
end;

function TRaDBPanelField.GetEditKind: TEditKind;
begin
   Result := FDBEditField.EditKind;
end;

procedure TRaDBPanelField.SetEditKind(Value: TEditKind);
begin
   GetCollection.Update(Self);
   FDBEditField.EditKind := Value;
end;

function TRaDBPanelField.GetButtonHint: String;
begin
   Result := FDBEditField.ButtonHint;
end;

procedure TRaDBPanelField.SetButtonHint(Value: String);
begin
   GetCollection.Update(Self);
   FDBEditField.ButtonHint := Value;
end;

function TRaDBPanelField.GetEnabled: boolean;
begin
   Result := FDBEditField.Enabled;
end;

procedure TRaDBPanelField.SetEnabled(Value: boolean);
begin
   GetCollection.Update(Self);
   FDBEditField.Enabled := Value;
end;

function TRaDBPanelField.GetOnClick: TNotifyEvent;
begin
   Result := FDBEditField.OnClick;
end;

procedure TRaDBPanelField.SetOnClick(Value: TNotifyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnClick := Value;
end;

function TRaDBPanelField.GetOnButtonClick: TNotifyEvent;
begin
   Result := FDBEditField.OnButtonClick;
end;

procedure TRaDBPanelField.SetOnButtonClick(Value: TNotifyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnButtonClick := Value;
end;

function TRaDBPanelField.GetOnDblClick: TNotifyEvent;
begin
   Result := FDBEditField.OnDblClick;
end;

procedure TRaDBPanelField.SetOnDblClick(Value: TNotifyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnDblClick := Value;
end;

function TRaDBPanelField.GetOnDragDrop: TDragDropEvent;
begin
   Result := FDBEditField.OnDragDrop;
end;

procedure TRaDBPanelField.SetOnDragDrop(Value: TDragDropEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnDragDrop := Value;
end;

function TRaDBPanelField.GetOnDragOver: TDragOverEvent;
begin
   Result := FDBEditField.OnDragOver;
end;

procedure TRaDBPanelField.SetOnDragOver(Value: TDragOverEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnDragOver := Value;
end;

function TRaDBPanelField.GetOnEndDrag: TEndDragEvent;
begin
   Result := FDBEditField.OnEndDrag;
end;

procedure TRaDBPanelField.SetOnEndDrag(Value: TEndDragEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnEndDrag := Value;
end;

function TRaDBPanelField.GetOnMouseDown: TMouseEvent;
begin
   Result := FDBEditField.OnMouseDown;
end;

procedure TRaDBPanelField.SetOnMouseDown(Value: TMouseEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnMouseDown := Value;
end;

function TRaDBPanelField.GetOnMouseMove: TMouseMoveEvent;
begin
   Result := FDBEditField.OnMouseMove;
end;

procedure TRaDBPanelField.SetOnMouseMove(Value: TMouseMoveEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnMouseMove := Value;
end;

function TRaDBPanelField.GetOnMouseUp: TMouseEvent;
begin
   Result := FDBEditField.OnMouseUp;
end;

procedure TRaDBPanelField.SetOnMouseUp(Value: TMouseEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnMouseUp := Value;
end;

function TRaDBPanelField.GetOnStartDrag: TStartDragEvent;
begin
   Result := FDBEditField.OnStartDrag;
end;

procedure TRaDBPanelField.SetOnStartDrag(Value: TStartDragEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnStartDrag := Value;
end;

function TRaDBPanelField.GetOnEnter: TNotifyEvent;
begin
   Result := FDBEditField.OnEnter;
end;

procedure TRaDBPanelField.SetOnEnter(Value: TNotifyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnEnter := Value;
end;

function TRaDBPanelField.GetOnExit: TNotifyEvent;
begin
   Result := FDBEditField.OnExit;
end;

procedure TRaDBPanelField.SetOnExit(Value: TNotifyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnExit := Value;
end;

function TRaDBPanelField.GetOnKeyDown: TKeyEvent;
begin
   Result := FDBEditField.OnKeyDown;
end;

procedure TRaDBPanelField.SetOnKeyDown(Value: TKeyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnKeyDown := Value;
end;

function TRaDBPanelField.GetOnKeyPress: TKeyPressEvent;
begin
   Result := FDBEditField.OnKeyPress;
end;

procedure TRaDBPanelField.SetOnKeyPress(Value: TKeyPressEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnKeyPress := Value;
end;

function TRaDBPanelField.GetOnKeyUp: TKeyEvent;
begin
   Result := FDBEditField.OnKeyUp;
end;

procedure TRaDBPanelField.SetOnKeyUp(Value: TKeyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnKeyUp := Value;
end;

function TRaDBPanelField.GetTabStop: boolean;
begin
   Result := FDBEditField.TabStop;
end;

procedure TRaDBPanelField.SetTabStop(Value: boolean);
begin
   GetCollection.Update(Self);
   FDBEditField.TabStop := Value;
end;

function TRaDBPanelField.GetOnReplaceField: TFieldNotifyEvent;
begin
   Result := FDBEditField.OnReplaceField;
end;

procedure TRaDBPanelField.SetOnReplaceField(const Value: TFieldNotifyEvent);
begin
   GetCollection.Update(Self);
   FDBEditField.OnReplaceField := Value;
end;

{TRaDBPanelFields}
constructor TRaDBPanelFields.Create(ADBPanelBox: TDBPanelBox;
                                         DBPanelFieldClass: TRaDBPanelFieldClass);
begin
  inherited Create(DBPanelFieldClass);
  FDBPanelBox := ADBPanelBox;
end;

function TRaDBPanelFields.Add: TRaDBPanelField;
begin
  Result := TRaDBPanelField(inherited Add);
end;

function TRaDBPanelFields.GetField(Index: Integer): TRaDBPanelField;
begin
  Result := TRaDBPanelField(inherited Items[Index]);
end;

function TRaDBPanelFields.GetOwner: TPersistent;
begin
  Result := FDBPanelBox;
end;

procedure TRaDBPanelFields.SetField(Index: Integer; Value: TRaDBPanelField);
begin
  TRaDBPanelField(inherited Items[Index]).Assign(Value);
end;

procedure TRaDBPanelFields.Update(Item: TCollectionItem);
begin
   if FDBPanelBox.Refreshing then exit;
   FDBPanelBox.FStoreFields := (Count > 0);
   if not FDBPanelBox.StoreFields then
       FDBPanelBox.StartRefresh;
end;

{TRaDBPanelDataLink}
procedure TRaDBPanelDataLink.ActiveChanged;
begin
  if (FDBPanel <> nil) then
         FDBPanel.SetVisibleAttributes;
end;

{TDBPanelBox}
constructor TDBPanelBox.Create(AOwner: TComponent);
begin
   Inherited Create(AOwner);
   if (AOwner <> nil) and (AOwner is TRaDBPanel) then
   begin
      FDBPanel := (AOwner as TRaDBPanel);
      Visible := False;
      Enabled := False;
   end;
   ControlStyle := ControlStyle - [csAcceptsControls];
   ControlStyle := ControlStyle + [csNoDesignVisible];
   FFields := TRaDBPanelFields.Create(Self, TRaDBPanelField);
end;

destructor TDBPanelBox.Destroy;
begin
   FFields.Free;
   inherited Destroy;
end;

function TDBPanelBox.FindNext(StartControl: TWinControl;
                 GoForward: Boolean; var WrapFlag: Integer): TWinControl;
var Control: TWinControl;
begin
  Control := Inherited FindNext(StartControl, GoForward, WrapFlag);
  if (WrapFlag > 0) and (GoForward) and (Parent <> nil) and
                                               (Parent is TRaDBPanel) then
  if FDBPanel <> nil then
  with FDBPanel do
  if Grid.CanFocus then
  begin
     WrapFlag := 0;
     Control := Grid;
     if Grid.FirstColumn >= 0 then
        Grid.SelectedIndex := Grid.FirstColumn;
  end;
  Result := Control;
end;

procedure TDBPanelBox.RefreshFields;

  procedure UpdateFields;
  var i: Integer;
      AWidth, AHeight: Integer;
  begin
     try
       FFields.BeginUpdate;
       for i := 0 to FFields.Count-1 do
       with FFields.Fields[i].FDBEditField do
       begin
          AWidth := Width;
          AHeight := Height;
          try
            try
               Changing := False;
               DataSource := Self.DataSource;
               Refresh(False);
               if Self.ReadOnly then
               begin
                  if EditKind = ekControl then
                      EditKind := ekCommon;
                  if FieldType = efLookUp then
                      EditKind := ekView;
               end;
            except
               Raise
            end;
          finally
            SetBounds(Left, Top, AWidth, AHeight);
          end;
       end;
     finally
       FFields.EndUpdate;
     end;
  end;

  procedure CreateFields;
  var i: Integer;
      Field: TField;
      DBEditField: TRaDBEdit;
      ATop, ALeft, AWidth, AHeight, LineHeight, LineDistance: integer;
      MaxLabelLength: Integer;
  begin
     LineDistance := 10;
     if (DataSource = nil) or
        (DataSource.DataSet = nil) or
        not DataSource.DataSet.Active then
        exit;
     ATop := OriginY;
     ALeft := OriginX;
     MaxLabelLength := 0;
     LineHeight := 0;
     for i := 0 to DataSource.DataSet.FieldCount - 1  do
     begin
        Field := DataSource.DataSet.Fields[i];
        if(Field.Visible) then
        begin
           DBEditField := FFields.Add.FDBEditField;
           DBEditField.Changing := True;
           DBEditField.TabStop :=
              not (Field.ReadOnly or
                       (Field.FieldKind in [fkCalculated, fkInternalCalc]));
           DBEditField.Changing := False;
           DBEditField.DataField := Field.FieldName;
           if ReadOnly then
           begin
              if DBEditField.EditKind = ekControl then
                  DBEditField.EditKind := ekCommon;
              if DBEditField.FieldType = efLookUp then
                 DBEditField.EditKind := ekView;
           end;
           if Orientation = orVertical then
           begin
              if FFields.Count > 1 then
              begin
                 ATop := ATop+LineDistance;
                 if FFields.Count > 1 then
                     Inc(ATop, FFields.Fields[FFields.Count-2].Height);
              end;
           end
             else
             if FFields.Count > 1 then
             begin
                ATop := FFields.Fields[FFields.Count-2].Top;
                ALeft := FFields.Fields[FFields.Count-2].Left;
                AWidth := FFields.Fields[FFields.Count-2].Width;
                AHeight := FFields.Fields[FFields.Count-2].Height;
                if LineHeight < AHeight+LineDistance then
                        LineHeight := AHeight+LineDistance;
                if ALeft + AWidth + LineDistance + DbEditField.Width >
                               ClientWidth-GetSystemMetrics(SM_CXVSCROLL) then
                begin
                   ALeft := OriginX;
                   ATop := ATop + LineHeight;
                end
                  else
                     ALeft := ALeft + AWidth + LineDistance;
             end;
           DBEditField.Top := ATop;
           DBEditField.Left := ALeft;
           if Orientation = orVertical then
           begin
              DBEditField.DistanceX := 8;
              if MaxLabelLength < DBEditField.CLabel.Width then
                 MaxLabelLength := DBEditField.CLabel.Width;
           end;
        end;
    end;
    i := 0;
    while i < FFields.Count do
    begin
       DBEditField := FFields[i].FDBEditField;
       if Orientation = orVertical then
       begin
          DBEditField.DistanceX :=
                         8 + MaxLabelLength - DBEditField.CLabel.Width;
       end;
       inc(i);
   end;
  end;

begin
   if FixedFields then Exit;
   try
     HorzScrollBar.Visible := False;
     VertScrollBar.Visible := False;
     Visible := False;
     ControlStyle := ControlStyle-[csNoDesignVisible];
     if not FStoreFields then
     begin
        FFields.BeginUpdate;
        try
          FFields.Clear;
          CreateFields;
        finally
          FFields.EndUpdate;
        end;
        if (FDBPanel <> nil) and not FDBPanel.FStyleChanging then
           if (FDBPanel.FBoxHeight < 0) and (Fields.Count > 0) then
           begin
              with FDBPanel.FSplitter do
              begin
                 if not Visible then
                 begin
                    if FDBPanel.FDBpanelGrid.Visible then
                    begin
                       Visible := True;
                       Enabled := True;
                       ControlStyle := ControlStyle-[csNoDesignVisible];
                    end;
                 end;
                 if Visible then
                 begin
                     Align := alBottom;
                     if ActualHeight+OriginY + 4 < FDBPanel.ClientHeight then
                        Self.Height := ActualHeight+OriginY
                       else
                        Self.Height := FDBPanel.ClientHeight*2 div 3;
                     Align := AlTop;
                 end;
              end;
           end;
        FStoreFields := False;
     end
       else
         UpdateFields;
     {!!!}
     if FStoreFields then FixedFields := True;
   finally
     HorzScrollBar.Visible := True;
     VertScrollBar.Visible := True;
     Visible := True;
     ControlStyle := ControlStyle-[csNoDesignVisible];
   end;
end;

{TDBPanelGrid}
constructor TDBPanelGrid.Create(AOwner: TComponent);
begin
   Inherited Create(AOwner);
   if (AOwner <> nil) and (AOwner is TRaDBPanel) then
   begin
      FDBPanel := (AOwner as TRaDBPanel);
      Visible := False;
      Enabled := False;
      ControlStyle := ControlStyle + [csNoDesignVisible];
      MinRowCount := 5;
   end;
end;

{TRaDBPanel}
constructor TRaDBPanel.Create(AOwner: TComponent);
begin
   Inherited Create(AOwner);
   ControlStyle := ControlStyle - [csSetCaption, csAcceptsControls];
   BevelWidth := 1;
   BevelInner := bvRaised;
   BevelOuter := bvLowered;
   FDBPanelBox := TDBPanelBox.Create(Self);
   FDBPanelBox.Parent := Self;
   FDBPanelBox.Align := alTop;
   FSplitter := TSplitter.Create(Self);
   with FSplitter do
   begin
      Parent := Self;
      Cursor := crVSplit;
      Align := alTop;
      MinSize := 1;
      Height := 3;
      Visible := False;
      Enabled := False;
      ControlStyle := ControlStyle+[csNoDesignVisible];
   end;
   FDBPanelGrid := TDBPanelGrid.Create(Self);
   FDBPanelGrid.Parent := Self;
   FDBPanelGrid.Align := alClient;
   FBoxHeight := -1;
   FBoxLink := TRaDBPanelDataLink.Create;
   FBoxLink.FDBPanel := Self;
   FGridLink := TRaDBPanelDataLink.Create;
   FGridLink.FDBPanel := Self;
   FCloseOnEscape := True;
end;

destructor TRaDBPanel.Destroy;
begin
   FBoxLink.Free;
   FGridLink.Free;
   Inherited Destroy;
end;

function TRaDBPanel.GetStoreFields: Boolean;
begin
   Result := FDBPanelBox.StoreFields;
   FStoreFields := Result;
end;

procedure TRaDBPanel.SetStoreFields(Value: Boolean);
begin
   FDBPanelBox.FStoreFields := Value;
   FStoreFields := Value;
   {!!!}
   if not FStoreFields then FDBPanelBox.FixedFields := False;
end;

function TRaDBPanel.GetActiveControl: TWinControl;
var
  Form: TCustomForm;
  Control: TWinControl;
begin
  Control := nil;
  Form := GetParentForm(Self);
  if Form <> nil then
     Control := Form.ActiveControl;
  Result := Control;
end;

function TRaDBPanel.GetActualWidth: integer;
begin
   Result := FDBPanelBox.ActualWidth;
   if FDBPanelBox.Visible then
     Result := Result + FDBPanelBox.OriginX;
   if Result < FDBPanelGrid.ActualWidth then
     Result := FDBPanelGrid.ActualWidth;
   Result := Result + BorderWidth*2;
   if BevelInner <> bvNone then
     Result := Result + BevelWidth*2;
   if BevelOuter <> bvNone then
     Result := Result + BevelWidth*2;
end;

function TRaDBPanel.GetActualHeight: integer;
begin
   Result := 0;
   if FDBPanelBox.Visible then
   begin
      Inc(Result, FDBPanelBox.ActualHeight);
      Inc(Result, FDBPanelBox.OriginY);
   end;

   if FDBPanelGrid.Visible then
      Inc(Result, FDBPanelGrid.ActualHeight);

   if FSplitter.Visible then
      Inc(Result, FSplitter.Height);
   Inc(Result, BorderWidth*2);
   if BevelInner <> bvNone then
      Inc(Result, BevelWidth*2);
   if BevelOuter <> bvNone then
      Inc(Result, BevelWidth*2);
end;

function TRaDBPanel.GetColumns: TDBGridColumns;
begin
   Result := FDBPanelGrid.Columns;
   FStoreColumns := FDBPanelGrid.StoreColumns;
end;

function TRaDBPanel.GetFields: TRaDBPanelFields;
begin
   Result := FDBPanelBox.FFields;
end;

function TRaDBPanel.GetDataSource: TDataSource;
begin
   case FPanelStyle of
      psBox: Result := FBoxLink.DataSource;
    else
      Result := FGridLink.DataSource;
   end;
end;

function TRaDBPanel.GetOrientation: TOrientation;
begin
   Result := FDBPanelBox.Orientation;
end;

procedure TRaDBPanel.SetOrientation(Value: TOrientation);
begin
   if not FDBPanelBox.StoreFields then
      FDBPanelBox.Orientation := Value;
end;

procedure TRaDBPanel.SetColumns(Value: TDBGridColumns);
begin
   FDBPanelGrid.Columns := Value;
end;

procedure TRaDBPanel.SetFields(Value: TRaDBPanelFields);
begin
   FDBPanelBox.FFields.Assign(Value);
end;

procedure TRaDBPanel.SetDataSource(Value: TDataSource);
begin
   case FPanelStyle of
     psBox:
        begin
           FBoxLink.DataSource := Value;
           FGridLink.DataSource := nil;
        end;
     psGrid:
        begin
           FBoxLink.DataSource := nil;
           FGridLink.DataSource := Value;
        end;
     psAuto:
           FGridLink.DataSource := Value;
   end;
   SetVisibleAttributes;
end;

procedure TRaDBPanel.CNCloseDBForm(var Message: TMessage);
begin
   SendMessage(FDBPanelBox.Handle, Message.Msg, Message.WParam, Message.LParam);
   SendMessage(FDBPanelGrid.Handle, Message.Msg, Message.WParam, Message.LParam);
end;

procedure TRaDBPanel.CMChildKey(var Message: TCMChildKey);
var Shift: TShiftState;

   function CloseForm(AModalResult: Integer): boolean;
   var Form: TCustomForm;
   begin
      Result := False;
      Form := GetParentForm(Self);
      if Form <> nil then
      begin
         SendMessage(Form.Handle, WM_CLOSE, AModalResult, 0);
         Result := True;
      end;
   end;

begin
  Shift := GetShiftState;
  with Message do
    if Sender <> Self then
    begin
       case CharCode of
          Word('1'):
               if Shift = [ssCtrl] then
               begin
                 if FDBPanelBox.CanFocus then
                     FDBPanelBox.DoKey(bgkFirstTab)
                    else
                 if FDBPanelGrid.CanFocus then
                 begin
                    if FDBPanelGrid.FirstColumn >= 0 then
                      FDBPanelGrid.SelectedIndex := FDBPanelGrid.FirstColumn;
                    FDBPanelGrid.SetFocus;
                 end;
                 Result := 1;
                 Exit;
               end;
          Word('2'):
               if Shift = [ssCtrl] then
               begin
                 if FDBPanelGrid.CanFocus then
                 begin
                    if FDBPanelGrid.FirstColumn >= 0 then
                      FDBPanelGrid.SelectedIndex := FDBPanelGrid.FirstColumn;
                    FDBPanelGrid.SetFocus;
                 end
                  else
                 if FDBPanelBox.CanFocus then
                     FDBPanelBox.DoKey(bgkFirstTab);
                 Result := 1;
                 Exit;
               end;
          VK_RETURN:
               if (Shift = [ssCtrl]) and CloseForm(-100) then
               begin
                  Result := 1;
                  Exit;
               end;
          VK_ESCAPE:
               if (Shift = []) and FCloseOnEscape and CloseForm(mrCancel) then
               begin
                  Result := 1;
                  Exit;
               end;
       end;
    end;
  inherited;
end;

procedure TRaDBPanel.CNPanelActivated(var Message: TMessage);
var Form: TCustomForm;
begin
   if not (csDesigning in ComponentState) then
   begin
      Form := GetParentForm(Self);
      if (Form <> nil) and Form.CanFocus then
      begin
         if Form.ActiveControl = FDBPanelGrid then
            Form.ActiveControl := nil;
         if Form.ActiveControl = nil then
            Form.ActiveControl := FindNextControl(nil, True, True, False);
         if Form.ActiveControl = FDBPanelGrid then
            FDBPanelGrid.SelectedIndex := FDBPanelGrid.FirstColumn;
      end;
   end;
end;

procedure TRaDBPanel.SetPanelStyle(Value: TRaDBPanelStyle);
var ADataSource: TDataSource;
begin
   if Value <> FPanelStyle then
   begin
      ADataSource := DataSource;
      FPanelStyle := Value;
      DataSource := ADataSource;
   end;
end;

procedure TRaDBPanel.SetBoxHeight(Value: Integer);
begin
   if Value <> FBoxHeight then
   begin
      if Value < 0 then
         Value := -1;
      FBoxHeight := Value;
      SetVisibleAttributes;
   end;
end;

procedure TRaDBPanel.SetReadOnly(Value: Boolean);
begin
   try
      FDBPanelBox.ReadOnly := Value;
      FDBPanelGrid.ReadOnly := Value;
      FReadOnly := Value;
      SetVisibleAttributes;
   except
      Raise;
   end;
end;

procedure TRaDBPanel.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) then
  begin
     if (FBoxLink <> nil) and (AComponent = FBoxLink.DataSource) then
        FBoxLink.DataSource := nil;
     if (FGridLink <> nil) and (AComponent = FGridLink.DataSource) then
        FGridLink.DataSource := nil;
  end;
end;

function TRaDBPanel.GetOnGridEnter: TNotifyEvent;
begin
   Result := FDBPanelGrid.OnEnter;
end;

function TRaDBPanel.GetOnGridExit: TNotifyEvent;
begin
   Result := FDBPanelGrid.OnExit;
end;

function TRaDBPanel.GetOnGridColEnter: TNotifyEvent;
begin
   Result := FDBPanelGrid.OnColEnter;
end;

function TRaDBPanel.GetOnGridColExit: TNotifyEvent;
begin
   Result := FDBPanelGrid.OnColExit;
end;

function TRaDBPanel.GetOnGridReplaceField: TFieldNotifyEvent;
begin
   Result := FDBPanelGrid.OnReplaceField;
end;

function TRaDBPanel.GetOnGridEditButtonClick: TNotifyEvent;
begin
   Result := FDBPanelGrid.OnEditButtonClick;
end;

function TRaDBPanel.GetOnGridCellClick: TDBGridClickEvent;
begin
   Result := FDBPanelGrid.OnCellClick;
end;

function TRaDBPanel.GetOnGridColumnMoved: TMovedEvent;
begin
   Result := FDBPanelGrid.OnColumnMoved;
end;

function TRaDBPanel.GetOnGridDrawDataCell: TDrawDataCellEvent;
begin
   Result := FDBPanelGrid.OnDrawDataCell;
end;

function TRaDBPanel.GetOnGridDrawColumnCell: TDrawColumnCellEvent;
begin
   Result := FDBPanelGrid.OnDrawColumnCell;
end;

procedure TRaDBPanel.SetOnGridEnter(Value: TNotifyEvent);
begin
   FDBPanelGrid.OnEnter := Value;
end;

procedure TRaDBPanel.SetOnGridExit(Value: TNotifyEvent);
begin
   FDBPanelGrid.OnExit := Value;
end;

procedure TRaDBPanel.SetOnGridColEnter(Value: TNotifyEvent);
begin
   FDBPanelGrid.OnColEnter := Value;
end;

procedure TRaDBPanel.SetOnGridColExit(Value: TNotifyEvent);
begin
   FDBPanelGrid.OnColExit := Value;
end;

procedure TRaDBPanel.SetOnGridReplaceField(Value: TFieldNotifyEvent);
begin
   FDBPanelGrid.OnReplaceField := Value;
end;

procedure TRaDBPanel.SetOnGridEditButtonClick(Value: TNotifyEvent);
begin
   FDBPanelGrid.OnEditButtonClick := Value;
end;

procedure TRaDBPanel.SetOnGridCellClick(Value: TDBGridClickEvent);
begin
   FDBPanelGrid.OnCellClick := Value;
end;

procedure TRaDBPanel.SetOnGridColumnMoved(Value: TMovedEvent);
begin
   FDBPanelGrid.OnColumnMoved := Value;
end;

procedure TRaDBPanel.SetOnGridDrawDataCell(Value: TDrawDataCellEvent);
begin
   FDBPanelGrid.OnDrawDataCell := Value;
end;

procedure TRaDBPanel.SetOnGridDrawColumnCell(Value: TDrawColumnCellEvent);
begin
   FDBPanelGrid.OnDrawColumnCell := Value;
end;

procedure TRaDBPanel.SetVisibleAttributes;
begin
  if not FStyleChanging then
  begin
     FStyleChanging := True;
     PostMessage(Handle, CN_STYLECHANGED, 0, 0);
  end;
end;

procedure TRaDBPanel.CNStyleChanged(var Message: TMessage);

  procedure LetControlVisible(Control: TWinControl; Value: boolean);
  begin
     with Control do
     begin
        Visible := Value;
        Enabled := Value;
        if Value then ControlStyle := ControlStyle - [csNoDesignVisible]
                 else ControlStyle := ControlStyle + [csNoDesignVisible];
        if Value then
        begin
           if Control = FDBPanelBox then
             if (FBoxHeight < 0) and (FDBPanelBox.Fields.Count > 0) then
                Control.Height := FDBPanelBox.ActualHeight+FDBPanelBox.OriginY
               else
                Control.Height := FBoxHeight;
        end
         else
          Control.Height := 0;
     end;
  end;

begin
   try
      FDBPanelGrid.Align := alNone;
      FDBPanelBox.Align := alNone;
      if FPanelStyle = psAuto then
      begin
         with FGridLink do
         if DataSet <> nil then
           FBoxLink.DataSource := GetMasterSource(DataSet)
          else
           FBoxLink.DataSource := nil;
      end;
      FDBPanelBox.DataSource := FBoxLink.DataSource;
      FDBPanelGrid.DataSource := FGridLink.DataSource;
      LetControlVisible(FDBPanelBox, FBoxLink.DataSource <> nil);
      LetControlVisible(FDBPanelGrid, FGridLink.DataSet <> nil);
      case FPanelStyle of
        psBox:
           begin
              FSplitter.Visible := False;
              FSplitter.Enabled := False;
              FSplitter.ControlStyle := FSplitter.ControlStyle+[csNoDesignVisible];
              FDBPanelBox.Align := alClient;
           end;
        psGrid:
           begin
              FSplitter.Visible := False;
              FSplitter.Enabled := False;
              FSplitter.ControlStyle := FSplitter.ControlStyle+[csNoDesignVisible];
              FDBPanelGrid.Align := alClient;
           end;
        psAuto:
           begin
             if FDBPanelBox.Height > Self.ClientHeight+3 then
             begin
                FDBPanelBox.Height := Self.ClientHeight*2 div 3;
                FSplitter.Top := FDBPanelBox.Top + FDBPanelBox.Height+1;
             end;

             FDBPanelBox.Align := alTop;
             if FDBPanelBox.Height > 0 then
             with FSplitter do
             begin
                Visible := True;
                Enabled := True;
                ControlStyle := ControlStyle-[csNoDesignVisible];
                Align := alBottom;
                Align := AlTop;
             end;
             if FSplitter.Visible then
                 FDBPanelGrid.Top := FSplitter.Top+FSplitter.Height+1;
             FDBPanelGrid.Align := alClient;
           end;
      end;
   finally
      FStyleChanging := False;
      PostMessage(Handle, CN_PANELACTIVATED, 0 , 0);
   end;
end;

end.
