(*////////////////////////////////////////////////////////////////////////////
//   Part of AlexSoft VCL/DLL Library.                                      //
//   All rights reserved. (c) Copyright 1998.                               //
//   Created by: Alex Rabichooc                                             //
//**************************************************************************//
//  Users of this unit must accept this disclaimer of warranty:             //
//    "This unit is supplied as is. The author disclaims all warranties,    //
//    expressed or implied, including, without limitation, the warranties   //
//    of merchantability and of fitness for any purpose.                    //
//    The author assumes no liability for damages, direct or                //
//    consequential, which may result from the use of this unit."           //
//                                                                          //
//  This Unit is donated to the public as public domain.                    //
//                                                                          //
//  This Unit can be freely used and distributed in commercial and          //
//  private environments provided this notice is not modified in any way.   //
//                                                                          //
//  If you do find this Unit handy and you feel guilty for using such a     //
//  great product without paying someone - sorry :-)                        //
//                                                                          //
//  Please forward any comments or suggestions to Alex Rabichooc at:        //
//                                                                          //
//  a_rabichooc@yahoo.com or alex@carmez.mldnet.com                         //
/////////////////////////////////////////////////////////////////////////////*)

unit dbedfld;

interface
uses Windows, Messages, SysUtils, Classes, Controls, Graphics, Forms, Dialogs,
     DB, StdCtrls, Buttons, DBCtrls, StdUtils {$IFDEF VER120}, dbxctrls{$ENDIF};

type
TRaDBEditType = (efEmpty, efEdit, efMemo, efImage, efLookUp,
                 efDate, efTime, efDataSet);

TEditKind = (ekCommon, ekView, ekControl);

TRaDBEdit = class;

TRaDBEdit = class(TWinControl)
  private
    FChanging: Boolean;
    FTabStop: Boolean;
    FEditKind: TEditKind;
    FControl: TWinControl;
    FButton: TSpeedButton;
    FDataLink: TFieldDataLink;
    FLabel: TLabel;
    FField: TField;
    FDistanceX: Integer;
    FDistanceY: Integer;
    FShowBlob: Boolean;
    FDeltaX: Integer;
    FDeltaY: Integer;
    FOnButtonClick: TNotifyEvent;
    FOnReplaceField: TFieldNotifyEvent;
    procedure CMDesignHitTest(var Mess: TCMDesignHitTest); message CM_DESIGNHITTEST;
    procedure CNReplaceField(var Message: TMessage); Message CN_REPLACEFIELD;
    procedure DoChanges(Sender: TObject);
    procedure DoButtonClick(Sender: TObject);
    procedure DoEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure DoEditKeyPress(Sender: TObject; var Key: Char);
    procedure DoEditKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    function GetGlyph: TBitmap;
    function GetNumGlyphs: TNumGlyphs;
    procedure SetNumGlyphs(Value: TNumGlyphs);
    function GetButtonHint: String;
    procedure SetButtonHint(Value: String);
    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetOnButtonClick: TNotifyEvent;
    procedure SetOnButtonClick(Value: TNotifyEvent);
    function GetField: TField;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetEditKind(Value: TEditKind);
    procedure SetDistanceX(Value: Integer);
    procedure SetDistanceY(Value: Integer);
    function GetMaxLength: Integer;
    procedure SetMaxLength(Value: Integer);
    function GetReadOnly: Boolean;
    procedure SetTabStop(Value: Boolean);
    function GetCaption: String;
    procedure SetCaption(Value: String);
    procedure SetGlyph(Value: TBitmap);
    function GetLabelFont: TFont;
    procedure SetLabelFont(AFont: TFont);
    procedure SetShowBlob(Value: Boolean);
    function GetLabelParentFont: Boolean;
    procedure SetLabelParentFont(Value: Boolean);
    procedure SetOnReplaceField(const Value: TFieldNotifyEvent);
    procedure DoOnClick(Sender: TObject);
    procedure DoOnDblClick(Sender: TObject);
    procedure DoOnDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure DoOnDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure DoOnEndDrag(Sender, Target: TObject; X, Y: Integer);
    procedure DoOnMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure DoOnMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure DoOnMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure DoOnStartDrag(Sender: TObject; var DragObject: TDragObject);
    procedure DoOnEnter(Sender: TObject);
    procedure DoOnExit(Sender: TObject);
  protected
    FCaption: String;
    procedure CreateControl;
    procedure Notification(AComponent: TComponent;
                                          Operation: TOperation); override;
    procedure SetName(const NewName: TComponentName); override;
    procedure SetParent(AParent: TWinControl); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ClickButton;
    procedure FreeControl;
    function FieldType: TRaDBEditType; virtual;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    property Field: TField read GetField;
    property Glyph: TBitmap read GetGlyph write SetGlyph;
    property NumGlyphs: TNumGlyphs read GetNumGlyphs write SetNumGlyphs default 1;
    property CLabel: TLabel read FLabel;
    procedure Refresh(Resize: boolean);
    property ReadOnly: Boolean read GetReadOnly;
    property Changing: Boolean read FChanging write FChanging;
  published
    property ShowBlob: Boolean read FShowBlob write SetShowBlob;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DataField: string read GetDataField write SetDataField;
    property DistanceX: Integer read FDistanceX write SetDistanceX;
    property DistanceY: Integer read FDistanceY write SetDistanceY;
    property MaxLength: Integer read GetMaxLength write SetMaxLength;
    property EditKind: TEditKind read FEditKind write SetEditKind;
    property Font;
    property LabelFont: TFont read GetLabelFont write SetLabelFont;
    property LabelParentFont: Boolean read GetLabelParentFont
                                      write SetLabelParentFont;
    property Caption: String read GetCaption write SetCaption;
    property ButtonHint: String read GetButtonHint write SetButtonHint;
    property OnButtonClick: TNotifyEvent read GetOnButtonClick
                                         write SetOnButtonClick;
    property OnReplaceField: TFieldNotifyEvent read FOnReplaceField write SetOnReplaceField;
    property ParentFont;
    property Enabled;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property TabStop: boolean read FTabStop write SetTabStop;
    property TabOrder;
    property ShowHint;
    property Visible;
end;

TxDBLookUpComboBox = {$IFDEF VER120}class(TfxDBLookUpComboBox)
                     {$ELSE}class(TDBLookUpComboBox)
                     {$ENDIF}
protected
    procedure KeyPress(var Key: Char); override;
    procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

{$IFNDEF PROFI}
TDBSmartImage = class(TDBImage);
{$ENDIF}

implementation
uses dbConsts, dbBoxGrd, dbTools, {$IFDEF PROFI}dbImage, {$ENDIF}dbPanel, DbXCnsts;

type

TxLabel = class(TLabel)
protected
  procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

TxButton = class(TSpeedButton)
protected
  procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

TxDBEdit = class(TDBEdit)
protected
  procedure KeyPress(var Key: Char); override;
  procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

TDBDateEdit = class(TxDBEdit)
protected
  procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

TxDBSmartImage = class(TDBSmartImage)
protected
  procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

TxDBRichEdit = class(TDBRichEdit)
protected
  procedure CMDesignHitTest(var Message: TCMDesignHitTest); message CM_DESIGNHITTEST;
end;

{$R DBEDFLD.RES}

procedure PerformHitTest(DBEdit: TRaDBEdit; var Mess: TCMDesignHitTest);
begin
  if (DBEdit.Parent is TDBPanelBox) and
     (DBEdit.Parent as TDBPanelBox).StoreFields then
  with Mess do
  begin
     if Keys=1 then
     begin
        Screen.Cursor := crHandPoint;
        if (DBEdit.FDeltaX = 0) and (DBEdit.FDeltaY = 0) then
        begin
           DBEdit.FDeltaX := XPos-1;
           DBEdit.FDeltaY := YPos-1;
        end;
        DBEdit.Left := DBEdit.Left+XPos-DBEdit.FDeltaX-1;
        DBEdit.Top := DBEdit.Top+YPos-DBEdit.FDeltaY-1;
     end
      else
      begin
         Screen.Cursor := crDefault;
         DBEdit.FDeltaX := 0;
         DBEdit.FDeltaY := 0;
         if Screen.ActiveControl = DBEdit.FControl then
         begin
           Application.MainForm.Show;
           GetParentForm(DBEdit).Show;
         end;
      end;
     Result := 1;
  end;
end;

{TxDBLookUpComboBox}
procedure TxDBLookUpComboBox.KeyPress(var Key: Char);
begin
  if not ListVisible and (Key = #27) then
    GetParentForm(Self).Perform(CM_DIALOGKEY, Byte(Key), 0)
    else
      Inherited;
end;

procedure TxDBLookUpComboBox.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TxLabel}
procedure TxLabel.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TxDBEdit}
procedure TxDBEdit.KeyPress(var Key: Char);
type PClass = ^TClass;
var AField: TField;
    SelfClass: TClass;
begin
  AField := nil;
  if (Datasource <> nil) and (DataSource.DataSet <> nil) and
     DataSource.DataSet.Active and Assigned(Field) then
  begin
     if (Field.FieldKind = fkData) and (Field.LookUpDataSet <> nil) then
        AField := Field.LookUpDataSet.FindField(Field.LookUpResultField);
  end;

  if AField = nil then
     AField := Field;

  SelfClass := PClass(Self)^;
  try
    PClass(Self)^ := Self.ClassParent.ClassParent;
    KeyPress(Key);
  finally
    PClass(Self)^ := SelfClass;
  end;

  if (Key in [#32..#255]) and (AField <> nil) and not AField.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      EditCanModify;
    #27:
      begin
        Reset;
        Key := #0;
      end;
  end;
end;

procedure TxDBEdit.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TxDBRichEdit}
procedure TxDBRichEdit.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TxDBSmartImage}
procedure TxDBSmartImage.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TDBDateEdit}
procedure TDBDateEdit.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TxButton}
procedure TxButton.CMDesignHitTest(var Message: TCMDesignHitTest);
begin
   if (Parent is TRaDBEdit) then
     PerformHitTest((Parent as TRaDBEdit), Message);
  if Message.Result = 0 then
     Inherited;
end;

{TRaDBEdit}
constructor TRaDBEdit.Create(AOwner: TComponent);
begin
  Inherited Create(AOwner);
  FButton := TxButton.Create(Self);
  FButton.Enabled := False;
  FButton.Visible := False;
  FButton.Glyph.LoadFromResourceName(HInstance,'EDITBUTTON');
  FButton.Parent := Self;
  FButton.ShowHint := True;
  if not (csDesigning in ComponentState) then
     FButton.OnClick := DoButtonClick;
  ControlStyle := ControlStyle + [csReplicatable];
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DoChanges;
  Width := 121;
  Height := 40;
  FDistanceY := 4;
  FTabStop := True;
  FLabel := TxLabel.Create(Self);
  FLabel.Parent := Self;
  if not (csDesigning in ComponentState) then
  begin
    FLabel.OnClick := DoOnClick;
    FLabel.OnDblClick := DoOnDblClick;
    FLabel.OnDragDrop := DoOnDragDrop;
    FLabel.OnDragOver := DoOnDragOver;
    FLabel.OnEndDrag := DoOnEndDrag;
    FLabel.OnMouseDown := DoOnMouseDown;
    FLabel.OnMouseMove := DoOnMouseMove;
    FLabel.OnMouseUp := DoOnMouseUp;
    FLabel.OnStartDrag := DoOnStartDrag;
  end
   else
     ControlStyle := ControlStyle + [csFramed];
end;

destructor TRaDBEdit.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TRaDBEdit.CMDesignHitTest(var Mess: TCMDesignHitTest);
begin
  PerformHitTest(Self, Mess);
  if Mess.Result = 0 then
     Inherited;
end;

procedure TRaDBEdit.CNReplaceField(var Message: TMessage);
var AField: TField;
    Form: TCustomForm;
begin
   AField := pointer(Message.LParam);
   if FControl.CanFocus then
   begin
     if Assigned(FOnReplaceField) then
        FOnReplaceField(AField)
       else
        ReplaceField(AField);
     FControl.SetFocus;
   end;
   Form := GetParentForm(Self);
   if (Form <> nil) and (Form.CanFocus) then
      Form.SetFocus;
end;

procedure TRaDBEdit.DoChanges(Sender: TObject);
const IAmHere: Boolean = False;
begin
   if (Parent is TRaDBBox) and (Parent as TRaDBBox).FixedFields then
      Exit;
   if IAmHere then Exit;
   IAmHere := True;
   try
     if (FControl = nil) or (Field <> FField) then
     begin
        Refresh(not ((FField = nil) and (Field.FieldName = DataField)));
        SetFieldAttributes(Field);
     end;
   finally
     IAmHere := False;
   end;
end;

procedure TRaDBEdit.DoEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(OnKeyDown) then
     OnKeyDown(Self, Key, Shift);
end;

procedure TRaDBEdit.DoEditKeyPress(Sender: TObject; var Key: Char);
begin
  if Assigned(OnKeyPress) then
     OnKeyPress(Self, Key);
end;

procedure TRaDBEdit.DoEditKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if Assigned(OnKeyUp) then
     OnKeyUp(Self, Key, Shift);
end;

procedure TRaDBEdit.DoButtonClick(Sender: TObject);
begin
   if (FControl <> nil) and (FControl.CanFocus) and not ReadOnly then
      FControl.SetFocus;
   if Assigned(FOnButtonClick) then
      FOnButtonClick(Self)
     else
      DoEditButtonClick(Field, Self);
end;

function TRaDBEdit.FieldType: TRaDBEditType;
var AField: TField;
begin
   AField := Field;
   if AField = nil then
      Result := efEmpty
     else
      begin
         Result := efEdit;
      end;
   if (Result = efEdit) and (AField.FieldKind = fkLookUp) and
      (EditKind <> ekView) then
      Result := efLookUp;
   if Result = efEdit then
     case AField.DataType of
       ftMemo, ftFmtMemo:
           Result := efMemo;
       ftGraphic, ftTypedBinary, ftBlob:
           Result := efImage;
       ftDate, ftDateTime:
           Result := efDate;
       {$IFNDEF VER110}
       {$ELSE}
       ftDataSet:
           Result := efDataSet;
       {$ENDIF}
     end;
end;

procedure TRaDBEdit.CreateControl;
begin
   case FieldType of
     efLookUp:
         begin
            FControl := TxDBLookUpComboBox.Create(Self);
            FControl.Parent := Self;
            with FControl as TxDBLookUpComboBox do
            begin
               DataSource := nil;
               DataField := Self.DataField;
               DataSource := Self.DataSource;
               TabStop := Self.TabStop;

               if not (csDesigning in Self.ComponentState) then
               begin
                 OnKeyDown := DoEditKeyDown;
                 OnKeyPress := Self.DoEditKeyPress;
                 OnKeyUp := Self.DoEditKeyUp;
                 OnClick := Self.DoOnClick;
                 OnDblClick := Self.DoOnDblClick;
                 OnDragDrop := Self.DoOnDragDrop;
                 OnDragOver := Self.DoOnDragOver;
                 OnEndDrag := Self.DoOnEndDrag;
                 OnMouseDown := Self.DoOnMouseDown;
                 OnMouseMove := Self.DoOnMouseMove;
                 OnMouseUp := Self.DoOnMouseUp;
                 OnStartDrag := Self.DoOnStartDrag;
                 OnEnter := Self.DoOnEnter;
                 OnExit := Self.DoOnExit;
               end;
            end;
         end;
    efDate:
         begin
            FControl := TDBDateEdit.Create(Self);
            if csDesigning in ComponentState then
              FControl.Enabled := False;
            FControl.Parent := Self;
            with FControl as TDBDateEdit do
            begin
               DataSource := nil;
               DataField := Self.DataField;
               DataSource := Self.DataSource;
               ReadOnly := Self.ReadOnly;
               TabStop := Self.TabStop;

               if not (csDesigning in Self.ComponentState) then
               begin
                 OnKeyDown := DoEditKeyDown;
                 OnKeyPress := Self.DoEditKeyPress;
                 OnKeyUp := Self.DoEditKeyUp;
                 OnClick := Self.DoOnClick;
                 OnDblClick := Self.DoOnDblClick;
                 OnDragDrop := Self.DoOnDragDrop;
                 OnDragOver := Self.DoOnDragOver;
                 OnEndDrag := Self.DoOnEndDrag;
                 OnMouseDown := Self.DoOnMouseDown;
                 OnMouseMove := Self.DoOnMouseMove;
                 OnMouseUp := Self.DoOnMouseUp;
                 OnStartDrag := Self.DoOnStartDrag;
                 OnEnter := Self.DoOnEnter;
                 OnExit := Self.DoOnExit;
               end;

               if EditKind = ekView then
               begin
                  ParentColor := True;
                  ReadOnly := True;
                  TabStop := False;
                  BorderStyle := bsNone;
               end
            end;
         end;
    efImage :
         begin
            FControl := TxDBSmartImage.Create(Self);
            FControl.Parent := Self;
            with FControl as TxDBSmartImage do
            begin
               AutoDisplay := ShowBlob;
               DataSource := nil;
               DataField := Self.DataField;
               DataSource := Self.DataSource;
               if FShowBlob then
               begin
                  Width := Picture.Width;
                  Height := Picture.Height;
               end
                 else
                 begin
                    Width := 0;
                    Height := 21;
                 end;
               Visible := FShowBlob;
               if Visible then
                  ControlStyle := ControlStyle - [csNoDesignVisible]
                 else
                  ControlStyle := ControlStyle + [csNoDesignVisible];
               ReadOnly := Self.ReadOnly;
               TabStop := Self.TabStop;

               if not (csDesigning in Self.ComponentState) then
               begin
                 OnKeyDown := DoEditKeyDown;
                 OnKeyPress := Self.DoEditKeyPress;
                 OnKeyUp := Self.DoEditKeyUp;
                 OnClick := Self.DoOnClick;
                 OnDblClick := Self.DoOnDblClick;
                 OnDragDrop := Self.DoOnDragDrop;
                 OnDragOver := Self.DoOnDragOver;
                 OnEndDrag := Self.DoOnEndDrag;
                 OnMouseDown := Self.DoOnMouseDown;
                 OnMouseMove := Self.DoOnMouseMove;
                 OnMouseUp := Self.DoOnMouseUp;
                 OnStartDrag := Self.DoOnStartDrag;
                 OnEnter := Self.DoOnEnter;
                 OnExit := Self.DoOnExit;
               end;

               if EditKind = ekView then
               begin
                  ParentColor := True;
                  TabStop := False;
                  ReadOnly := True;
                  BorderStyle := bsNone;
               end
            end;
         end;
    efMemo :
         begin
            FControl := TxDBRichEdit.Create(Self);
            FControl.Parent := Self;
            with FControl as TxDBRichEdit do
            begin
               AutoDisplay := ShowBlob;
               DataSource := nil;
               DataField := Self.DataField;
               DataSource := Self.DataSource;
               if FShowBlob then
               begin
                  Width := 300;
                  Height := 100;
               end
                 else
                 begin
                    Width := 0;
                    Height := 21;
                 end;
               Visible := FShowBlob;
               if Visible then
                  ControlStyle := ControlStyle - [csNoDesignVisible]
                 else
                  ControlStyle := ControlStyle + [csNoDesignVisible];
               ScrollBars := ssBoth;
               ReadOnly := Self.ReadOnly;
               TabStop := Self.TabStop;

               if not (csDesigning in Self.ComponentState) then
               begin
                 OnKeyDown := DoEditKeyDown;
                 OnKeyPress := Self.DoEditKeyPress;
                 OnKeyUp := Self.DoEditKeyUp;
                 OnClick := Self.DoOnClick;
                 OnDblClick := Self.DoOnDblClick;
                 OnDragDrop := Self.DoOnDragDrop;
                 OnDragOver := Self.DoOnDragOver;
                 OnEndDrag := Self.DoOnEndDrag;
                 OnMouseDown := Self.DoOnMouseDown;
                 OnMouseMove := Self.DoOnMouseMove;
                 OnMouseUp := Self.DoOnMouseUp;
                 OnStartDrag := Self.DoOnStartDrag;
                 OnEnter := Self.DoOnEnter;
                 OnExit := Self.DoOnExit;
               end;

               if EditKind = ekView then
               begin
                  ParentColor := True;
                  TabStop := False;
                  ReadOnly := True;
                  BorderStyle := bsNone;
               end
            end;
         end;
    efDataSet :
         begin
            FControl := TxDBEdit.Create(Self);
            FControl.Parent := Self;
            with FControl as TxDBEdit do
            begin
               DataSource := nil;
               DataField := Self.DataField;
               DataSource := Self.DataSource;
               Width := 0;
               Height := 21;
               Visible := False;
               ControlStyle := ControlStyle + [csNoDesignVisible];
               TabStop := False;
               ReadOnly := True;
               BorderStyle := bsNone;

               if not (csDesigning in Self.ComponentState) then
               begin
                 OnKeyDown := DoEditKeyDown;
                 OnKeyPress := Self.DoEditKeyPress;
                 OnKeyUp := Self.DoEditKeyUp;
                 OnClick := Self.DoOnClick;
                 OnDblClick := Self.DoOnDblClick;
                 OnDragDrop := Self.DoOnDragDrop;
                 OnDragOver := Self.DoOnDragOver;
                 OnEndDrag := Self.DoOnEndDrag;
                 OnMouseDown := Self.DoOnMouseDown;
                 OnMouseMove := Self.DoOnMouseMove;
                 OnMouseUp := Self.DoOnMouseUp;
                 OnStartDrag := Self.DoOnStartDrag;
                 OnEnter := Self.DoOnEnter;
                 OnExit := Self.DoOnExit;
               end;
            end;
         end;
       else
         begin
            FControl := TxDBEdit.Create(Self);
            FControl.Parent := Self;
            with FControl as TxDBEdit do
            begin
               DataSource := nil;
               DataField := Self.DataField;
               DataSource := Self.DataSource;
               ReadOnly := Self.ReadOnly;
               TabStop := Self.TabStop;
               if EditKind = ekView then
               begin
                  ParentColor := True;
                  TabStop := False;
                  ReadOnly := True;
                  BorderStyle := bsNone;
               end;

               if not (csDesigning in Self.ComponentState) then
               begin
                 OnKeyDown := DoEditKeyDown;
                 OnKeyPress := Self.DoEditKeyPress;
                 OnKeyUp := Self.DoEditKeyUp;
                 OnClick := Self.DoOnClick;
                 OnDblClick := Self.DoOnDblClick;
                 OnDragDrop := Self.DoOnDragDrop;
                 OnDragOver := Self.DoOnDragOver;
                 OnEndDrag := Self.DoOnEndDrag;
                 OnMouseDown := Self.DoOnMouseDown;
                 OnMouseMove := Self.DoOnMouseMove;
                 OnMouseUp := Self.DoOnMouseUp;
                 OnStartDrag := Self.DoOnStartDrag;
                 OnEnter := Self.DoOnEnter;
                 OnExit := Self.DoOnExit;
               end;
            end;
         end;
    end;
end;

function TRaDBEdit.GetGlyph: TBitmap;
begin
  Result := FButton.Glyph;
end;

procedure TRaDBEdit.SetGlyph(Value: TBitmap);
begin
  FButton.Glyph := Value;
end;

function TRaDBEdit.GetButtonHint: String;
begin
   Result := FButton.Hint;
end;

procedure TRaDBEdit.SetButtonHint(Value: String);
begin
   FButton.Hint := Value;
end;

function TRaDBEdit.GetNumGlyphs: TNumGlyphs;
begin
   Result := FButton.NumGlyphs;
end;

procedure TRaDBEdit.SetNumGlyphs(Value: TNumGlyphs);
begin
   FButton.NumGlyphs := Value;
end;

procedure TRaDBEdit.ClickButton;
begin
   if FButton.Enabled then
       FButton.Click;
end;

procedure TRaDBEdit.Refresh(Resize: boolean);
var AWidth, AHeight: Integer;
begin
  if FChanging then Exit;
  AWidth := 0;
  AHeight := 0;
  FreeControl;
  CreateControl;
  with FControl do
  begin
     if (Self.Field <> nil) then
     begin
        case FieldType of
          efEdit, efDate:
            Width := GetDefaultWidth(Font, Self.Field.DisplayWidth);
          efLookUp:
            Width := GetDefaultWidth(Font, Self.Field.DisplayWidth)+
                                                GetSystemMetrics(SM_CXVSCROLL);
        end;
        if not (FieldType in [efMemo, efImage]) then
             Width := Width+4;
        if FCaption = '' then
           FLabel.Caption := Self.Field.DisplayLabel;
        Self.MaxLength := Self.Field.DisplayWidth;
     end
      else
        if (Self.DataField <> '') then
          if (FCaption <> '') then
             FLabel.Caption := FCaption
              else
               FLabel.Caption := Self.DataField;
     if FDistanceX = 0 then
     begin
        Top := FLabel.Height + FDistanceY;
        AWidth := FLabel.Width;
     end
       else
        begin
           Left := FLabel.Width + FDistanceX;
           AHeight := FLabel.Height;
        end;

     FButton.Visible := (EditKind = ekControl) or
                        ((FieldType in [efMemo, efImage]) and not ShowBlob);
     FButton.Enabled := (EditKind = ekControl) or
                        ((FieldType in [efMemo, efImage]) and not ShowBlob);
     if FButton.Visible then
     begin
        FButton.Top := Top+2;
        FButton.Height := GetSystemMetrics(SM_CXVSCROLL);
        FButton.Width := FButton.Height;
        FButton.Left := Left+Width+1
     end
      else
        begin
           FButton.Left := Left+Width;
           FButton.Width := 0;
        end;
     if FButton.Left+FButton.Width > AWidth then
     begin
        AWidth := FButton.Left+FButton.Width;
     end;
     if Top + Height > AHeight then
        AHeight := Top+Height;
  end;
  FField := Field;
  ReSize := Resize and not (csLoading in ComponentState);
  if ReSize then
     SetBounds(Left, Top, AWidth, AHeight)
    else SetBounds(Left, Top, Width, Height);
end;

procedure TRaDBEdit.FreeControl;
begin
   if (FControl <> nil) and not (csDestroying in FControl.ComponentState) then
   begin
      FControl.Free;
      FControl := nil;
   end;
   if FCaption <> '' then
      FLabel.Caption := FCaption
      else
      FLabel.Caption := Self.Name;
end;

function TRaDBEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TRaDBEdit.Notification(AComponent: TComponent;
                                             Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = FDataLink.DataSource) then
      FDataLink.DataSource := nil;
end;

procedure TRaDBEdit.SetParent(AParent: TWinControl);
var ErrorMessage: String;
begin
   if (AParent <> nil) and
      (AParent is TRaDBBox) and
      (Owner <> AParent) and
      ((AParent as TRaDBBox).CreateMode = cmAuto) then
   begin
     ErrorMessage :=
       Format(SCantInsertComponent, [AParent.Name, Self.Name]);
     Raise EInvalidOperation.Create(ErrorMessage);
   end;
   Inherited SetParent(AParent);
end;

procedure TRaDBEdit.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TRaDBEdit.GetDataField: string;
begin
   Result := FDataLink.FieldName;
end;

procedure TRaDBEdit.SetDataField(const Value: string);
var i: Integer;
    AnEditKind: TEditKind;
begin
  if Value <> FDataLink.FieldName then
  begin
     FDataLink.FieldName := Value;
     if (Field <> nil) and
        (Field.ReadOnly or (Field.FieldKind in [fkCalculated, fkInternalCalc])) then
        FEditKind := ekView
       else
     if FieldType = efDate then
        FEditKind := ekControl
       else
     if FieldType = efDataSet then
        FEditKind := ekControl
       else
     if IsLookUpField(Field) then
     begin
        FEditKind := ekControl;
        if (DataSource <> nil) and
           (DataSource.DataSet <> nil) and
           (FDataLink.Field <> nil) then
           for i := 0 to DataSource.DataSet.FieldCount-1 do
             with DataSource.DataSet.Fields[i] do
             begin
                if UpperCase(FDataLink.Field.KeyFields) = UpperCase(FieldName) then
                begin
                   if (DataSource.DataSet.Fields[i].Visible) and
                      (DataSource.DataSet.Fields[i] <> Self.Field) then
                   begin
                     FEditKind := ekCommon;
                     TabStop := False;
                   end;
                   if DataSource.DataSet.Fields[i].ReadOnly or
                      (DataSource.DataSet.Fields[i].FieldKind in
                                           [fkCalculated, fkInternalCalc])then
                     FEditKind := ekView;
                   break;
                end;
             end;
     end
       else
       begin
          AnEditKind := ekCommon;
          if (DataSource <> nil) and (DataSource.DataSet <> nil) then
             for i := 0 to DataSource.DataSet.FieldCount-1 do
             with DataSource.DataSet.Fields[i] do
               if UpperCase(KeyFields) = UpperCase(FDataLink.FieldName) then
               begin
                  AnEditKind := ekControl;
                  break;
               end;
          FEditKind := AnEditKind;
       end;
       if FEditKind = ekView then
       begin
          FLabel.ParentFont := False;
          Font.Style := [fsBold];
          Font.Color := clNavy;
       end
         else
         begin
            ParentFont := True;
            FLabel.ParentFont := True;
         end;
       Refresh(True);
  end;
end;

function TRaDBEdit.GetOnButtonClick: TNotifyEvent;
begin
   Result := FOnButtonClick;
end;

procedure TRaDBEdit.SetOnButtonClick(Value: TNotifyEvent);
begin
   FOnButtonClick := Value;
end;

function TRaDBEdit.GetField: TField;
begin
  Result := FDataLink.Field;
  if Result = nil then
    if (FDataLink.DataSet <> nil) then
       Result := FDataLink.DataSet.FindField(FDataLink.FieldName);
end;

function TRaDBEdit.GetMaxLength: Integer;
begin
   case FieldType of
     efEdit: Result := (FControl as TxDBEdit).MaxLength;
    else
     Result := 0;
   end;
end;

procedure TRaDBEdit.SetMaxLength(Value: Integer);
begin
   if Value <> MaxLength then
   case FieldType of
     efEdit: (FControl as TxDBEdit).MaxLength := Value;
   end;
end;

function TRaDBEdit.GetReadOnly: Boolean;
begin
   Result := not TabStop;
end;

procedure TRaDBEdit.SetTabStop(Value: Boolean);
begin
   FTabStop := Value;
   FControl.TabStop := Value;
   if FControl is TxDBLookUpComboBox then
     (FControl as TxDBLookUpComboBox).ReadOnly := not Value
    else
   if FControl is TDBDateEdit then
     (FControl as TDBDateEdit).ReadOnly := not Value
    else
   if FControl is TDBSmartImage then
     (FControl as TDBSmartImage).ReadOnly := not Value
    else
   if FControl is TDBRichEdit then
     (FControl as TDBRichEdit).ReadOnly := not Value
    else
   if FControl is TxDBEdit then
     (FControl as TxDBEdit).ReadOnly := True;
end;

procedure TRaDBEdit.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
   Inherited SetBounds(ALeft, ATop, AWidth, AHeight);
   if (FControl <> nil) then
   begin
      FButton.Left := Width - FButton.Width;
      if FControl.Visible then
      begin
         if FButton.Width <> 0 then
           FControl.Width := FButton.Left - 1 - FControl.Left
          else
           FControl.Width := Width - FControl.Left;
      end;
      FControl.Height := Height - FControl.Top;
      Inherited SetBounds(Left, Top, Width, FControl.Top+FControl.Height);
   end;
end;

procedure TRaDBEdit.SetEditKind(Value: TEditKind);
begin
   FEditKind := Value;
   Refresh(True);
end;

procedure TRaDBEdit.SetDistanceX(Value: Integer);
begin
   if Value < 0 then
        Raise Exception.Create(SInvalidValue)
       else
       if Value <> FDistanceX then
       begin
          FDistanceX := Value;
          if FDistanceX = 0 then FDistanceY := 1
                            else FDistanceY := 0;
          Refresh(True);
       end;
end;

procedure TRaDBEdit.SetDistanceY(Value: Integer);
begin
   if not (Value in [0..100]) then
        Raise Exception.Create(SInvalidRange)
       else
       if Value <> FDistanceY then
       begin
          FDistanceY := Value;
          if FDistanceY = 0 then FDistanceX := 1
                            else FDistanceX := 0;
          Refresh(True);
       end;
end;

procedure TRaDBEdit.SetName(const NewName: TComponentName);
begin
   Inherited SetName(NewName);
   if not (csLoading in ComponentState) then
      Refresh(True);
end;

function TRaDBEdit.GetCaption: String;
begin
   Result := FLabel.Caption
end;

procedure TRaDBEdit.SetCaption(Value: String);
begin
   if Value <> FLabel.Caption then
   begin
      FCaption := Value;
      Refresh(True);
   end;
end;

function TRaDBEdit.GetLabelFont: TFont;
begin
   Result := FLabel.Font;
end;

procedure TRaDBEdit.SetLabelFont(AFont: TFont);
begin
   if (AFont <> nil) then
   begin
      FLabel.Font.Assign(AFont);
      Refresh(True);
   end;
end;

function TRaDBEdit.GetLabelParentFont: Boolean;
begin
   Result := FLabel.ParentFont;
end;

procedure TRaDBEdit.SetLabelParentFont(Value: Boolean);
begin
   FLabel.ParentFont := Value;
end;

procedure TRaDBEdit.SetShowBlob(Value: Boolean);
begin
   if Value <> FShowBlob then
   begin
      FShowBlob := Value;
      Refresh(True);
   end;
end;

procedure TRaDBEdit.SetOnReplaceField(const Value: TFieldNotifyEvent);
begin
  FOnReplaceField := Value;
end;

procedure TRaDBEdit.DoOnClick(Sender: TObject);
begin
  if Assigned(OnClick) then
     OnClick(Self);
end;

procedure TRaDBEdit.DoOnDblClick(Sender: TObject);
begin
  if Assigned(OnDblClick) then
    OnDblClick(Self);
end;

procedure TRaDBEdit.DoOnDragDrop(Sender, Source: TObject; X, Y: Integer);
begin
  if Assigned(OnDragDrop) then
  begin
     if (Sender is TControl) and ((Sender as TControl).Parent = Self) then
     begin
        X := X + (Sender as TControl).Left;
        Y := Y + (Sender as TControl).Top;
     end;
     OnDragDrop(Self, Source, X, Y);
  end;
end;

procedure TRaDBEdit.DoOnDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  if Assigned(OnDragOver) then
  begin
     if (Sender is TControl) and ((Sender as TControl).Parent = Self) then
     begin
        X := X + (Sender as TControl).Left;
        Y := Y + (Sender as TControl).Top;
     end;
     OnDragOver(Self, Source, X, Y, State, Accept);
  end;
end;

procedure TRaDBEdit.DoOnEndDrag(Sender, Target: TObject; X, Y: Integer);
begin
  if Assigned(OnEndDrag) then
    OnEndDrag(Self, Target, X, Y);
end;

procedure TRaDBEdit.DoOnEnter(Sender: TObject);
begin
  if Assigned(OnEnter) then
    OnEnter(Self);
end;

procedure TRaDBEdit.DoOnExit(Sender: TObject);
begin
  if Assigned(OnExit) then
    OnExit(Self);
end;

procedure TRaDBEdit.DoOnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(OnMouseDown) then
  begin
     if (Sender is TControl) and ((Sender as TControl).Parent = Self) then
     begin
        X := X + (Sender as TControl).Left;
        Y := Y + (Sender as TControl).Top;
     end;
     OnMouseDown(Self, Button, Shift, X, Y);
  end;
end;

procedure TRaDBEdit.DoOnMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  if Assigned(OnMouseMove) then
  begin
     if (Sender is TControl) and ((Sender as TControl).Parent = Self) then
     begin
        X := X + (Sender as TControl).Left;
        Y := Y + (Sender as TControl).Top;
     end;
     OnMouseMove(Self, Shift, X, Y);
  end;
end;

procedure TRaDBEdit.DoOnMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(OnMouseUp) then
  begin
     if (Sender is TControl) and ((Sender as TControl).Parent = Self) then
     begin
        X := X + (Sender as TControl).Left;
        Y := Y + (Sender as TControl).Top;
     end;
     OnMouseUp(Self, Button, Shift, X, Y);
  end;
end;

procedure TRaDBEdit.DoOnStartDrag(Sender: TObject;
  var DragObject: TDragObject);
begin
   if Assigned(OnStartDrag) then
     OnStartDrag(Self, DragObject);
end;

end.
