(*////////////////////////////////////////////////////////////////////////////
//   Part of AlexSoft VCL/DLL Library.                                      //
//   All rights reserved. (c) Copyright 1998.                               //
//   Created by: Alex Rabichooc                                             //
//**************************************************************************//
//  Users of this unit must accept this disclaimer of warranty:             //
//    "This unit is supplied as is. The author disclaims all warranties,    //
//    expressed or implied, including, without limitation, the warranties   //
//    of merchantability and of fitness for any purpose.                    //
//    The author assumes no liability for damages, direct or                //
//    consequential, which may result from the use of this unit."           //
//                                                                          //
//  This Unit is donated to the public as public domain.                    //
//                                                                          //
//  This Unit can be freely used and distributed in commercial and          //
//  private environments provided this notice is not modified in any way.   //
//                                                                          //
//  If you do find this Unit handy and you feel guilty for using such a     //
//  great product without paying someone - sorry :-)                        //
//                                                                          //
//  Please forward any comments or suggestions to Alex Rabichooc at:        //
//                                                                          //
//  a_rabichooc@yahoo.com or alex@carmez.mldnet.com                         //
/////////////////////////////////////////////////////////////////////////////*)

{---------------------------------------------------------------------------
  TRaDBBox - Data-aware component like TDBCtrlGrid
     properties
        DataSource: TDataSource;
           Defines DataSource of Component.
        ShowBlobs: Boolean;
           Determines whether Blob fileds will be shown automatically.
        AllowDelete: Boolean;
           Determines whether user can delete records from DataSet.
        AllowInsert: Boolean;
           Determines whether user can insert records into DataSet.
        ConfirmDelete: Boolean;
           Determines whether confirm dialog will be shown when user
           will try to delete a record from the DataSet.
        property ReadOnly: Boolean;
           If this property is True then user can't modify the DataSet.
        property CreateMode: TCreateMode;
           cmAuto - The Component automatically will create editing fields.
           cmManual - The User manually describes editing fields.
----------------------------------------------------------------------------}

unit dbBoxGrd;

interface
uses Windows, Messages, SysUtils, Classes, Controls, Graphics, Forms,
     DB, StdCtrls, DBCtrls, stdUtils;

type
  TRaDBBox = class;

  TPanelDataLink = class(TDataLink)
  private
    FdbBoxGrid: TRaDBBox;
  protected
    procedure ActiveChanged; override;
    procedure LayoutChanged; override;
  end;


  TRaDBBoxKey = (bgkNull, bgkEditMode, bgkPriorTab, bgkNextTab, bgkFirstTab,
    bgkLastTab, bgkLeft, bgkRight, bgkUp, bgkDown, bgkScrollUp, bgkScrollDown,
    bgkPageUp, bgkPageDown, bgkHome, bgkEnd, bgkInsert, bgkAppend, bgkDelete,
    bgkEditButton);

  TCreateMode = (cmAuto, cmManual);
  TOrientation = (orVertical, orHorizontal);
  TRaDBBox = class(TScrollBox)
  private
    FRefreshing: Boolean;
    FFixedFields: Boolean;
    FAllowInsert: Boolean;
    FAllowDelete: Boolean;
    FConfirmDelete: Boolean;
    FAllowMove: boolean;
    FReadOnly: Boolean;
    FCreateMode: TCreateMode;
    FDataLink: TPanelDataLink;
    FOrientation: TOrientation;
    FLabelFont: TFont;
    FOrigin: TPoint;
    FShowBlobs: Boolean;
    function CreateName(FName: String): String;
    procedure CMChildKey(var Message: TCMChildKey); message CM_CHILDKEY;
    procedure CNActiveChanged(var Message: TMessage); message CN_ACTIVECHANGED;
    procedure CNCloseDBForm(var Message: TMessage); Message CN_CLOSEDBFORM;
    function GetActualHeight: Integer;
    function GetActualWidth: Integer;
    function GetEditMode: Boolean;
    procedure SetOrigin(Value: TPoint);
    procedure SetOriginX(Value: Integer);
    procedure SetOriginY(Value: Integer);
    procedure SetReadOnly(Value: Boolean);
    procedure SetEditMode(Value: Boolean);
    procedure SetShowBlobs(Value: Boolean);
  protected
    function GetGridKey(var Key: Word; Shift: TShiftState): TRaDBBoxKey; virtual;
    procedure DoExit; override;
    function FindNext(StartControl: TWinControl; GoForward: Boolean;
                                     var WrapFlag: Integer): TWinControl; virtual;
    procedure Scroll(Inc: Integer);
    procedure SelectNext(GoForward: Boolean);
    procedure SelectLast;
    procedure SetCreateMode(Value: TCreateMode);
    function GetDataSource: TDataSource;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
                                          Operation: TOperation); override;
    procedure RefreshFields; virtual;
    procedure SetDataSource(Value: TDataSource); virtual;
    procedure SetOrientation(Value: TOrientation); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DoKey(Key: TRaDBBoxKey);
    procedure StartRefresh;
    procedure SetLabelFont(AFont: TFont);
    procedure SetToActualHeight;
    procedure SetToActualWidth;
    property FixedFields: Boolean read FFixedFields write FFixedFields;
    property EditMode: Boolean read GetEditMode write SetEditMode;
    property ActualHeight: Integer read GetActualHeight;
    property ActualWidth: Integer read GetActualWidth;
    property Refreshing: boolean read FRefreshing;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
  published
    property ShowBlobs: Boolean read FShowBlobs write SetShowBlobs;
    property AllowDelete: Boolean read FAllowDelete write FAllowDelete default True;
    property AllowInsert: Boolean read FAllowInsert write FAllowInsert default True;
    property AllowMove: Boolean read FAllowMove write FAllowMove default True;
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True;
    property ReadOnly: Boolean read FReadOnly write SetReadOnly;
    property Orientation: TOrientation read FOrientation write SetOrientation;
    property CreateMode: TCreateMode read FCreateMode write SetCreateMode;
    property OriginX: integer read FOrigin.X write SetOriginX;
    property OriginY: integer read FOrigin.Y write SetOriginY;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property Font;
    property LabelFont: TFont read FLabelFont write SetLabelFont;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
  end;

implementation

uses grids, dbEdFld, Dialogs, DbConsts, dbTools, DbXCnsts;

{TPanelDataLink}
procedure TPanelDataLink.ActiveChanged;
begin
  if (FdbBoxGrid <> nil) then
      FdbBoxGrid.StartRefresh;
end;

procedure TPanelDataLink.LayoutChanged;
begin
  Inherited LayoutChanged;
  if (FdbBoxGrid <> nil) then
      FdbBoxGrid.StartRefresh;
end;

{TRaDBBox}
constructor TRaDBBox.Create(AOwner: TComponent);
begin
   Inherited Create(AOwner);
   FDataLink := TPanelDataLink.Create;
   FDataLink.FdbBoxGrid := Self;
   FLabelFont := TFont.Create;
   Height := 100;
   Inherited TabStop := False;
   FAllowInsert := True;
   FAllowDelete := True;
   FConfirmDelete := True;
   FAllowMove := True;
   FOrigin.X := 10;
   FOrigin.Y := 10;
end;

destructor TRaDBBox.Destroy;
begin
   FDataLink.Free;
   FDataLink := nil;
   if FLabelFont <> nil then
      FLabelFont.Free;
   Inherited Destroy;
end;

function TRaDBBox.GetDataSource: TDataSource;
begin
   Result := FDataLink.DataSource;
end;

procedure TRaDBBox.Notification(AComponent: TComponent;
                                                      Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
     (AComponent = DataSource) then
      DataSource := nil;
end;

function TRaDBBox.CreateName(FName: String): String;
var i: Integer;
    AName: String;
begin
   AName := Format('%s', [FName]);
   i := 0;
   while Owner.FindComponent(AName) <> nil do
   begin
      inc(i);
      AName := Format('%s%d', [FName,i]);
   end;
   for i := 1 to Length(AName) do
     if not (AName[i] in ['A'..'Z', 'a'..'z', '0'..'9', '_']) then
       AName[i] := '_';
   Result := AName;
end;

procedure TRaDBBox.RefreshFields;

  procedure RemoveFields;
  begin
     while ComponentCount > 0 do
        Components[0].Free;
  end;

  procedure CreateFields;
  var i: Integer;
      Field: TField;
      DBEditField: TRaDBEdit;
      ATop, ALeft, AWidth, AHeight, LineHeight, LineDistance: integer;
      AVisible: Boolean;
      MaxLabelLength: Integer;
  begin
     LineDistance := 10;
     if not FDataLink.Active then
        exit;
     AVisible := Visible;
     try
        Visible := False;
        ATop := FOrigin.Y;
        ALeft := FOrigin.X;
        MaxLabelLength := 0;
        LineHeight := 0;
        for i := 0 to FDataLink.DataSet.FieldCount - 1  do
        begin
           Field := FDataLink.DataSet.Fields[i];
           if(Field.Visible) then
           begin
              DBEditField := TRaDBEdit.Create(Self);
              DBEditField.Parent := Self;
              DBEditField.ShowBlob := FShowBlobs;
              DBEditField.DataSource := DataSource;
              DBEditField.DataField := Field.FieldName;
              DBEditField.TabStop :=
                   not (Field.ReadOnly or
                           (Field.FieldKind in [fkCalculated, fkInternalCalc]));
              if ReadOnly then
              begin
                 if DBEditField.EditKind = ekControl then
                     DBEditField.EditKind := ekCommon;
                 if DBEditField.FieldType = efLookUp then
                     DBEditField.EditKind := ekView;
              end;
              DBEditField.Name := CreateName('df'+Field.FieldName);
              DBEditField.Top := ATop;
              DBEditField.Left := ALeft;
              DBEditField.Font := Font;
              if DBEditField.EditKind = ekView then
              begin
                DBEditField.Font.Style := [fsBold];
                DBEditField.Font.Color := clNavy;
              end;
              DBEditField.LabelFont := FLabelFont;
              if Orientation = orVertical then
              begin
                 DBEditField.DistanceX := 8;
                 if MaxLabelLength < DBEditField.CLabel.Width then
                    MaxLabelLength := DBEditField.CLabel.Width;
                 if ComponentCount > 1 then
                 begin
                    ATop := ATop+LineDistance;
                    if ComponentCount > 1 then
                        Inc(ATop,
                           (Components[ComponentCount-2] as TControl).Height);
                    DBEditField.Left := ALeft;
                    DBEditField.Top := ATop;
                 end;
              end
                else
                if ComponentCount > 1 then
                begin
                   ATop := (Components[ComponentCount-2] as TControl).Top;
                   ALeft := (Components[ComponentCount-2] as TControl).Left;
                   AWidth := (Components[ComponentCount-2] as TControl).Width;
                   AHeight := (Components[ComponentCount-2] as TControl).Height;
                   if LineHeight < AHeight+LineDistance then
                           LineHeight := AHeight+LineDistance;
                   if ALeft + AWidth + LineDistance + DbEditField.Width >
                            Self.ClientWidth-GetSystemMetrics(SM_CXVSCROLL) then
                   begin
                      ALeft := FOrigin.X;
                      ATop := ATop + LineHeight;
                   end
                     else
                        ALeft := ALeft + AWidth + LineDistance;
                   DBEditField.Top := ATop;
                   DBEditField.Left := ALeft;
                end;
           end;
       end;
       if Orientation = orVertical then
       begin
           i := 0;
           while i < ComponentCount do
           begin
              DBEditField := Components[i] as TRaDBEdit;
              DBEditField.DistanceX :=
                                8 + MaxLabelLength - DBEditField.CLabel.Width;
              inc(i);
           end;
       end;
      finally
       Visible := AVisible;
     end;
  end;

begin
   if (CreateMode = cmAuto) and not FFixedFields then
   begin
      RemoveFields;
      CreateFields;
   end;
end;

procedure TRaDBBox.SetOrientation(Value: TOrientation);
begin
   if Value <> FOrientation then
   begin
      FOrientation := Value;
      StartRefresh;
   end;
end;

procedure TRaDBBox.StartRefresh;
begin
  if not FRefreshing then
  begin
     FRefreshing := True;
     PostMessage(Handle, CN_ACTIVECHANGED, 0, 0);
  end;
end;

procedure TRaDBBox.CMChildKey(var Message: TCMChildKey);
var
  GridKey: TRaDBBoxKey;
begin
  with Message do
    if Sender <> Self then
    begin
      GridKey := GetGridKey(CharCode, GetShiftState);
      if Assigned(OnKeyDown) then OnKeyDown(Sender, CharCode, GetShiftState);
      if (GridKey <> bgkNull) or ReadOnly then
      begin
        DoKey(GridKey);
        Result := 1;
        Exit;
      end;
    end;
  inherited;
end;

procedure TRaDBBox.CNActiveChanged(var Message: TMessage);
begin
   RefreshFields;
   FRefreshing := False;
end;

procedure TRaDBBox.CNCloseDBForm(var Message: TMessage);
begin
   if (DataSource <> nil) and
      (DataSource.DataSet <> nil) then
     if Message.WParam = mrCancel then
        DataSource.DataSet.Cancel
       else
        DataSource.DataSet.CheckBrowseMode;
end;

procedure TRaDBBox.DoExit;
begin
   if (DataSource <> nil) and
      (DataSource.DataSet <> nil) and
      (DataSource.DataSet.Modified) then
     DataSource.DataSet.Post;
   Inherited DoExit;
end;

function TRaDBBox.FindNext(StartControl: TWinControl;
 GoForward: Boolean; var WrapFlag: Integer): TWinControl;
var
  I, StartIndex: Integer;
  List: TList;
begin
  List := TList.Create;
  try
    StartIndex := 0;
    I := 0;
    Result := StartControl;
    GetTabOrderList(List);
    if List.Count > 0 then
    begin
      StartIndex := List.IndexOf(StartControl);
      if StartIndex = -1 then
        if GoForward then
          StartIndex := List.Count - 1 else
          StartIndex := 0;
      I := StartIndex;
      repeat
        if GoForward then
        begin
          Inc(I);
          if I = List.Count then I := 0;
        end else
        begin
          if I = 0 then I := List.Count;
          Dec(I);
        end;
        Result := List[I];
      until (Result.CanFocus and Result.TabStop) or (I = StartIndex);
    end;
    WrapFlag := 0;
    if GoForward then
    begin
      if I <= StartIndex then WrapFlag := 1;
    end else
    begin
      if I >= StartIndex then WrapFlag := -1;
    end;
  finally
    List.Free;
  end;
end;

function TRaDBBox.GetActualHeight: Integer;
var i: Integer;
begin
   Result := 0;
   for i := 0 to ControlCount - 1 do
     if Result < Controls[i].Top + Controls[i].Height then
         Result := Controls[i].Top + Controls[i].Height;
   if BorderStyle <> bsNone then
      Inc(Result, 2);
end;

function TRaDBBox.GetActualWidth: Integer;
var i: Integer;
begin
   Result := 0;
   for i := 0 to ControlCount - 1 do
     if Result < Controls[i].Left + Controls[i].Width then
         Result := Controls[i].Left + Controls[i].Width;
   if BorderStyle <> bsNone then
      Inc(Result, 2);
end;

function TRaDBBox.GetEditMode: Boolean;
begin
  Result := not Focused and ContainsControl(FindControl(GetFocus));
end;

procedure TRaDBBox.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  DoKey(GetGridKey(Key, Shift));
end;

function TRaDBBox.GetGridKey(var Key: Word; Shift: TShiftState): TRaDBBoxKey;
var
  GridKey: TRaDBBoxKey;
  ParentForm: TCustomForm;
  IsLookup: Boolean;
begin
   IsLookup := false;
   ParentForm := GetParentForm(Self);
   if (ParentForm <> nil) and
      (ParentForm.ActiveControl <> nil) and
      (ParentForm.ActiveControl is TxDBLookUpComboBox) then
      IsLookup := (ParentForm.ActiveControl as TxDBLookUpComboBox).ListVisible;
  GridKey := bgkNull;
  if not (ParentForm.ActiveControl is TCustomGrid) then
  case Key of
    VK_F1: if Shift = [ssCtrl] then
                GridKey := bgkEditButton;
    VK_LEFT: if (ssCtrl in Shift) then
                GridKey := bgkFirstTab;
    VK_RIGHT: if (ssCtrl in Shift) then
                GridKey := bgkLastTab;
    VK_UP: if (Shift = []) and not IsLookup then GridKey := bgkUp;
    VK_DOWN: if (Shift = [])and not IsLookup then GridKey := bgkDown;
    VK_PRIOR: if not IsLookup then
                if ssCtrl in Shift then
                   GridKey := bgkHome
                  else
                   GridKey := bgkPageUp;
    VK_NEXT: if not IsLookup then
               if ssCtrl in Shift then
                  GridKey := bgkEnd
                 else
                  GridKey := bgkPageDown;
    VK_HOME: if ssCtrl in Shift then
               GridKey := bgkHome;
    VK_END: if ssCtrl in Shift then
               GridKey := bgkEnd;
    VK_RETURN:
        if (Shift = []) and not IsLookup then
           GridKey := bgkNextTab;
    VK_TAB:
      if not (ssCtrl in Shift) then
        if ssShift in Shift then
          GridKey := bgkPriorTab else
          GridKey := bgkNextTab;
    VK_F2: GridKey := bgkEditMode;
    VK_INSERT:
      if GetKeyState(VK_CONTROL) >= 0 then
        GridKey := bgkInsert else
        GridKey := bgkAppend;
    VK_DELETE: if GetKeyState(VK_CONTROL) < 0 then GridKey := bgkDelete;
    VK_F8: GridKey := bgkDelete;
  end;
  Result := GridKey;
end;

procedure TRaDBBox.SetEditMode(Value: Boolean);
var
  Control: TWinControl;
begin
  if GetEditMode <> Value then
    if Value then
    begin
      Control := FindNextControl(nil, True, True, False);
      if Control <> nil then Control.SetFocus;
    end else
      SetFocus;
end;

procedure TRaDBBox.SetOriginX(Value: Integer);
var NewOrigin: TPoint;
begin
   NewOrigin.X := Value;
   NewOrigin.Y := FOrigin.Y;
   SetOrigin(NewOrigin);
end;

procedure TRaDBBox.SetOriginY(Value: Integer);
var NewOrigin: TPoint;
begin
   NewOrigin.X := FOrigin.X;
   NewOrigin.Y := Value;
   SetOrigin(NewOrigin);
end;

procedure TRaDBBox.SetOrigin(Value: TPoint);
begin
   if (Value.x <> FOrigin.x) or (Value.y <> FOrigin.y) then
   begin
      FOrigin := Value;
      StartRefresh;
   end;
end;

procedure TRaDBBox.SetReadOnly(Value: Boolean);
begin
   if (Value <> FReadOnly) then
      if (CreateMode = cmAuto) then
   begin
      FReadOnly := Value;
      TabStop := FReadOnly;
      StartRefresh;
   end
     else
       Raise Exception.Create(SCantChangeReadOnly);
end;

procedure TRaDBBox.SetToActualHeight;
begin
   Height := ActualHeight;
end;

procedure TRaDBBox.SetToActualWidth;
begin
   Width := ActualWidth;
end;

procedure TRaDBBox.Scroll(Inc: Integer);
begin
  if FDataLink.Active and (Inc <> 0) then
    with FDataLink.DataSet do
      if State = dsInsert then
      begin
        UpdateRecord;
        if Modified then
        begin
          DisableControls;
          try
             Post;
             if (Inc > 0) and FAllowInsert and FAllowMove then
                  Append;
          finally
             EnableControls;
          end;
        end
          else
          if (Inc < 0) or not EOF then Cancel;
      end else
      begin
        CheckBrowseMode;
        DisableControls;
        try
          if FAllowMove then
             MoveBy(Inc);
          if (Inc = 1) and EOF and CanModify and FAllowInsert and FAllowMove then
             Append;
        finally
          EnableControls;
        end;
      end;
end;

procedure TRaDBBox.SetDataSource(Value: TDataSource);
begin
   if (Value <> FDataLink.DataSource) and
      ((FDataLink.DataSet = nil) or (FCreateMode = cmAuto)) then
   begin
      FDataLink.DataSource := Value;
      if Value <> nil then Value.FreeNotification(Self);
   end
    else
     if (FCreateMode <> cmAuto) and (Value <> FDataLink.DataSource) then
       Raise Exception.Create(SCantChangeDataSource);
end;

procedure TRaDBBox.DoKey(Key: TRaDBBoxKey);
var ParentForm: TCustomForm;
begin
  if FDataLink.Active then
  begin
    with FDataLink.DataSet do
      case Key of
        bgkEditMode: EditMode := not EditMode;
        bgkPriorTab: SelectNext(False);
        bgkNextTab: SelectNext(True);
        bgkFirstTab:
                begin
                   EditMode := False;
                   EditMode := True;
                end;
        bgkLastTab: SelectLast;
        bgkUp: SelectNext(False);
        bgkDown: SelectNext(True);
        bgkScrollUp: Scroll(-1);
        bgkScrollDown: Scroll(1);
        bgkPageUp: Scroll(-1);
        bgkPageDown: Scroll(1);
        bgkHome: if FAllowMove then First;
        bgkEnd:  if FAllowMove then Last;
        bgkInsert:
          if CanModify and FAllowInsert and FAllowMove then
          begin
            Insert;
            EditMode := True;
          end;
        bgkAppend:
          if CanModify and FAllowInsert and FAllowMove then
          begin
            Append;
            EditMode := True;
          end;
        bgkDelete:
          if CanModify and FAllowDelete and not ReadOnly and
             (not FConfirmDelete or
                    (MessageDlg(SDeleteRecordQuestion, mtConfirmation,
                                               mbOKCancel, 0) <> idCancel)) then
            Delete;
        bgkEditButton:
            begin
               ParentForm := GetParentForm(Self);
               if (ParentForm <> nil) and
                  (ParentForm.ActiveControl <> nil) and
                  (ParentForm.ActiveControl.Parent is TRaDBEdit) then
                  (ParentForm.ActiveControl.Parent as TRaDBEdit).ClickButton;
            end;
      end;
  end;
end;

procedure TRaDBBox.SelectLast;
var
  ParentForm: TCustomForm;
  ActiveControl, Control: TWinControl;
  WrapFlag: Integer;
begin
  ParentForm := GetParentForm(Self);
  if ParentForm <> nil then
  begin
    EditMode := False;
    EditMode := True;
    ActiveControl := ParentForm.ActiveControl;
    if ContainsControl(ActiveControl) then
    begin
      Control := FindNext(ActiveControl, False, WrapFlag);
      Control.SetFocus;
    end;
  end;
end;

procedure TRaDBBox.SelectNext(GoForward: Boolean);
var
  ParentForm: TCustomForm;
  ActiveControl, Control: TWinControl;
  WrapFlag: Integer;
begin
  ParentForm := GetParentForm(Self);
  if ParentForm <> nil then
  begin
    ActiveControl := ParentForm.ActiveControl;
    if ContainsControl(ActiveControl) then
    begin
      Control := FindNext(ActiveControl, GoForward, WrapFlag);
      if not (FDataLink.DataSet.State in dsEditModes) then
        SetFocus;
      try
        if WrapFlag <> 0 then Scroll(WrapFlag);
      except
        ActiveControl.SetFocus;
        raise;
      end;
      if not Control.CanFocus then
        Control := FindNext(Control, GoForward, WrapFlag);
      Control.SetFocus;
    end;
  end;
end;

procedure TRaDBBox.SetCreateMode(Value: TCreateMode);

   procedure SetOwnerToForm;
   var AField, NewField: TRaDBEdit;
       AName, ADataField: String;
       ADataSource: TDataSource;
       AVisible: boolean;
       ABounds: TRect;
       AEditKind: TEditKind;
       ParentForm: TCustomForm;
   begin
     ParentForm := GetParentForm(Self);
     if ParentForm = nil then
       exit;
     AVisible := Visible;
     try
        Visible := False;
        while ComponentCount > 0 do
        begin
           AField := Components[0] as TRaDBEdit;
           ABounds.Left := AField.Left;
           ABounds.Right := AField.Width;
           ABounds.Top := AField.Top;
           ABounds.Bottom := AField.Height;
           AName := CreateName(AField.Name);
           NewField := TRaDBEdit.Create(ParentForm);
           NewField.DistanceX := AField.DistanceX;
           NewField.DistanceY := AField.DistanceY;
           NewField.ShowBlob := AField.ShowBlob;
           NewField.Font := AField.Font;
           NewField.LabelFont := AField.LabelFont;
           ADataField := AField.DataField;
           ADataSource := AField.DataSource;
           NewField.TabStop := AField.TabStop;;
           AEditKind :=AField.EditKind;
           NewField.Parent := Self;
           AField.Free;
           NewField.Name := AName;
           NewField.DataField := ADataField;
           NewField.DataSource := ADataSource;
           NewField.EditKind := AEditKind;
           NewField.SetBounds(ABounds.Left,ABounds.Top,
                                  ABounds.Right,ABounds.Bottom);

        end;
     finally
        Visible := AVisible;
     end;
   end;

   procedure RemoveControls;
   begin
     while ControlCount > 0 do
        Controls[0].Free;
   end;

var AVisible: boolean;
begin
   if Value <> FCreateMode then
   begin
      FCreateMode := Value;
      case FCreateMode of
         cmAuto:
             begin
                AVisible := Visible;
                try
                  Visible := False;
                  RemoveControls;
                  StartRefresh;
                finally
                  Visible := AVisible;
                end;
             end;
         cmManual:
             SetOwnerToForm;
      end;
   end;
end;

procedure TRaDBBox.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var MustRefresh: Boolean;
begin
   MustRefresh := (FDataLink <> nil) and (AWidth <> Width) and
                  (CreateMode = cmAuto) and (FOrientation = orHorizontal);
   inherited SetBounds(ALeft, ATop, AWidth, AHeight);
   if MustRefresh then
      StartRefresh;
end;

procedure TRaDBBox.SetLabelFont(AFont: TFont);
begin
   if (AFont <> nil) then
   begin
      FLabelFont.Assign(AFont);
      StartRefresh;
   end;
end;

procedure TRaDBBox.SetShowBlobs(Value: Boolean);
begin
   if Value <> FShowBlobs then
   begin
      FShowBlobs := Value;
      StartRefresh;
   end;
end;

end.
