unit AHWord97;

interface

{
   These objects are based on the TWordObject demo in the directory
	  C:\Program Files\Borland\Delphi 3\Demos\Activex\Oleauto\Word8

   This is the Delphi 3 version

	 Feel free to add to this unit. My only condition is that you e-mail me your additions
	 and put in small comments indicating updates/changes (eg '<- added by JB 1/3/99')
	 and large comments where things are not obvious.

	 You are free to use this unit in any program, even commercial (although I'd like to know
  about where it is being used). You are not allowed to sell this, or a variation on it, as
  that would be dishonest. If you give away your source code in a program and use this unit,
  please keep this header so bugs/bug-fixes come to me.

	 Dr Allan Harkness
	 Scotland
  e-mail AHarkness@doctors.org.uk
      or allan@atheroma.freeserve.co.uk

  History: see Word97 Hx.txt
  Version 1.9
}

uses
	Windows, Classes, SysUtils, ActiveX, OleCtrls, ComObj, Word_TLB;

const
  wdTrue : Integer = -1;
  wdFalse : Integer = 0;

{$IFDEF VER120}
var
  EmptyParam : OLEVariant;
{$ENDIF}

type
  TWordDoc = class;

  TWordApp = class
  private
    FComGlobal : _Global;
    FComApp : _Application;
    FUsedExisting : Boolean;
    function GetCaption : String;
    procedure SetCaption(Value : String);
    function GetZoomPerCent : Integer;
    procedure SetZoomPerCent (Value : Integer);
    function GetVisible : Boolean;
    procedure SetVisible(Value : Boolean);
    function GetScreenUpdating : Boolean;
    procedure SetScreenUpdating (Value : Boolean);
    function GetWindowState : TOleEnum;
    procedure SetWindowState (Value : TOleEnum);
    function GetDocument(Index : Integer) : TWordDoc;
    function GetNoOfDocuments : Integer;
    procedure FreeDocuments;
  protected
    FDocuments : TList;
    procedure RemoveDoc(Index : Integer);
  public
    constructor Create(UseExisting : Boolean);
    constructor CreateFromOleObject(OleObject : OleVariant);
    destructor Destroy; override;
    destructor CloseApp(oeSaveChanges: TOleEnum);

    procedure DisableSystemCloseBox;
    function AddNewDoc(Template : String) : TWordDoc;
    function AddOpenDoc(DocName : String) : TWordDoc;
    function AddActiveDoc : TWordDoc;
    procedure CloseActiveDoc(oeSaveChanges: TOleEnum);
    procedure Move(oeUnit : TOleEnum; Count : integer);
    procedure MoveEnd(oeUnit : TOleEnum; Count : integer);
    procedure MoveStart(oeUnit : TOleEnum; Count : integer);
    procedure MoveRight(oeUnit : TOleEnum; Count : integer; Extend : TOleEnum );
    procedure MoveLeft(oeUnit : TOleEnum; Count : integer; Extend : TOleEnum );
    procedure MoveUp(oeUnit : TOleEnum ; Count : integer; Extend : TOleEnum );
    procedure MoveDown(oeUnit : TOleEnum ; Count : integer; Extend : TOleEnum );
    procedure GotoBookmark(Bookmark : String);
    procedure GoTo_(oeWhat, oeWhich : TOleEnum; oeCount: integer; oeName: String);
    procedure GoToNext(oeWhat : TOleEnum);
    procedure GoToPrevious(oeWhat : TOleEnum);
    procedure UpdateActiveDocFields;
    Procedure RunMacro(MacroName : string);
    procedure ScreenRefresh;
    procedure Cut;
    procedure Copy;
    procedure Paste;
    procedure Activate;
    procedure InsertText(Text : String);
    procedure PrintActiveDoc;
    procedure SaveActiveDocAs(Filename : String);
    procedure ZoomFullPage;
    procedure ZoomFullWidth;
    property Global : _Global read FComGlobal;
    property Application : _Application read FComApp;
    property UsedExisting : Boolean read FUsedExisting;
    property Caption : String read GetCaption write SetCaption;
    property Visible : Boolean read GetVisible write SetVisible;
    property ZoomPerCent : Integer read GetZoomPerCent write SetZoomPerCent;
    property ScreenUpdating : Boolean read GetScreenUpdating write SetScreenUpdating;
    property WindowState : TOleEnum read GetWindowState write SetWindowState;
    property Document [Index: Integer] : TWordDoc read GetDocument;
    property NoOfDocuments : Integer read GetNoOfDocuments;
  end;

  TWordRange = class;

  TWordDoc = class
  private
    FComDoc : _Document;
    FWordApp : TWordApp;
    function GetActive : Boolean;
    procedure SetActive(Value : Boolean);
    function GetRange(Index : Integer) : TWordRange;
    function GetNoOfRanges : Integer;
    function GetNoOfBookMarks : Integer;
    function GetBookmarkByName (BookmarkName: String) : Word_TLB.Bookmark;
    function GetBookmarkByIndex (Index: Integer) : Word_TLB.Bookmark;
    procedure SetBuiltInProperty(Index : TOleEnum; Const Value: Variant);
    function GetBuiltInProperty(Index : TOleEnum) : Variant;
    procedure SetCustomProperty(Index : String; Const Value : Variant);
    function GetCustomProperty(Index : String) : Variant;
    procedure FreeRangesAndRemoveDoc;
    function GetAutoTextEntries : OleVariant;
  protected
    FItemIndex : Integer;
    FRanges : TList;
    procedure RemoveRange(Index : Integer);
  public
    constructor CreateNewDoc(WordApp : TWordApp; Template : String);
    constructor CreateOpenDoc(WordApp : TWordApp; FileName : String);
    constructor CreateFromActiveDoc(WordApp : TWordApp);
    destructor Destroy; override;
    destructor CloseDoc(oeSaveChanges: TOleEnum);

    procedure Print;
    procedure PrintPreview;
    function AddRangeFromBookMark(BookmarkName : String) : TWordRange;
    function AddRangeFromSelection : TWordRange;
    function AddRangeFromDoc(iStart : integer; iEnd : integer) : TWordRange;
    function AddRangeFromRange(ComRange : Range) : TWordRange;
    function GoTo_(oeWhat, oeWhich : TOleEnum; oeCount: integer; oeName: String) : Range;
    function GoToNext(oeWhat : TOleEnum) : Range;
    function GoToPrevious(oeWhat : TOleEnum) : Range;
    function NoOfPages (IncludeFootnotesAndEndnotes : Boolean) : Integer;
    function NoOfWords (IncludeFootnotesAndEndnotes : Boolean) : Integer;
    procedure ReplaceBookmark (BookmarkName, ReplaceText : String; ReassignBookmark : Boolean);
    function BookmarkExists(BookmarkName : String): Boolean;
    procedure AddCustomProperty(Index: String; Value : Variant; oePropertyType : TOleEnum);
    procedure DeleteCustomProperty(Index: String);
    procedure DeleteRange(Index : Integer);
    function DocStillInWord : Boolean;
    procedure UpdateFields;
    procedure SaveAs(Filename : String);
    property WordApp : TWordApp read FWordApp;
    property Document : _Document read FComDoc write FComDoc;
    property Active : Boolean read GetActive write SetActive;
    property AutoTextEntries : OleVariant read GetAutoTextEntries;
    property Range [Index: Integer] : TWordRange read GetRange;
    property NoOfRanges : Integer read GetNoOfRanges;
    property BookmarkByIndex [Index: Integer] : Word_TLB.Bookmark read GetBookmarkByIndex;
    property Bookmark [BookmarkName: String] : Word_TLB.Bookmark read GetBookmarkByName;
    property NoOfBookmarks : Integer read GetNoOfBookmarks;
    property BuiltInProperty [Index : TOleEnum] : Variant read GetBuiltInProperty write SetBuiltInProperty;
    property CustomProperty [Index : String] : Variant read GetCustomProperty write SetCustomProperty;
    property ItemIndex : Integer read FItemIndex;
  end;

  TWordRange = class
  private
    FComRange : Word_TLB.Range;
    FWordDoc : TWordDoc;
  protected
    FItemIndex : Integer;
  public
    constructor CreateFromBookMark(WordDoc : TWordDoc; BookmarkName : String);
    constructor CreateFromSelection(WordDoc : TWordDoc);
    constructor CreateFromDoc(WordDoc : TWordDoc; iStart : integer; iEnd : integer);
    constructor CreateFromRange(WordDoc : TWordDoc; ComRange : Range);
    destructor Destroy; override;
    procedure Collapse(oeDirection : TOleEnum);
    function EndOf(oeUnit : TOleEnum; oeExtend : TOleEnum ) : Integer;
    function Expand(oeUnit : TOleEnum) : Integer;
    function GoTo_(oeWhat, oeWhich : TOleEnum; oeCount: integer; oeName: String) : Range;
    function GotoBookmark(BookmarkName : string) : range;
    function GoToNext(oeWhat : TOleEnum) : Range;
    function GoToPrevious(oeWhat : TOleEnum) : Range;
    function NoOfWords : Integer;
    procedure InsertAfter(Text : String);
    procedure InsertAutoText;
    procedure InsertGivenAutoText (AutoText : String; UseRichText : Boolean);
    procedure InsertBefore(Text : String);
    procedure InsertBreak(oeType : TOleEnum);
    procedure InsertParagraph;
    procedure InsertParagraphAfter;
    procedure InsertParagraphBefore;
    procedure InsertSymbol(CharacterNumber: Integer; Font: String;
                           Unicode: Boolean; oeBias : TOleEnum);
    function Move(oeUnit : TOleEnum; oeCount : integer) : Integer;
    function MoveWhile(Cset : String; Count : Integer) : Integer;
    function MoveUntil(Cset : String; Count : Integer) : Integer;
    function MoveStart(oeUnit : TOleEnum; oeCount : integer) : Integer;
    function MoveStartWhile(Cset : String; Count : Integer) : Integer;
    function MoveStartUntil(Cset : String; Count : Integer) : Integer;
    function MoveEnd(oeUnit : TOleEnum; oeCount : integer) : Integer;
    function MoveEndUntil(Cset : String; Count : Integer) : Integer;
    function MoveEndWhile(Cset : String; Count : Integer) : Integer;
    function Next(oeUnit : TOleEnum; oeCount : integer) : Range;
    function Previous(oeUnit : TOleEnum; oeCount : integer) : Range;
    function GetNextRange(oeUnit : TOleEnum; oeCount : integer) : Range;
    function GetPreviousRange(oeUnit : TOleEnum; oeCount : integer) : Range;
    procedure SetRange(iStart, iEnd : Integer);
    function StartOf(oeUnit : TOleEnum; oeExtend : TOleEnum ) : Integer;
    procedure CreateBookMark(Bookmark : String);
    procedure Select;
    procedure Cut;
    procedure Copy;
    procedure Paste;

    procedure SetStart(Value : Integer);
    function GetStart : Integer;
    procedure SetEnd(Value : Integer);
    function GetEnd : Integer;
    procedure SetText(Value : String);
    function GetText : String;
    procedure SetBold(Value : Boolean);
    function GetBold : Boolean;
    procedure SetItalic(Value : Boolean);
    function GetItalic : Boolean;
    procedure SetUnderline(Value : Boolean);
    function GetUnderline : Boolean;
    procedure SetCase(oeValue : TOleEnum);
    function GetCase :TOleEnum;
    procedure SetFont(fFont : _Font);
    function GetFont : _Font;
    procedure SetStyle(Style : String);
    function GetStyle : String;
    property Range : Range read FComRange write FComRange;
    property WordDoc : TWordDoc read FWordDoc;
    property Start : Integer read GetStart write SetStart;
    property End_ : Integer read GetEnd write SetEnd;
    property Text : String read GetText write SetText;
    property Bold : Boolean read GetBold write SetBold;
    property Italic : Boolean read GetItalic write SetItalic;
    property Underline : Boolean read GetUnderline write SetUnderline;
    property Case_ : TOleEnum read GetCase write SetCase;
    property Font : _Font read GetFont write SetFont;
    property Style : String read GetStyle write SetStyle;
    property ItemIndex : Integer read FItemIndex;
  end;

implementation

{ TWordApp implementation }

constructor TWordApp.Create(UseExisting : Boolean);
var
  TempIUnknown : IUnknown;
  Result: HResult;
begin
  inherited Create;
  FUsedExisting := False;
  if UseExisting then
  begin
    Result := GetActiveObject(CLASS_Application, nil, TempIUnknown);
    if Result = MK_E_UNAVAILABLE then // Word application does not exist
      FComApp := CoApplication_.Create
    else
    begin                       // Word application exists
      // make sure no other error occured while trying to get global class
      OleCheck(Result);
      // convert late bound IUnknown to early bound _Global
      OleCheck(TempIUnknown.QueryInterface(_Application, FComApp));
      FUsedExisting := True; // actually got an existing instance
    end;
  end
  else
    FComApp := CoApplication_.Create;
  FComGlobal := CoGlobal.Create;
	 FDocuments := TList.Create;
end;

constructor TWordApp.CreateFromOleObject(OleObject : OleVariant);
var
  tempDoc : TWordDoc;
begin
  inherited Create;
  FUsedExisting := False;
  try
    FComApp := IDISPATCH (OleObject.Application) as _Application;
    FComGlobal := CoGlobal.Create;
    FUsedExisting := True; // actually got an existing instance
  except
    raise Exception.Create ('Word not active in OLE container');
  end;
  FDocuments := TList.Create;
  tempDoc := TWordDoc.CreateFromActiveDoc (self); //throw away pointer (but kept in TList)
end;

procedure TWordApp.FreeDocuments;
var
	 i : Integer;
begin
	 if FDocuments.Count > 0 then
		 for i := FDocuments.Count - 1 downto 0 do // faster to free if we go backward
			 if Assigned (FDocuments [i]) then
				 TWordDoc (FDocuments [i]).Free;
	 FDocuments.Free;
end;

destructor TWordApp.Destroy;
begin
	 FreeDocuments;
	 inherited Destroy;
end;

destructor TWordApp.CloseApp;
//destructor TWordApp.CloseApp(oeSaveChanges: TOleEnum);
var
  ovSaveChanges,
	 OriginalFormat,
	 RouteDocument  : OleVariant;
begin
	 ovSaveChanges := oeSaveChanges;
	 OriginalFormat := EmptyParam;
	 RouteDocument := EmptyParam;
	 try
		 FComApp.Quit (ovSaveChanges, OriginalFormat, RouteDocument);
	 except
	 end;
	 FreeDocuments;
	 inherited Destroy;
end;

procedure TWordApp.DisableSystemCloseBox;
var
  hWordWIndow : HWND;
  hSysMenu    : HMENU;
begin
  // Find the Word 97 handle
  hWordWindow := FindWindow ('OpusApp', nil);
  if hWordWindow <> 0 then
  begin
    hSysMenu := GetSystemMenu (hWordWindow, false);
    EnableMenuItem (hSysMenu, SC_CLOSE, MF_BYCOMMAND or MF_GRAYED)
  end;
end;

procedure TWordApp.SetCaption(Value : String);
begin
  FComApp.Caption := Value;
end;

function TWordApp.GetCaption : String;
begin
	 Result := FComApp.Caption;
end;

procedure TWordApp.ZoomFullPage;
begin
	 FComApp.ActiveWindow.View.Zoom.PageFit := wdPageFitFullPage;
end;

procedure TWordApp.ZoomFullWidth;
begin
	 FComApp.ActiveWindow.View.Zoom.PageFit := wdPageFitBestFit;
end;

procedure TWordApp.SetZoomPerCent (Value : Integer);
begin
	 FComApp.ActiveWindow.View.Zoom.Percentage := Value;
end;

function TWordApp.GetZoomPerCent : Integer;
begin
	 Result := FComApp.ActiveWindow.View.Zoom.Percentage;
end;

procedure TWordApp.SetVisible(Value : Boolean);
begin
  FComApp.Visible := Value;
end;

function TWordApp.GetVisible : Boolean;
begin
  Result := FComApp.Visible;
end;

function TWordApp.GetDocument(Index : Integer) : TWordDoc;
begin
  Assert ((Index >= 0) and (Index < FDocuments.Count),
          'Index out of range for GetDocument (' + IntToStr (Index) + ')');
	 GetDocument := TWordDoc (FDocuments [Index]);
end;

function TWordApp.GetNoOfDocuments : Integer;
begin
  GetNoOfDocuments := FDocuments.Count;
end;

procedure TWordApp.RemoveDoc(Index : Integer);
// remove Document object from list (but do not free it)
// should rarely be used as onus then on developer to free this Document object
var
  i : Integer;
  wd : TWordDoc;
begin
  Assert ((Index >= 0) and (Index < FDocuments.Count),
          'Index out of range for RemoveDocument (' + IntToStr (Index) + ')');
  FDocuments.Delete (Index);
  i := Index;
	 while i < FDocuments.Count do
  begin
    wd := TWordDoc (FDocuments [i]);
    if Assigned (wd) then wd.FItemIndex := i;
    inc (i);
  end;
end;

procedure TWordApp.InsertText(Text : String);
begin
  FComApp.Selection.TypeText(Text);
end;

function TWordApp.AddNewDoc(Template : String) : TWordDoc;
var
	 wd : TWordDoc;
begin
   wd := TWordDoc.CreateNewDoc (Self, Template);
   AddNewDoc := wd;
end;

function TWordApp.AddOpenDoc (DocName : String) : TWordDoc;
var
	 wd : TWordDoc;
begin
	 wd := TWordDoc.CreateOpenDoc (Self, DocName);
	 AddOpenDoc := wd;
end;

function TWordApp.AddActiveDoc : TWordDoc;
// tries to see if active doc in list & just return it
// else try to get active doc from Word
// if no active doc then return nil and leave list alone
var
	 wd : TWordDoc;
	 i : Integer;
begin
	 wd := nil;
	 i := 0;
	 while (i < FDocuments.Count) and (wd = nil) do
	 begin
		 wd := TWordDoc (FDocuments [i]);
		 if not wd.Active then wd := nil;
		 inc (i)
	 end;
	 if wd = nil then
	 begin
		 wd := TWordDoc.CreateFromActiveDoc (Self);
		 if wd.FComDoc = nil then
		 begin
			 wd.Destroy;
			 wd := nil
		 end
	 end;
	 AddActiveDoc := wd;
end;

procedure TWordApp.CloseActiveDoc(oeSaveChanges: TOleEnum);
var
	 wd : TWordDoc;
begin
  ///Following will throw wobbler if no active doc! - debugged 27/6/99
	 ///AddActiveDoc.CloseDoc(oeSaveChanges);
  wd := AddActiveDoc;
  if wd <> nil then wd.CloseDoc (oeSaveChanges);
end;

procedure TWordApp.PrintActiveDoc;
begin
	 FComApp.PrintOut (EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
	  								 EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
										 EmptyParam, EmptyParam, EmptyParam)
end;

procedure TWordApp.SaveActiveDocAs(Filename : String);
var
  ovFilename : OleVariant;
begin
  ovFilename := Filename;
	 FComApp.ActiveDocument.SaveAs (ovFilename, EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
																  EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam);
end;

procedure TWordApp.Move(oeUnit : TOleEnum; Count : integer);
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;

	 FComApp.selection.Move(ovUnit, ovCount);
end;

procedure TWordApp.MoveEnd(oeUnit : TOleEnum; Count : integer);
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;

	 FComApp.selection.MoveEnd(ovUnit, ovCount);
end;

procedure TWordApp.MoveStart(oeUnit : TOleEnum; Count : integer);
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;

  FComApp.selection.MoveStart(ovUnit, ovCount);
end;

procedure TWordApp.MoveLeft(oeUnit : TOleEnum; Count : integer; Extend : TOleEnum );
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
	 ovExtend : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;
  ovExtend := Extend;

	 FComApp.selection.MoveLeft(ovUnit, ovCount, ovExtend);
end;

procedure TWordApp.MoveRight(oeUnit : TOleEnum; Count : integer; Extend : TOleEnum );
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
	 ovExtend : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;
  ovExtend := Extend;

	 FComApp.selection.MoveRight(ovUnit, ovCount, ovExtend);
end;

procedure TWordApp.MoveUp(oeUnit : TOleEnum ; Count : integer; Extend : TOleEnum );
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
	 ovExtend : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;
  ovExtend := Extend;

	 FComApp.selection.MoveUp(ovUnit, ovCount, ovExtend);
end;

procedure TWordApp.MoveDown(oeUnit : TOleEnum ; Count : integer; Extend : TOleEnum );
var
	 ovUnit : OleVariant;
	 ovCount : OleVariant;
	 ovExtend : OleVariant;
begin
	 ovUnit := oeUnit;
	 ovCount := Count;
  ovExtend := Extend;

	 FComApp.selection.MoveDown(ovUnit, ovCount, ovExtend);
end;

procedure TWordApp.GoTo_(oeWhat, oeWhich: TOleEnum; oeCount: integer; oeName: String);
var
  ovWhat, ovWhich, ovCount, ovName : OleVariant;
begin
  ovWhat := oeWhat;
  ovWhich := oeWhich;
  ovCount := ovCount;
  if ovName = '' then
		 ovName := EmptyParam
  else
    ovName := ovName;
  FComApp.Selection.GoTo_ (ovWhat, ovWhich, ovCount, ovName);
end;

procedure TWordApp.GoToNext(oeWhat: TOleEnum);
var
  ovWhat : OleVariant;
begin
  ovWhat := oeWhat;
  FComApp.Selection.GoToNext (ovWhat);
end;

procedure TWordApp.GoToPrevious(oeWhat: TOleEnum);
var
  ovWhat : OleVariant;
begin
  ovWhat := oeWhat;
  FComApp.Selection.GoToPrevious (ovWhat);
end;

procedure TWordApp.GotoBookmark(Bookmark : string);
var
	 What : OLEVariant;
  Which : OLEVariant;
  Count : OLEVariant;
  Name : OLEVariant;
begin
	 What := wdGoToBookmark;
  Which := EmptyParam;
	 Count := EmptyParam;
  Name := Bookmark;

  FComApp.Selection.GoTo_(What, Which, Count, Name);
end;

procedure TWordApp.Cut;
begin
  FComApp.Selection.Cut;
end;

procedure TWordApp.Copy;
begin
  FComApp.Selection.Copy;
end;

procedure TWordApp.Paste;
begin
  FComApp.Selection.Paste;
end;

procedure TWordApp.Activate;
begin
	FComApp.Activate;
end;

procedure TWordApp.UpdateActiveDocFields;
begin
	FComApp.ActiveDocument.Fields.Update;
end;

procedure TWordApp.RunMacro(MacroName : string);
begin
	FComApp.Run(MacroName);
end;

procedure TWordApp.ScreenRefresh;
begin
	 FComApp.ScreenRefresh;
end;

function TWordApp.GetScreenUpdating : Boolean;
begin
  GetScreenUpdating := FComApp.ScreenUpdating;
end;

procedure TWordApp.SetScreenUpdating (Value : Boolean);
begin
	 FComApp.ScreenUpdating := Value;
end;

function TWordApp.GetWindowState : TOleEnum;
begin
	 GetWindowState := FComApp.WindowState;
end;

procedure TWordApp.SetWindowState (Value : TOleEnum);
begin
	 FComApp.WindowState := Value;
end;

{ TWordDoc }

constructor TWordDoc.CreateNewDoc(WordApp : TWordApp; Template: String);
var
	 DocTemplate,
  NewTemplate : OleVariant;
begin
  Create;
  FItemIndex := -1;
  DocTemplate := Template;
  NewTemplate := False;
  FWordApp := WordApp;
  FComDoc := FWordApp.FComApp.Documents.Add (DocTemplate, NewTemplate);
  FItemIndex := FWordApp.NoOfDocuments;
  FWordApp.FDocuments.Add (self);
	 FRanges := TList.Create;
end;

constructor TWordDoc.CreateOpenDoc(WordApp : TWordApp; FileName: String);
var
	 ovFileName : OLEVariant;
begin
	 Create;
	 FItemIndex := -1;
	 ovFileName := FileName;
	 FWordApp := WordApp;
	 FComDoc := FWordApp.FComApp.Documents.Open (ovFileName, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
																						   EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam);
  FItemIndex := FWordApp.NoOfDocuments;
  FWordApp.FDocuments.Add (self);
  FRanges := TList.Create;
end;

constructor TWordDoc.CreateFromActiveDoc(WordApp : TWordApp);
begin
	 Create;
	 FItemIndex := -1;
	 FWordApp := WordApp;
	 if FWordApp.FComApp.Documents.Count = 0 then
		 FComDoc := nil // indicates that no valid document
	 else
	 begin
		 FComDoc := FWordApp.FComApp.ActiveDocument;
		 FItemIndex := FWordApp.NoOfDocuments;
		 FWordApp.FDocuments.Add (self);
	 end;
	 FRanges := TList.Create;
end;

function TWordDoc.DocStillInWord : Boolean;
// New function to check if document is still loaded in Word
// If user were to close document himself, further calls to FComDoc would
// generate an error. Therefor, you should check for the presence of the
// document before using the object, if user intervention could have occurred
// since the document was first created. This is most likely to be useful if
// you have a button on your program that closes Word or the document.
// NB if the user closes the document while your code is actively filling in
// the document, exceptions will still occur. Unless you check the document before
// every use of TWordDoc, you cannot get round this - you should warn users to leave
// well alone while your program does its magic.
var
  i : OleVariant;
  bPresent : boolean;
  docs : Word_TLB.Documents;
begin
  bPresent := False;
  i := 1;
  docs := FWordApp.FComApp.Documents;
  while (not bPresent) and (i <= docs.Count) do
  begin
    if docs.Item (i) = FComDoc then bPresent := True;
    inc (i);
  end;
  Result := bPresent;
end;

procedure TWordDoc.FreeRangesAndRemoveDoc;
var
	 i : Integer;
begin
	 if FRanges.Count > 0 then
		 for i := FRanges.Count - 1 downto 0 do // faster to free if we go backward
      if Assigned (FRanges [i]) then
        TWordRange (FRanges [i]).Free;
  FRanges.Free;
  if FItemIndex <> -1 then FWordApp.RemoveDoc (FItemIndex);
	 inherited;
end;

destructor TWordDoc.Destroy;
begin
	 FreeRangesAndRemoveDoc;
  inherited;
end;

destructor TWordDoc.CloseDoc;
var
  ovSaveChanges,
  OriginalFormat,
  RouteDocument  : OleVariant;
begin
  ovSaveChanges := oeSaveChanges;
  OriginalFormat := EmptyParam;
  RouteDocument := EmptyParam;
  FComDoc.Close (ovSaveChanges, OriginalFormat, RouteDocument);
  FreeRangesAndRemoveDoc;
  inherited Destroy;
end;

function TWordDoc.GetActive;
begin
  GetActive := (FWordApp.FComApp.ActiveDocument = FComDoc);
end;

procedure TWordDoc.SetActive(Value: Boolean);
begin
  FComDoc.Activate;
end;

procedure TWordDoc.Print;
begin
  FComDoc.PrintOut (EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
                    EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
                    EmptyParam, EmptyParam)
end;

procedure TWordDoc.PrintPreview;
begin
  FComDoc.PrintPreview
end;

procedure TWordDoc.SaveAs(Filename: String);
var
  DocName : OleVariant;
begin
  DocName := FileName;
	 FComDoc.SaveAs (DocName, EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam,
                  EmptyParam, EmptyParam, EmptyParam, EmptyParam, EmptyParam);
end;

procedure TWordDoc.UpdateFields;
begin
	 FComDoc.Fields.Update;
end;

function TWordDoc.AddRangeFromBookMark(BookmarkName : String) : TWordRange;
var
  wr : TWordRange;
begin
  wr := TWordRange.CreateFromBookMark (Self, BookMarkName);
  AddRangeFromBookMark := wr;
end;

function TWordDoc.AddRangeFromSelection : TWordRange;
var
  wr : TWordRange;
begin
  wr := TWordRange.CreateFromSelection (Self);
  AddRangeFromSelection := wr;
end;

function TWordDoc.AddRangeFromDoc(iStart : integer; iEnd : integer) : TWordRange;
var
	 wr : TWordRange;
begin
	 wr := TWordRange.CreateFromDoc (Self, iStart, iEnd);
	 AddRangeFromDoc := wr;
end;

function TWordDoc.AddRangeFromRange(ComRange : Range) : TWordRange;
var
	 wr : TWordRange;
begin
	 wr := TWordRange.CreateFromRange (Self, ComRange);
	 AddRangeFromRange := wr;
end;

function TWordDoc.GetRange(Index : Integer) : TWordRange;
begin
	 Assert ((Index >= 0) and (Index < FRanges.Count),
	 				 'Index out of range for GetRange (' + IntToStr (Index) + ')');
	 GetRange := TWordRange (FRanges [Index]);
end;

function TWordDoc.GetNoOfRanges : Integer;
begin
	 GetNoOfRanges := FRanges.Count;
end;

procedure TWordDoc.DeleteRange(Index : Integer);
// remove and free range object
begin
	 Assert ((Index >= 0) and (Index < FRanges.Count),
					 'Index out of range for DeleteRange (' + IntToStr (Index) + ')');
	 TWordRange (FRanges [Index]).Free;
end;

procedure TWordDoc.RemoveRange(Index : Integer);
// remove range object from list (but do not free it)
// should rarely be used as onus then on developer to free this range object
var
	 i : Integer;
	 wr : TWordRange;
begin
	 Assert ((Index >= 0) and (Index < FRanges.Count),
					 'Index out of range for RemoveRange (' + IntToStr (Index) + ')');
	 FRanges.Delete (Index);
	 i := Index;
	 while i < FRanges.Count do
	 begin
		 wr := TWordRange (FRanges [i]);
		 if Assigned (wr) then wr.FItemIndex := wr.FItemIndex - 1;
	   inc (i);
	 end;
end;

function TWordDoc.GoTo_(oeWhat, oeWhich: TOleEnum; oeCount: integer; oeName: String) : range;
var
	 ovWhat, ovWhich, ovCount, ovName : OleVariant;
begin
	 ovWhat := oeWhat;
	 ovWhich := oeWhich;
	 ovCount := ovCount;
	 if ovName = '' then
		 ovName := EmptyParam
	 else
		 ovName := ovName;
	 GoTo_ := FComDoc.Range (EmptyParam, EmptyParam).GoTo_ (ovWhat, ovWhich, ovCount, ovName);
end;

function TWordDoc.GoToNext(oeWhat: TOleEnum) : range;
var
	 ovWhat : OleVariant;
begin
	 ovWhat := oeWhat;
	 GotoNext := FComDoc.Range (EmptyParam, EmptyParam).GoToNext (ovWhat);
end;

function TWordDoc.GoToPrevious(oeWhat: TOleEnum) : range;
var
	 ovWhat : OleVariant;
begin
	 ovWhat := oeWhat;
	 GotoPrevious := FComDoc.Range (EmptyParam, EmptyParam).GoToPrevious (ovWhat);
end;

function TWordDoc.NoOfPages (IncludeFootnotesAndEndnotes : Boolean) : Integer;
var
  ovIncudeHF : OleVariant;
begin
  ovIncudeHF := IncludeFootnotesAndEndnotes;
  NoOfPages := FComDoc.ComputeStatistics (wdStatisticPages, ovIncudeHF)
end;

function TWordDoc.NoOfWords (IncludeFootnotesAndEndnotes : Boolean) : Integer;
var
  ovIncudeHF : OleVariant;
begin
  ovIncudeHF := IncludeFootnotesAndEndnotes;
  NoOfWords := FComDoc.ComputeStatistics (wdStatisticWords, ovIncudeHF)
end;

function TWordDoc.GetNoOfBookMarks : Integer;
begin
  GetNoOfBookMarks := FComDoc.Bookmarks.Count;
end;

function TWordDoc.GetBookmarkByIndex (Index: Integer) : Word_TLB.Bookmark;
var
  ovBookMarkName : OleVariant;
begin
  Result := nil;
  try
    ovBookMarkName := Index;
    Result := FComDoc.Bookmarks.Item(ovBookMarkName);
  except
    raise Exception.CreateFmt ('Bookmark No %d not found in document "%s"',
                               [Index, FComDoc.Name]);
  end;
end;

function TWordDoc.GetBookmarkByName (BookmarkName: String) : Word_TLB.Bookmark;
var
  ovBookMarkName : OleVariant;
begin
  Result := nil;
  try
    ovBookMarkName := BookmarkName;
    Result := FComDoc.Bookmarks.Item(ovBookMarkName);
  except
    raise Exception.CreateFmt ('Bookmark "%s" not found in document "%s"',
                               [BookmarkName, FComDoc.Name]);
  end;
end;

procedure TWordDoc.ReplaceBookmark (BookmarkName, ReplaceText : String; ReassignBookmark : Boolean);
var
	 ovBookMarkName : OleVariant;
	 TempRange : Word_TLB.Range;
	 ovRange : OleVariant;
begin
	 ovBookMarkName := BookMarkName;
	 try
		 TempRange := FComDoc.Bookmarks.Item (ovBookMarkName).Range;
		 TempRange.Text := ReplaceText;
		 ovRange := TempRange;
		 if ReassignBookMark then FComDoc.Bookmarks.Add (BookmarkName, ovRange);
	 except
		 raise Exception.CreateFmt ('Bookmark "%s" not found in document "%s"',
															  [BookMarkName, FComDoc.Name]);
	 end;
end;

//added by BDP 07/22/1999
function TWordDoc.BookmarkExists(BookmarkName : String): Boolean;
begin
	if FComDoc.Bookmarks.Exists(BookmarkName) then
		Result := True
	else
		Result := False;
end;

function TWordDoc.GetAutoTextEntries : OleVariant;
begin
	 GetAutoTextEntries := FComDoc.Get_AttachedTemplate.AutoTextEntries;
end;

procedure TWordDoc.SetBuiltInProperty(Index : TOleEnum; Const Value: Variant);
//added by BDP 05/08/1999 amended by AH 26/11/99
var
  DocumentProperties : OLEVariant;
  DocumentProperty   : OLEVariant;
  ovProp : OLEVariant;
begin
  try
    ovProp := Value;
    DocumentProperties := FComDoc.BuiltInDocumentProperties;
    DocumentProperty := DocumentProperties.Item [Index];
    DocumentProperty.Value := ovProp;
  except
    raise Exception.CreateFmt ('Error setting document property: "%d"', [Index]);
  end;
end;

function TWordDoc.GetBuiltInProperty(Index : TOleEnum) : Variant;
//added by BDP 05/08/1999 amended by AH 26/11/99
var
  DocumentProperties : OLEVariant;
  DocumentProperty   : OLEVariant;
begin
  try
    DocumentProperties := FComDoc.BuiltInDocumentProperties;
    DocumentProperty := DocumentProperties.Item [Index];
    Result := DocumentProperty.Value;
  except
    raise Exception.CreateFmt ('Error getting document property: "%d"', [Index]);
  end;
end;

procedure TWordDoc.SetCustomProperty(Index : String; Const Value : Variant);
//added by BDP 05/08/1999
var
  CustomProperties : OLEVariant;
  CustomProperty   : OLEVariant;
  ovProp : OLEVariant;
begin
  try
    ovProp := Value;
    CustomProperties := FComDoc.CustomDocumentProperties;
    CustomProperty := CustomProperties.Item [Index];
    CustomProperty.Value := ovProp;
  except
    raise Exception.CreateFmt ('Error setting custom property: "%s"', [Index]);
  end;
end;

function TWordDoc.GetCustomProperty(Index : String) : Variant;
//added by BDP 05/08/1999
var
  CustomProperties : OLEVariant;
  CustomProperty   : OLEVariant;
begin
  try
    CustomProperties := FComDoc.CustomDocumentProperties;
    CustomProperty := CustomProperties.Item [Index];
    Result := CustomProperty.Value;
  except
    raise Exception.CreateFmt ('Error getting custom property: "%s"', [Index]);
  end;
end;

procedure TWordDoc.AddCustomProperty(Index: String; Value : Variant; oePropertyType : TOleEnum);
var
  CustomProperties : OLEVariant;
begin
  try
    CustomProperties := FComDoc.CustomDocumentProperties;
    CustomProperties.Add(Index, wdFalse, oePropertyType, Value);
  except
    raise Exception.CreateFmt ('Error creating custom property: "%s"', [Index]);
  end;
end;

procedure TWordDoc.DeleteCustomProperty(Index: String);
var
  CustomProperties : OLEVariant;
  CustomProperty   : OLEVariant;
begin
  try
    CustomProperties := FComDoc.CustomDocumentProperties;
    CustomProperty := CustomProperties.Item [Index];
    if not VarIsNull (CustomProperty) then CustomProperty.Delete;
  except
    raise Exception.CreateFmt ('Error deleting custom property: "%s"', [Index]);
  end;
end;

{ TWordRange }

procedure TWordRange.Collapse(oeDirection: TOleEnum);
var
	 ovDirection : OleVariant;
begin
	 ovDirection := oeDirection;
	 FComRange.Collapse (ovDirection);
end;

constructor TWordRange.CreateFromBookMark(WordDoc: TWordDoc; BookmarkName: String);
var
	 ovBookMarkName : OleVariant;
begin
	 Create;
	 FItemIndex := -1;
	 ovBookMarkName := BookMarkName;
	 FWordDoc := WordDoc;
	 try
		 FComRange := FWordDoc.FComDoc.Bookmarks.Item (ovBookMarkName).Range;
		 FItemIndex := FWordDoc.NoOfRanges;
		 FWordDoc.FRanges.Add (self);
	 except
		 FComRange := nil;
		 raise Exception.CreateFmt ('Bookmark "%s" not found in document "%s"',
                               [BookMarkName, FWordDoc.FComDoc.Name]);
    // NB raising an exception here will cause TWordRange.Destroy to be called
    // The exception will be from trying to get an invalid bookmark
    // ie before ItemIndex is given a valid number, hence preset to -1
    // TWordRange.Destroy will only try to remove the range from the TWordDoc list
    // if ItemIndex <> -1
  end
end;

constructor TWordRange.CreateFromSelection(WordDoc : TWordDoc);
begin
  Create;
  FItemIndex := -1;
  FWordDoc := WordDoc;
  FComRange := FWordDoc.FComDoc.Application_.Selection.Range;
  FItemIndex := FWordDoc.NoOfRanges;
  FWordDoc.FRanges.Add (self);
end;

constructor TWordRange.CreateFromDoc(WordDoc : TWordDoc; iStart : integer; iEnd : integer);
var
  ovStart, ovEnd : OleVariant;
begin
  Create;
  FItemIndex := -1;
  ovStart := iStart;
  ovEnd := iEnd;
  FWordDoc := WordDoc;
  FComRange := FWordDoc.FComDoc.Range (ovStart, ovEnd);
  FItemIndex := FWordDoc.NoOfRanges;
  FWordDoc.FRanges.Add (self);
end;

constructor TWordRange.CreateFromRange(WordDoc : TWordDoc; ComRange : Range);
begin
  Create;
  FItemIndex := -1;
  FWordDoc := WordDoc;
  FComRange := ComRange;
  FItemIndex := FWordDoc.NoOfRanges;
  FWordDoc.FRanges.Add (self);
end;

destructor TWordRange.Destroy;
begin
  if FItemIndex <> -1 then FWordDoc.RemoveRange (FItemIndex);
  inherited;
end;

function TWordRange.EndOf(oeUnit, oeExtend: TOleEnum) : Integer;
var
  ovUnit,
  ovExtend : OleVariant;
begin
  ovUnit := oeUnit;
  ovExtend := oeExtend;
  EndOf := FComRange.EndOf (ovUnit, ovExtend);
end;

function TWordRange.Expand(oeUnit: TOleEnum) : Integer;
var
  ovUnit : OleVariant;
begin
  ovUnit := oeUnit;
  Expand := FComRange.Expand (ovUnit);
end;

function TWordRange.GetBold: Boolean;
begin
  GetBold := FComRange.Bold = wdTrue;
end;

function TWordRange.GetCase: TOleEnum;
begin
  GetCase := FComRange.Case_
end;

function TWordRange.GetEnd: Integer;
begin
  GetEnd := FComRange.End_;
end;

function TWordRange.GetFont: _Font;
begin
  GetFont := FComRange.Font;
end;

function TWordRange.GetItalic: Boolean;
begin
  GetItalic := FComRange.Italic = wdTrue;
end;

function TWordRange.GetStart: Integer;
begin
  GetStart := FComRange.Start;
end;

function TWordRange.GetStyle: String;
begin
  GetStyle := FComRange.Get_Style;
end;

function TWordRange.GetText: String;
begin
  GetText := FComRange.Text;
end;

function TWordRange.GetUnderline: Boolean;
begin
  GetUnderline := FComRange.Underline = wdTrue;
end;

function TWordRange.GoTo_(oeWhat, oeWhich: TOleEnum; oeCount: integer; oeName: String) : range;
var
  ovWhat, ovWhich, ovCount, ovName : OleVariant;
begin
  ovWhat := oeWhat;
  ovWhich := oeWhich;
  ovCount := ovCount;
  if ovName = '' then
    ovName := EmptyParam
  else
    ovName := ovName;
  FComRange := FComRange.GoTo_ (ovWhat, ovWhich, ovCount, ovName);
  Goto_ := FComRange;
end;

function TWordRange.GotoBookmark(BookmarkName : string) : range;
var
	 What : OLEVariant;
  Which : OLEVariant;
  Count : OLEVariant;
  Name : OLEVariant;
begin
	 What := wdGoToBookmark;
  Which := EmptyParam;
	 Count := EmptyParam;
  Name := BookmarkName;
  try
    FComRange := FComRange.GoTo_(What, Which, Count, Name);
  except
    FComRange := nil;
    raise Exception.CreateFmt ('Bookmark "%s" not found in document "%s"',
                               [BookMarkName, FWordDoc.FComDoc.Name]);
  end;
  GotoBookmark := FComRange;
end;

function TWordRange.GoToNext(oeWhat: TOleEnum) : range;
// moves range to next
var
  ovWhat : OleVariant;
begin
  ovWhat := oeWhat;
  FComRange := FComRange.GoToNext (ovWhat);
  GotoNext := FComRange;
end;

function TWordRange.GoToPrevious(oeWhat: TOleEnum) : range;
// moves range to previous
var
  ovWhat : OleVariant;
begin
  ovWhat := oeWhat;
  FComRange := FComRange.GoToPrevious (ovWhat);
  GotoPrevious := FComRange;
end;

function TWordRange.NoOfWords : Integer;
begin
  NoOfWords := FComRange.ComputeStatistics (wdStatisticWords)
end;

procedure TWordRange.InsertAfter(Text: String);
begin
  FComRange.InsertAfter (Text);
end;

procedure TWordRange.InsertAutoText;
begin
  FComRange.InsertAutoText
end;

procedure TWordRange.InsertGivenAutoText (AutoText : String; UseRichText : Boolean);
var
  ATemplate : OleVariant;
  AnAutoText : OleVariant;
begin
  // NB Word_TLB import seems to fail with Get_AttachedTemplate and returns an OleVariant
  // rather than a Template type. Therefor this bit of code is late bound and may not work in
  // other languages - sorry
  try
    ATemplate := FWordDoc.FComDoc.Get_AttachedTemplate;
    AnAutoText := ATemplate.AutoTextEntries.Item (AutoText);
    AnAutoText.Insert (FComRange, UseRichText);
  except
    on E : EOleSysError do
      if E.ErrorCode = 5941 then
        raise Exception.CreateFmt ('Error Autotext "%s" not found in template "%s"',
                                  [AutoText, ATemplate.Name])
      else
        raise Exception.CreateFmt ('Error inserting Autotext "%s" from template "%s"',
                                  [AutoText, ATemplate.Name])
  end;
end;

procedure TWordRange.InsertBefore(Text: String);
begin
  FComRange.InsertBefore (Text);
end;

procedure TWordRange.InsertBreak(oeType: TOleEnum);
var
  ovType : OleVariant;
begin
  ovType := oeType;
  FComRange.InsertBreak (ovType);
end;

procedure TWordRange.InsertParagraph;
begin
  FComRange.InsertParagraph
end;

procedure TWordRange.InsertParagraphAfter;
begin
  FComRange.InsertParagraphAfter
end;

procedure TWordRange.InsertParagraphBefore;
begin
  FComRange.InsertParagraphBefore
end;

procedure TWordRange.InsertSymbol(CharacterNumber: Integer; Font: String;
																	Unicode: Boolean; oeBias : TOleEnum);
var
  ovFont, ovUnicode, ovBias : OleVariant;
begin
  ovFont := Font;
  ovUnicode := UniCode;
  ovBias := oeBias;
  FComRange.InsertSymbol (CharacterNumber, ovFont, ovUnicode, ovBias);
end;

function TWordRange.Move(oeUnit: TOleEnum; oeCount: integer) : Integer;
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  Move := FComRange.Move (ovUnit, ovCount);
end;

function TWordRange.MoveEnd(oeUnit: TOleEnum; oeCount: integer) : Integer;
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  MoveEnd := FComRange.MoveEnd (ovUnit, ovCount);
end;

function TWordRange.MoveEndUntil(Cset: String; Count: Integer) : Integer;
var
  ovCset, ovCount : OleVariant;
begin
  ovCset := Cset;
  ovCount := Count;
  MoveEndUntil := FComRange.MoveEndUntil (ovCset, ovCount);
end;

function TWordRange.MoveEndWhile(Cset: String; Count: Integer) : Integer;
var
  ovCset, ovCount : OleVariant;
begin
  ovCset := Cset;
  ovCount := Count;
  MoveEndWhile := FComRange.MoveEndWhile (ovCset, ovCount);
end;

function TWordRange.MoveStart(oeUnit: TOleEnum; oeCount: integer) : Integer;
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  MoveStart := FComRange.MoveStart (ovUnit, ovCount);
end;

function TWordRange.MoveStartUntil(Cset: String; Count: Integer) : Integer;
var
  ovCset, ovCount : OleVariant;
begin
  ovCset := Cset;
  ovCount := Count;
  MoveStartUntil := FComRange.MoveStartUntil (ovCset, ovCount);
end;

function TWordRange.MoveStartWhile(Cset: String; Count: Integer) : Integer;
var
  ovCset, ovCount : OleVariant;
begin
  ovCset := Cset;
  ovCount := Count;
  MoveStartWhile := FComRange.MoveStartWhile (ovCset, ovCount);
end;

function TWordRange.MoveUntil(Cset: String; Count: Integer) : Integer;
var
  ovCset, ovCount : OleVariant;
begin
  ovCset := Cset;
  ovCount := Count;
  MoveUntil := FComRange.MoveUntil (ovCset, ovCount);
end;

function TWordRange.MoveWhile(Cset: String; Count: Integer) : Integer;
var
  ovCset, ovCount : OleVariant;
begin
  ovCset := Cset;
  ovCount := Count;
  MoveWhile := FComRange.MoveWhile (ovCset, ovCount);
end;

function TWordRange.Next(oeUnit: TOleEnum; oeCount: integer) : Range;
// moves range to next
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  FComRange := FComRange.Next (ovUnit, ovCount);
  Next := FComRange;
end;

function TWordRange.Previous(oeUnit: TOleEnum; oeCount: integer) : Range;
// moves range to previous
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  FComRange := FComRange.Previous (ovUnit, ovCount);
  Previous := FComRange;
end;

function TWordRange.GetNextRange(oeUnit: TOleEnum; oeCount: integer) : Range;
// returns a range next to this one (does not alter this range)
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  GetNextRange := FComRange.Next (ovUnit, ovCount);
end;

function TWordRange.GetPreviousRange(oeUnit: TOleEnum; oeCount: integer) : Range;
// returns a range prior to this one (does not alter this range)
var
  ovUnit,
  ovCount : OleVariant;
begin
  ovUnit := oeUnit;
  ovCount := oeCount;
  GetPreviousRange := FComRange.Previous (ovUnit, ovCount);
end;

procedure TWordRange.SetBold(Value: Boolean);
begin
  if Value then FComRange.Bold := wdTrue else FComRange.Bold := wdFalse;
end;

procedure TWordRange.SetCase(oeValue: TOleEnum);
begin
  FComRange.Case_ := oeValue;
end;

procedure TWordRange.SetEnd(Value: Integer);
begin
  FComRange.End_ := Value;
end;

procedure TWordRange.SetFont(fFont: _Font);
begin
  FComRange.Font := fFont;
end;

procedure TWordRange.SetItalic(Value: Boolean);
begin
  if Value then FComRange.Italic := wdTrue else FComRange.Italic := wdFalse;
end;

procedure TWordRange.SetRange(iStart, iEnd: Integer);
begin
  FComRange.SetRange (iStart, iEnd);
end;

procedure TWordRange.SetStart(Value: Integer);
begin
  FComRange.Start := Value;
end;

procedure TWordRange.SetStyle(Style: String);
var
  ovStyle : OleVariant;
begin
  ovStyle := Style;
  FComRange.Set_Style (ovStyle);
end;

procedure TWordRange.SetText(Value: String);
begin
  FComRange.Text := Value;
end;

procedure TWordRange.SetUnderline(Value: Boolean);
begin
  if Value then FComRange.Underline := wdTrue else FComRange.Underline := wdFalse;
end;

function TWordRange.StartOf(oeUnit : TOleEnum; oeExtend : TOleEnum ): Integer;
var
  ovUnit,
  ovExtend : OleVariant;
begin
  ovUnit := oeUnit;
  ovExtend := oeExtend;
  StartOf := FComRange.StartOf (ovUnit, ovExtend);
end;

procedure TWordRange.CreateBookMark(Bookmark : String);
var
  ovRange : OleVariant;
begin
  ovRange := FComRange;
  FWordDoc.FComDoc.Bookmarks.Add (BookMark, ovRange);
end;

procedure TWordRange.Select;
begin
	 FComRange.Select;
end;

procedure TWordRange.Cut;
begin
	 FComRange.Cut;
end;

procedure TWordRange.Copy;
begin
	 FComRange.Copy;
end;

procedure TWordRange.Paste;
begin
	 FComRange.Paste;
end;

initialization
{$IFDEF VER120}
	 TVarData (EmptyParam).VType  := varError;
	 TVarData (EmptyParam).VError := DISP_E_PARAMNOTFOUND;
{$ENDIF}
end.
