//======================== TAdvSplitter 1.1 =================================//
//                                                                           //
//  97/11/14                                                                 //
//  by Charles Bedard                                                        //
//                                                                           //
//       see AdvSplitter.txt for info on using this component                //
//                                                                           //
//     *** You can distribute or modify this code at will, but               //
//         please notify me about any change you make. The idea here         //
//         is to see if i triggered some nice ideas on improving             //
//         this kind of component                                            //
//                                                                           //
//  Changes from v1.0                                                        //
//                                                                           //
//     - added the ProportionalResize property                               //
//     - fixed the drawing when both panels are hidden                       //
//     - fixed the CTL3D inconsistency                                       //
//     - improved the Pane updating                                          //
//     - SwapPanes now works properly (a swap made at design time is         //
//       remains at run-time)                                                //
//===========================================================================//

unit AdvSplitter;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Buttons,
  Forms, Dialogs, ExtCtrls, DsgnIntf;

const
  BorderStyles: array[TBorderStyle] of LongInt = (0, WS_BORDER);

type
  TAdvFrameStyle = (ftNone, ftRaised, ftLowered, ftCarved, ftBump);
  TSplitterOrientation = (soHorizontal, soVertical);
  TSplitterStyle = (ssCheckered, ssSolid);

  { TAdvPane are Panels with a single property: FrameStyle,
    which sets the inherited bevels (Inner and Outer).}
  TAdvPane = class(TCustomPanel)
  private
    FFrameStyle : TAdvFrameStyle;
    FMinSize    : Word;
  protected
    procedure ReadState(Reader: TReader); override;
    procedure SetFrameStyle(value: TAdvFrameStyle);
  public
    constructor Create(AOwner: TComponent); override;
  published
    property BevelWidth;
    property Color;
    property FrameStyle : TAdvFrameStyle read FFrameStyle write SetFrameStyle;
    property Height stored False;
    property Left stored False;
    property Top stored False;
    property Width stored False;
    property MinSize : Word read FMinSize write FMinSize;
    property OnResize;
  end;

  TAdvSplitter = class;

  { TAdvPaneSetting is a simple wrapper of the Panes properties
    so that it shows as sub-properties in the Object Inspector }
  TAdvPaneSetting = class(TPersistent)
  private
    FIndex  : integer;
    FSplitter: TAdvSplitter;

    function GetVisible     : boolean;
    function GetColor       : TColor;
    function GetFrameStyle  : TAdvFrameStyle;
    function GetFrameWidth  : word;
    function GetMinSize     : word;

    procedure SetColor(value: TColor);
    procedure SetFrameStyle(value: TAdvFrameStyle);
    procedure SetFrameWidth(value: word);
    procedure SetMinSize(value: word);
    procedure SetVisible(value: boolean);
  public
    Constructor Create(ASplitter: TAdvSplitter; aIndex: Integer);
    property Visible   : boolean read GetVisible write SetVisible stored False;
  published
    property Color     : TColor read GetColor write SetColor stored False;
    property FrameStyle: TAdvFrameStyle read GetFrameStyle write SetFrameStyle stored False;
    property FrameWidth: word read GetFrameWidth write SetFrameWidth stored False;
    property MinSize   : Word read GetMinSize write SetMinSize stored False;
  end;


  TAdvSplitter = class(TCustomControl)
  private
    FAllowResize   : Boolean;
    FAutoUpdate    : Boolean;
    FBorderStyle   : TBorderStyle;
    FCanResize     : Boolean;
    FCheckeredBrush: HBRUSH;
    FOrientation   : TSplitterOrientation;
    FPosition      : Integer;
    FPane          : array[0..1] of TAdvPane;
    FPos           : TPoint;
    FPosPercent    : real;
    FProportionalResize : boolean;
    FSetting1,
    FSetting2      : TAdvPaneSetting;
    FSplitterColor : TColor;
    FSplitterStyle : TSplitterStyle;
    FThickness     : Integer;
    FOnResize      : TNotifyEvent;
    FOnPaneResize  : TNotifyEvent;
    { internal procedures }
    function  GetPane(index: Integer) : TAdvPane;
    procedure SetAutoUpdate(value: Boolean);
    procedure SetBorderStyle(value: TBorderStyle);
    procedure SetOrientation(value: TSplitterOrientation);
    procedure SetPosition(value: integer);
    procedure SetSplitterColor(value: TColor);
    procedure SetThickness(value: integer);
    procedure DrawSplitRect(X,Y: integer);
    procedure InvalidateSplitRect;
    procedure InvertSplitterRect(R: TRect);
    procedure CheckPosBounds(value: integer);
    procedure UpdatePanes;
    procedure CMCtl3DChanged(var msg: TMessage); message CM_CTL3DCHANGED;
    procedure CMDesignHitTest(var msg: TCMDesignHitTest); message CM_DESIGNHITTEST;
    procedure WMEraseBkGnd(var msg: TWMEraseBkGnd); message WM_ERASEBKGND;
    procedure WMSetCursor(var msg: TWMSetCursor); message WM_SETCURSOR;
    procedure WMSize(var msg: TWMSize); message WM_SIZE;
    procedure DoPaneResize(sender: TObject);
    procedure Resize;
  protected
    { VCL and Windows Messages procedures }
    procedure CreateParams(var params: TCreateParams); override;
    procedure CreateWnd; override;
    function GetChildOwner : TComponent; override;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X,Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X,Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X,Y: Integer); override;
    procedure Paint; override;
    procedure ReadState(Reader: TReader); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Center;
    procedure SwapPanes;
    property Pane[index: Integer]: TAdvPane read GetPane;
  published
    { properties }
    property AllowResize     : Boolean read FAllowResize write FAllowResize;
    property AutoUpdate      : Boolean read FAutoUpdate write SetAutoUpdate;
    property BorderStyle     : TBorderStyle read FBorderStyle write SetBorderStyle;
    property Orientation     : TSplitterOrientation read FOrientation write SetOrientation;
    property Pane1Setting    : TAdvPaneSetting read FSetting1 write FSetting1;
    property Pane2Setting    : TAdvPaneSetting read FSetting2 write FSetting2;
    property ProportionalResize : boolean read FProportionalResize write FProportionalResize;
    property Position        : Integer read FPosition write SetPosition;
    property SplitterColor   : TColor read FSplitterColor write SetSplitterColor;
    property SplitterStyle   : TSplitterStyle read FSplitterStyle write FSplitterStyle;
    property Thickness       : Integer read FThickness write SetThickness;

    { events }
    property OnPaneResize  : TNotifyEvent read FOnPaneResize write FOnPaneResize;
    property OnResize      : TNotifyEvent read FOnResize write FOnResize;
    { inherited properties }
    property Align;
    property Color;
    property Ctl3D;
    property Enabled;
    property ParentCtl3D;
    property ParentShowHint;
    property ShowHint;
    property Visible;
  end;

procedure Register;

implementation

constructor TAdvPane.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  Color        := clBtnFace;
  MinSize      := 10;
  BorderStyle  := bsNone;
  FrameStyle   := ftLowered;
  FullRepaint  := True;
end;

procedure TAdvPane.ReadState(Reader : TReader);
var
  S        : TAdvSplitter;
begin
  inherited ReadState(Reader);
  if Reader.Parent is TAdvSplitter then begin
    S := TAdvSplitter(Reader.Parent);
    if Assigned(S.FPane[Tag-1]) then
      S.FPane[Tag-1].Free;
    S.FPane[Tag-1] := Self;
  end;
end;

{ This is the most interesting procedure, as it sets how a
  Pane will look. There are many possibilities, but these
  are the ones i thought looked nice. }
procedure TAdvPane.SetFrameStyle(value: TAdvFrameStyle);
begin
  if FFrameStyle <> value then begin
    FFrameStyle := value;
    case FFrameStyle of
      ftNone    : begin
                    BevelInner := bvNone;
                    BevelOuter := bvNone;
                  end;
      ftRaised  : begin
                    BevelInner := bvRaised;
                    BevelOuter := bvRaised;
                  end;
      ftLowered : begin
                    BevelInner := bvLowered;
                    BevelOuter := bvLowered;
                  end;
      ftCarved  : begin
                    BevelInner := bvRaised;
                    BevelOuter := bvLowered;
                  end;
      ftBump    : begin
                    BevelInner := bvLowered;
                    BevelOuter := bvRaised;
                  end;
    end;
  end;
end;


{================= TAdvPaneSetting ==============================}
Constructor TAdvPaneSetting.Create(ASplitter: TAdvSplitter; aIndex: Integer);
begin
  inherited Create;
  FIndex := aIndex;
  FSplitter := ASplitter;
end;

function TAdvPaneSetting.GetColor  : TColor;
begin
 result := FSplitter.FPane[FIndex].Color;
end;

procedure TAdvPaneSetting.SetColor(value: TColor);
begin
  FSplitter.FPane[FIndex].Color := value;
end;

function TAdvPaneSetting.GetFrameStyle: TAdvFrameStyle;
begin
  result := FSplitter.FPane[FIndex].FrameStyle;
end;

procedure TAdvPaneSetting.SetFrameStyle(value: TAdvFrameStyle);
begin
  FSplitter.FPane[FIndex].FrameStyle := value;
  FSplitter.CheckPosBounds(FSplitter.FPosition);
  FSplitter.UpdatePanes;
end;

function TAdvPaneSetting.GetFrameWidth  : word;
begin
  result := FSplitter.FPane[FIndex].BevelWidth;
end;

procedure TAdvPaneSetting.SetFrameWidth(value: word);
begin
  FSplitter.FPane[FIndex].BevelWidth := value;
  FSplitter.CheckPosBounds(FSplitter.FPosition);
  FSplitter.UpdatePanes;
end;

function TAdvPaneSetting.GetMinSize: word;
begin
 result := FSplitter.FPane[FIndex].MinSize;
end;

procedure TAdvPaneSetting.SetMinSize(value: Word);
begin
  FSplitter.FPane[FIndex].MinSize := value;
  FSplitter.CheckPosBounds(FSplitter.FPosition);
  FSplitter.UpdatePanes;
end;

function TAdvPaneSetting.GetVisible : boolean;
begin
  result := FSplitter.FPane[FIndex].Visible;
end;

procedure TAdvPaneSetting.SetVisible(value: boolean);
begin
  FSplitter.FPane[FIndex].Visible := value;
  FSplitter.Invalidate;
end;


{============== TAdvSplitter =================================}

procedure TAdvSplitter.Center;
begin
  case Orientation of
    soHorizontal : Position := (ClientHeight - Thickness) div 2;
    soVertical   : Position := (ClientWidth - Thickness) div 2;
  end;
end;

procedure TAdvSplitter.CMCtl3DChanged(var msg: TMessage);
begin
  if (csLoading in ComponentState) then Exit;
  if NewStyleControls and (FBorderStyle = bsSingle) then RecreateWnd;
  inherited;
  UpdatePanes;
end;

procedure TAdvSplitter.CMDesignHitTest(var msg: TCMDesignHitTest);
begin
  if FCanResize then
    Msg.Result := 1
  else
    inherited;
end;

constructor TAdvSplitter.Create(AOwner: TComponent);
var
   i            : integer;
   grayPattern  : array [0..8] of WORD;
   CheckeredBmp : HBITMAP;
begin
  inherited Create(AOwner);
  Ctl3D  := True;
  Height := 100;
  Width  := 100;
  Exclude(FComponentStyle, csInheritable);
  if Classes.GetClass(TAdvPane.ClassName) = nil then
    Classes.RegisterClass(TAdvPane);

  FPane[0] := TAdvPane.Create(Self);
  FPane[0].Tag := 1;
  FPane[0].ParentCtl3D := True;
  FPane[0].Color := clBtnFace;
  FPane[0].FrameStyle := ftLowered;
  FPane[0].Parent   := Self;
  FPane[0].OnResize := self.DoPaneResize;
  FSetting1 := TAdvPaneSetting.Create(self,0);

  FPane[1] := TAdvPane.Create(Self);
  FPane[1].Tag := 2;
  FPane[1].ParentCtl3D := True;
  FPane[1].Color := clBtnFace;
  FPane[1].FrameStyle := ftLowered;
  FPane[1].Parent := Self;
  FPane[1].OnResize := self.DoPaneResize;
  FSetting2 := TAdvPaneSetting.Create(self,1);

  FAllowResize   := True;
  FAutoUpdate    := False;
  FBorderStyle   := bsNone;
  FPosition      := 50;
  FOrientation   := soVertical;
  FSplitterColor := clBtnFace;
  FThickness     := 2;
  FProportionalResize := False;

  FPos.X := -1;
  FPos.Y := -1;

  for i:= 0 to 8 do
    grayPattern[i] := WORD($5555 shl (i AND 1));
  CheckeredBmp  := CreateBitmap(8,8,1,1,@grayPattern);
  if CheckeredBmp <> 0 then
    FCheckeredBrush := CreatePatternBrush(CheckeredBmp);
  DeleteObject(CheckeredBmp);
end;

procedure TAdvSplitter.CreateParams(var Params : TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    Style := Style or BorderStyles[FBorderStyle];
    if NewStyleControls and Ctl3D and (FBorderStyle = bsSingle) then
    begin
      Style := Style and not WS_BORDER;
      ExStyle := ExStyle or WS_EX_CLIENTEDGE;
    end;
    WindowClass.style := WindowClass.style and not (CS_HREDRAW or CS_VREDRAW);
  end;
end;

procedure TAdvSplitter.CreateWnd;
begin
  inherited CreateWnd;
  UpdatePanes;
end;

destructor TAdvSplitter.Destroy;
begin
  if FPane[0] <> NIL then begin
     FPane[0].Free;
     FPane[0] := nil;
  end;
  if FPane[1] <> NIL then begin
     FPane[1].Free;
     FPane[1] := nil;
  end;
  FSetting1.Free;
  FSetting1 := NIL;
  FSetting2.Free;
  FSetting2 := NIL;
  DeleteObject(FCheckeredBrush);

  inherited Destroy;
end;

function TAdvSplitter.GetChildOwner: TComponent;
begin
  result := Self;
end;

procedure TAdvSplitter.GetChildren(Proc: TGetChildProc; Root: TComponent);
var
  i : Integer;
begin
  for i := 0 to 1 do
    Proc(TControl(FPane[i]));
end;

function TAdvSplitter.GetPane(Index: Integer): TAdvPane;
begin
  if index in [1..2] then
    result := FPane[index-1]
  else result := NIL;
end;

procedure TAdvSplitter.MouseDown(Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
begin
  inherited MouseDown(Button,Shift,X,Y);
  if FCanResize then begin
    if (Button = mbLeft) then begin
      SetCapture(Handle);
      DrawSplitRect(X,Y);
    end;
  end;
end;

procedure TAdvSplitter.MouseMove(Shift: TShiftState; X,Y: Integer);
begin
  inherited MouseMove(Shift,X,Y);
  case Orientation of
    soHorizontal : if (Y < FPane[0].MinSize) or
                      (Y > ClientHeight - FPane[1].MinSize) or
                      (Y = Position) then Exit;
    soVertical   : if (X < FPane[0].MinSize) or
                      (X > ClientWidth - FPane[1].MinSize) or
                      (X = Position) then Exit;
  end;
  if (GetCapture = Handle) and FCanResize then begin
    DrawSplitRect(X,Y);
    if AutoUpdate then
      Refresh;
  end;
end;

procedure TAdvSplitter.MouseUp(Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
begin
  if FCanResize then begin
    ReleaseCapture;
    FCanResize := False;
    DrawSplitRect(-1,-1);
    Refresh;
  end;
  inherited MouseUp(Button,Shift,X,Y);
end;

procedure TAdvSplitter.Paint;
var
  P, T,
  CW, CH : Integer;
begin
  inherited Paint;
  UpdatePanes;
  P  := Position;
  CW := ClientWidth;
  CH := ClientHeight;
  T  := Thickness;

  if not Ctl3D then begin
    Canvas.Pen.Color := SplitterColor;
    Canvas.Brush.Color := SplitterColor;
    case Orientation of
      soHorizontal : Canvas.Rectangle(0, P, CW, P+T);
      soVertical   : Canvas.Rectangle(P, 0, P+T, CH);
    end;
  end;
  { draw Splitter area if both Panes are visible}
  if (FPane[0].Visible) and (FPane[1].Visible) then begin
    Canvas.Pen.Color := SplitterColor;
    Canvas.Brush.Color := SplitterColor;
    case Orientation of
      soHorizontal : Canvas.Rectangle(0, P, CW, P+T);
      soVertical   : Canvas.Rectangle(P, 0, P+T, CH);
    end;
  end { if both panes are hidden, then draw the client rect }
  else if not(FPane[0].Visible or FPane[1].Visible) then begin
    Canvas.Pen.Color := self.Color;
    Canvas.Brush.Color := self.Color;
    Canvas.Rectangle(0,0,CW,CH);
  end;
end;

procedure TAdvSplitter.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
end;

procedure TAdvSplitter.DrawSplitRect(X,Y: Integer);
var
   R: TRect;
begin
  { Don't draw if both panes are not visible }
  if not( (FPane[0].Visible) and (FPane[1].Visible) ) then
    Exit;

  if AutoUpdate and (X > -1) and (Y > -1) then begin
    InvalidateSplitRect;
    case Orientation of
      soHorizontal : CheckPosBounds(Y);
      soVertical   : CheckPosBounds(X);
    end;
    InvalidateSplitRect;
    Exit;
  end;

  { Erase if FPos <> (-1,-1) }
  if (FPos.X > -1) or (FPos.Y > -1) then begin
    case Orientation of
      soHorizontal : R := Rect(0,FPos.Y,ClientWidth,Thickness);
      soVertical   : R := Rect(FPos.X,0,Thickness,ClientHeight);
    end;
    InvertSplitterRect(R);
  end;

  {FPos = new position}
  FPos.X := X;
  FPos.Y := Y;
  if not FCanResize then
    Exit;
  case Orientation of
    soHorizontal : begin
                     CheckPosBounds(FPos.Y);
                     FPos.Y := Position;
                     R := Rect(0,Position,ClientWidth,Thickness);
                   end;
    soVertical   : begin
                     CheckPosBounds(FPos.X);
                     FPos.X := Position;
                     R := Rect(Position,0,Thickness,ClientHeight);
                   end;
  end;
  InvertSplitterRect(R);
end;

procedure TAdvSplitter.InvertSplitterRect(R: TRect);
var
  CurrBrush : HBRUSH;
begin
  if (SplitterStyle = ssSolid) or (SplitterColor = clBlack) then
    PatBlt(Canvas.Handle,R.Left,R.Top,R.Right,R.Bottom,DSTINVERT)
  else begin
    CurrBrush := SelectObject(Canvas.Handle, FCheckeredBrush);
    PatBlt(Canvas.Handle,R.Left,R.Top,R.Right,R.Bottom,PATINVERT);
  end;
end;


procedure TAdvSplitter.InvalidateSplitRect;
var
  R : TRect;
begin
  case Orientation of
    soHorizontal : R := Rect(0, Position-2, ClientWidth, Position+Thickness+2);
    soVertical   : R := Rect(Position-2, 0, Position+Thickness+2, ClientHeight);
  end;
  InvalidateRect(Handle, @R, True);
end;

procedure TAdvSplitter.SetAutoUpdate(value: boolean);
begin
  if (value <> FAutoUpdate) then
    FAutoUpdate := value;
end;

procedure TAdvSplitter.SetBorderStyle(value: TBorderStyle);
begin
  if (value <> FBorderStyle) then begin
    FBorderStyle := value;
    RecreateWnd;
  end;
end;

procedure TAdvSplitter.SetOrientation(value: TSplitterOrientation);
begin
  if (value <> FOrientation) then begin
    FOrientation := value;
    if (csLoading in ComponentState) then
      Exit;
    if not HandleAllocated then
      Exit;
    UpdatePanes;
  end;
end;

procedure TAdvSplitter.SetPosition(value: Integer);
begin
  if (CsLoading in ComponentState) then begin
    FPosition := value;
    case Orientation of
       soHorizontal : FPosPercent := FPosition / Height;
       soVertical   : FPosPercent := FPosition / Width;
    end;
    Exit;
  end;
  if not HandleAllocated then Exit;
  CheckPosBounds(value);
  Invalidate;
end;

procedure TAdvSplitter.SetSplitterColor(value: TColor);
begin
  if (value <> FSplitterColor) then begin
    FSplitterColor := value;
    Refresh;
  end;
end;

procedure TAdvSplitter.SetThickness(value: integer);
begin
  if (value <> FThickness) then begin
    FThickness := Value;
    if (csLoading in ComponentState) then
      Exit;
    if not HandleAllocated then
      Exit;
    UpdatePanes;
  end;
end;

procedure TAdvSplitter.CheckPosBounds(value: integer);
var
  MinPos : Integer;
  MaxPos : Integer;
  PF     : TForm;
begin
  if FPane[0].FrameStyle <> ftNone then
     MinPos := FPane[0].MinSize + (4*FPane[0].BevelWidth)
  else MinPos := FPane[0].MinSize;

  if Orientation = soHorizontal then
     MaxPos := ClientHeight - Thickness - FPane[1].MinSize - 1
  else {soVertical}
     MaxPos := ClientWidth - Thickness - FPane[1].MinSize - 1;

  if FPane[1].FrameStyle <> ftNone then
     Dec(MaxPos,(4*FPane[1].BevelWidth)-1);

  if Value < MinPos then
     Value := MinPos;
  if Value > MaxPos then
     Value := MaxPos;

  FPosition := value;

  {notify the designer of the change}
  if csDesigning in ComponentState then begin
    PF := TForm(GetParentForm(Self));
    if Assigned(PF) and (PF.Designer <> nil) then
      PF.Designer.Modified;
  end;
end;

procedure TAdvSplitter.UpdatePanes;
var
  P      : Integer;
  S      : Integer;
  CW, CH : Integer;
begin
  if (csLoading in ComponentState) then
    Exit;
  if not HandleAllocated then
    Exit;
  CheckPosBounds(FPosition);
  P  := FPosition;
  CW := ClientWidth;
  CH := ClientHeight;
  S  := Thickness;
  case Orientation of
    soHorizontal : begin
                     if FPane[0].Visible then begin
                        if FPane[1].Visible then begin
                           FPane[0].SetBounds(0, 0, CW, P);
                           FPane[1].SetBounds(0, P+S, CW, CH-P-S);
                        end
                        else FPane[0].SetBounds(0, 0, CW,CH);
                     end
                     else if FPane[1].Visible then
                        FPane[1].SetBounds(0, 0, CW, CH);
                   end;
    soVertical :   begin
                     if FPane[0].Visible then begin
                        if FPane[1].Visible then begin
                           FPane[0].SetBounds(0, 0, P, CH);
                           FPane[1].SetBounds(P+S, 0, CW-P-S, CH);
                        end
                        else FPane[0].SetBounds(0,0,CW,CH);
                     end
                     else if FPane[1].Visible then
                        FPane[1].SetBounds(0,0,CW,CH);

                   end;
  end;

  FPane[0].invalidate;
  FPane[1].invalidate;

  DoPaneResize(self);
  { record proportions }
  case Orientation of
    soHorizontal : FPosPercent := Position / CH;
    soVertical   : FPosPercent := Position / CW;
  end;
end;


procedure TAdvSplitter.WMSetCursor(var msg: TWMSetCursor);
var
  Cur : hCursor;
  P   : TPoint;
begin
  Cur := 0;

  { Only Change cursor when Both Panes are visible and
    when we are over the Splitter Rect }
  if (Msg.HitTest = HTCLIENT) and
     (FPane[0].Visible) and
     (FPane[1].Visible) then begin
       GetCursorPos(P);
       P := ScreenToClient(P);
       case Orientation of
         soHorizontal : if Abs(Position - P.Y) <= Thickness then
                           Cur := Screen.Cursors[crVSplit];
         soVertical   : if Abs(Position - P.X) <= Thickness then
                           Cur := Screen.Cursors[crHSplit];
       end;
  end;

  FCanResize := (FAllowResize or (csDesigning in ComponentState)) and
                (Cur <> 0);
  if FCanResize then
    SetCursor(Cur)
  else inherited;
end;

procedure TAdvSplitter.WMEraseBkGnd(var msg: TWMEraseBkGnd);
begin
  Msg.Result := 1 {Dont Erase BackGround};
end;

procedure TAdvSplitter.WMSize(var Msg: TWMSize);
begin
  if ProportionalResize then
     { change position according to percentage of new dimensions}
       case Orientation of
         soHorizontal : Position := Round(ClientHeight * FPosPercent);
         soVertical   : Position := Round(ClientWidth * FPosPercent);
       end;
  UpdatePanes;
  inherited;
  Resize;
end;

procedure TAdvSplitter.DoPaneResize(sender: TObject);
begin
  if Assigned(FOnPaneResize) then
    FOnPaneResize(self);
end;

procedure TAdvSplitter.Resize;
begin
  if Assigned(FOnResize) then
    FOnResize(self);
end;


procedure TAdvSplitter.SwapPanes;
var
   T  : TAdvPane;
   i  : integer;
   PF : TCustomForm;
begin
  T        := FPane[0];
  FPane[0] := FPane[1];
  FPane[1] := T;

  FPane[0].Tag := 1;
  FPane[1].Tag := 2;

  RecreateWnd;
  if csDesigning in ComponentState then begin
    PF := TForm(GetParentForm(Self));
    if Assigned(PF) and (PF.Designer <> nil) then
      PF.Designer.Modified;
  end;
end;

{ ================ Design Time Component menu ================= }

const
  AdvSplitterVerbs : array[0..1] of String =
    ('Center Panes', 'Swap Panes');

  AdvSplitterNumVerbs = 2;

type
  TAdvSplitterEditor = class(TDefaultEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

procedure TAdvSplitterEditor.ExecuteVerb(Index: Integer);
var
  Splitter : TAdvSplitter;
begin
  if not(Component is TAdvSplitter) then
    Exit;
  Splitter := TAdvSplitter(Component);
  if Index = 0 then
    Splitter.Center
  else if Index = 1 then
    Splitter.SwapPanes;
end;

function TAdvSplitterEditor.GetVerb(Index: Integer): string;
begin
  Result := AdvSplitterVerbs[Index];
end;

function TAdvSplitterEditor.GetVerbCount: Integer;
begin
  Result := AdvSplitterNumVerbs;
end;



procedure Register;
begin
  RegisterComponents('Samples',[TAdvSplitter]);
  RegisterComponentEditor(TAdvSplitter, TAdvSplitterEditor);
end;


end.
