unit Main;

// This unit contains the implementation for a demo program using different virtual
// treeviews which show various features.
// Note: define the symbol "GraphicEx" if you have my GraphicEx library available
// (see www.lischke-online.de/Graphics.html) which allows to load more image formats into the application.
// Otherwise disable the conditional symbol to compile this demo.

{.$define GraphicEx}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, VirtualTrees, ExtDlgs, ImgList, JPEG,
  {$ifdef GraphicEx}GraphicEx,{$endif}
  Buttons, ExtCtrls, ComCtrls;

type
  TMainForm = class(TForm)
    PageControl1: TPageControl;
    CloseButton: TButton;
    TabSheet1: TTabSheet;
    TabSheet2: TTabSheet;
    VST1: TVirtualStringTree;
    GroupBox1: TGroupBox;
    AddRootButton: TButton;
    NodeCountEdit: TEdit;
    AddChildButton: TButton;
    ClearButton: TButton;
    Label1: TLabel;
    Label3: TLabel;
    GroupBox2: TGroupBox;
    SBCheckBox: TCheckBox;
    LoadBackgroundButton: TButton;
    OPD: TOpenPictureDialog;
    VDT1: TVirtualDrawTree;
    Label2: TLabel;
    SelectFolderButton: TButton;
    ImageFolderLabel: TLabel;
    SystemImages: TImageList;
    TabSheet3: TTabSheet;
    VST2: TVirtualStringTree;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    TreeImages: TImageList;
    Label8: TLabel;
    BitBtn1: TBitBtn;
    FontDialog1: TFontDialog;
    TabSheet4: TTabSheet;
    VST3: TVirtualStringTree;
    Label9: TLabel;
    Label10: TLabel;
    Label11: TLabel;
    AnimationImageList: TImageList;
    AnimationTimer: TTimer;
    TabSheet5: TTabSheet;
    VST4: TVirtualStringTree;
    TimerEdit: TEdit;
    UpDown1: TUpDown;
    Label12: TLabel;
    TabSheet6: TTabSheet;
    VST5: TVirtualStringTree;
    Label13: TLabel;
    Label14: TLabel;
    Label15: TLabel;
    Label16: TLabel;
    Label17: TLabel;
    CheckMarkCombo: TComboBox;
    Label18: TLabel;
    GridLineCheckBox: TCheckBox;
    procedure CloseButtonClick(Sender: TObject);
    procedure AddButtonClick(Sender: TObject);
    procedure ClearButtonClick(Sender: TObject);
    procedure LoadBackgroundButtonClick(Sender: TObject);
    procedure SBCheckBoxClick(Sender: TObject);
    procedure VST1GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var Text: WideString);
    procedure SelectFolderButtonClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure PageControl1Change(Sender: TObject);
    procedure VDT1InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
      var InitialStates: TVirtualNodeInitStates);
    procedure VDT1FreeNode(Sender: TBaseVirtualTree; Node: PVirtualNode);
    procedure VDT1DrawNode(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; ClipRect, NodeRect: TRect;
      Column: Integer);
    procedure VDT1GetNodeWidth(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; var NodeWidth: Integer);
    procedure VDT1Expanding(Sender: TBaseVirtualTree; Node: PVirtualNode; var Allowed: Boolean);
    procedure VDT1GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
      var Index: Integer);
    procedure VST1Change(Sender: TBaseVirtualTree; Node: PVirtualNode);
    procedure VST2GetNodeDataSize(Sender: TBaseVirtualTree; var NodeDataSize: Integer);
    procedure VST2GetDrawInfo(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var DrawInfo: TDrawInfo);
    procedure VST2GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var Text: WideString);
    procedure VST2InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
      var InitialStates: TVirtualNodeInitStates);
    procedure BitBtn1Click(Sender: TObject);
    procedure VST2GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
      var Index: Integer);
    procedure VST2InitChildren(Sender: TBaseVirtualTree; Node: PVirtualNode; var ChildCount: Cardinal);
    procedure VST2NewText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; Text: WideString);
    procedure VST3InitChildren(Sender: TBaseVirtualTree; Node: PVirtualNode; var ChildCount: Cardinal);
    procedure VST3InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
      var InitialStates: TVirtualNodeInitStates);
    procedure VST3GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var Text: WideString);
    procedure VST3GetHint(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var Text: WideString);
    procedure VST3GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
      var Index: Integer);
    procedure VST3Editing(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; var Allowed: Boolean);
    procedure VST3Change(Sender: TBaseVirtualTree; Node: PVirtualNode);
    procedure VST3CreateEditor(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; out EditLink: IVTEditLink);
    procedure VST3GetDrawInfo(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var DrawInfo: TDrawInfo);
    procedure VDT1GetHintSize(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; var R: TRect);
    procedure VDT1DrawHint(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; R: TRect; Column: Integer);
    procedure AnimationTimerTimer(Sender: TObject);
    procedure VST4GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
      var Index: Integer);
    procedure VST4InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
      var InitialStates: TVirtualNodeInitStates);
    procedure VST4InitChildren(Sender: TBaseVirtualTree; Node: PVirtualNode; var ChildCount: Cardinal);
    procedure TimerEditChange(Sender: TObject);
    procedure VST5BeforeItemErase(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; ItemRect: TRect;
      var Color: TColor; var EraseAction: TItemEraseAction);
    procedure VST5AfterItemPaint(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; ItemRect: TRect);
    procedure VST5FocusChanging(Sender: TBaseVirtualTree; OldNode, NewNode: PVirtualNode; OldColumn, NewColumn: Integer;
      var Allowed: Boolean);
    procedure VST5InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
      var InitialStates: TVirtualNodeInitStates);
    procedure VST5GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var Text: WideString);
    procedure VST5GetDrawInfo(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
      var DrawInfo: TDrawInfo);
    procedure CheckMarkComboChange(Sender: TObject);
    procedure GridLineCheckBoxClick(Sender: TObject);
  private
    FThumbSize: Integer;
    FExtensionsInitialized: Boolean;
    FExtensionList: TStringList;
    function FileCompare(Sender: TBaseVirtualTree; Node1, Node2: PVirtualNode; Column: Integer): Integer;
    procedure RescaleImage(Source, Target: TBitmap);
    function CanDisplay(const Name: String): Boolean;
  public
  end;

var
  MainForm: TMainForm;

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.DFM}

uses
  FileCtrl, ShellAPI, Mask;

type
  // node data record for the thumbnail treeview (VDT1)
  PImageData = ^TImageData;
  TImageData = record
    FullPath: String;
    IsFolder: Boolean;
    OpenIndex,
    CloseIndex: Integer;  // image indices into the system image list
    Image: TBitmap;
    // some image properties, preformatted
    Properties: String;
  end;

  // node data record for the general features treeview (VST2)
  PNodeData2 = ^TNodeData2;
  TNodeData2 = record
    Caption,
    StaticText,
    ForeignText: WideString;
    ImageIndex,
    Level: Integer;
  end;

  // describes the type of value a property tree node stores in its data property
  TValueType = (
    vtNone,
    vtString,
    vtPickString,
    vtNumber,
    vtPickNumber,
    vtMemo,
    vtDate
  );

  // node data record for the the document properties treeview (VST3)
  PPropertyData = ^TPropertyData;
  TPropertyData = record
    ValueType: TValueType;
    Value: WideString; // this value can actually be a date or a number too
    Changed: Boolean;
  end;

  // our own edit link to implement several different node editors 
  TPropertyEditLink = class(TInterfacedObject, IVTEditLink)
  private
    FEdit: TWinControl;      // one of the property editor classes 
    FTree: TBaseVirtualTree; // a back reference to the tree calling
    FNode: PVirtualNode;     // the node to be edited
    FColumn: Integer;        // the column of the node
    FOldEditProc: TWndMethod; // used to capture some important messages regardless of the
                             // type of edit control we use
    FStopping: Boolean;      // True if the link is currently cancelling/ending an edit action
  protected
    procedure DoEndEdit; virtual; // need this virtual method to solve inheritance problems
                                  // when deriving the grid editor link
    procedure EditWindowProc(var Message: TMessage);
    procedure EditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  public
    destructor Destroy; override;

    function BeginEdit: Boolean; stdcall;
    procedure CancelEdit; stdcall;
    function CanStop: Boolean; stdcall;
    procedure EndEdit; stdcall;
    function GetBounds: TRect; stdcall;
    function PrepareEdit(Tree: TBaseVirtualTree; Node: PVirtualNode; Column: Integer): Boolean; stdcall;
    procedure SetBounds(R: TRect); stdcall;
  end;

type
  TPropertyTextKind = (
    ptkText,
    ptkHint
  );

// the following constants provide the property tree with default data

const
  // types of editors to use for a certain node in VST3
  ValueTypes: array[0..1, 0..12] of TValueType = (
    (
      vtString,     // Title
      vtString,     // Theme
      vtPickString, // Category
      vtMemo,       // Keywords
      vtNone,       // Template
      vtNone,       // Page count
      vtNone,       // Word count
      vtNone,       // Character count
      vtNone,       // Lines
      vtNone,       // Paragraphs
      vtNone,       // Scaled
      vtNone,       // Links to update
      vtMemo),      // Comments
    (
      vtString,     // Author
      vtNone,       // Most recently saved by
      vtNumber,     // Revision number
      vtPickString, // Primary application
      vtString,     // Company name
      vtNone,       // Creation date
      vtDate,       // Most recently saved at
      vtNone,       // Last print
      vtNone,
      vtNone,
      vtNone,
      vtNone,
      vtNone)
  );

  // types of editors to use for a certain node in VST3
  DefaultValue: array[0..1, 0..12] of String = (
    (
      'Virtual Treeview',         // Title
      'native Delphi controls',   // Theme
      'Virtual Controls',         // Category
      'virtual, treeview, VCL',   // Keywords
      'no template used',         // Template
      '> 500',                    // Page count
      '?',                        // Word count
      '> 600.000',                // Character count
      '~17.300',                  // Lines
      '',                         // Paragraphs
      'False',                    // Scaled
      'www.lischke-online.de',    // Links to update
      'The best treeview ever.'), // Comments
    (
      'Dipl. Ing. Mike Lischke',  // Author
      'Mike Lischke',             // Most recently saved by
      '1.25',                     // Revision number
      'Delphi',                   // Primary application
      '',                         // Company name
      'July 1999',                // Creation date
      '12.11.2000',               // Most recently saved at
      '',                         // Last print
      '',
      '',
      '',
      '',
      '')
  );

  // fixed strings for property tree (VST3)
  PropertyTexts: array[0..1, 0..12, TPropertyTextKind] of String = (
    (// first (upper) subtree
     ('Title', 'Title of the file or document'),
     ('Theme', 'Theme of the file or document'),
     ('Category', 'Category of theme'),
     ('Keywords', 'List of keywords which describe the content of the file'),
     ('Template', 'Name of the template which was used to create the document'),
     ('Page count', 'Number of pages in the document'),
     ('Word count', 'Number of words in the document'),
     ('Character count', 'Number of characters in the document'),
     ('Lines', 'Number of lines in the document'),
     ('Paragraphs', 'Number of paragraphs in the document'),
     ('Scaled', 'Scaling of the document for output'),
     ('Links to update', 'Links which must be updated'),
     ('Comments', 'Description or comments for the file')
     ),
    (// second (lower) subtree
     ('Author', 'name of the author of the file or document'),
     ('Most recently saved by', 'Name of the person who has saved the document last'),
     ('Revision number', 'Revision number of the file or document'),
     ('Primary application', 'Name of the application which is primarily used to create this kind of file'),
     ('Company name', 'Name of the company or institution'),
     ('Creation date', 'Date when the file or document was created'),
     ('Most recently saved at', 'Date when the file or document was saved the last time'),
     ('Last print', 'Date when the file or document was printed the last time'),
     ('', ''),   // the remaining 5 entries are not used
     ('', ''),
     ('', ''),
     ('', ''),
     ('', '')
   )
  );

type
  PGridData = ^TGridData;
  TGridData = record
    ValueType: array[0..3] of TValueType; // one for each column
    Value: array[0..3] of Variant;
    Changed: Boolean;
  end;

  // our own edit link to implement several different node editors (grid tree)
  TGridEditLink = class(TPropertyEditLink, IVTEditLink)
  protected
    procedure DoEndEdit; override;
  public
    procedure EndEdit; stdcall;
    function PrepareEdit(Tree: TBaseVirtualTree; Node: PVirtualNode; Column: Integer): Boolean; stdcall;
  end;

//----------------- utility functions ----------------------------------------------------------------------------------

function HasChildren(const FileName: String): Boolean;

// determines whether FileName is a folder and whether it contains other file objects

var
  SR: TSearchRec;

begin
  // if we can find it directly via FindFirst then it is a file
  Result := FindFirst(FileName, faAnyFile, SR) = 0;
  if Result then
    FindClose(SR)
  else
  begin
    // FileName is a folder, try to find a subfolder or file
    if FindFirst(FileName + '*.*', faAnyFile, SR) = 0 then
    begin
      repeat
        Result := (SR.Name <> '..') and (SR.Name <> '.');
      until Result or (FindNext(SR) <> 0);
      FindClose(SR);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function GetIconIndex(Name: String; Flags: Cardinal): Integer;

// Returns the index of the system icon for the given file object.

var
  SFI: TSHFileInfo;

begin
  if SHGetFileInfo(PChar(Name), 0, SFI, SizeOf(TSHFileInfo), Flags) = 0 then
    Result := -1
  else
    Result := SFI.iIcon;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure GetOpenAndClosedIcons(Name: String; var Open, Closed: Integer);

begin
  Closed := GetIconIndex(Name, SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
  Open := GetIconIndex(Name, SHGFI_SYSICONINDEX or SHGFI_SMALLICON or SHGFI_OPENICON);
end;

//----------------- TPropertyEditLink ----------------------------------------------------------------------------------

// This implementation is used in VST3 to make a connection beween the tree and the actual edit window
// which might be a simple edit, a combobox or a memo etc.

destructor TPropertyEditLink.Destroy;

begin
  FEdit.Free;
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPropertyEditLink.DoEndEdit;

// simple redirection which can be overridden by the grid edit link to call the correct EndEdit method

begin
  EndEdit;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPropertyEditLink.EditWindowProc(var Message: TMessage);

// here we can capture some messages for keeping track of focus changes

begin
  case Message.Msg of
    WM_KILLFOCUS:
      if FEdit is TDateTimePicker then
      begin
        // when the user clicks on a dropped down calender we also get the kill focus message
        if not TDateTimePicker(FEdit).DroppedDown then
          DoEndEdit;
      end
      else
        DoEndEdit;
  else
    FOldEditProc(Message);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPropertyEditLink.EditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);

var
  CanAdvance: Boolean;

begin
  case Key of
    VK_RETURN,
    VK_UP,
    VK_DOWN:
      begin
        // consider special cases before finishing edit mode
        CanAdvance := Shift = [];
        if FEdit is TComboBox then
          CanAdvance := CanAdvance and not TComboBox(FEdit).DroppedDown;
        if FEdit is TDateTimePicker then
          CanAdvance :=  CanAdvance and not TDateTimePicker(FEdit).DroppedDown;

        if CanAdvance then
        begin
          DoEndEdit;
          with FTree do
          begin
            if Key = VK_UP then
              FocusedNode := GetPreviousVisible(FocusedNode)
            else
              FocusedNode := GetNextVisible(FocusedNode);
            Selected[FocusedNode] := True;
          end;
          Key := 0;
        end;
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPropertyEditLink.BeginEdit: Boolean;

begin
  Result := not FStopping;
  if Result then
  begin
    FEdit.Show;
    FEdit.SetFocus;
    // set a window procedure hook to get notified about important messages
    FOldEditProc := FEdit.WindowProc;
    FEdit.WindowProc := EditWindowProc;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPropertyEditLink.CancelEdit;

begin
  if not FStopping then
  begin
    FStopping := True;
    // restore the edit's window proc
    FEdit.WindowProc := FOldEditProc;
    FEdit.Hide;
    FTree.CancelEditNode;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPropertyEditLink.CanStop: Boolean;

begin
  Result := not FStopping;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPropertyEditLink.EndEdit;

var
  Data: PPropertyData;
  Buffer: array[0..1024] of Char;
  S: WideString;

begin
  if not FStopping then
  begin
    FStopping := True;
    // restore the edit's window proc
    FEdit.WindowProc := FOldEditProc;
    Data := FTree.GetNodeData(FNode);
    if FEdit is TComboBox then
      S := TComboBox(FEdit).Text
    else
    begin
      GetWindowText(FEdit.Handle, Buffer, 1024);
      S := Buffer;
    end;

    if S <> Data.Value then
    begin
      Data.Value := S;
      Data.Changed := True;
      FTree.InvalidateNode(FNode);
    end;
    FEdit.Hide;
    FTree.EndEditNode;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPropertyEditLink.GetBounds: TRect;

begin
  Result := FEdit.BoundsRect;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPropertyEditLink.PrepareEdit(Tree: TBaseVirtualTree; Node: PVirtualNode; Column: Integer): Boolean;

var
  Data: PPropertyData;

begin
  Result := not FStopping;
  if Result then
  begin
    FTree := Tree as TCustomVirtualStringTree;
    FNode := Node;
    FColumn := Column;

    // determine what edit type actually is needed
    FEdit.Free;
    FEdit := nil;
    Data := FTree.GetNodeData(Node);
    case Data.ValueType of
      vtString:
        begin
          FEdit := TEdit.Create(nil);
          with FEdit as TEdit do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value;
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtPickString:
        begin
          FEdit := TComboBox.Create(nil);
          with FEdit as TComboBox do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value;
            Items.Add(Text);
            Items.Add('Standard');
            Items.Add('Additional');
            Items.Add('Win32');
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtNumber:
        begin
          FEdit := TMaskEdit.Create(nil);
          with FEdit as TMaskEdit do
          begin
            Visible := False;
            Parent := Tree;
            EditMask := '9999';
            Text := Data.Value;
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtPickNumber:
        begin
          FEdit := TComboBox.Create(nil);
          with FEdit as TComboBox do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value;
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtMemo:
        begin
          FEdit := TComboBox.Create(nil);
          // in reality this should be a drop down memo but this requires
          // a special control
          with FEdit as TComboBox do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value;
            Items.Add(Data.Value);
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtDate:
        begin
          FEdit := TDateTimePicker.Create(nil);
          with FEdit as TDateTimePicker do
          begin
            Visible := False;
            Parent := Tree;
            CalColors.MonthBackColor := clWindow;
            CalColors.TextColor := clBlack;
            CalColors.TitleBackColor := clBtnShadow;
            CalColors.TitleTextColor := clBlack;
            CalColors.TrailingTextColor := clBtnFace;
            Date := StrToDate(Data.Value);
            OnKeyDown := EditKeyDown;
          end;
        end;
    else
      Result := False;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TPropertyEditLink.SetBounds(R: TRect);

begin
  if FTree.Name = 'VST3' then
    Inc(R.Left, 6);
  FEdit.BoundsRect := R;
end;

//----------------- TGridEditLink --------------------------------------------------------------------------------------

// By reusing some of the property edit implementation we can greatly simplify the grid edit link stuff.

procedure TGridEditLink.DoEndEdit;

begin
  EndEdit;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGridEditLink.EndEdit;

var
  Data: PGridData;
  Buffer: array[0..1024] of Char;
  S: WideString;
  I: Integer;

begin
  if not FStopping then
  begin
    FStopping := True;
    // restore the edit's window proc
    FEdit.WindowProc := FOldEditProc;
    Data := FTree.GetNodeData(FNode);
    if FEdit is TComboBox then
    begin
      S := TComboBox(FEdit).Text;
      if S <> Data.Value[FColumn - 2] then
      begin
        Data.Value[FColumn - 2] := S;
        Data.Changed := True;
      end;
    end
    else
      if FEdit is TMaskEdit then
      begin
        I := StrToInt(Trim(TMaskEdit(FEdit).EditText));
        if I <> Data.Value[FColumn - 2] then
        begin
          Data.Value[FColumn - 2] := I;
          Data.Changed := True;
        end;
      end
      else
      begin
        GetWindowText(FEdit.Handle, Buffer, 1024);
        S := Buffer;
        if S <> Data.Value[FColumn - 2] then
        begin
          Data.Value[FColumn - 2] := S;
          Data.Changed := True;
        end;
      end;

    if Data.Changed then
      FTree.InvalidateNode(FNode);
    FEdit.Hide;
    FTree.EndEditNode;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGridEditLink.PrepareEdit(Tree: TBaseVirtualTree; Node: PVirtualNode; Column: Integer): Boolean;

var
  Data: PGridData;

begin
  Result := not FStopping;
  if Result then
  begin
    FTree := Tree as TCustomVirtualStringTree;
    FNode := Node;
    FColumn := Column;

    // determine what edit type actually is needed
    FEdit.Free;
    FEdit := nil;
    Data := FTree.GetNodeData(Node);
    case Data.ValueType[FColumn - 2] of
      vtString:
        begin
          FEdit := TEdit.Create(nil);
          with FEdit as TEdit do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value[FColumn - 2];
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtPickString:
        begin
          FEdit := TComboBox.Create(nil);
          with FEdit as TComboBox do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value[FColumn - 2];
            // here you would usually do a lookup somewhere to get values for the combobox
            case FColumn of
              3:
                begin
                  Items.Add('John');
                  Items.Add('Mike');
                  Items.Add('Barney');
                  Items.Add('Tim');
                end;
              4:
                begin
                  Items.Add('Doe');
                  Items.Add('Lischke');
                  Items.Add('Miller');
                  Items.Add('Smith');
                end;
            end;
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtNumber:
        begin
          FEdit := TMaskEdit.Create(nil);
          with FEdit as TMaskEdit do
          begin
            Visible := False;
            Parent := Tree;
            EditMask := '9999;0; ';
            Text := Data.Value[FColumn - 2];
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtPickNumber:
        begin
          FEdit := TComboBox.Create(nil);
          with FEdit as TComboBox do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value[FColumn - 2];
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtMemo:
        begin
          FEdit := TComboBox.Create(nil);
          // in reality this should be a drop down memo but this requires
          // a special control
          with FEdit as TComboBox do
          begin
            Visible := False;
            Parent := Tree;
            Text := Data.Value[FColumn - 2];
            Items.Add(Data.Value[FColumn - 2]);
            OnKeyDown := EditKeyDown;
          end;
        end;
      vtDate:
        begin
          FEdit := TDateTimePicker.Create(nil);
          with FEdit as TDateTimePicker do
          begin
            Visible := False;
            Parent := Tree;
            CalColors.MonthBackColor := clWindow;
            CalColors.TextColor := clBlack;
            CalColors.TitleBackColor := clBtnShadow;
            CalColors.TitleTextColor := clBlack;
            CalColors.TrailingTextColor := clBtnFace;
            Date := StrToDate(Data.Value[FColumn - 2]);
            OnKeyDown := EditKeyDown;
          end;
        end;
    else
      Result := False;
    end;
  end;
end;

//----------------- TMainForm ------------------------------------------------------------------------------------------

function TMainForm.CanDisplay(const Name: String): Boolean;

// determines whether the given file is one we can display in the image tree

var
  Ext: String;
  I: Integer;

begin
  if not FExtensionsInitialized then
  begin
    FExtensionsInitialized := False;
    FExtensionList := TStringList.Create;
    {$ifdef GraphicEx}
    FileFormatList.GetExtensionList(FExtensionList);
    for I := 0 to FExtensionList.Count - 1 do
      FExtensionList[I] := '.' + FExtensionList[I];
    {$else}
    // GraphicEx is not used so add some default extensions
    with FExtensionList do
    begin
      Add('.bmp');
      Add('.ico');
      Add('.jpg');
      Add('.jpeg');
      Add('.wmf');
      Add('.emf');
    end;
    {$endif}
  end;

  Result := False;
  Ext := ExtractFileExt(Name);
  for I := 0 to FExtensionList.Count - 1 do
  begin
    Result := CompareText(Ext, FExtensionList[I]) = 0;
    if Result then
      Break;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.CloseButtonClick(Sender: TObject);

begin
  Close;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.AddButtonClick(Sender: TObject);

var
  Count: Cardinal;
  Start: Cardinal;
  
begin 
  Screen.Cursor := crHourGlass;
  with VST1 do
  try
    Start := GetTickCount;
    case (Sender as TButton).Tag of
      0: // add to root
        begin
          Count := StrToInt(NodeCountEdit.Text);
          RootNodeCount := RootNodeCount + Count;
        end;
      1: // add as child
        if Assigned(FocusedNode) then
        begin
          Count := StrToInt(NodeCountEdit.Text);
          ChildCount[FocusedNode] := ChildCount[FocusedNode] + Count;
          Expanded[FocusedNode] := True;
          InvalidateToBottom(FocusedNode);
        end;
    end;
    Label1.Caption := Format('Last operation duration: %d ms', [GetTickCount - Start]);
    Label3.Caption := Format('Nodes in tree: %d', [VST1.TotalCount]);
  finally
    Screen.Cursor := crDefault;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.ClearButtonClick(Sender: TObject);

var
  Start: Cardinal;

begin
  Screen.Cursor := crHourGlass;
  try
    Start := GetTickCount;
    VST1.Clear;
    Label1.Caption := Format('Last operation duration: %d ms', [GetTickCount - Start]);
    Label3.Caption := 'Nodes in tree: 0';
  finally
    Screen.Cursor := crDefault;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.LoadBackgroundButtonClick(Sender: TObject);

begin
  with OPD do
  begin
    if Execute then
    begin
      VST1.Background.LoadFromFile(FileName);
      if SBCheckBox.Checked then
        VST1.Invalidate
      else
        SBCheckBox.Checked := True;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SBCheckBoxClick(Sender: TObject);

begin
  if SBCheckBox.Checked then
    VST1.Options := VST1.Options + [voShowBackground]
  else
    VST1.Options := VST1.Options - [voShowBackground];
    
  VST1.Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST1GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
  var Text: WideString);

begin
  Text := Format('Level %d, Index %d', [Sender.GetNodeLevel(Node), Node.Index]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.SelectFolderButtonClick(Sender: TObject);

var
  Dir: String;

begin
  if SelectDirectory('Select folder to browse for images', '', Dir) then
  begin
    ImageFolderLabel.Caption := Dir;
    // fill root level of image tree
    VDT1.Clear;
    // only one root node which represents the selected folder
    VDT1.RootNodeCount := 1;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.CheckMarkComboChange(Sender: TObject);

begin
  VST2.CheckImageKind := TCheckImageKind(CheckMarkCombo.ItemIndex);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.FormCreate(Sender: TObject);

var
  SFI: TSHFileInfo;
  Buffer: array[0..MAX_PATH] of Char;

begin
  Randomize;
  
  GetWindowsDirectory(Buffer, MAX_PATH);
  ImageFolderLabel.Caption := Buffer;
  VDT1.NodeDataSize := SizeOf(TImageData);
  VST3.NodeDataSize := SizeOf(TPropertyData);
  VST5.NodeDataSize := SizeOf(TGridData);
  
  // displays initially the windows folder (see OnInitNode of the treeview)
  VDT1.RootNodeCount := 1;
  VDT1.Expanded[VDT1.GetFirstNode] := True;

  // node data size of the third tree is passed via OnGetNodeDataSize of that tree

  SystemImages.Handle := SHGetFileInfo('', 0, SFI, SizeOf(SFI), SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
  SystemImages.ShareImages := True;

  // thumb size
  FThumbSize := 100;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.PageControl1Change(Sender: TObject);

begin
  VST1.Clear;
  // disable timer (and stop so the animation) if the animation treeview is not
  // the one which is currently visible
  AnimationTimer.Enabled := PageControl1.Activepage = TabSheet5;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.RescaleImage(Source, Target: TBitmap);

// if source is in at least one dimension larger than the thumb size then rescale
// source but keep aspect ratio

var
  NewWidth,
  NewHeight: Integer;
  
begin
  if (Source.Width > FThumbSize) or (Source.Height > FThumbSize) then
  begin
    if Source.Width > Source.Height then
    begin
      NewWidth := FThumbSize;
      NewHeight := Round(FThumbSize * Source.Height / Source.Width);
    end
    else
    begin
      NewHeight := FThumbSize;
      NewWidth := Round(FThumbSize * Source.Width / Source.Height);
    end;

    Target.Width := NewWidth;
    Target.Height := NewHeight;
    SetStretchBltMode(Target.Canvas.Handle, HALFTONE);
    StretchBlt(Target.Canvas.Handle, 0, 0, NewWidth, NewHeight, Source.Canvas.Handle, 0, 0,
               Source.Width, Source.Height, SRCCOPY);
  end
  else
    Target.Assign(Source);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

var
  Data: PImageData;
  Picture: TPicture;

begin
  Data := Sender.GetNodeData(Node);
  if ParentNode = nil then
  begin
    // top level node, initialize first enumeration
    Data.FullPath := ImageFolderLabel.Caption;
    Data.IsFolder := True;
    GetOpenAndClosedIcons(Data.FullPath, Data.OpenIndex, Data.CloseIndex);
    if HasChildren(Data.FullPath) then
      Include(InitialStates, ivsHasChildren);
  end
  else
  begin
    Picture := TPicture.Create;
    if not Data.IsFolder then
    try
      try
        Data.Image := TBitmap.Create;
        Picture.LoadFromFile(Data.FullPath);
        if not (Picture.Graphic is TBitmap) then
        begin
          // Some extra steps needed to keep non TBitmap descentants alive when scaling.
          // This is needed because when accessing Picture.Bitmap all non-TBitmap content
          // will simply be erased (definitly the wrong action, but we can't do anything
          // to prevent this). Hence we must explicitly draw the graphic to a bitmap.
          with Data.Image do
          begin
            Width := Picture.Width;
            Height := Picture.Height;
            Canvas.Draw(0, 0, Picture.Graphic);
          end;
          Picture.Bitmap.Assign(Data.Image);
        end;
        RescaleImage(Picture.Bitmap, Data.Image);

        // collect some additional image properties
        case Picture.Bitmap.PixelFormat of
          pf1bit:
            Data.Properties := '2 colors, ' + Data.Properties;
          pf4bit:
            Data.Properties := '16 colors, ' + Data.Properties;
          pf8bit:
            Data.Properties := '256 colors, ' + Data.Properties;
          pf15bit:
            Data.Properties := '32K colors, ' + Data.Properties;
          pf16bit:
            Data.Properties := '64K colors, ' + Data.Properties;
          pf24bit:
            Data.Properties := '16M colors, ' + Data.Properties;
          pf32bit:
            Data.Properties := '16M+ colors, ' + Data.Properties;
        end;
        Data.Properties := Format('%dx%d pixels, ', [Picture.Width, Picture.Height]) + Data.Properties;

        if Cardinal(Data.Image.Height) + 4 > TVirtualDrawTree(Sender).DefaultNodeHeight then
          Sender.NodeHeight[Node] := Data.Image.Height + 4;
      except
        Data.Image.Free;
        Data.Image := nil;
      end;
    finally
      Picture.Free;
    end
    else    
      if HasChildren(Data.FullPath) then
        Include(InitialStates, ivsHasChildren);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1FreeNode(Sender: TBaseVirtualTree; Node: PVirtualNode);

var
  Data: PImageData;

begin
  Data := Sender.GetNodeData(Node);
  Data.Image.Free;
  Finalize(Data^); // clear string data 
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1DrawNode(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; ClipRect,
  NodeRect: TRect; Column: Integer);

var
  Data: PImageData;
  X, Y: Integer;

begin
  Data := Sender.GetNodeData(Node);
  case Column of
    0:
      begin
        Y := NodeRect.Top + (NodeRect.Bottom - NodeRect.Top - Canvas.TextHeight('Wg')) div 2 + 2;
        Canvas.TextOut(NodeRect.Left + 2, Y, ExtractFileName(Data.FullPath));
      end;
    1:
      begin
        if Assigned(Data.Image) then
        begin
          X := NodeRect.Left + (VDT1.Header.Columns[1].Width - Data.Image.Width) div 2 + 2;
          Canvas.Draw(X, NodeRect.Top + 2, Data.Image);
        end;
      end;
    2:
      if Assigned(Data.Image) then
      begin
        Y := NodeRect.Top + (NodeRect.Bottom - NodeRect.Top - Canvas.TextHeight('Wg')) div 2 + 2;
        Canvas.TextOut(NodeRect.Left + 2, Y, Data.Properties);
      end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1GetNodeWidth(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer;
  var NodeWidth: Integer);

var
  Data: PImageData;

begin
  Data := Sender.GetNodeData(Node);
  case Column of
    0:
      begin
        if Node.Parent = Sender.RootNode then
          NodeWidth := Canvas.TextWidth(Data.FullPath) + 4
        else
          NodeWidth := Canvas.TextWidth(ExtractFileName(Data.FullPath)) + 4;
      end;
    1:
      begin
        if Assigned(Data.Image) then
          NodeWidth := Data.Image.Width;
      end;
    2:
      NodeWidth := VDT1.Header.Columns[Column].Width; 
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TMainForm.FileCompare(Sender: TBaseVirtualTree; Node1, Node2: PVirtualNode; Column: Integer): Integer;

// used to sort the image draw tree
 
var
  Data1,
  Data2: PImageData;

begin
  Data1 := Sender.GetNodeData(Node1);
  Data2 := Sender.GetNodeData(Node2);
  // folder are always before files
  if Data1.IsFolder <> Data2.IsFolder then
  begin
    // one of both is a folder the other a file
    if Data1.IsFolder then
      Result := -1
    else
      Result := 1;
  end
  else // both are of same type (folder or file)
    Result := CompareText(Data1.FullPath, Data2.FullPath);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1Expanding(Sender: TBaseVirtualTree; Node: PVirtualNode; var Allowed: Boolean);

// Called just before a node with children (only folder nodes can have children) is expanded.
// We use this event (instead of InitChildren) because we have to enumerate all file objects in the
// node's folder anyway.

var
  Data,
  ChildData: PImageData;
  SR: TSearchRec;
  ChildNode: PVirtualNode;
  NewName: String;
  
begin
  if not Sender.ChildrenInitialized[Node] then
  begin
    Data := Sender.GetNodeData(Node);
    if FindFirst(IncludeTrailingBackslash(Data.FullPath) + '*.*', faAnyFile, SR) = 0 then
    begin
      Screen.Cursor := crHourGlass;
      try
        repeat
          if (SR.Name <> '.') and (SR.Name <> '..') then
          begin
            NewName := IncludeTrailingBackslash(Data.FullPath) + SR.Name;
            if (SR.Attr and faDirectory <> 0) or CanDisplay(NewName) then
            begin
              ChildNode := Sender.AddChild(Node);
              ChildData := Sender.GetNodeData(ChildNode);
              ChildData.FullPath := NewName;
              ChildData.IsFolder := (SR.Attr and faDirectory <> 0);
              if not ChildData.IsFolder then
                ChildData.Properties := Format('%n KB', [SR.Size / 1024]);
              GetOpenAndClosedIcons(ChildData.FullPath, ChildData.OpenIndex, ChildData.CloseIndex);
            end;
          end;
        until FindNext(SR) <> 0;

        // finally sort node
        if Sender.ChildCount[Node] > 0 then
          Sender.Sort(Node, 0, True, FileCompare, False);
      finally
        FindClose(SR);
        Screen.Cursor := crDefault;
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
  var Index: Integer);

var
  Data: PImageData;
  
begin
  if (Kind in [ikNormal, ikSelected]) and (Column = 0) then
  begin
    Data := Sender.GetNodeData(Node);
    if Sender.Expanded[Node] then
      Index := Data.OpenIndex
    else
      Index := Data.CloseIndex;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST1Change(Sender: TBaseVirtualTree; Node: PVirtualNode);

begin
  Label6.Caption := Format('Selected: %d', [VST1.SelectedCount]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2GetNodeDataSize(Sender: TBaseVirtualTree; var NodeDataSize: Integer);

// Returns the size of a node record. Since this size is fixed already at creation time it
// would make sense to avoid this event and assign the value in OnCreate of the form (see there
// for the other trees). But this is a demo program, so I want to show this way too.
// Note the -1 value in VST2.NodeDataSize which primarily causes this event to be fired.

begin
  NodeDataSize := SizeOf(TNodeData2);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2GetDrawInfo(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer;
  TextType: TVSTTextType; var DrawInfo: TDrawInfo);

var
  Data: PNodeData2;

begin
  Data := Sender.GetNodeData(Node);
  case Column of
    0: // main column
      case TextType of
        ttNormal:
          if Data.Level = 0 then
            DrawInfo.Font.Style := DrawInfo.Font.Style + [fsBold];
        ttStatic:
          begin
            DrawInfo.Font.Color := clBlue;
            DrawInfo.Font.Style := DrawInfo.Font.Style - [fsBold];
          end;
      end;
    1: // image column (there is no text)
      ;
    2: // language column (just use default text)
      ;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
  var Text: WideString);

// Returns the text as it is store in the nodes data record, we could also just return a
// dynamically created string, which would be very memory saving, but not of much use unless
// it is static and does never change.

var
  Data: PNodeData2;

begin
  Data := Sender.GetNodeData(Node);
  Text := '';
  case Column of
    0: // main column (has two different captions)
      case TextType of
        ttNormal:
          Text := Data.Caption;
        ttStatic:
          Text := Data.StaticText;
      end;
    1: // no text in the image column
      ;
    2:
      if TextType = ttNormal then
        Text := Data.ForeignText;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

var
  Data: PNodeData2;

begin
  Data := Sender.GetNodeData(Node);
  with Data^ do
  begin
    Level := Sender.GetNodeLevel(Node);
    if Level < 4 then
    begin
      Include(InitialStates, ivsHasChildren);
      if Level = 0 then
        Include(InitialStates, ivsExpanded);
    end;

    Caption := Format('Level %d, Index %d', [Level, Node.Index]);
    if Level in [0, 3] then
      StaticText := '(blue)';

    ForeignText := '';
    case Data.Level of
      1:
        begin
          ForeignText := WideChar($3042);
          ForeignText := ForeignText + WideChar($3045) + WideChar($30A5) + WideChar($30A6) + WideChar($30A7);
        end;
      2:
        begin
          ForeignText := WideChar($0430);
          ForeignText := ForeignText + WideChar($0431) + WideChar($0432) + WideChar($0433) + WideChar($0434);
        end;
      3:
        begin
          ForeignText := WideChar($FB31);
          ForeignText := ForeignText + WideChar($FB32) + WideChar($FB33) + WideChar($FB34) + WideChar($FB35);
        end;
      4:
        begin
          ForeignText := WideChar($2104);
          ForeignText := '"nichts" is unmglich ' + ForeignText;
        end;
    end;
    Node.CheckType := TCheckType(4 - Data.Level);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.BitBtn1Click(Sender: TObject);

begin
  with FontDialog1 do
  begin
    Font := VST2.Font;
    if Execute then
      VST2.Font := Font;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
  var Index: Integer);

var
  Data: PNodeData2;

begin
  // for this demo only the normal image is shown, you can easily change this for the state and overlay image
  if Kind in [ikNormal, ikSelected] then
  begin
    Data := Sender.GetNodeData(Node);
    case Column of
      -1, // general case
      0: // main column
        Index := Data.Level;
      1: // image only column
        if Sender.FocusedNode = Node then
          Index := 6;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2InitChildren(Sender: TBaseVirtualTree; Node: PVirtualNode; var ChildCount: Cardinal);

// the tree is set to 5 levels a 5 children (~4000 nodes)

var
  Data: PNodeData2;

begin
  Data := Sender.GetNodeData(Node);
  if Data.Level < 4 then
    ChildCount := 5;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST2NewText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; Text: WideString);

// the caption of a node has been changed, keep this in the node record

var
  Data: PNodeData2;

begin
  Data := Sender.GetNodeData(Node);
  Data.Caption := Text;
end;

//----------------------------------------------------------------------------------------------------------------------

// I just want to simulate in VST3 the document properties dialog (property sheet)
// so most of tree data is hard coded here.

procedure TMainForm.VST3InitChildren(Sender: TBaseVirtualTree; Node: PVirtualNode; var ChildCount: Cardinal);

begin
  case Node.Index of
    0:
      ChildCount := 13;
    1:
      ChildCount := 8;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

var
  Data: PPropertyData;
  
begin
  if ParentNode = nil then
  begin
    InitialStates := [ivsHasChildren, ivsExpanded];
  end
  else
  begin
    Data := Sender.GetNodeData(Node);
    Data.ValueType := ValueTypes[ParentNode.Index, Node.Index];
    Data.Value := DefaultValue[ParentNode.Index, Node.Index];
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
  var Text: WideString);

var
  Data: PPropertyData;
  
begin
  if TextType = ttNormal then
    case Column of
      0:
        if Node.Parent = Sender.RootNode then
        begin
          // root nodes
          if Node.Index = 0 then
            Text := 'Description'
          else
            Text := 'Origin';
        end
        else
          Text := PropertyTexts[Node.Parent.Index, Node.Index, ptkText];
      1:
        begin
          Data := Sender.GetNodeData(Node);
          Text := Data.Value;
        end;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3GetHint(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
  var Text: WideString);

begin
  if (Column = 0) and (Node.Parent <> Sender.RootNode) then
    Text := PropertyTexts[Node.Parent.Index, Node.Index, ptkHint];
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
  var Index: Integer);

var
  Data: PPropertyData;

begin
  if (Kind in [ikNormal, ikSelected]) and (Column = 0) then
  begin
    if Node.Parent = Sender.RootNode then
      Index := 12 // root nodes, this is an open folder
    else
    begin
      Data := Sender.GetNodeData(Node);
      if Data.ValueType <> vtNone then
        Index := 14
      else
        Index := 13;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3Editing(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; var Allowed: Boolean);

var
  Data: PPropertyData;

begin
  with Sender do
  begin
    Data := GetNodeData(Node);
    Allowed := (Node.Parent <> RootNode) and (Column = 1) and (Data.ValueType <> vtNone);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3Change(Sender: TBaseVirtualTree; Node: PVirtualNode);

begin
  with Sender do
  begin
    if Assigned(Node) and (Node.Parent <> RootNode) then
    begin
      // Note: the test whether a node can really be edited is done in the OnEditing event
      EditNode(Node, 1);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3CreateEditor(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer;
  out EditLink: IVTEditLink);

begin
  if Sender = VST3 then
    EditLink := TPropertyEditLink.Create
  else
    EditLink := TGridEditLink.Create;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST3GetDrawInfo(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer;
  TextType: TVSTTextType; var DrawInfo: TDrawInfo);

var
  Data: PPropertyData;

begin
  // make the root nodes underlined and changed nodes bold
  if Node.Parent = Sender.RootNode then
    DrawInfo.Font.Style := [fsUnderline]
  else
  begin
    Data := Sender.GetNodeData(Node);
    if Data.Changed then
      DrawInfo.Font.Style := [fsBold]
    else
      DrawInfo.Font.Style := [];
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1GetHintSize(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; var R: TRect);

// Draw trees must manage parts of the hints themselves. Here we return the size of the
// hint window we want to show or an empty rectangle in the case we don't want a hint at all.

var
  Data: PImageData;

begin
  Data := Sender.GetNodeData(Node);
  if Assigned(Data) and Assigned(Data.Image) and (Column = 1) then
    R := Rect(0, 0, 2 * Data.Image.Width, 2 * Data.Image.Height)
  else
    R := Rect(0, 0, 0, 0);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VDT1DrawHint(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; R: TRect; Column: Integer);

// Here we actually paint the hint. It is the image in a larger size.

var
  Data: PImageData;

begin
  Data := Sender.GetNodeData(Node);
  if Assigned(Data) and Assigned(Data.Image) and (Column = 1) then
  begin
    SetStretchBltMode(Canvas.Handle, HALFTONE);
    StretchBlt(Canvas.Handle, 0, 0, 2 * Data.Image.Width, 2 * Data.Image.Height, Data.Image.Canvas.Handle, 0, 0,
               Data.Image.Width, Data.Image.Height, SRCCOPY);
  end;
end;

//----------------- Animation treeview ---------------------------------------------------------------------------------

type
  TAnimationEntry = record
    Start,
    Count,
    Current: Cardinal;
  end;

var
  Animation: array[0..3] of TAnimationEntry = (
    (Start:  4; Count:  6; Current: 0),
    (Start: 10; Count: 12; Current: 0),
    (Start: 22; Count:  5; Current: 0),
    (Start: 27; Count: 12; Current: 0)
  );

procedure TMainForm.AnimationTimerTimer(Sender: TObject);

// used to keep the animations going

var
  I: Integer;

begin
  for I := 0 to 3 do
    with Animation[I] do
      Current := (Current + 1) mod Count;
  VST4.Invalidate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST4GetImageIndex(Sender: TBaseVirtualTree; Node: PVirtualNode; Kind: TVTImageKind; Column: Integer;
  var Index: Integer);

begin
  if Node.Parent = Sender.RootNode then
  begin
    Index := Node.Index;
  end
  else
    // For the animation we have calculated the image index for each node group in the
    // timer event. The current position plus the start image index give the image index
    // to show. To make the tree even more interesting each node in a group is set to one
    // image ahead to its previous sibling (with wrap around).
    with Animation[Node.Parent.Index] do
      Index := Start + (Current + Node.Index) mod Count;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST4InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

begin
  if ParentNode = nil then
    Include(InitialStates, ivsHasChildren);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST4InitChildren(Sender: TBaseVirtualTree; Node: PVirtualNode; var ChildCount: Cardinal);

begin
  ChildCount := 5;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.TimerEditChange(Sender: TObject);

begin
  AnimationTimer.Interval := StrToInt(TimerEdit.Text);
end;

//----------------- Virtual Treeview as grid ---------------------------------------------------------------------------

procedure TMainForm.VST5BeforeItemErase(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; ItemRect: TRect;
  var Color: TColor; var EraseAction: TItemEraseAction);

// some nodes should get a different background color

begin
  with Canvas do
  begin
    if Node.Index mod 6 = 0 then
      Color := $49DDEF // $70A33F // $436BFF
    else
      Color := VST5.Color;
    EraseAction := eaColor;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST5AfterItemPaint(Sender: TBaseVirtualTree; Canvas: TCanvas; Node: PVirtualNode; ItemRect: TRect);

begin
  with Canvas do
  begin
    // simulate a fixed column by filling column 1 (0, the main column, isn't visible at all)
    // with an edge similar to those of TCustomGrid 
    ItemRect.Right := ItemRect.Left + 12;
    // the passed rectangle is the client area to paint into without scrolling taking into account
    OffsetRect(ItemRect, Sender.OffsetX, 0);

    DrawEdge(Handle, ItemRect, BDR_RAISEDINNER, BF_RECT or BF_MIDDLE);
    if Node = Sender.FocusedNode then
      TreeImages.Draw(Canvas, ItemRect.Left, ItemRect.Top, 17);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST5FocusChanging(Sender: TBaseVirtualTree; OldNode, NewNode: PVirtualNode; OldColumn,
  NewColumn: Integer; var Allowed: Boolean);

begin
  // do not allow focusing the indicator column (which simulates a fixed column)
  Allowed := NewColumn > 1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST5InitNode(Sender: TBaseVirtualTree; ParentNode, Node: PVirtualNode;
  var InitialStates: TVirtualNodeInitStates);

// prepares all nodes for editing

var
  Data: PGridData;
  D: TDateTime;

begin
  // we use parts of the "infrastructure" of the property tree for the nodes of the grid tree
  Data := Sender.GetNodeData(Node);

  // these are the editor kinds used in the grid tree
  Data.ValueType[0] := vtNumber;
  Data.ValueType[1] := vtPickString;
  Data.ValueType[2] := vtPickString;
  Data.ValueType[3] := vtDate;

  // fill some default values
  Data.Value[0] := Variant(Node.Index);
  Data.Value[1] := 'John';
  Data.Value[2] := 'Doe';
  // a date value slightly randomized around today,
  // need the way using a local variable to tell the compiler we are not using a float as
  // variant, but a datetime
  D := Date + Random(14) - 7;
  Data.Value[3] := D;

  if Sender.FocusedColumn < 2 then
    Sender.FocusedColumn := 2;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST5GetText(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer; TextType: TVSTTextType;
  var Text: WideString);

var
  Data: PGridData;

begin
  if Column > 1 then
  begin
    Data := Sender.GetNodeData(Node);
    Text := Data.Value[Column - 2];
  end
  else
    Text := '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.VST5GetDrawInfo(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: Integer;
  TextType: TVSTTextType; var DrawInfo: TDrawInfo);

var
  Data: PGridData;

begin
  Data := Sender.GetNodeData(Node);
  if Data.Changed then
    DrawInfo.Font.Style := [fsBold];
  if (Column - 2) = (Integer(Node.Index) mod (VST5.Header.Columns.Count - 1)) then
    DrawInfo.Background := $E0E0E0;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TMainForm.GridLineCheckBoxClick(Sender: TObject);

begin
  if GridLineCheckBox.Checked then
    VST5.Options := VST5.Options + [voShowHorzGridLines, voShowVertGridLines]
  else
    VST5.Options := VST5.Options - [voShowHorzGridLines, voShowVertGridLines];
end;

//----------------------------------------------------------------------------------------------------------------------

end.
