unit TsList;

 {*--------------------------------------------------------------------------*
  COPYRIGHT 2001 BY TS-INFORMATIK, GERMANY

  Author  :   Thomas Suess
  Date    :   09.10.2001
  Version :   0.0.1T

  Email   :   th.suess@t-online.de
  Web     :   ts-informatik.de

  Modified:


  Abstract: Procedures and functions to build static and dynamic lists. You
            can also use it as priority queues and s.o.


  Notice:  This Delphi unit is basically FREEWARE.
          (Please send me a mail if you found a bug or if you have a good idea)
  *--------------------------------------------------------------------------*}


{ --------------------------------------------------------------------------- }
{ +++ Short Description +++++++++++++++++++++++++++++++++++++++++++++++++++++ }
{ --------------------------------------------------------------------------- }
{
Static List (ARRAY)
-------------------
1. Declare ARRAY (Frame/Userdata Memory) in Application unit
   Type
    // do not use open arrays/strings in the userdata record !!
    tUserData = record
                  s1 : string [100];
                  a1 : array [0..9] of char;
                  id : integer;
                end;
  Var
    aPool    : array [0..4] of tTsFrame;
    auPool   : array [0..4] of tUserData;
    pHeader  : tTsHeader;    // array pool header
    u1Header : tTsHeader;    // user header
    u2Header : tTsHeader;    // user header

2. Init Array Pool

   for i := 0 to High(aPool) do
    if i <= High (auPool) then
      aPool [i].UD := @auPool[i];

  InitArrayPool (pHeader, @aPool, High(aPool), FALSE);
  InitHeaderTs (u1Header, 0, 0, false);
  InitHeaderTs (u2Header, 0, 0, false);

3. User 1 (u1Header) and 2 or more can used one or more array pools.
   All procedures and functions with the remark ARRAY or ARRAY / LIST can
   used for static (array) list.


Dynamic List
------------
1. Declare userdata type and List - Header
   Type
    // do not use open arrays/strings in the userdata record !!
    tUserData = record
                  s1 : string [100];
                  a1 : array [0..9] of char;
                  id : integer;
                end;

    pUser     = ^tUserData;

  Var
    u3Header : tTsHeader;    // user header
    uFrame   : tTsFrame;

2. Init user Header
   InitHeaderTs (u3Header, 0, 0, true);

3. All procedures and functions with the remark LIST or ARRAY / LIST can
   used for dynamic list. The dynamic memory for userdata must be managed
   by the application.

     InitFrameTs (@uFrame, 1, 0, 0, false, nil);
     New (pUser);
     pUser^.s1 := 'Thomas Suess' + #0;
     ok := SetULD (u3Header, uframe, pUser);
     .
     .
     if GetxULD (u3Header, uFrame, 0) then
       if uFrame.UD <> nil then dispose (uFrame.UD);

++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ }


interface

uses Windows;

type
  // pointer to frame
  tTsFramePtr = ^tTsFrame;

  // frame record
  tTsFrame    =  record
                   Prio      : Integer;   // High Prio = MaxInteger
                   Id        : Integer;   // user data
                   HWnd      : Integer;   // user data
                   DisposeUD : Boolean;   // only used for dyn. list
                   UD        : Pointer;   // pointer to user data
                   Next      : tTsFramePtr;
                  end;
  // header record
  tTsHeader   =  record
                   Count     : Integer;  // Act. count [0,1..] for frames
                   MaxCount  : Integer;  // used only for static list
                   HEvent    : Integer;  // Event handle
                   DynList   : Boolean;  // TRUE if dynamic list
                   Next      : tTsFramePtr;
                 end;


{ -- ARRAY ------------------------------------------------------------ }
Procedure InitArrayPool  ( Var  header : tTsHeader ;
                                p      : Pointer   ;
                                max    : Integer   ;
                                udnil  : Boolean  );

{ -- ARRAY ------------------------------------------------------------ }
Procedure ClearUAHeader  ( Var pHeader : tTsHeader   ;
                           Var uHeader : tTsHeader  );


{ -- ARRAY ------------------------------------------------------------ }
function GetxUAD  ( Var pHeader : tTsHeader ;
                    Var uHeader : tTsHeader ;
                    Var frame   : tTsFrame  ;
                        ix      : Integer   ) : Boolean;


{ -- ARRAY ------------------------------------------------------------ }
function SetUAD  ( Var pHeader : tTsHeader ;
                   Var uHeader : tTsHeader ;
                   Var frame   : tTsFrame  ;
                       pUD     : Pointer   ;
                       UDLen   : Cardinal) : Boolean;


{ -- ARRAY ------------------------------------------------------------ }
function SendUAEvent  (  Var pHeader : tTsHeader ;
                         Var uHeader : tTsHeader ;
                         Var frame   : tTsFrame  ;
                             pUD     : Pointer   ;
                             UDLen   : Cardinal) : Boolean;


{ -- LIST ------------------------------------------------------------- }
Procedure ClearULHeader  ( Var uHeader : tTsHeader  );


{ -- LIST ------------------------------------------------------------- }
function GetxULD  ( Var uHeader : tTsHeader ;
                    Var frame   : tTsFrame  ;
                        ix      : Integer   ) : Boolean;



{ -- LIST ------------------------------------------------------------- }
function SetULD  ( Var uHeader : tTsHeader ;
                   Var frame   : tTsFrame  ;
                       pUD     : Pointer ) : Boolean;


{ -- LIST ------------------------------------------------------------- }
function SendULEvent  (  Var uHeader : tTsHeader ;
                         Var frame   : tTsFrame  ;
                             pUD     : Pointer  ): Boolean;


{ -- ARRAY / LIST ----------------------------------------------------- }
Procedure InitFrameTs  ( p     : tTsFramePtr ;
                         prio  : Integer     ;
                         id    : Integer     ;
                         hwnd  : Integer     ;
                         dp    : Boolean     ;
                         ud    : Pointer    );


{ -- ARRAY / LIST ----------------------------------------------------- }
Procedure InitHeaderTs  ( Var h     : tTsHeader ;
                              max   : Integer   ;
                              event : Integer   ;
                              dyn   : Boolean  );


{ -- ARRAY / LIST ----------------------------------------------------- }
function ReadxTs   ( Var uHeader : tTsHeader ;
                     Var frame   : tTsFrame  ;
                         ix      : Integer   ) : Boolean;


{ -- ARRAY / LIST ----------------------------------------------------- }
function CreateEventTs ( Var uHeader : tTsHeader ) : Boolean;


{ -- ARRAY / LIST ----------------------------------------------------- }
function WaitEventTs ( Var uHeader : tTsHeader;
                           Time    : Integer  ) : Cardinal;



implementation

Var
  CSTsList : TRTLCriticalSection;



{ ======================== LOCAL PROCEDURE ============================


{ --------------------------------------------------------------------- }
Procedure PutInList  ( Var  header : tTsHeader    ;
                            p      : tTsFramePtr );

{ ---------------------------------------------------------------------
  Input    : p       => pointer to frame
  Output   :
  InOut    : header  => ListHeader

  FUNCTION : Add/Insert frame to list
  --------------------------------------------------------------------- }
Var
  ActPtr, OldPtr : tTsFramePtr;
  i              : Integer;

begin // PutInList
  try
    EnterCriticalSection (CSTsList);
    Inc (header.Count);
    ActPtr := header.Next; OldPtr := @header; i := 0;
    If ActPtr = nil Then
    begin // first frame after header
      header.Next       := p;
      header.Next^.Next := nil;
    end
    else
    begin
      While (ActPtr^.Next <> nil) AND (ActPtr^.Prio >= p^.Prio) DO
      begin  // search in list
        OldPtr := ActPtr;
        ActPtr := OldPtr^.Next;
        Inc (i);
      end;
      If ActPtr^.Prio < p^.Prio Then
      begin  // insert frame for ActPtr
        p^.Next      := ActPtr;
        if i <> 0 then
          OldPtr^.Next := p
        else
          header.Next  := p;
      end
      else
      begin // add frame
        p^.Next      := nil;
        ActPtr^.Next := p;
      end;
    end;
  finally
    LeaveCriticalSection (CSTsList);
  end;
end;


{ --------------------------------------------------------------------- }
Procedure GetFromList  ( Var  header : tTsHeader    ;
                         Var  p      : tTsFramePtr  ;
                              ix     : Integer      ;
                              del    : Boolean     );
{ ---------------------------------------------------------------------
  Input    : ix      => Index
             del     => TRUE : delete frame in list
  Output   :
  InOut    : header  => ListHeader
             p       => Pointer to frame
  FUNCTION : Get a frame from the List.
  --------------------------------------------------------------------- }
Var
  ActPtr, OldPtr  : tTsFramePtr;
  i               : Integer;

begin (* GetFromList *)
  try
    EnterCriticalSection (CSTsList);
    p := nil;
    IF (header.Next <> nil) AND (header.Count <> 0) Then
    begin
      ActPtr := header.Next; OldPtr := nil; i := 0;
      While (ActPtr^.Next <> nil) AND ( i <> ix) DO
      begin  // search in list
        OldPtr := ActPtr;
        ActPtr := OldPtr^.Next;
        Inc (i);
      end;
      If i = ix Then
      begin  // ix (Index) found
        p := ActPtr;
        If del Then
        begin  // delete frame
          Dec (header.Count);
          If i = 0 Then
            header.Next := header.Next^.Next // first frame
          else if i = header.Count Then      // last frame
            OldPtr^.Next := nil
          else
            OldPtr^.Next := ActPtr^.Next;    // ix frame
        end;
      end;
    end;
  finally
    LeaveCriticalSection (CSTsList);
  end;
end;




{ ===================== PROCEDURE ONLY FOR ARRAY ======================


{ --------------------------------------------------------------------- }
Procedure InitArrayPool  ( Var  header : tTsHeader ;
                                p      : Pointer   ;
                                max    : Integer   ;
                                udnil  : Boolean  );

{ ---------------------------------------------------------------------
  Input    : p       => Pointer to array frame pool
             max     => init parameter (maxcount)
  Output   :
  InOut    : header  => PoolHeader

  FUNCTION : Init static array pool.
  --------------------------------------------------------------------- }
Var
  tPtr   : PChar;
  udptr  : PChar;
  i      : Integer;

begin (* InitArrayPool *)
  InitHeaderTs  ( header, max, 0, FALSE);
  tPtr := p;
  for i := 0 to max do
  begin
    if udnil then udptr := nil
    else udptr := tTsFramePtr(tPtr).UD;
    InitFrameTs (tTsFramePtr(tPtr),0, 0, 0, FALSE, udptr);
    PutInList (header, tTsFramePtr(tPtr));
    tPtr := tPtr + SizeOf (tTsFrame);
  end;
end;



{ --------------------------------------------------------------------- }
Procedure ClearUAHeader  ( Var pHeader : tTsHeader   ;
                           Var uHeader : tTsHeader  );
{ ---------------------------------------------------------------------
  Input    :
  Output   :
  InOut    : pHeader  : Pool Header
             uHeader  : User Header

  FUNCTION : Clear user array header. Put all user frames -> pool
  --------------------------------------------------------------------- }
Var
  p : tTsFramePtr;

begin // ClearUserArray - Header
  GetFromList (uHeader, p, 0, TRUE);
  While p <> nil DO
  begin
    InitFrameTs (p, 0, 0, 0, FALSE, p.UD);
    PutInList   (pHeader, p);
    GetFromList (uHeader, p, 0, TRUE);
  end;
end;


{ --------------------------------------------------------------------- }
function GetxUAD  ( Var pHeader : tTsHeader ;
                    Var uHeader : tTsHeader ;
                    Var frame   : tTsFrame  ;
                        ix      : Integer   ) : Boolean;

{ ---------------------------------------------------------------------
  Input    : ix       : User farme Index
  Output   : Result   : TRUE/FALSE
  InOut    : pHeader  : Pool Header
             uHeader  : User Header
             frame    : frame data

  FUNCTION : Get index frame form user array list and return
             it to the array pool
  --------------------------------------------------------------------- }
Var
  p : tTsFramePtr;

begin // GetIndexUserArrayData
  Result := FALSE;
  GetFromList (uHeader, p, ix, TRUE);
  If p <> nil THEN
  begin
    frame  := p^;
    Result := TRUE;
    InitFrameTs (p, 0, 0, 0, FALSE, p^.UD);
    PutInList   (pHeader, p);
  end;
end;


{ --------------------------------------------------------------------- }
function SetUAD  ( Var pHeader : tTsHeader ;
                   Var uHeader : tTsHeader ;
                   Var frame   : tTsFrame  ;
                       pUD     : Pointer   ;
                       UDLen   : Cardinal  ) : Boolean;

{ ---------------------------------------------------------------------
  Input    : pUD      : pointer to User Data
  Output   : Result   : TRUE/FALSE
  InOut    : pHeader  : Pool Header
             uHeader  : User Header
             frame    : frame data

  FUNCTION : set user frame (array) from pool to user list
  --------------------------------------------------------------------- }

Var
  p     : tTsFramePtr;
  i     : Cardinal;
  p1,p2 : ^byte;

begin // SetUserArrayData
  Result := FALSE;
  GetFromList (pHeader, p, 0, TRUE);
  If p <> nil THEN
  begin
    p^.Prio      := frame.Prio;
    p^.Id        := frame.Id;
    p^.HWnd      := frame.HWnd;
    p^.DisposeUD := frame.DisposeUD;
    if (p^.UD <> nil) AND (pUD <> nil) then
    begin // copy user data (memory)
      p1 := p^.UD; p2 := pUD;
      for i := 0 to UDLen-1 do
      begin
        p1^ := p2^;
        Inc (p1); Inc (P2);
     end;
    end;
    PutInList  (uHeader, p);
    Result      := TRUE;
  end;
end;


{ --------------------------------------------------------------------- }
function SendUAEvent  (  Var pHeader : tTsHeader ;
                         Var uHeader : tTsHeader ;
                         Var frame   : tTsFrame  ;
                             pUD     : Pointer   ;
                             UDLen   : Cardinal) : Boolean;

{ ---------------------------------------------------------------------
  Input    : pUD      : pointer to User Data
  Output   : Result   : TRUE/FALSE
  InOut    : pHeader  : Pool Header
             uHeader  : User Header
             frame    : frame data

  FUNCTION : set user frame (array) from pool to user list and send event
  --------------------------------------------------------------------- }

begin // Send User (array) event
  Result := FALSE;
  if uHeader.HEvent <> 0 then
    if SetUAD(pHeader, uHeader, frame, pUD, UDLen) then
      Result := SetEvent(uHeader.HEvent);
end;



{ ===================== PROCEDURE ONLY FOR LIST ======================


{ --------------------------------------------------------------------- }
Procedure ClearULHeader  ( Var uHeader : tTsHeader  );
{ ---------------------------------------------------------------------
  Input    :
  Output   :
  InOut    : uHeader  : user Header

  FUNCTION : Clear user list header.
  --------------------------------------------------------------------- }
Var
  p : tTsFramePtr;

begin // ClearUserList - Header
  if uHeader.DynList then
  begin
    GetFromList (uHeader, p, 0, TRUE);
    While p <> nil DO
    begin
      if p^.UD <> nil then Dispose (p^.UD); // dealloc user mem
      Dispose (p);                          // dealloc frame mem
      GetFromList (uHeader, p, 0, TRUE);    // get next
    end;
  end;
end;



{ --------------------------------------------------------------------- }
function GetxULD  ( Var uHeader : tTsHeader ;
                    Var frame   : tTsFrame  ;
                        ix      : Integer   ) : Boolean;

{ ---------------------------------------------------------------------
  Input    : ix       : User farme Index
  Output   : Result   : TRUE/FALSE
  InOut    : uHeader  : User Header
             frame    : frame data

  FUNCTION : Get index frame form user list and dealloc memory
  --------------------------------------------------------------------- }
Var
  p : tTsFramePtr;

begin // GetIndexUserListData
  Result := FALSE;
  if uHeader.DynList then
  begin
    GetFromList (uHeader, p, ix, TRUE);
    If p <> nil THEN
    begin
      frame  := p^;
      Result := TRUE;
      if p^.DisposeUD AND (p^.UD <> nil) then Dispose (p^.UD);
      Dispose (p);
    end;
  end;
end;



{ --------------------------------------------------------------------- }
function SetULD  ( Var uHeader : tTsHeader ;
                   Var frame   : tTsFrame  ;
                       pUD     : Pointer ) : Boolean;

{ ---------------------------------------------------------------------
  Input    : pUD      : pointer to User Data
  Output   : Result   : TRUE/FALSE
  InOut    : uHeader  : User Header
             frame    : frame data

  FUNCTION : set user frame (list) to new memory
  --------------------------------------------------------------------- }
Var
  p  : tTsFramePtr;

begin // SetUserListData
  Result := FALSE;
  if uHeader.DynList then
  begin
    New (p);
    If p <> nil THEN
    begin
      p^.Prio      := frame.Prio;
      p^.Id        := frame.Id;
      p^.HWnd      := frame.HWnd;
      p^.DisposeUD := frame.DisposeUD;
      p^.UD        := pUD;
      PutInList  (uHeader, p);
      Result       := TRUE;
    end;
  end;
end;


{ --------------------------------------------------------------------- }
function SendULEvent  (  Var uHeader : tTsHeader ;
                         Var frame   : tTsFrame  ;
                             pUD     : Pointer ) : Boolean;

{ ---------------------------------------------------------------------
  Input    : pUD      : pointer to User Data
  Output   : Result   : TRUE/FALSE
  InOut    : uHeader  : User Header
             frame    : frame data

  FUNCTION : set user frame (list) and event
  --------------------------------------------------------------------- }

begin // Send User (list) event
  Result := FALSE;
  if uHeader.HEvent <> 0 then
    if SetULD(uHeader, frame, pUD) then
      Result := SetEvent(uHeader.HEvent);
end;


{ ==================== PROCEDURE FOR ARRAY AND LIST ====================


{ --------------------------------------------------------------------- }
Procedure InitFrameTs  ( p    : tTsFramePtr ;
                         prio : Integer     ;
                         id   : Integer     ;
                         hwnd : Integer     ;
                         dp   : Boolean     ;
                         ud   : Pointer    );
{ ---------------------------------------------------------------------
  Input    : p     => Pointer to new frame
             prio  => List prio (Max Integer = high prio)
             id    => (user) parameter
             hwnd  => (user) parameter
             dp    => TURE : delete dynamic user memory
             udnil => TRUE : set UD to nil
  Output   :
  InOut    :

  FUNCTION : Init new farme.
  --------------------------------------------------------------------- }

begin // InitFrameTs
  p^.Prio      := prio;
  p^.Id        := id;
  p^.HWnd      := hwnd;
  p^.DisposeUD := dp;
  p^.UD        := ud;
  p^.Next      := nil;
end;


{ --------------------------------------------------------------------- }
Procedure InitHeaderTs  ( Var h     : tTsHeader ;
                              max   : Integer   ;
                              event : Integer   ;
                              dyn   : Boolean  );

{ ---------------------------------------------------------------------
  Input    : max   ==> max frame (only for array typ)
             Event ==> Event Handle
             Dyn   ==> TRUE if dyn. List
  Output   :
  InOut    : h  ==> new header

  FUNCTION : Init new header.
  --------------------------------------------------------------------- }

begin // InitHeaderTs
  h.Count    := 0;
  h.MaxCount := max;
  h.HEvent   := event;
  h.DynList  := dyn;
  h.Next     := nil;
end;



{ --------------------------------------------------------------------- }
function ReadxTs   ( Var uHeader : tTsHeader ;
                     Var frame   : tTsFrame  ;
                         ix      : Integer   ) : Boolean;

{ ---------------------------------------------------------------------
  Input    : ix       : User farme Index
  Output   : Result   : TRUE/FALSE
  InOut    : uHeader  : User Header
             frame    : frame data

  FUNCTION : Read index frame from user array/list
  --------------------------------------------------------------------- }
Var
  p : tTsFramePtr;

begin // ReadIndexUserData Array/List
  Result := FALSE;
  GetFromList (uHeader, p, ix, FALSE);
  If p <> nil THEN
  begin
    frame  := p^;
    Result := TRUE;
  end;
end;



{ --------------------------------------------------------------------- }
function CreateEventTs ( Var uHeader : tTsHeader ) : Boolean;

{ ---------------------------------------------------------------------
  Input    :
  Output   : Result   : TRUE/FALSE
  InOut    : uHeader  : user Header

  FUNCTION : Create new event
  --------------------------------------------------------------------- }

begin // Create Event Ts
  Result := FALSE;
  uHeader.HEvent := CreateEvent (nil, true, false, '');
  if uHeader.HEvent <> 0 then
    Result := TRUE;
end;



{ --------------------------------------------------------------------- }
function WaitEventTs ( Var uHeader : tTsHeader;
                           Time    : Integer  ) : Cardinal;

{ ---------------------------------------------------------------------
  Input    : Time     : Timeout for waitstate
  Output   : Result   : Wait access code
  InOut    : uHeader  : user Header

  FUNCTION : Wait for Event
  --------------------------------------------------------------------- }

begin // Wait for event Ts
  Result := WAIT_FAILED;
  if uHeader.HEvent <> 0 then
    Result := WaitForSingleObject(uHeader.HEvent, Time);
end;


initialization
  InitializeCriticalSection(CSTsList);
end.
