
{*******************************************************}
{                                                       }
{       Borland Delphi Visual Component Library         }
{       Screen Saver                                    }
{                                                       }
{       Copyright (c) 2000 Vino Rodrigues               }
{       vinorodrigues@yahoo.com                         }
{                                                       }
{       Version History:                                }
{       ----------------                                }
{                                                       }
{       1.3  30-Aug-01                                  }
{            First time I included "version history".   }
{                                                       }
{*******************************************************}

unit ScrSav;

{$UNDEF DEBUG}
{.$DEFINE DEBUG}

{$IFDEF DEBUG}
  {$D+} // DEBUG
  {$C+}	// ASSERTIONS
  {$O-}	// OPTIMIZATION
  {$Q+}	// OVERFLOWCHECKS
  {$R+}	// RANGECHECKS
  {$Y+} // SYMBOL AND X-REF INFO
  {$L+} // LOCAL SYMBOL INFO
{$ELSE}
  {$D-} // DEBUG
  {$C-}	// ASSERTIONS
  {$Q-} // OVERFLOWCHECKS
  {$R-} // RANGECHECKS
  {$Y-} // SYMBOL AND X-REF INFO
  {$L-} // LOCAL SYMBOL INFO
{$ENDIF}

{$P+,S-,W-,R-,T-,H+,X+}
{$C PRELOAD}

interface

uses
  Classes, SysUtils, Forms, Dialogs, Windows, Messages, Graphics, Controls,
  Registry;

const  // windows api default's
{ TODO -oVino :
Check WinAPI for actual values of PBT_*
Values here are assumed... }
  PBT_APMQUERYSUSPEND = 0;
  PBT_APMSUSPEND =      4;

const  // defaults
  DEF_PASSWORD_DELAY =  2;
  DEF_IGNORE_MOUSE =    2;
  DEF_MOUSE_THRESHOLD = 5;

type
  { TScreenSaver }

  TCloseOnEvent = (coActivation, coMouseMove, coMouseDown, coKeyDown, coPowerSuspend, coMonitorSuspend);
  TCloseOn = set of TCloseOnEvent;
  TChangePasswordEvent = procedure(Sender: TObject; var CallChangePw: Boolean) of object;
  TPasswordQueryEvent = procedure(Sender: TObject; var CallVerifyPw, CanClose: Boolean) of object;

  TScreenSaver = class(TDataModule)
  private
    MousePos: TPoint;
    StartTime: TDateTime;

    FCanvas: TCanvas;
    FClientRect: TRect;
    FCloseOn: TCloseOn;
    FDesktopBitmap: TBitmap;
    FIsDialogActive: Boolean;
    FHandle: HWND;
    FIgnoreMouse: Integer;
    FMouseThreshold: Integer;
    FMuteSound: Boolean;
    FPasswordDelay: Cardinal;
    FSaveDesktop: Boolean;

    FOnActivate: TNotifyEvent;
    FOnChangePassword: TChangePasswordEvent;
    FOnConfigure: TNotifyEvent;
    FOnDeactivate: TNotifyEvent;
    FOnHelp: THelpEvent;
    FOnPaint: TNotifyEvent;
    FOnPasswordQuery: TPasswordQueryEvent;
    FDisallowSuspend: Boolean;

    function GetUpdateRect: TRect;
    procedure SetIsDialogActive(const Value: Boolean);

    function GetAbout: Longint;
    procedure SetAbout(const Value: Longint);
  protected
    function DoQueryPassword: Boolean;
    procedure DoPaint;
    procedure LoadDesktopBitmap;
    procedure LoadWindowsPlus;
    procedure SaveMousePos;
  public
    constructor CreateNew(AOwner: TComponent; Dummy: Integer = 0); override;
    destructor Destroy; override;

    function CanClose: Boolean;
    procedure Invalidate; virtual;
    procedure InvalidateRect(const Rect: TRect); virtual;
    procedure Paint; virtual;
    procedure Repaint; virtual;
    procedure Update; virtual;

    property Canvas: TCanvas read FCanvas;
    property ClientRect: TRect read FClientRect;
    property DesktopBitmap: TBitmap read FDesktopBitmap;
    property DisallowSuspend: Boolean read FDisallowSuspend write FDisallowSuspend default False;
    property IsDialogActive: Boolean read FIsDialogActive write SetIsDialogActive;
    property Handle: HWND read FHandle;
    property IgnoreMouse: Integer read FIgnoreMouse write FIgnoreMouse default DEF_IGNORE_MOUSE;
    property PasswordDelay: Cardinal read FPasswordDelay write FPasswordDelay default DEF_PASSWORD_DELAY;
    property MouseThreshold: Integer read FMouseThreshold write FMouseThreshold default DEF_MOUSE_THRESHOLD;
    property MuteSound: Boolean read FMuteSound write FMuteSound default False;
    property UpdateRect: TRect read GetUpdateRect;
  published
    property CloseOn: TCloseOn read FCloseOn write FCloseOn default [coActivation, coMouseMove, coMouseDown, coKeyDown, coPowerSuspend, coMonitorSuspend];
    property SaveDesktop: Boolean read FSaveDesktop write FSaveDesktop;

    property OnChangePassword: TChangePasswordEvent read FOnChangePassword write FOnChangePassword;
    property OnConfigure: TNotifyEvent read FOnConfigure write FOnConfigure;
    property OnHelp: THelpEvent read FOnHelp write FOnHelp;
    property OnPasswordQuery: TPasswordQueryEvent read FOnPasswordQuery write FOnPasswordQuery;
    property OnPaint: TNotifyEvent read FOnPaint write FOnPaint;
    property OnActivate: TNotifyEvent read FOnActivate write FOnActivate;
    property OnDeactivate: TNotifyEvent read FOnDeactivate write FOnDeactivate;

    property About: Longint read GetAbout write SetAbout stored False;
  end;

  { TScreenSaverApplication }

  TSaverMode = (ssUnknown, ssConfig, ssSaver, ssPreview, ssChangePasswd, ssHelp);

  TScreenSaverApplication = class(TComponent)
  private
    FComponentClass: TComponentClass;
    FIsLoading: Boolean;
    FMode: TSaverMode;
    FModule: TScreenSaver;
    FParamHandle: THandle;
    FOnException: TExceptionEvent;

    procedure OnExceptionHandler(Sender: TObject; E: Exception);
    function GetHelpFile: string;
    function GetOnHelp: THelpEvent;
    function GetTitle: string;
    procedure SetHelpFile(const Value: string);
    procedure SetOnHelp(const Value: THelpEvent);
    procedure SetTitle(const Value: string);
  protected
    procedure DoChangePassword;
    procedure DoGetMode(var Mode: TSaverMode; var ParamHandle: THandle); dynamic;
    procedure DoHandleException(E: Exception); dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    procedure CreateForm(InstanceClass: TComponentClass; var Reference); virtual;
    procedure Initialize; virtual;
    procedure Run; virtual;
    property Title: string read GetTitle write SetTitle;
    property HelpFile: string read GetHelpFile write SetHelpFile;
    property OnHelp: THelpEvent read GetOnHelp write SetOnHelp;

    procedure ChangePassword;
    procedure Configure;
    procedure Help;
    function HelpCommand(Command: Integer; Data: Longint): Boolean;
    function HelpContext(Context: THelpContext): Boolean;
    function HelpJump(const JumpID: string): Boolean;
    function IsSaverRunning: Boolean;
    procedure RunSaver(const InPreview: Boolean);
    property Mode: TSaverMode read FMode;

    property OnException: TExceptionEvent read FOnException write FOnException;
  end;

var
  Application: TScreenSaverApplication = nil;

implementation

const
  COMPONENT_VER_LOW =  1;
  COMPONENT_VER_HIGH = 0;

{ Resource }

resourcestring
  sOnlyOneClassInstance = 'Only one instance of TScreenSaver' +
    ' allowed in a TScreenSaverApplication';
  sCannotChangePassword = 'PwdChangeProc not found. Cannot change password';

{ TScreenSaver }

constructor TScreenSaver.CreateNew(AOwner: TComponent; Dummy: Integer);
begin
  inherited CreateNew(AOwner, Dummy);
  Application.FModule := Self;
  FPasswordDelay := DEF_PASSWORD_DELAY;
  FIgnoreMouse := DEF_IGNORE_MOUSE;
  FMouseThreshold := DEF_MOUSE_THRESHOLD;
  FMuteSound := False;
  FCloseOn := [coActivation, coMouseMove, coMouseDown, coKeyDown,
    coPowerSuspend, coMonitorSuspend];
  FIsDialogActive := False;
end;

destructor TScreenSaver.Destroy;
begin
  if Assigned(FDesktopBitmap) then FDesktopBitmap.Free;
  inherited Destroy;
end;

function TScreenSaver.CanClose: Boolean;
const
  sREGSTR_DESKTOP = 'Control Panel\Desktop';
  sUSE_PASSWORD = 'ScreenSaveUsePassword';
var
  dt: TDateTime;
  D, H, N, S, Z: Word;
  el: Cardinal;
  reg: TRegistry;
  pw: Boolean;
  CallVerifyPw: Boolean;
begin
  Result := True;

  if Win32Platform <> VER_PLATFORM_WIN32_NT then
  begin
    if StartTime <> 0 then
    begin
      dt := Now - StartTime;
      try
        D := Trunc(Int(dt)); DecodeTime(dt, H, N, S, Z);
        el := S + ((N + ((H + (D * 24)) * 60)) * 60);
      except
        el := FPasswordDelay + 1;
      end;
      if el <= FPasswordDelay then Exit;
    end;

    pw := False;
    reg := TRegistry.Create;
    try
      reg.RootKey := HKEY_CURRENT_USER;
      try
        if reg.OpenKey(sREGSTR_DESKTOP, False) then
          try pw := reg.ReadInteger(sUSE_PASSWORD) <> 0 except end;
      except end;
    finally
      reg.CloseKey;
      reg.Free;
    end;
    if not pw then Exit;

    FIsDialogActive := True;
    {$IFNDEF DEBUG}
    ShowCursor(True);
    {$ENDIF}
    if Assigned(FOnPasswordQuery) then
    begin
      CallVerifyPw := False;
      FOnPasswordQuery(Self, CallVerifyPw, Result);
    end
    else
      CallVerifyPw := True;
    if CallVerifyPw then
      Result := DoQueryPassword;
    SaveMousePos;
    {$IFNDEF DEBUG}
    ShowCursor(False);
    {$ENDIF}
    FIsDialogActive := False;
  end;
end;

procedure TScreenSaver.Invalidate;
begin
  InvalidateRect(FClientRect);
end;

procedure TScreenSaver.InvalidateRect(const Rect: TRect);
begin
  if FHandle <> 0 then
    Windows.InvalidateRect(FHandle, @Rect, True);
end;

procedure TScreenSaver.Paint;
begin
  if Assigned(FOnPaint) then
    FOnPaint(Self)
  else
    DoPaint;
end;

procedure TScreenSaver.Repaint;
begin
  Invalidate;
  Update;
end;

procedure TScreenSaver.Update;
begin
  if FHandle <> 0 then UpdateWindow(FHandle);
end;

function TScreenSaver.DoQueryPassword: Boolean;
type  TPasswdFunc = function (Parent: THandle): Boolean; stdcall;
var
  SysDir: string;
  i: Integer;
  PwdModule: THandle;
  PwdFunc: TPasswdFunc;
begin
  Result := True;
  if Win32Platform = VER_PLATFORM_WIN32_NT then Exit;

  SetLength(SysDir, MAX_PATH);
  i := GetSystemDirectory(PChar(SysDir), MAX_PATH);
  SetLength(SysDir, i);
  if (Length(SysDir) > 0) and (SysDir[Length(SysDir)] <> '\') then
    SysDir := SysDir + '\';

  PwdModule := LoadLibrary(PChar(SysDir + 'PASSWORD.CPL'));
  if PwdModule <> 0 then
  begin
    PwdFunc := GetProcAddress(PwdModule, 'VerifyScreenSavePwd');
    if Assigned(PwdFunc) then
      Result := PwdFunc(FHandle);
    FreeLibrary(PwdModule);
  end;
end;

procedure TScreenSaver.DoPaint;
begin
  if Canvas <> nil then
    with Canvas do
      if Assigned(FDesktopBitmap) then
        CopyRect(UpdateRect, FDesktopBitmap.Canvas, UpdateRect)
      else
      begin
        Canvas.Brush.Color := clBlack;
        FillRect(UpdateRect);
      end;
end;

procedure TScreenSaver.LoadDesktopBitmap;
begin
  if Assigned(FDesktopBitmap) then FDesktopBitmap.Free;

  FDesktopBitmap := TBitmap.Create;
  with FDesktopBitmap do
  begin
    Width := Screen.Width;
    Height := Screen.Height;
  end;
  BitBlt(FDesktopBitmap.Canvas.Handle, 0, 0, Screen.Width, Screen.Height,
    GetDC(GetDesktopWindow), 0, 0, SrcCopy);
end;

procedure TScreenSaver.LoadWindowsPlus;
const
  sREGSTR_SCREENSAVERS =
    'Software\Microsoft\Windows\CurrentVersion\Screen Savers';
  sPASSWORD_DELAY = 'Password Delay';
  sMOUSE_THRESHOLD = 'Mouse Threshold';
  sMUTE_SOUND = 'Mute Sound';
var  reg: TRegistry;
begin
  reg := TRegistry.Create;
  try
    reg.RootKey := HKEY_CURRENT_USER;
    try
      if reg.OpenKey(sREGSTR_SCREENSAVERS, False) then
      begin
        try FPasswordDelay := reg.ReadInteger(sPASSWORD_DELAY); except end;
        try FMouseThreshold := reg.ReadInteger(sMOUSE_THRESHOLD); except end;
        try FMuteSound := reg.ReadBool(sMUTE_SOUND); except end;
      end;
    except end;
  finally
    reg.CloseKey;
    reg.Free;
  end;
end;

procedure TScreenSaver.SaveMousePos;
begin
  GetCursorPos(MousePos);
  ScreenToClient(FHandle, MousePos);
end;

function TScreenSaver.GetUpdateRect: TRect;
begin
  if FHandle <> 0 then
    Windows.GetUpdateRect(FHandle, Result, False)
  else
    Result := ClientRect;
end;

procedure TScreenSaver.SetIsDialogActive(const Value: Boolean);
begin
  FIsDialogActive := Value;
  if not Value then SaveMousePos;
end;

function TScreenSaver.GetAbout: Longint;
begin
  Result := MakeLong(COMPONENT_VER_LOW, COMPONENT_VER_HIGH);
end;

procedure TScreenSaver.SetAbout(const Value: Integer);
begin
end;

{ TScreenSaverApplication }

constructor TScreenSaverApplication.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FIsLoading := True;
end;

procedure TScreenSaverApplication.CreateForm(InstanceClass: TComponentClass;
  var Reference);
begin
  if InstanceClass.InheritsFrom(TScreenSaver) then
  begin
    if FComponentClass <> nil then
      raise Exception.CreateRes(@sOnlyOneClassInstance);
    if (FComponentClass = nil) or (FComponentClass <> InstanceClass) then
      FComponentClass := TComponentClass(InstanceClass);
    TComponent(Reference) := FComponentClass.Create(Self);
  end
  else
    Forms.Application.CreateForm(InstanceClass, Reference);
end;

procedure TScreenSaverApplication.Initialize;
begin
  DoGetMode(FMode, FParamHandle);
  Forms.Application.Initialize;
  Forms.Application.ShowMainForm := False;
end;

procedure TScreenSaverApplication.Run;
begin
  (* if FMode = ssUnknown then Exit; *)

  if not Forms.Application.Terminated then
  begin
    Forms.Application.OnException := OnExceptionHandler;

    case FMode of
      ssSaver: if not IsSaverRunning then RunSaver(False);
      ssPreview: if not IsSaverRunning then RunSaver(True);
      ssConfig, ssUnknown: Configure;
      ssChangePasswd: ChangePassword;
      ssHelp: Help;
    end;

    Forms.Application.OnException := nil;
  end;
end;

procedure TScreenSaverApplication.ChangePassword;
var  CallChangePw: Boolean;
begin
  if Win32Platform <> VER_PLATFORM_WIN32_NT then
  begin
    if FModule <> nil then
    begin
      if Assigned(FModule.OnChangePassword) then
      begin
        CallChangePw := False;
        FModule.OnChangePassword(Self, CallChangePw);
      end
      else
        CallChangePw := True;
      if CallChangePw then DoChangePassword;
    end
    else
      DoChangePassword;
  end;
end;

procedure TScreenSaverApplication.Configure;
begin
  if FModule <> nil then
    if Assigned(FModule.OnConfigure) then
    begin
      FModule.OnConfigure(Self);
      Exit;
    end;

  Forms.Application.ShowMainForm := True;
  Forms.Application.Run;
end;

procedure TScreenSaverApplication.Help;
var  CallHelp: Boolean;
begin
  if FModule <> nil then
    if Assigned(FModule.OnHelp) then
    begin
      CallHelp := True;
      FModule.OnHelp(HELP_FINDER, 0, CallHelp);
      if not CallHelp then Exit;
    end;

  if not HelpCommand(HELP_FINDER, 0) then
    Configure;
end;

function TScreenSaverApplication.HelpCommand(Command,
  Data: Integer): Boolean;
begin
  Result := Forms.Application.HelpCommand(Command, Data);
end;

function TScreenSaverApplication.HelpContext(
  Context: THelpContext): Boolean;
begin
  Result := Forms.Application.HelpContext(Context);
end;

function TScreenSaverApplication.HelpJump(const JumpID: string): Boolean;
begin
  Result := Forms.Application.HelpJump(JumpID);
end;

function TScreenSaverApplication.IsSaverRunning: Boolean;
var
  Running, Dummy: Boolean;
  hfw: HWND;
  wl: Longint;
  rc: TRect;
begin
  Result := False;

  if Win32Platform <> VER_PLATFORM_WIN32_NT then
  begin
    SystemParametersInfo(SPI_SCREENSAVERRUNNING, Ord(False), @Running, 0);
    if Running then
    begin
      SystemParametersInfo(SPI_SCREENSAVERRUNNING, Ord(True), @Dummy, 0);
      Result := True;
    end;
  end
  else
  begin
    hfw := GetForegroundWindow;
    if hfw = 0 then
      Result := True
    else
    begin
      wl := GetWindowLong(hfw, GWL_STYLE);
      if ((wl and $F0000000) <> (WS_POPUP or WS_VISIBLE)) then Exit;
      GetWindowRect(hfw, rc);
      if ((rc.Right - rc.Left) <> GetSystemMetrics(SM_CXSCREEN)) or
        ((rc.Bottom - rc.Top) <> GetSystemMetrics(SM_CYSCREEN)) then
        Exit;
      Result := True;
    end;
  end;
end;

function PrevWndProc(Wnd: HWnd; Msg: Integer; wParam: Word;
  lParam: Integer): Integer; far; stdcall;
begin
  case Msg of
    WM_CLOSE: PostMessage(Wnd, WM_QUIT, 0, 0);

    WM_PAINT:
      if (Application <> nil) and (Application.FModule <> nil) then
        Application.FModule.Paint;
  end;

  Result := DefWindowProc(Wnd, Msg, wParam, lParam);
end;

function SaverWndProc(Wnd: HWnd; Msg: Integer; wParam: Word;
  lParam: Integer): Integer; far; stdcall;
var  MMsg: TWMMouseMove absolute Msg;
begin
  case Msg of
    WM_CLOSE:
      if (Application <> nil) and (Application.FModule <> nil) then
      begin
        with Application.FModule do
        begin
          if FIsDialogActive then
          begin
            Result := 0;
            Exit;
          end;
          if CanClose then
          begin
            if Assigned(FOnDeactivate) then
              FOnDeactivate(Application.FModule);
            PostMessage(Wnd, WM_QUIT, 0, 0);
          end
          else
          begin
            Result := 0;
            Exit;
          end;
        end;
      end
      else
        PostMessage(Wnd, WM_QUIT, 0, 0);

    WM_ACTIVATE:
      if (Application <> nil) and (Application.FModule <> nil) then
        if Application.FIsLoading then
        begin
          Application.FIsLoading := False;
          with Application.FModule do
          begin
            SaveMousePos;
            StartTime := Now;
            if Assigned(FOnActivate) then FOnActivate(Application.FModule);
          end;
        end
        else
          if Application.FModule.FIsDialogActive then
          begin
            Result := 0;
            Exit;
          end
          else
            if coActivation in Application.FModule.FCloseOn then
              PostMessage(Wnd, WM_CLOSE, 0, 0);

    WM_ACTIVATEAPP:
      if (Application <> nil) and (Application.FModule <> nil) then
        if not Application.FIsLoading then
          if Application.FModule.FIsDialogActive then
          begin
            Result := 0;
            Exit;
          end
          else
            if coActivation in Application.FModule.FCloseOn then
              PostMessage(Wnd, WM_CLOSE, 0, 0);

    WM_KEYDOWN,
    WM_SYSKEYDOWN:
      if (Application <> nil) and (Application.FModule <> nil) then
        if Application.FModule.FIsDialogActive then
        begin
          Result := 0;
          Exit;
        end
        else
          if coKeyDown in Application.FModule.FCloseOn then
            PostMessage(Wnd, WM_CLOSE, 0, 0);

    WM_LBUTTONDOWN,
    WM_MBUTTONDOWN,
    WM_RBUTTONDOWN:
      if (Application <> nil) and (Application.FModule <> nil) then
        if Application.FModule.FIsDialogActive then
        begin
          Result := 0;
          Exit;
        end
        else
          if coMouseDown in Application.FModule.FCloseOn then
            PostMessage(Wnd, WM_CLOSE, 0, 0);

    WM_MOUSEMOVE:
      if (Application <> nil) and (Application.FModule <> nil) then
        if Application.FModule.FIsDialogActive then
        begin
          Result := 0;
          Exit;
        end
        else
          if coMouseMove in Application.FModule.FCloseOn then
            if Application.FModule.FIgnoreMouse > 0 then
              Dec(Application.FModule.FIgnoreMouse)
            else
              if ( (Abs(Application.FModule.MousePos.x - MMsg.XPos) +
                Abs(Application.FModule.MousePos.y - MMsg.YPos)) >
                Application.FModule.FMouseThreshold ) then
                PostMessage(Wnd, WM_CLOSE, 0, 0);

    WM_SYSCOMMAND:
      begin
        if (wParam = SC_SCREENSAVE) then
        begin
          Result := 0;
          Exit;
        end;
        if (wParam = SC_MONITORPOWER) then
          if (Application <> nil) and (Application.FModule <> nil) then
            if not Application.FModule.FIsDialogActive then
              if (coMonitorSuspend in Application.FModule.FCloseOn) then
                PostMessage(Wnd, WM_CLOSE, 0, 0);
      end;

    WM_PAINT:
      if (Application <> nil) and (Application.FModule <> nil) then
        Application.FModule.Paint;

    WM_POWERBROADCAST:
      if (Application <> nil) and (Application.FModule <> nil) then
        if not Application.FModule.FIsDialogActive then
        begin
          if (wParam = PBT_APMQUERYSUSPEND) then
            if (lParam and $01 <> 0) and
              Application.FModule.FDisallowSuspend then
            begin
              Result := BROADCAST_QUERY_DENY;
              Exit;
            end;
          if (wParam = PBT_APMSUSPEND) and
            (coPowerSuspend in Application.FModule.FCloseOn) then
            PostMessage(Wnd, WM_CLOSE, 0, 0);
        end;
  end;

  Result := DefWindowProc(Wnd, Msg, wParam, lParam);
end;

procedure TScreenSaverApplication.RunSaver(const InPreview: Boolean);
var
  WndClass: TWndClass;
  exstyle, style: Cardinal;
  {$IFNDEF DEBUG}
  Dummy: Boolean;
  {$ENDIF}
begin
  if FModule = nil then
    Exit;
  if InPreview then
    if not IsWindow(FParamHandle) then
      Exit;

  if InPreview then
    while not IsWindowVisible(FParamHandle) do
      Forms.Application.ProcessMessages;

  with WndClass do
  begin
    if InPreview then
      style := CS_PARENTDC
    else
      style := CS_HREDRAW or CS_VREDRAW;
    if InPreview then
      lpfnWndProc := @PrevWndProc
    else
      lpfnWndProc := @SaverWndProc;
    cbClsExtra := 0;
    cbWndExtra := 0;
    hInstance := SysInit.hInstance;
    hIcon := 0;
    if InPreview then
      hCursor := LoadCursor(0, IDC_ARROW)
    else
      {$IFDEF DEBUG}
      hCursor := LoadCursor(0, IDC_APPSTARTING);
      {$ELSE}
      hCursor := LoadCursor(0, IDC_WAIT);
      {$ENDIF}
    hbrBackground := HBRUSH(GetStockObject(BLACK_BRUSH));
    lpszMenuName := nil;
    lpszClassName := 'ScrnSaveClass';
  end;
  Windows.RegisterClass(WndClass);

  with FModule do
  begin
    if FSaveDesktop then LoadDesktopBitmap;
    if not InPreview then LoadWindowsPlus;

    if InPreview then
    begin
      GetWindowRect(FParamHandle, FClientRect);
      FClientRect := Rect(0, 0,
        FClientRect.Right - FClientRect.Left,
        FClientRect.Bottom - FClientRect.Top);

      exstyle := 0;
      style := WS_CHILD or WS_VISIBLE or WS_DISABLED;
    end
    else
    begin
      FClientRect := Rect(0, 0, GetSystemMetrics(SM_CXSCREEN),
        GetSystemMetrics(SM_CYSCREEN));

      {$IFDEF DEBUG}
      exstyle := 0;
      {$ELSE}
      exstyle := WS_EX_TOPMOST;
      {$ENDIF}
      style := WS_POPUP or WS_VISIBLE;
    end;
    FHandle := CreateWindowEx(exstyle, WndClass.lpszClassName, nil,
      style, 0, 0, FClientRect.Right, FClientRect.Bottom, FParamHandle,
      0, hInstance, nil);

    FCanvas := TCanvas.Create;
    FCanvas.Handle := GetDC(FHandle);

    if InPreview then
    begin
      FIsLoading := False;
      if Assigned(FOnActivate) then
        FOnActivate(FModule);
    end
    else
    begin
      {$IFNDEF DEBUG}
      SetWindowPos(FHandle, HWND_TOPMOST, 0, 0, 0, 0,
        SWP_NOSIZE or SWP_NOMOVE);
      SystemParametersInfo(SPI_SCREENSAVERRUNNING, Ord(True), @Dummy, 0);
      ShowCursor(False);
      SetCapture(FHandle);
      {$ENDIF}
    end;

    repeat
      Forms.Application.ProcessMessages;
      if InPreview then
        if not IsWindowVisible(FParamHandle) then
          Forms.Application.Terminate;
    until Forms.Application.Terminated;

    if InPreview then
    begin
      if Assigned(FOnDeactivate) then
        FOnDeactivate(FModule);
    end
    else
    begin
      {$IFNDEF DEBUG}
      SystemParametersInfo(SPI_SCREENSAVERRUNNING, Ord(False), @Dummy, 0);
      ReleaseCapture;
      ShowCursor(True);
      {$ENDIF}
    end;

    FClientRect := Rect(0, 0, 0, 0);
    FCanvas.Free;  FCanvas := nil;
    DestroyWindow(FHandle);  FHandle := 0;
    Windows.UnregisterClass(WndClass.lpszClassName, hInstance);
  end;
end;

procedure TScreenSaverApplication.DoChangePassword;
type
  TPasswdFunc = function (lpcRegkeyname: PChar; hwnd: THandle;
    uiReserved1, uiReserved2: Integer): Integer; stdcall;
var
  SysDir: string;
  i: Integer;
  PwdModule: THandle;
  PwdFunc: TPasswdFunc;
begin
  SetLength(SysDir, MAX_PATH);
  i := GetSystemDirectory(PChar(SysDir), MAX_PATH);
  SetLength(SysDir, i);
  if (Length(SysDir) > 0) and (SysDir[Length(SysDir)] <> '\') then
    SysDir := SysDir + '\';

  i := 0;
  PwdModule := LoadLibrary(PChar(SysDir + 'MPR.DLL'));
  if PwdModule <> 0 then begin
    PwdFunc := GetProcAddress(PwdModule, 'PwdChangePasswordA');
    if Assigned(PwdFunc) then
      PwdFunc('SCRSAVE', Application.FParamHandle, 0, 0)
    else
      i := -1;
    FreeLibrary(PwdModule);
  end
  else
    i := -1;

  if i <> 0 then raise Exception.CreateRes(@sCannotChangePassword);
end;

procedure TScreenSaverApplication.DoGetMode(var Mode: TSaverMode;
  var ParamHandle: THandle);
var
  S: string;
  i: Integer;
begin
  Mode := ssUnknown;
  ParamHandle := 0;

  if ParamCount > 0 then
  begin
    S := UpperCase(ParamStr(1));
    if (S[1] = '/') or (S[1] = '-') then S := Copy(S, 2, MAXINT);

    i := Pos(':', S);
    if i > 0 then
    begin
      try
        ParamHandle := StrToInt(Copy(S, i+1, MAXINT));
      except end;
      S := Copy(S, 1, i-1);
    end;

    if S = 'S' then
      Mode := ssSaver
    else
      if (S = 'P') or (S = 'L') then
        Mode := ssPreview
      else
        if S = 'C' then
          Mode := ssConfig
        else
          if S = 'A' then
            Mode := ssChangePasswd
          else
            if (S = 'H') or (S = '?') or (S = 'HELP') then
              Mode := ssHelp;

    if ParamCount > 1 then
    begin
      try
        ParamHandle := StrToInt(ParamStr(2));
      except end;
    end;
  end;
end;

procedure TScreenSaverApplication.DoHandleException(E: Exception);
begin
  Forms.Application.ShowException(E);
end;

procedure TScreenSaverApplication.OnExceptionHandler(Sender: TObject;
  E: Exception);
begin
  if Assigned(FOnException) then
    FOnException(Sender, E)
  else
    DoHandleException(E);
end;

function TScreenSaverApplication.GetHelpFile: string;
begin
  Result := Forms.Application.HelpFile;
end;

function TScreenSaverApplication.GetOnHelp: THelpEvent;
begin
  Result := Forms.Application.OnHelp;
end;

function TScreenSaverApplication.GetTitle: string;
begin
  Result := Forms.Application.Title;
end;

procedure TScreenSaverApplication.SetHelpFile(const Value: string);
begin
  Forms.Application.HelpFile := Value;
end;

procedure TScreenSaverApplication.SetOnHelp(const Value: THelpEvent);
begin
  Forms.Application.OnHelp := Value;
end;

procedure TScreenSaverApplication.SetTitle(const Value: string);
begin
  Forms.Application.Title := Value;
end;

{ Initialization & Finalization }

procedure InitApplication;
begin
  Application := TScreenSaverApplication.Create(nil);
end;

procedure DoneApplication;
begin
  Application.Free;
  Application := nil;
end;

initialization
  InitApplication;

finalization
  DoneApplication;

end.
