unit ReportField2;

{
Author:          Juergen Ruemmler
EMail:           juergen@ruemmler-net.de
Version:         2.0 from 22.08.2000
Legal issues:    Copyright (C) 1999-2000 by Juergen Ruemmler
                 Brockdorffstr. 27a, 22149 Hamburg, Germany

                 TReportField is Freeware.

Description:

   TReportField provide different Borderstyles and can build
   each character in a separate frame. Additional you can get
   and put the Styleproperties of TReportField to save on disk.

   For the different ways please have a lock at the demo.


History:

  Version    Date         Comment
  -------    ---------    -------
  1.0        11.11.1999   Initialversion

  1.1        12.11.1999   Position and Size added to TReportFieldVar,
                          NotifyEvent OnStyleChange added

  1.2        10.05.2000   Property DividerOutline added,
                          procedure Print added to print on any Canvas

  1.3        17.08.2000   Bug in Print-Procedure killed

  2.0        22.08.2000   BorderStyle-Properties individual for left,right,top,bottom
                          Styleproperties-Structure is inkompatible with Version < 2
                          Procedures SetStyle and GetStyle removed and Procedures
                             SaveToStream and ReadFromStream added

Additional:

  Ill be happy, if you write me an email, if you like this component.
  If you use this component to create an new one, please sent it me and
  Im happy too.
  If you find a bug please send me a describtion.

  Good luck and have a nice day!
}

interface

Uses StdCtrls, Classes, Controls, Windows, Graphics, Dialogs, SysUtils;

Type
  TCrossOutStyle = (coNone, coSimple, coDouble);
  TDividerHeight = (dhNone, dhHalf, dhFull);

  TBorderTB = class(TPersistent)
  private
    FWidth: Byte;
    FStyle: TPenStyle;
    FOnChange: TNotifyEvent;
    procedure SetWidth(Value: Byte);
    procedure SetStyle(Value: TPenStyle);
  published
    property Width: Byte Read FWidth Write SetWidth;
    property Style: TPenStyle Read FStyle Write SetStyle;
    property OnChange: TNotifyEvent Read FOnChange Write FOnChange;
  end;

  TBorderLR = class(TPersistent)
  private
    FWidth: Byte;
    FStyle: TPenStyle;
    FHeight: TDividerHeight;
    FOnChange: TNotifyEvent;
    procedure SetWidth(Value: Byte);
    procedure SetStyle(Value: TPenStyle);
    procedure SetHeight(Value: TDividerHeight);
  published
    property Width: Byte Read FWidth Write SetWidth;
    property Style: TPenStyle Read FStyle Write SetStyle;
    property Height: TDividerHeight Read FHeight Write SetHeight;
    property OnChange: TNotifyEvent Read FOnChange Write FOnChange;
  end;

  TBorder = class(TPersistent)
  private
    FTop,
    FBottom: TBorderTB;
    FLeft,
    FRight: TBorderLR;
    FOnChange: TNotifyEvent;
    procedure BorderChange(Sender: TObject);
  public
    constructor create;
    destructor destroy; override;
  published
    property Top: TBorderTB Read FTop Write FTop;
    property Bottom: TBorderTB Read FBottom Write FBottom;
    property Left: TBorderLR Read FLeft Write FLeft;
    property Right: TBorderLR Read FRight Write FRight;
    property OnChange: TNotifyEvent Read FOnChange Write FOnChange;
  end;

  TCrossOut = class(TPersistent)
  private
    FStyle: TCrossOutStyle;
    FWidth: Byte;
    FOnChange: TNotifyEvent;
    procedure SetStyle(Value: TCrossOutStyle);
    procedure SetWidth(Value: Byte);
  published
    property Style: TCrossOutStyle Read FStyle Write SetStyle;
    property Width: Byte Read FWidth Write SetWidth;
    property OnChange: TNotifyEvent Read FOnChange Write FOnChange;
  end;

  TDivider = class(TPersistent)
  private
    FStyle: TPenStyle;
    FWidth: Byte;
    FHeight: TDividerHeight;
    FOnChange: TNotifyEvent;
    procedure SetHeight(Value: TDividerHeight);
    procedure SetStyle(Value: TPenStyle);
    procedure SetWidth(Value: Byte);
  published
    property Height: TDividerHeight Read FHeight Write SetHeight;
    property Style: TPenStyle Read FStyle Write SetStyle;
    property Width: Byte Read FWidth Write SetWidth;
    property OnChange: TNotifyEvent Read FOnChange Write FOnChange;
  end;

  TReportField2 = class(TCustomLabel)
  private
    FAlignment: TAlignment;
    FMaxLength: Integer;
    FFixLength: Boolean;
    FColor: TColor;
    FBorder: TBorder;
    FCrossOut: TCrossOut;
    FDivider: TDivider;
    FOnChange: TNotifyEvent;
    loading: Boolean;
    function GetText: TCaption;
    function GetTransparent: Boolean;
    procedure SetText(const Value: TCaption);
    procedure SetAlignment(Value: TAlignment);
    procedure SetColor(const Value: TColor);
    procedure SetFixLength(const Value: Boolean);
    procedure SetMaxLength(const Value: Integer);
    procedure SetTransparent(const Value: Boolean);
    procedure ClientChange(Sender: TObject);
  protected
    procedure Paint; override;
  public
    procedure Print(Can: TCanvas; Zoom: Double; RTop, RLeft: Integer);
    procedure SaveToStream(Stream: TStream);
    procedure ReadFromStream(Stream: TStream);
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Text: TCaption read GetText write SetText;
    property Border: TBorder Read FBorder Write FBorder;
    property CrossOut: TCrossOut Read FCrossOut Write FCrossOut;
    property Divider: TDivider Read FDivider Write FDivider;
    property MaxLength: Integer Read FMaxLength Write SetMaxLength;
    property FixLength: Boolean Read FFixLength Write SetFixLength;
    property Font;
    property Alignment: TAlignment Read FAlignment Write SetAlignment;
    property BackColor: TColor Read FColor Write SetColor;
    property Enabled;
    property PopupMenu;
    property ShowHint;
    property Transparent: Boolean Read GetTransparent Write SetTransparent;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStyleChange: TNotifyEvent Read FOnChange Write FOnChange;
  end;

procedure Register;

implementation

{*.DCR}

procedure Register;
begin
  RegisterComponents('Extra', [TReportField2]);
end;

// *****************************************************************
//          T B o r d e r T B
// *****************************************************************

procedure TBorderTB.SetWidth(Value: Byte);
begin
  if Value <> FWidth then begin
    FWidth:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

procedure TBorderTB.SetStyle(Value: TPenStyle);
begin
  if Value <> FStyle then begin
    FStyle:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

// *****************************************************************
//          T B o r d e r L R
// *****************************************************************

procedure TBorderLR.SetWidth(Value: Byte);
begin
  if Value <> FWidth then begin
    FWidth:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

procedure TBorderLR.SetStyle(Value: TPenStyle);
begin
  if Value <> FStyle then begin
    FStyle:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

procedure TBorderLR.SetHeight(Value: TDividerHeight);
begin
  if Value <> FHeight then begin
    FHeight:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

// *****************************************************************
//          T B o r d e r
// *****************************************************************

constructor TBorder.Create;
begin
  inherited Create;
  FTop:=TBorderTB.Create;
  FTop.FWidth:=1;
  FTop.FStyle:=psSolid;
  FTop.OnChange:=BorderChange;
  FBottom:=TBorderTB.Create;
  FBottom.FWidth:=1;
  FBottom.FStyle:=psSolid;
  FBottom.OnChange:=BorderChange;
  FLeft:=TBorderLR.Create;
  FLeft.FWidth:=1;
  FLeft.FStyle:=psSolid;
  FLeft.FHeight:=dhFull;
  FLeft.OnChange:=BorderChange;
  FRight:=TBorderLR.Create;
  FRight.FWidth:=1;
  FRight.FStyle:=psSolid;
  FRight.FHeight:=dhFull;
  FRight.OnChange:=BorderChange;
end;

destructor TBorder.Destroy;
begin
  FTop.Free;
  FBottom.Free;
  FLeft.Free;
  FRight.Free;
  inherited Destroy;
end;

procedure TBorder.BorderChange(Sender: TObject);
begin
  if Assigned(OnChange) then OnChange(Self);
end;

// *****************************************************************
//          T C r o s s O u t
// *****************************************************************

procedure TCrossOut.SetStyle(Value: TCrossOutStyle);
begin
  if Value <> FStyle then begin
    FStyle:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

procedure TCrossOut.SetWidth(Value: Byte);
begin
  if Value <> FWidth then begin
    FWidth:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

// *****************************************************************
//          T D i v i d e r
// *****************************************************************

procedure TDivider.SetHeight(Value: TDividerHeight);
begin
  if Value <> FHeight then begin
    FHeight:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

procedure TDivider.SetStyle(Value: TPenStyle);
begin
  if Value <> FStyle then begin
    FStyle:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

procedure TDivider.SetWidth(Value: Byte);
begin
  if Value <> FWidth then begin
    FWidth:=Value;
    if Assigned(OnChange) then OnChange(Self);
  end;
end;

// *****************************************************************
//          T R e p o r t F i e l d 2
// *****************************************************************

constructor TReportField2.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  loading:=True;
  FMaxLength:=20;
  FFixLength:=True;
  FAlignment:=taLeftJustify;
  FBorder:=TBorder.Create;
  FBorder.OnChange:=ClientChange;
  FCrossOut:=TCrossOut.Create;
  FCrossOut.Width:=1;
  FCrossOut.FStyle:=coNone;
  FCrossOut.OnChange:=ClientChange;
  FDivider:=TDivider.Create;
  FDivider.FStyle:=psSolid;
  FDivider.FWidth:=1;
  FDivider.FHeight:=dhHalf;
  FDivider.OnChange:=ClientChange;
  AutoSize:=False;
  Layout:=tlCenter;
  ShowAccelChar:=False;
  Color:=clWhite;
  FColor:=clWhite;
  loading:=False;
end;

destructor TReportField2.Destroy;
begin
  FBorder.Free;
  FCrossOut.Free;
  FDivider.Free;
  inherited Destroy;
end;

procedure TReportField2.ClientChange(Sender: TObject);
begin
  if loading then exit;
  Paint;
  if Assigned(OnStyleChange) then OnStyleChange(Self);
end;

procedure TReportField2.SaveToStream(Stream: TStream);

  procedure WriteText(txt: String);
  var i: integer;
      b: Byte;
  begin
    for i:=1 to length(txt) do begin
      b:=ord(txt[i]);
      Stream.WriteBuffer(b, 1);
    end;
  end;

var h: string;
    i: integer;
    fs: TFontStyles;
    bo: boolean;
    al: TAlignment;
    co: TColor;
    b: Byte;
begin
  try
    // Self
    h:=Self.Name;
    while length(h) < 40 do h:=h + #32;
    WriteText(h);
    h:=Self.Text;
    while length(h) < 100 do h:=h + #32;
    WriteText(h);
    h:=Self.Hint;
    while length(h) < 80 do h:=h + #32;
    WriteText(h);
    // Self.Font
    h:=Self.Font.Name;
    while length(h) < 40 do h:=h + #32;
    WriteText(h);
    i:=Font.Size;                  Stream.WriteBuffer(i, SizeOf(i));
    fs:=Font.Style;                Stream.WriteBuffer(fs, SizeOf(fs));
    co:=Font.Color;                Stream.WriteBuffer(co, SizeOf(co));
    i:=Top;                        Stream.WriteBuffer(i, SizeOf(i));
    i:=Left;                       Stream.WriteBuffer(i, SizeOf(i));
    i:=Width;                      Stream.WriteBuffer(i, SizeOf(i));
    i:=Height;                     Stream.WriteBuffer(i, SizeOf(i));
    bo:=Transparent;               Stream.WriteBuffer(bo, SizeOf(bo));
    al:=Alignment;                 Stream.WriteBuffer(al, SizeOf(al));
    i:=MaxLength;                  Stream.WriteBuffer(i, SizeOf(i));
    bo:=FixLength;                 Stream.WriteBuffer(bo, SizeOf(bo));
    co:=BackColor;                 Stream.WriteBuffer(co, SizeOf(co));
    b:=Divider.Width;              Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Divider.Height);        Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Divider.Style);         Stream.WriteBuffer(b, SizeOf(b));
    b:=CrossOut.Width;             Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(CrossOut.Style);        Stream.WriteBuffer(b, SizeOf(b));
    b:=Border.Top.Width;           Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Border.Top.Style);      Stream.WriteBuffer(b, SizeOf(b));
    b:=Border.Bottom.Width;        Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Border.Bottom.Style);   Stream.WriteBuffer(b, SizeOf(b));
    b:=Border.Left.Width;          Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Border.Left.Style);     Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Border.Left.Height);    Stream.WriteBuffer(b, SizeOf(b));
    b:=Border.Right.Width;         Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Border.Right.Style);    Stream.WriteBuffer(b, SizeOf(b));
    b:=Ord(Border.Right.Height);   Stream.WriteBuffer(b, SizeOf(b));
  except
  end;
end;

procedure TReportField2.ReadFromStream(Stream: TStream);

  function ReadText(Len: Integer): String;
  var b: byte;
      h: String;
      i: integer;
  begin
    h:='';
    for i:=1 to Len do begin
      Stream.ReadBuffer(b, 1);
      h:=h + chr(b);
    end;
    Result:=h;
  end;

var h: string;
    i: integer;
    fs: TFontStyles;
    bo: boolean;
    al: TAlignment;
    co: TColor;
    b: Byte;
begin
  if (Stream.Size - Stream.Position) < 311 then exit;
  //
  try
    loading:=True;
    h:=ReadText(40);
    repeat
      if length(h) > 0 then
        if h[length(h)] = #32 then h:=copy(h, 1, length(h) - 1);
    until (length(h) = 0) or (h[length(h)] <> #32);
    try
      Name:=h;
    except
    end;
    h:=ReadText(100);
    repeat
      if length(h) > 0 then
        if h[length(h)] = #32 then h:=copy(h, 1, length(h) - 1);
    until (length(h) = 0) or (h[length(h)] <> #32);
    Caption:=h;
    h:=ReadText(80);
    repeat
      if length(h) > 0 then
        if h[length(h)] = #32 then h:=copy(h, 1, length(h) - 1);
    until (length(h) = 0) or (h[length(h)] <> #32);
    Hint:=h;
    h:=ReadText(40);
    repeat
      if length(h) > 0 then
        if h[length(h)] = #32 then h:=copy(h, 1, length(h) - 1);
    until (length(h) = 0) or (h[length(h)] <> #32);
    Font.Name:=h;
    Stream.ReadBuffer(i, SizeOf(i));    Font.Size:=i;
    Stream.ReadBuffer(fs, SizeOf(fs));  Font.Style:=fs;
    Stream.ReadBuffer(co, SizeOf(co));  Font.Color:=co;
    Stream.ReadBuffer(i, SizeOf(i));    Top:=i;
    Stream.ReadBuffer(i, SizeOf(i));    Left:=i;
    Stream.ReadBuffer(i, SizeOf(i));    Width:=i;
    Stream.ReadBuffer(i, SizeOf(i));    Height:=i;
    Stream.ReadBuffer(bo, SizeOf(bo));  Transparent:=bo;
    Stream.ReadBuffer(al, SizeOf(al));  FAlignment:=al;
    Stream.ReadBuffer(i, SizeOf(i));    FMaxLength:=i;
    Stream.ReadBuffer(bo, SizeOf(bo));  FFixLength:=bo;
    Stream.ReadBuffer(co, SizeOf(co));  FColor:=co; Color:=co;
    Stream.ReadBuffer(b, SizeOf(b));    Divider.Width:=b;
    Stream.ReadBuffer(b, SizeOf(b));    Divider.Height:=TDividerHeight(b);
    Stream.ReadBuffer(b, SizeOf(b));    Divider.Style:=TPenStyle(b);
    Stream.ReadBuffer(b, SizeOf(b));    CrossOut.Width:=b;
    Stream.ReadBuffer(b, SizeOf(b));    CrossOut.Style:=TCrossOutStyle(b);
    Stream.ReadBuffer(b, SizeOf(b));    Border.Top.Width:=b;
    Stream.ReadBuffer(b, SizeOf(b));    Border.Top.Style:=TPenStyle(b);
    Stream.ReadBuffer(b, SizeOf(b));    Border.Bottom.Width:=b;
    Stream.ReadBuffer(b, SizeOf(b));    Border.Bottom.Style:=TPenStyle(b);
    Stream.ReadBuffer(b, SizeOf(b));    Border.Left.Width:=b;
    Stream.ReadBuffer(b, SizeOf(b));    Border.Left.Style:=TPenStyle(b);
    Stream.ReadBuffer(b, SizeOf(b));    Border.Left.Height:=TDividerHeight(b);
    Stream.ReadBuffer(b, SizeOf(b));    Border.Right.Width:=b;
    Stream.ReadBuffer(b, SizeOf(b));    Border.Right.Style:=TPenStyle(b);
    Stream.ReadBuffer(b, SizeOf(b));    Border.Right.Height:=TDividerHeight(b);
    loading:=False;
    Paint;
    if Assigned(OnStyleChange) then OnStyleChange(Self);
  except
    loading:=False;
  end;
end;

function TReportField2.GetText: TCaption;
begin
  Result:=Caption;
end;

procedure TReportField2.SetText(const Value: TCaption);
begin
  if Caption <> Value then begin
    Caption:=Value;
    if not loading then begin
      Paint;
      if Assigned(OnStyleChange) then OnStyleChange(Self);
    end;
  end;
end;

procedure TReportField2.SetAlignment(Value: TAlignment);
begin
  if FAlignment <> Value then begin
    FAlignment:=Value;
    if not loading then begin
      Paint;
      if Assigned(OnStyleChange) then OnStyleChange(Self);
    end;
  end;
end;

function TReportField2.GetTransparent: Boolean;
begin
  Result := not (csOpaque in ControlStyle);
end;

procedure TReportField2.SetTransparent(const Value: Boolean);
begin
  if Transparent <> Value then begin
    if Value then
      ControlStyle:=ControlStyle - [csOpaque]
    else
      ControlStyle:=ControlStyle + [csOpaque];
    if not loading then begin
      Paint;
      if Assigned(OnStyleChange) then OnStyleChange(Self);
    end;
  end;
end;

procedure TReportField2.SetColor(const Value: TColor);
begin
  if FColor <> Value then begin
    FColor:=Value;
    Color:=FColor;
    if not loading then begin
      Paint;
      if Assigned(OnStyleChange) then OnStyleChange(Self);
    end;
  end;
end;

procedure TReportField2.SetFixLength(const Value: Boolean);
begin
  if Value and (FMaxLength < 1) then begin
    if csDesigning in ComponentState then
      MessageDlg('Invalid Value for FixLength, MaxLength must be > 0!',
                 mtInformation, [mbOK], 0);
    exit;
  end;
  if Value <> FFixLength then begin
    FFixLength:=Value;
    if not loading then begin
      Paint;
      if Assigned(OnStyleChange) then OnStyleChange(Self);
    end;
  end;
end;

procedure TReportField2.SetMaxLength(const Value: Integer);
begin
  if Value < 0 then exit;
  if FFixLength and (Value < 1) then begin
    if csDesigning in ComponentState then
      MessageDlg('Invalid Value for MaxLength, FixLength must be False!',
                 mtInformation, [mbOK], 0);
    exit;
  end;
  if Value < 0 then exit;
  if Value <> FMaxLength then begin
    FMaxLength:=Value;
    if not loading then begin
      Paint;
      if Assigned(OnStyleChange) then OnStyleChange(Self);
    end;
  end;
end;

procedure TReportField2.Print(Can: TCanvas; Zoom: Double; RTop, RLeft: Integer);
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
  WordWraps: array[Boolean] of Word = (0, DT_WORDBREAK);
var
  Rect: TRect;
  I, j, c1, c2, cw: Integer;
  zr: Double;
begin
  RLeft:=RLeft + Round(Left * Zoom);
  RTop:=RTop + Round(Top * Zoom);
  with Can do begin
    if not Transparent then begin
      Brush.Color:=Self.Color;
      Brush.Style:=bsSolid;
      Rect.Left:=RLeft;
      Rect.Top:=RTop;
      Rect.Right:=RLeft + Round(ClientWidth * Zoom);
      Rect.Bottom:=RTop + Round(ClientHeight * Zoom);
      Can.FillRect(Rect);
    end;
    Brush.Style:=bsClear;
    // now here we draw the lines
    // Border
    Pen.Width:=1;
    // Linker Rand
    Pen.Style:=Border.Left.Style;
    For i:=0 to Round(Border.Left.Width * Zoom) - 1 do begin
      MoveTo(RLeft + i, RTop + Round(ClientHeight * Zoom - 1));
      if Border.Left.Height = dhHalf then
        LineTo(RLeft + i, RTop + Round((ClientHeight div 2) * Zoom));
      if Border.Left.Height = dhFull then LineTo(RLeft + i, RTop);
    end;
    // Rechter Rand
    Pen.Style:=Border.Right.Style;
    for i:=0 to Round(Border.Right.Width * Zoom) - 1 do begin
      MoveTo(RLeft + Round(ClientWidth * Zoom - 1) - I,
             RTop + Round(ClientHeight * Zoom - 1));
      if Border.Right.Height = dhHalf then
        LineTo(RLeft + Round((ClientWidth) * Zoom - 1) - I,
               RTop + Round((ClientHeight div 2) * Zoom));
      if Border.Right.Height = dhFull then
        LineTo(RLeft + Round(ClientWidth * Zoom - 1) - I, RTop);
    end;
    // Oberer Rand
    Pen.Style:=Border.Top.Style;
    for i:=0 to Round(Border.Top.Width * Zoom) - 1 do begin
      MoveTo(RLeft + Round(ClientWidth * Zoom - 1), RTop + I);
      LineTo(RLeft, RTop + I);
    end;
    // Unterer Rand
    Pen.Style:=Border.Bottom.Style;
    for i:=0 to Round(Border.Bottom.Width * Zoom) - 1 do begin
      MoveTo(RLeft, RTop + Round(ClientHeight * Zoom - 1) - I);
      LineTo(RLeft + Round(ClientWidth * Zoom - 1),
             RTop + Round(ClientHeight * Zoom - 1) - I);
    end;
    // Divider
    c1:=1;
    c2:=FMaxLength - 1;
    Pen.Style:=Divider.Style;
    if FFixLength and (Divider.Style <> psClear) then begin
      zr:=(ClientWidth - Border.Left.Width - Border.Right.Width
                       - (Divider.Width * (FMaxLength - 1))) / FMaxLength;
      for I:=c1 to c2 do begin
        cw:=Round(zr * I + (Divider.Width * (I - 1)));
        if I > 0 then begin
          for j:= 0 to Round(Divider.Width * Zoom - 1) do begin
            MoveTo(RLeft + j + Round((cw + Border.Left.Width) * Zoom),
                   RTop + Round(ClientHeight * Zoom - 1));
            case Divider.Height of
              dhFull: LineTo(RLeft + j + Round((cw + Border.Left.Width) * Zoom),
                             RTop);
              dhHalf: LineTo(RLeft + j + Round((cw + Border.Left.Width) * Zoom),
                             RTop + Round(ClientHeight div 2 * Zoom));
            end;
          end;
        end;
      end;
    end;
    // CrossOut
    if CrossOut.Style In [coSimple, coDouble] then begin
      Pen.Style:=psSolid;
      cw:=Round(CrossOut.Width div 2 * Zoom);
      for i:=0 to Round(CrossOut.Width * Zoom - 1) do begin
        MoveTo(RLeft + i - cw, RTop + Round(ClientHeight * Zoom - 1));
        LineTo(RLeft + i - cw + Round(ClientWidth * Zoom - 1), RTop);
        if CrossOut.Style = coDouble then begin
          MoveTo(RLeft - cw + i, RTop);
          LineTo(RLeft - cw + i + Round(ClientWidth * Zoom - 1),
                 RTop + Round(ClientHeight * Zoom - 1));
        end;
      end;
    end;
    // Text
    can.Font.Name:=Self.Font.Name;
    can.Font.Size:=Self.Font.Size;
    can.Font.Style:=Self.Font.Style;
    if FFixLength and (Length(GetText) > 0) then
      begin
        cw:=Length(GetText);
        case FAlignment of
          taCenter: begin c1:=1 + (FMaxLength - cw) div 2; c2:=c1 + cw - 1; end;
          taRightJustify: begin c2:=FMaxLength; c1:=c2 - cw + 1; end;
        else
          begin c1:=1; c2:=cw; end;
        end;
        zr:=(ClientWidth - Border.Left.Width - Border.Right.Width
                         - (Divider.Width * (FMaxLength - 1))) / FMaxLength;
        for I:=c1 to c2 do begin
          Rect.Left:=RLeft +
               Round((Border.Left.Width + ((zr + Divider.Width) * (i - 1))) * Zoom) +
               Round(((zr * Zoom) - Can.TextWidth(GetText[i - c1 + 1])) / 2);
          Rect.Top:=RTop +
               ((Round((ClientHeight - Border.Top.Width - Border.Bottom.Width)
                 * Zoom) - Can.TextHeight(GetText)) div 2) +
               Round(Border.Top.Width * Zoom);
          TextOut(Rect.Left, Rect.Top, GetText[I - c1 + 1]);
        end;
      end
    else
      begin
        Rect.Top:=RTop +
             ((Round((ClientHeight - Border.Top.Width - Border.Bottom.Width)
               * Zoom) - Can.TextHeight(GetText)) div 2) +
             Round(Border.Top.Width * Zoom);
        case FAlignment of
          taCenter:
            Rect.Left:=RLeft +
               ((Round((ClientWidth - Border.Left.Width - Border.Right.Width) * Zoom)
                 - Can.TextWidth(GetText)) div 2) +
               Round(Border.Left.Width * Zoom);
          taRightJustify:
            Rect.Left:=RLeft + Round((ClientWidth - Border.Right.Width) * Zoom - 1)
                             - Can.TextWidth(GetText);
        else
          Rect.Left:=RLeft + Round(Border.Left.Width * Zoom);
        end;
        TextOut(Rect.Left, Rect.Top, GetText);
      end;
  end;
end;

procedure TReportField2.Paint;
const
  Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var
  Rect: TRect;
  I, j, c1, c2, cw: Integer;
  zr: Double;
  h: String;
begin
  Canvas.Font:=Font;
  with Canvas do begin
    if not Transparent then begin
      Brush.Color:=Self.Color;
      Brush.Style:=bsSolid;
      FillRect(ClientRect);
    end;
    Brush.Style:=bsClear;
    // now here we draw the lines
    // Border
    Pen.Width:=1;
    // Linker Rand
    Pen.Style:=Border.Left.Style;
    for i:=0 to Border.Left.Width - 1 do begin
      MoveTo(I, ClientHeight - 1);
      if Border.Left.Height = dhHalf then LineTo(I, ClientHeight div 2);
      if Border.Left.Height = dhFull then LineTo(I, -1);
    end;
    // Rechter Rand
    Pen.Style:=Border.Right.Style;
    for i:=0 to Border.Right.Width - 1 do begin
      MoveTo(ClientWidth - 1 - I, ClientHeight - 1);
      if Border.Right.Height = dhHalf then
        LineTo(ClientWidth - 1 - I, ClientHeight div 2);
      if Border.Right.Height = dhFull then LineTo(ClientWidth - 1 - I, -1);
    end;
    // Oberer Rand
    Pen.Style:=Border.Top.Style;
    for i:=0 to Border.Top.Width - 1 do begin
      MoveTo(ClientWidth - 1, I);
      LineTo(-1, I);
    end;
    // Unterer Rand
    Pen.Style:=Border.Bottom.Style;
    for i:=0 to Border.Bottom.Width - 1 do begin
      MoveTo(-1, ClientHeight - 1 - I);
      LineTo(ClientWidth, ClientHeight - 1 - I);
    end;
    // Divider
    c1:=1;
    c2:=MaxLength - 1;
    Pen.Style:=Divider.Style;
    if FFixLength and (Divider.Style <> psClear) then begin
      zr:=(ClientWidth - Border.Left.Width - Border.Right.Width
                      - (Divider.Width * (FMaxLength - 1))) / FMaxLength;
      for I:=c1 to c2 do begin
        cw:=Round(zr * i + (Divider.Width * (i - 1)));
        if I > 0 then begin
          for j:=0 to Divider.Width - 1 do begin
            MoveTo(cw + j + Border.Left.Width, ClientHeight - 1);
            case Divider.Height of
              dhFull: LineTo(cw + j + Border.Left.Width, 0);
              dhHalf: LineTo(cw + j + Border.Left.Width, ClientHeight div 2);
            end;
          end;
        end;
      end;
    end;
    // CrossOut
    if CrossOut.Style In [coSimple, coDouble] then begin
      Pen.Style:=psSolid;
      cw:=CrossOut.Width div 2;
      for i:=0 to CrossOut.Width - 1 do begin
        MoveTo(-cw + i, ClientHeight - 1);
        LineTo(ClientWidth - 1 - cw + i, 0);
        if CrossOut.Style = coDouble then begin
          MoveTo(-cw + i, 0);
          LineTo(ClientWidth - 1 - cw + i, ClientHeight - 1);
        end;
      end;
    end;
    // Text
    h:=GetText;
    if trim(h) = '' then h:='Rjgh';
    if FFixLength and (Length(GetText) > 0) then
      begin
        cw:=Length(GetText);
        case Alignment of
          taCenter: begin c1:=1 + (FMaxLength - cw) div 2; c2:=c1 + cw - 1; end;
          taRightJustify: begin c2:=FMaxLength; c1:=c2 - cw + 1; end;
        else
          begin c1:=1; c2:=cw; end;
        end;
        zr:=(ClientWidth - Border.Left.Width - Border.Right.Width
                         - (Divider.Width * (FMaxLength - 1))) / FMaxLength;
        for I:=c1 to c2 do begin
          Rect.Left:=Round(Border.Left.Width + ((zr + Divider.Width) * (i - 1)) +
                     ((zr - TextWidth(GetText[i - c1 + 1])) / 2));
          Rect.Top:=Border.Top.Width + ((ClientHeight - Border.Top.Width
                    - Border.Bottom.Width - TextHeight(h)) div 2);
          TextOut(Rect.Left, Rect.Top, GetText[I - c1 + 1]);
        end;
      end
    else
      begin
        Rect.Top:=Border.Top.Width + ((ClientHeight - Border.Top.Width
                  - Border.Bottom.Width - TextHeight(h)) div 2);
        case FAlignment of
          taCenter:
            Rect.Left:=Border.Left.Width + ((ClientWidth - Border.Left.Width
                       - Border.Right.Width - TextWidth(GetText)) div 2);
          taRightJustify:
            Rect.Left:=ClientWidth - 1 - Border.Right.Width - TextWidth(GetText);
        else
          Rect.Left:=Border.Left.Width;
        end;
        TextOut(Rect.Left, Rect.Top, GetText);
      end;
  end;
end;

end.

