unit PBWatcherUnit;
//Author: Poul Bak
//Copyright  2000 - 2004 : Bak-O-Soft (Poul Bak). All rights reserved.
//http://bak-o-soft.dk/
//Mailto:info@bak-o-soft.dk
//Version: 10.10.00.00

interface

{$INCLUDE PBDefines.inc}

uses
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
	Menus, StdCtrls, ComCtrls, ExtCtrls, ShellAPI, ClipBrd, Registry, Printers,
	Buttons, TypInfo
	{$IFNDEF COMPILER_MAX_3}
		, ImgList
		{$IFNDEF COMPILER_MAX_5} , Variants {$ENDIF}
	{$ENDIF};

const
	_PBMAX_VARIABLES = 1000;
	WM_PBTOOLTRAYICON = WM_USER + 2555;
	HIDEWATCHFORM = 1;
	SHOWWATCHFORM = 2;

type
	TPBIncludeTime = (itPBNone, itPBTime, itPBTickCount);

	TPBWatchRecord = record
		VarName : string;
		WatchCounter, TotalCounter, LastTime : Cardinal;
	end;

	TPBWatchForm = class(TForm)
		Menu1: TMainMenu;
		File1: TMenuItem;
		Open1: TMenuItem;
		Save1: TMenuItem;
		N1: TMenuItem;
		Print1: TMenuItem;
		Settings1: TMenuItem;
		Enabled1: TMenuItem;
		PauseBetweenWatches1: TMenuItem;
		OpenDialog1: TOpenDialog;
		SaveDialog1: TSaveDialog;
		PrintDialog1: TPrintDialog;
		StayOnTop1: TMenuItem;
		AutoAdd1: TMenuItem;
		WatchObject1: TMenuItem;
		WatchPointer1: TMenuItem;
    WatchPoint1: TMenuItem;
    WatchRect1: TMenuItem;
		AddToWatches1: TMenuItem;
		AutoAdd2: TMenuItem;
		WatchCharArray1: TMenuItem;
    WatchIntArray1: TMenuItem;
    WatchDoubleArray1: TMenuItem;
    WatchBoolArray1: TMenuItem;
    WatchColor1: TMenuItem;
    WatchCursor1: TMenuItem;
		N13: TMenuItem;
		AutoSavewatches1: TMenuItem;
		Includetime1: TMenuItem;
		Includecounter1: TMenuItem;
		Maxwatches1: TMenuItem;
		Skipwatches1: TMenuItem;
		Help1: TMenuItem;
		Action1: TMenuItem;
		ClearWatch1: TMenuItem;
		ResetCounterPause1: TMenuItem;
		N2: TMenuItem;
		FindDialog1: TFindDialog;
		Findtext1: TMenuItem;
		RemoveWatches: TMenuItem;
    AddToUsesClause: TMenuItem;
		AutoAdd3: TMenuItem;
		WatchStrings1: TMenuItem;
    WatchComponents1: TMenuItem;
		N14: TMenuItem;
		RestoreClipBoard1: TMenuItem;
    WatchByteArray1: TMenuItem;
    WatchMenuItem1: TMenuItem;
    WatchMenu1: TMenuItem;
    WatchDateTime1: TMenuItem;
		N19: TMenuItem;
		OpenDesignModewatchfile1: TMenuItem;
		OpenRunModewatchfile1: TMenuItem;
		N20: TMenuItem;
		SameSettings: TMenuItem;
		N21: TMenuItem;
		ConditionalWatch: TMenuItem;
    WatchExtendedArray1: TMenuItem;
		WatchShortIntArray1: TMenuItem;
    ImageList1: TImageList;
    PageControl1: TPageControl;
		TabSheet1: TTabSheet;
		TabSheet2: TTabSheet;
    WatchEdit: TRichEdit;
    ScrollBox1: TScrollBox;
    WatchImage: TImage;
		StaticText1: TStaticText;
    WatchBitmap1: TMenuItem;
    WatchCanvasRect1: TMenuItem;
    RemoveFromUsesClause: TMenuItem;
    BitmapEdit: TEdit;
		BitmapUpDown: TUpDown;
		WatchImageList1: TMenuItem;
    PBBackColorDialog: TColorDialog;
		BackColor: TMenuItem;
    TransparentBitmap: TMenuItem;
    N27: TMenuItem;
		WatchSet1: TMenuItem;
    WatchEnum1: TMenuItem;
    Watch1: TMenuItem;
    N3: TMenuItem;
    WatchTimer: TMenuItem;
		ResetTimer: TMenuItem;
    WatchObjectList1: TMenuItem;
    BitmapHeight1: TMenuItem;
    BitmapWidth1: TMenuItem;
		procedure CopyAllToWatcher;
		function GetEditWindow : TForm;
		procedure SendKeys(Keys : string);
		procedure SendVKey(Key : Word);
		procedure InsertLine(Sender : TObject; WatchProc : string);
		procedure Enabled1Click(Sender: TObject);
		procedure PasteAllFromWatcher;
		procedure PauseBetweenWatches1Click(Sender: TObject);
		procedure Open1Click(Sender: TObject);
		procedure Save1Click(Sender: TObject);
		procedure Print1Click(Sender: TObject);
		procedure StayOnTop1Click(Sender: TObject);
		procedure ResetCounterPause1Click(Sender: TObject);
		procedure FormShowHide(Sender: TObject);
		procedure AddToWatches1Click(Sender: TObject);
		procedure WndProc(var Msg : TMessage); override;
		procedure AutoSavewatches1Click(Sender: TObject);
		procedure Includecounter1Click(Sender: TObject);
		procedure Includetime1Click(Sender: TObject);
		procedure Skipwatches1Click(Sender: TObject);
		procedure Maxwatches1Click(Sender: TObject);
		procedure Help1Click(Sender: TObject);
		procedure Findtext1Click(Sender: TObject);
		procedure FindDialog1Find(Sender: TObject);
		procedure FindDialog1Close(Sender: TObject);
		procedure RemoveWatchesClick(Sender: TObject);
		procedure ClearWatch1Click(Sender: TObject);
		procedure AddToUsesClauseClick(Sender: TObject);
		procedure RestoreClipBoard1Click(Sender: TObject);
		procedure OpenDesignModewatchfile1Click(Sender: TObject);
		procedure OpenRunModewatchfile1Click(Sender: TObject);
		procedure SameSettingsClick(Sender: TObject);
		procedure FormResize(Sender: TObject);
		procedure GetClipBoard;
		procedure RestoreClipBoard;
		procedure ConditionalWatchClick(Sender: TObject);
		procedure PageControl1Change(Sender: TObject);
		procedure BitmapEditChange(Sender: TObject);
		procedure LoadWatches(FileDir, FileName : string);
		procedure SaveWatches(FileDir, FileName : string);
		procedure Recreate(const OnTop : TFormStyle);
		procedure RemoveFromUsesClauseClick(Sender: TObject);
		procedure BackColorClick(Sender: TObject);
		procedure SetBackColors;
		procedure TransparentBitmapClick(Sender: TObject);
		procedure FormCreate(Sender: TObject);
		procedure AddSubMenuClick(Sender: TObject);
		procedure AutoAddClick(Sender: TObject);
    procedure WatchImageMouseMove(Sender: TObject; Shift: TShiftState; X,
			Y: Integer);
    procedure BitmapHeight1Click(Sender: TObject);
    procedure BitmapWidth1Click(Sender: TObject);
	private
		TempEnabled : Boolean;
	end;

	TNotiFyIconDataEx = record
		cbSize: DWORD;
		Wnd: HWND;
		uID: UINT;
		uFlags: UINT;
		uCallbackMessage: UINT;
		hIcon: HICON;
		szTip: array [0..127] of AnsiChar;
		dwState: DWORD;
		dwStateMask: DWORD;
		szInfo: array[0..255]of AnsiChar;
		case integer of
			0: (uTimeout: UINT);
			1: (uVersion: UINT;
					szInfoTitle: array[0..63]of AnsiChar;
					dwInfoFlags: DWORD;
					guidItem : TGuid;
				 )
	end;

	TPBWatchTrayIcon = class(TPersistent)
	private
		FHWnd : HWnd;
		FIconData : TNotifyIconDataEx;
		FVisible : Boolean;
		function ShowIcon : Boolean;
		function HideIcon :Boolean;
		procedure WndProc(var Msg : TMessage);
	public
		constructor Create;
		destructor Destroy; override;
	end;

	TPBWatcher = class(TPersistent)
	private
		{ Private declarations }
		FEnabled, FVisible, FStayOnTop : Boolean;
		FAutoSaveWatch, FIncludeCounter, FAddToWatches : Boolean;
		FPauseBetweenWatches, FMaxWatches, FSkipWatches, FVarNumber : Cardinal;
		FTime, FBitmapHeight, FBitmapWidth : Cardinal;
		FIncludeTime : TPBIncludeTime;
		FFileDir, FFileName, FHelpFile, FConditionalWatch : string;
		FPBWatchForm: TPBWatchForm;
		FPBWatchTrayIcon : TPBWatchTrayIcon;
		FWatchRecords : array[0.._PBMAX_VARIABLES] of TPBWatchRecord;
		function BoolToStr(Value : Boolean) : string;
		function Proceed(Name : string) : Boolean;
		function ReadColor : TColor;
		function ReadTransparent : Boolean;
		procedure UpdateWatch(Name, Value : string; const Save : Boolean);
		procedure SetAddToWatches(Value : Boolean);
		procedure SetAutoSaveWatch(Value : Boolean);
		procedure SetBitmapHeight(Value : Cardinal);
		procedure SetBitmapWidth(Value : Cardinal);
		procedure SetColor(Value : TColor);
		procedure SetEnabled(Value : Boolean);
		procedure SetIncludeCounter(Value : Boolean);
		procedure SetIncludeTime(Value : TPBIncludeTime);
		procedure SetMaxWatches(Value : Cardinal);
		procedure SetPauseBetweenWatches(Value : Cardinal);
		procedure SetSkipWatches(Value : Cardinal);
		procedure SetStayOnTop(Value : Boolean);
		procedure SetTransparent(Value : Boolean);
		procedure SetVisible(Value : Boolean);
		procedure LoadSettings(Mode : string);
		procedure SaveSettings(Mode : string);
		procedure SaveAllSettings;
	protected
		{ Protected declarations }
	public
		{ Public declarations }
		constructor Create;
		destructor Destroy; override;
		procedure Clear;
		procedure ResetCounter;
		procedure Watch(const VarName : string; const Value : Variant);
		procedure WatchEnum(const VarName : string; const Value;
			const Info : PTypeInfo);
		procedure WatchSet(const VarName : string; const Value;
			const Info : PTypeInfo);
		procedure WatchObject(const VarName : string; const Value : TObject);
		procedure WatchPointer(const VarName : string; const Value : Pointer;
			const Length : integer);
		procedure WatchPoint(const VarName : string; const Value : TPoint);
		procedure WatchRect(const VarName : string; const Value : TRect);
		procedure WatchColor(const VarName : string; const Value : TColor);
		procedure WatchCursor(const VarName : string; const Value : TCursor);
		procedure WatchMenuItem(const VarName : string; const Value : TMenuItem);
		procedure WatchDateTime(const VarName : string; const Value : TDateTime);
		procedure WatchBitmap(const VarName : string; const Value : TBitmap);
		procedure WatchCanvasRect(const VarName : string; const Value : TCanvas;
			const Rect0 : TRect);
		procedure WatchTimer(const VarName : string);
		procedure ResetTimer;
		procedure WatchCharArray(const VarName : string; const Values : array of Char);
		procedure WatchIntArray(const VarName : string; const Values : array of integer);
		procedure WatchByteArray(const VarName : string; const Values : array of Byte);
		procedure WatchShortIntArray(const VarName : string; const Values : array of ShortInt);
		procedure WatchDoubleArray(const VarName : string; const Values : array of Double);
		procedure WatchExtendedArray(const VarName : string; const Values : array of Extended);
		procedure WatchBoolArray(const VarName : string; const Values : array of Boolean);
		procedure WatchStrings(const VarName : string; const Value : TStrings);
		procedure WatchComponents(const VarName : string; const Value : TComponent);
		procedure WatchMenu(const VarName : string; const Value : TComponent);
		procedure WatchImageList(const VarName : string; const Value : TImagelist);
		property AddToWatches : Boolean read FAddToWatches write SetAddToWatches;
		property AutoSaveWatch : Boolean read FAutoSaveWatch write SetAutoSaveWatch;
		property BackgroundColor : TColor read ReadColor write SetColor;
		property BitmapHeight : Cardinal read FBitmapHeight write SetBitmapHeight;
		property BitmapWidth : Cardinal read FBitmapWidth write SetBitmapWidth;
		property FileDir : string read FFileDir write FFileDir;
		property FileName : string read FFileName write FFileName;
		property Enabled : Boolean read FEnabled write SetEnabled;
		property IncludeCounter : Boolean read FIncludeCounter write SetIncludeCounter;
		property IncludeTime : TPBIncludeTime read FIncludeTime write SetIncludeTime;
		property MaxWatches : Cardinal read FMaxWatches write SetMaxWatches;
		property PauseBetweenWatches : Cardinal read FPauseBetweenWatches write SetPauseBetweenWatches;
		property SkipWatches : Cardinal read FSkipWatches write SetSkipWatches;
		property StayOnTop : Boolean read FStayOnTop write SetStayOnTop;
		property TransparentBitmap : Boolean read ReadTransparent write SetTransparent;
		property Visible : Boolean read FVisible write SetVisible;
	end;

function PBWatcher : TPBWatcher;

implementation

{$R *.DFM}

type
	TPBWatchMenuAdder = class(TPersistent)
	private
		PBWatchMenu, N1, ViewMenu : TMenuItem;
		constructor Create;
		destructor Destroy; override;
		procedure PBWatchMenuClick(Sender : TObject);
	end;

var
	PBWatcherObject : TPBWatcher = nil;
	PBWatchMenuAdder : TPBWatchMenuAdder = nil;
	PBWatcherHideShowMessage : DWord;
	IsDesigning, HasFoundOne, IsCreating, FromMessage, TempVisible : Boolean;
	RefCount : integer;
	Mode : string;
	ClipText : PChar = nil;
	ColorBitmap : TBitmap;

function PBWatcher : TPBWatcher;
begin
	if RefCount = 0 then PBWatcherObject := TPBWatcher.Create;
	Result := PBWatcherObject;
end;

procedure PrintWysiwygBitmap(const Bitmap0 : TBitmap);
var
	PPIX, PPIY : integer;
	TempBitmap : TBitmap;
	Rect0 : TRect;
begin
	TempBitmap := TBitmap.Create;
	with Printer do
	try
		PPIX := GetDeviceCaps(Handle, LOGPIXELSX);
		PPIY := GetDeviceCaps(Handle, LOGPIXELSY);
		Rect0 := Bounds(0, 0, Trunc(Bitmap0.Width * PPIX / Screen.PixelsPerInch),
			Trunc(Bitmap0.Height * PPIY / Screen.PixelsPerInch));
		TempBitmap.Width := Rect0.Right;
		TempBitmap.Height := Rect0.Bottom;
		TempBitmap.Canvas.StretchDraw(Rect0, Bitmap0);
		Title := 'PBWatcher ' + DateTimeToStr(Now);
		BeginDoc;
		Canvas.Draw(0, 0, TempBitmap);
		EndDoc;
	finally
		TempBitmap.Free;
	end;
end;

//  ----------------- PBWatcher object ------------------------
constructor TPBWatcher.Create;
var
	Reg : TRegistry;
	Buffer : array[0..MAX_PATH] of Char;
	Keys : TStringList;
	t : integer;
	SearchPath : string;
begin
	inherited Create;
	IsCreating := True;
	Inc(RefCount);
	PBWatcherObject := Self;
	FPBWatchForm := TPBWatchForm.Create(nil);
	Enabled := True;
	FVisible := False;
	TempVisible := True;
	StayOnTop := False;
	AutoSaveWatch := False;
	SkipWatches := 0;
	MaxWatches := 0;
	PauseBetweenWatches := 0;
	AddToWatches := True;
	IncludeTime := itPBNone;
	IncludeCounter := False;
	GetTempPath(MAX_PATH, Buffer);
	FFileDir := String(Buffer);
	PBWatcherHideShowMessage := RegisterWindowMessage('PBWatchHideShowDesigning');
	if IsDesigning then FPBWatchTrayIcon := nil
	else FPBWatchTrayIcon := TPBWatchTrayIcon.Create;
	ColorBitmap := TBitmap.Create;
	ColorBitmap.Height := 10;
	ColorBitmap.Width := 10;
	ColorBitmap.TransparentColor := clBtnFace;
	with FPBWatchForm do
	begin
		Left := Screen.Width div 2;
		Top := -2;
		Height := 145;
		Width := Left - 20;
		BackColor.Bitmap := ColorBitmap;
		if IsDesigning then Mode := 'Design'
		else
		begin
			Mode := 'Run';
			AutoAdd1.Enabled := False;
			AutoAdd2.Enabled := False;
			AutoAdd3.Enabled := False;
			AddToUsesClause.Enabled := False;
			RemoveFromUsesClause.Enabled := False;
			RemoveWatches.Enabled := False;
			RestoreClipBoard1.Enabled := False;
			ConditionalWatch.Enabled := False;
		end;
		Caption := 'PBWatcher - [' + Mode + 'mode]';
		FFileName := 'PBWatcher.txt';
	end;
	LoadSettings(Mode);
	Reg := TRegistry.Create;
	with Reg do
	begin
		if not FileExists(FHelpFile) then
		begin
			GetWindowsDirectory(Buffer, MAX_PATH);
			if FileExists(string(Buffer) + '\Help\PBWatcher.chm')
				and FileExists(string(Buffer) + '\hh.exe')
				then FHelpFile := string(Buffer) + '\Help\PBWatcher.chm'
			else if FileExists(string(Buffer) + '\Help\PBWatcher.htm')
				then FHelpFile := string(Buffer) + '\Help\PBWatcher.htm'
			else if OpenKey('\Software\Borland\Delphi', False) then
			begin
				Keys := TStringList.Create;
				GetKeyNames(Keys);
				for t := Keys.Count - 1 downto 0 do
				begin
					if Pos(Copy(Keys[t], 1, 2), '3.|4.|5.|6.|7.|8.|9.') <> 0 then
					begin
						if OpenKey(Keys[t] + '\Library', False) then
						begin
							if ValueExists('SearchPath') then
							begin
								SearchPath := ReadString('SearchPath');
								FHelpFile := FileSearch('PBWatcher.chm', SearchPath);
								if (FHelpFile = '') or not FileExists(string(Buffer) + '\hh.exe')
								 then FHelpFile := FileSearch('PBWatcher.htm', SearchPath);
							end
							else if ValueExists('Search Path') then
							begin
								SearchPath := ReadString('Search Path');
								FHelpFile := FileSearch('PBWatcher.chm', SearchPath);
								if (FHelpFile = '') or not FileExists(string(Buffer) + '\hh.exe')
								 then FHelpFile := FileSearch('PBWatcher.htm', SearchPath);
							end;
						end;
					end;
				end;
				Keys.Free;
			end;
		end;
		Free;
	end;
	if FHelpFile = '' then FPBWatchForm.Help1.Enabled := False
	else FPBWatchForm.HelpFile := FHelpFile;
	Visible := TempVisible;
	IsCreating := False;
	FTime := GetTickCount;
end;

destructor TPBWatcher.Destroy;
begin
	SaveAllSettings;
	FPBWatchTrayIcon.Free;
	ColorBitmap.Free;
	FPBWatchForm.Free;
	Dec(RefCount);
	inherited Destroy;
end;

function TPBWatcher.BoolToStr(Value :  Boolean) : string;
begin
	if Value = True then Result := 'True'
	else Result := 'False';
end;

function TPBWatcher.Proceed(Name : string) : Boolean;
var
	t : integer;
begin
	Result := False;
	t := 0;
	while (FWatchRecords[t].VarName <> Name) and (FWatchRecords[t].VarName <> '')
		and (t < _PBMAX_VARIABLES) do Inc(t);
	if t <= _PBMAX_VARIABLES then with FWatchRecords[t] do
	begin
		FVarNumber := t;
		if FPBWatchForm.Enabled1.Checked	and ((TotalCounter >= FSkipWatches) or (FSkipWatches = 0))
			and ((WatchCounter < FMaxWatches) or (FMaxWatches = 0))
			and (GetTickCount >= LastTime + FPauseBetweenWatches) then Result := True;
		Inc(TotalCounter);
	end;
end;

procedure TPBWatcher.UpdateWatch(Name, Value : string; const Save : Boolean);
var
	TimeString, CounterString, TempName : string;
	FirstChar : Char;
begin
	if Length(Name) > 0 then FirstChar := Name[1]
	else FirstChar := #0;
	if (FirstChar in ['|', ' ']) then TempName := Copy(Name, 2, Length(Name) - 1)
	else TempName := Name;
	if FVarNumber <= _PBMAX_VARIABLES then with FWatchRecords[FVarNumber] do
	begin
		VarName := TempName;
		Inc(WatchCounter);
		LastTime := GetTickCount;
	end;
	if FIncludeTime = itPBTime then TimeString := ' {' + TimeToStr(Time) + '}'
	else if FIncludeTime = itPBTickCount then TimeString := ' {' + IntToStr(GettickCount) + '}'
	else TimeString := '';
	if FIncludeCounter and (FVarNumber <= _PBMAX_VARIABLES) then
	begin
		if FWatchRecords[FVarNumber].WatchCounter <> FWatchRecords[FVarNumber].TotalCounter
			then CounterString := '  {#' + IntToStr(FWatchRecords[FVarNumber].WatchCounter)
			+ '/' + IntToStr(FWatchRecords[FVarNumber].TotalCounter) + '}'
		else CounterString := '  {#' + IntToStr(FWatchRecords[FVarNumber].WatchCounter) + '}';
	end
	else CounterString  := '';
	with FPBWatchForm.WatchEdit do
	begin
		try
			if FAddToWatches then
			begin
				if FirstChar = ' ' then Lines.Add('');
				if FirstChar = '|' then Lines[Lines.Count - 1] := Lines[Lines.Count - 1]
					+ '   ' + TempName + '=' + Value + CounterString + TimeString
				else Lines.Add(TempName + '=' + Value + CounterString + TimeString);
			end
			else Lines.Values[TempName] := Value + CounterString + TimeString;
		except
			Self.Enabled := False;
			ShowMessage('PBWatcher Error: Can''t add more watches!'#10#13
				+'PBWatcher is disabled.'#10#13'(See MaxWatches)');
		end;
	end;
	if Save then
	begin
		if FFileDir[Length(FFileDir)] = '\'
			then FPBWatchForm.WatchEdit.Lines.SaveToFile(FFileDir + FFileName)
		else FPBWatchForm.WatchEdit.Lines.SaveToFile(FFileDir + '\' + FFileName);
	end;
end;

procedure TPBWatcher.Watch(const VarName : string; const Value : Variant);
var
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		if (VarType(Value) and varBoolean) = varBoolean
			then ValueString := BoolToStr(Value)
		else ValueString := VarToStr(Value);
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchEnum(const VarName : string; const Value;
	const Info : PTypeInfo);
var
	ValueString : string;
	S : Byte;
begin
	if Proceed(VarName) then with FPBWatchForm do
	begin
		S := Byte(Value);
		ValueString := GetEnumName(Info, S);
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchSet(const VarName : string; const Value;
	const Info : PTypeInfo);
var
	ValueString : string;
	S, i : integer;
	SetTypeData, BaseTypeData: PTypeData;
	BaseTypeInfo: {$IFDEF COMPILER_MAX_3} PTypeInfo {$ELSE} PPTypeInfo {$ENDIF};
begin
	if Proceed(VarName) then with FPBWatchForm do
	begin
		SetTypeData := GetTypeData(Info);
		{$IFDEF COMPILER_MAX_3}
			BaseTypeInfo := SetTypeData^.CompType^;
			BaseTypeData := GetTypeData(BaseTypeInfo);
		{$ELSE}
			BaseTypeInfo := SetTypeData^.CompType;
			BaseTypeData := GetTypeData(BaseTypeInfo^);
		{$ENDIF}
		S := Byte(Value);
		ValueString := '[';
		for i := BaseTypeData^.MinValue to BaseTypeData^.MaxValue do
		begin
			if (S and 1) = 1 then
			{$IFDEF COMPILER_MAX_3}
				ValueString := ValueString + GetEnumName(BaseTypeInfo, i) + ',';
			{$ELSE}
				ValueString := ValueString + GetEnumName(BaseTypeInfo^, i) + ',';
			{$ENDIF}
			S := S shr 1;
		end;
		if ValueString[Length(ValueString)] = ','
			then Delete(ValueString, Length(ValueString), 1);
		ValueString := ValueString + ']';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchBitmap(const VarName : string; const Value : TBitmap);
var
	ValueString, TempName : string;
	TempBitmap, TempMask : TBitmap;
	Rect0 : TRect;
	ImageNumber : integer;
	FirstChar : Char;
begin
	TempMask := nil;
	if Proceed(VarName) then with FPBWatchForm do
	begin
		if Value <> nil then
		begin
			TempBitmap := TBitmap.Create;
			if ImageList1.Count = 0 then
			begin
				if (BitmapHeight = 0) then ImageList1.Height := Value.Height + 2
				else ImageList1.Height := BitmapHeight + 2;
				if (BitmapWidth = 0) then ImageList1.Width := Value.Width + 2
				else ImageList1.Width := BitmapWidth + 2;
			end;
			with TempBitmap do
			begin
				Height := ImageList1.Height;
				Width := ImageList1.Width;
				Canvas.Brush.Color := PBBackColorDialog.Color;
				Canvas.FillRect(Canvas.ClipRect);
				Canvas.Draw(1, 1, Value);
				IntersectRect(Rect0, Bounds(0, 0, Width, Height), Bounds(0, 0,
					Value.Width + 2, Value.Height + 2));
				Canvas.Brush.Color := clBlack;
				Canvas.FrameRect(Rect0);
			end;
			if FPBWatchForm.TransparentBitmap.Checked then
			begin
				Tempmask := TBitmap.Create;
				TempMask.Assign(TempBitmap);
				TempMask.Mask(Value.TransparentColor);
			end;
			if FAddToWatches then	ImageNumber := ImageList1.Add(TempBitmap, TempMask)
			else
			begin
				if Length(VarName) > 0 then FirstChar := VarName[1]
				else FirstChar := #0;
				if (FirstChar in ['|', ' '])
					then TempName := Copy(VarName, 2, Length(VarName) - 1)
				else TempName := VarName;
				ImageNumber := StrToIntDef(WatchEdit.Lines.Values[TempName], -1);
				if ImageNumber <> -1 then ImageList1.Replace(ImageNumber, TempBitmap,
					TempMask)
				else ImageNumber := ImageList1.Add(TempBitmap, TempMask);
			end;
			TempMask.Free;
			TempBitmap.Free;
			if ImageList1.Count > 0 then with BitmapUpDown do
			begin
				Max := ImageList1.Count - 1;
				Min := 0;
			end
			else with BitmapUpDown do
			begin
				Max := -1;
				Min := -1;
				Position := -1;
				BitmapEdit.ReadOnly := True;
			end;
			if ImageNumber <> -1 then BitmapUpDown.Position := ImageNumber;
			BitmapEditChange(Self);
			if BitmapUpDown.Max > 0 then BitmapEdit.ReadOnly := False;
			ValueString := 'Image Number ' + IntToStr(ImageNumber);
		end
		else ValueString := 'nil';
		UpdateWatch(VarName, ValueString, False);
		if FAutoSaveWatch then SaveWatches(FFileDir, FFileName);
	end;
end;

procedure TPBWatcher.WatchCanvasRect(const VarName : string; const Value : TCanvas;
	const Rect0 : TRect);
var
	TempBitmap : TBitmap;
begin
	TempBitmap := TBitmap.Create;
	with TempBitmap do
	try
		Width := Rect0.Right - Rect0.Left;
		Height := Rect0.Bottom - Rect0.Top;
		Canvas.CopyRect(Bounds(0, 0, Width, Height), Value, Rect0);
		WatchBitmap(VarName, TempBitmap);
	finally
		TempBitmap.Free;
	end;
end;

procedure TPBWatcher.WatchTimer(const VarName : string);
var
	TimeElapsed : Cardinal;
	TempName : string;
begin
	TimeElapsed := GetTickCount - FTime;
	if VarName = '' then TempName := 'Time elapsed'
	else TempName := VarName;
	Watch(TempName, IntToStr(TimeElapsed) + ' mSec');
end;

procedure TPBWatcher.ResetTimer;
begin
	FTime := GetTickCount;
end;

procedure TPBWatcher.WatchCharArray(const VarName : string; const Values : array of Char);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ', ';
			if IsCharAlphaNumeric(Values[t])
				then ValueString := ValueString + '''' + Values[t] + ''''
			else ValueString := ValueString + '#' + IntToStr(Ord(Values[t]));
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchIntArray(const VarName : string; const Values : array of integer);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ', ';
			ValueString := ValueString + IntToStr(Values[t]);
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchByteArray(const VarName : string; const Values : array of Byte);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ', ';
			ValueString := ValueString + IntToStr(Values[t]);
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchExtendedArray(const VarName : string; const Values : array of Extended);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ',  ';
			ValueString := ValueString + FloatToStr(Values[t]);
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchShortIntArray(const VarName : string; const Values : array of ShortInt);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ',  ';
			ValueString := ValueString + IntToStr(Values[t]);
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchDoubleArray(const VarName : string; const Values : array of Double);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ',  ';
			ValueString := ValueString + FloatToStr(Values[t]);
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchBoolArray(const VarName : string; const Values : array of Boolean);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := '(';
		for t := Low(Values) to High(Values) do
		begin
			if t > Low(Values) then ValueString := ValueString + ', ';
			ValueString := ValueString + BoolToStr(Values[t]);
		end;
		ValueString := ValueString + ')';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchObject(const VarName : string; const Value : TObject);
var
	ValueString : string;
	Ancestor : TClass;
begin
	if Proceed(VarName) then
	begin
		if Value = nil then ValueString := 'nil'
		else if Value is TComponent then ValueString := (Value as TComponent).Name
			+ ' [' + Value.ClassName + ']'
		else ValueString := ' [' + Value.ClassName + ']';
		if Value <> nil then
		begin
			Ancestor := Value.ClassParent;
			while Ancestor <> nil do
			begin
				ValueString := ValueString + ', [' + Ancestor.ClassName + ']';
				Ancestor := Ancestor.ClassParent;
			end;
		end;
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchPointer(const VarName : string; const Value : Pointer;
	const Length : integer);
var
	t : integer;
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		if Value = nil then ValueString := 'nil'
		else if Length = 0 then ValueString := '$' + IntToHex(Integer(Value), 8)
		else
		begin
			ValueString := '';
			for t := Integer(Value) to Integer(Value) + Length - 1 do
			begin
				if t > Integer(Value) then ValueString := ValueString + ', ';
				if IsCharAlphaNumeric(PChar(t)^)
					then ValueString := ValueString + '''' + PChar(t)^ + ''''
				else ValueString := ValueString + '#' + IntToStr(Ord(PChar(t)^));
			end;
		end;
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchPoint(const VarName : string; const Value : TPoint);
var
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		ValueString := ' X:' + IntToStr(Value.x) + ' Y:' + IntToStr(Value.y);
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchRect(const VarName : string; const Value : TRect);
var
	ValueString : string;
begin
	if Proceed(VarName) then with Value do
	begin
		ValueString := ' Left:' + IntToStr(Left) + ' Top:' + IntToStr(Top)
			+ ' Right:' + IntToStr(Right) + ' Bottom:' + IntToStr(Bottom);
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchColor(const VarName : string; const Value : TColor);
begin
	if Proceed(VarName)
		then UpdateWatch(VarName, ColorToString(Value),	FAutoSaveWatch);
end;

procedure TPBWatcher.WatchCursor(const VarName : string; const Value : TCursor);
begin
	if Proceed(VarName)
		then UpdateWatch(VarName, CursorToString(Value), FAutoSaveWatch);
end;

procedure TPBWatcher.WatchMenuItem(const VarName : string; const Value : TMenuItem);
var
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		if Value <> nil then
		begin
			ValueString := 'Name:' + Value.Name + ', Caption:' + Value.Caption
				+ ', Shortcut:'	+ ShortCutToText(Value.ShortCut) + ', Checked:'
				+ BoolToStr(Value.Checked) + ', Enabled:' + BoolToStr(Value.Enabled)
				+ 'Visible:' + BoolToStr(Value.Visible) + ', Hint:' + Value.Hint + ', OnClick:';
			if Assigned(Value.OnClick) then ValueString := ValueString + '$'
				+ IntToHex(Integer(@@Value.OnClick), 8)
			else ValueString := ValueString + 'nil';
		end
		else ValueString := 'nil';
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchDateTime(const VarName : string; const Value : TDateTime);
var
	ValueString : string;
begin
	if Proceed(VarName) then
	begin
		if Value > 1 then ValueString := FormatDateTime('dddd dddddd', Value) + ' ';
		ValueString := ValueString + FormatDateTime('tt', Value);
		UpdateWatch(VarName, ValueString, FAutoSaveWatch);
	end;
end;

procedure TPBWatcher.WatchStrings(const VarName : string; const Value : TStrings);
var
	t : integer;
begin
	if Proceed(VarName) then
	begin
		if not FAddToWatches then FPBWatchForm.WatchEdit.Clear;
		if Value = nil then UpdateWatch(VarName, '', FAutoSaveWatch)
		else
		begin
			UpdateWatch(VarName, '', False);
			for t := 0 to Value.Count - 1 do
			begin
				UpdateWatch('[' + IntToStr(t) + ']', Value[t], False);
				if Value.Objects[t] <> nil
					then WatchObject('Objects[' + IntToStr(t) + ']', Value.Objects[t]);
			end;
			UpdateWatch('', '', FAutoSaveWatch);
		end;
	end;
end;

procedure TPBWatcher.WatchComponents(const VarName : string; const Value : TComponent);
var
	t : integer;
begin
	if Proceed(VarName) then
	begin
		if not FAddToWatches then FPBWatchForm.WatchEdit.Clear;
		WatchObject(VarName, Value);
		if Value <> nil then for t := 0 to Value.ComponentCount - 1
			do WatchObject('Components[' + IntToStr(t)	+ ']', Value.Components[t]);
	end;
end;

procedure TPBWatcher.WatchMenu(const VarName : string; const Value : TComponent);
var
	t : integer;
begin
	if Proceed(VarName) then
	begin
		if not FAddToWatches then FPBWatchForm.WatchEdit.Clear;
		WatchObject(VarName, Value);
		if Value is TMenu then
		begin
			for t := 0 to (Value as TMenu).Items.Count - 1
				do WatchMenuItem('Items[' + IntToStr(t)	+ ']', (Value as TMenu).Items[t]);
		end
		else if Value is TMenuItem then
		begin
			for t := 0 to (Value as TMenuItem).Count - 1
				do WatchMenuItem('Items[' + IntToStr(t)	+ ']', (Value as TMenuItem)[t]);
		end;
	end;
end;

procedure TPBWatcher.WatchImageList(const VarName : string;
	const Value : TImagelist);
var
	TempBitmap, NoMaskBitmap : TBitmap;
	Rect0 : TRect;
	t, StartImageNumber, EndImageNumber : integer;
begin
	if Proceed(VarName) then with FPBWatchForm do
	begin
		if Value <> nil then
		begin
			if not FAddToWatches then Clear;
			StartImageNumber := ImageList1.Count;
			if ImageList1.Count = 0 then
			begin
				if (BitmapHeight = 0) then ImageList1.Height := Value.Height + 2
				else ImageList1.Height := BitmapHeight + 2;
				if (BitmapWidth = 0) then ImageList1.Width := Value.Width + 2
				else ImageList1.Width := BitmapWidth + 2;
			end;
			TempBitmap := TBitmap.Create;
			TempBitmap.Height := ImageList1.Height;
			TempBitmap.Width := ImageList1.Width;
			for t := 0 to Value.Count - 1 do
			begin
				TempBitmap.Canvas.Brush.Color := PBBackColorDialog.Color;
				TempBitmap.Canvas.FillRect(Rect(0, 0, TempBitmap.Width, TempBitmap.Height));
				if FPBWatchForm.TransparentBitmap.Checked
					then Value.Draw(TempBitmap.Canvas, 1, 1, t)
				else
				begin
					NoMaskBitmap := TBitmap.Create;
					Value.GetBitmap(t, NoMaskBitmap);
					TempBitmap.Canvas.Draw(1, 1, NoMaskBitmap);
					NoMaskBitmap.Free;
				end;
				IntersectRect(Rect0, Bounds(0, 0, Value.Width + 2, Value.Height + 2),
					Bounds(0, 0, ImageList1.Width, ImageList1.Height));
				TempBitmap.Canvas.Brush.Color := clBlack;
				TempBitmap.Canvas.FrameRect(Rect0);
				if FPBWatchForm.TransparentBitmap.Checked
					then ImageList1.AddMasked(TempBitmap, PBBackColorDialog.Color)
				else ImageList1.Add(TempBitmap, nil);
			end;
			TempBitmap.Free;
			if ImageList1.Count > 0 then with BitmapUpDown do
			begin
				Max := ImageList1.Count - 1;
				Min := 0;
			end
			else with BitmapUpDown do
			begin
				Max := -1;
				Min := -1;
				Position := -1;
				BitmapEdit.ReadOnly := True;
			end;
			EndImageNumber := ImageList1.Count - 1;
			if EndImageNumber <> -1 then BitmapUpDown.Position := EndImageNumber;
			BitmapEditChange(Self);
			if BitmapUpDown.Max > 0 then BitmapEdit.ReadOnly := False;
			UpdateWatch(VarName, 'Image Numbers[' + IntToStr(StartImageNumber) + '..'
				+ IntToStr(EndImageNumber) + ']', FAutoSaveWatch);
		end
		else WatchObject(VarName, nil);
	end;
end;

procedure TPBWatcher.Clear;
begin
	FPBWatchForm.ClearWatch1Click(Self);
end;

procedure TPBWatcher.ResetCounter;
begin
	FPBWatchForm.ResetCounterPause1Click(Self);
end;

procedure TPBWatcher.SetAddToWatches(Value : Boolean);
begin
	if FAddToWatches <> Value then
	begin
		FAddToWatches := Value;
		FPBWatchForm.AddToWatches1.Checked := Value;
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetAutoSaveWatch(Value : Boolean);
begin
	if FAutoSaveWatch <> Value then
	begin
		FAutoSaveWatch := Value;
		FPBWatchForm.AutoSavewatches1.Checked := Value;
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetBitmapHeight(Value : Cardinal);
begin
	if FBitmapHeight <> Value then
	begin
		FBitmapHeight := Value;
		FPBWatchForm.BitmapHeight1.Caption := 'Bitmap height: ' + IntToStr(Value);
		Clear;
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetBitmapWidth(Value : Cardinal);
begin
	if FBitmapWidth <> Value then
	begin
		FBitmapWidth := Value;
		FPBWatchForm.BitmapWidth1.Caption := 'Bitmap width: ' + IntToStr(Value);
		Clear;
		if not IsCreating then SaveAllSettings;
	end;
end;

function TPBWatcher.ReadColor : TColor;
begin
	Result := FPBWatchForm.PBBackColorDialog.Color;
end;

procedure TPBWatcher.SetColor(Value : TColor);
begin
	if Value <> FPBWatchForm.PBBackColorDialog.Color
		then FPBWatchForm.PBBackColorDialog.Color := Value;
end;


procedure TPBWatcher.SetEnabled(Value : Boolean);
begin
	if FEnabled <> Value then
	begin
		FEnabled := Value;
		FPBWatchForm.Enabled1.Checked := Value;
		FPBWatchForm.ResetCounterPause1Click(Self);
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetIncludeCounter(Value : Boolean);
begin
	if FIncludeCounter <> Value then
	begin
		FIncludeCounter := Value;
		FPBWatchForm.IncludeCounter1.Checked := Value;
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetIncludeTime(Value : TPBIncludeTime);
begin
	if FIncludeTime <> Value then
	begin
		FIncludeTime := Value;
		if Value = itPBNone then	FPBWatchForm.IncludeTime1.Caption := 'Include time (= itPBNone)'
		else if Value = itPBTime then	FPBWatchForm.IncludeTime1.Caption := 'Include time (= itPBTime)'
		else FPBWatchForm.IncludeTime1.Caption := 'Include time (= itPBTickCount)';
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetMaxWatches(Value : Cardinal);
begin
	if FMaxWatches <> Value then
	begin
		FMaxWatches := Value;
		FPBWatchform.Maxwatches1.Caption := '&Max watches (= ' + IntToStr(Value) + ')';
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetPauseBetweenWatches(Value : Cardinal);
begin
	if FPauseBetweenWatches <> Value then
	begin
		FPauseBetweenWatches := Value;
		FPBWatchform.PauseBetweenWatches1.Caption := '&Pause between watches (= ' + IntToStr(Value) + ')';
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetSkipWatches(Value : Cardinal);
begin
	if FSkipWatches <> Value then
	begin
		FSkipWatches := Value;
		FPBWatchform.Skipwatches1.Caption := '&Skip watches (= ' + IntToStr(Value) + ')';
		if not IsCreating then SaveAllSettings;
	end;
end;

procedure TPBWatcher.SetStayOnTop(Value : Boolean);
begin
	if FStayOnTop <> Value then
	begin
		FStayOnTop := Value;
		FPBWatchForm.StayOnTop1.Checked := Value;
		if Value then FPBWatchForm.Recreate(fsStayOntop)
		else FPBWatchForm.Recreate(fsNormal);
		if not IsCreating then SaveAllSettings;
	end;
end;

function TPBWatcher.ReadTransparent : Boolean;
begin
	Result := FPBWatchForm.TransparentBitmap.Checked;
end;

procedure TPBWatcher.SetTransparent(Value : Boolean);
begin
	if FPBWatchForm.TransparentBitmap.Checked <> Value
		then FPBWatchForm.TransparentBitmap.Checked := Value;
end;

procedure TPBWatcher.SetVisible(Value : Boolean);
begin
	if FVisible <> Value then
	begin
		FVisible := Value;
		FPBWatchForm.Visible := Value;
	end;
end;

procedure TPBWatcher.LoadSettings(Mode : string);
var
	Reg : TRegistry;
	PWndPlacement : PWindowPlacement;
begin
	Reg := TRegistry.Create;
	with Reg do
	begin
		if OpenKey('\Software\Borland\Delphi\PBWatcher\' + FPBWatchForm.Caption, True) then
		begin
			with FPBWatchForm do
			begin
				GetMem(PWndPlacement, SizeOf(TWindowPlacement));
				PWndPlacement^.length := SizeOf(TWindowPlacement);
				GetWindowPlacement(Handle, PWndPlacement);
				if ValueExists('ShowCmd') then PWndPlacement^.showCmd := ReadInteger('ShowCmd')
				else PWndPlacement^.showCmd := SW_SHOWNORMAL;
				if ValueExists('Left') then PWndPlacement^.rcNormalPosition.Left := ReadInteger('Left')
				else PWndPlacement^.rcNormalPosition.Left := Left;
				if ValueExists('Top') then PWndPlacement^.rcNormalPosition.Top := ReadInteger('Top')
				else PWndPlacement^.rcNormalPosition.Top := Top;
				if ValueExists('Right') then PWndPlacement^.rcNormalPosition.Right := ReadInteger('Right')
				else PWndPlacement^.rcNormalPosition.Right := Left + Width;
				if ValueExists('Bottom') then PWndPlacement^.rcNormalPosition.Bottom := ReadInteger('Bottom')
				else PWndPlacement^.rcNormalPosition.Bottom := Top + Height;
				SetWindowPlacement(Handle, PWndPlacement);
				FreeMem(PWndPlacement);
				if ValueExists('RestoreClipBoard') then RestoreClipBoard1.Checked := ReadBool('RestoreClipBoard');
				if ValueExists('SameSettings') then SameSettings.Checked := ReadBool('SameSettings');
				if ValueExists('ActivePage') then PageControl1.ActivePage :=
					PageControl1.FindChildControl(ReadString('ActivePage')) as TTabSheet;
				PageControl1Change(Self);
				if ValueExists('Backcolor')
					then PBBackColorDialog.Color := TColor(ReadInteger('BackColor'));
				SetBackColors;
				if ValueExists('TransparentBitmap')
					then TransparentBitmap.Checked := ReadBool('TransparentBitmap');
			end;
			if ValueExists('AddToWatches') then AddToWatches := ReadBool('AddToWatches');
			if ValueExists('AutoSaveWatch') then AutoSaveWatch := ReadBool('AutoSaveWatch');
			if ValueExists('ConditionalWatch') then FConditionalWatch := ReadString('ConditionalWatch');
			if ValueExists('Enabled') then Enabled := ReadBool('Enabled');
			if ValueExists('FileDir') then FFileDir := ReadString('FileDir');
			if ValueExists('FileName') then FFileName := ReadString('FileName');
			if ValueExists('HelpFile') then FHelpFile := ReadString('HelpFile');
			if ValueExists('IncludeCounter') then IncludeCounter := ReadBool('IncludeCounter');
			if ValueExists('IncludeTime') then IncludeTime := TPBIncludeTime(ReadInteger('IncludeTime'));
			if ValueExists('MaxWatches') then MaxWatches := ReadInteger('MaxWatches');
			if ValueExists('SkipWatches') then SkipWatches := ReadInteger('SkipWatches');
			if ValueExists('PauseBetweenWatches') then PauseBetweenWatches := ReadInteger('PauseBetweenWatches');
			if ValueExists('StayOnTop') then StayOnTop := ReadBool('StayOnTop');
			if ValueExists('Visible') then TempVisible := ReadBool('Visible');
			if ValueExists('BitmapHeight') then BitmapHeight := ReadInteger('BitmapHeight');
			if ValueExists('BitmapWidth') then BitmapWidth := ReadInteger('BitmapWidth');
		end;
		Free;
	end;
	if Length(FConditionalWatch) > 25
		then FPBWatchForm.ConditionalWatch.Caption := 'Conditional watch (='''
		+ Copy(FConditionalWatch, 1, 22) + '...'')'
	else FPBWatchForm.ConditionalWatch.Caption := 'Conditional watch (='''
		+ FConditionalWatch + ''')';
	FPBWatchForm.WatchImage.Align := alClient;
	FPBWatchForm.WatchImage.Align := alNone;
end;

procedure TPBWatcher.SaveSettings(Mode : string);
var
	Reg : TRegistry;
	PWndPlacement : PWindowPlacement;
begin
	Reg := TRegistry.Create;
	with Reg do
	begin
		if OpenKey('\Software\Borland\Delphi\PBWatcher\PBWatcher - [' + Mode + 'mode]', True) then
		begin
			with FPBWatchForm do
			begin
				GetMem(PWndPlacement, SizeOf(TWindowPlacement));
				PWndPlacement^.length := SizeOf(TWindowPlacement);
				GetWindowPlacement(Handle, PWndPlacement);
				if FVisible then WriteInteger('ShowCmd', PWndPlacement^.showCmd)
				else WriteInteger('ShowCmd', 0);
				WriteInteger('Left', PWndPlacement^.rcNormalPosition.Left);
				WriteInteger('Top', PWndPlacement^.rcNormalPosition.Top);
				WriteInteger('Right', PWndPlacement^.rcNormalPosition.Right);
				WriteInteger('Bottom', PWndPlacement^.rcNormalPosition.Bottom);
				FreeMem(PWndPlacement);
				WriteBool('RestoreClipBoard', RestoreClipBoard1.Checked);
				WriteBool('SameSettings', SameSettings.Checked);
				WriteString('ActivePage', PageControl1.ActivePage.Name);
				WriteInteger('BackColor', Integer(PBBackColorDialog.Color));
				WriteBool('TransparentBitmap', TransparentBitmap.Checked);
			end;
			WriteBool('AddToWatches', FAddToWatches);
			WriteBool('AutoSaveWatch', FAutoSaveWatch);
			WriteString('ConditionalWatch', FConditionalWatch);
			WriteBool('Enabled', FEnabled);
			WriteString('FileDir', FileDir);
			WriteString('FileName', FFileName);
			WriteString('HelpFile', FHelpFile);
			WriteBool('IncludeCounter', FIncludeCounter);
			WriteInteger('IncludeTime', Integer(FIncludeTime));
			WriteInteger('MaxWatches', FMaxWatches);
			WriteInteger('PauseBetweenWatches', FPauseBetweenWatches);
			WriteInteger('SkipWatches', FSkipWatches);
			WriteBool('StayOnTop', FStayOnTop);
			WriteBool('Visible', FVisible);
			WriteInteger('BitmapHeight', FBitmapHeight);
			WriteInteger('BitmapWidth', FBitmapWidth);
			CloseKey;
		end;
	end;
	Reg.Free;
end;

procedure TPBWatcher.SaveAllSettings;
begin
	if FPBWatchForm.SameSettings.Checked then
	begin
		SaveSettings('Run');
		SaveSettings('Design');
	end
	else SaveSettings(Mode);
end;

//  ----------------- PBWatchForm ------------------------
procedure TPBWatchForm.FormCreate(Sender: TObject);
begin
	WatchEdit.MaxLength := 16777216; // 16 Mb (don't eat all the memory)
end;

procedure TPBWatchForm.Enabled1Click(Sender: TObject);
begin
	PBWatcherObject.Enabled := not PBWatcherObject.Enabled;
end;

procedure TPBWatchForm.AddToWatches1Click(Sender: TObject);
begin
	PBWatcherObject.AddToWatches := not PBWatcherObject.AddToWatches;
end;

procedure TPBWatchForm.AutoSavewatches1Click(Sender: TObject);
begin
	PBWatcherObject.AutoSaveWatch := not PBWatcherObject.AutoSaveWatch;
end;

procedure TPBWatchForm.BitmapHeight1Click(Sender: TObject);
var
	InputStr : string;
begin
	InputStr := InputBox('Bitmap height:',
		'Type the max height of watched bitmaps!'#10#13'A value of 0 (zero)'
		+ ' will autosize the bitmaps'#10#13'- determined by the first in the list.',
		IntToStr(PBWatcher.BitmapHeight));
	if InputStr <> IntToStr(PBWatcher.BitmapHeight)
		then PBWatcher.BitmapHeight := StrToIntDef(InputStr, 0);
end;

procedure TPBWatchForm.BitmapWidth1Click(Sender: TObject);
var
	InputStr : string;
begin
	InputStr := InputBox('Bitmap width:',
		'Type the max width of watched bitmaps!'#10#13'A value of 0 (zero)'
		+ ' will autosize the bitmaps'#10#13'- determined by the first in the list.',
		IntToStr(PBWatcher.BitmapWidth));
	if InputStr <> IntToStr(PBWatcher.BitmapWidth)
		then PBWatcher.BitmapWidth := StrToIntDef(InputStr, 0);
end;

procedure TPBWatchForm.Includecounter1Click(Sender: TObject);
begin
	PBWatcherObject.IncludeCounter := not PBWatcherObject.IncludeCounter;
end;

procedure TPBWatchForm.Includetime1Click(Sender: TObject);
begin
	if PBWatcherObject.IncludeTime = itPBNone then PBWatcherObject.IncludeTime := itPBTime
	else if PBWatcherObject.IncludeTime = itPBTime then PBWatcherObject.IncludeTime := itPBTickCount
	else PBWatcherObject.IncludeTime := itPBNone;
end;

procedure TPBWatchForm.PauseBetweenWatches1Click(Sender: TObject);
begin
	PBWatcherObject.PauseBetweenWatches := StrToIntDef(InputBox
		('Pause between watches', 'Minimum milliseconds between watches:',
		IntToStr(PBWatcherObject.FPauseBetweenWatches)),
		PBWatcherObject.FPauseBetweenWatches);
end;

procedure TPBWatchForm.Skipwatches1Click(Sender: TObject);
begin
	PBWatcherObject.Skipwatches := StrToIntDef(InputBox
		('Skip watches', 'Number of watches to skip at start:',
		IntToStr(PBWatcherObject.FSkipWatches)),
		PBWatcherObject.FSkipWatches);
end;

procedure TPBWatchForm.Maxwatches1Click(Sender: TObject);
begin
	PBWatcherObject.Maxwatches := StrToIntDef(InputBox
		('Max watches', 'Maximum number of watches:',
		IntToStr(PBWatcherObject.FMaxWatches)),
		PBWatcherObject.FMaxWatches);
end;

procedure TPBWatchForm.ClearWatch1Click(Sender: TObject);
begin
	WatchEdit.Clear;
	ImageList1.Clear;
	BitmapUpDown.Max := -1;
	BitmapUpDown.Min := -1;
	BitmapUpDown.Position := -1;
	BitmapEdit.ReadOnly := True;
	BitmapEditChange(Self);
end;

procedure TPBWatchForm.Open1Click(Sender: TObject);
begin
	with PBWatcherObject do
	begin
		OpenDialog1.InitialDir := FFileDir;
		OpenDialog1.FileName := FFileName;
		if OpenDialog1.Execute then
		begin
			FFileDir := ExtractFilePath(OpenDialog1.FileName);
			FFileName := ChangeFileExt(ExtractFileName(OpenDialog1.FileName),'.txt');
			SaveAllSettings;
			LoadWatches(FFileDir, FFileName);
		end;
	end;
end;

procedure TPBWatchForm.Save1Click(Sender: TObject);
begin
	with PBWatcherObject do
	begin
		SaveDialog1.InitialDir := FFileDir;
		SaveDialog1.FileName := FFileName;
		if SaveDialog1.Execute then
		begin
			FFileDir := ExtractFilePath(SaveDialog1.FileName);
			FFileName := ChangeFileExt(ExtractFileName(SaveDialog1.FileName),'.txt');
			PBWatcherObject.SaveAllSettings;
			SaveWatches(FFileDir, FFileName);
		end;
	end;
end;

procedure TPBWatchForm.Print1Click(Sender: TObject);
var
	TempBitmap : TBitmap;
begin
	TempBitmap := TBitmap.Create;
	try
		if PrintDialog1.Execute then
		try
			if PageControl1.ActivePage = TabSheet1
				then WatchEdit.Print('PBWatcher ' + DateTimeToStr(Now))
			else PrintWysiwygBitmap(WatchImage.Picture.Bitmap);
		except
			MessageBeep(0);
		end;
	finally
		TempBitmap.Free;
	end;
end;

procedure TPBWatchForm.StayOnTop1Click(Sender: TObject);
begin
	PBWatcherObject.StayOnTop := not PBWatcherObject.StayOnTop;
end;

procedure TPBWatchForm.Recreate(const OnTop : TFormStyle);
begin
	FormStyle := OnTop;
	RecreateWnd;
end;

procedure TPBWatchForm.ResetCounterPause1Click(Sender: TObject);
var
	t : integer;
begin
	for t := 0 to _PBMAX_VARIABLES do with PBWatcherObject.FWatchRecords[t] do
	begin
		TotalCounter := 0;
		WatchCounter := 0;
		LastTime := 0;
	end;
end;

procedure TPBWatchForm.FormShowHide(Sender: TObject);
begin
	if PBWatcherObject <> nil then
	begin
		PBWatcherObject.FVisible := Visible;
		if FromMessage then FromMessage := False
		else if (not IsCreating) and not (csDestroying in ComponentState)
			then PBWatcherObject.SaveAllSettings;
		if IsDesigning then
		begin
			if PBWatchMenuAdder <> nil then PBWatchMenuAdder.PBWatchMenu.Checked := Visible;
		end;
		if Visible then BringToFront;
	end;
end;

procedure TPBWatchForm.FormResize(Sender: TObject);
begin
	if not (csDestroying in ComponentState) and (PBWatcherObject <> nil) then
	begin
		if not IsCreating then PBWatcherObject.SaveAllSettings;
		WatchEdit.ScrollBars := WatchEdit.ScrollBars; 
	end;
end;

procedure TPBWatchForm.SendKeys(Keys : string);
var
	t, KeyScan : SmallInt;
begin
	if Keys <> '' then
	begin
		for t := 1 to Length(Keys) do
		begin
			KeyScan := VkKeyScan(Keys[t]);
			SendVKey(KeyScan);
		end;
	end;
end;

procedure TPBWatchForm.SendVKey(Key : Word);
begin
	if Win32Platform = VER_PLATFORM_WIN32_NT then
	begin
		if (HiByte(Key) and 1) = 1 then Keybd_Event(VK_LSHIFT, MapVirtualKey(VK_LSHIFT, 0), 0, 0);
		if (HiByte(Key) and 2) = 2 then Keybd_Event(VK_LCONTROL, MapVirtualKey(VK_LCONTROL, 0), 0, 0);
		if (HiByte(Key) and 4) = 4 then Keybd_Event(VK_LMENU, MapVirtualKey(VK_LMENU, 0), 0, 0);
		Keybd_Event(LoByte(Key), MapVirtualKey(LoByte(Key), 0), 0, 0);
		Keybd_Event(LoByte(Key), MapVirtualKey(LoByte(Key), 0), KEYEVENTF_KEYUP, 0);
		if (HiByte(Key) and 4) = 4 then Keybd_Event(VK_LMENU, MapVirtualKey(VK_LMENU, 0), KEYEVENTF_KEYUP, 0);
		if (HiByte(Key) and 2) = 2 then Keybd_Event(VK_LCONTROL, MapVirtualKey(VK_LCONTROL, 0), KEYEVENTF_KEYUP, 0);
		if (HiByte(Key) and 1) = 1 then Keybd_Event(VK_LSHIFT, MapVirtualKey(VK_LSHIFT, 0), KEYEVENTF_KEYUP, 0);
	end
	else
	begin
		if (HiByte(Key) and 1) = 1 then Keybd_Event(VK_SHIFT, MapVirtualKey(VK_SHIFT, 0), 0, 0);
		if (HiByte(Key) and 2) = 2 then Keybd_Event(VK_CONTROL, MapVirtualKey(VK_CONTROL, 0), 0, 0);
		if (HiByte(Key) and 4) = 4 then Keybd_Event(VK_MENU, MapVirtualKey(VK_MENU, 0), 0, 0);
		Keybd_Event(LoByte(Key), MapVirtualKey(LoByte(Key), 0), 0, 0);
		Keybd_Event(LoByte(Key), MapVirtualKey(LoByte(Key), 0), KEYEVENTF_KEYUP, 0);
		if (HiByte(Key) and 4) = 4 then Keybd_Event(VK_MENU, MapVirtualKey(VK_MENU, 0), KEYEVENTF_KEYUP, 0);
		if (HiByte(Key) and 2) = 2 then Keybd_Event(VK_CONTROL, MapVirtualKey(VK_CONTROL, 0), KEYEVENTF_KEYUP, 0);
		if (HiByte(Key) and 1) = 1 then Keybd_Event(VK_SHIFT, MapVirtualKey(VK_SHIFT, 0), KEYEVENTF_KEYUP, 0);
	end;
end;

procedure TPBWatchForm.InsertLine(Sender : TObject; WatchProc : string);
var
	EditWindow : TForm;
	VarName, TempVarName, Line : string;
	t : integer;
	MenuItem : TMenuItem;
begin
	if Sender is TMenuItem then
	begin
		MenuItem := TMenuItem(Sender);
		EditWindow := GetEditWindow;
		if EditWindow <> nil then
		begin
			EditWindow.SetFocus;
			if EditWindow.Active then
			begin
				if RestoreClipBoard1.Checked then GetClipBoard;
				ClipBoard.Clear;
				Application.ProcessMessages;
				SendVKey(512 + Ord('C'));
				Application.ProcessMessages;
				TempVarName := ClipBoard.AsText;
				VarName := '';
				t := 1;
				while t <= Length(TempVarName) do
				begin
					if TempVarName[t] = '''' then VarName := VarName + ''''''
					else if TempVarName[t] = ' ' then VarName := VarName + '_'
					else VarName := VarName + TempVarName[t];
					Inc(t);
				end;
				Line := PBWatcherObject.FConditionalWatch;
				if Line <> '' then if Line[Length(Line)] <> ' ' then Line := Line + ' ';
				if WatchProc = 'WatchCanvasRect' then Line := Line
					+ 'PBWatcher.WatchCanvasRect(''' + VarName + ''', ' + ClipBoard.AsText
					+ ', ' + ClipBoard.AsText + '.ClipRect);'
				else if WatchProc = 'WatchEnum' then Line := Line
					+ 'PBWatcher.WatchEnum(''' + VarName + ''', ' + ClipBoard.AsText
					+ ', TypeInfo(T' + ClipBoard.AsText + '));'
				else if WatchProc = 'WatchSet' then Line := Line
					+ 'PBWatcher.WatchSet(''' + VarName + ''', ' + ClipBoard.AsText
					+ ', TypeInfo(T' + ClipBoard.AsText + '));'
				else if (WatchProc = 'WatchTimer') then Line := Line
					+ 'PBWatcher.WatchTimer(''' + VarName + ''');'
				else if (WatchProc = 'ResetTimer')
					then Line := Line + 'PBWatcher.ResetTimer;'
				else if (WatchProc = 'WatchPointer') then Line := Line
					+ 'PBWatcher.WatchPointer(''' + VarName + ''', '
					+ ClipBoard.AsText + ', 0);'
				else Line := Line + 'PBWatcher.' + WatchProc + '(''' + VarName + ''', '
					+ ClipBoard.AsText + ');';
				if MenuItem.Name = 'AddClipBoard' then ClipBoard.AsText := Line + #13#10
				else
				begin
					if MenuItem.Name = 'AddAbove' then SendVKey(VK_UP);
					SendVKey(VK_END);
					SendVKey(VK_RETURN);
					SendVKey(VK_HOME);
					SendKeys(Line);
					SendVKey(VK_HOME);
					if RestoreClipBoard1.Checked then RestoreClipBoard
					else ClipBoard.AsText := Line + #13#10;
				end;
			end;
		end;
	end;
end;

procedure TPBWatchForm.RemoveWatchesClick(Sender: TObject);
var
	EditWindow : TForm;
	t : integer;
begin
	EditWindow := GetEditWindow;
	if EditWindow <> nil then
	begin
		EditWindow.SetFocus;
		Application.ProcessMessages;
		if EditWindow.Active then
		begin
			CopyAllToWatcher;
			EditWindow.Cursor := crHourGlass;
			Screen.Cursor := crHourGlass;
			with WatchEdit do
			begin
				SelLength := 0;
				t := 0;
				while t < Lines.Count do
				begin
					if Pos('PBWatcher.', Lines[t]) <> 0 then Lines.Delete(t)
					else Inc(t);
				end;
				SelectAll;
				CopyToClipboard;
				Clear;
				Lines.EndUpdate;
			end;
			EditWindow.SetFocus;
			if EditWindow.Active then
			begin
				PasteAllFromWatcher;
				ShowMessage('PBWatches have been removed.'#10#13
					+'You can ''undo'' if you want.');
			end;
		end;
		Screen.Cursor := crDefault;
		EditWindow.Cursor := crDefault;
	end;
end;

procedure TPBWatchForm.AddToUsesClauseClick(Sender: TObject);
var
	EditWindow : TForm;
	Find0, Find1, Find2 : integer;
begin
	EditWindow := GetEditWindow;
	if EditWindow <> nil then
	begin
		EditWindow.SetFocus;
		if EditWindow.Active then
		begin
			CopyAllToWatcher;
			EditWindow.Cursor := crHourGlass;
			Screen.Cursor := crHourGlass;
			with WatchEdit do
			begin
				Find0 := FindText(#10'uses', 0, Length(Text), [stWholeWord]);
				if Find0 = -1 then Find0 := FindText(#13'uses', 0, Length(Text),
					[stWholeWord]);
				if Find0 <> -1 then
				begin
					Find2 := FindText(';', Find0, Length(Text), []);
					Find1 := FindText('PBWatcherUnit', Find0, Find2 - Find0, []);
					if Find1 <> -1 then
					begin
						Clear;
						Exit;
					end;
					if Find2 <> -1 then
					begin
						SelStart := Find2;
						SelLength := 0;
						SelText := ', PBWatcherUnit';
					end;
				end;
				SelectAll;
				CopyToClipboard;
				Clear;
				Lines.EndUpdate;
			end;
			EditWindow.SetFocus;
			if EditWindow.Active then	PasteAllFromWatcher;
		end;
		Screen.Cursor := crDefault;
	end;
end;

procedure TPBWatchForm.RemoveFromUsesClauseClick(Sender: TObject);
var
	EditWindow : TForm;
	Find0, Find1, Find2, Find3, Find4 : integer;
	Sep : Char;
begin
	EditWindow := GetEditWindow;
	if EditWindow <> nil then
	begin
		EditWindow.SetFocus;
		if EditWindow.Active then
		begin
			CopyAllToWatcher;
			EditWindow.Cursor := crHourGlass;
			Screen.Cursor := crHourGlass;
			with WatchEdit do
			begin
				SelLength := 0;
				Find0 := FindText(#10'uses', 0, Length(Text), [stWholeWord]);
				if Find0 = -1 then Find0 := FindText(#13'uses', 0, Length(Text),
					[stWholeWord]);
				if Find0 <> -1 then
				begin
					Find4 := FindText(';', Find0, Length(Text), []);
					Find2 := FindText('PBWatcherUnit', Find0, Find4 - Find0, []);
					Find3 := FindText(',', Find2, Find4 - Find2, []);
					Find1 := FindText(',', Find0, Find2 - Find0, []);
					if (Find2 <> -1) and (Find4 > Find2) then
					begin
						SelStart := Find2;
						SelLength := 13;
						ClearSelection;
						if Find1 <> -1 then
						repeat
							SelStart := SelStart - 1;
							SelLength := 1;
							if SelLength = 0 then SelLength := 2;
							Sep := SelText[1];
							if (Sep in [',', ' ', #9, #10, #13]) then	ClearSelection;
						until Sep = ','
						else if Find3 <> -1 then
						repeat
							SelLength := 1;
							if SelLength = 0 then SelLength := 2;
							Sep := SelText[Length(SelText)];
							if (Sep in [',', ' ', #9, #10, #13]) then	ClearSelection;
						until not (Sep in [',', ' ', #9, #10, #13]);
					end;
				end;
				SelectAll;
				CopyToClipboard;
				Clear;
				Lines.EndUpdate;
			end;
			EditWindow.SetFocus;
			if EditWindow.Active then
			begin
				PasteAllFromWatcher;
				ShowMessage('PBWatcherUnit has been removed from uses clause.'#10#13
					+'You can ''undo'' if you want.');
			end;
		end;
		Screen.Cursor := crDefault;
	end;
end;

procedure TPBWatchForm.WndProc(var Msg : TMessage);
begin
	with Msg do
	begin
		if not (csDestroying in ComponentState) and (PBWatcherObject <> nil) then
		begin
			if (Msg = PBWatcherHideShowMessage) then
			begin
				if IsDesigning then
				begin
					if (wParam = SHOWWATCHFORM) then
					begin
						IsCreating := True;
						PBWatcherObject.LoadSettings(Mode);
						IsCreating := False;
						if TempVisible and (not Visible) then
						begin
							FromMessage := True;
							Show;
							if PBWatcherObject.FAutoSaveWatch then OpenRunModewatchfile1Click(Self);
						end;
					end
					else if (wParam = HIDEWATCHFORM) then
					begin
						TempVisible := Visible;
						if Visible then
						begin
							FromMessage := True;
							Hide;
						end;
					end;
				end;
			end
			else if (Msg = WM_MOVE) and (not IsCreating) then PBWatcherObject.SaveAllSettings;
		end;
	end;
	inherited WndProc(Msg);
end;

procedure TPBWatchForm.Help1Click(Sender: TObject);
begin
	if FileExists(PBWatcherObject.FHelpFile) then ShellExecute(Handle, nil,
		PChar(PBWatcherObject.FHelpFile), nil, nil, SW_SHOWNORMAL);
end;

procedure TPBWatchForm.Findtext1Click(Sender: TObject);
begin
	TempEnabled := PBWatcherObject.Enabled;
	PBWatcherObject.Enabled := False;
	HasFoundOne := False;
	FindDialog1.Execute;
end;

procedure TPBWatchForm.FindDialog1Find(Sender: TObject);
var
	FoundAt: LongInt;
	StartPos, ToEnd: integer;
	SearchHow : TSearchTypes;
begin
	with WatchEdit do
	begin
		if SelLength <> 0 then StartPos := SelStart + SelLength
		else StartPos := 0;
		ToEnd := Length(Text) - StartPos;
		SearchHow := [];
		if (frMatchCase	in FindDialog1.Options) then SearchHow := [stMatchCase];
		if (frWholeWord	in FindDialog1.Options) then SearchHow :=
			SearchHow + [stWholeWord];
		FoundAt := FindText(FindDialog1.FindText, StartPos, ToEnd, SearchHow);
		if FoundAt <> -1 then
		begin
			HasFoundOne := True;
			SetFocus;
			SelStart := FoundAt;
			SelLength := Length(FindDialog1.FindText);
		end
		else if HasFoundOne then
		begin
			FoundAt := FindText(FindDialog1.FindText, 0, Length(Text), SearchHow);
			HasFoundOne := True;
			SetFocus;
			SelStart := FoundAt;
			SelLength := Length(FindDialog1.FindText);
		end
		else SelLength := 0;
	end;
end;

procedure TPBWatchForm.FindDialog1Close(Sender: TObject);
begin
	if TempEnabled then PBWatcherObject.Enabled := True;
	HasFoundOne := False;
end;

procedure TPBWatchForm.RestoreClipBoard1Click(Sender: TObject);
begin
	RestoreClipBoard1.Checked := not RestoreClipBoard1.Checked;
	if not IsCreating then PBWatcherObject.SaveAllSettings;
end;

procedure TPBWatchForm.CopyAllToWatcher;
var
	PText : PChar;
	ClipBoardHandle : THandle;
begin
	if RestoreClipBoard1.Checked then GetClipBoard;
{$IFDEF COMPILER_MAX_5}
	SendVKey(512 + VK_HOME);
	Application.ProcessMessages;
	SendVKey(512 + 256 + VK_END);
{$ELSE}
	SendVKey(512 + Ord('A'));
{$ENDIF}
	Application.ProcessMessages;
	SendVKey(512 + Ord('C'));
	Application.ProcessMessages;
	SendVKey(512 + VK_HOME);
	Screen.Cursor := crHourGlass;
	with WatchEdit do
	begin
		Lines.BeginUpdate;
		Clear;
		ClipBoard.Open;
		try
			ClipBoardHandle := Clipboard.GetAsHandle(CF_TEXT);
			PText := GlobalLock(ClipBoardHandle);
			Text := StrPas(PText);
			GlobalUnlock(ClipBoardHandle);
		finally
			ClipBoard.Close;
		end;
		Application.ProcessMessages;
		Screen.Cursor := crHourGlass;
	end;
end;

procedure TPBWatchForm.PasteAllFromWatcher;
begin
{$IFDEF COMPILER_MAX_5}
	SendVKey(512 + VK_HOME);
	Application.ProcessMessages;
	SendVKey(512 + 256 + VK_END);
{$ELSE}
	SendVKey(512 + Ord('A'));
{$ENDIF}
	Application.ProcessMessages;
	SendVKey(512 + Ord('V'));
	Application.ProcessMessages;
	ClipBoard.Clear;
	SendVKey(512 + VK_HOME);
	Application.ProcessMessages;
	if RestoreClipBoard1.Checked then RestoreClipBoard;
end;

function TPBWatchForm.GetEditWindow : TForm;
var
	WinHandle1, WinHandle2 : HWnd;
	PText : array[0..256] of Char;
begin
	WinHandle1 := GetWindow(Self.Handle, GW_HWNDFIRST);
	repeat
		WinHandle2 := GetWindow(WinHandle1, GW_HWNDNEXT);
		if WinHandle2 <> 0 then
		begin
			WinHandle1 := WinHandle2;
			GetClassName(WinHandle2, PText, 255);
		end;
	until (PText = 'TEditWindow') or (WinHandle2 = 0);
	Result := nil;
	if (PText = 'TEditWindow') and (WinHandle2 <> 0) then
	begin
		if (FindControl(WinHandle2) is TForm)
			then Result := FindControl(WinHandle2) as TForm
		else Beep;
	end
	else Beep;
end;

procedure TPBWatchForm.OpenDesignModewatchfile1Click(Sender: TObject);
var
	Reg : TRegistry;
	FileDir, FileName : string;
begin
	if IsDesigning
		then LoadWatches(PBWatcherObject.FFileDir, PBWatcherObject.FFileName)
	else
	begin
		Reg := TRegistry.Create;
		with Reg do
		begin
			if OpenKey('\Software\Borland\Delphi\PBWatcher\PBWatcher - [Designmode]', False) then
			begin
				if ValueExists('FileDir') then FileDir := ReadString('FileDir');
				if ValueExists('FileName') then FileName := ReadString('FileName');
				LoadWatches(FileDir, FileName);
			end
			else MessageBeep(0);
			Free;
		end;
	end;
end;

procedure TPBWatchForm.OpenRunModewatchfile1Click(Sender: TObject);
var
	Reg : TRegistry;
	FileDir, FileName : string;
begin
	if not IsDesigning
		then LoadWatches(PBWatcherObject.FFileDir, PBWatcherObject.FFileName)
	else
	begin
		Reg := TRegistry.Create;
		with Reg do
		begin
			if OpenKey('\Software\Borland\Delphi\PBWatcher\PBWatcher - [Runmode]', False) then
			begin
				if ValueExists('FileDir') then FileDir := ReadString('FileDir');
				if ValueExists('FileName') then FileName := ReadString('FileName');
				LoadWatches(FileDir, FileName);
			end
			else MessageBeep(0);
			Free;
		end;
	end;
end;

procedure TPBWatchForm.SameSettingsClick(Sender: TObject);
begin
	SameSettings.Checked := not SameSettings.Checked;
	if not IsCreating then
	begin
		PBWatcherObject.SaveSettings('Run');
		PBWatcherObject.SaveSettings('Design');
	end;
end;

procedure TPBWatchForm.GetClipBoard;
var
	ClipHandle : THandle;
	PTemp : PChar;
begin
	StrDispose(ClipText);
	ClipText := nil;
	ClipBoard.Open;
	try
		ClipHandle := Clipboard.GetAsHandle(CF_TEXT);
		PTemp := GlobalLock(ClipHandle);
		ClipText := StrNew(PTemp);
		GlobalUnlock(ClipHandle);
	finally
		Clipboard.Close;
	end;
end;

procedure TPBWatchForm.RestoreClipBoard;
begin
	if ClipText <> nil then
	begin
		ClipBoard.SetTextBuf(ClipText);
		StrDispose(ClipText);
		ClipText := nil;
	end;
end;

procedure TPBWatchForm.ConditionalWatchClick(Sender: TObject);
begin
	PBWatcherObject.FConditionalWatch := InputBox('Conditional watch:',
		'Type the prefix to ''AutoAdd'':', PBWatcherObject.FConditionalWatch);
	if Length(PBWatcherObject.FConditionalWatch) > 25
		then ConditionalWatch.Caption := 'Conditional watch (='''
		+ Copy(PBWatcherObject.FConditionalWatch, 1, 22) + '...'')'
	else ConditionalWatch.Caption := 'Conditional watch (='''
		+ PBWatcherObject.FConditionalWatch + ''')';
	PBWatcherObject.SaveAllSettings;
end;

procedure TPBWatchForm.TransparentBitmapClick(Sender: TObject);
begin
	TransparentBitmap.Checked := not TransparentBitmap.Checked;
	PBWatcherObject.SaveAllSettings;
end;

procedure TPBWatchForm.BackColorClick(Sender: TObject);
begin
	if PBBackColorDialog.Execute then
	begin
		SetBackcolors;
		BitmapEditChange(Self);
		PBWatcherObject.SaveAllSettings;
	end;
end;

procedure TPBWatchForm.SetBackColors;
begin
	ScrollBox1.Color := PBBackColorDialog.Color;
	WatchImage.Picture.Bitmap.TransparentColor := PBBackColorDialog.Color;
{$IFNDEF COMPILER_MAX_3}
	ColorBitmap.Canvas.Brush.Color := PBBackColorDialog.Color;
	ColorBitmap.Canvas.FillRect(ColorBitmap.Canvas.ClipRect);
	ColorBitmap.Canvas.Brush.Color := clBlack;
	ColorBitmap.Canvas.FrameRect(ColorBitmap.Canvas.ClipRect);
	BackColor.Bitmap := ColorBitmap;
{$ENDIF}
end;

procedure TPBWatchForm.PageControl1Change(Sender: TObject);
begin
	if PageControl1.ActivePage = TabSheet1 then
	begin
		StaticText1.Visible := False;
		BitmapEdit.Visible := False;
		BitmapUpDown.Visible := False;
	end
	else
		begin
		StaticText1.Visible := True;
		BitmapEdit.Visible := True;
		BitmapUpDown.Visible := True;
	end;
	if Sender = PageControl1 then PBWatcherObject.SaveAllSettings;
end;

procedure TPBWatchForm.BitmapEditChange(Sender: TObject);
begin
	with WatchImage do
	begin
		Height := ImageList1.Height;
		Width := ImageList1.Width;
		Picture.Bitmap.Height := ImageList1.Height;
		Picture.Bitmap.Width := ImageList1.Width;
		Canvas.Brush.Color := PBBackColorDialog.Color;
		Canvas.FillRect(Rect(0, 0, Width, Height));
		if StrToIntDef(BitmapEdit.Text, -1) < BitmapUpdown.Min
			then BitmapEdit.Text := IntToStr(BitmapUpDown.Min)
		else if StrToIntDef(BitmapEdit.Text, -1) > BitmapUpdown.Max
			then BitmapEdit.Text := IntToStr(BitmapUpDown.Max);
		if BitmapUpDown.Position <> -1 then
		begin
			if TransparentBitmap.Checked
				then ImageList1.Draw(Canvas, 0, 0, BitmapUpDown.Position)
			else
			begin
				ImageList1.GetBitmap(BitmapUpDown.Position, WatchImage.Picture.Bitmap);
			end;
		end
		else
		begin
			Picture.Assign(nil);
			Align := alClient;
			Align := alNone;
		end;
		Repaint;
	end;
end;

procedure TPBWatchForm.LoadWatches(FileDir, FileName : string);
var
	TempBitmap, TempBitmap2 : TBitmap;
	t, ImageCounter : integer;
	Rect0 : TRect;
begin
	TempBitmap := TBitmap.Create;
	TempBitmap2 := TBitmap.Create;
	ClearWatch1Click(Self);
	try
		try
			if FileDir[Length(FileDir)] = '\' then
			begin
				WatchEdit.Lines.LoadFromFile(FileDir + FileName);
				if FileExists(FileDir	+ ChangeFileExt(FileName, '.bmp'))
					then TempBitmap.LoadFromFile(FileDir	+ ChangeFileExt(FileName, '.bmp'));
			end
			else
			begin
				WatchEdit.Lines.LoadFromFile(FileDir + '\' + FileName);
				if FileExists(FileDir + '\'	+ ChangeFileExt(FileName, '.bmp'))
					then TempBitmap.LoadFromFile(FileDir + '\'
					+ ChangeFileExt(FileName, '.bmp'));
			end;
		except
			MessageBeep(0);
		end;
		if not TempBitmap.Empty then
		begin
			if (Integer(TempBitmap.Canvas.Pixels[0, 0]) = 0)
				or (Integer(TempBitmap.Canvas.Pixels[1, 0]) <> 0)
				or (Integer(TempBitmap.Canvas.Pixels[0, 1]) <> 0)
				then ImageCounter := 1
			else ImageCounter := Integer(TempBitmap.Canvas.Pixels[0, 0]);
			ImageList1.Height := TempBitmap.Height;
			ImageList1.Width := TempBitmap.Width div ImageCounter;
			TempBitmap2.Height := ImageList1.Height;
			TempBitmap2.Width := ImageList1.Width;
			for t := 1 to ImageCounter do
			begin
				Rect0 := Bounds((t - 1) * ImageList1.Width, 0, ImageList1.Width,
					ImageList1.Height);
				TempBitmap2.Canvas.CopyRect(Bounds(0, 0, ImageList1.Width,
					ImageList1.Height), TempBitmap.Canvas, Rect0);
				ImageList1.Add(TempBitmap2, nil);
			end;
			if ImageList1.Count > 0 then with BitmapUpDown do
			begin
				Max := ImageList1.Count - 1;
				Min := 0;
				Position := Max;
				if Position > 0 then BitmapEdit.ReadOnly := False
				else BitmapEdit.ReadOnly := True;
			end;
			BitmapEditChange(Self);
		end;
	finally
		TempBitmap2.Free;
		TempBitmap.Free;
	end;
end;

procedure TPBWatchForm.SaveWatches(FileDir, FileName : string);
var
	TempBitmap, TempBitmap2 : TBitmap;
begin
	TempBitmap := TBitmap.Create;
	TempBitmap2 := TBitmap.Create;
	try
		if ImageList1.Count > 0 then
		begin
			TempBitmap2.Handle := ImageList1.GetImageBitmap;
			if TempBitmap2.Handle <> 0 then with TempBitmap do
			begin
				Assign(TempBitmap2);
				Dormant;
				Height := ImageList1.Height;
				Width := ImageList1.Width * ImageList1.Count;
				Canvas.Pixels[0, 0] := TColor(ImageList1.Count);
			end;
		end;
		if FileDir[Length(FileDir)] = '\' then
		begin
			WatchEdit.Lines.SaveToFile(FileDir + FileName);
			if not TempBitmap.Empty
				then TempBitmap.SaveToFile(FileDir	+ ChangeFileExt(FileName, '.bmp'));
		end
		else
		begin
			WatchEdit.Lines.SaveToFile(FileDir + '\' + FileName);
			if not TempBitmap.Empty
				then TempBitmap.SaveToFile(FileDir + '\'
				+ ChangeFileExt(FileName, '.bmp'));
		end;
	finally
		TempBitmap.Free;
		TempBitmap2.Free;
	end;
end;

procedure TPBWatchForm.AddSubMenuClick(Sender: TObject);
var
	TopMenuItem, MenuItem : TMenuItem;
	t : integer;
begin
	if Sender is TMenuItem then
	begin
		TopMenuItem := TMenuItem(Sender);
		for t := 0 to TopMenuItem.Count - 1 do
		begin
			MenuItem := TopMenuItem.Items[t];
			while MenuItem.Count > 0 do MenuItem.Delete(0);
			MenuItem.Add(NewItem('Add &above', 0, False, True, AutoAddClick, 0,
				'AddAbove'));
			MenuItem.Add(NewItem('Add &below', 0, False, True, AutoAddClick, 0,
				'AddBelow'));
			MenuItem.Add(NewItem('Add &Clipboard', 0, False, True, AutoAddClick, 0,
				'AddClipBoard'));
		end;
	end;
end;

procedure TPBWatchForm.AutoAddClick(Sender: TObject);
var
	WatchProc : string;
	MenuItem : TMenuItem;
begin
	if Sender is TMenuItem then
	begin
		MenuItem := TMenuItem(Sender).Parent;
		WatchProc := StripHotKey(MenuItem.Caption);
		InsertLine(Sender, WatchProc);
	end;
end;

procedure TPBWatchForm.WatchImageMouseMove(Sender: TObject;
	Shift: TShiftState; X, Y: Integer);
var
	Color : TColor;
begin
	Color := WatchImage.Canvas.Pixels[X, Y];
	WatchImage.Hint := ' R:' + IntToStr(GetRValue(Color))
		+ ' G:' + IntToStr(GetGValue(Color))
		+ ' B:' + IntToStr(GetBValue(Color));
//	Application.CancelHint;
	Application.ActivateHint(ClientToScreen(Point(X, Y)));
end;

//  ----------------- PBWatchTrayIcon ------------------------
constructor TPBWatchTrayIcon.Create;
var
	DesignModeWindow : HWnd;
begin
	inherited;
	FHWnd := AllocateHWnd(WndProc);
	ZeroMemory(@FIconData, SizeOf(TNotifyIconDataEx));
	with FIconData do
	begin
		cbSize := SizeOf(TNotifyIconDataEx);
		wnd := FHWnd;
		uID := 0;
		uFlags := NIF_MESSAGE + NIF_ICON + NIF_TIP;
		szTip := 'PBWatcher - click to show/hide !';
		hIcon := PBWatcherObject.FPBWatchForm.Icon.Handle;
		uCallbackMessage := WM_PBTOOLTRAYICON;
	end;
	ShowIcon;
	if not IsDesigning then
	begin
		DesignModeWindow := FindWindow('TPBWatchForm', 'PBWatcher - [DesignMode]');
		if DesignModeWindow <> 0 then SendMessage(DesignModeWindow, PBWatcherHideShowMessage, HIDEWATCHFORM, 0);
	end;
end;

destructor TPBWatchTrayIcon.Destroy;
var
	DesignModeWindow : HWnd;
begin
	if not IsDesigning then
	begin
		DesignModeWindow := FindWindow('TPBWatchForm', 'PBWatcher - [DesignMode]');
		if DesignModeWindow <> 0 then SendMessage(DesignModeWindow, PBWatcherHideShowMessage, SHOWWATCHFORM, 0);
	end;
	if FVisible then HideIcon;
	DeAllocateHWnd(FHWnd);
	inherited Destroy;
end;

procedure TPBWatchTrayIcon.WndProc(var Msg : TMessage);
begin
	with Msg do
	begin
		if (Msg = WM_PBTOOLTRAYICON) and (lParam = WM_LBUTTONDOWN)
			then with PBWatcherObject.FPBWatchForm do Visible := not Visible
		else Result := DefWindowProc(FHWnd, Msg, wParam, lParam);
	end;
end;

function TPBWatchTrayIcon.ShowIcon;
begin
	Result := Shell_NotifyIcon(NIM_ADD,@FIconData);
	FVisible := Result;
end;

function TPBWatchTrayIcon.HideIcon;
begin
	Result := Shell_NotifyIcon(NIM_DELETE,@FIconData);
	FVisible := Result;
end;

//  ----------------- PBWatchMenuAdder --------------------------
constructor TPBWatchMenuAdder.Create;
begin
	inherited;
	if Application.FindComponent('AppBuilder') <> nil
		then ViewMenu := Application.FindComponent('AppBuilder').FindComponent('ViewsMenu') as TMenuItem
	else ViewMenu := nil;
	if ViewMenu <> nil then
	begin
		N1 := NewLine;
		ViewMenu.Add(N1);
		PBWatchMenu := NewItem('PBWatcher Window', ShortCut(Word('W'), [ssCtrl]),
		False, True, PBWatchMenuClick, 0, 'PBWatchMenu');
		ViewMenu.Add(PBWatchMenu);
	end;
end;

destructor TPBWatchMenuAdder.Destroy;
begin
	if ViewMenu <> nil then
	begin
		if PBWatchMenu <> nil then
		begin
			if ViewMenu.IndexOf(PBWatchMenu) <> -1
				then ViewMenu.Delete(ViewMenu.IndexOf(PBWatchMenu));
			PBWatchMenu.Free;
			PBWatchMenu := nil;
		end;
		if N1 <> nil then
		begin
			if ViewMenu.IndexOf(N1) <> -1
				then ViewMenu.Delete(ViewMenu.IndexOf(N1));
			N1.Free;
			N1 := nil;
		end;
	end;
	inherited;
end;

procedure TPBWatchMenuAdder.PBWatchMenuClick(Sender : TObject);
begin
	PBWatchMenu.Checked := not PBWatchMenu.Checked;
	if PBWatcherObject <> nil then
	begin
		PBWatcherObject.FPBWatchForm.Visible := PBWatchMenu.Checked;
	end;
end;

//  ------------------- Initialization --------------------------
initialization
begin
	if CmdLine = nil then IsDesigning := True
	else if Pos('\DELPHI32.EXE', UpperCase(ParamStr(0))) > 0 then IsDesigning := True;
	if (RefCount = 0) then
	begin
		if IsDesigning and (PBWatchMenuAdder = nil)
			then PBWatchMenuAdder := TPBWatchMenuAdder.Create;
		if PBWatcherObject = nil then PBWatcherObject := TPBWatcher.Create;
	end
	else Inc(RefCount);
end;

//  ------------------- Finalization ----------------------------
finalization
begin
	if RefCount <= 1 then
	begin
		StrDispose(ClipText);
		PBWatcherObject.Free;
		PBWatcherObject := nil;
		if IsDesigning then
		begin
			PBWatchMenuAdder.Free;
			PBWatchMenuAdder := nil;
		end;
	end
	else Dec(RefCount);
end;

end.

