unit LaydockDBNavigator;
{$R-,H+,X+}
interface
uses Windows, SysUtils, Messages, Classes, Controls, Forms, Dialogs, DBCtrls,
     Graphics, StdCtrls, ExtCtrls, Mask, ComCtrls, Db, Math, Buttons;

resourcestring
  SFirstRecord = ' ';
  SPriorRecord = ' ';
  SNextRecord = ' ';
  SLastRecord = ' ';
  SInsertRecord = ' ';
  SDeleteRecord = ' ';
  SEditRecord = ' ';
  SPostEdit = ' ';
  SCancelEdit = ' ';
  SRefreshRecord = ' ';
  SDeleteRecordQuestion = '  ?';
  SDeleteMultipleRecordsQuestion = '    ?';
  SRecordNotFound = '  ';

  SCFirstRecord = '';
  SCPriorRecord = '';
  SCNextRecord = '';
  SCLastRecord = '';
  SCInsertRecord = '';
  SCDeleteRecord = '';
  SCEditRecord = '';
  SCPostEdit = '';
  SCCancelEdit = '';
  SCRefreshRecord = '';

{ TFieldDataLink }
type
  TFieldDataLink = class(TDataLink)
  private
    FField: TField;
    FFieldName: string;
    FControl: TComponent;
    FEditing: Boolean;
    FModified: Boolean;
    FOnDataChange: TNotifyEvent;
    FOnEditingChange: TNotifyEvent;
    FOnUpdateData: TNotifyEvent;
    FOnActiveChange: TNotifyEvent;
    function GetCanModify: Boolean;
    procedure SetEditing(Value: Boolean);
    procedure SetField(Value: TField);
    procedure SetFieldName(const Value: string);
    procedure UpdateField;
    procedure UpdateRightToLeft;
  protected
    procedure ActiveChanged; override;
    procedure EditingChanged; override;
    procedure FocusControl(Field: TFieldRef); override;
    procedure LayoutChanged; override;
    procedure RecordChanged(Field: TField); override;
    procedure UpdateData; override;
  public
    constructor Create;
    function Edit: Boolean;
    procedure Modified;
    procedure Reset;
    property CanModify: Boolean read GetCanModify;
    property Control: TComponent read FControl write FControl;
    property Editing: Boolean read FEditing;
    property Field: TField read FField;
    property FieldName: string read FFieldName write SetFieldName;
    property OnDataChange: TNotifyEvent read FOnDataChange write FOnDataChange;
    property OnEditingChange: TNotifyEvent read FOnEditingChange write FOnEditingChange;
    property OnUpdateData: TNotifyEvent read FOnUpdateData write FOnUpdateData;
    property OnActiveChange: TNotifyEvent read FOnActiveChange write FOnActiveChange;
  end;

  TMyNavButton = class;
  TMyNavDataLink = class;
  TMyNavGlyph = (ngEnabled, ngDisabled);
  TMyNavigateBtn = (nbFirst, nbPrior, nbNext, nbLast,
                  nbInsert, nbDelete, nbEdit, nbPost, nbCancel, nbRefresh);
  TMyButtonSet = set of TMyNavigateBtn;
  TMyNavButtonStyle = set of (nsAllowTimer, nsFocusRect);

  ENavClick = procedure (Sender: TObject; Button: TMyNavigateBtn) of object;

{ TDBLaydockNavigator }

  TDBLaydockNavigator = class (TCustomPanel)
  private
    FDataLink: TMyNavDataLink;
    FVisibleButtons: TMyButtonSet;
    FHints: TStrings;
    FDefHints: TStrings;
    FDefCaptions: TStrings;
    FCaptions: TStrings;
    ButtonWidth: Integer;
    MinBtnSize: TPoint;
    FOnNavClick: ENavClick;
    FBeforeAction: ENavClick;
    FocusedButton: TMyNavigateBtn;
    FConfirmDelete: Boolean;
    FFlat: Boolean;
    FShowCaption: Boolean;
    FProgressBar : TProgressBar;
    FLabel : TLabel;
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure MyMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ClickHandler(Sender: TObject);
    function GetDataSource: TDataSource;
    function GetHints: TStrings;
    procedure HintsChanged(Sender: TObject);
    function GetCaptions: TStrings;
    procedure CaptionsChanged(Sender: TObject);
    procedure InitButtons;
    procedure InitHints;
    procedure InitCaptions;
    procedure InitProgressBar;
    procedure InitLabel;
    procedure SetDataSource(Value: TDataSource);
    procedure SetFlat(Value: Boolean);
    procedure SetShowCaption(Value: Boolean);
    procedure SetHints(Value: TStrings);
    procedure SetCaptions(Value: TStrings);
    procedure SetSize(var W: Integer; var H: Integer);
    procedure SetVisible(Value: TMyButtonSet);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure WMWindowPosChanging(var Message: TWMWindowPosChanging); message WM_WINDOWPOSCHANGING;
  protected
    Buttons: array[TMyNavigateBtn] of TMyNavButton;
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    procedure CalcMinSize(var W, H: Integer);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TMyNavigateBtn); virtual;
  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property VisibleButtons: TMyButtonSet read FVisibleButtons write SetVisible
      default [nbFirst, nbPrior, nbNext, nbLast, nbInsert, nbDelete,
        nbEdit, nbPost, nbCancel, nbRefresh];
    property Align;
    property Anchors;
    property Constraints;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property Flat: Boolean read FFlat write SetFlat default False;
    property ShowCaption: Boolean read FShowCaption write SetShowCaption default False;
    property Ctl3D;
    property Hints: TStrings read GetHints write SetHints;
    property Captions: TStrings read GetCaptions write SetCaptions;
    property ParentCtl3D;
    property ParentColor;
    property ParentShowHint;
    property PopupMenu;
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property BeforeAction: ENavClick read FBeforeAction write FBeforeAction;
    property OnClick: ENavClick read FOnNavClick write FOnNavClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;
  end;

{ TMyNavButton }
  TMyNavButton = class(TSpeedButton)
  private
    FIndex: TMyNavigateBtn;
    FNavStyle: TMyNavButtonStyle;
    FRepeatTimer: TTimer;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    destructor Destroy; override;
    property NavStyle: TMyNavButtonStyle read FNavStyle write FNavStyle;
    property Index : TMyNavigateBtn read FIndex write FIndex;
  end;

{ TMyNavDataLink }

  TMyNavDataLink = class(TDataLink)
  private
    FNavigator: TDBLaydockNavigator;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(ANav: TDBLaydockNavigator);
    destructor Destroy; override;
  end;

procedure Register;

implementation

{$R LaydockDBNavigator}

procedure Register;
begin
  RegisterComponents('Laydock',[TDBLaydockNavigator]);
end;

{ TFieldDataLink }

constructor TFieldDataLink.Create;
begin
  inherited Create;
  VisualControl := True;
end;

procedure TFieldDataLink.SetEditing(Value: Boolean);
begin
  if FEditing <> Value then
  begin
    FEditing := Value;
    FModified := False;
    if Assigned(FOnEditingChange) then FOnEditingChange(Self);
  end;
end;

procedure TFieldDataLink.SetFieldName(const Value: string);
begin
  if FFieldName <> Value then
  begin
    FFieldName :=  Value;
    UpdateField;
  end;
end;

procedure TFieldDataLink.SetField(Value: TField);
begin
  if FField <> Value then
  begin
    FField := Value;
    EditingChanged;
    RecordChanged(nil);
    UpdateRightToLeft;
  end;
end;

procedure TFieldDataLink.UpdateField;
begin
  if Active and (FFieldName <> '') then
  begin
    if Assigned(FControl) then
      SetField(GetFieldProperty(DataSource.DataSet, FControl, FFieldName)) else
      SetField(DataSource.DataSet.FieldByName(FFieldName));
  end else
    SetField(nil);
end;

procedure TFieldDataLink.UpdateRightToLeft;
var
  IsRightAligned: Boolean;
  AUseRightToLeftAlignment: Boolean;
begin
  if Assigned(FControl) and (FControl is TWinControl) then
    with FControl as TWinControl do
      if IsRightToLeft then
      begin
        IsRightAligned :=(GetWindowLong(Handle, GWL_EXSTYLE) and WS_EX_RIGHT) = WS_EX_RIGHT;
        AUseRightToLeftAlignment := DBUseRightToLeftAlignment(TControl(FControl), Field);
        if (IsRightAligned and (not AUseRightToLeftAlignment)) or
           ((not IsRightAligned) and AUseRightToLeftAlignment) then
          Perform(CM_RECREATEWND, 0, 0);
      end;
end;

function TFieldDataLink.Edit: Boolean;
begin
  if CanModify then inherited Edit;
  Result := FEditing;
end;

function TFieldDataLink.GetCanModify: Boolean;
begin
  Result := not ReadOnly and (Field <> nil) and Field.CanModify;
end;

procedure TFieldDataLink.Modified;
begin
  FModified := True;
end;

procedure TFieldDataLink.Reset;
begin
  RecordChanged(nil);
end;

procedure TFieldDataLink.ActiveChanged;
begin
  UpdateField;
  if Assigned(FOnActiveChange) then FOnActiveChange(Self);
end;

procedure TFieldDataLink.EditingChanged;
begin
  SetEditing(inherited Editing and CanModify);
end;

procedure TFieldDataLink.FocusControl(Field: TFieldRef);
begin
  if (Field^ <> nil) and (Field^ = FField) and (FControl is TWinControl) then
    if TWinControl(FControl).CanFocus then
    begin
      Field^ := nil;
      TWinControl(FControl).SetFocus;
    end;
end;

procedure TFieldDataLink.RecordChanged(Field: TField);
begin
  if (Field = nil) or (Field = FField) then
  begin
    if Assigned(FOnDataChange) then FOnDataChange(Self);
    FModified := False;
  end;
end;

procedure TFieldDataLink.LayoutChanged;
begin
  UpdateField;
end;

procedure TFieldDataLink.UpdateData;
begin
  if FModified then
  begin
    if (Field <> nil) and Assigned(FOnUpdateData) then FOnUpdateData(Self);
    FModified := False;
  end;
end;

{ TDBLaydockNavigator }

var
  BtnTypeName: array[TMyNavigateBtn] of PChar = ('FIRST', 'PRIOR', 'NEXT',
    'LAST', 'INSERT', 'DELETE', 'EDIT', 'POST', 'CANCEL', 'REFRESH');
  BtnHintId: array[TMyNavigateBtn] of Pointer = (@SFirstRecord, @SPriorRecord,
    @SNextRecord, @SLastRecord, @SInsertRecord, @SDeleteRecord, @SEditRecord,
    @SPostEdit, @SCancelEdit, @SRefreshRecord);
  BtnCaptionId: array[TMyNavigateBtn] of Pointer = (@SCFirstRecord, @SCPriorRecord,
    @SCNextRecord, @SCLastRecord, @SCInsertRecord, @SCDeleteRecord, @SCEditRecord,
    @SCPostEdit, @SCCancelEdit, @SCRefreshRecord);

constructor TDBLaydockNavigator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption]+ [csOpaque];
  if not NewStyleControls then ControlStyle := ControlStyle + [csFramed];
  FDataLink := TMyNavDataLink.Create(Self);
  FVisibleButtons := [nbFirst, nbPrior, nbNext, nbLast, nbInsert, nbDelete, nbEdit, nbPost, nbCancel, nbRefresh];
  FHints := TStringList.Create;
  FCaptions := TStringList.Create;
  FProgressBar := TProgressBar.Create(self);
  FLabel := TLabel.Create(Self);
  TStringList(FCaptions).OnChange := CaptionsChanged;
  TStringList(FHints).OnChange := HintsChanged;
  InitButtons;
  InitHints;
  if FShowCaption then
    InitCaptions;
  InitLabel;
  InitProgressBar;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 240;
  Height := 25;
  ButtonWidth := 0;
  FocusedButton := nbFirst;
  FConfirmDelete := True;
  FullRepaint := False;
  FLabel.Transparent := True;
  FLabel.Alignment := taCenter;
  FLabel.AutoSize := False;
  FLabel.Font.Style := FLabel.Font.Style + [fsBold];
end;

destructor TDBLaydockNavigator.Destroy;
begin
  FDefHints.Free;
  FDefCaptions.Free;
  FDataLink.Free;
  FHints.Free;
  FCaptions.Free;
  FLabel.Free;
  FProgressBar.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TDBLaydockNavigator.InitProgressBar;
begin
  InsertControl(FProgressBar);
  FProgressBar.onMouseDown := MyMouseDown;
  FProgressBar.Top := 13;
  FProgressBar.Height := 10;
  FProgressBar.Width := Width;
end;

procedure TDBLaydockNavigator.InitLabel;
begin
  InsertControl(FLabel);
  FLabel.Top := 0;
  FLabel.Height := 13;
  FLabel.Width := Width;
end;

procedure TDBLaydockNavigator.InitButtons;
var
  I: TMyNavigateBtn;
  Btn: TMyNavButton;
  X: Integer;
  ResName: string;
begin
  X := 0;
  MinBtnSize := Point(20, 43);
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TMyNavButton.Create (Self);
    Btn.Caption := '';
    Btn.Flat := Flat;
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.SetBounds (X, 23, MinBtnSize.X, MinBtnSize.Y);
    FmtStr(ResName, 'dbmyn_%s', [BtnTypeName[I]]);
    Btn.Glyph.LoadFromResourceName(HInstance, ResName);
    Btn.NumGlyphs := 2;
    Btn.Enabled := True;
    Btn.OnClick := ClickHandler;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
  Buttons[nbPrior].NavStyle := Buttons[nbPrior].NavStyle + [nsAllowTimer];
  Buttons[nbNext].NavStyle  := Buttons[nbNext].NavStyle + [nsAllowTimer];
end;

procedure TDBLaydockNavigator.InitHints;
var
  I: Integer;
  J: TMyNavigateBtn;
begin
  if not Assigned(FDefHints) then
  begin
    FDefHints := TStringList.Create;
    for J := Low(Buttons) to High(Buttons) do
      FDefHints.Add(LoadResString(BtnHintId[J]));
  end;
  for J := Low(Buttons) to High(Buttons) do
    Buttons[J].Hint := FDefHints[Ord(J)];
  J := Low(Buttons);
  for I := 0 to (FHints.Count - 1) do
  begin
    if FHints.Strings[I] <> '' then Buttons[J].Hint := FHints.Strings[I];
    if J = High(Buttons) then Exit;
    Inc(J);
  end;
end;

procedure TDBLaydockNavigator.InitCaptions;
var
  I: Integer;
  J: TMyNavigateBtn;
begin
  if not Assigned(FDefCaptions) then
  begin
    FDefCaptions := TStringList.Create;
    for J := Low(Buttons) to High(Buttons) do
      FDefCaptions.Add(LoadResString(BtnCaptionId[J]));
  end;
  for J := Low(Buttons) to High(Buttons) do
    Buttons[J].Caption := FDefCaptions[Ord(J)];
  J := Low(Buttons);
  for I := 0 to (FCaptions.Count - 1) do
  begin
    if (FCaptions.Strings[I] <> '') and (not FShowCaption) then
      Buttons[J].Caption := FCaptions.Strings[I];
    if J = High(Buttons) then Exit;
    Inc(J);
  end;
end;

procedure TDBLaydockNavigator.HintsChanged(Sender: TObject);
begin
  InitHints;
end;

procedure TDBLaydockNavigator.CaptionsChanged(Sender: TObject);
begin
  InitCaptions;
end;

procedure TDBLaydockNavigator.SetFlat(Value: Boolean);
var
  I: TMyNavigateBtn;
begin
  if FFlat <> Value then
  begin
    FFlat := Value;
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Flat := Value;
  end;
end;

procedure TDBLaydockNavigator.SetShowCaption(Value: Boolean);
var
  i : TMyNavigateBtn;
begin
  if FShowCaption <> Value then
    begin
      FShowCaption := Value;
      for I := Low(Buttons) to High(Buttons) do
        if FShowCaption then
          InitCaptions
        else
          Buttons[I].Caption := ''
    end;
end;

procedure TDBLaydockNavigator.SetHints(Value: TStrings);
begin
  if Value.Text = FDefHints.Text then
    FHints.Clear
  else
    FHints.Assign(Value);
end;

function TDBLaydockNavigator.GetHints: TStrings;
begin
  if (csDesigning in ComponentState) and not (csWriting in ComponentState) and
     not (csReading in ComponentState) and (FHints.Count = 0) then
    Result := FDefHints else
    Result := FHints;
end;

procedure TDBLaydockNavigator.SetCaptions(Value: TStrings);
begin
  if Value.Text = FDefCaptions.Text then
    FCaptions.Clear
  else
    FCaptions.Assign(Value);
end;

function TDBLaydockNavigator.GetCaptions: TStrings;
begin
  if (csDesigning in ComponentState) and not (csWriting in ComponentState) and
     not (csReading in ComponentState) and (FCaptions.Count = 0) then
    Result := FDefCaptions else
    Result := FCaptions;
end;

procedure TDBLaydockNavigator.GetChildren(Proc: TGetChildProc; Root: TComponent);
begin
end;

procedure TDBLaydockNavigator.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TDBLaydockNavigator.SetVisible(Value: TMyButtonSet);
var
  I: TMyNavigateBtn;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  SetSize(W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TDBLaydockNavigator.CalcMinSize(var W, H: Integer);
var
  Count: Integer;
  I: TMyNavigateBtn;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
    if Buttons[I].Visible then
      Inc(Count);
  if Count = 0 then Inc(Count);

  W := Max(W, Count * MinBtnSize.X);
  H := Max(H, MinBtnSize.Y);

  if Align = alNone then W := (W div Count) * Count;
end;

procedure TDBLaydockNavigator.SetSize(var W: Integer; var H: Integer);
var
  Count: Integer;
  I: TMyNavigateBtn;
  Space, Temp, Remain: Integer;
  X,Y: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  CalcMinSize(W, H);

  Count := 0;
  for I := Low(Buttons) to High(Buttons) do
    if Buttons[I].Visible then
      Inc(Count);
  if Count = 0 then Inc(Count);

  ButtonWidth := W div Count;
  Temp := Count * ButtonWidth;
  if Align = alNone then W := Temp;

  X := 0;
  Y := 23;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec(Temp, Remain);
        if Temp < 0 then
        begin
          Inc(Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds(X, Y, ButtonWidth + Space, Height-y);
      Inc(X, ButtonWidth + Space);
    end
    else
      Buttons[I].SetBounds (Width + 1, Y, ButtonWidth, Height-y);
  end;
  FLabel.SetBounds(0,0,Width,13);
  FProgressBar.SetBounds(0,13,Width,10);
end;

procedure TDBLaydockNavigator.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  if not HandleAllocated then SetSize(W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TDBLaydockNavigator.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;
  W := Width;
  H := Height;
  SetSize(W, H);
end;

procedure TDBLaydockNavigator.WMWindowPosChanging(var Message: TWMWindowPosChanging);
begin
  inherited;
  if (SWP_NOSIZE and Message.WindowPos.Flags) = 0 then
    CalcMinSize(Message.WindowPos.cx, Message.WindowPos.cy);
end;

procedure TDBLaydockNavigator.ClickHandler(Sender: TObject);
begin
  BtnClick (TMyNavButton (Sender).Index);
  if Enabled and FDataLink.Active then
    begin
      FProgressBar.Max := FDataLink.DataSource.DataSet.RecordCount;
      FProgressBar.Position := FDataLink.DataSource.DataSet.RecNo;
      FLabel.Caption := IntToStr(FDataLink.DataSource.DataSet.RecNo)+'  '+IntToStr(FDataLink.DataSource.DataSet.RecordCount);
    end;
end;

procedure TDBLaydockNavigator.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TMyNavigateBtn;
begin
  OldFocus := FocusedButton;
  FocusedButton := TMyNavButton (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TDBLaydockNavigator.BtnClick(Index: TMyNavigateBtn);
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then
  begin
    if not (csDesigning in ComponentState) and Assigned(FBeforeAction) then
      FBeforeAction(Self, Index);
    with DataSource.DataSet do
    begin
      case Index of
        nbPrior: Prior;
        nbNext: Next;
        nbFirst: First;
        nbLast: Last;
        nbInsert: Insert;
        nbEdit: Edit;
        nbCancel: Cancel;
        nbPost: Post;
        nbRefresh: Refresh;
        nbDelete:
          if not FConfirmDelete or
            (MessageDlg(SDeleteRecordQuestion, mtConfirmation,
            mbOKCancel, 0) <> idCancel) then Delete;
      end;
    end;
  end;
  if not (csDesigning in ComponentState) and Assigned(FOnNavClick) then
    FOnNavClick(Self, Index);
end;

procedure TDBLaydockNavigator.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TDBLaydockNavigator.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TDBLaydockNavigator.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TMyNavigateBtn;
  OldFocus: TMyNavigateBtn;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TDBLaydockNavigator.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TDBLaydockNavigator.DataChanged;
var
  UpEnable, DnEnable: Boolean;
begin
  UpEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.BOF;
  DnEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.EOF;
  Buttons[nbFirst].Enabled := UpEnable;
  Buttons[nbPrior].Enabled := UpEnable;
  Buttons[nbNext].Enabled := DnEnable;
  Buttons[nbLast].Enabled := DnEnable;
  Buttons[nbDelete].Enabled := Enabled and FDataLink.Active and
    FDataLink.DataSet.CanModify and
    not (FDataLink.DataSet.BOF and FDataLink.DataSet.EOF);
  if Enabled and FDataLink.Active then
    begin
      FProgressBar.Max := FDataLink.DataSource.DataSet.RecordCount;
      FProgressBar.Position := FDataLink.DataSource.DataSet.RecNo;
      FLabel.Caption := IntToStr(FDataLink.DataSource.DataSet.RecNo)+'  '+IntToStr(FDataLink.DataSource.DataSet.RecordCount);
    end;
end;

procedure TDBLaydockNavigator.EditingChanged;
var
  CanModify: Boolean;
begin
  CanModify := Enabled and FDataLink.Active and FDataLink.DataSet.CanModify;
  Buttons[nbInsert].Enabled := CanModify;
  Buttons[nbEdit].Enabled := CanModify and not FDataLink.Editing;
  Buttons[nbPost].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbCancel].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbRefresh].Enabled := CanModify;
  if Enabled and FDataLink.Active then
    begin
      FProgressBar.Max := FDataLink.DataSource.DataSet.RecordCount;
      FProgressBar.Position := FDataLink.DataSource.DataSet.RecNo;
      FLabel.Caption := IntToStr(FDataLink.DataSource.DataSet.RecNo)+'  '+IntToStr(FDataLink.DataSource.DataSet.RecordCount);
    end;
end;

procedure TDBLaydockNavigator.ActiveChanged;
var
  I: TMyNavigateBtn;
begin
  if not (Enabled and FDataLink.Active) then
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Enabled := False
  else
  begin
    DataChanged;
    EditingChanged;
    FProgressBar.Max := FDataLink.DataSource.DataSet.RecordCount;
    FProgressBar.Position := FDataLink.DataSource.DataSet.RecNo;
    FLabel.Caption := IntToStr(FDataLink.DataSource.DataSet.RecNo)+'  '+IntToStr(FDataLink.DataSource.DataSet.RecordCount);
  end;
end;

procedure TDBLaydockNavigator.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TDBLaydockNavigator.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TDBLaydockNavigator.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TDBLaydockNavigator.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  SetSize(W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitHints;
  ActiveChanged;
end;

procedure TDBLaydockNavigator.MyMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var
  N : Real;
begin
  if Enabled and FDataLink.Active then
    Begin
      N := FDataLink.DataSource.DataSet.RecordCount / FProgressBar.Width;
      N := N*X;
      FDataLink.DataSource.DataSet.First;
      FDataLink.DataSource.DataSet.MoveBy(Round(N));
      FProgressBar.Position := FDataLink.DataSource.DataSet.RecNo;
      FLabel.Caption := IntToStr(FDataLink.DataSource.DataSet.RecNo)+'  '+IntToStr(FDataLink.DataSource.DataSet.RecordCount);
    end;
end;

{TMyNavButton}

destructor TMyNavButton.Destroy;
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy;
end;

procedure TMyNavButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  if nsAllowTimer in FNavStyle then
  begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);

    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True;
  end;
end;

procedure TMyNavButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
                                  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled  := False;
end;

procedure TMyNavButton.TimerExpired(Sender: TObject);
begin
  FRepeatTimer.Interval := RepeatPause;
  try
    Click;
  except
    FRepeatTimer.Enabled := False;
    raise;
  end;
end;

procedure TMyNavButton.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TDBLaydockNavigator (Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    Canvas.Brush.Style := bsSolid;
    Font.Color := clBtnShadow;
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

{ TMyNavDataLink }

constructor TMyNavDataLink.Create(ANav: TDBLaydockNavigator);
begin
  inherited Create;
  FNavigator := ANav;
  VisualControl := True;
end;

destructor TMyNavDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy;
end;

procedure TMyNavDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged;
end;

procedure TMyNavDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged;
end;

procedure TMyNavDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged;
end;

end.
