unit LaydockButton;
interface

uses
  SysUtils, Messages, Windows, Classes, Controls, Forms, CommCtrl , Graphics, Dialogs,
  ExtCtrls, StdCtrls, Printers, Consts;

type
  PPalEntriesArray = ^TPalEntriesArray; {for palette re-construction}
  TPalEntriesArray = array[0..0] of TPaletteEntry;

type
  TGradientStyle = (gsHorizontal, gsVertical, gsRectangle,
					gsVertCenter, gsHorizCenter,gsLeftTop,gsRightTop,
					gsLeftTopRightBottom,gsRightTopLeftBottom,
					gsElliptic,gsNone,gsImage,gsTextOnly);
type
  TTextStyle = (tsNormal, tsRaised, tsInset);

type
  TShadeType = (stLight, stHeavy);

type
  TTextLayout = (tlLeftRight, tlUpDown);

type
  TFrameLayout = (flNone, flUp, flDown);
  TPosChangedEvent = procedure (Sender: TObject; CurLeft, CurTop: Longint) of object;

type
  TLaydockBtn = class(TCustomControl)
  private
    FDefault : Boolean;
    FClicksDisabled : Boolean;
    FActive      : Boolean;
    FFrameSize   : Integer ;
    FBorderColor : TColor;
    FImage       : TBitmap;
    FBtnHiliteClr : TColor;
    FBtnShadowClr : TColor;
    FTxtHiliteClr : TColor;
    FTxtShadowClr : TColor;
    FBeginClr : TColor;
    FEndClr : TColor;
    FSwapClr : Boolean;
    FMouseDownEffect : Boolean ;
    FGradientStyle : TGradientStyle;
    FTextStyle : TTextStyle;
    FShadeType : TShadeType;
    FTextLayout: TTextLayout ;
    mDn, mUp : Boolean;
    bm : TBitmap;
    TmpClr : TColor;
    FTimer: TTimer;
    FBlink: Boolean;
    FShowBorder : Boolean;
    FInterval:Word;
    FFontColor: TColor;
    FShowColor: TColor;
    FResizing: Integer;
    SavedCursor: TCursor ;
    CursorSaveOk:Boolean ;
    FResizeBorder: Integer;
    FDesignMode: Boolean;
    MouseDownX : Integer ;
    MouseDownY : Integer ;
    FEditSelected : Boolean ;
    FPosMoved : Boolean ;
    FSwareCommand : String ;
    FOnPosChanged : TPosChangedEvent ;

    procedure Paint; override;
    procedure SetDefault(Value: Boolean);
    procedure CMTextChanged(var Msg:TMessage); message CM_TEXTCHANGED;
    procedure WMEraseBkgnd(Var Msg : TMessage); message WM_ERASEBKGND;
    procedure CMFocusChanged(var Message: TCMFocusChanged); message CM_FOCUSCHANGED;
    procedure SetSwareCtlStyle(ADefault: Boolean); virtual;
    procedure ChangedFont(Sender:TObject);
    procedure SetBlink(Value: Boolean);
    procedure SetInterval(Value: Word);
    procedure SetDesignMode(Value: Boolean);
    procedure SetEditSelected(Value: Boolean);
    procedure SetResizeBorder(Value: Integer);
    procedure SetSwareCommand(Value: String);
    procedure MouseDown(Button:TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift:TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button:TMouseButton; Shift:TShiftState; X, Y: Integer); override;
    procedure SetBtnHilite(Value: TColor);
    procedure SetBtnShadow(Value: TColor);
    procedure SetTxtHilite(Value: TColor);
    procedure SetTxtShadow(Value: TColor);
    procedure SetBeginClr(Value : TColor);
    procedure SetEndClr(Value : TColor);
    procedure SetSwap(Value : Boolean);
    procedure SetBorder(Value : Boolean);
    procedure SetMouseDownEffect(Value:Boolean);
    procedure SetStyle(Value: TTextStyle);
    procedure SetShade(Value: TShadeType);
    procedure ChangedImage(Sender:TObject);
    procedure SetImage(Value:TBitmap);
    procedure SetLayout(Value: TTextLayout);
    procedure SetFrameSize(Value: Integer);
    procedure SetBorderColor(Value: TColor);
    procedure DoRaise(offset: Integer);
    procedure DoInset(offset: Integer);
    procedure DoNorm(offset: Integer);
    procedure SetGradient(Value : TGradientStyle);
    procedure DoElliptic(fr, fg, fb, dr, dg, db : Integer);
    procedure DoFocusRect;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
  protected
    procedure WndProc(var Message: TMessage); override;
    procedure WCtlBlink(Sender:TObject) ; dynamic;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    property BtnHiliteClr    : TColor          read FBtnHiliteClr    write SetBtnHilite       default clBtnHighLight;
    property BtnShadowClr    : TColor          read FBtnShadowClr    write SetBtnShadow       default clBtnShadow;
    property TxtHiliteClr    : TColor          read FTxtHiliteClr    write SetTxtHilite       default clBtnHighLight;
    property TxtShadowClr    : TColor          read FTxtShadowClr    write SetTxtShadow       default clBtnShadow;
    property BeginColor      : TColor          read FBeginClr        write SetBeginClr        default clNavy;
    property EndColor        : TColor          read FEndClr          write SetEndClr          default clAqua;
    property SwapColors      : Boolean         read FSwapClr         write SetSwap            default False;
    property ShowBorder      : Boolean         read FShowBorder      write SetBorder          default False;
    property MouseDownEffect : Boolean         read FMouseDownEffect write SetMouseDownEffect default False;
    property GradientStyle   : TGradientStyle  read FGradientStyle   write SetGradient        default gsNone;
    property TextStyle       : TTextStyle      read FTextStyle       write SetStyle           default tsNormal;
    property TextShadeType   : TShadeType      read FShadeType       write SetShade           default stLight;
    property TextLayout      : TTextLayout     read FTextLayout      write SetLayout          default tlLeftRight;
    property FrameSize       : Integer         read FFrameSize       write SetFrameSize       default 3 ;
    property BorderColor     : TColor          read FBorderColor     write SetBorderColor     default clBlack ;
    property Image           : TBitmap         read FImage           write SetImage ;
    property Default         : Boolean         read FDefault         write SetDefault         default False;
    property Blink           : Boolean         read FBlink           write SetBlink;
    property Interval        : Word            read FInterval        write SetInterval;
    property ResizeBorder    : Integer         read FResizeBorder    write SetResizeBorder;
    property DesignMode      : Boolean         read FDesignMode      write SetDesignMode;
    property EditSelected    : Boolean         read FEditSelected    write SetEditSelected;
    property SwareCommand    : String          read FSwareCommand    write SetSwareCommand;

    property Anchors;
    property Caption;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseUp;
    property Visible;
    property Enabled;
    property Font;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseMove;
    property OnDragDrop ;
    property OnDragOver ;
    property OnEndDrag ;
    property OnStartDrag ;
    property OnPosChanged : TPosChangedEvent read FOnPosChanged write FOnPosChanged;
  end;
procedure BltTBitmapAsDib(DestDc : hdc;x : word;y : word;Width : word;Height : word;bm : TBitmap);
procedure PrintBitmap(Bitmap : Graphics.TBitmap;X, Y, pWidth, pHeight: Integer);
procedure SwareGradientDrawArea(Canvas:TCanvas ; _Type,Left,Top,Right,Bottom:integer; BeginColor,EndColor:TColor);
procedure SwareGradientDrawLine(Canvas:TCanvas;  _Type,Left,Top,Right,Bottom,width:integer; BeginColor,EndColor: TColor;Style:Integer) ;
procedure SwareGradientDrawPLine(Canvas:TCanvas;  _Type,width:Integer;Points:array of TPoint ; BeginColor,EndColor: TColor;Style:Integer) ;
procedure SwareGradientDrawRect(Canvas:TCanvas; _Type,Left,Top,Right,Bottom,width:Integer; BeginColor,EndColor:TColor) ;
procedure SwareGradientDrawEllipseDefault(Canvas:TCanvas; _Type,Left , Top , Right , Bottom,Pwidth:Integer; BeginColor,EndColor:TColor) ;
procedure SwareGradientDrawEllipse(Canvas:TCanvas; _Type,Left,Top,Right,Bottom,Pwidth:Integer; BeginColor,EndColor:TColor;ParentLeft,ParentTop:Integer) ;
procedure SwareGradientDrawTriangle(Canvas:TCanvas; _Type,Left,Top,Right,Bottom,Pwidth:Integer; BeginColor,EndColor:TColor) ;
procedure DrawButton(Canvas:TCanvas;Left,Top,Right,Bottom:Integer;
                    Caption,FontName:PChar;FontColor,BackColor:TColor;
                    FontSize,FrameSize:Integer;Horz,mDown:boolean; Image : TBitmap);
procedure FocusRect(Canvas:TCanvas; FocusArea:TRect);

function  minint(a,b:Integer):Integer ;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Laydock',[TLaydockBtn]);
end;

function  minint(a,b:Integer):Integer;
begin
    if a>b then Result:=b else Result :=b ;
end;

procedure BltTBitmapAsDib(DestDc : hdc;   {Handle of where to blt}
                          x : word;       {Bit at x}
                          y : word;       {Blt at y}
                          Width : word;   {Width to stretch}
                          Height : word;  {Height to stretch}
                          bm : TBitmap);  {the TBitmap to Blt}
var
  OriginalWidth :LongInt;               {width of BM}
  dc : hdc;                             {screen dc}
  IsPaletteDevice : bool;               {if the device uses palettes}
  IsDestPaletteDevice : bool;           {if the device uses palettes}
  BitmapInfoSize : integer;             {sizeof the bitmapinfoheader}
  lpBitmapInfo : PBitmapInfo;           {the bitmap info header}
  hBm : hBitmap;                        {handle to the bitmap}
  hPal : hPalette;                      {handle to the palette}
  OldPal : hPalette;                    {temp palette}
  hBits : THandle;                      {handle to the DIB bits}
  pBits : pointer;                      {pointer to the DIB bits}
  lPPalEntriesArray : PPalEntriesArray; {palette entry array}
  NumPalEntries : integer;              {number of palette entries}
  i : integer;                          {looping variable}
begin
{$IFOPT R+}
  {$DEFINE CKRANGE}
  {$R-}
{$ENDIF}

 {Save the original width of the bitmap}
  OriginalWidth := bm.Width;

 {Get the screen's dc to use since memory dc's are not reliable}
  dc := GetDc(0);
 {Are we a palette device?}
  IsPaletteDevice :=
    GetDeviceCaps(dc, RASTERCAPS) and RC_PALETTE = RC_PALETTE;
 {Give back the screen dc}
  ReleaseDc(0, dc);

 {Allocate the BitmapInfo structure}
  if IsPaletteDevice then
    BitmapInfoSize := sizeof(TBitmapInfo) + (sizeof(TRGBQUAD) * 255)
  else
    BitmapInfoSize := sizeof(TBitmapInfo);
  GetMem(lpBitmapInfo, BitmapInfoSize);

 {Zero out the BitmapInfo structure}
  FillChar(lpBitmapInfo^, BitmapInfoSize, #0);

 {Fill in the BitmapInfo structure}
  lpBitmapInfo^.bmiHeader.biSize := sizeof(TBitmapInfoHeader);
  lpBitmapInfo^.bmiHeader.biWidth := OriginalWidth;
  lpBitmapInfo^.bmiHeader.biHeight := bm.Height;
  lpBitmapInfo^.bmiHeader.biPlanes := 1;
  if IsPaletteDevice then
    lpBitmapInfo^.bmiHeader.biBitCount := 8
  else
    lpBitmapInfo^.bmiHeader.biBitCount := 24;
  lpBitmapInfo^.bmiHeader.biCompression := BI_RGB;
  lpBitmapInfo^.bmiHeader.biSizeImage :=
    ((lpBitmapInfo^.bmiHeader.biWidth *
      longint(lpBitmapInfo^.bmiHeader.biBitCount)) div 8) *
      lpBitmapInfo^.bmiHeader.biHeight;
  lpBitmapInfo^.bmiHeader.biXPelsPerMeter := 0;
  lpBitmapInfo^.bmiHeader.biYPelsPerMeter := 0;
  if IsPaletteDevice then begin
    lpBitmapInfo^.bmiHeader.biClrUsed := 256;
    lpBitmapInfo^.bmiHeader.biClrImportant := 256;
  end else begin
    lpBitmapInfo^.bmiHeader.biClrUsed := 0;
    lpBitmapInfo^.bmiHeader.biClrImportant := 0;
  end;

 {Take ownership of the bitmap handle and palette}
  hBm := bm.ReleaseHandle;
  hPal := bm.ReleasePalette;

 {Get the screen's dc to use since memory dc's are not reliable}
  dc := GetDc(0);

  if IsPaletteDevice then begin
   {If we are using a palette, it must be}
   {selected into the dc during the conversion}
    OldPal := SelectPalette(dc, hPal, TRUE);
   {Realize the palette}
    RealizePalette(dc);
  end;
 {Tell GetDiBits to fill in the rest of the bitmap info structure}
  GetDiBits(dc,
            hBm,
            0,
            lpBitmapInfo^.bmiHeader.biHeight,
            nil,
            TBitmapInfo(lpBitmapInfo^),
            DIB_RGB_COLORS);

 {Allocate memory for the Bits}
  hBits := GlobalAlloc(GMEM_MOVEABLE,
                       lpBitmapInfo^.bmiHeader.biSizeImage);
  pBits := GlobalLock(hBits);
 {Get the bits}
  GetDiBits(dc,
            hBm,
            0,
            lpBitmapInfo^.bmiHeader.biHeight,
            pBits,
            TBitmapInfo(lpBitmapInfo^),
            DIB_RGB_COLORS);


  if IsPaletteDevice then begin
   {Lets fix up the color table for buggy video drivers}
    GetMem(lPPalEntriesArray, sizeof(TPaletteEntry) * 256);
   {$IFDEF VER100}
      NumPalEntries := GetPaletteEntries(hPal,
                                         0,
                                         256,
                                         lPPalEntriesArray^);
   {$ELSE}
      NumPalEntries := GetSystemPaletteEntries(dc,
                                               0,
                                               256,
                                               lPPalEntriesArray^);
   {$ENDIF}
    for i := 0 to (NumPalEntries - 1) do begin
      lpBitmapInfo^.bmiColors[i].rgbRed :=
        lPPalEntriesArray^[i].peRed;
      lpBitmapInfo^.bmiColors[i].rgbGreen :=
        lPPalEntriesArray^[i].peGreen;
      lpBitmapInfo^.bmiColors[i].rgbBlue :=
        lPPalEntriesArray^[i].peBlue;
    end;
    FreeMem(lPPalEntriesArray, sizeof(TPaletteEntry) * 256);
  end;

  if IsPaletteDevice then begin
   {Select the old palette back in}
    SelectPalette(dc, OldPal, TRUE);
   {Realize the old palette}
    RealizePalette(dc);
  end;

 {Give back the screen dc}
  ReleaseDc(0, dc);

 {Is the Dest dc a palette device?}
  IsDestPaletteDevice :=
    GetDeviceCaps(DestDc, RASTERCAPS) and RC_PALETTE = RC_PALETTE;


  if IsPaletteDevice then begin
   {If we are using a palette, it must be}
   {selected into the dc during the conversion}
    OldPal := SelectPalette(DestDc, hPal, TRUE);
   {Realize the palette}
    RealizePalette(DestDc);
  end;

 {Do the blt}
  StretchDiBits(DestDc,
                x,
                y,
                Width,
                Height,
                0,
                0,
                OriginalWidth,
                lpBitmapInfo^.bmiHeader.biHeight,
                pBits,
                lpBitmapInfo^,
                DIB_RGB_COLORS,
                SrcCopy);

  if IsDestPaletteDevice then begin
   {Select the old palette back in}
    SelectPalette(DestDc, OldPal, TRUE);
   {Realize the old palette}
    RealizePalette(DestDc);
  end;

 {De-Allocate the Dib Bits}
  GlobalUnLock(hBits);
  GlobalFree(hBits);

 {De-Allocate the BitmapInfo}
  FreeMem(lpBitmapInfo, BitmapInfoSize);

 {Set the ownership of the bimap handles back to the bitmap}
  bm.Handle := hBm;
  bm.Palette := hPal;

  {Turn range checking back on if it was on when we started}
{$IFDEF CKRANGE}
  {$UNDEF CKRANGE}
  {$R+}
{$ENDIF}
end;

procedure PrintBitmap(Bitmap : Graphics.TBitmap;X, Y, pWidth, pHeight: Integer);
begin
    BltTBitmapAsDib(printer.Canvas.Handle,X,Y,pWidth, pHeight,Bitmap);
end ;

procedure SwareGradientDrawArea(Canvas:TCanvas ; _Type,Left,Top,Right,Bottom:integer; BeginColor,EndColor:TColor);
var
  frect , frect2 : TRect ;
          i , loopcount , halfloopcount : Integer;
  fr,fg,fb,dr,dg,db ,cr,cg,cb:Integer ;
  Temppoints:array[0..255] of TPoint ;
  frgn : HRGN ;
  bm : TBitMap ;
begin
  bm := TBitMap.Create ;
  bm.Width := Right - Left ;
  bm.Height := Bottom-Top ;
  fr := (BeginColor and $000000FF) ;
  fg := ((BeginColor shr 8) and $000000ff) ;
  fb := ((BeginColor shr 16) and $000000ff) ;
  dr := ((EndColor and $000000ff) - fr) ;
  dg := (((EndColor shr 8) and $000000ff) - fg) ;
  db := (((EndColor shr 16) and $000000ff) - fb) ;
  if (_Type>10) then _Type:=2 ;
  if (_Type<0) then _Type:=2 ;
  case (_Type) of
    0: begin
         frect.Top := Top ;
         frect.Bottom := Bottom ;
         frect.Left := Left ;
         frect.Right := Right ;
         Canvas.Brush.Color := BeginColor ;
         Canvas.FillRect(frect) ;
       end;
    1: begin
         frect.Top := Top ;
         frect.Bottom := Bottom ;
         for i:=0 to 255 do
           begin
             frect.Left := (Left+i*(Right - Left)div 255) ;
             frect.Right := Left+(i+1)*(Right - Left)div 255 ;
             cr := (fr+i*dr div 255) ;
             cg := (fg+i*dg div 255) ;
             cb := (fb+i*db div 255) ;
             Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
             Canvas.FillRect(frect) ;
           end;
         end;
    2: begin
         frect.left := Left ;
         frect.right := Right ;
         for i:=0 to 255 do
           begin
             frect.top := Top+i*(Bottom - Top) div 255 ;
             frect.bottom := Top+(i+1)*(Bottom - Top) div 255 ;
             cr := (fr+i*dr div 255) ;
             cg := (fg+i*dg div 255) ;
             cb := (fb+i*db div 255) ;
             Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
             Canvas.FillRect(frect) ;
           end;
         end;
    3: begin
         loopcount := abs(Bottom-Top) ;
         if ((Right - Left)>(Bottom-Top)) then loopcount:=abs(Right - Left) ;
         halfloopcount:= loopcount div 2 ;
         for i:=0 to halfloopcount do
           begin
             cr := (fr+i*dr div halfloopcount) ;
             cg := (fg+i*dg div halfloopcount) ;
             cb := (fb+i*db div halfloopcount) ;
             frect.Top := Top+i*(Bottom - Top) div loopcount ;
             frect.Bottom := Bottom-i*(Bottom - Top) div loopcount ;
             frect.Left := Left+i*(Right - Left) div loopcount ;
             frect.Right := Right-i*(Right - Left) div loopcount ;
             Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
             Canvas.FrameRect(frect) ;
           end;
           Canvas.FillRect(frect) ;
         end;
		4:
		begin
			frect.left := Left ;
			frect.right := Right ;
			frect2 := frect ;
            for i:=0 to 254 do
			begin
				frect.top := Top+i*(Bottom - Top) div 510 ;
				frect.bottom := Top+(i+1)*(Bottom - Top) div 510 ;
				cr := (fr+i*dr div 255) ;
				cg := (fg+i*dg div 255) ;
                cb := (fb+i*db div 255) ;
            	Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
                Canvas.FillRect(frect) ;
                frect2.top := Bottom - (i)*(Bottom - Top) div 510 ;
				frect2.bottom := Bottom - (i+1)*(Bottom - Top) div 510 ;
				Canvas.FillRect(frect2) ;
			end;
            frect2.top := frect.bottom ;
			Canvas.FillRect(frect2) ;
        end;
		5:
        begin
			frect.top := Top ;
			frect.bottom := Bottom ;
			frect2 := frect ;
			for i:=0 to 254 do
            begin
				frect.left := Left+i*(Right - Left) div 510 ;
                frect.right := Left+(i+1)*(Right - Left) div 510 ;
                cr := (fr+i*dr div 255) ;
				cg := (fg+i*dg div 255) ;
				cb := (fb+i*db div 255) ;
				Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
				Canvas.FillRect(frect) ;
                frect2.left := Right-i*(Right - Left) div 510 ;
				frect2.right := Right-(i+1)*(Right - Left) div 510 ;
                Canvas.FillRect(frect2) ;
			end;
			frect2.left := frect.right ;
			Canvas.FillRect(frect2) ;
		end;
		6:
		begin
			Canvas.Pen.Width := 1 ;
			halfloopcount:=128 ;
			loopcount:= halfloopcount*2 ;
			bm.Canvas.Brush.Style := bsSolid ;
			for i:=0 to loopcount do
			begin
				cr := (fr+i*dr div loopcount) ;
				cg := (fg+i*dg div loopcount) ;
				cb := (fb+i*db div loopcount) ;
				bm.Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
				if (i<halfloopcount) then
				begin
					Temppoints[0].x:= 0 ; Temppoints[0].y:= i*(Bottom-Top) div halfloopcount;
					Temppoints[1].x:= 0 ; Temppoints[1].y:= (i+1)*(Bottom-Top) div halfloopcount;
					Temppoints[3].x:= i*(Right-Left) div halfloopcount;  Temppoints[3].y:= 0;
					Temppoints[2].x:= (i+1)*(Right-Left) div halfloopcount;     Temppoints[2].y:= 0;
					frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
					PaintRgn(bm.Canvas.Handle, frgn) ;
					DeleteObject(frgn) ;
				end
				else
				begin
					Temppoints[0].x:= (i-halfloopcount)*(Right - Left) div halfloopcount;  Temppoints[0].y:= Bottom-Top;
					Temppoints[1].x:= (i+1-halfloopcount)*(Right - Left) div halfloopcount;  Temppoints[1].y:= Bottom-Top;
					Temppoints[3].x:= Right - Left;  Temppoints[3].y:= (i-halfloopcount)*(Bottom-Top) div halfloopcount;
					Temppoints[2].x:= Temppoints[3].x;     Temppoints[2].y:= (i+1-halfloopcount)*(Bottom-Top) div halfloopcount;
					frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
					PaintRgn(bm.Canvas.Handle, frgn) ;
					DeleteObject(frgn) ;
				end;
			end;
			Canvas.Draw(Left,Top , bm) ;
		end;
		7:
		begin
			Canvas.Pen.Width := 1 ;
			halfloopcount:=128 ;
			loopcount:= halfloopcount*2 ;
			bm.Canvas.Brush.Style := bsSolid ;
			for i:=0 to loopcount do
			begin
				cr := (fr+i*dr div loopcount) ;
				cg := (fg+i*dg div loopcount) ;
				cb := (fb+i*db div loopcount) ;
				bm.Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
				if (i<halfloopcount) then
				begin
					Temppoints[0].x:= Right-Left;  Temppoints[0].y:= i*(Bottom-Top) div halfloopcount;
					Temppoints[1].x:= Right-Left;  Temppoints[1].y:= (i+1)*(Bottom-Top) div halfloopcount;
					Temppoints[2].x:= Right-Left - (i+1)*(Right-Left) div halfloopcount;  Temppoints[2].y:= 0;
					Temppoints[3].x:= Right-Left - i*(Right-Left) div halfloopcount;    Temppoints[3].y:= 0;
					frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
					PaintRgn(bm.Canvas.Handle, frgn) ;
					DeleteObject(frgn) ;
				end
				else
				begin
					Temppoints[0].x:= Right-Left - (i-halfloopcount)*(Right-Left) div halfloopcount;  Temppoints[0].y:= Bottom-Top ;
					Temppoints[1].x:= Right-Left - (i+1-halfloopcount)*(Right-Left) div halfloopcount;  Temppoints[1].y:= Bottom-Top ;
					Temppoints[3].x:= 0;  Temppoints[3].y:= (i-halfloopcount)*(Bottom-Top) div halfloopcount;;
					Temppoints[2].x:= 0;    Temppoints[2].y:= (i+1-halfloopcount)*(Bottom-Top) div halfloopcount;;
					frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
					PaintRgn(bm.Canvas.Handle, frgn) ;
					DeleteObject(frgn) ;
				end;
			end;
			Canvas.Draw(Left,Top , bm) ;
		end;
		8:
		begin
			Canvas.Pen.Width := 1 ;
			loopcount:= 128 ;
			bm.Canvas.Brush.Style := bsSolid ;
			for i:=0 to loopcount do
			begin
				cr := (fr+i*dr div loopcount) ;
				cg := (fg+i*dg div loopcount) ;
				cb := (fb+i*db div loopcount) ;
				bm.Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
					Temppoints[0].x:= 0 ; Temppoints[0].y:= i*(Bottom-Top) div loopcount;
					Temppoints[1].x:= 0 ; Temppoints[1].y:= (i+1)*(Bottom-Top) div loopcount;
					Temppoints[3].x:= i*(Right-Left) div loopcount;  Temppoints[3].y:= 0;
					Temppoints[2].x:= (i+1)*(Right-Left) div loopcount;     Temppoints[2].y:= 0;
					frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
					PaintRgn(bm.Canvas.Handle, frgn) ;
					DeleteObject(frgn) ;

					Temppoints[0].x:= Right-Left-i*(Right - Left) div loopcount;  Temppoints[0].y:= Bottom-Top;
					Temppoints[1].x:= Right-Left-(i+1)*(Right - Left) div loopcount;  Temppoints[1].y:= Bottom-Top;
					Temppoints[2].x:= Right - Left;  Temppoints[2].y:= Bottom-Top-(i+1)*(Bottom-Top) div loopcount;
					Temppoints[3].x:= Temppoints[2].x;     Temppoints[3].y:= Bottom-Top-i*(Bottom-Top) div loopcount;
					frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
					PaintRgn(bm.Canvas.Handle, frgn) ;
					DeleteObject(frgn) ;
			end;
			Canvas.Draw(Left,Top , bm) ;
		end;
		9:
		begin
			Canvas.Pen.Width := 1 ;
			loopcount:= 128 ;
			bm.Canvas.Brush.Style := bsSolid ;
			for i:=0 to loopcount do
                    begin
                            cr := (fr+i*dr div loopcount) ;
                            cg := (fg+i*dg div loopcount) ;
                            cb := (fb+i*db div loopcount) ;
                            bm.Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
                            Temppoints[0].x:= Right-Left;  Temppoints[0].y:= i*(Bottom-Top) div loopcount;
                            Temppoints[1].x:= Right-Left;  Temppoints[1].y:= (i+1)*(Bottom-Top) div loopcount;
                            Temppoints[2].x:= Right-Left - (i+1)*(Right-Left) div loopcount;  Temppoints[2].y:= 0;
                            Temppoints[3].x:= Right-Left - i*(Right-Left) div loopcount;    Temppoints[3].y:= 0;
                            frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
                            PaintRgn(bm.Canvas.Handle, frgn) ;
                            DeleteObject(frgn) ;

                            Temppoints[0].x:= i*(Right-Left) div loopcount;  Temppoints[0].y:= Bottom-Top ;
                            Temppoints[1].x:= (i+1)*(Right-Left) div loopcount;  Temppoints[1].y:= Bottom-Top ;
                            Temppoints[2].x:= 0;  Temppoints[2].y:= Bottom-Top-(i+1)*(Bottom-Top) div loopcount;;
                            Temppoints[3].x:= 0;    Temppoints[3].y:= Bottom-Top-i*(Bottom-Top) div loopcount;;
                            frgn:=CreatePolygonRgn(Temppoints,4,ALTERNATE);
                            PaintRgn(bm.Canvas.Handle, frgn) ;
                            DeleteObject(frgn) ;
                end;
			Canvas.Draw(Left,Top , bm) ;
		end;
	end;
			bm.Free ;
end;

procedure SwareGradientDrawLine(Canvas:TCanvas;  _Type,Left,Top,Right,Bottom,width:Integer; BeginColor,EndColor: TColor;Style:Integer) ;
var
	i : Integer;
	fr,fg,fb,dr,dg,db ,cr,cg,cb : Integer ;
	x1,y1,x2,y2 , width2 : Integer;
begin
	if (_Type=0) then
	begin
		Canvas.Pen.Color := BeginColor ;
		Canvas.Pen.Width := width ;
		Canvas.Pen.Style := TPenStyle(Style) ;
		Canvas.MoveTo(Left,Top) ;
       	Canvas.LineTo(Right,Bottom) ;
		Exit ;
    end ;

	fr := (BeginColor and $000000FF) ;
	fg := ((BeginColor shr 8) and $000000ff) ;
	fb := ((BeginColor shr 16) and $000000ff) ;
    dr := ((EndColor and $000000ff) - fr) ;
	dg := (((EndColor shr 8) and $000000ff) - fg) ;
	db := (((EndColor shr 16) and $000000ff) - fb) ;

   	Canvas.Pen.Width := 1 ;
	width2 := width div 2+1 ;
	if (abs(Right - Left)>abs(Bottom - Top)) then
    begin
		if (Left>Right) then
        begin
			x1 := Left+width2 ;
			x2 := Right-width2 ;
        end
		else
        begin
			x1 := Left-width2 ;
			x2 := Right+width2 ;
        end;
	    for i:=0 to width2 do
		begin
			y1 := Top+i ;
    	    y2 := Bottom+i ;
	        cr := (fr+i*dr div width2) ;
    	    cg := (fg+i*dg div width2) ;
			cb := (fb+i*db div width2) ;
			Canvas.Pen.Color := TColor(RGB(cr,cg,cb)) ;
			Canvas.MoveTo(x1,y1) ;
			Canvas.LineTo(x2,y2) ;
			y1 := Top-i ;
    	    y2 := Bottom-i ;
			Canvas.MoveTo(x1,y1) ;
			Canvas.LineTo(x2,y2) ;
        end;
	end
    else
	begin
        if (Top>Bottom) then
        begin
			y1 := Top+width2 ;
			y2 := Bottom-width2 ;
		end
		else
        begin
			y1 := Top-width2 ;
			y2 := Bottom+width2 ;
        end;
	    for i:=0 to width2 do
		begin
			x1 := Left+i ;
			x2 := Right+i ;
			cr := (fr+i*dr div width2) ;
			cg := (fg+i*dg div width2) ;
			cb := (fb+i*db div width2) ;
			Canvas.Pen.Color := TColor(RGB(cr,cg,cb)) ;
			Canvas.MoveTo(x1,y1) ;
			Canvas.LineTo(x2,y2) ;
			x1 := Left-i ;
			x2 := Right-i ;
			Canvas.MoveTo(x1,y1) ;
			Canvas.LineTo(x2,y2) ;
		end;
	end;
end;

procedure SwareGradientDrawPLineCount(Canvas:TCanvas;  _Type,width:Integer;Points:array of TPoint ; _Count:Integer ;BeginColor,EndColor: TColor;Style:Integer) ;
var
	i , j , k : Integer ;
	fr,fg,fb,dr,dg,db ,cr,cg,cb : Integer ;
	width2 : Integer ;
	Temppoints:array[0..255] of TPoint ;
	LineDir:array[0..255] of Integer ;
	KDir:array[0..1] of Integer ;
	Count : Integer ;
begin
	Count:=_Count ;
	if Count>255 then Count:=255 ;
	if Count<2 then
	begin
		Exit ;
	end;
	Canvas.Pen.Mode := pmCopy ;
	if (_Type=0) then
	begin
		Canvas.Pen.Color := BeginColor ;
		Canvas.Pen.Width := width ;
		Canvas.Pen.Style := TPenStyle(Style) ;
		Canvas.MoveTo(Points[0].x,Points[0].y) ;
		Canvas.Polyline(Slice(Points,Count)) ;
		Exit ;
	end ;

	for i:=0 to Count-2 do
	begin
        LineDir[i]:=0 ;
		if (abs(Points[i+1].x - Points[i].x)>abs(Points[i+1].y - Points[i].y)) then
		begin
			if (Points[i].x>Points[i+1].x) then
            begin
    		    LineDir[i]:=1 ;
			end
            else
			begin
				LineDir[i]:=2 ;
			end;
        end
		else
		begin
            if (Points[i].y>Points[i+1].y) then
			begin
				LineDir[i]:=3 ;
			end
            else
            begin
				LineDir[i]:=4 ;
            end;
		end;
	end;

    fr := (EndColor and $000000FF) ;
	fg := ((EndColor shr 8) and $000000ff) ;
    fb := ((EndColor shr 16) and $000000ff) ;
	dr := ((BeginColor and $000000ff) - fr) ;
    dg := (((BeginColor shr 8) and $000000ff) - fg) ;
    db := (((BeginColor shr 16) and $000000ff) - fb) ;

   	Canvas.Pen.Width := 1 ;
	width2 := width div 2+1 ;
	KDir[0]:=1 ;
	KDir[1]:=-1 ;
    for j:=0 to width2 do
	begin
		cr := (fr+j*dr div width2) ;
   	    cg := (fg+j*dg div width2) ;
		cb := (fb+j*db div width2) ;
		Canvas.Pen.Color := TColor(RGB(cr,cg,cb)) ;
		for k:=0 to 1 do
        begin
			case LineDir[0] of
			1:
				begin TempPoints[0].x:=Points[0].x+width2 ;
				TempPoints[0].y:=Points[0].y+j*KDir[k] ; end ;
			2:
				begin TempPoints[0].x:=Points[0].x-width2 ;
				TempPoints[0].y:=Points[0].y-j*KDir[k] ; end ;
			3:
				begin TempPoints[0].x:=Points[0].x-j*KDir[k] ;
				TempPoints[0].y:=Points[0].y+width2 ; end ;
            4:
                begin TempPoints[0].x:=Points[0].x+j*KDir[k] ;
				TempPoints[0].y:=Points[0].y-width2 ; end ;
            end;
			for i:=1 to Count-2 do
			begin
				TempPoints[i].x:=Points[i].x+j*KDir[k] ;
				TempPoints[i].y:=Points[i].y+j*KDir[k] ;
				case LineDir[i-1] of
				1:
					case LineDir[i] of
						1,3:
						begin TempPoints[i].x:=Points[i].x-j*KDir[k] ;
						TempPoints[i].y:=Points[i].y+j*KDir[k] ; end ;
						4:
						begin TempPoints[i].x:=Points[i].x+j*KDir[k] ;
						TempPoints[i].y:=Points[i].y+j*KDir[k] ; end ;
					end;
				2:
					case LineDir[i] of
						2,3:
						begin TempPoints[i].x:=Points[i].x-j*KDir[k] ;
						TempPoints[i].y:=Points[i].y-j*KDir[k] ; end ;
						4:
						begin TempPoints[i].x:=Points[i].x+j*KDir[k] ;
						TempPoints[i].y:=Points[i].y-j*KDir[k] ; end ;
					end;
				3:
					case LineDir[i] of
						1,3:
						begin TempPoints[i].x:=Points[i].x-j*KDir[k] ;
						TempPoints[i].y:=Points[i].y+j*KDir[k] ; end ;
						2:
						begin TempPoints[i].x:=Points[i].x-j*KDir[k] ;
						TempPoints[i].y:=Points[i].y-j*KDir[k] ; end ;
					end;
				4:
					case LineDir[i] of
						1,4:
						begin TempPoints[i].x:=Points[i].x+j*KDir[k] ;
						TempPoints[i].y:=Points[i].y+j*KDir[k] ; end ;
						2:
						begin TempPoints[i].x:=Points[i].x+j*KDir[k] ;
						TempPoints[i].y:=Points[i].y-j*KDir[k] ; end ;
					end;
				end;
			end ;
			case LineDir[Count-2] of
			1:
				begin TempPoints[Count-1].x:=Points[Count-1].x-width2-1 ;
				TempPoints[Count-1].y:=Points[Count-1].y+j*KDir[k] ; end;
			2:
				begin TempPoints[Count-1].x:=Points[Count-1].x+width2+1 ;
				TempPoints[Count-1].y:=Points[Count-1].y-j*KDir[k] ; end ;
			3:
				begin TempPoints[Count-1].x:=Points[Count-1].x-j*KDir[k] ;
				TempPoints[Count-1].y:=Points[Count-1].y-width2-1 ; end ;
			4:
				begin TempPoints[Count-1].x:=Points[Count-1].x+j*KDir[k] ;
				TempPoints[Count-1].y:=Points[Count-1].y+width2+1 ; end ;
			end;
			Canvas.MoveTo(TempPoints[0].x,TempPoints[0].y) ;
			Canvas.Polyline(Slice(TempPoints,Count)) ;
		end;
	end;
end;

procedure SwareGradientDrawPLine(Canvas:TCanvas;  _Type,width:Integer;Points:array of TPoint ; BeginColor,EndColor: TColor;Style:Integer) ;
begin
     SwareGradientDrawPLineCount(Canvas,_Type,width,Points,High(Points),BeginColor,EndColor,Style) ;
end;

procedure SwareGradientDrawRect(Canvas:TCanvas; _Type,Left,Top,Right,Bottom,width:Integer; BeginColor,EndColor:TColor) ;
var
	i ,j , x1 ,x2 , y1 , y2 : Integer;
	fr,fg,fb,dr,dg,db ,cr,cg,cb : Integer;
	Width2 : Integer;
begin
	if (_Type=0) then
	begin
		if (abs(Right-Left)<=(width)) then _Type := 1 ;
		if (abs(Bottom-Top)<=(width)) then _Type := 1 ;
	end;
	if (Left>Right) then
	begin
		x1 := Right ;
		x2 := Left ;
	end
	else
	begin
		x2 := Right ;
		x1 := Left ;
	end;
	if (Top>Bottom) then
	begin
		y1 := Bottom ;
		y2 := Top ;
	end
	else
	begin
		y2 := Bottom ;
		y1 := Top ;
	end;
	Width2 := width div 2+1 ;
	Canvas.Pen.Mode := pmCopy	;
	if (_Type=0) then
	begin
		if (BeginColor = EndColor) then
		begin
			Canvas.Pen.Color := BeginColor ;
			Canvas.Pen.Width := width ;
			Canvas.Brush.Style := bsClear ;
			Canvas.Rectangle(x1,y1,x2+1,y2+1);
		end
		else
		begin
			fr := (BeginColor and $000000FF) ;
			fg := ((BeginColor shr 8) and $000000ff) ;
			fb := ((BeginColor shr 16) and $000000ff) ;
			dr := ((EndColor and $000000ff) - fr) ;
			dg := (((EndColor shr 8) and $000000ff) - fg) ;
			db := (((EndColor shr 16) and $000000ff) - fb) ;
			Canvas.Pen.Width := 1 ;
			Canvas.Brush.Style := bsSolid ;
            x1:=x1+Width2;   y1:=y1+Width2;
            x2:=x2-Width2;   y2:=y2-Width2;
			for i:=0 to Width2 do
			begin
				cr := (fr+i*dr div Width2) ;
				cg := (fg+i*dg div Width2) ;
				cb := (fb+i*db div Width2) ;
				j:=Width2-i ;
				Canvas.Brush.Color := TColor(RGB(cr,cg,cb)) ;
				Canvas.FrameRect(Rect(x1-j,y1-j,x2+j+1,y2+j+1));
				Canvas.FrameRect(Rect(x1+j,y1+j,x2-j+1,y2-j+1));
			end;
		end ;
		Canvas.Brush.Style := bsSolid ;
	end
	else
    if _Type<10 then SwareGradientDrawArea(Canvas , _Type , x1 , y1 , x2+1 , y2+1 , BeginColor , EndColor)
    else begin
            SwareGradientDrawRect(Canvas , 0 , x1 , y1 , x2+1 , y2+1 , width , BeginColor,EndColor) ;
            x1:=x1+Width2;   y1:=y1+Width2;
            x2:=x2-Width2;   y2:=y2-Width2;
            SwareGradientDrawArea(Canvas , 1 , x1+1 , y1+1 , x2 , y2 , BeginColor , BeginColor);
         end;
end;

procedure SwareGradientDrawEllipseDefault(Canvas:TCanvas; _Type,Left , Top , Right , Bottom,Pwidth:Integer; BeginColor,EndColor:TColor) ;
var
	Width, Height ,Width2 , PenWidth , i: Integer ;
	fr,fg,fb,dr,dg,db,cr,cg,cb : Integer ;
	frgn : HRGN ;
	EllipseRect : TRect ;
	bm : TBitMap ;
begin
	Width := Right - Left ;
	Height := Bottom-Top ;
	if (_Type = 0) then   // No gradient
	begin
		PenWidth := PWidth ;
		if (PWidth>Width) then PenWidth :=Width ;
		if (PWidth>Height) then PenWidth :=Height ;
		Width2 :=PenWidth div 2 ;
		Canvas.Pen.Width := PenWidth ;
		Canvas.Ellipse(Width2, Width2, Width-Width2, Height-Width2);
	end
	else
	begin
		bm := TBitMap.Create ;
		bm.Width := Width ;
		bm.Height := Height ;
		bm.Canvas.CopyRect(Rect(0,0,Width,Height),Canvas,Rect(Left,Top,Right,Bottom)) ;
		fr := (BeginColor and $000000FF) ;
		fg := ((BeginColor shr 8) and $000000ff) ;
		fb := ((BeginColor shr 16) and $000000ff) ;
		dr := ((EndColor and $000000ff) - fr) ;
		dg := (((EndColor shr 8) and $000000ff) - fg) ;
		db := (((EndColor shr 16) and $000000ff) - fb) ;
		bm.Canvas.Brush.Style := bsSolid ;
		bm.Canvas.Pen.Width := 1 ;
		bm.Canvas.Pen.Style := psClear ;
		bm.Canvas.Pen.Mode := pmCopy ;
		if (Width>Height) then Width2 := Height div 2 else Width2 := Width div 2 ;
		for i:=0 to Width2 do
		begin
			cr := (fr+i*dr div Width2) ;
			cg := (fg+i*dg div Width2) ;
			cb := (fb+i*db div Width2) ;
			EllipseRect.Left := i ;
			EllipseRect.Top  := i ;
			EllipseRect.Right := Width - i ;
			EllipseRect.Bottom := Height - i ;
			bm.Canvas.Brush.Color :=TColor(RGB(cr,cg,cb)) ;
			frgn:=CreateEllipticRgnIndirect(EllipseRect);
			PaintRgn(bm.Canvas.Handle, frgn) ;
			DeleteObject(frgn) ;
		end;
		bm.Canvas.Ellipse(EllipseRect.Left,EllipseRect.Top,EllipseRect.Right,EllipseRect.Bottom);
		Canvas.Draw(Left,Top,bm);
		bm.Free ;
	end;
end;

procedure SwareGradientDrawEllipse(Canvas:TCanvas; _Type,Left , Top , Right , Bottom,Pwidth:Integer; BeginColor,EndColor:TColor;ParentLeft,ParentTop:Integer) ;
var
	Width, Height ,Width2 , PenWidth : Integer ;
	frgn : HRGN ;
begin
	Width := Right - Left ;
	Height := Bottom-Top ;
	if (_Type = 0) then   // No gradient
	begin
		PenWidth := PWidth ;
		if (PWidth>Width) then PenWidth :=Width ;
		if (PWidth>Height) then PenWidth :=Height ;
		Width2 :=PenWidth div 2 ;
		Canvas.Pen.Width := PenWidth ;
		Canvas.Ellipse(Width2, Width2, Width-Width2, Height-Width2);
	end
	else
	begin
		frgn:=CreateEllipticRgnIndirect(Rect(Left+ParentLeft,Top+ParentTop,Right+ParentLeft,Bottom+ParentTop));
		SelectClipRgn(Canvas.Handle, frgn) ;
		SwareGradientDrawRect(Canvas,_Type,Left,Top,Right,Bottom,Pwidth,BeginColor,EndColor) ;
		SelectClipRgn(Canvas.Handle, 0) ;
		DeleteObject(frgn) ;
	end;
end;

procedure SwareGradientDrawTriangle(Canvas:TCanvas; _Type,Left,Top,Right,Bottom,Pwidth:Integer; BeginColor,EndColor:TColor) ;
var
	Width2 , i , PenWidth , Width , Height : Integer ;
	fr,fg,fb,dr,dg,db,cr,cg,cb : Integer ;
	triangle : array[0..2] of TPoint;
begin
	Width := Right - Left ;
	Height := Bottom - Top ;
	if (_Type = 0) then   // No gradient
    begin
		PenWidth := Pwidth ;
		if (Pwidth>Width) then PenWidth :=Width ;
        if (Pwidth>Height) then PenWidth :=Height ;
        Width2 :=PenWidth div 2 ;
		Canvas.Pen.Width := PenWidth ;
        triangle[0].x := Left + Width div 2 ;
		triangle[0].y := Top + Width2 ;
        triangle[1].x := Left + Width2 ;
		triangle[1].y := Bottom-trunc(Width2*Height/Width)-1 ;
        triangle[2].x := Right-Width2-1 ;
		triangle[2].y := Bottom-trunc(Width2*Height/Width)-1 ;
		Canvas.Polygon(triangle);
	end
	else
	begin
		fr := (BeginColor and $000000FF) ;
		fg := ((BeginColor shr 8) and $000000ff) ;
		fb := ((BeginColor shr 16) and $000000ff) ;
		dr := ((EndColor and $000000ff) - fr) ;
		dg := (((EndColor shr 8) and $000000ff) - fg) ;
		db := (((EndColor shr 16) and $000000ff) - fb) ;
		Canvas.Brush.Style := bsSolid ;
		Canvas.Pen.Width :=1 ;
		Canvas.Pen.Style := psClear ;
		Canvas.Pen.Mode := pmCopy ;
		if (Width>Height) then Width2 := Height div 2 else Width2 := Width div 2 ;
		for i:=0 to Width2 do
		begin
			cr := (fr+i*dr div Width2) ;
			cg := (fg+i*dg div Width2) ;
			cb := (fb+i*db div Width2) ;
			triangle[0].x := Left + Width div 2 ;
			triangle[0].y := Top + i ;
			triangle[1].x := Left + trunc(i*Width/Height) ;
			triangle[1].y := Bottom - trunc(i*Height/Width)-1 ;
			triangle[2].x := Right-trunc(i*Width/Height)-1 ;
			triangle[2].y := Bottom - trunc(i*Height/Width)-1 ;
			Canvas.Brush.Color :=TColor(RGB(cr,cg,cb)) ;
			Canvas.Pen.Color :=Canvas.Brush.Color ;
			Canvas.Polygon(triangle);
		end;
		Canvas.Pen.Style := psSolid ;
	end;
end;

procedure DrawButton(Canvas:TCanvas;Left,Top,Right,Bottom:Integer;
                    Caption:Pchar;
                    FontName:PChar;FontColor,BackColor:TColor;
                    FontSize,FrameSize:Integer;Horz,mDown:boolean; Image : TBitmap);
var
  rct , RectTemp : TRect;
  offset , Width , Height, W , H : Integer ;
begin
  Canvas.Font.Name := FontName;
  Canvas.Font.Size := FontSize;
  Canvas.Font.Color := FontColor;
  rct := Rect(Left,Top,Right,Bottom);
  SwareGradientDrawArea(Canvas,0,Left,Top,Right,Bottom,BackColor,BackColor);
  Canvas.Brush.Style := bsClear;
        if (mDown=FALSE)then begin
          offset := 0;
          Frame3D(Canvas,rct,clBtnHighLight,clBtnShadow,FrameSize);
        end
        else begin
          offset := 1;
          Frame3D(Canvas,rct,clBtnShadow,clBtnHighLight,FrameSize);
        end;
        Canvas.Brush.Color:=clBlack;
        rct.Left:=rct.Left-FrameSize ;
        rct.Top:=rct.Top-FrameSize ;
        rct.Right:=rct.Right+FrameSize ;
        rct.Bottom:=rct.Bottom+FrameSize ;
        Canvas.FrameRect(rct);
    if (not (FontColor=BackColor)) then
    begin
        Width := Right - Left ;         // My Butt
        Height := Bottom - Top ;
        Canvas.Brush.Style := bsClear;
        Canvas.Font.Color := FontColor;
        if (Horz=TRUE) then
        begin
            Canvas.TextOut(Left+(Width-Canvas.TextWidth(Caption) div 2)+offset,
            Top+((Height-Canvas.TextHeight(Caption)) div 2)+offset,Caption);
        end
        else
        begin
            W := (Width-Canvas.TextWidth('00')) div 2 ;
            H := (Height-((Canvas.TextHeight(Caption)*Length(Caption)) div 2)) div 2 ;
            RectTemp:=Rect((Left+W)+offset,(Top+H)+offset,Right-W+offset,Bottom-H+offset);
            DrawText(Canvas.Handle,Caption,Length(Caption),RectTemp,DT_EXPANDTABS or DT_WORDBREAK);
            Canvas.TextOut(Left+((Width-Canvas.TextWidth(Caption)) div 2)+offset,
            Top+((Height-Canvas.TextHeight(Caption)) div 2)+offset,Caption);
        end;
    end;
    Canvas.Brush.Style := bsSolid;
end;

procedure FocusRect(Canvas:TCanvas; FocusArea:TRect);
var r :Integer ;
begin
    Canvas.Pen.Width := 1 ;
    Canvas.Pen.Style := psDot ;
    Canvas.Pen.Mode := pmNot ;
    Canvas.Brush.Style := bsClear;
    Canvas.Rectangle(FocusArea.Left,FocusArea.Top,FocusArea.Right,FocusArea.Bottom);
    r := 1 ;
    Canvas.Pen.Width := 2 ;
    Canvas.Pen.Style := psSolid ;
    if ((abs(FocusArea.Left-FocusArea.Right)>5) and (abs(FocusArea.Top-FocusArea.Bottom)>5)) then
    begin
        Canvas.Rectangle(FocusArea.Left+1,FocusArea.Top+1,FocusArea.Left+r+r+1+1,FocusArea.Top+r+r+1+1);
        Canvas.Rectangle(FocusArea.Left+1,FocusArea.Bottom-r-r-1-1,FocusArea.Left+r+r+1+1,FocusArea.Bottom-1);
        Canvas.Rectangle(FocusArea.Right-r-1-r-1,FocusArea.Top+1,FocusArea.Right-1,FocusArea.Top+r+r+1+1);
        Canvas.Rectangle(FocusArea.Right-r-1-r-1,FocusArea.Bottom-r-r-r-1,FocusArea.Right-1,FocusArea.Bottom-1);
    end
    else
    begin
//        Canvas.Rectangle(FocusArea.Left-r,FocusArea.Top-r,FocusArea.Left+r+1,FocusArea.Top+r+1);
//        Canvas.Rectangle(FocusArea.Left-r,FocusArea.Bottom-r,FocusArea.Left+r+1,FocusArea.Bottom+r+1);
//        Canvas.Rectangle(FocusArea.Right-r-1,FocusArea.Top-r-1,FocusArea.Right+r,FocusArea.Top+r);
//        Canvas.Rectangle(FocusArea.Right-r-1,FocusArea.Bottom-r-1,FocusArea.Right+r,FocusArea.Bottom+r);
    end;
    Canvas.Brush.Style := bsSolid;
    Canvas.Pen.Width := 1 ;
    Canvas.Pen.Style := psSolid ;
    Canvas.Pen.Mode := pmCopy ;
end;

function InitColorStatusControl(CC: Integer): Boolean;
var
  ICC: TInitCommonControlsEx;
begin
  ICC.dwSize := SizeOf(TInitCommonControlsEx);
  ICC.dwICC := CC;
  Result := InitCommonControlsEx(ICC);
  if not Result then InitCommonControls;
end;

constructor TLaydockBtn.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FBtnHiliteClr := clBtnHighLight;   //Default Hilite Color
  FBtnShadowClr := clBtnShadow;      //Default Shadow Color
  FTxtHiliteClr := clGray;   //Default Text Hilite
  FTxtShadowClr := clBtnShadow;      //Default Text Shadow
  FBeginClr := clSilver;            //Start Color is at the Top of button
  FEndClr := clGray;             //Stop Color is at the Bottom of button
  Height := 22;                   //Default button size
  Width := 100;
  bm := TBitmap.Create;          //Create the internal BMP
  bm.Canvas.Pen.Style:= psSolid; //These may already be the default,
  bm.Canvas.Pen.Mode:= pmCopy;   // but just in case...

  FImage := TBitmap.Create;
  FImage.OnChange:=ChangedImage;

  mDn := FALSE;                   //Mouse is UP to begin with
  mUp := TRUE;

//  ControlStyle := ControlStyle + [csOpaque];   //This reduces flicker
  FSwapClr := True;               //By default, we swap colors on Mouse Down
  FMouseDownEffect := TRUE ;
  FTextStyle := tsInset;         //No 3D text
  FShadeType := stLight;          //Light text shading by default
  FGradientStyle := gsVertical;   //Top & Bottom to Center for gradient Fill default
  FFrameSize := 2;
  FBorderColor := clBlack ;

  Font.Color := clYellow ;
  FFontColor := Font.Color;
  FShowColor := Font.Color;

  Font.OnChange := ChangedFont ;

  FResizeBorder := 4 ;
  FDesignMode := FALSE ;
  FPosMoved := FALSE ;
  FEditSelected := FALSE ;
  CursorSaveOk := FALSE ;
  TabStop := True;
  FBlink := false ;
  FInterval :=500 ;
  FTimer := TTimer.Create(Self) ;
  FTimer.Enabled :=FBlink ;
  FTimer.OnTimer :=WCtlBlink ;
  FTimer.Interval :=FInterval ;
end;

//WndProc Added by victor Chen
procedure TLaydockBtn.WndProc(var Message: TMessage);
begin
  case Message.Msg of
    WM_LBUTTONDOWN, WM_RBUTTONDOWN, WM_LBUTTONDBLCLK:
      if not (csDesigning in ComponentState) and not Focused then
        begin
          FClicksDisabled := True;
          Windows.SetFocus(Handle);
          FClicksDisabled := False;
          if not Focused then Exit;
        end;
    CN_COMMAND:
      if FClicksDisabled then Exit;
  end;
  inherited WndProc(Message);
end;

procedure TLaydockBtn.CMFocusChanged(var Message: TCMFocusChanged);
begin
  with Message do
    if Sender is TLaydockBtn then
      FActive := Sender = Self
    else
      FActive := FDefault;
  SetSwareCtlStyle(FActive);
  Invalidate;
  inherited;
end;

procedure TLaydockBtn.Paint;
var
  FromR, FromG, FromB : Integer; //These are the separate color values for RGB
  DiffR, DiffG, DiffB : Integer; // of color values.
  rct : TRect;                   //Rectangle used to draw frame around button
  offset : Integer;              //Used for Caption location during Mouse Down

 {To speed things up and reduce flicker, I use a Bitmap to draw the button in
 its entirety, ten BitBlt it to the canvas of the control.}
begin
	if (FDesignMode=FALSE) then
    begin
        if (CursorSaveOk=FALSE) then
		begin
			SavedCursor := Cursor ;
			CursorSaveOk := TRUE ;
		end ;
	end;

  bm.Canvas.Font := Font;     // Set the Bitmap font to match control font
  bm.Canvas.Font.Color := FShowColor; //Keep Bitmap in synch with settings.
  bm.Width := Width;          //Set BMP dimensions to match control's
  bm.Height := Height;
  rct := Rect(0,0,Width,Height);  //Set rectangle size for later use
  case FGradientStyle of
	   gsNone      : SwareGradientDrawArea(bm.Canvas,0,0,0,Width,Height,FBeginClr,FBeginClr);
	   gsTextOnly  : SwareGradientDrawArea(bm.Canvas,0,0,0,Width,Height,FBeginClr,FBeginClr) ;
	   gsHorizontal: SwareGradientDrawRect(bm.Canvas,1,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsVertical  : SwareGradientDrawRect(bm.Canvas,2,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsElliptic  :
	   begin
            FromR := FBeginClr and $000000ff;  //Strip out separate RGB values
            FromG := (FBeginClr shr 8) and $000000ff;
            FromB := (FBeginClr shr 16) and $000000ff;
            DiffR := (FEndClr and $000000ff) - FromR;   //Find the difference
            DiffG := ((FEndClr shr 8) and $000000ff) - FromG;
            DiffB := ((FEndClr shr 16) and $000000ff) - FromB;
            DoElliptic(FromR, FromG, FromB, DiffR, DiffG, DiffB);
	   end;
	   gsRectangle : SwareGradientDrawRect(bm.Canvas,3,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsVertCenter: SwareGradientDrawRect(bm.Canvas,4,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsHorizCenter:SwareGradientDrawRect(bm.Canvas,5,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsLeftTop    :SwareGradientDrawRect(bm.Canvas,6,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsRightTop   :SwareGradientDrawRect(bm.Canvas,7,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsLeftTopRightBottom: SwareGradientDrawRect(bm.Canvas,8,0,0,Width,Height,0,FBeginClr,FEndClr) ;
	   gsRightTopLeftBottom: SwareGradientDrawRect(bm.Canvas,9,0,0,Width,Height,0,FBeginClr,FEndClr) ;
  end;
//    MyButton
  FImage.Transparent := True;

  bm.Canvas.Brush.Style := bsClear;  //Gradient is done, time for Hilite-Shadow
  case FGradientStyle of
	gsImage :
        begin
            bm.Canvas.Brush.Style:=bsSolid;
            bm.Canvas.Brush.Color:=clBtnFace;
    	    bm.Canvas.FillRect(Rect(0, 0, Width,Height));
            if (mUp=TRUE) then Frame3D(bm.Canvas,rct,FBtnHiliteClr,FBtnShadowClr,2)
            else Frame3D(bm.Canvas,rct,FBtnShadowClr,FBtnHiliteClr,2);
            bm.Canvas.Brush.Color:=clBlack;
    	    bm.Canvas.FrameRect(Rect(0, 0, Width,Height));
            bm.Canvas.Draw((bm.Width-FImage.Width) div 2,(bm.Height-FImage.Height) div 2,FImage);
        end;
//        BitBlt(bm.Canvas.Handle,0,0,Width,Height,FImage.Canvas.Handle,0,0,SRCCOPY);
	gsTextOnly : begin
          if mUp = TRUE then offset := 0 else offset := 1;
          DoNorm(offset);
        end ;
       else
	  begin
        if ((mUp=TRUE) or (FMouseDownEffect=False)) then
          begin       //If Mouse up do normal button
            offset := 1;  //offset is used to move the Caption down a notch when the button is pressed.
            Frame3D(bm.Canvas,rct,FBtnHiliteClr,FBtnShadowClr,FFrameSize); //Draw frame
            bm.Canvas.Draw(((bm.Width-FImage.Width-bm.Canvas.TextWidth(Caption)) div 2)+offset,((bm.Height-FImage.Height) div 2)+offset, FImage);
          end
        else
          begin
            offset := 2; //If mouse down, we want caption to 'move' and reverse the frame
            Frame3D(bm.Canvas,rct,FBtnShadowClr,FBtnHiliteClr,FFrameSize);
            bm.Canvas.Draw(((bm.Width-FImage.Width-bm.Canvas.TextWidth(Caption)) div 2)+offset,((bm.Height-FImage.Height) div 2)+offset, FImage);
          end;
            bm.Canvas.Brush.Color:=FBorderColor;
            rct.Left:=rct.Left-FFrameSize ;
            rct.Top:=rct.Top-FFrameSize ;
            rct.Right:=rct.Right+FFrameSize ;
            rct.Bottom:=rct.Bottom+FFrameSize ;
            bm.Canvas.FrameRect(rct);

        if FTextStyle = tsRaised then DoRaise(offset);  //Based on style, we draw
        if FTextStyle = tsInset then DoInset(offset);   // the caption on the
        if FTextStyle = tsNormal then DoNorm(offset);   // button
	  end;
    end ;

  if FDesignMode = FALSE then begin if FActive then DoFocusRect ; end ;
  if ((FDesignMode = TRUE) and (FEditSelected = TRUE)) then
  begin
	bm.Canvas.Brush.Style := bsClear;
	bm.Canvas.Pen.Color := clBlue ;
	bm.Canvas.Pen.Mode := pmCopy ;
	bm.Canvas.Pen.Width := 1 ;
	bm.Canvas.Pen.Style := psDot ;
	bm.Canvas.Rectangle(1,1,Width-1,Height-1) ;
	bm.Canvas.Pen.Style := psSolid ;
	bm.Canvas.Brush.Style := bsSolid;
  end;

  BitBlt(Canvas.Handle,0,0,Width,Height,bm.Canvas.Handle,0,0,SRCCOPY);
end;

procedure TLaydockBtn.SetDesignMode(Value: boolean);
begin
  if FDesignMode <> Value then
  begin
	FDesignMode := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  if FDesignMode and Enabled then
  begin
    if FEditSelected=False then
    begin
        FEditSelected:=True;
        Repaint;
    end;
  end;
end;

procedure TLaydockBtn.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  if Enabled then
  begin
    if FEditSelected=True then
    begin
        FEditSelected:=False;
        Invalidate;
    end;
  end;
end;

procedure TLaydockBtn.SetEditSelected(Value: boolean);
begin
  if FEditSelected <> Value then
  begin
	FEditSelected := Value;
	Invalidate;
  end;
end;

procedure TLaydockBtn.SetSwareCommand(Value: String);
begin
	FSwareCommand := Value;
end;

procedure TLaydockBtn.SetResizeBorder(Value: Integer);
begin
  if FResizeBorder <> Value then
  begin
    FResizeBorder := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetBtnHilite(Value: TColor);
begin
  if FBtnHiliteClr <> Value then begin
    FBtnHiliteClr := Value;
	Invalidate;  //Invalidate tells windows to do a Paint.  Our control
  end;           // goes through its Paint procedure.
end;

procedure TLaydockBtn.SetBtnShadow(Value: TColor);
begin
  if FBtnShadowClr <> Value then begin
    FBtnShadowClr := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetTxtHilite(Value: TColor);
begin
  if FTxtHiliteClr <> Value then begin
    FTxtHiliteClr := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetTxtShadow(Value: TColor);
begin
  if FTxtShadowClr <> Value then begin
    FTxtShadowClr := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetBeginClr(Value : TColor);
begin
  if FBeginClr <> Value then begin
    FBeginClr := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetEndClr(Value : TColor);
begin
  if FEndClr <> Value then begin
    FEndClr := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetSwap(Value : Boolean);
begin
  if FSwapClr <> Value then
    FSwapClr := Value;
end;

procedure TLaydockBtn.SetBorder(Value : Boolean);
begin
  if FShowBorder <> Value then
    FShowBorder := Value;
end;

procedure TLaydockBtn.SetMouseDownEffect(Value : Boolean);
begin
  if FMouseDownEffect <> Value then
    FMouseDownEffect := Value;
end;

procedure TLaydockBtn.WCtlBlink(Sender : TObject) ;
begin
      if FShowColor = FFontColor then
		FShowColor := FBeginClr
      else
		FShowColor := FFontColor ;
    Invalidate ;
end;

procedure TLaydockBtn.ChangedFont(Sender:TObject);
begin
    FFontColor := Font.Color;
    FShowColor := Font.Color;
    Invalidate ;
end;

procedure TLaydockBtn.SetBlink(Value: Boolean);
begin
  if FBlink <> Value then
  begin
    FBlink := Value;
    FTimer.Enabled := Value;
    if not FTimer.Enabled then
    begin
        FShowColor := FFontColor ;
        Invalidate ;
    end;
  end;
end;

procedure TLaydockBtn.SetInterval(Value: Word);
begin
  if FInterval <> Value then
  begin
    FInterval := Value;
    FTimer.Interval := Value;
  end;
end;

procedure TLaydockBtn.MouseDown(Button:TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if ((FDesignMode = TRUE) and (FEditSelected=TRUE)) then
    begin
      if (Button=mbLeft) then
	begin
          mDn := TRUE;
          mUp := FALSE;
          MouseDownX := X ;
          MouseDownY := Y ;
	end;
      Invalidate;
      Exit ;
    end;
  mDn := TRUE;
  mUp := FALSE;
  if FSwapClr = TRUE then
    begin
      TmpClr := FBeginClr;
      FBeginClr := FEndClr;
      FEndClr := TmpClr;
    end;
  Invalidate;
  inherited;
end;

procedure TLaydockBtn.MouseMove(Shift: TShiftState;
	  X, Y: Integer);
begin
  if ((FDesignMode = TRUE) and (FEditSelected=TRUE)) then
  begin
	if (mDn=TRUE) then
	begin
	if ((MouseDownX=X) and (MouseDownY=Y)) then Exit ;
	FPosMoved:=TRUE;
		case FReSizing of
            0:
                begin
                    Cursor := crDefault ;
                    Left := Left+(X-MouseDownX) ;
                    Top := Top+(Y-MouseDownY) ;
                end;
			1:
                begin
                    Width := Width-(X-MouseDownX) ;
                    Height := Height-(Y-MouseDownY) ;
                    Left := Left+(X-MouseDownX) ;
					Top := Top+(Y-MouseDownY) ;
                end;
            2:
                begin
                    Width := Width-(X-MouseDownX) ;
                    Left := Left+(X-MouseDownX) ;
                end;
			3:
				begin
					Width := Width-(X-MouseDownX) ;
                    Height := Y ;
                    Left := Left+(X-MouseDownX) ;
				end;
            4:
                begin
                    Height := Y ;
                end;
            5:
                begin
                    Width := X ;
					Height := Y ;
                end;
            6:
                begin
                    Width := X ;
				end;
            7:
                begin
                    Width := X ;
                    Height := Height-(Y-MouseDownY) ;
                    Top := Top+(Y-MouseDownY) ;
                end;
			8:
				begin
					Height := Height-(Y-MouseDownY) ;
                    Top := Top+(Y-MouseDownY) ;
                end;
        end;
        if (Width<1) then Width := 1 ;
        if (Height<1) then Height := 1 ;
		Exit ;
    end
    else
    begin
	FReSizing := 0 ;
        Cursor := crDefault ;
        Exit ;
        if (X<FResizeBorder) then
		begin
            if (Y<FResizeBorder) then begin Cursor := crSizeNWSE ; FReSizing := 1 ; Exit ;end;
            if (Y>(Height-FResizeBorder)) then begin Cursor := crSizeNESW ;  FReSizing := 3 ; Exit ;end;
            Cursor := crSizeWE ;  FReSizing := 2; Exit ;
        end;
		if (X>Width-FResizeBorder) then
        begin
            if (Y<FResizeBorder) then begin Cursor := crSizeNESW ;  FReSizing := 7 ; Exit ;end;
            if (Y>(Height-FResizeBorder)) then begin Cursor := crSizeNWSE ;  FReSizing := 5 ; Exit ;end;
            Cursor := crSizeWE ;  FReSizing := 6 ; Exit ;
        end;
        if (Y<FResizeBorder) then begin Cursor := crSizeNS ;  FReSizing := 8 ; Exit ;end;
		if (Y>(Height-FResizeBorder)) then begin Cursor := crSizeNS ;  FReSizing := 4 ; Exit ;end;
		FReSizing := 0 ;
        Cursor := crDefault ;
        Exit ;
    end;
  end;
    Cursor := SavedCursor ;
    inherited;
//  Invalidate;
end;

procedure TLaydockBtn.MouseUp;
begin
  mUp := TRUE;
  mDn := FALSE;
  if ((FDesignMode = TRUE) and (FEditSelected=TRUE)) then
  begin
    Invalidate;
    if Assigned(FOnPosChanged) and FPosMoved=TRUE then FOnPosChanged(Self, Left,Top);
    FPosMoved :=FALSE ;
    Exit ;
  end;
  inherited;
  if FSwapClr = TRUE then
    begin
      TmpClr := FBeginClr;
      FBeginClr := FEndClr;
      FEndClr := TmpClr;
  end;
  Invalidate;
end;

procedure TLaydockBtn.CMTextChanged(var Msg:TMessage);
begin
  Inherited;
  Invalidate;
end;

procedure TLaydockBtn.WMEraseBkgnd(var Msg:TMessage);
begin
  Msg.Result := 0;  //We intercept this message and tell
end;                // Windows it processed OK.

procedure TLaydockBtn.SetSwareCtlStyle(ADefault: Boolean);
const
  BS_MASK = $000F;
var
  Style: Word;
begin
  if HandleAllocated then
  begin
    if ADefault then Style := BS_DEFPUSHBUTTON else Style := BS_PUSHBUTTON;
    if GetWindowLong(Handle, GWL_STYLE) and BS_MASK <> Style then
      SendMessage(Handle, BM_SETSTYLE, Style, 1);
  end;
end;

procedure TLaydockBtn.SetDefault(Value: Boolean);
var
  Form: TCustomForm;
begin
  FDefault := Value;
  if HandleAllocated then
  begin
    Form := GetParentForm(Self);
    if Form <> nil then
      Form.Perform(CM_FOCUSCHANGED, 0, Longint(Form.ActiveControl));
  end;
end;

procedure TLaydockBtn.SetGradient(Value : TGradientStyle);
begin
  if FGradientStyle <> Value then begin
    FGradientStyle := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetStyle(Value: TTextStyle);
begin
  if FTextStyle <> Value then begin
    FTextStyle := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetShade(Value: TShadeType);
begin
  {When user selects a shade, set the variable to new type and do
   an invalidate to force a re-paint of the component.}
  if FShadeType <> Value then begin
    FShadeType := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetLayout(Value: TTextLayout);  //Add by Victor Chen
begin
  if FTextLayout <> Value then begin
    FTextLayout := Value;
	Invalidate;
  end;
end;

procedure TLaydockBtn.SetFrameSize(Value: Integer);  //Add by Victor Chen
begin
  if FFrameSize <> Value then
  begin
    FFrameSize := Value;
    if FFrameSize > 10 then FFrameSize := 3;
    if FFrameSize < 1 then FFrameSize := 3;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetBorderColor(Value: TColor);  //Add by Victor Chen
begin
  if FBorderColor <> Value then
  begin
    FBorderColor := Value;
    Invalidate;
  end;
end;

procedure TLaydockBtn.SetImage(Value: TBitmap);
begin
  if Assigned(Value) then
    FImage.Assign(Value)
  else
  begin
    FImage.Height:=0;
    FImage.Width:=0;
  end;
end;

procedure TLaydockBtn.ChangedImage(Sender:TObject);
begin
  Invalidate;
end;

procedure TLaydockBtn.DoElliptic(fr, fg, fb, dr, dg, db : Integer);
var
  I: Integer;
  R, G, B : Byte;
  Pw, Ph : Real;
  x1,y1,x2,y2 : Real;
{The elliptic is a bit different, since I had to use real numbers. I cut down
 on the number (to 155 instead of 255) of iterations in an attempt to speed
 things up, to no avail.  I think it just takes longer for windows to draw an
 ellipse as opposed to a rectangle.}
begin
  bm.Canvas.Pen.Style := psClear;
  bm.Canvas.Pen.Mode := pmCopy;
  x1 := 0 - (bm.Width / 4);
  x2 := bm.Width + (bm.Width / 4);
  y1 := 0 - (bm.Height / 4);
  y2 := bm.Height + (bm.Height / 4);
  Pw := ((bm.Width / 4) + (bm.Width / 2)) / 155;
  Ph := ((bm.Height / 4) + (bm.Height / 2)) / 155;
  for I := 0 to 155 do begin         //Make ellipses of color
    x1 := x1 + Pw;
    x2 := X2 - Pw;
	y1 := y1 + Ph;
    y2 := y2 - Ph;
    R := fr + MulDiv(I, dr, 155);    //Find the RGB values
    G := fg + MulDiv(I, dg, 155);
	B := fb + MulDiv(I, db, 155);
	bm.Canvas.Brush.Color := R or (G shl 8) or (b shl 16);   //Plug colors into brush
	bm.Canvas.Ellipse(Trunc(x1),Trunc(y1),Trunc(x2),Trunc(y2));
  end;
  bm.Canvas.Pen.Style := psSolid;
end;

procedure TLaydockBtn.DoRaise(offset: Integer);
var
   StrText  :array[0..255] of char ;
   RectTemp :TRect ;
   CapLen   :Integer;
begin
    CapLen:=GetTextBuf(StrText , Sizeof(StrText)) ;
    bm.Canvas.Brush.Style := bsClear;
    bm.Canvas.Font.Color := FTxtHiliteClr;
    if (FTextLayout=tlLeftRight) then
      begin
        if FImage <> nil then
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)-1+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)-1+offset,Caption)
        else
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)) div 2)-1+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)-1+offset,Caption);
        if FShadeType = stHeavy then
          begin
            bm.Canvas.Font.Color := FTxtShadowClr;
            if FImage <> nil then
              bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)+1+offset,
              ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+1+offset,Caption)
            else
              bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)) div 2)+1+offset,
              ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+1+offset,Caption);
          end;
        bm.Canvas.Font.Color := FShowColor;
        if FImage <> nil then
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+offset,Caption)
        else
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)) div 2)+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+offset,Caption);
          bm.Canvas.Brush.Style := bsSolid;
      end
    else
      begin
        RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)-1+offset,
        ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)-1+offset,
        bm.Width,bm.Height);
        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK) ;
        if FShadeType = stHeavy then
        begin
            bm.Canvas.Font.Color := FTxtShadowClr;
        RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)+1+offset,
        ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)+1+offset,
        bm.Width,bm.Height);
        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK) ;
        end;
        bm.Canvas.Font.Color := FShowColor;
        RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)+offset,
        ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)+offset,
        bm.Width,bm.Height);
        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK) ;
    end;
    bm.Canvas.Brush.Style := bsSolid;
end;

procedure TLaydockBtn.DoInset(offset: Integer);
var
   StrText  :array[0..255] of char ;
   RectTemp :TRect ;
   CapLen   :Integer;
begin
    CapLen:=GetTextBuf(StrText , Sizeof(StrText)) ;
  bm.Canvas.Brush.Style := bsClear;
    if (FTextLayout=tlLeftRight) then
    begin
        if FShadeType = stHeavy then begin
        bm.Canvas.Font.Color := FTxtShadowClr;
        if FImage <> nil then
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)-1+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)-1+offset,Caption)
        else
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)) div 2)-1+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)-1+offset,Caption);
        end;
        bm.Canvas.Font.Color := FTxtHiliteClr;
        if FImage <> nil then
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)+1+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+1+offset,Caption)
        else
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width) div 2)+1+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+1+offset,Caption);
        bm.Canvas.Font.Color := FShowColor;
        if FImage <> nil then
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+offset,Caption)
        else
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)) div 2)+offset,
          ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+offset,Caption)
    end
    else
    begin
        if FShadeType = stHeavy then begin
        bm.Canvas.Font.Color := FTxtShadowClr;
        RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)-1+offset,
        ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)-1+offset,
        bm.Width,bm.Height);
        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK) ;
        end;
        bm.Canvas.Font.Color := FTxtHiliteClr;
        RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)+1+offset,
        ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)+1+offset,
        bm.Width,bm.Height);
        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK) ;
        bm.Canvas.Font.Color := FShowColor;
        RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)+offset,
        ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)+offset,
        bm.Width,bm.Height);
        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK) ;
    end;
  bm.Canvas.Brush.Style := bsSolid;
end;

procedure TLaydockBtn.DoNorm(offset: Integer);
var
   StrText  :array[0..255] of char ;
   RectTemp :TRect ;
   CapLen   :integer;
begin
    CapLen:=GetTextBuf(StrText , Sizeof(StrText)) ;
    bm.Canvas.Brush.Style := bsClear;
    bm.Canvas.Font.Color := FShowColor;
    if (FTextLayout=tlLeftRight) then
      begin
        if FImage <> nil then
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)+FImage.Width+3) div 2)+offset,
         ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+offset,Caption)
        else
          bm.Canvas.TextOut(((bm.Width-bm.Canvas.TextWidth(Caption)) div 2)+offset,
         ((bm.Height-bm.Canvas.TextHeight(Caption)) div 2)+offset,Caption)
      end
    else
      begin
        if FImage <> nil then
          RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')+FImage.Width+3) div 2)+offset,
          ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)+offset, bm.Width,bm.Height)
        else
          RectTemp:=Rect(((bm.Width-bm.Canvas.TextWidth('00')) div 2)+offset,
          ((bm.Height-((bm.Canvas.TextHeight(Caption)*CapLen) div 2)) div 2)+offset, bm.Width,bm.Height);

        DrawText(bm.Canvas.Handle,StrText,CapLen,RectTemp,DT_EXPANDTABS or DT_WORDBREAK);
    end;
    bm.Canvas.Brush.Style := bsSolid;
end;

procedure TLaydockBtn.DoFocusRect ;
var
   RectTemp :TRect ;
   i        :integer;
begin
     case FGradientStyle of
         gsImage,gsTextOnly : i:=2;
     else i:= Round(FFrameSize*1.6);
     end;

       begin
        RectTemp.Left:=i ;
        RectTemp.Top:=i ;
        RectTemp.Right:=bm.Width-i ;
        RectTemp.Bottom:=bm.Height-i ;
        if((RectTemp.Right>RectTemp.Left) and (RectTemp.Bottom>RectTemp.Top)) then
        begin
            bm.Canvas.DrawFocusRect(RectTemp);
        end;
       end;
end;

destructor TLaydockBtn.Destroy;
begin
  bm.Free;       //Cleanup the bitmap - no memory leaks
  FImage.Free;
  FTimer.Free;
  inherited Destroy;
end;

end.


