{*****************************************************************************
 Component   :  LsIconEditorMain is a part of the LsIconEditor
                (an Icon property editor for Delphi).

 Version     :  3.10 for Delphi versions 3, 4, 5 & 6.

 Author      :  Leo D. Shih  e-mail: <ldshih@ecn.ab.ca>

 Copyright   :  (C)1999-2002 by Leo D. Shih, all rights reserved.

 DISCLAIMER  :  Current version of TLsIconEditor (Software) is distributed
                as freeware, without warranties of any kind. either expressed
                or implied.  In no event shall the author be liable for any
                problems or damages that may result from the use of this
                software.

 Last Update :  May 2002
*****************************************************************************}


{$INCLUDE LSCOMP.INC}
{$IFDEF D6_OR_HIGHER} //3.1>
  {$WARN SYMBOL_PLATFORM OFF}
  {$WARN UNIT_PLATFORM OFF}
{$ENDIF} //3.1<


unit LsIconEditorMainD3;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls, ComCtrls, Buttons, StdCtrls, ShellAPI, ShlObj, FileCtrl, Menus,
  ComObj, Registry
  {$IFDEF D4_OR_HIGHER},
     ImgList, ToolWin, ActiveX
  {$ENDIF};


type

  TCursorOrIcon = packed record
    Reserved: Word;
    wType: Word;
    Count: Word;
  end;

  PIconRec = ^TIconRec;
  TIconRec = packed record
    Width: Byte;
    Height: Byte;
    Colors: Word;
    Reserved1: Word;
    Reserved2: Word;
    DIBSize: Longint;
    DIBOffset: Longint;
  end;

  TLsIconEditorForm = class(TForm)
    Panel1: TPanel;
    FileView: TListView;
    IconView: TListView;
    DirCombo: TComboBox;
    Panel2: TPanel;
    Image1: TImage;
    ToolBar1: TToolBar;
    BtnPrev: TToolButton;
    BtnFileMask: TToolButton;
    BtnClear: TToolButton;
    BtnSave: TToolButton;
    BtnSelect: TToolButton;
    BtnCancel: TToolButton;
    ImageList_n: TImageList;
    ImageList_h: TImageList;
    PopupMenu1: TPopupMenu;
    AllFiles1: TMenuItem;
    StatusBar1: TStatusBar;
    IconList: TImageList;
    Masked1: TMenuItem;
    SaveDialog1: TSaveDialog;
    BtnAbout: TToolButton;
    Image2: TImage;
    Splitter1: TSplitter;
    procedure FormCreate(Sender: TObject);
    procedure DisplayHint(Sender: TObject);
    procedure CreateImages;
    procedure DirComboUpdate(cPath: TFileName);
    procedure DirComboChange(Sender: TObject);
    procedure DirComboDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure DirComboClick(Sender: TObject);
    procedure FileViewBuildList(cPath: String; FMask: string);
    procedure FileViewAddFiles(cDir: String; Mask: String);  //Attr: DWord);
    procedure FileViewDblClick(Sender: TObject);
    procedure FileViewClick(Sender: TObject);
    procedure AllFiles1Click(Sender: TObject);
    procedure Masked1Click(Sender: TObject);
    procedure IconViewLoadIcons(IFName: string);
    procedure BtnClearClick(Sender: TObject);
    procedure IconViewClick(Sender: TObject);
    procedure InitializeBitmapInfoHeader(Bitmap: HBITMAP;
      var bi: TBitmapInfoHeader; Colors: Integer);
    procedure GetDIBSizes(Bitmap: HBITMAP; var InfoHeaderSize: DWORD;
      var ImageSize: DWORD; Colors: Integer);
    function GetDIB(Bitmap: HBITMAP; Palette: HPALETTE; var BitmapInfo;
      var Bits; Colors: Integer): Boolean;
    procedure SaveToStream(Stream: TStream; Icon: HIcon; SaveLength: Boolean;
      Colors: Integer);
    procedure BtnSaveClick(Sender: TObject);
    procedure SaveToIcoFile(Sender: TObject);
    procedure SaveToBmpFile(Sender: TObject);
    procedure BtnAboutClick(Sender: TObject);
    procedure BtnCancelClick(Sender: TObject);
    procedure BtnSelectClick(Sender: TObject);
    procedure BtnPrevClick(Sender: TObject);

  private
    { Private declarations }
    InitDirPath: string;
    NumberOfIcons: LongInt;
    sImages: TImageList;  //3.1
    Path: TFileName;
    SelPath,
    SelectedDir: TFileName;
    Drives: set of 0..25;
    FileMask: string;
    MaskOn: Boolean;
    DirOnly: Boolean;

  public
    { Public declarations }
    IconFName,
      IconFExt: string;
    IsBmp,
      IsIco: Boolean;
    IcoColors,
      BmpHeight,
      BmpWidth: integer;
  end;

var
  LsIconEditorForm: TLsIconEditorForm;

implementation

{$R *.DFM}

uses
  SaveIconDlgD3, AboutIconEditorD3;

Const
  MaskStr = '*.cpl;*.dll;*.drv;*.exe;*.ico;*.ocx';  //3.1
  
{ ====================================================================== }
{                          Global routines                               }
{ ====================================================================== }

function NumPos(a: Char; b: string; c: integer): integer;
var
  i, az: integer;
begin
  Result := MaxInt;
  if Length(b) > 0 then
  begin
    az := 0;
    for i := 1 to Length(b) do
      if b[i] = a then
      begin
        inc(az);
        if az = c then
        begin
          Result := i;
          exit;
        end;
      end;
  end;
end;  //NumPos

function GetCount(a: Char; b: String): integer;
var
  i: integer;
begin
  Result := 0;
  if Length(b) > 0 then
    for i := 1 to Length(b) do
      if b[i] = a then
        inc(Result);
end;  //GetCount

function AddSlash(Path: string): string;
begin
  if Path = '' then exit;
  if Path[Length(Path)] <> '\' then
    Result := Path + '\'
  else
    Result := Path;
end;  //AddSlash

function DelSlash(Path: String): string;
begin
  Result := Path;
  if Path <> '' then
   if Path[Length(Path)] = '\' then
     Delete(Result, Length(Path), 1);
end;  //DelSlash

function SortProc(I1, I2: TListItem; Para: integer): integer; stdcall;
var
  c1, c2, s1, s2: string;
begin
  Result := 0;
  c1 := ansiuppercase(i1.caption);
  c2 := ansiuppercase(i2.caption);
  if i1.subitems.count <= para then s1 := '' else s1 := i1.subitems[para];
  if i2.subitems.count <= para then s2 := '' else s2 := i2.subitems[para];
  if s2 > s1 then result := -1
  else if s1 > s2 then result := 1
  else if s1 = s2 then begin
    if c2 > c1 then result := -1;
    if c1 > c2 then result := 1;
  end;
end; //sortproc

Function GetWinDir: string;
var
  p: pChar;
begin
  GetMem(p, MAX_PATH);
  GetWindowsDirectory(p, MAX_PATH);
  Result := StrPas(p);
  FreeMem(p);
end;  //GetWinDir

{ ==== End of Global routines ==== }

{ ====================================================================== }
{                           LsIconEditor                                 }
{ =======================================================================}

procedure TLsIconEditorForm.FormCreate(Sender: TObject);
begin
  Integer(Drives) := GetLogicalDrives;
  CreateImages;
  FileMask := MaskStr;
  MaskOn := True; //False;
//  Path := AddSlash(GetCurrentDir);  //3.1
  Path := AddSlash(GetWinDir);  //3.1
  DirComboUpdate(Path);
  FileViewBuildList(Path, FileMask);
  Application.OnHint := DisplayHint;
end;  //FormCreate

procedure TLsIconEditorForm.DisplayHint(Sender: TObject);
begin
  Statusbar1.SimpleText := GetLongHint(Application.Hint);
end;  //DisplayHint

// Retrieves icons from selected file. then load them into IconView
procedure TLsIconEditorForm.IconViewLoadIcons(IFName: string);
var
  x: integer;
  Icon: TIcon;
  pFName: array[0..255] of char;
  oldCursor: TCursor;
  ListItem: TListItem;
begin
  if FileExists(IFName) then begin
    OldCursor := Screen.Cursor;
    Screen.CurSor := crHourGlass;
    NumberOfIcons := ExtractIcon(hInstance, StrPCopy(pFName, IFName), $FFFFFFFF);
    IconList.Clear;
    IconList.Height := 32;
    IconList.Width := 32;
    Image1.Picture := nil;
    with IconView do begin
      Items.Clear;
      ViewStyle := vsIcon;
      LargeImages := IconList;
      Items.BeginUpdate;
      try
        for x := 0 to NumberOfIcons - 1 do begin
          Icon := TIcon.Create;
          Icon.Handle := ExtractIcon(hInstance, pFName, x);
          IconList.AddIcon(Icon);
          ListItem := Items.Add;
          ListItem.Caption := Format('%d', [x]);
          ListItem.ImageIndex := x;
          Icon.Free;
        end;
      finally
        Items.EndUpdate;
        IconView.Selected := nil;
        Screen.Cursor := OldCursor;
      end;
    end; // with
  end; // if
//  BtnState;
end; //LoadIcons

{ ====================================================================== }
{   Procedures and functions listed below are used to save an icon to    }
{   a file stream either in 16-colors, 256-colors or hi-Color            }
{ ====================================================================== }

procedure TLsIconEditorForm.InitializeBitmapInfoHeader(Bitmap: HBITMAP;
  var bi: TBitmapInfoHeader; Colors: Integer);
var
  ds: TDIBSection;
  Bytes: Integer;

  function BytesPerScanline(PixelsPerScanline, BitsPerPixel,
    Alignment: Longint): Longint;
  begin
    Dec(Alignment);
    Result := ((PixelsPerScanline * BitsPerPixel) + Alignment) and not Alignment;
    Result := Result div 8;
  end;

begin
  ds.dsbmih.biSize := 0;
  Bytes := GetObject(Bitmap, SizeOf(ds), @ds);
  if Bytes = 0 then
    ShowMessage('Invalid Bitmap')
  else if (Bytes >= (sizeof(ds.dsbm) + sizeof(ds.dsbmih))) and
    (ds.dsbmih.biSize >= DWORD(sizeof(ds.dsbmih))) then
    bi := ds.dsbmih
  else
  begin
    FillChar(bi, sizeof(bi), 0);
    with bi, ds.dsbm do
    begin
      biSize := SizeOf(bi);
      biWidth := bmWidth;
      biHeight := bmHeight;
    end;
  end;
  if Colors <> 0 then
    case Colors of
      2     : bi.biBitCount := 1;
      16    : bi.biBitCount := 4;
      256   : bi.biBitCount := 8;
      65536 : bi.biBitCount := 16;  //3.1
    end
  else bi.biBitCount := ds.dsbm.bmBitsPixel * ds.dsbm.bmPlanes;
  bi.biPlanes := 1;
  if bi.biSizeImage = 0 then
    bi.biSizeImage := BytesPerScanLine(bi.biWidth, bi.biBitCount, 32) *
      Abs(bi.biHeight);
end; //InitializeBitmapInfoHeader

procedure TLsIconEditorForm.GetDIBSizes(Bitmap: HBITMAP; var InfoHeaderSize: DWORD;
  var ImageSize: DWORD; Colors: Integer);
var
  bi: TBitmapInfoHeader;
begin
  InitializeBitmapInfoHeader(Bitmap, bi, Colors);
  if bi.biBitCount > 8 then
  begin
    InfoHeaderSize := SizeOf(TBitmapInfoHeader);
    if (bi.biCompression and BI_BITFIELDS) <> 0 then
      Inc(InfoHeaderSize, 12);
  end
  else
    InfoHeaderSize := SizeOf(TBitmapInfoHeader) + SizeOf(TRGBQuad) *
      (1 shl bi.biBitCount);
  ImageSize := bi.biSizeImage;
end; //GetDIBSizes

function TLsIconEditorForm.GetDIB(Bitmap: HBITMAP; Palette: HPALETTE; var BitmapInfo;
  var Bits; Colors: Integer): Boolean;
var
  OldPal: hPalette;
  DC: HDC;
begin
  InitializeBitmapInfoHeader(Bitmap, TBitmapInfoHeader(BitmapInfo), Colors);
  OldPal := 0;
  DC := CreateCompatibleDC(0);
  try
    if Palette <> 0 then
    begin
      OldPal := SelectPalette(DC, Palette, False);
      RealizePalette(DC);
    end;
    Result := GetDIBits(DC, Bitmap, 0, TBitmapInfoHeader(BitmapInfo).biHeight,
              @Bits, TBitmapInfo(BitmapInfo), DIB_RGB_COLORS) <> 0;
  finally
    if OldPal <> 0 then SelectPalette(DC, OldPal, False);
    DeleteDC(DC);
  end;
end; //GetDIB

procedure TLsIconEditorForm.SaveToStream(Stream: TStream; Icon: HIcon;
  SaveLength: Boolean; Colors: Integer);
var
  ii: TIconInfo;
  MonoInfoSize, ColorInfoSize: DWORD;
  MonoBitsSize, ColorBitsSize: DWORD;
  MonoInfo, MonoBits, ColorInfo, ColorBits: Pointer;
  ci: TCursorOrIcon;
  List: TIconRec;
  Length: Longint;
begin
  FillChar(ci, SizeOf(ci), 0);
  FillChar(List, SizeOf(List), 0);
  if not GetIconInfo(Icon, ii) then
    ShowMessage('ERROR - unable to retrive IconInfo');
  try
    GetDIBSizes(ii.hbmMask, MonoInfoSize, MonoBitsSize, 2);
    GetDIBSizes(ii.hbmColor, ColorInfoSize, ColorBitsSize, Colors);
    MonoInfo := nil;
    MonoBits := nil;
    ColorInfo := nil;
    ColorBits := nil;
    try
      MonoInfo := AllocMem(MonoInfoSize);
      MonoBits := AllocMem(MonoBitsSize);
      ColorInfo := AllocMem(ColorInfoSize);
      ColorBits := AllocMem(ColorBitsSize);
      GetDIB(ii.hbmMask, 0, MonoInfo^, MonoBits^, 2);
      GetDIB(ii.hbmColor, 0, ColorInfo^, ColorBits^, Colors);
      if SaveLength then
      begin
        Length := SizeOf(ci) + SizeOf(List) + ColorInfoSize +
          ColorBitsSize + MonoBitsSize;
        Stream.Write(Length, SizeOf(Length));
      end;
      with ci do
      begin
        ci.wType := 1; //RC3_ICON;
        ci.Count := 1;
      end;
      Stream.Write(ci, SizeOf(ci));
      with List, PBitmapInfoHeader(ColorInfo)^ do
      begin
        Width := biWidth;
        Height := biHeight;
        Colors := biPlanes * biBitCount;
        DIBSize := ColorInfoSize + ColorBitsSize + MonoBitsSize;
        DIBOffset := SizeOf(ci) + SizeOf(List);
      end;
      Stream.Write(List, SizeOf(List));
      with PBitmapInfoHeader(ColorInfo)^ do
        Inc(biHeight, biHeight); { color height includes mono bits }
      Stream.Write(ColorInfo^, ColorInfoSize);
      Stream.Write(ColorBits^, ColorBitsSize);
      Stream.Write(MonoBits^, MonoBitsSize);
    finally
      FreeMem(ColorInfo, ColorInfoSize);
      FreeMem(ColorBits, ColorBitsSize);
      FreeMem(MonoInfo, MonoInfoSize);
      FreeMem(MonoBits, MonoBitsSize);
    end;
  finally
    DeleteObject(ii.hbmColor);
    DeleteObject(ii.hbmMask);
  end;
end; //SaveToStream

//Save the selected icon to an Icon file or a Bitmap file
procedure TLsIconEditorForm.BtnSaveClick(Sender: TObject);
begin
  if (IconView.Selected = nil) or
    (Image1.Picture.Icon.Empty) then
  begin
    ShowMessage('No icon selected');
    exit;
  end;

  with SaveIconForm do
  begin
    ShowModal;
    if ModalResult = mrOK then
    begin
      if IsIco then
      begin
        IcoColors := Colors;
        SaveToIcoFile(Sender);
      end
      else if IsBmp then
      begin
        SpEdtHeight.Update;
        SpEdtWidth.Update;
        BmpHeight := SpEdtHeight.Value;
        BmpWidth := SpEdtWidth.Value;
        SaveToBmpFile(Sender);
      end;
    end
    else exit;
  end;
end;  //BtnSaveClick

// Save selected icon to an .ico file
procedure TLsIconEditorForm.SaveToIcoFile(Sender: TObject);
var
  PEIcon: TIcon;
  FStream: TFileStream;
begin
  PEIcon := Image1.Picture.Icon;
  with SaveDialog1 do
  begin
    Title := 'Save selected icon to *.ico file';
    Filter := GraphicFilter(TIcon);
    DefaultExt := GraphicExtension(TIcon);
    InitialDir := InitDirPath;
    if SaveDialog1.Execute then
    begin
      if (SaveDialog1.FileName = '') or
        (IconView.Selected = nil) then exit;
      FStream := TFileStream.Create(SaveDialog1.FileName, fmCreate or fmOpenWrite);
      try
        SaveToStream(FStream, PEIcon.Handle, False, IcoColors);
      finally
        FStream.Free;
      end;
      InitDirPath := AddSlash(ExtractFilePath(SaveDialog1.FileName));
    end;
  end;
end; //SaveToIcoFile

// Convert the selected icon to bitmap then save it to a .bmp file
procedure TLsIconEditorForm.SaveToBmpFile(Sender: TObject);
var
  Bmp1,
    Bmp2: TBitmap;
  Icon1: TIcon;
  Rect1: TRect;
//  Image2: TImage;
//  BitsperPixel : integer;
begin
  Bmp1 := TBitmap.Create;
  Bmp2 := TBitmap.Create;
  Icon1 := Image1.Picture.Icon;
  try
    with Rect1 do
    begin
      Top := 0;
      Left := 0;
      Right := BmpWidth;
      Bottom := BmpHeight;
    end;

    with Bmp1 do
    begin
      width := Icon1.Width;
      Height := Icon1.Height;
      canvas.draw(0, 0, Icon1);
    end;

    with SaveIconForm do
    begin
      if RBtn16Color.Checked then
        Bmp2.PixelFormat := pf4bit
      else if RBtn256Color.Checked then
        Bmp2.PixelFormat := pf8bit
      else if RBtnHiColor.Checked then
        Bmp2.PixelFormat := pf24bit;
    end;

    with Bmp2 do
    begin
      Width := BmpWidth;
      Height := BmpHeight;
      Canvas.StretchDraw(Rect1, Bmp1);
    end;
    with Image2 do
    begin
      Width := BmpWidth;
      Height := BmpHeight;
      Visible := False;
      Picture := nil;
      Picture.Assign(Bmp2);
    end;
    with SaveDialog1 do
    begin
      Title := 'Save selected icon to *.bmp file';
      Filter := GraphicFilter(TBitmap);
      DefaultExt := GraphicExtension(TBitmap);
      InitialDir := InitDirPath;
      if SaveDialog1.Execute then
      begin
        if (SaveDialog1.FileName = '') then exit;
        Image2.Picture.Bitmap.SaveToFile(SaveDialog1.FileName);
        InitDirPath := AddSlash(ExtractFilePath(SaveDialog1.FileName));
      end;
    end;
  finally
    Bmp1.Free;
    Bmp2.Free;
  end;
end; //SaveToBmpFile


{ ====================================================================== }
{                      PathCombo and FileView                            }
{ ====================================================================== }

procedure TLsIconEditorForm.CreateImages;
var
  SysIL: UInt;
  SFI: TSHFileInfo;
begin
  sImages := TImageList.Create(self);
  SysIL := SHGetFileInfo('', 0, SFI, SizeOf(SFI), SHGFI_SYSICONINDEX
    or SHGFI_SMALLICON);
  if SysIL <> 0 then
  begin
    sImages.Handle := SysIL;
    sImages.ShareImages := True;
  end;
  FileView.SmallImages := sImages;  //3.1
end;  //CreateImages

procedure TLsIconEditorForm.DirComboUpdate(cPath: TFileName);
var
  Drv: byte;
  selDrv: byte;
  isDrive: boolean;
  i,
  cnt: integer;
begin
  selPath := AddSlash(cPath);
  DirCombo.items.beginupdate;
  DirCombo.items.clear;
  i := -1;
  isDrive := cPath = 'My Computer';
  selDrv := ord(upcase(cPath[1])) - ord('A');
  DirCombo.items.add('0:My Computer');
  for Drv := 0 to 25 do
    if Drv in drives then
    begin
      DirCombo.items.add('1:' + char(Drv + ord('A')) + ':\');
      if (not isDrive) and (Drv = selDrv) then
      begin
//          if cPath[length(cPath) - 1] = ':' then
//            i := DirCombo.items.count - 1
//          else
        for cnt := 1 to getcount('\', cPath) - 1 do
          DirCombo.items.add(IntToStr(cnt + 1) + ':'
                + copy(cPath, 1, numpos('\', cPath, cnt + 1)));
        i := DirCombo.items.count - 1;
      end;
      if isDrive then
        i := 0;
    end;  //if
  if i > -1 then
    DirCombo.items[i] := 'T' + DirCombo.items[i];
  DirCombo.itemindex := -1;
  DirCombo.items.endupdate;
  DirCombo.itemindex := i;
end; //DirComboUpdate

procedure TLsIconEditorForm.DirComboChange(Sender: TObject);
var
  DirStr: string;
begin
  if DirCombo.droppeddown then exit;
  if DirCombo.itemindex <= 0 then exit;
  DirStr := DirCombo.items[DirCombo.itemindex];
  DirStr := copy(DirStr, pos(':', DirStr) + 1, MaxInt);
  if (DirStr <> 'My Computer') and (not DirectoryExists(DirStr)) then
  begin
    ShowMessage('"' + DirStr + ' "  is inaccessible');
    DirComboUpdate(SelPath);
    Path := SelPath;
  end
  else begin
    Path := DirStr;
//    SelectedDir := Path;
    DirComboUpdate(Path);
    FileViewBuildList(Path, FileMask);
  end;
end; //DirComboChange

procedure TLsIconEditorForm.DirComboDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
var
  DirName,
    DirStr: string;
  sfi: tshfileinfo;
  tab: integer;
  mode: integer;
begin
  fillchar(sfi, sizeof(tshfileinfo), 0);
  with DirCombo, DirCombo.canvas do
  begin
    Canvas.Font.Name := 'MS Sans Serif';
    Canvas.Font.Size := 8;
    mode := SHGFI_ICON or SHGFI_SMALLICON or SHGFI_DISPLAYNAME;
    DirStr := items[index];
    if DirStr[1] = 'T' then
    begin
      DirStr := copy(DirStr, 2, maxint);
      mode := mode or SHGFI_OPENICON;
    end;
    sImages.drawingstyle := dsTransparent;
    if odselected in state then
      sImages.drawingstyle := dsSelected;
    fillrect(rect);
    if DirStr = '0:My Computer' then
    begin
      sfi.szDisplayname := 'My Computer';  //Drives';
      sfi.iIcon := 15;
    end
    else
      SHGetFileInfo(pchar(copy(DirStr, pos(':', DirStr) + 1, maxint)), 0,
        sfi, sizeof(TSHFileInfo), mode);
    DirName := sfi.szDisplayName;
    tab := StrToInt(copy(DirStr, 1, pos(':', DirStr) - 1)) * 8;

    if not droppeddown then tab := 0;

    TextOut(Rect.Left + 20 + tab, Rect.Top + 2, DirName);
    sImages.draw(canvas, Rect.Left + 2 + tab, Rect.Top + 2, sfi.iIcon);
    sImages.drawingstyle := dsNormal;
  end;
end;  //DirComboDrawItem

procedure TLsIconEditorForm.DirComboClick(Sender: TObject);
begin
  with DirCombo do
  begin
    if ItemIndex = -1 then exit;
    if (ItemIndex = 0) then
    begin
      Path := 'My Computer';
      SelectedDir := 'My Computer';
      DirComboUpdate(SelPath);   //Path);
    end
    else begin
      Path := Items[ItemIndex];
      SelectedDir := Path;
      FileViewBuildList(Path, FileMask);
    end;
  end;
end;  //DirComboClick

procedure TLsIconEditorForm.FileViewBuildList(cPath: String; FMask: string);
var
  MaskPtr: PChar;
  Ptr: PChar;
begin
  FileView.Items.BeginUpdate;
  FileView.Items.Clear;
  if MaskOn = True then
  begin
    DirOnly := True;
    FileViewAddFiles(cPath, '*.*');
    DirOnly := False;
    // parse the FMask string, then process each mask in terms  //3.1>
    MaskPtr := PChar(FMask);
    while MaskPtr <> nil do
    begin
      Ptr := StrScan(MaskPtr, ';');
      if Ptr <> nil then
        Ptr^ := #0;
      FileViewAddFiles(cPath, StrPas(MaskPtr));
      if Ptr <> nil then
      begin
        Ptr^ := ';';
        inc(Ptr);
      end;
      MaskPtr := Ptr;
    end;  //3.1<
  end else
  begin
    DirOnly := False;
    FileViewAddFiles(cPath, '*.*');
  end;
  FileView.Items.EndUpdate;
end;  //FileViewBuildList

procedure TLsIconEditorForm.FileViewAddFiles(cDir: String; Mask: String);
var
  CurDir,
  FName,
    FileName,
    FileOrDir: string;
  pFName: array[0..255] of char;  //3.1
  sfi: TSHFileInfo;
  hFindFile: THandle;
  Win32FD: TWin32FindData;
  OldCur: TCursor;
begin
  OldCur := Screen.Cursor;
  CurDir := AddSlash(cDir);
  hFindFile := FindFirstFile(PChar(CurDir + Mask), Win32FD);
  if hFindFile <> INVALID_HANDLE_VALUE then
  try
    Screen.Cursor := crHourGlass;
    repeat
      with Win32FD do
      begin
        // if DirOnly, display folders only
        if (not DirOnly) or (dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY > 0) then  //3.1
        begin
          if (dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY > 0) then
            FileOrDir := 'dir'
          else
            FileOrDir := 'file';

          FName := StrPas(Win32FD.cFileName);
          FileName := CurDir + FName;

          if (MaskOn = True) and (FileOrDir = 'file') and
            (UpperCase(ExtractFileExt(FName)) <> '.ICO') then  //3.1>
          begin
            if ExtractIcon(hInstance, StrPCopy(pFName, FileName), $FFFFFFFF) <= 0 then
              continue;
          end;  //3.1<

          if (FName = '.') or (FName = '..') then continue;
          // Get Shell information about this file
          SHGetFileInfo(PChar(FileName), 0, sfi, SizeOf(sfi),
                        SHGFI_SYSICONINDEX or SHGFI_DISPLAYNAME);

          with FileView.Items.Add do
          begin
            Caption := FName;
            ImageIndex := sfi.iIcon;
            SubItems.Add(FileName);
            SubItems.Add(FileOrDir);
          end;  //with FileView
        end;  //if (not DirOnly ...  //3.1
      end;  //with Win32FD
    until not FindNextFile(hFindFile, Win32FD);
  finally
    Windows.FindClose(hFindFile);
    FileView.CustomSort(@SortProc, 1);
    Screen.Cursor := OldCur;
  end;
end;  //FileViewAddFiles

procedure TLsIconEditorForm.FileViewDblClick(Sender: TObject);
var
  CurDir,
  NewDir: TFileName;
begin
  CurDir := '';
  NewDir := '';
  with FileView do
  begin
    if Selected = nil then
      exit;
    if Selected.SubItems[1] = 'dir' then
    begin
      CurDir := Selected.SubItems[0];
      NewDir := AddSlash(CurDir);
      if NewDir <> '' then
      begin
        DirComboUpDate(NewDir);
        FileViewBuildList(NewDir, FileMask);
        Path := NewDir;
      end
    end;
  end;
end;  //FileViewDblClick

procedure TLsIconEditorForm.BtnPrevClick(Sender: TObject);
var
  CurPath, NewPath: string;
begin
  CurPath := '';
  NewPath := '';
  CurPath := DelSlash(Path);
  NewPath := ExtractFilePath(CurPath);
  if NewPath <> '' then
  begin
    DirComboUpDate(NewPath);
    FileViewBuildList(NewPath, FileMask);
    Path := NewPath;
  end;
end;  //BtnPrevClick


procedure TLsIconEditorForm.FileViewClick(Sender: TObject);
begin
  if (FileView.Selected = nil) then exit;
  if (FileView.Selected.SubItems[1] <> 'dir') then
  begin
    IconFName := FileView.Selected.SubItems[0];
//    ShowMessage(IconFName);
    IconViewLoadIcons(IconFName);
  end;
end;  //FileViewClick

procedure TLsIconEditorForm.IconViewClick(Sender: TObject);
var
  pFName: array[0..255] of Char;
begin
  if IconView.Selected = nil then
    exit;
  if (not FileExists(IconFName)) then
    exit;
  IconFExt := ExtractFileExt(IconFName);
  if UpperCase(IconFExt) = 'ICO' then
    Image1.Picture.Icon.LoadFromFile(IconFName)
  else
    Image1.Picture.Icon.Handle := ExtractIcon(hInstance,
                                              StrPCopy(pFName, IconFName),
                                              IconView.Selected.ImageIndex);
end;  //IconViewClick

{ ====================================================================== }
{                  OnClick routines for ToolBar Buttons                  }
{ ====================================================================== }

procedure TLsIconEditorForm.AllFiles1Click(Sender: TObject);
begin
  MaskON := False;
  FileMask := '*.*';
  FileViewBuildList(Path, FileMask);
  BtnFileMask.Down := False; //True;
  AllFiles1.Checked := True;
  Masked1.Checked := False;
end;  //AllFiles1Click

procedure TLsIconEditorForm.Masked1Click(Sender: TObject);
begin
  MaskOn := True;
  FileMask := MaskStr;   //'*.cpl;*.dll;*.drv;*.exe;*.ico;*.ocx';  //3.1
  FileViewBuildList(Path, FileMask);
  BtnFileMask.Down := True;
  Masked1.Checked := True;
  AllFiles1.Checked := False;
end;  //Masked1Click

procedure TLsIconEditorForm.BtnClearClick(Sender: TObject);
begin
  IconView.Items.Clear;
  Image1.Picture := nil;
end;  //BtnClearClick

procedure TLsIconEditorForm.BtnAboutClick(Sender: TObject);
begin
  AboutForm.Show;
end;  //BtnAboutClick

procedure TLsIconEditorForm.BtnSelectClick(Sender: TObject);
begin
  if (IconView.Selected = nil) or
    (Image1.Picture.Icon.Empty) then
  begin
    ShowMessage('No icon selected');
    exit;
  end;
//  ShowMessage('Selected icon will be Icon Property' + #13 +
//              'for the application form');
  ModalResult := mrOK;
end;


procedure TLsIconEditorForm.BtnCancelClick(Sender: TObject);
begin
  ModalResult := mrCancel;
  Close;
end;

end.
