{********************************************************************}
{                                                                    }
{                      pFIBErrorHandler                              }
{        component for handling IB errors                            }
{                                                                    }
{     Copyright (c) 04.2000 by Serge Buzadzhy                        }
{     email:  serge_buzadzhy@mail.ru,                                }
{             FidoNet: 2:467/44.37                                   }
{                                                                    }
{                                                                    }
{********************************************************************}


unit pFIBErrorHandler;

interface
 uses sysutils,Classes,FIB,pFIBDatabase, ib_errorcodes,
  ibase,IB_Intf, ib_externals, StdConsts,
  DB;


 type

    TOptionErrorHandler=(oeException ,oeForeignKey,oeLostConnect);
    TKindIBError =(keNoError,keException ,keForeignKey,keLostConnect,
     keSecurity, keOther
    );

    TOnFIBErrorEvent = procedure (Sender:TObject;ErrorValue:EFIBError;
                       KindIBError:TKindIBError;
                        var DoRaise:boolean
                       ) of object;

    TOptionsErrorHandler= set of TOptionErrorHandler;

    TpFibErrorHandler= class (TComponent)
    private
     FLastError:TKindIBError;
     FOnFIBErrorEvent:TOnFIBErrorEvent;
     FOptions:TOptionsErrorHandler;
     FExceptionNumber:integer;
     procedure DefaultOnError(Sender:TObject;ErrorValue:EFIBError;
                        var DoRaise:boolean);
    public
     constructor Create(AOwner:TComponent);override;
     destructor Destroy;override;
     procedure DoOnErrorEvent(Sender:TObject;ErrorValue:EFIBError;
                        var DoRaise:boolean); dynamic; // for internal use
     procedure DoOnLostConnect(ErrorValue:EFIBError);
     property ExceptionNumber:integer read FExceptionNumber;
     property LastError:TKindIBError read FLastError;
    published
     property OnFIBErrorEvent:TOnFIBErrorEvent read FOnFIBErrorEvent write FOnFIBErrorEvent;
     property Options:TOptionsErrorHandler read FOptions write FOptions
      default [oeException,oeLostConnect]
     ;
    end;

implementation

constructor TpFibErrorHandler.Create(AOwner:TComponent);//override;
begin
 if ErrorHandlerRegistered then
  raise Exception.Create('Error handler instance already present');
 inherited Create(AOwner);
 RegisterErrorHandler(Self);
 Options:=[oeException,oeLostConnect];
 FLastError:=keNoError;
end;

destructor TpFibErrorHandler.Destroy;//override;
begin
 UnRegisterErrorHandler;
 inherited Destroy;
end;

procedure TpFibErrorHandler.DefaultOnError(Sender:TObject;
                        ErrorValue:EFIBError;
                        var DoRaise:boolean);
const EPrefix='EXCEPTION ';
var p:integer;
    IBErrorCode: Long;

begin
  IBErrorCode := StatusVectorArray[1];
  FExceptionNumber:=-1;
  FLastError:=keOther;
  with ErrorValue do
  if SQLCode=-836 then // Is Developer Exception
  begin
     FLastError:=keOther;
     if oeException in Options then begin
      p:=Pos(EPrefix,UpperCase(ErrorValue.Message));
      if p>0 then
       ErrorValue.Message:=Copy(ErrorValue.Message,p+10,MaxInt);
      p:=Pos(EPrefix,UpperCase(ErrorValue.Message));
      if p>0 then
       ErrorValue.Message:=Copy(ErrorValue.Message,p+10,MaxInt);
      p:=Pos('.',UpperCase(ErrorValue.Message));
      if p>0 then
      try
       FExceptionNumber:=StrToInt(Copy(ErrorValue.Message,1,p-1));
       ErrorValue.Message:=Copy(ErrorValue.Message,p+1,MaxInt);
      except
      end;
     end;  
  end   // end Developer Exception
  else
  if IBErrorCode=isc_lost_db_connection then
  if oeLostConnect in Options then
  begin
      FLastError:=keLostConnect;
      DoOnLostConnect(ErrorValue)
  end
  else
  if SQLCode=-551 then FLastError:=keSecurity;
end;

type THack=class(TpFIBDatabase);


procedure TpFibErrorHandler.DoOnLostConnect(ErrorValue:EFIBError);
var i:integer;
    Actions:TOnLostConnectActions;
begin
 for i:=0 to Pred(pDataBaseCount) do
 with THack(pDataBase(i)) do begin
  if not Connected then Continue;
  Actions:=laCloseConnect;
  DoOnLostConnect(pDataBase(i),ErrorValue,Actions);
 end;
end;

procedure TpFibErrorHandler.DoOnErrorEvent(
           Sender:TObject;ErrorValue:EFIBError;
              var DoRaise:boolean
           ); //dynamic;
begin
 DefaultOnError(Sender,ErrorValue,DoRaise);
 if Assigned(FOnFIBErrorEvent) then
  FOnFIBErrorEvent(Sender,ErrorValue,LastError,DoRaise);
end;



end.

