{********************************************************************}
{                                                                    }
{ TpFIBDatabase                                                      }
{        Patch for TFIBDatabase from Free IB components              }
{                                                                    }
{     Copyright (c) 07.1999 by Serge Buzadzhy                        }
{     email:  serge_buzadzhy@mail.ru,                                }
{             FidoNet: 2:467/44.37                                   }
{                                                                    }
{                                                                    }
{********************************************************************}


unit pFIBDatabase;

interface
{$I FIBPlus.INC}

uses
  Windows, Messages, SysUtils, Classes,
  {$IFNDEF FOR_CONSOLE}
  Forms,   Dialogs   ,extctrls, // IS GUI units
  {$ENDIF}
  DB,  ibase,IB_Intf, ib_externals,FIB,FIBDatabase;

type


  TFIBLoginEvent =
   procedure(Database: TFIBDatabase; LoginParams: TStrings; var DoConnect:boolean )
  of object;


  TOnLostConnectActions =(laTerminateApp,laCloseConnect,laIgnore,laWaitRestore);
  TFIBLostConnectEvent =
   procedure(Database: TFIBDatabase; E:EFIBError;var Actions:TOnLostConnectActions)
  of object;

  TFIBRestoreConnectEvent =   procedure of object;

  TEndTrEvent=procedure(EndingTR:TFIBTransaction;
   Action: TTransactionAction; Force: Boolean)of object;

  TpFIBDatabase = class(TFIBDatabase)
  private
   {$IFNDEF FOR_CONSOLE}
    vTimer    : TTimer;
   {$ENDIF} 
    FAliasName:string;
    FRewriteAlias:boolean;
    FOnLogin :TFIBLoginEvent;
    FOnLostConnect:TFIBLostConnectEvent;
    FOnErrorRestoreConnect:TFIBLostConnectEvent;
    FAfterRestoreConnect:TFIBRestoreConnectEvent;
    FBeforeStartTr:TNotifyEvent ;
    FAfterStartTr :TNotifyEvent ;
    FBeforeEndTr  :TEndTrEvent  ;
    FAfterEndTr   :TEndTrEvent  ;

    procedure SetAliasName(Value:string);
    function  GetWaitRC:Cardinal;
    procedure SetWaitRC(Value:Cardinal);
  protected
    procedure Loaded; override;
    procedure InternalClose(Force: Boolean); override;
    //    virtual  FIBDatabase
    procedure CloseLostConnect;
    procedure DoOnLostConnect
     (Database: TFIBDatabase; E:EFIBError;var Actions:TOnLostConnectActions);dynamic;
    procedure DoOnErrorRestoreConnect
     (Database: TFIBDatabase; E:EFIBError;var Actions:TOnLostConnectActions);dynamic;
    procedure DoAfterRestoreConnect;dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function  ReadParamsFromAlias:boolean;dynamic;
    procedure WriteParamsToAlias ;dynamic;
    procedure Open;override; //    virtual  FIBDatabase
    function  ExTestConnected(Actions:TOnLostConnectActions): Boolean;
    procedure WaitRestoreConnect;
    procedure StopWaitRestoreConnect;
    procedure RestoreConnect(Sender:TObject);
    procedure ForceCloseTransactions;
  published
    property AliasName:string  read FAliasName write SetAliasName ;
    property SaveDBParams:boolean  read FRewriteAlias write FRewriteAlias default
     true ;
    property WaitForRestoreConnect:Cardinal  read GetWaitRC write SetWaitRC
     default 30000
    ;
    property OnLogin :TFIBLoginEvent  read FOnLogin write FOnLogin ;
    property OnLostConnect:TFIBLostConnectEvent  read FOnLostConnect write FOnLostConnect;
    property OnErrorRestoreConnect:TFIBLostConnectEvent  read FOnErrorRestoreConnect write FOnErrorRestoreConnect;
    property AfterRestoreConnect:TFIBRestoreConnectEvent  read FAfterRestoreConnect write FAfterRestoreConnect;
    property BeforeStartTransaction:TNotifyEvent  read FBeforeStartTr write FBeforeStartTr;
    property AfterStartTransaction :TNotifyEvent  read FAfterStartTr  write FAfterStartTr;
    property BeforeEndTransaction  :TEndTrEvent   read FBeforeEndTr   write FBeforeEndTr;
    property AfterEndTransaction   :TEndTrEvent   read FAfterEndTr    write FAfterEndTr;

  end;

  TTPBMode=(tpbDefault,tpbReadCommitted,tpbRepeatableRead);

  TpFIBTransaction = class(TFIBTransaction)
  private
   FTPBMode:TTPBMode;
   FBeforeStart:TNotifyEvent;
   FAfterStart :TNotifyEvent;
   FBeforeEnd  :TEndTrEvent;
   FAfterEnd   :TEndTrEvent;
   function StoreTRParams:boolean;
  protected
   procedure EndTransaction(Action: TTransactionAction; Force: Boolean); override;
  public
   constructor Create(AOwner:TComponent); override;
   procedure StartTransaction; override;
  published
   property BeforeStart:TNotifyEvent  read FBeforeStart write FBeforeStart;
   property AfterStart :TNotifyEvent  read FAfterStart  write FAfterStart;
   property BeforeEnd  :TEndTrEvent  read FBeforeEnd   write FBeforeEnd;
   property AfterEnd   :TEndTrEvent  read FAfterEnd    write FAfterEnd;
   property TPBMode:TTPBMode  read FTPBMode write FTPBMode default tpbReadCommitted;
   property TRParams stored StoreTRParams;
  end;


function pDataBaseCount:integer;
function pDataBase(Index:integer):TpFibDataBase;

implementation

uses RegUtils,FIBQuery;

const RegRoot='FIBC_Software';


type  THackTransaction = class(TFIBTransaction)
      end;

      THackFIBQuery  = class (TFIBQuery)
      end;

var vDataBases:TList;

function pDataBaseCount:integer;
begin
 Result:=vDataBases.Count
end;

function pDataBase(Index:integer):TpFibDataBase;
begin
 Result:=nil;
 if (Index<0) or (Index>=vDataBases.Count) then  Exit;
 Result:=TpFibDataBase(vDataBases[Index])
end;

constructor TpFIBDatabase.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FRewriteAlias:=true;
   FAliasName:='';
   {$IFNDEF FOR_CONSOLE}
    vTimer          := TTimer.Create(Self);
    vTimer.Enabled  := False;
    vTimer.Interval := 30000; // 
    vTimer.OnTimer  := RestoreConnect;
   {$ENDIF}
   vDataBases.Add(Self)
end;

destructor TpFIBDatabase.Destroy;
begin
 vDataBases.Remove(Self);
 inherited Destroy;
end;

procedure TpFIBDatabase.Loaded;
begin
 inherited Loaded;
end;

procedure TpFIBDatabase.SetAliasName(Value:string);
begin
 if not (csLoading in ComponentState) then  CheckInactive;
 FAliasName:=Value
end;

function  TpFIBDatabase.GetWaitRC:Cardinal;
begin
 {$IFNDEF FOR_CONSOLE}
  Result:=vTimer.Interval
 {$ENDIF}
end;

procedure TpFIBDatabase.SetWaitRC(Value:Cardinal);
begin
 {$IFNDEF FOR_CONSOLE}
  vTimer.Interval:=Value
 {$ENDIF} 
end;

// Alias Works

function TpFIBDatabase.ReadParamsFromAlias:boolean; //dynamic;
var    Values :Variant;
       i:integer;
begin
Values :=
 DefReadFromRegistry(['Software',RegRoot,'Aliases',FAliasName],
   ['Database Name',
    DPBConstantNames[isc_dpb_user_name],
    DPBConstantNames[isc_dpb_lc_ctype],
    DPBConstantNames[isc_dpb_sql_role_name]
   ]
 );
 Result:=VarType(Values)<>varBoolean;
 if not Result then Exit; //
 for i:=0 to  3 do begin
  if Values[1,i] then
   case i of
    0:  DBName:=Values[0,i];
    1:  DBParamByDPB[isc_dpb_user_name]     :=Values[0,i];
    2:  DBParamByDPB[isc_dpb_lc_ctype]      :=Values[0,i];
    3:  DBParamByDPB[isc_dpb_sql_role_name] :=Values[0,i];
   end
  else
   if i=0 then Result:=false
 end;
end;

procedure TpFIBDatabase.WriteParamsToAlias ;//dynamic;
begin
 DefWriteToRegistry(['Software',RegRoot,'Aliases',FAliasName],
   ['Database Name'
   ,
    DPBConstantNames[isc_dpb_user_name],
    DPBConstantNames[isc_dpb_lc_ctype],
    DPBConstantNames[isc_dpb_sql_role_name]
   ],
   [DBName
   ,
    DBParamByDPB[isc_dpb_user_name],
    DBParamByDPB[isc_dpb_lc_ctype],
    DBParamByDPB[isc_dpb_sql_role_name]
   ]
 );
end;

procedure TpFIBDatabase.ForceCloseTransactions;
var i:integer;
begin
  for i := 0 to FTransactions.Count - 1 do begin
    try
      if FTransactions[i] <> nil then
        Transactions[i].OnDatabaseDisconnecting(Self);
    except
    end;
  end;
end;

procedure TpFIBDatabase.Open;     //override;
var DoConnect,NeedAlias:boolean;
begin
  CheckInactive;
  {$IFNDEF FOR_CONSOLE}
  NeedAlias:=FAliasName<>'';
  if  NeedAlias then
   if not ReadParamsFromAlias then
    DBName:=InputBox('??', 'Enter database name', DBName);
  {$ENDIF}  
  DoConnect:=true;
  if Assigned(FOnLogin) then FOnLogin(Self,DBParams,DoConnect);
  if not DoConnect then    Exit;
  inherited Open;
  if  FRewriteAlias and NeedAlias then WriteParamsToAlias;
end;

// Lost Connection works

procedure TpFIBDatabase.InternalClose(Force: Boolean); //override;
var Actions:TOnLostConnectActions;
begin
 Actions:=laCloseConnect;
 if Connected then
  if ExTestConnected(Actions)  then   inherited InternalClose(Force);
end;

function TpFIBDatabase.ExTestConnected(Actions:TOnLostConnectActions): Boolean;
begin
  result := Connected;
  if result then begin
    try
      BaseLevel;
    except
     on E:EFIBError do begin
       Result := False;
       DoOnLostConnect(Self,E,Actions);
     end
    end;
  end;
end;

procedure TpFIBDatabase.DoOnLostConnect
     (Database: TFIBDatabase; E:EFIBError;var Actions:TOnLostConnectActions);//dynamic;
begin
  if Assigned(FOnLostConnect) then FOnLostConnect(Self,E,Actions);
  if Actions=laTerminateApp then begin
   CloseLostConnect;
//   ShowMessage(E.Message);
   {$IFNDEF FOR_CONSOLE}
   Application.Terminate
   {$ENDIF}
  end;                         
  if  Actions in [laCloseConnect,laWaitRestore] then   CloseLostConnect;
  if  Actions =laWaitRestore then WaitRestoreConnect;
end;

procedure TpFIBDatabase.DoOnErrorRestoreConnect
     (Database: TFIBDatabase; E:EFIBError;var Actions:TOnLostConnectActions);//dynamic;
begin
  if Assigned(FOnErrorRestoreConnect) then FOnErrorRestoreConnect(Self,E,Actions);
end;

procedure TpFIBDatabase.DoAfterRestoreConnect ;//dynamic;
begin
  if Assigned(FAfterRestoreConnect) then    FAfterRestoreConnect ;
end;

procedure TpFIBDatabase.CloseLostConnect;
var i:integer;
begin
 //   IB Api
 // Let's avoid of calls IB Api
   FHandle:=nil;
   FreeFIBTLGlobals;
    for i := 0 to Pred(DataSetCount) do
     try
      if DataSets[i] <> nil then
       if DataSets[i].Owner is TFIBQuery then
        with THackFIBQuery(DataSets[i].Owner) do begin
          FHandle:=nil;
          Close;
          FPrepared:=false;
        end;
     except
 //      raise;
     end;
    for i:=0 to Pred(TransactionCount) do     try
     if (Transactions[i] <> nil)  and (Transactions[i].Active) then
     with THackTransaction(Transactions[i]) do begin
       FHandleIsShared:=true;
       EndTransaction(TACommit,true)
     end;
    except
//      raise
    end;
end;



procedure TpFIBDatabase.RestoreConnect(Sender:TObject);
var Actions:TOnLostConnectActions;
begin
  if Connected then Exit;
  try
   Actions  :=laIgnore;
   Connected:=true;
  {$IFNDEF FOR_CONSOLE}
   vTimer.Enabled:=false;
  {$ENDIF} 
   DoAfterRestoreConnect ;
  except
   On E:EFIBError do begin
    DoOnErrorRestoreConnect(Self,E,Actions) ;
    if Actions=laTerminateApp then begin
      {$IFNDEF FOR_CONSOLE}
        ShowMessage(E.Message);
        Application.Terminate
      {$ENDIF}        
    end;
   end
  end;
end;

procedure TpFIBDatabase.WaitRestoreConnect;
begin
  {$IFNDEF FOR_CONSOLE}
  vTimer.Enabled :=true;
  {$ENDIF}
end;

procedure TpFIBDatabase.StopWaitRestoreConnect;
begin
 {$IFNDEF FOR_CONSOLE}
  vTimer.Enabled := false
 {$ENDIF} 
end;


/// TpFIBTransaction

constructor TpFIBTransaction.Create(AOwner: TComponent); //override;
begin
 inherited Create(AOwner);
 FTPBMode :=tpbReadCommitted;
end;

procedure TpFIBTransaction.StartTransaction; //override;
begin
 if InTransaction then Exit;
 with TRParams do
  if  FTPBMode in  [tpbReadCommitted, tpbRepeatableRead] then begin
   Clear;
   Add('write');
   Add('nowait');
   Add('rec_version');
   if FTPBMode=tpbReadCommitted then Add('read_committed');
  end;
 if (DefaultDataBase is TpFIBDataBase) and
  Assigned( TpFIBDataBase(DefaultDataBase ).FBeforeStartTr )
 then
  TpFIBDataBase(DefaultDataBase ).FBeforeStartTr(Self);
 if Assigned(FBeforeStart) then FBeforeStart(Self);

 inherited StartTransaction;
 
 if Assigned(FAfterStart)  then FAfterStart(Self);
 if (DefaultDataBase is TpFIBDataBase) and
  Assigned( TpFIBDataBase(DefaultDataBase ).FAfterStartTr )
 then
  TpFIBDataBase(DefaultDataBase ).FAfterStartTr(Self);
end;

procedure TpFIBTransaction.EndTransaction(Action: TTransactionAction; Force: Boolean);// override;
begin
 if (DefaultDataBase is TpFIBDataBase) and
  Assigned( TpFIBDataBase(DefaultDataBase ).FBeforeEndTr )
 then
  TpFIBDataBase(DefaultDataBase ).FBeforeEndTr(Self,Action, Force);
 if Assigned(FBeforeEnd) then FBeforeEnd(Self,Action, Force);
 inherited EndTransaction(Action, Force);
 if Assigned(FAfterEnd)  then FAfterEnd(Self,Action, Force);
 if (DefaultDataBase is TpFIBDataBase) and
  Assigned( TpFIBDataBase(DefaultDataBase ).FAfterEndTr )
 then
  TpFIBDataBase(DefaultDataBase ).FAfterEndTr(Self,Action, Force);
end;

function TpFIBTransaction.StoreTRParams:boolean;
begin
 Result:=FTPBMode=tpbDefault
end;

initialization
 vDataBases:=TList.Create;
finalization
 vDataBases.Free
end.


