unit FIBSQLMonitor;

interface

{$I FIBPlus.INC}
uses
  SysUtils, Windows, Messages, Classes, StdConsts, FIBQuery,
  FIBDataSet, FIBDatabase,
 {$IFNDEF FOR_CONSOLE}
  Dialogs, Forms, Controls
 {$ENDIF}



   {$IFDEF  INC_SERVICE_SUPPORT}

  , IB_Services
   {$ENDIF}
  ;

const
  WM_MIN_FIBSQL_MONITOR = WM_USER;
  WM_MAX_FIBSQL_MONITOR = WM_USER + 512;
  WM_FIBSQL_SQL_EVENT = WM_MIN_FIBSQL_MONITOR + 1;

type
  TFIBSQLMonitorHook = class;
  TCustomFIBSQLMonitor = class;


  (*
   * TFIBSQLMonitor
   *)
  TSQLEvent = procedure(EventText: String) of object;

  TCustomFIBSQLMonitor = class(TComponent)
  private
    FAtom: TAtom;
    FHWnd: HWND;
    FThread: THandle;
    FOnSQLEvent: TSQLEvent;
  protected
    procedure MonitorHandler(var Msg: TMessage); virtual;
    property OnSQL: TSQLEvent read FOnSQLEvent write FOnSQLEvent;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

  TFIBSQLMonitor = class(TCustomFIBSQLMonitor)
  published
    property OnSQL;
  end;

  (*
   * TFIBSQLMonitorHook
   *)
  TFIBSQLMonitorHook = class(TObject)
  private
    FSharedBuffer,                   // MMF for shared memory.
    FWriteLock,                      // Only one writer at a time.
    FWriteEvent,                     // The SQL log has been written.
    FWriteFinishedEvent,             // The SQL log write is finished.
    FReadEvent,                      // All readers are ready
    FReadFinishedEvent: THandle;     // The SQL log read is now finished.
    FBuffer: PChar;                  // The shared buffer.
    FMonitorCount: PInteger;         // Number of registered monitors.
    FReaderCount: PInteger;          // Number of monitors currently reading
    FBufferSize: PInteger;           // Size of written buffer.
  protected
    procedure Lock;
    procedure Unlock;
    procedure BeginWrite;
    procedure EndWrite;
    procedure BeginRead;
    procedure EndRead;
    procedure WriteSQLData(Text: String);
  public
    constructor Create;
    destructor Destroy; override;
    function MonitorCount: Integer;
    procedure RegisterMonitor;
    procedure UnregisterMonitor;
    function ReadSQLData: String;
    procedure SQLPrepare(qry: TFIBQuery); virtual;
    procedure SQLExecute(qry: TFIBQuery); virtual;
    procedure SQLFetch(qry: TFIBQuery); virtual;
    procedure DBConnect(db: TFIBDatabase); virtual;
    procedure DBDisconnect(db: TFIBDatabase); virtual;
    procedure TRStart(tr: TFIBTransaction); virtual;
    procedure TRCommit(tr: TFIBTransaction); virtual;
    procedure TRCommitRetaining(tr: TFIBTransaction); virtual;
    procedure TRRollback(tr: TFIBTransaction); virtual;
    procedure TRRollbackRetaining(tr: TFIBTransaction); virtual;

   {$IFDEF  INC_SERVICE_SUPPORT}
    procedure ServiceAttach(Service: TpFIBCustomService); virtual;
    procedure ServiceDetach(Service: TpFIBCustomService); virtual;
    procedure ServiceQuery(Service: TpFIBCustomService); virtual;
    procedure ServiceStart(Service: TpFIBCustomService); virtual;
   {$ENDIF}

  end;

function MonitorHook: TFIBSQLMonitorHook;
procedure EnableMonitoring;
procedure DisableMonitoring;
function MonitoringEnabled: Boolean;

procedure Register;

implementation

uses
  FIB, StdFuncs;

procedure Register;
begin
//  RegisterComponents(FIBPalette, [TFIBSQLMonitor]);
end;



var
  bMonitoringEnabled: Boolean;

procedure EnableMonitoring;
begin
  bMonitoringEnabled := True;
end;

procedure DisableMonitoring;
begin
  bMonitoringEnabled := False;
end;

function MonitoringEnabled: Boolean;
begin
  result := bMonitoringEnabled;
end;

(*
 * TCustomFIBSQLMonitor
 *)

procedure FIBSQLM_Thread(Arg: TCustomFIBSQLMonitor); stdcall;
var
  st: String;
  len: Integer;
  FBuffer: PChar;
begin
  while (Arg.FHWnd <> 0) do begin
    st := MonitorHook.ReadSQLData;
    if Arg.FHWnd <> 0 then begin
      len := Length(st);
      GetMem(FBuffer, len + SizeOf(Integer));
      Move(len, FBuffer[0], SizeOf(Integer));
      Move(st[1], FBuffer[SizeOf(Integer)], len);
      PostMessage(
        Arg.FHWnd,
        WM_FIBSQL_SQL_EVENT,
        WPARAM(Arg),
        LPARAM(FBuffer));
    end;
  end;
  ExitThread(0)
end;

{function InitWndProc(HWindow: HWnd; Message, WParam: Longint;
  LParam: Longint): Longint; stdcall;}
function FIBSQLM_WindowProc(_hWnd: HWND; Msg: UINT; _wParam: WPARAM;
  _lParam: LPARAM): LRESULT; stdcall;
var
  MsgRec: TMessage;
begin
  MsgRec.Msg := Msg;
  MsgRec.WParam := _wParam;
  MsgRec.LParam := _lParam;
  case Msg of
    WM_CREATE:
      result := 0;
    else begin
      if ((Msg >= WM_MIN_FIBSQL_MONITOR) and
          (Msg <= WM_MAX_FIBSQL_MONITOR)) then begin
        try
          TCustomFIBSQLMonitor(_wParam).MonitorHandler(MsgRec);
        except
          ;
        end;
        result := MsgRec.Result;
      end else
        result := DefWindowProc(_hWnd, Msg, _wParam, _lParam);
    end;
  end;
end;

var
  MonitorClass: TWndClass = (
    style: 0;
    lpfnWndProc: @FIBSQLM_WindowProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: nil;
    lpszClassName: 'TCustomFIBSQLMonitor'
  );

constructor TCustomFIBSQLMonitor.Create(AOwner: TComponent);
var
  TempClass: TWndClass;
  ThreadID: DWORD;
begin
  inherited;
  MonitorClass.hInstance := HInstance;
  if not GetClassInfo(HInstance, MonitorClass.lpszClassName,
           TempClass) then begin
    FAtom := Windows.RegisterClass(MonitorClass);
    if FAtom = 0 then
      FIBError(feWindowsAPIError, [GetLastError, GetLastError]);
  end;
  FHWnd := CreateWindow(PChar(FAtom), '', 0, 0, 0, 0, 0,
             0, 0, HInstance, nil);
  if FHWnd = 0 then
    FIBError(feWindowsAPIError, [GetLastError, GetLastError]);
  MonitorHook.RegisterMonitor;
  FThread := CreateThread(nil, 0, @FIBSQLM_Thread, Pointer(Self), 0, ThreadID);
  if FThread = 0 then
    FIBError(feWindowsAPIError, [GetLastError, GetLastError]);
end;

destructor TCustomFIBSQLMonitor.Destroy;
begin
  MonitorHook.UnregisterMonitor;
  DestroyWindow(FHWnd);
  FHWnd := 0;
  if WaitForSingleObject(FThread, 1000) = WAIT_TIMEOUT then
    CloseHandle(FThread);
  Windows.UnregisterClass(PChar(FAtom), HInstance);
  FAtom := 0;
  inherited;
end;

procedure TCustomFIBSQLMonitor.MonitorHandler(var Msg: TMessage);
var
  st: String;
begin
  if (Msg.Msg = WM_FIBSQL_SQL_EVENT) then begin
    if (Assigned(FOnSQLEvent)) then begin
      SetString(st, PChar(Msg.LParam) + SizeOf(Integer),
                    PInteger(Msg.LParam)^);
      FreeMem(PChar(Msg.LParam));
      FOnSQLEvent(st);
    end;
  end else
    Msg.Result := DefWindowProc(FHWnd, Msg.Msg, Msg.WParam, Msg.LParam);
end;

(*
 * TFIBSQLMonitorHook
 *)

const
  MonitorHookNames: array[0..5] of String = (
    'FIB.SQL.MONITOR.Mutex',
    'FIB.SQL.MONITOR.SharedMem',
    'FIB.SQL.MONITOR.WriteEvent',
    'FIB.SQL.MONITOR.WriteFinishedEvent',
    'FIB.SQL.MONITOR.ReadEvent',
    'FIB.SQL.MONITOR.ReadFinishedEvent'
  );
  cMonitorHookSize = 1024;
  cMaxBufferSize = cMonitorHookSize - (4 * SizeOf(Integer));
  cDefaultTimeout = 2000; // 2 seconds

constructor TFIBSQLMonitorHook.Create;
var
  MapError: Integer;

  function CreateLocalEvent(Idx: Integer; InitialState: Boolean): THandle;
  begin
    result := CreateEvent(nil, True, InitialState, PChar(MonitorHookNames[Idx]));
    if result = 0 then
      FIBError(feCannotCreateSharedResource, [GetLastError]);
  end;

begin
  (*
   * Create the MMF with the initial size, and
   * create all events with an initial state of non-signalled.
   *)
  FWriteLock := CreateMutex(nil, False, PChar(MonitorHookNames[0]));
  if (FWriteLock = 0) then
    FIBError(feCannotCreateSharedResource, [GetLastError]);
  Lock; // Serialize the creation of memory mapped files.
  try
    FWriteEvent := CreateLocalEvent(2, False);
    FWriteFinishedEvent := CreateLocalEvent(3, True);
    FReadEvent := CreateLocalEvent(4, False);
    FReadFinishedEvent := CreateLocalEvent(5, False);
    (*
     * Set up the MMF
     *)
    FSharedBuffer := CreateFileMapping(
                       $FFFFFFFF, nil, PAGE_READWRITE, 0, cMonitorHookSize,
                       PChar(MonitorHookNames[1]));
    if (FSharedBuffer = 0) then
      FIBError(feCannotCreateSharedResource, [GetLastError]);
    MapError := GetLastError;
    FBuffer := MapViewOfFile(FSharedBuffer, FILE_MAP_ALL_ACCESS, 0, 0, 0);
    FMonitorCount := PInteger(FBuffer + cMonitorHookSize - SizeOf(Integer));
    FReaderCount := PInteger(PChar(FMonitorCount) - SizeOf(Integer));
    FBufferSize := PInteger(PChar(FReaderCount) - SizeOf(Integer));
    if MapError <> ERROR_ALREADY_EXISTS then begin
      (*
       * MMF initialization
       *)
      FMonitorCount^ := 0;
      FReaderCount^ := 0;
      FBufferSize^ := 0;
    end;
  finally
    Unlock;
  end;
end;

destructor TFIBSQLMonitorHook.Destroy;
begin
  Lock;
  try
    UnmapViewOfFile(FBuffer);
    CloseHandle(FSharedBuffer);
    CloseHandle(FWriteEvent);
    CloseHandle(FWriteFinishedEvent);
    CloseHandle(FReadEvent);
    CloseHandle(FReadFinishedEvent);
  finally
    Unlock;
    CloseHandle(FWriteLock);
  end;
  inherited;
end;

procedure TFIBSQLMonitorHook.Lock;
begin
  WaitForSingleObject(FWriteLock, INFINITE);
end;

procedure TFIBSQLMonitorHook.Unlock;
begin
  ReleaseMutex(FWriteLock);
end;

procedure TFIBSQLMonitorHook.BeginWrite;
begin
  Lock;
end;

procedure TFIBSQLMonitorHook.EndWrite;
begin
  (*
   * 1. Wait to end the write until all registered readers have
   *    started to wait for a write event
   * 2. Block all of those waiting for the write to finish.
   * 3. Block all of those waiting for all readers to finish.
   * 4. Unblock all readers waiting for a write event.
   * 5. Wait until all readers have finished reading.
   * 6. Now, block all those waiting for a write event.
   * 7. Unblock all readers waiting for a write to be finished.
   * 8. Unlock the mutex.
   *)
  while WaitForSingleObject(FReadEvent, cDefaultTimeout) = WAIT_TIMEOUT do begin
    if FMonitorCount^ > 0 then
      Dec(FMonitorCount^);
    if (FReaderCount^ = FMonitorCount^ - 1) or (FMonitorCount^ = 0) then
      SetEvent(FReadEvent);
  end;
  ResetEvent(FWriteFinishedEvent);
  ResetEvent(FReadFinishedEvent);
  SetEvent(FWriteEvent); // Let all readers pass through.
  while WaitForSingleObject(FReadFinishedEvent,
          cDefaultTimeout) = WAIT_TIMEOUT do begin
    if (FReaderCount^ = 0) or (InterlockedDecrement(FReaderCount^) = 0) then
      SetEvent(FReadFinishedEvent);
  end;
  ResetEvent(FWriteEvent);
  SetEvent(FWriteFinishedEvent);
  Unlock;
end;

procedure TFIBSQLMonitorHook.BeginRead;
begin
  (*
   * 1. Wait for the "previous" write event to complete.
   * 2. Increment the number of readers.
   * 3. if the reader count is the number of interested readers, then
   *    inform the system that all readers are ready.
   * 4. Finally, wait for the FWriteEvent to signal.
   *)
  WaitForSingleObject(FWriteFinishedEvent, INFINITE);
  InterlockedIncrement(FReaderCount^);
  if FReaderCount^ = FMonitorCount^ then
    SetEvent(FReadEvent);
  WaitForSingleObject(FWriteEvent, INFINITE);
end;

procedure TFIBSQLMonitorHook.EndRead;
begin
  if InterlockedDecrement(FReaderCount^) = 0 then
    SetEvent(FReadFinishedEvent);
end;

procedure TFIBSQLMonitorHook.RegisterMonitor;
begin
  Lock;
  try
    Inc(FMonitorCount^);
  finally
    Unlock;
  end;
end;

procedure TFIBSQLMonitorHook.UnregisterMonitor;
begin
  Lock;
  try
    Dec(FMonitorCount^);
  finally
    Unlock;
  end;
end;

function TFIBSQLMonitorHook.MonitorCount: Integer;
begin
  Lock;
  try
    result := FMonitorCount^;
  finally
    Unlock;
  end;
end;

function TFIBSQLMonitorHook.ReadSQLData: String;
begin
  BeginRead;
  try
    SetString(result, FBuffer, FBufferSize^);
  finally
    EndRead;
  end;
end;

procedure TFIBSQLMonitorHook.SQLPrepare(qry: TFIBQuery);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    if qry.Owner is TFIBDataSet then
      st := TFIBDataSet(qry.Owner).Name
    else
      st := qry.Name;
    st := st + ': [Prepare] ' + qry.SQL.Text + CRLF;
    try
     st := st + '  Plan: ' + qry.Plan;
    except
     st := st + '<Nothing plan>'
    end;
    WriteSQLData(st);
  end;
end;

type
 THackFQuery=class(TFIBQuery);

procedure TFIBSQLMonitorHook.SQLExecute(qry: TFIBQuery);
var
  st: String;
  i: Integer;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    if qry.Owner is TFIBDataSet then
      st := TFIBDataSet(qry.Owner).Name
    else
      st := qry.Name;
    st := st + ': [Execute] ' + THackFQuery(qry).FProcessedSQL.Text;
    if qry.Params.Count > 0 then begin
      for i := 0 to qry.Params.Count - 1 do begin
        st := st + CRLF + '  ' + qry.Params[i].Name + ' = ';
        try
          if qry.Params[i].IsNull then
            st := st + '<NULL>';
          st := st + qry.Params[i].AsString;
        except
          st := st + '<Can''t print value>';
        end;
      end;
    end;
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.SQLFetch(qry: TFIBQuery);
var
  st: String;
  i:integer;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    if qry.Owner is TFIBDataSet then
      st := TFIBDataSet(qry.Owner).Name
    else
      st := qry.Name;
    st := st + ': [Fetch] ' + qry.SQL.Text + CRLF;
    for i:=0 to Pred(qry.Current.Count) do
    begin
     st:=st+qry.Fields[i].Name+' = ';
     if qry.Fields[i].IsNull then
       st := st + 'NULL'
     else
       st := st + qry.Fields[i].asString;

     st:=st+CRLF;
    end;
    st:=CRLF+st;
    if (qry.EOF) then
      st := st + CRLF + '  End of file reached';
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.DBConnect(db: TFIBDatabase);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := db.Name + ': [Connect]';
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.DBDisconnect(db: TFIBDatabase);
var
  st: String;
begin
  if (Self = nil) then exit;
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := db.Name + ': [Disconnect]';
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.TRStart(tr: TFIBTransaction);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := tr.Name + ': [Start transaction]';
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.TRCommit(tr: TFIBTransaction);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := tr.Name + ': [Commit (Hard commit)]';
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.TRCommitRetaining(tr: TFIBTransaction);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := tr.Name + ': [Commit retaining (Soft commit)]';
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.TRRollbackRetaining(tr: TFIBTransaction);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := tr.Name + ': [Rollback retaining (Soft Rollback)]'; 
    WriteSQLData(st);
  end;
end;


procedure TFIBSQLMonitorHook.TRRollback(tr: TFIBTransaction);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0)then begin
    st := tr.Name + ': [Rollback]';
    WriteSQLData(st);
  end;
end;

{$IFDEF  INC_SERVICE_SUPPORT}

procedure TFIBSQLMonitorHook.ServiceAttach(Service: TpFIBCustomService);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0) then begin
    st := service.Name + ': [Attach]'; {do not localize}
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.ServiceDetach(Service: TpFIBCustomService);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0) then begin
    st := service.Name + ': [Detach]'; {do not localize}
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.ServiceQuery(Service: TpFIBCustomService);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0) then begin
    st := service.Name + ': [Query]'; {do not localize}
    WriteSQLData(st);
  end;
end;

procedure TFIBSQLMonitorHook.ServiceStart(Service: TpFIBCustomService);
var
  st: String;
begin
  if bMonitoringEnabled and (MonitorCount > 0) then begin
    st := service.Name + ': [Start]'; {do not localize}
    WriteSQLData(st);
  end;
end;
{$ENDIF}

procedure TFIBSQLMonitorHook.WriteSQLData(Text: String);
var
  i, len: Integer;

begin
  {$IFNDEF FOR_CONSOLE}
  Text := '[Application: ' + Application.Title + ']' + CRLF + CRLF + Text;
  {$ENDIF}
  Lock;
  try
    i := 1;
    len := Length(Text);
    while (len > 0) do begin
      BeginWrite;
      try
        FBufferSize^ := Min(len, cMaxBufferSize);
        Move(Text[i], FBuffer[0], FBufferSize^);
        Inc(i, cMaxBufferSize);
        Dec(len, cMaxBufferSize);
      finally
        EndWrite;
      end;
    end;
  finally
    Unlock;
  end;
end;

var
  _MonitorHook: TFIBSQLMonitorHook;
  bDone: Boolean;

function MonitorHook: TFIBSQLMonitorHook;
begin
  if (_MonitorHook = nil) and (not bDone) then
    _MonitorHook := TFIBSQLMonitorHook.Create;
  result := _MonitorHook;
end;

initialization

  bMonitoringEnabled := True;
  _MonitorHook := nil;
  bDone := False;

finalization

  bDone := True;
  _MonitorHook.Free;
  _MonitorHook := nil;

end.
