(*
 * FIBQuery.pas - (Implementation of TFIBQuery component)
 *  copyright (c) 1998 by
 *    Gregory Deatz
 *    gdeatz@hlmdd.com
 *
 * Please see the file FIBLicense.txt for full license information.
 *)
// Changed  07-09.1999 by Serge Buzadzhy Serge_Buzadzhy@mail.ru

unit FIBQuery;

(*
 * Compiler defines
 *)
{$A+}                           (* Aligned records: On *)
{$B-}                           (* Short circuit boolean expressions: Off *)
{$G+}                           (* Imported data: On *)
{$H+}                           (* Huge Strings: On *)
{$J-}                           (* Modification of Typed Constants: Off *)
{$M+}                           (* Generate run-time type information: On *)
{$Q-}                           (* Overflow checks: Off *)
{$R-}                           (* Range checks: Off *)
{$T+}                           (* Typed address: On *)
{$U+}                           (* Pentim-safe FDIVs: On *)
{$X+}                           (* Extended syntax: On *)
{$Z1}                           (* Minimum Enumeration Size: 1 Byte *)

interface

{$I FIBPlus.INC}
uses
  Windows, SysUtils, Classes, ibase,IB_Intf, ib_externals,
  DB, FIB, FIBDatabase, StdConsts, StdFuncs,IB_ErrorCodes,
  {$IFDEF SUPPORT_ARRAY_FIELD}
   pFIBArray
  {$ENDIF}
  ;

type
  TFIBQuery = class;
  TFIBXSQLDA = class;

  (* TFIBXSQLVAR *)
  TFIBXSQLVAR = class(TObject)
  protected
    FIndex: Integer;
    FModified: Boolean;
    FName: String;
    FQuery: TFIBQuery;
    FVariantFalse,
    FVariantTrue: Variant;
    FXSQLVAR: PXSQLVAR;       // Point to the PXSQLVAR in the owner object
    FParent: TFIBXSQLDA;
// Added variables
    FIsMacro:boolean;
    FQuoted :boolean;

 {$IFDEF SUPPORT_ARRAY_FIELD}
    vFIBArray:TpFIBArray;
 {$ENDIF}
    function AdjustScale(Value: Integer; Scale: Integer): Double;
    function AdjustScaleToCurrency(Value: Comp; Scale: Integer): Currency;

    function GetAsCurrency: Currency;
    function GetAsComp: Comp;    
{$IFDEF VER130}
    function GetAsInt64: Int64;
{$ENDIF}
    function GetAsDateTime: TDateTime;
    function GetAsDouble: Double;
    function GetAsFloat: Float;
    function GetAsLong: Long;
    function GetAsPointer: Pointer;
    function GetAsQuad: TISC_QUAD;
    function GetAsShort: Short;
    function GetAsString: String;
    function GetAsVariant: Variant;
    function GetAsXSQLVAR: PXSQLVAR;
    function GetIsNull: Boolean;
    function GetIsNullable: Boolean;
    function GetSize: Integer;
    function GetSQLType: Integer;

//  Array Support
 {$IFDEF SUPPORT_ARRAY_FIELD}
    procedure CheckArrayType;
    function GetDimensionCount:Integer;
    function GetDimension(Index: Integer): TISC_ARRAY_BOUND;
    function GetSliceSize:integer;
    function GetElementType:TFieldType;
    function GetArraySize:integer;
  {$ENDIF}
//
    procedure SetAsCurrency(Value: Currency);
    procedure SetAsComp(Value: comp); //patchInt64A
{$IFDEF VER130}
    procedure SetAsInt64(Value: Int64);
{$ENDIF}    
    procedure SetAsDateTime(Value: TDateTime);
    procedure SetAsTime(Value: TDateTime);
    procedure SetAsDate(Value: TDateTime);
    procedure SetAsDouble(Value: Double);
    procedure SetAsFloat(Value: Float);
    procedure SetAsLong(Value: Long);
    procedure SetAsPointer(Value: Pointer);
    procedure SetAsQuad(Value: TISC_QUAD);
    procedure SetAsShort(Value: Short);
    procedure SetAsString(Value: String);
    procedure SetAsVariant(Value: Variant);
    procedure SetAsXSQLVAR(Value: PXSQLVAR);
    procedure SetIsNull(Value: Boolean);
    procedure SetIsNullable(Value: Boolean);


  public
    constructor Create(AParent: TFIBXSQLDA);
    destructor Destroy; override;
    procedure Assign(Source: TFIBXSQLVAR);
    procedure LoadFromFile(const FileName: String);
    procedure LoadFromStream(Stream: TStream);
    procedure SaveToFile(const FileName: String);
    procedure SaveToStream(Stream: TStream);
// Array Support
{$IFDEF SUPPORT_ARRAY_FIELD}
    function  GetArrayElement(Indexes: array of Integer):Variant;
    function  GetArrayValues:Variant;
    procedure SetArrayValue(Value:Variant);
{$ENDIF}
//
    property AsCurrency: Currency read GetAsCurrency write SetAsCurrency;
    property AsComp: comp read GetAsComp write SetAsComp;     
{$IFDEF VER130}
    property AsInt64: Int64 read GetAsInt64 write SetAsInt64;
{$ENDIF}
    property AsDateTime: TDateTime read GetAsDateTime write SetAsDateTime;
    property AsDate: TDateTime read GetAsDateTime write SetAsDate;
    property AsTime: TDateTime read GetAsDateTime write SetAsTime;        
    property AsDouble: Double read GetAsDouble write SetAsDouble;
    property AsFloat: Float read GetAsFloat write SetAsFloat;
    property AsInteger: Integer read GetAsLong write SetAsLong;
    property AsLong: Long read GetAsLong write SetAsLong;
    property AsPointer: Pointer read GetAsPointer write SetAsPointer;
    property AsQuad: TISC_QUAD read GetAsQuad write SetAsQuad;
    property AsShort: Short read GetAsShort write SetAsShort;
    property AsString: String read GetAsString write SetAsString;
    property AsVariant: Variant read GetAsVariant write SetAsVariant;
    property AsXSQLVAR: PXSQLVAR read GetAsXSQLVAR write SetAsXSQLVAR;
    property Data: PXSQLVAR read FXSQLVAR write FXSQLVAR;
    property IsNull: Boolean read GetIsNull write SetIsNull;
    property IsNullable: Boolean read GetIsNullable write SetIsNullable;
    property Index: Integer read FIndex;
    property Modified: Boolean read FModified write FModified;
    property Name: String read FName;
    property Size: Integer read GetSize;
    property SQLType: Integer read GetSQLType;
    property Value: Variant read GetAsVariant write SetAsVariant;
    property VariantFalse: Variant read FVariantFalse write FVariantFalse;
    property VariantTrue: Variant read FVariantFalse write FVariantTrue;
// Added properties
    property IsMacro:boolean  read FIsMacro write FIsMacro;
    property Quoted :boolean  read FQuoted  write FQuoted;
//  Array Support
 {$IFDEF SUPPORT_ARRAY_FIELD}
    property FIBArray:TpFIBArray read vFIBArray;
    property DimensionCount:integer read GetDimensionCount;
    property Dimension[Index: Integer]: TISC_ARRAY_BOUND read GetDimension;
    property ElementType:TFieldType read GetElementType;
    property ArraySize:Integer read GetArraySize;
 {$ENDIF}        

  end;
  
  TFIBXSQLVARArray = array[0..0] of TFIBXSQLVAR;
  PFIBXSQLVARArray = ^TFIBXSQLVARArray;

  (* TFIBXSQLVAR *)
  TFIBXSQLDA = class(TObject)
  protected
    FCount: Integer;
    FNames: TStrings;
    FQuery: TFIBQuery;
    FSize: Integer;
    FXSQLDA: PXSQLDA;
    FXSQLVARs: PFIBXSQLVARArray; // array of FIBXQLVARs
    function GetModified: Boolean;
    function GetNames: String;
    function GetRecordSize: Integer;
    function GetXSQLDA: PXSQLDA;
    function GetXSQLVAR(Idx: Integer): TFIBXSQLVAR;
    function GetXSQLVARByName(Idx: String): TFIBXSQLVAR;
    procedure Initialize;
    procedure SetCount(Value: Integer);
  public
    constructor Create;
    destructor Destroy; override;
    procedure AddName(FieldName: String; Idx: Integer);
    property AsXSQLDA: PXSQLDA read GetXSQLDA;
    property ByName[Idx: String]: TFIBXSQLVAR read GetXSQLVARByName;
    property Count: Integer read FCount write SetCount;
    property Modified: Boolean read GetModified;
    property Names: String read GetNames;
    property RecordSize: Integer read GetRecordSize;
    property Vars[Idx: Integer]: TFIBXSQLVAR read GetXSQLVAR; default;
  end;

  (* TFIBBatch - basis for batch input and batch output objects. *)
  TFIBBatch = class(TObject)
  protected
    FFilename: String;
    FColumns: TFIBXSQLDA;
    FParams: TFIBXSQLDA;
  public
    procedure ReadyStream; virtual; abstract;
    property Columns: TFIBXSQLDA read FColumns;
    property Filename: String read FFilename write FFilename;
    property Params: TFIBXSQLDA read FParams;
  end;

  (* TFIBBatchInputStream - see FIBMiscellaneous for good examples. *)
  TFIBBatchInputStream = class(TFIBBatch)
  public
    function ReadParameters: Boolean; virtual; abstract;
  end;
  TFIBBatchInputStreamClass = class of TFIBBatchInputStream;

  (* TFIBBatchOutputStream - see FIBMiscellaneous for good examples. *)
  TFIBBatchOutputStream = class(TFIBBatch)
  public
    function WriteColumns: Boolean; virtual; abstract;
  end;
  TFIBBatchOutputStreamClass = class of TFIBBatchOutputStream;

  (* TFIBQuery *)
  TFIBSQLTypes = (SQLUnknown, SQLSelect, SQLInsert,
                  SQLUpdate, SQLDelete, SQLDDL,
                  SQLGetSegment, SQLPutSegment,
                  SQLExecProcedure, SQLStartTransaction,
                  SQLCommit, SQLRollback,
                  SQLSelectForUpdate, SQLSetGenerator);

  TOnSQLFetch  =procedure (RecordNumber:integer;   var StopFetching:boolean
  ) of object;
                  
  TFIBQuery = class(TComponent)
  protected
    FBase: TFIBBase;
    FBOF,                          // At BOF?
    FEOF,                          // At EOF
    FGoToFirstRecordOnExecute,     // Automatically position record on first record after executing
    FOpen,                         // Is a cursor open?
    FPrepared: Boolean;            // Has the query been prepared?
    FRecordCount: Integer;         // How many records have been read so far?
    FHandle: TISC_STMT_HANDLE;     // Once prepared, this accesses the SQL Query
    FCursor: String;               // Cursor name...
    FOnSQLChanging: TNotifyEvent;  // Call this when the SQL is changing.
    FSQL: TStrings;                // SQL Query (by user)
    FParamCheck: Boolean;          // Check for parameters? (just like TQuery)
    FProcessedSQL: TStrings;       // SQL Query (pre-processed for param labels)
    FSQLParams,                    // Any parameters to the query.
    FSQLRecord: TFIBXSQLDA;        // The current record
    FSQLType: TFIBSQLTypes;        // Select, update, delete, insert, create, alter, etc...
    //Added variables
    vPrepUserSQL:string;
    FUserSQLParams:TFIBXSQLDA;
    FProcExecuted:boolean;
    FOnSQLFetch:TOnSQLFetch;
    procedure DatabaseDisconnecting(Sender: TObject);
    function GetDatabase: TFIBDatabase;
    function GetDBHandle: PISC_DB_HANDLE;
    function GetEOF: Boolean;
    function GetFieldByName(FieldName: String): TFIBXSQLVAR;
    function GetFields(const Idx: Integer): TFIBXSQLVAR;
    function GetFieldIndex(FieldName: String): Integer;
    function GetPlan: String;
    function GetRecordCount: Integer;
    function GetRowsAffected: Integer;
    function GetSQLParams: TFIBXSQLDA;
    function GetTransaction: TFIBTransaction;
    function GetTRHandle: PISC_TR_HANDLE;
    procedure PreprocessSQL(const SQLText:string;IsUserSQL:boolean);
    procedure SetDatabase(Value: TFIBDatabase);
    procedure SetSQL(Value: TStrings);
    procedure SetTransaction(Value: TFIBTransaction);
    procedure SQLChanging(Sender: TObject);
    procedure SQLChange(Sender: TObject);
    procedure TransactionEnding(Sender: TObject);
 // Added procedures

    procedure SynchronizeParams;
    function  SetMacro(const SQLText:string;UseDefaults:boolean):string;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure BatchInput(InputObject: TFIBBatchInputStream);
    procedure BatchOutput(OutputObject: TFIBBatchOutputStream);
    function Call(ErrCode: ISC_STATUS; RaiseError: Boolean): ISC_STATUS;
    procedure CheckClosed;           // raise error if query is not closed.
    procedure CheckOpen;             // raise error if query is not open.
    procedure CheckValidStatement;   // raise error if statement is invalid.
    procedure Close;               // close the query.
    function Current: TFIBXSQLDA;
    procedure ExecQuery;                // ExecQuery the query.
    procedure FreeHandle;
    function Next: TFIBXSQLDA;
    procedure Prepare;                  // Prepare the query.
// Routine from Serge Buzadzhy
{$IFDEF SUPPORT_ARRAY_FIELD}
    procedure PrepareArrayFields;
{$ENDIF}
    procedure SetParamValues(ParamValues: array of Variant);
    procedure ExecWP(ParamValues: array of Variant);
                                           // Exec Query with ParamValues


//
    function  OK_SQLText:string;

    
    function  IsProc :boolean;
    function ParamByName(const ParamName:string): TFIBXSQLVAR;
    property BOF: Boolean read FBOF;
    property DBHandle: PISC_DB_HANDLE read GetDBHandle;
    property EOF: Boolean read GetEOF;
    property FieldByName[FieldName: String]: TFIBXSQLVAR read GetFieldByName;default;
    property Fields[const Idx: Integer]: TFIBXSQLVAR read GetFields;
    property FieldIndex[FieldName: String]: Integer read GetFieldIndex;
    property Open: Boolean read FOpen;
    property Params: TFIBXSQLDA read GetSQLParams;
    property Plan: String read GetPlan;
    property Prepared: Boolean read FPrepared;
    property RecordCount: Integer read GetRecordCount;
    property RowsAffected: Integer read GetRowsAffected;
    property SQLType: TFIBSQLTypes read FSQLType;
    property TRHandle: PISC_TR_HANDLE read GetTRHandle;
    property ProcExecuted:boolean read FProcExecuted write FProcExecuted;
    property OnSQLFetch:TOnSQLFetch read FOnSQLFetch write FOnSQLFetch; // for internal use
  published
    property Database: TFIBDatabase read GetDatabase write SetDatabase;
    property GoToFirstRecordOnExecute: Boolean read FGoToFirstRecordOnExecute
                                               write FGoToFirstRecordOnExecute
                                               default True;
    property ParamCheck: Boolean read FParamCheck write FParamCheck;
    property SQL: TStrings read FSQL write SetSQL;
    property Transaction: TFIBTransaction read GetTransaction write SetTransaction;
    (* Events *)
    property OnSQLChanging: TNotifyEvent read FOnSQLChanging write FOnSQLChanging;
  end;


const ExecProcPrefix='EXECUTE ';


implementation

uses
  FIBMiscellaneous, FIBSQLMonitor,StrUtil,
 //Added uses
  FIBDataSet;


(* TFIBXSQLVAR *)
constructor TFIBXSQLVAR.Create(AParent: TFIBXSQLDA);
begin
  FParent := AParent;
  FVariantFalse := 0;
  FVariantTrue  := 1;

  FIsMacro     :=false;
  FQuoted      :=false;
 {$IFDEF SUPPORT_ARRAY_FIELD}
  vFIBArray   :=nil;
 {$ENDIF}
end;

destructor TFIBXSQLVAR.Destroy; //override;
begin
 {$IFDEF SUPPORT_ARRAY_FIELD}
 if Assigned(vFIBArray) then vFIBArray.Free;
 {$ENDIF}
 inherited Destroy;
end;
function TFIBXSQLVAR.AdjustScale(Value: Integer; Scale: Integer): Double;
var
  Scaling, i: Integer;
  Val: Double;
begin
  Scaling := 1; Val := Value;
  if Scale > 0 then begin
    for i := 1 to Scale do Scaling := Scaling * 10;
    Result := Val * Scaling;
  end else if Scale < 0 then begin
    for i := -1 downto Scale do Scaling := Scaling * 10;
    Result := Val / Scaling;
  end else
    Result := Val;
end;

function TFIBXSQLVAR.AdjustScaleToCurrency(Value: Comp; Scale: Integer): Currency;
var
  Scaling: Extended;
  i: Integer;
  FractionText, PadText, CurrText: string;
begin
  try
  Scaling := 1;
  if Scale > 0 then
  begin
    for i := 1 to Scale do
      Scaling := Scaling * 10;
    result := Value * Scaling;
  end
  else
    if Scale < 0 then
    begin
      for i := -1 downto Scale do
        Scaling := Scaling * 10;
      FractionText := CompToStr(CompMod(Value, Scaling));
      for i := Length(FractionText) to -Scale -1 do
        PadText := '0' + PadText;
      if Value < 0 then
        CurrText := '-' + CompToStr(CompDiv(Value, Scaling)) +
DecimalSeparator + PadText + FractionText
      else
        CurrText := CompToStr(CompDiv(Value, Scaling)) + DecimalSeparator +
PadText + FractionText;
        result := StrToCurr(CurrText);
    end
    else
      result := Value;
  except
    Result := 0;
  end;
end; 

procedure TFIBXSQLVAR.Assign(Source: TFIBXSQLVAR);
var
  szBuff: PChar;
  s_bhandle, d_bhandle: TISC_BLOB_HANDLE;
  bSourceBlob, bDestBlob: Boolean;
  iSegs, iMaxSeg, iSize: Long;
  iBlobType: Short;
begin
  szBuff := nil;
  bSourceBlob := True;
  bDestBlob := True;
  s_bhandle := nil;
  d_bhandle := nil;
  try
    if (Source.IsNull) then begin
      IsNull := True;
      exit;
    end else if (FXSQLVAR^.sqltype and (not 1) = SQL_ARRAY) or
       (Source.FXSQLVAR^.sqltype and (not 1) = SQL_ARRAY) then
      exit; // arrays not supported.
    if (FXSQLVAR^.sqltype and (not 1) <> SQL_BLOB) and
       (Source.FXSQLVAR^.sqltype and (not 1) <> SQL_BLOB) then begin
      AsXSQLVAR := Source.AsXSQLVAR;
      exit;
    end else if (Source.FXSQLVAR^.sqltype and (not 1) <> SQL_BLOB) then begin
      szBuff := nil;
      FIBAlloc(szBuff, 0, Source.FXSQLVAR^.sqllen);
      Move(Source.FXSQLVAR^.sqldata[0], szBuff[0], Source.FXSQLVAR^.sqllen);
      bSourceBlob := False;
      iSize := Source.FXSQLVAR^.sqllen;
    end else if (FXSQLVAR^.sqltype and (not 1) <> SQL_BLOB) then
      bDestBlob := False;

    if bSourceBlob then begin
      // read the blob
      Source.FQuery.Call(isc_open_blob2(StatusVector, Source.FQuery.DBHandle,
        Source.FQuery.TRHandle, @s_bhandle, PISC_QUAD(Source.FXSQLVAR.sqldata),
        0, nil), True);
      try
        FIBMiscellaneous.GetBlobInfo(@s_bhandle, iSegs, iMaxSeg, iSize,
          iBlobType);
        szBuff := nil;
        FIBAlloc(szBuff, 0, iSize);
        FIBMiscellaneous.ReadBlob(@s_bhandle, szBuff, iSize);
      finally
        Source.FQuery.Call(isc_close_blob(StatusVector, @s_bhandle), True);
      end;
    end;

    if bDestBlob then begin
      // write the blob
      FQuery.Call(isc_create_blob2(StatusVector, FQuery.DBHandle,
        FQuery.TRHandle, @d_bhandle, PISC_QUAD(FXSQLVAR.sqldata),
        0, nil), True);
      try
        FIBMiscellaneous.WriteBlob(@d_bhandle, szBuff, iSize);
      finally
        FQuery.Call(isc_close_blob(StatusVector, @d_bhandle), True);
      end;
    end else begin
      // just copy the buffer
      FXSQLVAR.sqltype := SQL_TEXT;
      FXSQLVAR.sqllen := iSize;
      FIBAlloc(FXSQLVAR.sqldata, iSize, iSize);
      Move(szBuff[0], FXSQLVAR^.sqldata[0], iSize);
    end;
  finally
    FIBAlloc(szBuff, 0, 0);
  end;
end;
{$IFDEF VER130}
function TFIBXSQLVAR.GetAsInt64: Int64;
begin
  Result := 0;
  if not IsNull then
    case FXSQLVAR^.sqltype and (not 1) of
      SQL_TEXT, SQL_VARYING: begin
        try
          Result := StrToInt64(AsString);
        except
          on E: Exception do FIBError(feInvalidDataConversion, [nil]);
        end;
      end;
      SQL_SHORT:
        Result := Trunc(AdjustScale(Int64(PShort(FXSQLVAR^.sqldata)^),
                                    FXSQLVAR^.sqlscale));
      SQL_LONG:
        Result := Trunc(AdjustScale(Int64(PLong(FXSQLVAR^.sqldata)^),
                                    FXSQLVAR^.sqlscale));
      SQL_INT64:
        Result := Trunc(AdjustScale(PInt64(FXSQLVAR^.sqldata)^,
                                    FXSQLVAR^.sqlscale));
      SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT:
        Result := Trunc(AsDouble);
      else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;
{$ENDIF}


function TFIBXSQLVAR.GetAsCurrency: Currency;
begin
  if (FQuery.Database.SQLDialect < 3)
   or (FXSQLVAR^.sqltype and (not 1)<>SQL_INT64)
  then
    result := GetAsDouble
  else
    result := AdjustScaleToCurrency(PComp(FXSQLVAR^.sqldata)^,
               FXSQLVAR^.sqlscale
              );

end;

function TFIBXSQLVAR.GetAsComp: Comp;
begin
//  InitFPU;
  Result := 0;
  if not IsNull then
    if (FXSQLVAR^.sqltype and (not 1))<>SQL_INT64 then
     Result:= AsDouble
    else
     Result := PComp(FXSQLVAR^.sqldata)^*Degree10(FXSQLVAR^.sqlscale);
end;


function TFIBXSQLVAR.GetAsDateTime: TDateTime;
var
  tm_date: TCTimeStructure;
begin
  Result := 0;
  if not IsNull then
    case FXSQLVAR^.sqltype and (not 1) of
      SQL_TEXT, SQL_VARYING: begin
        try
          Result := StrToDate(AsString);
        except
          on E: EConvertError do FIBError(feInvalidDataConversion, [nil]);
        end;
      end;
      SQL_TYPE_TIME: begin
        isc_decode_sql_time(PISC_TIME(FXSQLVAR^.sqldata), @tm_date);
        try
          result := EncodeTime(Word(tm_date.tm_hour), Word(tm_date.tm_min),
                               Word(tm_date.tm_sec), 0)
        except
          on E: EConvertError do begin
            FIBError(feInvalidDataConversion, [nil]);
          end;
        end;
      end;
      SQL_TYPE_DATE: begin
        isc_decode_sql_date(PISC_DATE(FXSQLVAR^.sqldata), @tm_date);
        try
          result := EncodeDate(Word(tm_date.tm_year + 1900), Word(tm_date.tm_mon + 1),
                               Word(tm_date.tm_mday));
        except
          on E: EConvertError do begin
           FIBError(feInvalidDataConversion, [nil]);
          end;
        end;
      end;
      SQL_TIMESTAMP: begin
        isc_decode_date(PISC_QUAD(FXSQLVAR^.sqldata), @tm_date);
        try
          result := EncodeDate(Word(tm_date.tm_year + 1900), Word(tm_date.tm_mon + 1),
                              Word(tm_date.tm_mday));
          if result >= 0 then
            result := result + EncodeTime(Word(tm_date.tm_hour), Word(tm_date.tm_min),
                                          Word(tm_date.tm_sec), 0)
          else
            result := result - EncodeTime(Word(tm_date.tm_hour), Word(tm_date.tm_min),
                                          Word(tm_date.tm_sec), 0)
        except
          on E: EConvertError do begin
            FIBError(feInvalidDataConversion, [nil]);
          end;
        end;
      end;
     else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;



                                                                       
function TFIBXSQLVAR.GetAsDouble: Double;
begin
  Result := 0;
  if not IsNull then
    case FXSQLVAR^.sqltype and (not 1) of
      SQL_TEXT, SQL_VARYING: begin
        try
          Result := StrToFloat(AsString);
        except
          on E: Exception do FIBError(feInvalidDataConversion, [nil]);
        end;
      end;
      SQL_SHORT:
        Result := AdjustScale(Long(PShort(FXSQLVAR^.sqldata)^),
                              FXSQLVAR^.sqlscale);
      SQL_LONG:
        Result := AdjustScale(PLong(FXSQLVAR^.sqldata)^, FXSQLVAR^.sqlscale);
      SQL_FLOAT:
        Result := PFloat(FXSQLVAR^.sqldata)^;
      SQL_INT64:
        Result := PComp(FXSQLVAR^.sqldata)^*Degree10(FXSQLVAR^.sqlscale);
      SQL_DOUBLE, SQL_D_FLOAT:
        Result := PDouble(FXSQLVAR^.sqldata)^;
      else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;

function TFIBXSQLVAR.GetAsFloat: Float;
begin
  Result := 0;
  try
    Result := AsDouble;
  except
    on E: EOverflow do
      FIBError(feInvalidDataConversion, [nil]);
  end;
end;

function TFIBXSQLVAR.GetAsLong: Long;
begin
  Result := 0;
  if not IsNull then
    case FXSQLVAR^.sqltype and (not 1) of
      SQL_TEXT, SQL_VARYING: begin
        try
          Result := StrToInt(AsString);
        except
          on E: Exception do FIBError(feInvalidDataConversion, [nil]);
        end;
      end;
      SQL_SHORT:
        Result := Trunc(AdjustScale(Long(PShort(FXSQLVAR^.sqldata)^),
                                    FXSQLVAR^.sqlscale));
      SQL_LONG: begin
        Result := Trunc(AdjustScale(PLong(FXSQLVAR^.sqldata)^, FXSQLVAR^.sqlscale));
      end;
      SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT:
        Result := Trunc(AsDouble);
      else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;

function TFIBXSQLVAR.GetAsPointer: Pointer;
begin
  if not IsNull then
    Result := FXSQLVAR^.sqldata
  else
    Result := nil;
end;

function TFIBXSQLVAR.GetAsQuad: TISC_QUAD;
begin
  Result.gds_quad_high := 0;
  Result.gds_quad_low := 0;
  if not IsNull then
    case FXSQLVAR^.sqltype and (not 1) of
      SQL_BLOB, SQL_ARRAY, SQL_QUAD:
        Result := PISC_QUAD(FXSQLVAR^.sqldata)^;
      else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;

function TFIBXSQLVAR.GetAsShort: Short;
begin
  Result := 0;
  try
    Result := AsLong;
  except
    on E: Exception do FIBError(feInvalidDataConversion, [nil]);
  end;
end;


function TFIBXSQLVAR.GetAsString: String;
var
  sz: PChar;
  str_len: Integer;
  bs     :TFIBBlobStream;
begin
  Result := '';
  (* Check null, if so return a default string *)

  if not IsNull then
    case FXSQLVar^.sqltype and (not 1) of
      SQL_ARRAY:
        Result := '(Array)';
      SQL_BLOB:
      
        try
        //It is D.Kuzmenko idea. 
          Result:='(Blob)'; { return on error }
          bs := TFIBBlobStream.Create;
          try
            bs.Mode := bmRead;
            bs.Database := FQuery.Database;
            bs.Transaction := FQuery.Transaction;
            bs.BlobID := AsQuad;
            Result:=bs.AsString;
          finally
            bs.Free;
          end;
        except  
        end;


      SQL_TEXT, SQL_VARYING: begin
        sz := FXSQLVAR^.sqldata;
        if (FXSQLVar^.sqltype and (not 1) = SQL_TEXT) then
          str_len := FXSQLVar^.sqllen
        else begin
          str_len := isc_vax_integer(FXSQLVar^.sqldata, 2);
          Inc(sz, 2);
        end;
        SetString(Result, sz, str_len);
      end;
      SQL_TYPE_DATE, SQL_DATE:
        Result := DateToStr(AsDateTime);
      SQL_TYPE_TIME:
        Result := TimeToStr(AsTime);
      SQL_SHORT, SQL_LONG:
        if FXSQLVAR^.sqlscale <> 0 then
          Result := FloatToStr(AsDouble)
        else
          Result := IntToStr(AsLong);
      SQL_INT64:
        if FXSQLVAR^.sqlscale = 0 then
          Result := CompToStr(AsComp)
        else if FXSQLVAR^.sqlscale >= (-4) then
          Result := CurrToStr(AsCurrency)
        else
          Result := FloatToStr(AsDouble);          
      SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT:
        Result := FloatToStr(AsDouble);
      else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;

function TFIBXSQLVAR.GetAsVariant: Variant;
begin
  if IsNull then
    Result := NULL
  (* Check null, if so return a default string *)
  else case FXSQLVar^.sqltype and (not 1) of
      SQL_ARRAY:
        Result := '(Array)';
      SQL_BLOB:
        Result := '(Blob)';
      SQL_TEXT, SQL_VARYING:
        Result := AsString;
      SQL_TYPE_DATE, SQL_TYPE_TIME,SQL_DATE:
        Result := AsDateTime;
      SQL_SHORT, SQL_LONG:
        if FXSQLVAR^.sqlscale <> 0 then
          Result := AsDouble
        else
          Result := AsLong;
      SQL_INT64:
        if FXSQLVAR^.sqlscale = 0 then
          Result := AsCurrency
        else if FXSQLVAR^.sqlscale >= (-4) then
          Result := AsCurrency
        else
          Result := AsDouble;          
      SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT:
        Result := AsDouble;
      else
        FIBError(feInvalidDataConversion, [nil]);
    end;
end;

function TFIBXSQLVAR.GetAsXSQLVAR: PXSQLVAR;
begin
  Result := FXSQLVAR;
end;

function TFIBXSQLVAR.GetIsNull: Boolean;
begin
  Result := IsNullable and (FXSQLVAR^.sqlind^ = -1);
end;

function TFIBXSQLVAR.GetIsNullable: Boolean;
begin
  Result := (FXSQLVAR^.sqltype and 1 = 1);
end;

procedure TFIBXSQLVAR.LoadFromFile(const FileName: String);
var
  fs: TFileStream;
begin
  fs := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  try
    LoadFromStream(fs);
  finally
    fs.Free;
  end;
end;

procedure TFIBXSQLVAR.LoadFromStream(Stream: TStream);
var
  bs: TFIBBlobStream;
begin
  bs := TFIBBlobStream.Create;
  try
    bs.Mode := bmWrite;
    bs.Database := FQuery.Database;
    bs.Transaction := FQuery.Transaction;
    Stream.Seek(0, soFromBeginning);
    bs.LoadFromStream(Stream);
    bs.Finalize;
    AsQuad := bs.BlobID;
  finally
    bs.Free;
  end;
end;

procedure TFIBXSQLVAR.SaveToFile(const FileName: String);
var
  fs: TFileStream;
begin
  fs := TFileStream.Create(FileName, fmCreate or fmShareExclusive);
  try
    SaveToStream(fs);
  finally
    fs.Free;
  end;
end;

procedure TFIBXSQLVAR.SaveToStream(Stream: TStream);
var
  bs: TFIBBlobStream;
begin
  bs := TFIBBlobStream.Create;
  try
    bs.Mode := bmRead;
    bs.Database := FQuery.Database;
    bs.Transaction := FQuery.Transaction;
    bs.BlobID := AsQuad;
    bs.SaveToStream(Stream);
  finally
    bs.Free;
  end;
end;

function TFIBXSQLVAR.GetSize: Integer;
begin
  Result := FXSQLVAR^.sqllen;
end;

// Array Support

{$IFDEF SUPPORT_ARRAY_FIELD}
procedure TFIBXSQLVAR.CheckArrayType;
begin
 if not ((FXSQLVAR^.sqltype and (not 1) = SQL_ARRAY))
 or not Assigned(vFIBArray)
  then
   FIBError(feNotIsArrayField,[FXSQLVAR^.relName+'.'+FXSQLVAR^.SQlName]);
end;

function TFIBXSQLVAR.GetDimensionCount:Integer;
begin
 CheckArrayType;
 Result:=vFIBArray.DimensionCount
end;


function TFIBXSQLVAR.GetElementType:TFieldType;
begin
 CheckArrayType;
 Result:=vFIBArray.ArrayType
end;

function TFIBXSQLVAR.GetDimension(Index: Integer): TISC_ARRAY_BOUND;
begin
 CheckArrayType;
 Result:=vFIBArray.Dimension[Index]
end;

function TFIBXSQLVAR.GetSliceSize:integer;
begin
  CheckArrayType;
  Result:=vFIBArray.ArraySize
end;



procedure TFIBXSQLVAR.SetArrayValue(Value:Variant);
begin
 CheckArrayType;
 vFIBArray.SetArrayValue(Value, FXSQLVAR^.sqldata,
  FQuery.DBHandle,FQuery.TRHandle
 );
 ASQuad:=PISC_QUAD(FXSQLVAR^.sqldata)^
end;

function TFIBXSQLVAR.GetArrayValues:Variant;
begin
 CheckArrayType;
 Result:=false;
 if not Assigned(vFIBArray) then Exit;
 Result:= vFIBArray.GetArrayValues(FXSQLVAR^.sqldata,
             FQuery.DBHandle, FQuery.TRHandle
          );
end;

function TFIBXSQLVAR.GetArrayElement(Indexes: array of Integer):Variant;
begin
 CheckArrayType;
 Result:=GetArrayValues[Indexes[0]]
end;

function TFIBXSQLVAR.GetArraySize:integer;
begin
 CheckArrayType;
 Result:=vFIBArray.ArraySize
end;
{$ENDIF}
// End Array Support

function TFIBXSQLVAR.GetSQLType: Integer;
begin
  Result := FXSQLVAR^.sqltype and (not 1);
end;

{$IFDEF VER130}
procedure TFIBXSQLVAR.SetAsInt64(Value: Int64);
begin
  if IsNullable then
    IsNull := False;
  FXSQLVAR^.sqltype := SQL_INT64 or (FXSQLVAR^.sqltype and 1);
  FXSQLVAR^.sqlscale := 0;
  FXSQLVAR^.sqllen := SizeOf(Long);
  FIBAlloc(FXSQLVAR^.sqldata, 0, FXSQLVAR^.sqllen);
  PInt64(FXSQLVAR^.sqldata)^ := Value;
  FModified := True;
end;
{$ENDIF}
procedure TFIBXSQLVAR.SetAsCurrency(Value: Currency);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if FQuery.Database.SQLDialect < 3 then
   SetAsDouble(Value)
  else
  begin
    if IsNullable then
      IsNull := False;
    for i := 0 to FParent.FCount - 1 do
      if FParent.FNames[i] = FName then
      begin
        xvar := FParent[i];
        xvar.FXSQLVAR^.sqltype := SQL_INT64 or (xvar.FXSQLVAR^.sqltype and 1);
        xvar.FXSQLVAR^.sqlscale := -4;
        xvar.FXSQLVAR^.sqllen := SizeOf(Currency);
        FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
        PCurrency(xvar.FXSQLVAR^.sqldata)^ := Value;
        xvar.FModified := True;
      end;
  end;
end;
 
procedure TFIBXSQLVAR.SetAsComp(Value: comp); //patchInt64A
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    xvar.FXSQLVAR^.sqltype := SQL_INT64 or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqlscale := 0;
    xvar.FXSQLVAR^.sqllen := SizeOf(comp);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    PComp(xvar.FXSQLVAR^.sqldata)^ := Value;
    xvar.FModified := True;
  end;
end; //patchInt64A


procedure TFIBXSQLVAR.SetAsTime(Value: TDateTime);
var
  tm_date: TCTimeStructure;
  Hr, Mt, S, Ms: Word;
begin
  if FQuery.Database.SQLDialect < 3 then
  begin
    AsDateTime := Value;
    exit;
  end;
  if IsNullable then
    IsNull := False;
  FXSQLVAR^.sqltype := SQL_TYPE_TIME or (FXSQLVAR^.sqltype and 1);
  DecodeTime(Value, Hr, Mt, S, Ms);
  with tm_date do begin
    tm_sec := S;
    tm_min := Mt;
    tm_hour := Hr;
    tm_mday := 0;
    tm_mon := 0;
    tm_year := 0;
  end;
  FXSQLVAR^.sqllen := SizeOf(ISC_TIME);
  FIBAlloc(FXSQLVAR^.sqldata, 0, FXSQLVAR^.sqllen);
  isc_encode_sql_time(@tm_date, PISC_TIME(FXSQLVAR^.sqldata));
  FModified := True;
end;

procedure TFIBXSQLVAR.SetAsDate(Value: TDateTime);
var
  tm_date: TCTimeStructure;
  Yr, Mn, Dy: Word;
begin
  if FQuery.Database.SQLDialect < 3 then
  begin
    AsDateTime := Value;
    exit;
  end;
  if IsNullable then
    IsNull := False;
  FXSQLVAR^.sqltype := SQL_TYPE_DATE or (FXSQLVAR^.sqltype and 1);
  DecodeDate(Value, Yr, Mn, Dy);
  with tm_date do begin
    tm_sec := 0;
    tm_min := 0;
    tm_hour := 0;
    tm_mday := Dy;
    tm_mon := Mn - 1;
    tm_year := Yr - 1900;
  end;
  FXSQLVAR^.sqllen := SizeOf(ISC_DATE);
  FIBAlloc(FXSQLVAR^.sqldata, 0, FXSQLVAR^.sqllen);
  isc_encode_sql_date(@tm_date, PISC_DATE(FXSQLVAR^.sqldata));
  FModified := True;
end;


procedure TFIBXSQLVAR.SetAsDateTime(Value: TDateTime);
var
  tm_date: TCTimeStructure;
  Yr, Mn, Dy, Hr, Mt, S, Ms: Word;
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    DecodeDate(Value, Yr, Mn, Dy);
    DecodeTime(Value, Hr, Mt, S, Ms);
    with tm_date do begin
      tm_sec := S;
      tm_min := Mt;
      tm_hour := Hr;
      tm_mday := Dy;
      tm_mon := Mn - 1;
      tm_year := Yr - 1900;
    end;
    xvar.FXSQLVAR^.sqltype := SQL_DATE or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqllen := SizeOf(TISC_QUAD);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    isc_encode_date(@tm_date, PISC_QUAD(xvar.FXSQLVAR^.sqldata));
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsDouble(Value: Double);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    xvar.FXSQLVAR^.sqltype := SQL_DOUBLE or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqllen := SizeOf(Double);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    PDouble(xvar.FXSQLVAR^.sqldata)^ := Value;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsFloat(Value: Float);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    xvar.FXSQLVAR^.sqltype := SQL_FLOAT or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqllen := SizeOf(Float);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    PSingle(xvar.FXSQLVAR^.sqldata)^ := Value;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsLong(Value: Long);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    xvar.FXSQLVAR^.sqltype := SQL_LONG or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqllen := SizeOf(Long);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    PLong(xvar.FXSQLVAR^.sqldata)^ := Value;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsPointer(Value: Pointer);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable and (Value = nil) then
    IsNull := True
  else begin
    IsNull := False;
    for i := 0 to FParent.FCount - 1 do
               if FParent.FNames[i] = FName then begin
      xvar := FParent[i];
      xvar.IsNull := False;
      Move(Value^, xvar.FXSQLVAR^.sqldata, xvar.FXSQLVAR^.sqllen);
      xvar.FModified := True;
    end;
  end;
end;

procedure TFIBXSQLVAR.SetAsQuad(Value: TISC_QUAD);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    if (xvar.FXSQLVAR^.sqltype and (not 1) <> SQL_BLOB) and
       (xvar.FXSQLVAR^.sqltype and (not 1) <> SQL_ARRAY)
    then
    xvar.FXSQLVAR^.sqltype := SQL_BLOB or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqllen := SizeOf(TISC_QUAD);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    PISC_QUAD(xvar.FXSQLVAR^.sqldata)^ := Value;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsShort(Value: Short);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if IsNullable then
    IsNull := False;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    xvar.FXSQLVAR^.sqltype := SQL_SHORT or (xvar.FXSQLVAR^.sqltype and 1);
    xvar.FXSQLVAR^.sqllen := SizeOf(Short);
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen);
    PShort(xvar.FXSQLVAR^.sqldata)^ := Value;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsString(Value: String);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
  bs: TFIBBlobStream;
begin
  if IsNullable then
    IsNull := False;
  if (FXSQLVAR^.sqltype AND SQL_BLOB) = SQL_BLOB then begin
    bs := TFIBBlobStream.Create;
    try
      bs.Mode := bmWrite;
      bs.Database := FQuery.Database;
      bs.Transaction := FQuery.Transaction;
      bs.Truncate;
      if Length(Value) > 0 then
        bs.Write(Value[1], Length(Value));
      bs.Finalize;
      AsQuad := bs.BlobID;
    finally
      bs.Free;
    end;
  end else for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    if (xvar.FXSQLVAR^.sqlname = 'DB_KEY') or
       (xvar.FXSQLVAR^.sqlname = 'RDB$DB_KEY') then
      Move(Value[1], xvar.FXSQLVAR^.sqldata^, xvar.FXSQLVAR^.sqllen)
    else begin
     if (FXSQLVAR^.sqltype and (not 1) = SQL_TIMESTAMP) then begin
      SetAsDateTime(StrToDateTime(Value))
     end else begin
      xvar.FXSQLVAR^.sqltype := SQL_TEXT or (xvar.FXSQLVAR^.sqltype and 1);
      xvar.FXSQLVAR^.sqllen := Length(Value);
      FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, xvar.FXSQLVAR^.sqllen + 1);
      if (Length(Value) > 0) then
        Move(Value[1], xvar.FXSQLVAR^.sqldata^, xvar.FXSQLVAR^.sqllen);
     end
    end;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetAsVariant(Value: Variant);
var vt:integer;
begin
  vt:=VarType(Value);
  if (FXSQLVAR^.sqltype and (not 1) = SQL_TIMESTAMP) or
   (FXSQLVAR^.sqltype and (not 1) = SQL_TYPE_DATE)
  then
   if vt=varDouble then vt:=varDate;

  if VarIsNull(Value) then
    IsNull := True
  else case vt of
    varEmpty, varNull:
      IsNull := True;
    varSmallint, varInteger, varByte:
      AsLong := Value;
    varSingle, varDouble:
      AsDouble := Value;
    varCurrency:
      AsCurrency := Value; 
    varBoolean:
      if Value then
        AsVariant := VariantTrue
      else
        AsVariant := VariantFalse;
    varDate:
     case SQLType of
      SQL_TYPE_TIME: AsTime:=Value;
      SQL_TYPE_DATE: AsDate:=Value;
     else 
      AsDateTime := Value;
     end;
    varOleStr, varString:
      AsString := Value;
    varArray:
    {$IFDEF SUPPORT_ARRAY_FIELD}
     SetArrayValue(Value)
    {$ENDIF}
    ;
    varByRef, varDispatch, varError, varUnknown, varVariant:
      FIBError(feNotPermitted, [nil]);
  end;
end;

procedure TFIBXSQLVAR.SetAsXSQLVAR(Value: PXSQLVAR);
var
  sqlind: PShort;
  sqldata: PChar;
  local_sqllen: Integer;
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    sqlind := xvar.FXSQLVAR^.sqlind;
    sqldata := xvar.FXSQLVAR^.sqldata;
    Move(Value^, xvar.FXSQLVAR^, SizeOf(TXSQLVAR));
    xvar.FXSQLVAR^.sqlind := sqlind;
    xvar.FXSQLVAR^.sqldata := sqldata;
    if (Value^.sqltype and 1 = 1) then begin
      if (xvar.FXSQLVAR^.sqlind = nil) then
        FIBAlloc(xvar.FXSQLVAR^.sqlind, 0, SizeOf(Short));
      xvar.FXSQLVAR^.sqlind^ := Value^.sqlind^;
    end else if (xvar.FXSQLVAR^.sqlind <> nil) then
      FIBAlloc(xvar.FXSQLVAR^.sqlind, 0, 0);
    if ((xvar.FXSQLVAR^.sqltype and (not 1)) = SQL_VARYING) then
      local_sqllen := xvar.FXSQLVAR^.sqllen + 2
    else
      local_sqllen := xvar.FXSQLVAR^.sqllen;
    FIBAlloc(xvar.FXSQLVAR^.sqldata, 0, local_sqllen);
    Move(Value^.sqldata[0], xvar.FXSQLVAR^.sqldata[0], local_sqllen);
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetIsNull(Value: Boolean);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  if (Value and not IsNullable) then
    IsNullable := True;
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    if Value then
      xvar.FXSQLVAR^.sqlind^ := -1
    else if ((not Value) and xvar.IsNullable) then
      xvar.FXSQLVAR^.sqlind^ := 0;
    xvar.FModified := True;
  end;
end;

procedure TFIBXSQLVAR.SetIsNullable(Value: Boolean);
var
  i: Integer;
  xvar: TFIBXSQLVAR;
begin
  for i := 0 to FParent.FCount - 1 do
             if FParent.FNames[i] = FName then begin
    xvar := FParent[i];
    if (Value <> xvar.IsNullable) then begin
      if Value then begin
        xvar.FXSQLVAR^.sqltype := xvar.FXSQLVAR^.sqltype or 1;
        FIBAlloc(xvar.FXSQLVAR^.sqlind, 0, SizeOf(Short));
      end else begin
        xvar.FXSQLVAR^.sqltype := xvar.FXSQLVAR^.sqltype and (not 1);
        FIBAlloc(xvar.FXSQLVAR^.sqlind, 0, 0);
      end;
    end;
  end;
end;

(* TFIBXSQLDA *)
constructor TFIBXSQLDA.Create;
begin
  FNames := TStringList.Create;
  FSize := 0;
end;

destructor TFIBXSQLDA.Destroy;
var
  i: Integer;
begin
  FNames.Free;
  if FXSQLDA <> nil then begin
    for i := 0 to FSize - 1 do with FXSQLDA^.sqlvar[i] do begin
      FIBAlloc(sqldata, 0, 0);
      FIBAlloc(sqlind, 0, 0);
      FXSQLVARs^[i].Free;
    end;
    FIBAlloc(FXSQLDA, 0, 0);
    FIBAlloc(FXSQLVARs, 0, 0);
    FXSQLDA := nil;
  end;
  inherited;
end;


procedure TFIBXSQLDA.AddName(FieldName: String; Idx: Integer);
var
  fn: String;
begin
  if Assigned(FQuery) and Assigned(FQuery.Database) then
   if FQuery.Database.SQLDialect=1 then
    fn := FormatIdentifier(1, FieldName)
   else
    fn := FieldName
  else
    fn := FormatIdentifier(1, FieldName);
//  fn := FieldName;
  while FNames.Count <= Idx do  FNames.Add('');
  FNames[Idx] := fn;
  FXSQLVARs^[Idx].FName := fn;
  FXSQLVARs^[Idx].FIndex := Idx;
end;

function TFIBXSQLDA.GetModified: Boolean;
var
  i: Integer;
begin
  Result := False;
  for i := 0 to FCount - 1 do if FXSQLVARs^[i].Modified then begin
    Result := True;
    exit;
  end;

end;

function TFIBXSQLDA.GetNames: String;
begin
  Result := FNames.Text;
end;

function TFIBXSQLDA.GetRecordSize: Integer;
begin
  Result := SizeOf(TFIBXSQLDA) + XSQLDA_LENGTH(FSize);
end;

function TFIBXSQLDA.GetXSQLDA: PXSQLDA;
begin
  Result := FXSQLDA;
end;

function TFIBXSQLDA.GetXSQLVAR(Idx: Integer): TFIBXSQLVAR;
begin
  if (Idx < 0) or (Idx >= FCount) then
    FIBError(feXSQLDAIndexOutOfRange, [nil]);
  Result := FXSQLVARs^[Idx]
end;

function TFIBXSQLDA.GetXSQLVARByName(Idx: String): TFIBXSQLVAR;
var
  s: String;
  i, Cnt: Integer;

    function GetName(Index: LongInt): string;
    begin
// compatibility with IB4..5 field names conventions
     with FQuery.Database do
      if UpperOldNames and (SQLDialect>1) and
         not EasyNeedQuote(Idx) then
         Result := AnsiUpperCase(FNames[Index])
      else Result := FNames[Index];
    end;

begin
  if Assigned(FQuery) and Assigned(FQuery.Database) then
   if FQuery.Database.SQLDialect=1 then
    s := FormatIdentifier(1, Idx)
   else begin
     with FQuery.Database do
      if UpperOldNames  and
       not EasyNeedQuote(Idx) then s := AnsiUpperCase(Idx)
     else s := Idx
   end
  else
    s := FormatIdentifier(1, Idx);
  
  i := 0;
  Cnt := FNames.Count;
  while (i < Cnt) and (GetName(i) <> s) do Inc(i); // Wizard
  if i = Cnt then
    Result := nil
  else
    Result := GetXSQLVAR(i);
end;

procedure TFIBXSQLDA.Initialize;
var
  i, j, j_len: Integer;
  NamesWereEmpty: Boolean;
  st: String;
begin
  NamesWereEmpty := (FNames.Count = 0);
  if FXSQLDA <> nil then begin
    for i := 0 to FCount - 1 do begin
      with FXSQLVARs^[i].Data^ do begin
        if NamesWereEmpty then begin
          st := String(aliasname);
          if st = '' then begin
            st := 'F_';
            aliasname_length := 2;
            j := 1; j_len := 1;
            StrPCopy(aliasname, st + IntToStr(j));
          end else begin
            StrPCopy(aliasname, st);
            j := 0; j_len := 0;
          end;
          while GetXSQLVARByName(String(aliasname)) <> nil do begin
            Inc(j); j_len := Length(IntToStr(j));
            if j_len + aliasname_length > 31 then
              StrPCopy(aliasname,
                       Copy(st, 1, 31 - j_len) +
                       IntToStr(j))
            else
              StrPCopy(aliasname, st + IntToStr(j));
          end;
          Inc(aliasname_length, j_len);
          AddName(String(aliasname), i);
        end;
        case sqltype and (not 1) of
          SQL_TEXT, SQL_TYPE_DATE, SQL_TYPE_TIME, SQL_TIMESTAMP,
          SQL_BLOB, SQL_ARRAY, SQL_QUAD, SQL_SHORT,
          SQL_LONG, SQL_INT64, SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT:

          begin
            FIBAlloc(sqldata, 0, sqllen);
          end;
          SQL_VARYING: begin
            FIBAlloc(sqldata, 0, sqllen + 2);
          end;
          else
            FIBError(feUnknownSQLDataType, [sqltype and (not 1)])
        end;
        if (sqltype and 1 = 1) then
          FIBAlloc(sqlind, 0, SizeOf(Short))
        else if (sqlind <> nil) then
          FIBAlloc(sqlind, 0, 0);
      end;
    end;
  end;
end;

procedure TFIBXSQLDA.SetCount(Value: Integer);
var
  i, OldSize: Integer;
begin
  FNames.Clear;
  FCount := Value;
  if FSize > 0 then
    OldSize := XSQLDA_LENGTH(FSize)
  else
    OldSize := 0;
  if FCount > FSize then begin
    FIBAlloc(FXSQLDA, OldSize, XSQLDA_LENGTH(FCount));
    FIBAlloc(FXSQLVARs, FSize * SizeOf(TFIBXSQLVAR), FCount * SizeOf(TFIBXSQLVAR));
    FXSQLDA^.version := SQLDA_VERSION1;
    for i := 0 to FCount - 1 do begin
      if i >= FSize then begin
        FXSQLVARs^[i] := TFIBXSQLVAR.Create(Self);
        FXSQLVARs^[i].FQuery := FQuery;
      end;
      FXSQLVARs^[i].FXSQLVAR := @FXSQLDA^.sqlvar[i];
    end;
    FSize := FCount;
  end;
  if FSize > 0 then begin
    FXSQLDA^.sqln := Value;
    FXSQLDA^.sqld := Value;
  end;
end;

(* TFIBQuery *)
constructor TFIBQuery.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FGoToFirstRecordOnExecute := True;
  FBase := TFIBBase.Create(Self);
  FBase.OnDatabaseDisconnecting := DatabaseDisconnecting;
  FBase.OnTransactionEnding := TransactionEnding;
  FBOF := False;
  FEOF := False;
  FPrepared := False;
  FRecordCount := 0;
  FSQL := TStringList.Create;
  TStringList(FSQL).OnChanging := SQLChanging;
  FProcessedSQL := TStringList.Create;
  FHandle := nil;
  FSQLParams := TFIBXSQLDA.Create;
  FSQLParams.FQuery := Self;
  FSQLRecord := TFIBXSQLDA.Create;
  FSQLRecord.FQuery := Self;
  FUserSQLParams:=TFIBXSQLDA.Create;
  FUserSQLParams.FQuery := Self;
  FSQLType := SQLUnknown;
  FParamCheck := True;
  FCursor := Name + RandomString(8);
  FProcExecuted:=false;

end;

destructor TFIBQuery.Destroy;
begin
  if (FOpen) then
    Close;
  if (FHandle <> nil) then
    FreeHandle;
  FSQL.Free;
  FProcessedSQL.Free;
  FBase.Free;
  FSQLParams.Free;
  FSQLRecord.Free;
  FUserSQLParams.Free;
  inherited;
end;

procedure TFIBQuery.BatchInput(InputObject: TFIBBatchInputStream);
begin
  if not Prepared then
    Prepare;
  InputObject.FParams := Self.Params;
  InputObject.ReadyStream;
  if FSQLType in [SQLInsert, SQLUpdate, SQLDelete, SQLExecProcedure] then
    while InputObject.ReadParameters do
      ExecQuery;
end;

procedure TFIBQuery.BatchOutput(OutputObject: TFIBBatchOutputStream);
begin
  CheckClosed;
  if not Prepared then
    Prepare;
  if FSQLType = SQLSelect then begin
    try
      ExecQuery;
      OutputObject.FColumns := Self.FSQLRecord;
      OutputObject.ReadyStream;
      if not FGoToFirstRecordOnExecute then
        Next;
      while (not Eof) and (OutputObject.WriteColumns) do
        Next;
    finally
      Close;
    end;
  end;
end;

procedure TFIBQuery.CheckClosed;
begin
  if FOpen then FIBError(feDatasetOpen, [nil]);
end;

procedure TFIBQuery.CheckOpen;
begin
  if not FOpen then FIBError(feDatasetClosed, [nil]);
end;

procedure TFIBQuery.CheckValidStatement;
begin
  FBase.CheckTransaction;
  if (FHandle = nil) then
    FIBError(feInvalidStatementHandle, [nil]);
end;

procedure TFIBQuery.Close;
var
  isc_res: ISC_STATUS;
begin
  try
    if (FHandle <> nil) and (SQLType = SQLSelect) and FOpen then begin
      isc_res := Call(
                   isc_dsql_free_statement(StatusVector, @FHandle, DSQL_close),
                   False);
      if (StatusVector^ = 1) and (isc_res > 0) and
        not CheckStatusVector(
              [isc_bad_stmt_handle, isc_dsql_cursor_close_err]) then
        IBError(Self);
    end;
  finally
    FEOF := False;
    FBOF := False;
    FOpen := False;
    FProcExecuted:=false;
    FRecordCount := 0;
  end;
end;

function TFIBQuery.Call(ErrCode: ISC_STATUS; RaiseError: Boolean): ISC_STATUS;
begin
  Result := 0;
  if Transaction <> nil then
    Result := Transaction.Call(ErrCode, False);
  if RaiseError and (ErrCode > 0) then
    IbError(Self);
end;

function TFIBQuery.Current: TFIBXSQLDA;
begin
  Result := FSQLRecord;
end;

procedure TFIBQuery.DatabaseDisconnecting(Sender: TObject);
begin
  if (FHandle <> nil) then begin
    Close;
    FreeHandle;
  end;
end;

// String routine
function ReduceBlanks(const S: string): string;
var i,cntB :integer;
    InQuote:boolean;
    QuoteChar:char;
begin
 Result:='';cntB:=0;
 InQuote:=false;
 QuoteChar:='"';
 for i:=1 to Length(S) do
 begin
  if not InQuote then begin
   InQuote:= (S[i] in ['''','"']);
   if InQuote then QuoteChar:=S[i];
  end
  else
   InQuote:= (S[i] <> QuoteChar);
  if (S[i]=' ') and not InQuote then begin
   Inc(cntB);
   if cntB=1 then   Result:=Result+' '
  end
  else begin
   Result:=Result+Copy(S,i,1);
   cntB:=0;
  end;

 end;
end;


function ReplaceCIStr(const S, Srch, Replace: string): string;
var
  I: Integer;
  Source: string;
begin
  Source := ReduceBlanks(S);
  Result := '';
  repeat
    I := PosCI(Srch, Source);
    if (I > 0) and (Source[i-1] in ['<','>'])
     and  (Srch <> #13) and (Srch <> #10) and (Srch[1]<>'@')
    then I:=0;
//       "<=;>=" ,     ,
//  
    if (I > 0)  then begin
      Result := Result + Copy(Source, 1, I - 1) + Replace;
      Source := Copy(Source, I + Length(Srch), MaxInt);
    end
    else Result := Result + Source;
  until I <= 0;
end;

const
  BeginWhere =' WHERE ';

function WhereCount(SQLText:string):integer;
var p:integer;
begin
//   Where clause
//  SQLText:=ReplaceCIStr(SQLText, #13#10,' ');
  Result:=0;
  p:=PosCI(BeginWhere,SQLText);
  while p>0 do begin
   Inc(Result);
   SQLText:=Copy(SQLText,p+7,10000);
   p:=PosCI(BeginWhere,SQLText);
  end;
end;


function  GetWhereClause(SQLText:string;N:integer):string;
var p,p1,p2:integer;
begin
//  N  where clause
// Returns N  where clause
  Result:='';
  p1:=0;  p:=PosCI(BeginWhere,SQLText);
  while  (p>0) and (p1<N) do begin
   Inc(p1);
   SQLText:=Copy(SQLText,p+7,10000);
   p:=Pos(BeginWhere,SQLText);
  end;
  if (p1<N) then Exit;
  p:=1; p1:=0; p2:=0;
  // p1= count of '('; p2= count of ')'
  while (p<Length(SQLText)) and (p2<=p1) do begin
   if SQLText[p]='(' then Inc(p1)
   else
   if SQLText[p]=')' then Inc(p2);
   Inc(p)
  end;
  if p2>p1 then SQLText:=Copy(SQLText,1,p-1);
  Result:=SQLText;
  p:=Pos(' GROUP ',SQLText);
  if p>0 then begin
   Result:=Copy(SQLText,1,p-1); Exit
  end
  else begin
   p:=PosCI(' PLAN ',SQLText);
   if p>0 then begin
    Result:=Copy(SQLText,1,p-1); Exit
   end
   else begin
    p:=PosCI(' ORDER ',SQLText);
    if p>0 then  Result:=Copy(SQLText,1,p-1);
   end
  end
end;


//

function  TFIBQuery.OK_SQLText:string;
var wcls:TStringList;
    wcl:string;
    p,i,j,wcnt:integer;
    ExistNullParams:boolean;
const  StrIsNull=' Is Null ';
begin
  Result:=vPrepUserSQL; ExistNullParams:=false;
  Result:=ReplaceCIStr(Result, #13#10,' ');

  Result:=SetMacro(Result,false);
  wcnt  :=WhereCount(Result);
  if (wcnt=0) or (Params.Count=0) then Exit;
  for i:=0 to Pred(Params.Count) do begin
    ExistNullParams:=Params[i].isNull;
    if ExistNullParams then Break;
  end;
  if not ExistNullParams then Exit;

  wcls:=TStringList.Create;
//  wcls.Capacity:=wcnt;
  try
   for j:=1 to wcnt do
      wcls.Add(GetWhereClause(Result,j));
   for i:=0 to Pred(Params.Count) do
    if Params[i].isNull then
     for j:=1 to wcnt do begin
      wcl:=wcls[j-1];
      p:=PosCI(Params[i].Name,wcl);
      if (p>1) and ((wcl[p-1] = '?') or (wcl[p-1]=':'))
       or ((p>2)and ((wcl[p-1] = '"') and ((wcl[p-2] = '?') or (wcl[p-2]=':'))) )
       then begin
       Result:=
        ReplaceCIStr(Result, '=?'+Params[i].Name+' ', StrIsNull);
       Result:=
        ReplaceCIStr(Result, '=?"'+Params[i].Name+'" ', StrIsNull);
       Result:=
        ReplaceCIStr(Result, '=:'+Params[i].Name+' ', StrIsNull);
       Result:=
        ReplaceCIStr(Result, '=:"'+Params[i].Name+'" ', StrIsNull);

       Result:=
        ReplaceCIStr(Result, '= ?'+Params[i].Name+' ', StrIsNull);
       Result:=
        ReplaceCIStr(Result, '= ?"'+Params[i].Name+'" ', StrIsNull);

       Result:=
        ReplaceCIStr(Result, '= :'+Params[i].Name+' ', StrIsNull);
       Result:=
        ReplaceCIStr(Result, '= :"'+Params[i].Name+'" ', StrIsNull);
      end;
     end;
   finally
    wcls.Free
   end;
end;

procedure TFIBQuery.SynchronizeParams;
var i:integer;
    pc:integer;
    pt :integer; //parameterType
begin
 pc:=Pred(Params.Count);
 for i:=0 to pc do
   if FSQLParams.ByName[Params[i].Name]<>nil then
   with FSQLParams.ByName[Params[i].Name] do
   begin
    if (Params[i].FXSQLVar^.sqltype and (not 1) <> SQL_BLOB) and
       (Params[i].FXSQLVar^.sqltype and (not 1) <> SQL_ARRAY) then
    begin
     if Params[i].FXSQLVar^.sqltype and (not 1) = SQL_INT64 then
     begin
      if Params[i].FXSQLVar^.sqlscale = 0 then
       AsComp:=Params[i].asComp
      else
       AsVariant:=Params[i].asVariant
     end
     else
     if (FXSQLVAR^.sqltype and (not 1) = SQL_TIMESTAMP) or
       (FXSQLVAR^.sqltype and (not 1) = SQL_TYPE_DATE)
     then begin
       pt:=Params[i].FXSQLVAR^.sqltype and (not 1);
       if (pt=SQL_DOUBLE) or (pt= SQL_FLOAT)or (pt= SQL_D_FLOAT)
       then
        Params[i].asDateTime:=Params[i].Value;
       if Params[i].AsString<>'' then
        SetAsDateTime(StrToDateTime(Params[i].AsString))
       else
        AsVariant:= Params[i].asVariant;
     end
     else
     if (FXSQLVAR^.sqltype and (not 1) = SQL_TYPE_TIME) then
     begin
       pt:=Params[i].FXSQLVAR^.sqltype and (not 1);
       if (pt=SQL_DOUBLE) or (pt= SQL_FLOAT)or (pt= SQL_D_FLOAT)
       then
        Params[i].asDateTime:=Params[i].Value;

       if Params[i].AsString<>'' then
        SetAsTime(StrToTime(Params[i].AsString))
       else
        AsVariant:= Params[i].asVariant;
     end
     else
      AsVariant:= Params[i].asVariant;
    end
    else
     AsQuad:=Params[i].AsQuad;
   end;
end;

type THackMonitorHook=class(TFIBSQLMonitorHook);

function  TFIBQuery.SetMacro(const SQLText:string;UseDefaults:boolean):string;
var i:integer;
    BoundMacro:string;
begin
//    -   CLRF
 Result:=SQLText;
 if Pos('@',Result)=0 then Exit; 
  for i := 0 to Pred(Params.Count) do
  if Params[i].isMacro then
  begin
     if Params[i].Quoted then BoundMacro:='"' else BoundMacro:='';
      Result:=ReplaceCIStr(Result,'@'+Params[i].Name+' ',
      ' '+BoundMacro+Params[i].asString+BoundMacro+' ');
      Result:=ReplaceCIStr(Result,'@'+Params[i].Name+#13#10,
       ' '+BoundMacro+Params[i].asString+BoundMacro+' ')
  end
end;

procedure TFIBQuery.ExecQuery;
var
  st:string;
  fetch_res: ISC_STATUS;
begin
  CheckClosed;
  Prepare;
  CheckValidStatement;
  SynchronizeParams;
  if MonitorHook<>nil then
   MonitorHook.SQLExecute(Self);
  try
   case FSQLType of
    SQLSelect: begin
      Call(isc_dsql_execute2(StatusVector,
                            TRHandle,
                            @FHandle,
                            DataBase.SQLDialect,
                            FSQLParams.AsXSQLDA,
                            nil), True);
      Call(
        isc_dsql_set_cursor_name(StatusVector, @FHandle, PChar(FCursor), 0),
        True);
      FOpen := True;
      FBOF := True;
      FEOF := False;
      FRecordCount := 0;
      if FGoToFirstRecordOnExecute then
        Next;
    end;
    SQLExecProcedure: begin
      fetch_res := Call(isc_dsql_execute2(StatusVector,
                            TRHandle,
                            @FHandle,
                            DataBase.SQLDialect,
                            FSQLParams.AsXSQLDA,
                            FSQLRecord.AsXSQLDA), False);
      if (fetch_res <> 0) and
        (fetch_res<>isc_deadlock) and (fetch_res<>isc_except) then begin 
        (*
         * Sometimes a prepared stored procedure gets confused (on the
         * InterBase side of things...).
         * This code is meant to try to work around the problem simply
         * by "retrying" the stuff.
         *)
        isc_dsql_prepare(StatusVector, TRHandle, @FHandle, 0,
                         PChar(FProcessedSQL.Text), 1, nil);
        Call(isc_dsql_execute2(StatusVector,
                            TRHandle,
                            @FHandle,
                            DataBase.SQLDialect,
                            FSQLParams.AsXSQLDA,
                            FSQLRecord.AsXSQLDA), True);
      end
      else
      if (fetch_res <> 0) then  IbError(Self) ;
     FProcExecuted:=true;      
    end else
      Call(isc_dsql_execute(StatusVector,
                           TRHandle,
                           @FHandle,
                           DataBase.SQLDialect,
                           FSQLParams.AsXSQLDA), True);

  end;
  except
   On E:Exception do begin
    if MonitorHook<>nil then
    with THackMonitorHook(MonitorHook) do    //Added Source
      if MonitoringEnabled and (MonitorCount > 0) then begin
       if Owner is TFIBDataSet then   st:=Owner.Name else st:=Name;
        WriteSQLData(st + ': [Execute] ' + E.Message);
      end;
    raise;
   end
  end;
end;

function TFIBQuery.GetEOF: Boolean;
begin
  Result := FEOF or not FOpen;
end;

function TFIBQuery.GetFieldByName(FieldName: String): TFIBXSQLVAR;
var
  i: Integer;
begin
  i := GetFieldIndex(FieldName);
  if (i < 0) then
    FIBError(feFieldNotFound, [FieldName]);
  Result := GetFields(i);
end;

function TFIBQuery.GetFields(const Idx: Integer): TFIBXSQLVAR;
begin
  if (Idx < 0) or (Idx >= FSQLRecord.Count) then
    FIBError(feFieldNotFound, ['Field No: ' + IntToStr(Idx)]);
  Result := FSQLRecord[Idx];
end;

function TFIBQuery.GetFieldIndex(FieldName: String): Integer;
begin
  if (FSQLRecord.ByName[FieldName] = nil) then
    Result := -1
  else
    Result := FSQLRecord.ByName[FieldName].Index;
end;

function TFIBQuery.Next: TFIBXSQLDA;
var
  fetch_res: ISC_STATUS;
  StopFetching:boolean;
begin
  Result := nil;
  if not FEOF then begin
    CheckOpen;
    // Go to the next record...
    StopFetching:=false;
    if Assigned(FOnSQLFetch) then
     FOnSQLFetch(FRecordCount,StopFetching);
    if StopFetching then Abort;
    
    fetch_res :=
      Call(isc_dsql_fetch(StatusVector, @FHandle, Database.SQLDialect, FSQLRecord.AsXSQLDA), False);
    if (fetch_res = 100) or (CheckStatusVector([isc_dsql_cursor_err])) then begin
      FEOF := True;
      //FOpen := False;
    end else if (fetch_res > 0) then begin
      try
       IbError(Self);
      except
        Close;
        raise ;
      end;
    end else begin
      Inc(FRecordCount);
      FBOF := False;
      Result := FSQLRecord;
    end;
    if MonitorHook<>nil then
     MonitorHook.SQLFetch(Self);

  end
end;

procedure TFIBQuery.FreeHandle;
var
  isc_res: ISC_STATUS;
begin
  try
    FSQLRecord.Count := 0;
    if FHandle <> nil then begin
      isc_res :=
        Call(isc_dsql_free_statement(StatusVector, @FHandle, DSQL_drop), False);
      if (StatusVector^ = 1) and (isc_res > 0) and (isc_res <> isc_bad_stmt_handle) then
        IbError(Self);
    end;
  finally
    FPrepared := False;
    FHandle := nil;
  end;
end;

function TFIBQuery.GetDatabase: TFIBDatabase;
begin
  Result := FBase.Database;
end;

function TFIBQuery.GetDBHandle: PISC_DB_HANDLE;
begin
  Result := FBase.DBHandle;
end;

function TFIBQuery.GetPlan: String;
var
  Result_buffer: array[0..16384] of Char;
  Result_length, i: Integer;
  info_request: Char;
begin
  if (not Prepared) or
     (not (FSQLType in [SQLSelect, SQLSelectForUpdate, SQLExecProcedure,
                        SQLUpdate, SQLDelete])) then
    Result := ''
  else begin
    info_request := Char(isc_info_sql_get_plan);
    Call(isc_dsql_sql_info(StatusVector, @FHandle, 2, @info_request,
                           SizeOf(Result_buffer), Result_buffer), True);
    if (Result_buffer[0] <> Char(isc_info_sql_get_plan)) then
      FIBError(feUnknownError, [nil]);
    Result_length := isc_vax_integer(@Result_buffer[1], 2);
    SetString(Result, nil, Result_length);
    for i := 1 to Result_length do
      Result[i] := Result_buffer[i + 2];
    Result := Trim(Result);
  end;
end;

function TFIBQuery.GetRecordCount: Integer;
begin
  Result := FRecordCount;
end;

function TFIBQuery.GetRowsAffected: integer;
var
  Result_buffer: array[0..1048] of Char;
  info_request: Char;
begin
  if not Prepared then
    Result := -1
  else begin
    info_request := Char(isc_info_sql_records);
    if isc_dsql_sql_info(StatusVector, @FHandle, 1, @info_request,
                         SizeOf(Result_buffer), Result_buffer) > 0 then
      IbError(Self);
    if (Result_buffer[0] <> Char(isc_info_sql_records)) then
      FIBError(feUnknownError, [nil]);
    case SQLType of
    SQLUpdate:   Result := isc_vax_integer(@Result_buffer[6], 4);
    SQLDelete:   Result := isc_vax_integer(@Result_buffer[13], 4);
    SQLInsert:   Result := isc_vax_integer(@Result_buffer[27], 4);
    else         Result := -1 ;
    end ;
  end;
end;

function TFIBQuery.GetSQLParams: TFIBXSQLDA;
begin
//  if not Prepared then    Prepare; // Added comment
//   Result := FSQLParams
  if FUserSQLParams.FXSQLDA=nil then SQLChange(nil);
  Result := FUserSQLParams;
end;

function TFIBQuery.GetTransaction: TFIBTransaction;
begin
  Result := FBase.Transaction;
end;

function TFIBQuery.GetTRHandle: PISC_TR_HANDLE;
begin
  Result := FBase.TRHandle;
end;

(*
 * Preprocess SQL
 *  Using FSQL, process the typed SQL and put the process SQL
 *  in FProcessedSQL and parameter names in FSQLParams
 *)
const
  ParamNameChars=['A'..'Z', 'a'..'z', '0'..'9', '_', '$','%','#','.'];

procedure TFIBQuery.PreprocessSQL(const SQLText:string;IsUserSQL:boolean);
const
  DefParCount=50;
var
  cCurChar, cNextChar, cQuoteChar: Char;
  sSQL, sProcessedSQL, sParamName: String;
  i, iLenSQL, iCurState, iSQLPos: Integer;
  iCurParamState:byte;
  slNames: TStrings;

  Parms:TFIBXSQLDA;
  OldParamValues:Variant;
  OldParamCount:integer;
  PCount,j:integer;

  MacroDefValues: Variant;


  procedure AddToProcessedSQL(cChar: Char);
  begin
//    sProcessedSQL[iSQLPos] := cChar;
    sProcessedSQL:=sProcessedSQL+cChar;
    Inc(iSQLPos);
  end;

const
  DefaultState = 0;
  CommentState = 1;
  QuoteState = 2;
  ParamState = 3;
  MacroState =4;
  ParamDefaultState = 0;
  ParamQuoteState = 1;

procedure RegParamName(AddQuote:boolean);
var j:integer;
begin
 iCurParamState := ParamDefaultState;
 iCurState := DefaultState;
 Inc(PCount);
 slNames.Add(sParamName);
 if IsUserSQL then begin
  if AddQuote then    AddToProcessedSQL('"');
  for j:=1 to Length(sParamName) do begin
   if (j=1) and (sParamName[j]='@') then Continue;
   AddToProcessedSQL(sParamName[j]);
  end;
  if AddQuote then    AddToProcessedSQL('"');
 end;
 AddToProcessedSQL(' ');
 sParamName := '';
end;

begin
  OldParamCount:=0;
  slNames       := TStringList.Create;
  PCount:=0;
  try
    (* Do some initializations of variables *)
    cQuoteChar := '''';
    sSQL :=SQLText;
    iLenSQL := Length(sSQL); //+MaxParams
//    SetString(sProcessedSQL, nil, iLenSQL);
    sProcessedSQL:='';
    MacroDefValues:=VarArrayCreate([0,DefParCount],varOleStr);
//    for i:=0 to MaxParams do MacroDefValues[i]:='';

    i := 1;
    iSQLPos := 1;
    iCurState := DefaultState;
    iCurParamState := ParamDefaultState;
    (*
     * Now, traverse through the SQL string, character by character,
     * picking out the parameters and formatting correctly for InterBase.
     *)
    while (i <= iLenSQL) do begin
      // Get the current token and a look-ahead.
      cCurChar := sSQL[i];
      if i = iLenSQL then
        cNextChar := #0
      else
        cNextChar := sSQL[i + 1];
      // Now act based on the current state.
      case iCurState of
        DefaultState: begin
          case cCurChar of
            '''', '"': begin
              cQuoteChar := cCurChar;
              iCurState := QuoteState;
            end;
            '?', ':': begin
              iCurState := ParamState;
              AddToProcessedSQL('?');
            end;
            '@':
             if IsUserSQL then  begin
              iCurState := MacroState;
              AddToProcessedSQL('@');
             end;
            '/': if (cNextChar = '*') then begin
              AddToProcessedSQL(cCurChar);
              Inc(i);
              iCurState := CommentState;
            end;
          end;
        end;
        CommentState: begin
          if (cNextChar = #0) then
            FIBError(feSQLParseError, ['EOF in comment detected'])
          else if (cCurChar = '*') then begin
            if (cNextChar = '/') then
              iCurState := DefaultState;
          end;
        end;
        QuoteState: begin
          if (cNextChar = #0) then
            FIBError(feSQLParseError, ['EOF in string detected'])
          else if (cCurChar = cQuoteChar) then begin
            if (cNextChar = cQuoteChar) then begin
              AddToProcessedSQL(cCurChar);
              Inc(i);
            end else
              iCurState := DefaultState;
          end;
        end;
        ParamState,MacroState: begin
         if iCurParamState = ParamDefaultState then
          if cCurChar = '"' then begin
           iCurParamState := ParamQuoteState;
           Inc(i);         Continue;
          end;
          // Step 1, collect the name of the parameter
          if (cCurChar in ParamNameChars) or
           (iCurParamState = ParamQuoteState)
          then
              sParamName := sParamName + cCurChar
          else
            FIBError(feSQLParseError, ['Parameter name expected']);
          // Step 2, determine if the parameter name is finished.
          if  (cNextChar='"') and (iCurParamState = ParamQuoteState)
          then begin
              RegParamName(true);
              Inc(i,2);
              Continue;
          end;

          if not (cNextChar in ParamNameChars)
          and not (iCurParamState = ParamQuoteState)
           then begin
            Inc(i);
            if (iCurState=MacroState) or true then begin
               if PCount>DefParCount then
                VarArrayRedim(MacroDefValues, PCount);
               j:=Pos('%',sParamName);
               if j>0 then begin
                 MacroDefValues[PCount]:=
                   Copy(sParamName,j+1,255) ;
                 sParamName   :='@'+Copy(sParamName,1,j-1);
               end
               else MacroDefValues[PCount]:='';
            end;
            RegParamName(false);
          end;
        end;
      end;
      if not (iCurState in  [ParamState,MacroState]) then
        AddToProcessedSQL(sSQL[i]);
      Inc(i);
    end; // end while
    AddToProcessedSQL(#0);
// Save Old Params Routine

    if not IsUserSQL then begin
     Parms:=FSQLParams;
     sProcessedSQL:=SetMacro(sProcessedSQL,false);
    end
    else
     Parms:=FUserSQLParams;
// if Assigned(DataBase) then begin
    if IsUserSQL then begin


    // Save Old  Param values
     OldParamCount:=slNames.Count;
     OldParamValues:=
      VarArrayCreate([0, Pred(OldParamCount),0, 1], varVariant);
      for i := 0 to Pred(OldParamCount) do  begin
       if slNames[i][1]<>'@' then
        OldParamValues[i,0]:=slNames[i]
       else
        OldParamValues[i,0]:=Copy(slNames[i],2,255);
        if (MacroDefValues[i]<>'') then begin
         if string(MacroDefValues[i])[1]='#' then
          OldParamValues[i,1]:=Copy(MacroDefValues[i],2,255)
         else
          OldParamValues[i,1]:=MacroDefValues[i];
         if csDesigning in ComponentState then Continue
        end;
       if Parms.ByName[OldParamValues[i,0]]=nil then Continue;
       if (Parms.ByName[OldParamValues[i,0]].sqltype and (not 1) <> SQL_BLOB) and
        (Parms.ByName[OldParamValues[i,0]].sqltype and (not 1) <> SQL_ARRAY) then
         OldParamValues[i,1]:=Parms.ByName[OldParamValues[i,0]].Value
       else
         OldParamValues[i,0]:='<BUZZ>' // BlobParams can't restore after Change SQL text
      end;
    end;
    Parms.Count := slNames.Count;
    for i := 0 to slNames.Count - 1 do begin
      sParamName:=slNames[i];
      if sParamName[1]<>'@' then begin
       Parms.AddName(sParamName, i);
      end
      else begin
       Parms.AddName(Copy(sParamName,2,255), i);
       Parms[i].IsMacro:=true;
       Parms[i].Quoted :=(MacroDefValues[i]<>'') and (string(MacroDefValues[i])[1]='#');
      end;
    end;
    for i := 0 to Pred(Parms.Count)  do begin
     Parms[i].IsNullable:=true;
     Parms[i].IsNull:=true
    end;
     // Restore Old  Param values
    if IsUserSQL then
     if OldParamCount>0 then
      for i := 0 to Pred(OldParamCount) do    begin
       if Parms.ByName[OldParamValues[i,0]]<>nil then
       with Parms.ByName[OldParamValues[i,0]] do
       begin
         Value:=OldParamValues[i,1];
       end
      end;
//  end;

  if not IsUserSQL then
    FProcessedSQL.Text := sProcessedSQL
  else
    vPrepUserSQL       := sProcessedSQL ;
  finally
    slNames.Free;
  end;
end;

procedure TFIBQuery.SetDatabase(Value: TFIBDatabase);
begin
  if (Value <> FBase.Database) and FPrepared then
    FreeHandle;
  FBase.Database := Value;
  if Value<>nil then begin
   TStringList(FSQL).OnChange   := SQLChange; // Added Source
 //  SQLChange(nil)
  end;
end;

// Array Support
{$IFDEF SUPPORT_ARRAY_FIELD}
procedure TFIBQuery.PrepareArrayFields;
var i:integer;
    v:TFIBXSQLVAR;
    vISC_ARRAY_DESC:TISC_ARRAY_DESC;
    vArrayType:TFieldType;
begin
 for i:=0 to Pred(Current.Count) do begin
   v:=Current.Vars[i];
   with v,v.FXSQLVAR^ do
   if (v.FXSQLVAR^.sqltype and (not 1) = SQL_ARRAY) then try
    isc_array_lookup_bounds(StatusVector, DBHandle, TRHandle,
      relName, sqlname, @vISC_ARRAY_DESC);
    vArrayType:=ftUnknown;
    with vISC_ARRAY_DESC do
    begin
      case array_desc_dtype of
        blr_int64 : vArrayType            := ftBytes;
        blr_text, Blr_Varying: vArrayType := ftString;
        Blr_Short:             vArrayType := ftSmallint;
        Blr_Long:              vArrayType := ftInteger;
        Blr_Float, Blr_Double,
        Blr_D_Float:           vArrayType := ftFloat;
        Blr_Date:              vArrayType := ftDateTime;
        blr_sql_date:          vArrayType := ftDate;
        blr_sql_time:          vArrayType := ftTime;
      end;

      vFIBArray:=TpFIBArray.Create(FXSQLVAR,
        vISC_ARRAY_DESC,relName, sqlname,vArrayType
      );

    end;
  except
   IbError(Self)
  end
 end;
end;
{$ENDIF}
procedure TFIBQuery.Prepare;
var
  stmt_len: Integer;
  res_buffer: array[0..7] of Char;
  type_item: Char;
  st:string;
  //Added Variables
  OldRunningText,SQLText:string;
begin
  CheckClosed;
  FBase.CheckDatabase;
  FBase.CheckTransaction;
  if (ParamCheck) and (Params.Count>0) then begin
   if vPrepUserSQL='' then SQLChange(nil);
   SQLText:=OK_SQLText;
   OldRunningText:=FProcessedSQL.Text;
   PreprocessSQL(SQLText,false);
   if (OldRunningText<>FProcessedSQL.Text) then FreeHandle;
  end
  else
  if FProcessedSQL.Text <> FSQL.Text then begin
    FProcessedSQL.Text := FSQL.Text;
    FreeHandle
  end;
  if FPrepared then
    exit;
  if MonitorHook<>nil then   MonitorHook.SQLPrepare(Self);
  if (FProcessedSQL.Text = '') then
    FIBError(feEmptyQuery, [nil]);
  try
    Call(isc_dsql_alloc_statement2(StatusVector, DBHandle,
                                    @FHandle), True);
    Call(isc_dsql_prepare(StatusVector, TRHandle, @FHandle, 0,
               PChar(FProcessedSQL.Text), DataBase.SQlDialect, nil), True);
    (* After preparing the statement, query the stmt type and possibly
      create a FSQLRecord "holder" *)
    (* Get the type of the statement *)
    type_item := Char(isc_info_sql_stmt_type);
    Call(isc_dsql_sql_info(StatusVector, @FHandle, 2, @type_item,
                         SizeOf(res_buffer), res_buffer), True);
    if (res_buffer[0] <> Char(isc_info_sql_stmt_type)) then
      FIBError(feUnknownError, [nil]);
    stmt_len := isc_vax_integer(@res_buffer[1], 2);
    FSQLType := TFIBSQLTypes(isc_vax_integer(@res_buffer[3], stmt_len));
    (* Done getting the type *)
    case FSQLType of
      SQLGetSegment,
      SQLPutSegment,
      SQLStartTransaction: begin
        FreeHandle;
        FIBError(feNotPermitted, [nil]);
      end;
      SQLCommit,
      SQLRollback,
      SQLDDL, SQLSetGenerator,
      SQLInsert, SQLUpdate, SQLDelete, SQLSelect, SQLSelectForUpdate,
      SQLExecProcedure: begin
        (* We already know how many inputs there are, so... *)
        if (FSQLParams.FXSQLDA <> nil) and
           (Call(isc_dsql_describe_bind(StatusVector, @FHandle, DataBase.SQlDialect,
                                        FSQLParams.FXSQLDA), False) > 0) then
          IbError(Self);
        FSQLParams.Initialize;
        if FSQLType in [SQLSelect, SQLSelectForUpdate,
                        SQLExecProcedure] then begin
          (* Allocate an initial output descriptor (with one column) *)
          FSQLRecord.Count := 1;
          (* Using isc_dsql_describe, get the right size for the columns... *)
          Call(isc_dsql_describe(StatusVector, @FHandle, DataBase.SQlDialect, FSQLRecord.FXSQLDA), True);
          if FSQLRecord.FXSQLDA^.sqld > FSQLRecord.FXSQLDA^.sqln then begin
            FSQLRecord.Count := FSQLRecord.FXSQLDA^.sqld;
            Call(isc_dsql_describe(StatusVector, @FHandle, DataBase.SQlDialect, FSQLRecord.FXSQLDA), True);
          end else if FSQLRecord.FXSQLDA^.sqld = 0 then
            FSQLRecord.Count := 0;
          FSQLRecord.Initialize;
        end;
      end;
    end;
    {$IFDEF SUPPORT_ARRAY_FIELD}
    PrepareArrayFields;
    {$ENDIF}

    FPrepared := True;
  except
    on E: Exception do begin
    if MonitorHook<>nil then
      with THackMonitorHook(MonitorHook) do    //Added Source
      if MonitoringEnabled and (MonitorCount > 0) then begin
       if Owner is TFIBDataSet then   st:=Owner.Name else st:=Name;
        WriteSQLData(st + ': [Prepare] ' + E.Message);
      end;
      if (FHandle <> nil) then
        FreeHandle;
      raise;
    end;
  end;
end;

procedure TFIBQuery.SetSQL(Value: TStrings);
begin
  FSQL.Assign(Value);
end;

procedure TFIBQuery.SetTransaction(Value: TFIBTransaction);
begin
  FBase.Transaction := Value;
end;

procedure TFIBQuery.SQLChange(Sender: TObject); // Added source
begin
// For register Params
  if not ParamCheck  or (FSQL.Text='') then
    FProcessedSQL.Text := FSQL.Text
  else begin
    PreprocessSQL(FSQL.Text,true);
    PreprocessSQL(FSQL.Text,false);
  end;
end;

procedure TFIBQuery.SQLChanging(Sender: TObject);
begin
  CheckClosed;
  if Assigned(OnSQLChanging) then
    OnSQLChanging(Self);
  if FHandle <> nil then FreeHandle;
end;


procedure TFIBQuery.TransactionEnding(Sender: TObject);
begin
  if Transaction.State in [tsDoRollback,tsDoCommit] then
   if (FOpen) then    Close;
end;

//// Routine work
procedure TFIBQuery.SetParamValues(ParamValues: array of Variant);
var i :integer;
    pc:integer;
begin
// Exec Query with ParamValues
 if High(ParamValues)<Pred(Params.Count) then
  pc:=High(ParamValues)
 else
  pc:=Pred(Params.Count);
 for i:=Low(ParamValues)  to pc do
  Params[i].AsVariant:=ParamValues[i];
end;

procedure TFIBQuery.ExecWP(ParamValues: array of Variant);
begin
// Exec Query with ParamValues
 SetParamValues(ParamValues);
 ExecQuery
end;


function  TFIBQuery.ParamByName(const ParamName:string): TFIBXSQLVAR;
begin
   Result:=Params.ByName[ParamName];
   if (Result=nil) and IsProc then Result:=FieldByName[ParamName];
end;

function  TFIBQuery.IsProc :boolean;
begin
  Result:= Copy(Trim(UpperCase(SQL.Text)),1,8)=ExecProcPrefix
end;

end.
