(*
 * FIBDataSet.pas - (Implementation of DataSet descendants)
 *  copyright (c) 1998 by
 *    Gregory Deatz
 *    gdeatz@hlmdd.com
 *
 * Please see the file FIBLicense.txt for full license information.
 *)
// Changed by Serge Buzadzhy 04-12.1999

unit FIBDataSet;
                            
(*
 * Compiler defines
 *)

interface

{$I FIBPlus.INC}
uses
  Windows, SysUtils, Classes, ibase,IB_Intf, ib_externals, FIB,
  FIBDatabase, FIBQuery, Db, StdFuncs, StdConsts, FIBMiscellaneous
 {$IFNDEF FOR_CONSOLE}
  ,
  Forms,Controls // IS GUI units
 {$ENDIF} 
 ;


const
  BufferCacheSize    =  32;  // Allocate cache in this many record chunks
  UniCache           =  2;   // Uni-directional cache is 2 records big.

type
  TFIBCustomDataSet = class;
  TFIBDataSet = class;

  PDateTime = ^TDateTime;
  TBlobDataArray =array[0..0] of TFIBBlobStream;
  PBlobDataArray = ^TBlobDataArray;



  (*
   * Support types for TFIBCustomDataSet
   *)
  TFieldData = record
    fdDataType: Short;
    fdDataScale: Short;    
    fdNullable: Boolean;
    fdIsNull: Boolean;
    fdDataSize: Short;
    fdDataLength: Short;
    fdDataOfs: Integer;
  end;
  PFieldData = ^TFieldData;

  TCachedUpdateStatus = (
                         cusUnmodified, cusModified, cusInserted,
                         cusDeleted, cusUninserted
                        );
  TRecordData = record
    rdBookmarkFlag: TBookmarkFlag;
    rdFieldCount: Short;
    rdRecordNumber: Long;
    rdCachedUpdateStatus: TCachedUpdateStatus;
    rdUpdateStatus: TUpdateStatus;
    rdSavedOffset: DWORD;
    rdFields: array[1..1] of TFieldData;
  end;
  PRecordData = ^TRecordData;

  (*
   * TFIBStringField allows us to have strings longer strings than 8196
   *)
  TFIBStringField = class(TStringField)
  private
  public
    constructor Create(AOwner: TComponent); override;
    class procedure CheckTypeSize(Value: Integer); override;
    function GetAsString: string; override;
    function GetAsVariant: Variant; override;
    function GetValue(var Value: string): Boolean;
    procedure SetAsString(const Value: string); override;
  published
  end;
//

  TFIBIntegerField = class(TIntegerField)
   private
    FIsTimeField:Boolean;
   protected
     function  GetAsBoolean: Boolean; override;
     procedure SetAsBoolean(Value: Boolean); override;
     procedure GetText(var Text: string; DisplayText: Boolean); override;
     procedure SetText(const Value: string); override;
     function  GetHours:integer;
     function  GetMinutes:integer;
     function  GetSeconds:integer;
   public
     constructor Create(AOwner: TComponent); override;
     function IsValidChar(InputChar: Char): Boolean; override;
     property Hours:integer read GetHours;
     property Minutes:integer read GetMinutes;
     property Seconds:integer read GetSeconds;
   published
     property IsTimeField:Boolean  read FIsTimeField write FIsTimeField default false;
   end;

  TFIBSmallIntField =class(TSmallintField)
   protected
     function GetAsBoolean: Boolean; override;
     procedure SetAsBoolean(Value: Boolean); override;
   end;

  TFIBFloatField =class(TFloatField)
   private
    FRoundByScale:boolean;
    function GetScale:integer;
   protected
     procedure SetAsFloat(Value: Double); override;
   public
     constructor Create(AOwner: TComponent); override;
     property Scale:integer read GetScale;
   published
     property RoundByScale:boolean read FRoundByScale write FRoundByScale default true;
   end;

  TFIBBCDField = class(TBCDField)
  private
    FRoundByScale:boolean;
    function GetScale:integer;
    function  ServerType:integer;
  protected
    class procedure CheckTypeSize(Value: Integer); override;
    function  GetAsCurrency: Currency; override;
    function  GetAsString: string; override;
    function  GetAsVariant: Variant; override;
{$IFDEF VER130}
    function  GetDataSize: Integer; override;
{$ELSE}
    function  GetDataSize: Word; override;
{$ENDIF}
    procedure GetText(var Text: string; DisplayText: Boolean); override;
    function  GetValue(var Value: Currency): Boolean;
    procedure SetAsString(const Value: string); override;    
    procedure SetAsCurrency(Value: Currency); override;
    function  GetAsComp: comp; virtual;
    procedure SetAsComp(Value: comp); virtual;

  public
    constructor Create(AOwner: TComponent); override;
    property Scale:integer read GetScale;
    property AsComp: Comp read GetAsComp write SetAsComp;
  published
    property Size default 8;
    property RoundByScale:boolean read FRoundByScale write FRoundByScale;

  end;


  {$IFDEF SUPPORT_ARRAY_FIELD}
   TFIBArrayField=class(TBytesField)
   private
    FOldValueBuffer:PChar;
    function GetFIBXSQLVAR:TFIBXSQLVAR;
   protected
    procedure GetText(var Text: string; DisplayText: Boolean); override;
    function  GetDimCount:integer;
    function  GetElementType:TFieldType;
    function  GetDimension(Index:integer):TISC_ARRAY_BOUND;
    function  GetArraySize:integer;
    procedure SaveOldBuffer   ;
    procedure RestoreOldBuffer;
    function  GetArrayId:TISC_QUAD;
    function GetAsVariant: Variant; override;
    procedure SetAsVariant(const Value: Variant); override;
   public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property DimensionCount:integer read GetDimCount;
    property ElementType:TFieldType read GetElementType;
    property Dimension[Index: Integer]: TISC_ARRAY_BOUND read GetDimension;
    property ArraySize:Integer read GetArraySize;
    property ArrayID:TISC_QUAD read GetArrayId;
   end;
  {$ENDIF}




  (*
   * TFIBDataLink
   *)
  TFIBDataLink = class(TDataLink)
  protected
    FDataSet: TFIBCustomDataSet;
  protected
    procedure ActiveChanged; override;
    procedure RecordChanged(Field: TField); override;
    procedure CheckBrowseMode; override;
  public
    constructor Create(ADataSet: TFIBCustomDataSet);
    destructor Destroy; override;
  end;

  (*
   * TFIBCustomDataSet - declaration
   *)
  TFIBUpdateAction = (uaFail, uaAbort, uaSkip, uaRetry, uaApply, uaApplied);

  TFIBUpdateErrorEvent = procedure(DataSet: TDataSet; E: EFIBError;
    UpdateKind: TUpdateKind; var UpdateAction: TFIBUpdateAction) of object;
  TFIBUpdateRecordEvent = procedure(DataSet: TDataSet; UpdateKind: TUpdateKind;
    var UpdateAction: TFIBUpdateAction) of object;

  TFIBUpdateRecordTypes = set of TCachedUpdateStatus;
//buzz
  TOnFetchRecord  =procedure (FromQuery: TFIBQuery;RecordNumber:integer;
   var StopFetching:boolean
  ) of object;


  TDispositionFieldType= (dfNormal,dfRRecNumber,dfWRecNumber,dfRWRecNumber);  
  TExtLocateOption =(eloCaseInsensitive, eloPartialKey,eloWildCards,
   eloInSortedDS
  );
  TExtLocateOptions=set of TExtLocateOption;
  TpFIBDsOption=
   (poTrimCharFields,poAllowChangeSqls,poAppendMode,poRefreshAfterPost,
     poStartTransaction,poAutoFormatFields,poProtectedEdit
   );
  TpFIBDsOptions= set of TpFIBDsOption;


  TFIBCustomDataSet = class(TDataset)
  protected
    (*
     * Fields, and internal objects
     *)
    FBase: TFIBBase;          (* Manages database and transaction *)
    FBlobCacheOffset: Integer;
    FBlobStreamList: TList;
    FBufferChunks: Integer;
    FBufferCache,
    FOldBufferCache: PChar;
    FBufferChunkSize,
    FCacheSize,
    FOldCacheSize: Integer;
    FBPos,
    FOBPos,
    FBEnd,
    FOBEnd: DWord;
    FCachedUpdates: Boolean;
    FCalcFieldsOffset: Integer;
    FCurrentRecord: Long;
    FDeletedRecords: Long; (* How many records have been deleted? *)
    FSourceLink: TFIBDataLink;
    FModelBuffer,          (* Model buffer for initializing record buffers. *)
    FOldBuffer: PChar;     (* For storing edited records *)
    FOpen: Boolean;           (* Is the dataset open? *)
    FPrepared: Boolean;
    FQDelete,
    FQInsert,
    FQRefresh,
    FQSelect,
    FQUpdate: TFIBQuery;      (* Dataset management queries *)
    FRecordBufferSize: Integer;
    FRecordCount: Integer;
    FRecordSize: Integer;
    vDisableScrollCount:integer;
    (*
     * Event fields...
     *)
    FDatabaseDisconnecting,
    FDatabaseDisconnected,
    FDatabaseFree: TNotifyEvent;
    FOnUpdateError: TFIBUpdateErrorEvent;
    FOnUpdateRecord: TFIBUpdateRecordEvent;
    FTransactionEnding,
    FTransactionEnded,
    FTransactionFree: TNotifyEvent;
    FUpdatesPending: Boolean;
    FUpdateRecordTypes: TFIBUpdateRecordTypes;
    FUniDirectional: Boolean;
    FOnGetRecordError:  TDataSetErrorEvent;
    vInspectRecno:integer;
    vTypeDispositionField:TDispositionFieldType;
    vFilteredRecCounts:integer;
    FOptions:TpFIBDsOptions;     
  protected
    function GetXSQLVAR(Fld:TField):TXSQLVAR;
    function GetFieldScale(Fld:TNumericField):Short;

    (*
     * Routines for managing access to the database, etc... They have
     * nothing to do with TDataset.
     *)
    function AdjustCurrentRecord(Buffer: Pointer; GetMode: TGetMode): TGetResult;
    function CanEdit: Boolean; virtual;
    function CanInsert: Boolean; virtual;
    function CanDelete: Boolean; virtual;

    procedure CheckEditState;
    procedure CheckInsertMode(Buffer: Pointer); 
    procedure ClearMemoryCache;
    procedure UpdateBlobInfo(Buff: Pointer;IsPost:boolean);
    (*
     * When copying a given record buffer, should we overwrite
     * the pointers to "memory" or should we just copy the
     * contents?
     *)
    procedure CopyRecordBuffer(Source, Dest: Pointer);
    procedure DoDatabaseDisconnecting(Sender: TObject);
    procedure DoDatabaseDisconnected(Sender: TObject);
    procedure DoDatabaseFree(Sender: TObject);
    procedure DoTransactionEnding(Sender: TObject);
    procedure DoTransactionEnded(Sender: TObject);
    procedure DoTransactionFree(Sender: TObject);
    procedure FetchCurrentRecordToBuffer(Qry: TFIBQuery; RecordNumber: Integer;
      Buffer: PChar); 
    function GetActiveBuf: PChar;
    function GetDatabase: TFIBDatabase;
    function GetDBHandle: PISC_DB_HANDLE;
    function GetDeleteSQL: TStrings;
    function GetInsertSQL: TStrings;
    function GetParams: TFIBXSQLDA;
    function GetRefreshSQL: TStrings;
    function GetSelectSQL: TStrings;
    function GetStatementType: TFIBSQLTypes;
    function GetUpdateSQL: TStrings;
    function GetTransaction: TFIBTransaction;
    function GetTRHandle: PISC_TR_HANDLE;
    procedure InternalDeleteRecord(Qry: TFIBQuery; Buff: Pointer);virtual;// Added virtual
    function  InternalLocate(const KeyFields: string; const KeyValues: Variant;
      Options: TExtLocateOptions): Boolean; virtual;
    procedure InternalPostRecord(Qry: TFIBQuery; Buff: Pointer); virtual;// Added virtual
    procedure InternalRefreshRow(Qry: TFIBQuery; Buff:Pointer); // Added (Buff:Pointer);
    procedure InternalRevertRecord(RecordNumber: Integer);
    function  IsVisibleStat(Buffer: PChar): Boolean;
    function  IsVisible(Buffer: PChar): Boolean; virtual; // Added virtual
    procedure SaveOldBuffer(Buffer: PChar);
    procedure SetBufferChunks(Value: Integer);
    procedure SetDatabase(Value: TFIBDatabase);
    procedure SetDeleteSQL(Value: TStrings);
    procedure SetInsertSQL(Value: TStrings);
    procedure SetQueryParams(Qry: TFIBQuery; Buffer: Pointer);
    procedure SetRefreshSQL(Value: TStrings);
    procedure SetSelectSQL(Value: TStrings);
    procedure SetUpdateSQL(Value: TStrings);
    procedure SetTransaction(Value: TFIBTransaction);
    procedure SetUpdateRecordTypes(Value: TFIBUpdateRecordTypes);
    procedure SetUniDirectional(Value: Boolean);
    procedure SourceChanged;
    procedure SourceDisabled;
    procedure SQLChanging(Sender: TObject);
    function AdjustPosition(FCache: PChar; Offset: DWORD;
      Origin: Integer): Integer;
    procedure ReadCache(FCache: PChar; Offset: DWORD; Origin: Integer;
      Buffer: PChar);
    procedure WriteCache(FCache: PChar; Offset: DWORD; Origin: Integer;
      Buffer: PChar);
    procedure ReadRecordCache(RecordNumber: Integer; Buffer: PChar;
      ReadOldBuffer: Boolean);
    procedure WriteRecordCache(RecordNumber: Integer; Buffer: PChar);
    function GetNewBuffer:PChar;
    function GetOldBuffer:PChar;
    function   IsFiltered:boolean;dynamic;
  protected
    (*
     * Routines from TDataset that need to be overridden to use the IB API
     * directly.
     *)
    function AllocRecordBuffer: PChar; override; (* abstract *)
    procedure DoBeforeCancel; override;
    procedure DoBeforeDelete; override;
    procedure DoBeforeEdit; override;
    procedure DoBeforeInsert; override;
    procedure DoBeforeScroll; override;
    procedure DoAfterScroll; override;    
    procedure DoBeforePost; override;    
    procedure  DoAfterInsert;   override;    
    procedure DoOnPostError(DataSet: TDataSet; E: EDatabaseError; var Action: TDataAction); virtual;
    procedure FreeRecordBuffer(var Buffer: PChar); override; (* abstract *)
    procedure GetBookmarkData(Buffer: PChar; Data: Pointer); override; (* abstract *)
    function GetBookmarkFlag(Buffer: PChar): TBookmarkFlag; override; (* abstract *)
    function GetCanModify: Boolean; override;
    function GetDataSource: TDataSource; override;
    function GetFieldClass(FieldType: TFieldType): TFieldClass; override;
    function GetRecNo: Integer; override;
    function GetRecord(Buffer: PChar; GetMode: TGetMode;
      DoCheck: Boolean): TGetResult; override; (* abstract *)
    function GetRecordCount: Integer; override;
    function GetRecordSize: Word; override; (* abstract *)
    procedure InternalAddRecord(Buffer: Pointer; Append: Boolean); override; (* abstract *)
    procedure InternalCancel; override;
    procedure InternalClose; override; (* abstract *)
    procedure InternalDelete; override; (* abstract *)
    procedure InternalFirst; override; (* abstract *)
    procedure InternalGotoBookmark(Bookmark: Pointer); override; (* abstract *)
    procedure InternalHandleException; override; (* abstract *)
    procedure InternalInitFieldDefs; override; (* abstract *)
    procedure InternalInitRecord(Buffer: PChar); override; (* abstract *)
    procedure InternalLast; override; (* abstract *)
    procedure InternalOpen; override; (* abstract *)
    procedure InternalPost; override; (* abstract *)
    procedure InternalRefresh; override;
    procedure InternalSetToRecord(Buffer: PChar); override; (* abstract *)
    function IsCursorOpen: Boolean; override; (* abstract *)
    procedure SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag); override;
    procedure SetBookmarkData(Buffer: PChar; Data: Pointer); override; (* abstract *)
//WARNING!!! Only Delphi 4, after update pack 2!!!!!!!!
{$IFDEF VER120}
    procedure SetBlockReadSize(Value: Integer); override;
{$ENDIF}
    procedure SetCachedUpdates(Value: Boolean);
    procedure SetDataSource(Value: TDataSource);
    procedure SetFieldData(Field: TField; Buffer: Pointer); override; (* abstract *)
    procedure SetRecNo(Value: Integer); override;
  protected
//buzz
   FBeforeFetchRecord: TOnFetchRecord;
   FAfterFetchRecord : TOnFetchRecord;
   procedure SynchronizeBuffers(const ExecEvent,ToFibCach:boolean);   
   procedure DoOnSelectFetch(RecordNumber:integer;   var StopFetching:boolean);

    (*
     * Properties that are protected in TFIBCustomDataSet, but should be,
     * at some level, made visible. These are good candidates for
     * being made *public*.

     *)

    property Params: TFIBXSQLDA read GetParams;
    property Prepared: Boolean read FPrepared;
    property QDelete: TFIBQuery read FQDelete;
    property QInsert: TFIBQuery read FQInsert;
    property QRefresh: TFIBQuery read FQRefresh;
    property QSelect: TFIBQuery read FQSelect;
    property QUpdate: TFIBQuery read FQUpdate;
    property StatementType: TFIBSQLTypes read GetStatementType;
    property UpdatesPending: Boolean read FUpdatesPending;
    (*
     * Properties that are protected in TFIBCustomDataSet, but should be
     * (at some level). These should probably be published.
     *)
    property BufferChunks: Integer read FBufferChunks write SetBufferChunks;
    property CachedUpdates: Boolean read FCachedUpdates write SetCachedUpdates;
    property DeleteSQL: TStrings read GetDeleteSQL write SetDeleteSQL;
    property InsertSQL: TStrings read GetInsertSQL write SetInsertSQL;
    property RefreshSQL: TStrings read GetRefreshSQL write SetRefreshSQL;
    property SelectSQL: TStrings read GetSelectSQL write SetSelectSQL;
    property UniDirectional: Boolean read FUniDirectional write SetUniDirectional;
    property UpdateRecordTypes: TFIBUpdateRecordTypes read FUpdateRecordTypes
                                                      write SetUpdateRecordTypes;
    property UpdateSQL: TStrings read GetUpdateSQL write SetUpdateSQL;
    (* -- Events *)
    property DatabaseDisconnecting: TNotifyEvent read FDatabaseDisconnecting
                                                 write FDatabaseDisconnecting;
    property DatabaseDisconnected: TNotifyEvent read FDatabaseDisconnected
                                                write FDatabaseDisconnected;
    property DatabaseFree: TNotifyEvent read FDatabaseFree
                                        write FDatabaseFree;
    property OnUpdateError: TFIBUpdateErrorEvent read FOnUpdateError
                                                 write FOnUpdateError;
    property OnUpdateRecord: TFIBUpdateRecordEvent read FOnUpdateRecord
                                                   write FOnUpdateRecord;
    property TransactionEnding: TNotifyEvent read FTransactionEnding
                                             write FTransactionEnding;
    property TransactionEnded: TNotifyEvent read FTransactionEnded
                                            write FTransactionEnded;
    property TransactionFree: TNotifyEvent read FTransactionFree
                                           write FTransactionFree;

  public

    (* public declarations *)
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ApplyUpdates;
    procedure BatchInput(InputObject: TFIBBatchInputStream);
    procedure BatchOutput(OutputObject: TFIBBatchOutputStream);
    function CachedUpdateStatus: TCachedUpdateStatus;
    procedure CancelUpdates;
    procedure CheckDatasetClosed;
    procedure CheckDatasetOpen;
    procedure CheckNotUniDirectional;
    procedure FetchAll;
    procedure Prepare; virtual;//Added virtual
    procedure RecordModified(Value: Boolean);
    procedure RevertRecord;
    procedure Undelete;
    procedure DisableScrollEvents;
    procedure EnableScrollEvents;
{$IFDEF SUPPORT_ARRAY_FIELD}
    function  ArrayFieldValue(Field:TField):Variant;
    procedure SetArrayValue(Field:TField;Value:Variant);
    function  GetElementFromValue( Field:TField;
               Indexes:array of integer):Variant;

    procedure SetArrayElementValue(Field:TField;Value:Variant;
     Indexes:array of integer
    );
{$ENDIF}
    procedure SwapRecords(Recno1,Recno2:integer);
    procedure MoveRecord(OldRecno,NewRecno:integer);    
  public
    (* public routines overridden from TDataSet *)
    function CompareBookmarks(Bookmark1, Bookmark2: TBookmark): Integer; override;
    function CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream; override;
    function GetCurrentRecord(Buffer: PChar): Boolean; override;
{$IFNDEF VER100 }
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; overload; override;
    function GetFieldData(FieldNo: Integer; Buffer: Pointer): Boolean; overload; override;
{$ELSE}
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; override; (* abstract *)
{$ENDIF}
    function  RecordFieldValue(Field:TField;RecNumber:integer):Variant;
    function Locate(const KeyFields: String; const KeyValues: Variant;
      Options: TLocateOptions): Boolean; override;
    function ExtLocate(const KeyFields: String; const KeyValues: Variant;
      Options: TExtLocateOptions): Boolean;

    function LocateNext(const KeyFields: String; const KeyValues: Variant;
      Options: TLocateOptions): Boolean; // Sister function to Locate
    function ExtLocateNext(const KeyFields: String; const KeyValues: Variant;
      Options: TExtLocateOptions): Boolean; // Sister function to ExtLocate


    function Lookup(const KeyFields: string; const KeyValues: Variant;
      const ResultFields: string): Variant; override;
{$IFDEF VER100 }
    procedure Translate(Src, Dest: PChar; ToOem: Boolean); override;
{$ELSE}
    function Translate(Src, Dest: PChar; ToOem: Boolean): Integer; override;
{$ENDIF}
    function UpdateStatus: TUpdateStatus; {$ifNdef VER100} override; {$endif}
    function IsSequenced: Boolean; override;        // Scroll bar

  public
    (* Public properties *)
    property DBHandle: PISC_DB_HANDLE read GetDBHandle;
    property TRHandle: PISC_TR_HANDLE read GetTRHandle;
  published
    (*
     * Published properties implemented in TFIBCustomDataSet
     *)
    (* -- Properties *)
    property Database: TFIBDatabase read GetDatabase write SetDatabase;
    property Transaction: TFIBTransaction read GetTransaction

                                          write SetTransaction;
    property BeforeFetchRecord: TOnFetchRecord read FBeforeFetchRecord
     write FBeforeFetchRecord;
    property AfterFetchRecord : TOnFetchRecord read FAfterFetchRecord
     write FAfterFetchRecord;
    property OnGetRecordError:TDataSetErrorEvent
     read FOnGetRecordError write FOnGetRecordError;
    property  Options:TpFIBDsOptions read FOptions write FOptions
           default
           [poTrimCharFields,poAllowChangeSqls,poStartTransaction,poAutoFormatFields,
            poRefreshAfterPost
           ];

  end;

  TFIBDataSet = class(TFIBCustomDataSet)
  public
    (*
     * Properties made public out of TFIBCustomDataSet
     *)
    property Params;
    property Prepared;
    property QDelete;
    property QInsert;
    property QRefresh;
    property QSelect;
    property QUpdate;
    property StatementType;
    property UpdatesPending;
  public
    (*
     * Public properties derived from TDataset
     *   NOTE: properties Bof and Eof *must* be spelled as is in order
     *         for them to work with C++-Builder. Delphi doesn't care
     *         about case, so I would suggest that you leave the spellings
     *         as is if you anticipate using FreeIBComponents with
     *         C++-Builder.
     *)
    property Bof;
    property Bookmark;
    property DefaultFields;
    property Designer;
    property Eof;
    property FieldCount;
    property FieldDefs;
    property Fields;
    property FieldValues;
    property Found;
    property Modified;
    property RecordCount;
    property State;
  published
    (*
     * Published out of TFIBCustomDataSet
     *)
    (* -- Properties *)
    property BufferChunks;
    property CachedUpdates;
    property DeleteSQL;
    property InsertSQL;
    property RefreshSQL;
    property SelectSQL;
    property UniDirectional;
    property UpdateRecordTypes;
    property UpdateSQL;
    (* -- Events *)
    property DatabaseDisconnecting;
    property DatabaseDisconnected;
    property DatabaseFree;
    property OnUpdateError;
    property OnUpdateRecord;
    property TransactionEnding;
    property TransactionEnded;
    property TransactionFree;
  published
    (*
     * Published out of TDataset
     *)
    (* -- Properties *)
    property Active;
    property AutoCalcFields;
    property DataSource read GetDataSource write SetDataSource;
    (* -- Events *)
    property AfterCancel;
    property AfterClose;
    property AfterDelete;
    property AfterEdit;
    property AfterInsert;
    property AfterOpen;
    property AfterPost;
    property AfterScroll;
    property BeforeCancel;
    property BeforeClose;
    property BeforeDelete;
    property BeforeEdit;
    property BeforeInsert;
    property BeforeOpen;
    property BeforePost;
    property BeforeScroll;
    property OnCalcFields;
    property OnDeleteError;
    property OnEditError;
    property OnNewRecord;
    property OnPostError;
    {$IFDEF VER130}
    property BeforeRefresh;
    property AfterRefresh;
    property ObjectView;
    {$ENDIF}
  end;

  (* TFIBDSBlobStream *)
  TFIBDSBlobStream = class(TStream)
  protected
    FField: TField;
    FBlobStream: TFIBBlobStream;
  public
    constructor Create(AField: TField; ABlobStream: TFIBBlobStream;
      Mode: TBlobStreamMode);
    function Read(var Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    procedure SetSize(NewSize: Longint); override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;


(*
 * Support routines
 *)
function RecordDataLength(n: Integer): Long;

type
  TFIBFilterType = (ftByField, ftCopy);

procedure FilterIn(FromDS, ToDS: TFIBCustomDataSet; How: TFIBFilterType;
  FromFieldList, ToFieldList: array of Integer;
  RefreshTo: Boolean);
procedure FilterOut(FromDS: TFIBCustomDataSet);
(* Clear the entire record cache, and do everything short of
   closing the data set--but don't delete anything, etc.. *)
procedure ClearRecordCache(FromDS: TFIBCustomDataSet);
(*
 * Sort the data set
 *)

procedure Sort(DataSet: TFIBCustomDataSet; Fields: array of const;
  Ordering: array of Boolean);



(*
 * More constants
 *)
const
  DefaultFieldClasses: array[ftUnknown..ftTypedBinary] of TFieldClass = (
    nil,                (* ftUnknown *)
    TFIBStringField,    (* ftString *)
    TFIBSmallIntField,     (* ftSmallint *)
    TFIBIntegerField,      (* ftInteger *)
    TWordField,         (* ftWord *)
    TBooleanField,      (* ftBoolean *)
    TFIBFloatField,     (* ftFloat *)
    TCurrencyField,     (* ftCurrency *)
    TFIBBCDField,          (* ftBCD *)
    TDateField,         (* ftDate *)
    TTimeField,         (* ftTime *)
    TDateTimeField,     (* ftDateTime *)
  {$IFDEF SUPPORT_ARRAY_FIELD}
    TFIBArrayField,
  {$ELSE}
    TBytesField,        (* ftBytes *)
  {$ENDIF}
    TVarBytesField,     (* ftVarBytes *)
    TAutoIncField,      (* ftAutoInc *)
    TBlobField,         (* ftBlob *)
    TMemoField,         (* ftMemo *)
    TGraphicField,      (* ftGraphic *)
    TBlobField,         (* ftFmtMemo *)
    TBlobField,         (* ftParadoxOle *)
    TBlobField,         (* ftDBaseOle *)
    TBlobField          (* ftTypedBinary *)
    );

{$IFNDEF FOR_CONSOLE}
var FIBHourglassCursor:TCursor = crHourglass;
{$ENDIF}

procedure Register;

implementation
uses StrUtil,FIBConsts,pFIBDataSet;

procedure Register;
begin
  RegisterClass(TFIBStringField);
  RegisterClass(TFIBIntegerField);
  RegisterClass(TFIBSmallIntField);
  RegisterClass(TFIBFloatField);
  RegisterClass(TFIBBCDField);
  {$IFDEF SUPPORT_ARRAY_FIELD}
   RegisterClass(TFIBArrayField);
  {$ENDIF} 
end;

(*
 * TFIBStringField - implementation
 *)
constructor TFIBStringField.Create(AOwner: TComponent);
begin
  inherited;
end;

class procedure TFIBStringField.CheckTypeSize(Value: Integer);
begin
  (*
   * Just don't check. Any string size is valid.
   *)
end;

function TFIBStringField.GetAsString: string;
begin
  if not GetValue(Result) then Result := '';
end;

function TFIBStringField.GetAsVariant: Variant;
var
  S: string;
begin
  if GetValue(S) then Result := S else Result := Null;
end;

function TFIBStringField.GetValue(var Value: string): Boolean;
var
  Buffer: PChar;
begin
  Buffer := nil;
  FIBAlloc(Buffer, 0, Size + 1);
  try
    Result := GetData(Buffer);
    if Result then begin
      Value := String(Buffer);
      if Transliterate and (Value <> '') then
        DataSet.Translate(PChar(Value), PChar(Value), False);
    end
  finally
    FIBAlloc(Buffer, 0, 0);
  end;
end;

procedure TFIBStringField.SetAsString(const Value: string);
var
  Buffer: PChar;
begin
  Buffer := nil;
  FIBAlloc(Buffer, 0, Size + 1);
  try
    StrLCopy(Buffer, PChar(Value), Size);
    if Transliterate then
      DataSet.Translate(Buffer, Buffer, True);
    SetData(Buffer);
  finally
    FIBAlloc(Buffer, 0, 0);
  end;
end;

(*
 * TFIBIntegerField - implementation
 *)

 constructor TFIBIntegerField.Create(AOwner: TComponent); //override;
 begin
  inherited Create(AOwner);
  FIsTimeField:=false
 end;

 function TFIBIntegerField.GetAsBoolean: Boolean; //override;
 begin
   Result:=AsInteger>0
 end;

 procedure TFIBIntegerField.SetAsBoolean(Value: Boolean); //override;
 begin
  if Value then  AsInteger:=1 else AsInteger:=0
 end;

 function TFIBIntegerField.IsValidChar(InputChar: Char): Boolean; //override;
 begin
  Result:=inherited IsValidChar(InputChar);
  if not Result and FIsTimeField then Result:=InputChar=TimeSeparator
 end;

 procedure TFIBIntegerField.GetText(var Text: string; DisplayText: Boolean); //override;
 var MSec,hours,minutes:integer;
 begin
  if (FIsTimeField)and (GetAsString<>'') then begin
   MSec    :=StrToInt(GetAsString);
   minutes :=MSec div 60;
   hours   :=minutes div 60;
   minutes :=minutes-(hours*60);
   if hours<10 then Text:='0'+IntToStr(hours) else Text:=IntToStr(hours);
   Text:=Text+ TimeSeparator;
   if minutes<10 then Text:=Text+'0';
   Text:=Text+IntToStr(minutes);
//   Result:=hours+0.01*minutes;
  end
  else
  inherited GetText(Text,DisplayText);
 end;

 procedure DecodeTimeStr(TimeStr:string ;var Hours, Minutes, Sec, MSec:Word);
 var PosEnd,i:integer;
     StrValue:string;
 begin
  Hours:=0; Minutes:=0; Sec:=0; MSec:=0;i:=0;
  repeat
   PosEnd:=Pos(TimeSeparator,TimeStr);
   if PosEnd<>0 then StrValue:=Copy(TimeStr,1,PosEnd-1)
   else StrValue:= TimeStr;
   if StrValue<>'' then
    case i of
     0:Hours  :=StrToInt(StrValue);
     1:Minutes:=StrToInt(StrValue);
     2:Sec    :=StrToInt(StrValue);
     3:MSec   :=StrToInt(StrValue);
    end;
    Inc(i);
    TimeStr:=Copy(TimeStr,PosEnd+1,255);
  until (PosEnd=0) or (i=4);
 end;

 procedure TFIBIntegerField.SetText(const Value: string); //override;
 var MSec,hours,minutes,Sec:Word;
     aValue:integer;
 begin
  if (FIsTimeField)and (Value<>'') then begin
//   T    :=StrToTime(Value);
   DecodeTimeStr(Value, Hours, Minutes, Sec, MSec);
   aValue:=(Hours*60*60+Minutes*60+Sec)+MSec*10;
   SetAsInteger(aValue)
  end
  else
  inherited SetText(Value);
 end;

 function  TFIBIntegerField.GetHours:integer;
 begin
  Result :=(AsInteger div 60) div 60;
 end;

 function  TFIBIntegerField.GetMinutes:integer;
 begin
   Result :=(AsInteger div 60)-Hours*60;
 end;

 function  TFIBIntegerField.GetSeconds:integer;
 begin
   Result :=AsInteger-Hours*60-Minutes*60;
 end;

(*
 * TFIBSmallIntField - implementation
 *)


 function TFIBSmallIntField.GetAsBoolean: Boolean; //override;
 begin
   Result:=AsInteger>0
 end;

 procedure TFIBSmallIntField.SetAsBoolean(Value: Boolean); //override;
 begin
  if Value then  AsInteger:=1 else AsInteger:=0
 end;


constructor TFIBFloatField.Create(AOwner: TComponent); //override;
begin
 inherited Create(AOwner);
 FRoundByScale:=true;
// FScale:=TFIBCustomDataSet(DataSet).GetFieldScale(Self);
end;

function TFIBFloatField.GetScale:integer;
begin
 if (FieldKind=fkData) then
  result:=TFIBCustomDataSet(DataSet).GetFieldScale(Self)
 else
  result:=-15;
end;

function RoundDbl(Value: Double;Decimals:integer): Double;
var 
    i:integer;
    st:double;
begin
 st:=1;
 for i:=1 to Decimals do st:=10*st;
 Result:=System.Int(Value)+Round(Frac(Value)*st)/st;
end;

procedure TFIBFloatField.SetAsFloat(Value: Double); //override;
begin
 if FRoundByScale  and (Scale<>0) then
  inherited
   SetAsFloat(RoundDbl(Value,-Scale))
 else
  inherited SetAsFloat(Value)
end;
//Array support
{$IFDEF SUPPORT_ARRAY_FIELD}

 constructor TFIBArrayField.Create(AOwner: TComponent); //override;
 begin
  inherited Create(AOwner);
  FOldValueBuffer:=nil;
 end;

 destructor TFIBArrayField.Destroy; //override;
 begin
  FIBAlloc(FOldValueBuffer, 0, 0);
  inherited Destroy;
 end;

 procedure TFIBArrayField.GetText(var Text: string; DisplayText: Boolean);
 begin
  if IsNull then
   Text:='(Array)'
  else
   Text:='(ARRAY)'
 end;
 
 function TFIBArrayField.GetFIBXSQLVAR:TFIBXSQLVAR;
 begin
   if DataSet=nil then Result:=nil
   else
   with TFIBDataSet(DataSet).QSelect do begin
    if not Prepared then Prepare;
     Result:=FieldByName[FieldName]
   end;
 end;

 function TFIBArrayField.GetDimCount:integer;
 begin
    if GetFIBXSQLVAR=nil then
     Result:=0
    else
     Result:=GetFIBXSQLVAR.DimensionCount
 end;

 function TFIBArrayField.GetElementType:TFieldType;
 begin
    if GetFIBXSQLVAR=nil then
     Result:=ftUnknown
    else
     Result:=GetFIBXSQLVAR.ElementType
 end;

 function TFIBArrayField.GetDimension(Index:integer):TISC_ARRAY_BOUND;
 begin
    if GetFIBXSQLVAR=nil then
     FibError(feInvalidColumnIndex,[nil])
    else
     Result:=GetFIBXSQLVAR.Dimension[Index]
 end;

 function TFIBArrayField.GetArraySize:integer;
 begin
    if GetFIBXSQLVAR=nil then
     Result:=0
    else
     Result:=GetFIBXSQLVAR.ArraySize
 end;

function  TFIBArrayField.GetArrayId:TISC_QUAD;
 var OldValueBuffer:PChar;
begin
  FIBAlloc(OldValueBuffer,0,ArraySize);
  try
   GetData(OldValueBuffer);
   Result:=PISC_QUAD(OldValueBuffer)^;
  finally
   FIBAlloc(OldValueBuffer,0,0)
  end;
end;



function TFIBArrayField.GetAsVariant:Variant;
begin
 Result:=0;
 if DataSet <>nil then
  Result:=TFIBDataSet(DataSet).ArrayFieldValue(Self)
end;

procedure TFIBArrayField.SetAsVariant(const Value: Variant); //override;
begin
 if DataSet <>nil then
  TFIBDataSet(DataSet).SetArrayValue(Self,Value)
end;

procedure  TFIBArrayField.SaveOldBuffer;
begin
  FIBAlloc(FOldValueBuffer,0,ArraySize);
  GetData(FOldValueBuffer);
end;

procedure TFIBArrayField.RestoreOldBuffer;
begin
  SetData(FOldValueBuffer);
end;

{$ENDIF}


{ TFIBBCDField }

constructor TFIBBCDField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetDataType(ftBCD);
  Size := 8;
  FRoundByScale:=true;
end;

class procedure TFIBBCDField.CheckTypeSize(Value: Integer);
begin
{ No need to check as the base type is currency, not BCD }
end;

function TFIBBCDField.GetScale:integer;
begin
 if (FieldKind=fkData) then
  result:=TFIBCustomDataSet(DataSet).GetFieldScale(Self)
 else
  result:=-15;
end;

function TFIBBCDField.GetAsCurrency: Currency;
begin
  if not GetValue(Result) then
    Result := 0;
end;

function TFIBBCDField.GetAsString: string;
var
  C: System.Currency;
  TC: comp;
begin
  if Scale = 0 then 
  begin
    if GetData(@TC) then
      Result := CompToStr(TC)
    else
      Result := '';
  end
  else
  begin
   if GetValue(C) then
    Result := CurrToStr(C)
   else
    Result := '';
  end
end;


function TFIBBCDField.GetAsVariant: Variant;
var
  C: System.Currency;
begin
  if GetValue(C) then
    Result := C
  else
    Result := Null;
end;

{$IFDEF VER130}
function TFIBBCDField.GetDataSize: Integer;
{$ELSE}
function TFIBBCDField.GetDataSize: Word;
{$ENDIF}
begin
  Result := 8;
end;


function  TFIBBCDField.ServerType:integer;
var F:TFIBXSQLVAR;
begin
 Result:=SQL_DOUBLE;
 with TFibDataSet(DataSet).QSelect do begin
  F:=FieldByName[FieldName];
  if F<>nil then Result:=F.Data^.sqltype and (not 1);
 end;
end;

type THackDataSet= class(TDataSet);


procedure TFIBBCDField.GetText(var Text: string; DisplayText: Boolean);
var
  Format: TFloatFormat;
  FmtStr: string;
  Digits: Integer;
  C: System.Currency;
  TC: comp; //patchInt64B
  {$IFNDEF VER130}
   BCD: array[0..255] of Byte;
   Success:boolean;
  {$ENDIF}
begin
  if Scale = 0 then //patchInt64B start
  begin
    if GetData(@TC) then
      Text := CompToStr(TC)
    else
      Text := '';
  end
  else begin
    {$IFDEF VER130}
    if GetData(@C) then
    {$ELSE}
    Success:=false;
    if ServerType=SQL_INT64 then
      Success:=GetData(@C)
    else
    if GetData(@BCD) then
       Success:=THackDataSet(DataSet).BCDToCurr(@BCD, C);
    if  Success then
    {$ENDIF}
    begin
      if DisplayText or (EditFormat = '') then
        FmtStr := DisplayFormat else
        FmtStr := EditFormat;
      if FmtStr = '' then
      begin
        if currency then
        begin
          if DisplayText then
            Format := ffCurrency
          else
            Format := ffFixed;
          Digits := CurrencyDecimals;
        end
        else begin
          Format := ffGeneral;
          Digits := 0;
        end;
        Text := CurrToStrF(C, Format, Digits);
      end
      else
        Text := FormatCurr(FmtStr, C);
    end
    else
      Text := '';
  end;
end;

function TFIBBCDField.GetValue(var Value: Currency): Boolean;
var
  TC: comp; 
begin
  if Scale = 0 then //patchInt64B start
  begin
    Result := GetData(@TC);
    Value := TC;
  end
  else
  begin //patchInt64B end

{$IFDEF VER130}
  Result := GetData(@Value);
{$ELSE}
  if ServerType=SQL_INT64 then
    Result := GetData(@Value)
  else
    Result :=inherited GetValue(Value);
{$ENDIF}
  end
end;

procedure TFIBBCDField.SetAsString(const Value: string); //override;
begin
 if Value = '' then Clear else
 if (ServerType=SQL_INT64)  then
 begin
  if (Scale=0) then   SetAsComp(StrToComp(Value))
  else
   SetAsCurrency(StrToCurr(Value))
 end
 else
  inherited SetAsString(Value)
end;

procedure TFIBBCDField.SetAsCurrency(Value: Currency);
var
  TC: comp;
{$IFNDEF VER130}
  BCD: array[0..255] of Byte;
{$ENDIF}
begin

  if Scale = 0 then //patchInt64B start
  begin
    TC := Value;
    SetData(@TC);
  end
  else begin
   if (MinValue <> 0) or (MaxValue <> 0) then
   begin
    Value:= RoundDbl(Value,-Scale);
    if (Value < MinValue) or (Value > MaxValue) then
      RangeError(Value, MinValue, MaxValue);
   end;

  {$IFDEF VER130}
    SetData(@Value);
  {$ELSE}
   if ServerType=SQL_INT64 then
    SetData(@Value)
   else begin
    THackDataSet(DataSet).CurrToBCD(Value, @BCD, 0,-Scale);
    SetData(@BCD)
   end;
  {$ENDIF}
  end;
end;

function TFIBBCDField.GetAsComp: comp; 
begin
  if Scale = 0 then
  begin
    if not GetData(@Result) then
      Result := 0;
  end
  else
    Result := AsCurrency;
end;

procedure TFIBBCDField.SetAsComp(Value: comp);
begin
  if Scale = 0 then
    SetData(@Value)
  else
    AsCurrency := Value;
end;


(*
 * TFIBDataLink - implementation
 *)
constructor TFIBDataLink.Create(ADataSet: TFIBCustomDataSet);
begin
  inherited Create;
  FDataSet := ADataSet;
end;

destructor TFIBDataLink.Destroy;
begin
  FDataSet.FSourceLink := nil;
  inherited;
end;

procedure TFIBDataLink.ActiveChanged;
begin
  if Active then
    FDataSet.SourceChanged
  else if not Active then
    FDataSet.SourceDisabled;
end;

procedure TFIBDataLink.CheckBrowseMode;
begin
  if FDataSet.Active then FDataSet.CheckBrowseMode;
end;

procedure TFIBDataLink.RecordChanged(Field: TField);
begin
  if (Field = nil) and FDataSet.Active then
    FDataSet.SourceChanged;
end;

(*
 * TFIBCustomDataSet - implementation
 *)

constructor TFIBCustomDataSet.Create(AOwner: TComponent);
begin
  inherited;
  FBase := TFIBBase.Create(Self);
  FCurrentRecord := -1;
  FDeletedRecords := 0;
  FBufferChunks := BufferCacheSize;

  FBlobStreamList := TList.Create;
  FSourceLink := TFIBDataLink.Create(Self);
  FQDelete := TFIBQuery.Create(Self);
  FQDelete.OnSQLChanging := SQLChanging;
  FQDelete.GoToFirstRecordOnExecute := False;
  FQDelete.Name:='DeleteQuery';
  FQInsert := TFIBQuery.Create(Self);
  FQInsert.OnSQLChanging := SQLChanging;
  FQInsert.GoToFirstRecordOnExecute := False;
  FQInsert.Name:='InsertQuery';
  FQRefresh := TFIBQuery.Create(Self);
  FQRefresh.OnSQLChanging := SQLChanging;
  FQRefresh.GoToFirstRecordOnExecute := False;
  FQRefresh.Name:='RefreshQuery';  
  FQSelect := TFIBQuery.Create(Self);
  FQSelect.OnSQLChanging := SQLChanging;
  FQSelect.OnSQLFetch    :=DoOnSelectFetch;

  FQSelect.GoToFirstRecordOnExecute := False;
  FQSelect.Name:='SelectQuery';  
  FQUpdate := TFIBQuery.Create(Self);
  FQUpdate.OnSQLChanging := SQLChanging;
  FQUpdate.GoToFirstRecordOnExecute := False;
  FQUpdate.Name:='UpdateQuery';
  FUpdateRecordTypes := [cusUnmodified, cusModified, cusInserted];
  vDisableScrollCount:=0;  
  (*
   * Bookmark size -
   *  A bookmark is just an integer as far as I'm concerned, so set the
   *  bookmark size to SizeOf(Integer)
   *)
  BookmarkSize := SizeOf(Integer);
  (* Events... *)
  FBase.OnDatabaseDisconnecting := DoDatabaseDisconnecting;
  FBase.OnDatabaseDisconnected := DoDatabaseDisconnected;
  FBase.OnDatabaseFree := DoDatabaseFree;
  FBase.OnTransactionEnding := DoTransactionEnding;
  FBase.OnTransactionEnded := DoTransactionEnded;
  FBase.OnTransactionFree := DoTransactionFree;

  FOptions:=[poTrimCharFields,poAllowChangeSqls,poStartTransaction,poAutoFormatFields
  ,  poRefreshAfterPost
  ];
  vFilteredRecCounts:=0;
  vTypeDispositionField:=dfNormal;

end;

destructor TFIBCustomDataSet.Destroy;
begin
  inherited;
  FSourceLink.Free;
  FBase.Free;
  ClearMemoryCache;
  FBlobStreamList.Free;
  FIBAlloc(FBufferCache, 0, 0);
  FIBAlloc(FOldBufferCache, 0, 0);
  FCacheSize := 0;
  FOldCacheSize := 0;
end;

function TFIBCustomDataSet.GetXSQLVAR(Fld:TField):TXSQLVAR;
var fPos:integer;
    curName:string;
begin
  if (Fld=nil) or (Fld.FieldKind <>fkData) then Exit;
  fPos:=0;
  with QSelect do
   while fPos<Pred(Current.Count) do
   with Current[fPos].Data^ do
   begin
    SetString(curName, aliasname, aliasname_length);
    if Fld.FieldName=curName then Break;
    Inc(fPos)
   end;
  if fPos<>QSelect.Current.Count then   Result:=QSelect.Current[fPos].Data^
  else Result.sqltype:=-1
end;

function  TFIBCustomDataSet.GetFieldScale(Fld:TNumericField):Short;
begin
 Result:=100;
 with GetXSQLVAR(Fld) do
   if sqltype<>-1 then Result:=sqlscale
end;

(*
 * Routines for managing access to the database, etc... They have
 * nothing to do with TDataset. (All TDataset overrides are listed
 * later)
 * I did my best to put everything in alphabetical order, but don't
 * get angry is something got out of place...
 *)

function TFIBCustomDataSet.AdjustCurrentRecord(Buffer: Pointer;
  GetMode: TGetMode): TGetResult;
begin
  (*
   * Skip over all invisible records.
   *)
  while not IsVisible(Buffer) do begin
    if GetMode = gmPrior then begin
      Dec(FCurrentRecord);
      if FCurrentRecord = -1 then begin
        result := grBOF;
        exit;
      end;
      ReadRecordCache(FCurrentRecord, Buffer, False);
    end else begin
      Inc(FCurrentRecord);
      if (FCurrentRecord = FRecordCount) then begin
        if (not FQSelect.EOF) and (FQSelect.Next <> nil) then begin
          FetchCurrentRecordToBuffer(FQSelect, FCurrentRecord, Buffer);
          Inc(FRecordCount);
        end else begin
          result := grEOF;
          exit;
        end;
      end else
        ReadRecordCache(FCurrentRecord, Buffer, False);
    end;
  end;
  result := grOK;
end;

procedure TFIBCustomDataSet.ApplyUpdates;
var
  CurBookmark: String;
  Buffer: PRecordData;
  CurUpdateTypes: TFIBUpdateRecordTypes;
  UpdateAction: TFIBUpdateAction;
  UpdateKind: TUpdateKind;
  bRecordsSkipped: Boolean;
begin
  if State in [dsEdit, dsInsert] then
    Post;
  FBase.CheckDatabase;
  FBase.CheckTransaction;
  DisableControls;
  CurBookmark := Bookmark;
  CurUpdateTypes := FUpdateRecordTypes;
  FUpdateRecordTypes := [cusModified, cusInserted, cusDeleted];
  try
    First;
    bRecordsSkipped := False;
    while not EOF do begin
      Buffer := PRecordData(GetActiveBuf);
      case Buffer^.rdCachedUpdateStatus of
        cusModified:
          UpdateKind := ukModify;
        cusInserted:
          UpdateKind := ukInsert;
        else
          UpdateKind := ukDelete;
      end;

      (*
       * Do the OnUpdateRecord stuff, if necessary
       *)
      if (Assigned(FOnUpdateRecord)) then begin
        UpdateAction := uaFail;
        FOnUpdateRecord(Self, UpdateKind, UpdateAction);
      end else
        UpdateAction := uaApply;
      case UpdateAction of
        uaFail:
          FIBError(feUserAbort, [nil]);
        uaAbort:
          SysUtils.Abort;
        uaApplied: begin
          Buffer^.rdCachedUpdateStatus := cusUnmodified;
          Buffer^.rdUpdateStatus       := usUnmodified;
          WriteRecordCache(Buffer^.rdRecordNumber, PChar(Buffer));
        end;
        uaSkip:
          bRecordsSkipped := True;
      end;

      (*
       * Now, do the update, if the user allows it
       *)
      while (UpdateAction in [uaApply, uaRetry]) do begin
        try
          case Buffer^.rdCachedUpdateStatus of
            cusModified:
              InternalPostRecord(FQUpdate, Buffer);
            cusInserted:
              InternalPostRecord(FQInsert, Buffer);
            cusDeleted:
              InternalDeleteRecord(FQDelete, Buffer);
          end;
          UpdateAction := uaApplied;
        except
          (*
           * If there is an exception, then allow the user
           * to intervene (decide what to do about it).
           *)
          on E: EFIBError do begin
            UpdateAction := uaFail;
            if Assigned(FOnUpdateError) then
              FOnUpdateError(Self, E, UpdateKind, UpdateAction);
            case UpdateAction of
              uaFail: raise;
              uaAbort: raise EAbort(E.Message);
              uaSkip: bRecordsSkipped := True;
            end;
          end;
        end;
      end;
      Next;
    end;
    FUpdatesPending := bRecordsSkipped;
  finally
    FUpdateRecordTypes := CurUpdateTypes;
    Bookmark := CurBookmark;
    EnableControls;
  end;
end;

procedure TFIBCustomDataSet.BatchInput(InputObject: TFIBBatchInputStream);
begin
  FQSelect.BatchInput(InputObject);
end;

procedure TFIBCustomDataSet.BatchOutput(OutputObject: TFIBBatchOutputStream);
var
  Qry: TFIBQuery;
  i  : integer;
begin
  Qry := TFIBQuery.Create(Self);
  try
    Qry.Database := FBase.Database;
    Qry.Transaction := FBase.Transaction;
    Qry.SQL.Assign(FQSelect.SQL);
    Qry.Prepare;
    for i:=0 to Pred(Params.Count) do begin
     Qry.Params[i].Assign(Params[i])
    end;
    Qry.BatchOutput(OutputObject);
  finally
    Qry.Free;
  end;
end;

procedure TFIBCustomDataSet.CancelUpdates;
var
  CurUpdateTypes: TFIBUpdateRecordTypes;
begin
  if State in [dsEdit, dsInsert] then
    Post;
  if FCachedUpdates and FUpdatesPending then begin
    DisableControls;
    CurUpdateTypes := UpdateRecordTypes;
    UpdateRecordTypes := [cusModified, cusInserted, cusDeleted];
    try
      First;
      while not EOF do begin
        RevertRecord;
        Next;
      end;
    finally
      UpdateRecordTypes := CurUpdateTypes;
      First;
      FUpdatesPending := False;
      EnableControls;
    end;
  end;
end;

procedure TFIBCustomDataSet.CheckDatasetClosed;
begin
  if FOpen then
    FIBError(feDataSetOpen, [nil]);
end;

procedure TFIBCustomDataSet.CheckDatasetOpen;
begin
  if not FOpen then
    FIBError(feDataSetClosed, [nil]);
end;

procedure TFIBCustomDataSet.CheckNotUniDirectional;
begin
  if UniDirectional then
    FIBError(feDataSetUniDirectional, [nil]);
end;

function TFIBCustomDataSet.CanEdit: Boolean;
var
  Buff: PRecordData;
begin
  Buff := PRecordData(GetActiveBuf);
  result := (FQUpdate.SQL.Text <> '') or
            ((Buff <> nil) and (Buff^.rdCachedUpdateStatus = cusInserted) and
             (FCachedUpdates));
end;

function TFIBCustomDataSet.CanInsert: Boolean;
begin
  result := FQInsert.SQL.Text <> '';
end;

function TFIBCustomDataSet.CanDelete: Boolean;
begin
  result := FQDelete.SQL.Text <> ''
end;


procedure TFIBCustomDataSet.CheckEditState;
begin
  case State of
    dsEdit: if not CanEdit then FIBError(feCannotUpdate, [nil]);
    dsInsert: if not CanInsert then FIBError(feCannotInsert, [nil]);
  end;
end;

procedure TFIBCustomDataSet.CheckInsertMode(Buffer: Pointer);
begin
  with PRecordData(Buffer)^ do
    if (State = dsInsert) and (not Modified) then begin
      rdRecordNumber := FRecordCount;
      FCurrentRecord := FRecordCount;
    end;
end;

procedure TFIBCustomDataSet.ClearMemoryCache;
var
  i: Integer;
begin
  for i := 0 to FBlobStreamList.Count - 1 do begin
    TFIBBlobStream(FBlobStreamList[i]).Free;
    FBlobStreamList[i] := nil;
  end;
  FBlobStreamList.Pack;
end;

procedure TFIBCustomDataSet.CopyRecordBuffer(Source, Dest: Pointer);
begin
  Move(Source^, Dest^, FRecordBufferSize);
end;

procedure TFIBCustomDataSet.DoDatabaseDisconnecting(Sender: TObject);
begin
  if Active then begin
    if FCachedUpdates then
      FetchAll
    else
      Active := False;
  end;
  FPrepared := False;
  if Assigned(FDatabaseDisconnecting) then
    FDatabaseDisconnecting(Sender);
end;

procedure TFIBCustomDataSet.DoDatabaseDisconnected(Sender: TObject);
begin
  if Assigned(FDatabaseDisconnected) then
    FDatabaseDisconnected(Sender);
end;

procedure TFIBCustomDataSet.DoDatabaseFree(Sender: TObject);
begin
  if Assigned(FDatabaseFree) then
    FDatabaseFree(Sender);
end;

procedure TFIBCustomDataSet.DoTransactionEnding(Sender: TObject);
begin
  if Transaction.State in [tsDoRollback,tsDoCommit] then
  if Active then begin
    if (FCachedUpdates)  and not (csDestroying in ComponentState) then
      FetchAll
    else
      Active := False;
  end;
  if Assigned(FTransactionEnding) then
    FTransactionEnding(Sender);
end;

procedure TFIBCustomDataSet.DoTransactionEnded(Sender: TObject);
begin
  if Assigned(FTransactionEnded) then
    FTransactionEnded(Sender);
end;

procedure TFIBCustomDataSet.DoTransactionFree(Sender: TObject);
begin
  if Assigned(FTransactionFree) then
    FTransactionFree(Sender);
end;

(*
 * Read the record from FQSelect.Current into the record buffer
 * Then write it to the cache (temporary file).
 *)
procedure TFIBCustomDataSet.FetchCurrentRecordToBuffer(Qry: TFIBQuery;
  RecordNumber: Integer; Buffer: PChar);
var
  p: PRecordData;
  pbd: PBlobDataArray;
  i, j: Integer;
  LocalData: Pointer;
  LocalDate, LocalDouble: Double;
  LocalInt: Integer;
  LocalCurrency:Currency;
  LocalComp: comp;   
  StopFetching:boolean;
begin
  StopFetching:=false;
  p := PRecordData(Buffer);
  (* Make sure blob cache is empty *)
  pbd := PBlobDataArray(Buffer + FBlobCacheOffset);
  if RecordNumber > -1 then
    for i := 0 to BlobFieldCount - 1 do
      pbd^[i] := nil;
  (* Get record information *)
  p^.rdBookmarkFlag := bfCurrent;
  p^.rdFieldCount := FQSelect.Current.Count; // Original source : Qry.Current.Count
  p^.rdRecordNumber := RecordNumber;
  p^.rdUpdateStatus := usUnmodified;
  p^.rdCachedUpdateStatus := cusUnmodified;
  p^.rdSavedOffset := $FFFFFFFF;
  (* Load up the fields *)
  for i := 0 to Qry.Current.Count - 1 do begin
    if (Qry = FQSelect) then
      j := i + 1
    else
      j := FQSelect.FieldIndex[Qry.Current[i].Name] + 1;
    if j > 0 then with p^,p^.rdFields[j] do begin
      fdDataType := Qry.Current[i].Data^.sqltype and (not 1);
      fdDataScale:= Qry.Current[i].Data^.sqlscale;
        
      if ((fdDataType = SQL_LONG) or
          (fdDataType = SQL_SHORT)) and
         (Qry.Current[i].Data^.sqlscale < 0) then
        fdDataType := SQL_DOUBLE;
      fdNullable :=(Qry.Current[i].Data^.sqltype and 1 = 1);
      fdIsNull :=
        (fdNullable and (Qry.Current[i].Data^.sqlind^ = -1));
      LocalData := Qry.Current[i].Data^.sqldata;
      case fdDataType of
        SQL_DATE: begin
          fdDataSize := SizeOf(TDateTime);
          if RecordNumber >= 0 then
            LocalDate := TimeStampToMSecs(
                           DateTimeToTimeStamp(Qry.Current[i].AsDateTime));
          LocalData := PChar(@LocalDate);
        end;
        SQL_TYPE_DATE:
        begin
          fdDataSize := SizeOf(TDateTime);
          if RecordNumber >= 0 then
            LocalInt := DateTimeToTimeStamp(Qry.Current[i].AsDateTime).Date;
          LocalData := PChar(@LocalInt);
        end;
        SQL_TYPE_TIME:
        begin
          fdDataSize := SizeOf(TDateTime);
          if RecordNumber >= 0 then
            LocalInt := DateTimeToTimeStamp(Qry.Current[i].AsDateTime).Time;
          LocalData := PChar(@LocalInt);
        end;
        
        SQL_SHORT, SQL_LONG: begin
          fdDataSize := SizeOf(Integer);
          if RecordNumber >= 0 then
            LocalInt := Qry.Current[i].AsLong;
          LocalData := PChar(@LocalInt);
        end;
        SQL_DOUBLE, SQL_FLOAT, SQL_D_FLOAT: begin
          fdDataSize := SizeOf(Double);
          if RecordNumber >= 0 then
            LocalDouble := Qry.Current[i].AsDouble;
          LocalData := PChar(@LocalDouble);
        end;
        SQL_INT64:
        begin
          if (fdDataScale = 0) then
          begin
               begin
                 fdDataSize := SizeOf(Comp);
                 if RecordNumber >= 0 then
                   LocalComp := Qry.Current[i].AsComp;
                   LocalData := PChar(@LocalComp);
               end
          end
          else
          if (fdDataScale >= (-4)) then
               begin
                 fdDataSize := SizeOf(Currency);
                 if RecordNumber >= 0 then
                   LocalCurrency := Qry.Current[i].AsCurrency;
                   LocalData := PChar(@LocalCurrency);
               end
               else begin
                  fdDataSize := SizeOf(Double);
                  if RecordNumber >= 0 then
                    LocalDouble := Qry.Current[i].AsDouble;
                  LocalData := PChar(@LocalDouble);
               end
        end;
        SQL_VARYING: begin
          fdDataSize := Qry.Current[i].Data^.sqllen;
          fdDataLength := isc_vax_integer(Qry.Current[i].Data^.sqldata, 2);
          if RecordNumber >= 0 then begin
            if (fdDataLength = 0) then
              LocalData := nil
            else
              LocalData := @Qry.Current[i].Data^.sqldata[2];
          end;
        end;
        else (* SQL_TEXT, SQL_BLOB, SQL_ARRAY, SQL_QUAD *) begin
          fdDataSize := Qry.Current[i].Data^.sqllen;
          if (fdDataType = SQL_TEXT) then
            fdDataLength := fdDataSize;
        end;
      end;
      if RecordNumber < 0 then begin
        fdIsNull := True;
        fdDataOfs := FRecordSize;
        Inc(FRecordSize, fdDataSize);
      end else begin
        if fdDataType = SQL_VARYING then begin
          if LocalData <> nil then
            Move(LocalData^, Buffer[fdDataOfs],
                             fdDataLength)
        end else
          Move(LocalData^, Buffer[rdFields[j].fdDataOfs],
                           fdDataSize)
      end;
    end;
  end;
  // Make sure that the "old" buffer doesn't point to anything at all.
  WriteRecordCache(RecordNumber, PChar(p));
  if Assigned(FAfterFetchRecord) then FAfterFetchRecord(Qry,RecordNumber,StopFetching);
  if StopFetching then Abort
end;

function TFIBCustomDataSet.GetActiveBuf: PChar;
begin
  case State of
    dsCalcFields:
      result := CalcBuffer;
    else if not FOpen then
      result := nil
    else
      result := ActiveBuffer;
  end;
end;

function TFIBCustomDataSet.CachedUpdateStatus: TCachedUpdateStatus;
begin
  if Active then
    result := PRecordData(GetActiveBuf)^.rdCachedUpdateStatus
  else
    result := cusUnmodified;
end;

function TFIBCustomDataSet.GetDatabase: TFIBDatabase;
begin
  result := FBase.Database;
end;

function TFIBCustomDataSet.GetDBHandle: PISC_DB_HANDLE;
begin
  result := FBase.DBHandle;
end;

function TFIBCustomDataSet.GetDeleteSQL: TStrings;
begin
  result := FQDelete.SQL;
end;

function TFIBCustomDataSet.GetInsertSQL: TStrings;
begin
  result := FQInsert.SQL;
end;

function TFIBCustomDataSet.GetParams: TFIBXSQLDA;
begin
  result := FQSelect.Params;
end;

function TFIBCustomDataSet.GetRefreshSQL: TStrings;
begin
  result := FQRefresh.SQL;
end;

function TFIBCustomDataSet.GetSelectSQL: TStrings;
begin
  result := FQSelect.SQL;
end;

function TFIBCustomDataSet.GetStatementType: TFIBSQLTypes;
begin
  result := FQSelect.SQLType;
end;

function TFIBCustomDataSet.GetUpdateSQL: TStrings;
begin
  result := FQUpdate.SQL;
end;

function TFIBCustomDataSet.GetTransaction: TFIBTransaction;
begin
  result := FBase.Transaction;
end;

function TFIBCustomDataSet.GetTRHandle: PISC_TR_HANDLE;
begin
  result := FBase.TRHandle;
end;

procedure TFIBCustomDataSet.InternalDeleteRecord(Qry: TFIBQuery; Buff: Pointer);
begin
  SetQueryParams(FQDelete, Buff);
  FQDelete.ExecQuery;
  with PRecordData(Buff)^ do begin
    rdUpdateStatus := usDeleted;
    rdCachedUpdateStatus := cusUnmodified;
  end;
  WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
end;

{$WARNINGS OFF}
function TFIBCustomDataSet.InternalLocate(const KeyFields: string;
  const KeyValues: Variant; Options: TExtLocateOptions): Boolean;
type TArrow=(arForward,arBackward);
var
  fl: TList;
  CurBookmark: String;
  fld, val: Variant;
  fld_cnt: Integer;
  vType,rc,rc1 :Integer;
  Arrow:TArrow;

function CompareValues:boolean;
var i:integer;
begin
      i := 0;
      result := True; Arrow :=arForward;
      while (result and (i < fld_cnt)) do begin
        if fld_cnt > 1 then begin
         val := KeyValues[i];
         vType:=VarType(val);
        end;
        fld := TField(fl[i]).Value;
        result := not (VarIsNull(val) or VarIsNull(fld));
        if result then
          try
            fld := VarAsType(fld, vType);
          except
            on E: EVariantError do result := False;
          end;
        if result then begin
          if TField(fl[i]).DataType = ftString then begin
            if (eloCaseInsensitive in Options) then begin
              fld := AnsiUpperCase(fld);
             if fld_cnt > 1 then
              val := AnsiUpperCase(val);
            end;
            fld := TrimRight(fld);
            if fld_cnt > 1 then
             val := TrimRight(val);
            if (eloPartialKey in Options) then
              result := result and (Pos(val, fld) = 1)
            else
            if  (eloWildCards in Options) then
             Result:= WildStringCompare( val,fld )
            else
              result := result and (val = fld);
          end else
            result := (val = fld);
         if not Result then
          if val > fld then Arrow:=arForward else Arrow:=arBackward;
        end;
        Inc(i);
      end;
end;

begin
  fl := TList.Create;
  try
    GetFieldList(fl, KeyFields);
    fld_cnt := fl.Count;
    CurBookmark := Bookmark;
    result := False;
    if fld_cnt<2 then begin
     val  :=KeyValues;
     vType:=VarType(val);
     if (vType =  varString) or (vType =varOleStr)
     then begin
      val := TrimRight(val);
      if  (eloCaseInsensitive in Options) then
       val := AnsiUpperCase(val);
     end;
    end;
    if (eloInSortedDS in Options) then begin
      Recno:=RecordCount;
      Result:=CompareValues;
      if Result then Exit;
      if Arrow=arForward then  begin
       rc:=Recno;
       Last;
       rc1:=Recno;
       Recno:=rc;
      end
      else begin
       rc1:=Recno;
       Recno:=1;
       rc   :=1
      end;
    end;  
    while ((not result) and (not EOF or (eloInSortedDS in Options))) do
    begin
      if (eloInSortedDS in Options) then begin
       if  (rc>rc1) then Break;
       Recno:=Trunc((rc1+rc)/2);
      end;
      Result:=CompareValues;
      if not result then
       if not (eloInSortedDS in Options) then        Next
       else begin
         if Arrow=arForward then rc:=Recno+1 else rc1:=Recno-1;
       end;
    end;
    if not result then
      Bookmark := CurBookmark
    else
      CursorPosChanged;
  finally
    fl.Free;
  end;
end;
{$WARNINGS ON}

procedure TFIBCustomDataSet.UpdateBlobInfo(Buff: Pointer;IsPost:boolean);
//this is cut from InternalPostRecord
var
  i, j, k: Integer;
  pbd: PBlobDataArray;
begin
  pbd := PBlobDataArray(PChar(Buff) + FBlobCacheOffset);
  j := 0;
  for i := 0 to FieldCount - 1 do
    if Fields[i].IsBlob then begin
      k := Fields[i].FieldNo;
      if pbd^[j] <> nil then begin
        if IsPost then
         pbd^[j].Finalize
        else
         pbd^[j].Cancel;
        PISC_QUAD(
          PChar(Buff) + PRecordData(Buff)^.rdFields[k].fdDataOfs)^ :=
          pbd^[j].BlobID;
        PRecordData(Buff)^.rdFields[k].fdIsNull := pbd^[j].Size = 0;
      end;
      Inc(j);
    end;
end;


procedure TFIBCustomDataSet.InternalPostRecord(Qry: TFIBQuery; Buff: Pointer);
begin
  // Update blob info first
  UpdateBlobInfo(Buff,true);
  // Blobs are finished.
  SetQueryParams(Qry, Buff);
  Qry.ExecQuery;
  PRecordData(Buff)^.rdUpdateStatus := usUnmodified;
  PRecordData(Buff)^.rdCachedUpdateStatus := cusUnmodified;
  SetModified(False);
  WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
  if FQRefresh.SQL.Text <> '' then
    InternalRefreshRow(FQRefresh,Buff);
end;

procedure TFIBCustomDataSet.InternalRefreshRow(Qry: TFIBQuery; Buff:Pointer);// Added (Buff:Pointer);
var
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState: Integer;
 {$ENDIF} 
  ofs: DWORD;
begin
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
 {$ENDIF} 
  try
    if (Qry.SQL.Text <> '') and (Buff <> nil) then begin
     if  not (Qry.Open  or Qry.ProcExecuted)
     then begin
      SetQueryParams(Qry, Buff);
      Qry.ExecQuery;
     end;
     if Qry.Open then
      try
        if (Qry.SQLType = SQLExecProcedure) or
           (Qry.Next <> nil) then begin
          ofs := PRecordData(Buff)^.rdSavedOffset;
          FetchCurrentRecordToBuffer(Qry,
                                     PRecordData(Buff)^.rdRecordNumber,
                                     Buff);
          if (ofs <> $FFFFFFFF) then begin
            PRecordData(Buff)^.rdSavedOffset := ofs;
            WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
            SaveOldBuffer(Buff);
          end;
        end;
      finally
        Qry.Close;
      end;
    end else
      FIBError(feCannotRefresh, [nil]);
  finally
  {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := iCurScreenState;
  {$ENDIF}  
  end;
end;

procedure TFIBCustomDataSet.InternalRevertRecord(RecordNumber: Integer);
var
  NewBuffer, OldBuffer: PRecordData;
begin
  NewBuffer := nil;
  OldBuffer := nil;
  NewBuffer := PRecordData(AllocRecordBuffer);
  OldBuffer := PRecordData(AllocRecordBuffer);
  try
    ReadRecordCache(RecordNumber, PChar(NewBuffer), False);
    ReadRecordCache(RecordNumber, PChar(OldBuffer), True);
    case NewBuffer^.rdCachedUpdateStatus of
      cusInserted: begin
        NewBuffer^.rdCachedUpdateStatus := cusUninserted;
        Inc(FDeletedRecords);
      end;
      cusModified,
      cusDeleted: begin
        if (NewBuffer^.rdCachedUpdateStatus = cusDeleted) then
          Dec(FDeletedRecords);
        CopyRecordBuffer(OldBuffer, NewBuffer);
      end;
    end;
    WriteRecordCache(RecordNumber, PChar(NewBuffer));
  finally
    FreeRecordBuffer(PChar(NewBuffer));
    FreeRecordBuffer(PChar(OldBuffer));
  end;
end;

(*
 * A visible record is one that is not truly deleted, and it is also
 * listed in the FUpdateRecordTypes set.
 *)
function  TFIBCustomDataSet.IsVisibleStat(Buffer: PChar): Boolean;
begin
  result := (PRecordData(Buffer)^.rdCachedUpdateStatus in FUpdateRecordTypes)
            and
            (not ((PRecordData(Buffer)^.rdCachedUpdateStatus = cusUnmodified)
                  and
                  (PRecordData(Buffer)^.rdUpdateStatus = usDeleted)));
end;

function TFIBCustomDataSet.IsVisible(Buffer: PChar): Boolean;
begin
  result :=IsVisibleStat(Buffer)
end;

function TFIBCustomDataSet.LocateNext(const KeyFields: String;
  const KeyValues: Variant; Options: TLocateOptions): Boolean;
begin
  DoBeforeScroll;
  DisableControls;
  DisableScrollEvents;
  Result :=false;
  try
    Result :=
     InternalLocate(KeyFields, KeyValues, TExtLocateOptions(Options));
  finally
    EnableControls;
    EnableScrollEvents;
    if Result then DoAfterScroll;
  end;
end;

function TFIBCustomDataSet.ExtLocateNext(const KeyFields: String; const KeyValues: Variant;
      Options: TExtLocateOptions): Boolean; 
begin
  DoBeforeScroll;
  DisableControls;
  DisableScrollEvents;
  Result :=false;
  try
    Result := InternalLocate(KeyFields, KeyValues, Options);
  finally
    EnableControls;
    EnableScrollEvents;
    if Result then DoAfterScroll;    
  end;
end;

procedure TFIBCustomDataSet.Prepare;
{$IFNDEF FOR_CONSOLE}
var
  iCurScreenState: Integer;
{$ENDIF}  
begin
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
 {$ENDIF}
  try
    FBase.CheckDatabase;
    FBase.CheckTransaction;
    if FQSelect.SQL.Text <> '' then begin
      if not FQSelect.Prepared then
        FQSelect.Prepare;
      if csDesigning in ComponentState then begin
       if (FQDelete.SQL.Text <> '') and (not FQDelete.Prepared) then
        FQDelete.Prepare;
       if (FQInsert.SQL.Text <> '') and (not FQInsert.Prepared) then
         FQInsert.Prepare;
       if (FQRefresh.SQL.Text <> '') and (not FQRefresh.Prepared) then
         FQRefresh.Prepare;
       if (FQUpdate.SQL.Text <> '') and (not FQUpdate.Prepared) then
        FQUpdate.Prepare;
      end;

      FPrepared := True;
      InternalInitFieldDefs;
    end else
      FIBError(feEmptyQuery, [nil]);
  finally
   {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := iCurScreenState;
   {$ENDIF}
  end;
end;

procedure TFIBCustomDataSet.RecordModified(Value: Boolean);
begin
  SetModified(Value);
end;

procedure TFIBCustomDataSet.RevertRecord;
var
  Buff: PRecordData;
begin
  if FCachedUpdates and FUpdatesPending then begin
    Buff := PRecordData(GetActiveBuf);
    InternalRevertRecord(Buff^.rdRecordNumber);
    ReadRecordCache(Buff^.rdRecordNumber, PChar(Buff), False);
    DataEvent(deRecordChange, 0);
  end;
end;

procedure TFIBCustomDataSet.SaveOldBuffer(Buffer: PChar);
var
  OldBuffer: Pointer;
  procedure CopyOldBuffer;
  begin
    CopyRecordBuffer(Buffer, OldBuffer);
    if BlobFieldCount > 0 then
      FillChar(
        PChar(OldBuffer)[FBlobCacheOffset],
        BlobFieldCount * SizeOf(TFIBBlobStream),
        0);
  end;

begin
  if (Buffer <> nil) and (PRecordData(Buffer)^.rdRecordNumber >= 0) then begin
    OldBuffer := AllocRecordBuffer;
    try
      if (PRecordData(Buffer)^.rdSavedOffset = $FFFFFFFF) then begin
        PRecordData(Buffer)^.rdSavedOffset :=
          AdjustPosition(FOldBufferCache, 0, FILE_END);
        CopyOldBuffer;
        WriteCache(
                   FOldBufferCache,
                   0,
                   FILE_CURRENT,
                   OldBuffer
                  );
        WriteCache(
                   FBufferCache,
                   PRecordData(Buffer)^.rdRecordNumber * FRecordBufferSize,
                   FILE_BEGIN,
                   Buffer
                  );
      end else begin
        //ReadRecordCache(PRecordData(Buffer)^.rdRecordNumber, OldBuffer, True);
        CopyOldBuffer;
        WriteCache(
                   FOldBufferCache,
                   PRecordData(Buffer)^.rdSavedOffset,
                   FILE_BEGIN,
                   OldBuffer
                  );
      end;
    finally
      FreeRecordBuffer(PChar(OldBuffer));
    end;
  end;
end;

procedure TFIBCustomDataSet.SetBufferChunks(Value: Integer);
begin
  if (Value <= 0) then
    FBufferChunks := BufferCacheSize
  else
    FBufferChunks := Value;
end;

procedure TFIBCustomDataSet.SetDatabase(Value: TFIBDatabase);
begin
  if FBase.Database <> Value then begin
    (* Check that the dataset is closed *)
    CheckDatasetClosed;
    (* Unset the database property of all owned components *)
    FBase.Database := Value;
    FQDelete.Database := Value;
    FQInsert.Database := Value;
    FQRefresh.Database := Value;
    FQSelect.Database := Value;
    FQUpdate.Database := Value;
  end;
end;

procedure TFIBCustomDataSet.SetDeleteSQL(Value: TStrings);
begin
  CheckDatasetClosed;
  FQDelete.SQL.Assign(Value);
end;

procedure TFIBCustomDataSet.SetInsertSQL(Value: TStrings);
begin
  CheckDatasetClosed;
  FQInsert.SQL.Assign(Value);
end;

procedure TFIBCustomDataSet.SetQueryParams(Qry: TFIBQuery; Buffer: Pointer);
var
  i, j: Integer;
  cr, data: PChar;
  fn, st: String;
  OldBuffer: Pointer;
  ts: TTimeStamp;
begin
  if (Buffer = nil) then
    FIBError(feBufferNotSet, [nil]);
  if (not FPrepared) then
    Prepare;
  OldBuffer := AllocRecordBuffer;
  try
    if PRecordData(Buffer)^.rdRecordNumber<0 then Exit;
    ReadRecordCache(PRecordData(Buffer)^.rdRecordNumber, OldBuffer, Qry<>QInsert);
    for i := 0 to Qry.Params.Count - 1 do begin
      fn := Qry.Params[i].Name;
      if (PosCI('OLD_', fn) = 1) then begin
        fn := Copy(fn, 5, Length(fn));
        cr := OldBuffer;
      end else if (PosCI('NEW_', fn) = 1) then begin
        fn := Copy(fn, 5, Length(fn));
        cr := Buffer;
      end else
        cr := Buffer;
      j := FQSelect.FieldIndex[fn] + 1;
      // Begin Added Source
      // for getting params from SelectSQL
      if j=0 then begin
        if (Qry<>FQSelect) and (Params.ByName[fn]<>nil) then
        Qry.Params.ByName[fn].Assign(Params.ByName[fn])
      end;
      //End  added Source
      if (j > 0) then with PRecordData(cr)^ do
        if rdFields[j].fdIsNull then
          Qry.Params[i].IsNull := True
        else begin
          Qry.Params[i].IsNull := False;
          data := cr + rdFields[j].fdDataOfs;
          case rdFields[j].fdDataType of
            SQL_TEXT, SQL_VARYING: begin
              SetString(st, data, rdFields[j].fdDataLength);
              Qry.Params[i].Data^.sqltype := SQL_TEXT or (Qry.Params[i].Data^.sqltype and 1);//Added
              Qry.Params[i].AsString := st;
            end;
            SQL_FLOAT, SQL_DOUBLE, SQL_D_FLOAT:
              Qry.Params[i].AsDouble := PDouble(data)^;
            SQL_SHORT, SQL_LONG:
              Qry.Params[i].AsLong := PLong(data)^;
            SQL_INT64:
            begin
              if rdFields[j].fdDataScale = 0 then
              {$IFDEF VER130}
                Qry.Params[i].AsInt64 := PInt64(data)^
              {$ELSE}
                Qry.Params[i].AsComp := PComp(data)^
              {$ENDIF}
              else if rdFields[j].fdDataScale >= (-4) then begin
                Qry.Params[i].AsCurrency := PCurrency(data)^;
//                Qry.Params[i].FXSQLVAR^.sqlscale:=rdFields[j].fdDataScale
              end
              else
                Qry.Params[i].AsDouble := PDouble(data)^;
            end;
            SQL_BLOB, SQL_ARRAY, SQL_QUAD: begin
              Qry.Params[i].AsQuad := PISC_QUAD(data)^;
            end;
            SQL_TYPE_DATE:
            begin
              ts.Date := PInt(data)^;
              ts.Time := 0;
              Qry.Params[i].AsDate :=
                TimeStampToDateTime(ts);
            end;
            SQL_TYPE_TIME:
            begin
              ts.Date := 0;
              ts.Time := PInt(data)^;
              Qry.Params[i].AsTime :=
                TimeStampToDateTime(ts);
            end;
            SQL_DATE:
              Qry.Params[i].AsDateTime :=
                TimeStampToDateTime(
                  MSecsToTimeStamp(PDouble(data)^));

          end;
        end;
    end;
  finally
    FreeRecordBuffer(PChar(OldBuffer));
  end;
end;

procedure TFIBCustomDataSet.SetRefreshSQL(Value: TStrings);
begin
  CheckDatasetClosed;
  FQRefresh.SQL.Assign(Value);
end;

procedure TFIBCustomDataSet.SetSelectSQL(Value: TStrings);
begin
  CheckDatasetClosed;
  FQSelect.SQL.Assign(Value);
end;

procedure TFIBCustomDataSet.SetUpdateSQL(Value: TStrings);
begin
  CheckDatasetClosed;
  FQUpdate.SQL.Assign(Value);
end;

procedure TFIBCustomDataSet.SetTransaction(Value: TFIBTransaction);
begin
  if (FBase.Transaction <> Value) then begin
    CheckDatasetClosed;
    FBase.Transaction := Value;
    FQDelete.Transaction := Value;
    FQInsert.Transaction := Value;
    FQRefresh.Transaction := Value;
    FQSelect.Transaction := Value;
    FQUpdate.Transaction := Value;
  end;
end;

procedure TFIBCustomDataSet.SetUpdateRecordTypes(Value: TFIBUpdateRecordTypes);
begin
  FUpdateRecordTypes := Value;
  if Active then First;
end;

procedure TFIBCustomDataSet.SetUniDirectional(Value: Boolean);
begin
  CheckDatasetClosed;
  FUniDirectional := Value;
end;

procedure TFIBCustomDataSet.SourceChanged;
begin
  DisableControls;
  try
    if SelectSQL.Text <> '' then begin
      if Active and (Params.Count > 0) then begin
        Active := False;
        Active := True;
      end;
    end;
  finally
    EnableControls;
  end;
end;

procedure TFIBCustomDataSet.SourceDisabled;
begin
  if Active then
    Close;
end;

procedure TFIBCustomDataSet.SQLChanging(Sender: TObject);
begin
  CheckDatasetClosed;
  FieldDefs.Clear;
  FPrepared := False;
end;

(*
 * I can "undelete" uninserted records (make them "inserted" again).
 * I can "undelete" cached deleted (the deletion hasn't yet occurred).
 *)
procedure TFIBCustomDataSet.Undelete;
var
  Buff: PRecordData;
begin
  CheckActive;
  Buff := PRecordData(GetActiveBuf);
  with Buff^ do begin
    if rdCachedUpdateStatus = cusUninserted then begin
      rdCachedUpdateStatus := cusInserted;
      Dec(FDeletedRecords);
    end else if (rdUpdateStatus = usDeleted) and
            (rdCachedUpdateStatus = cusDeleted) then begin
      rdCachedUpdateStatus := cusUnmodified;
      rdUpdateStatus := usUnmodified;
      Dec(FDeletedRecords);
    end;
    WriteRecordCache(rdRecordNumber, PChar(Buff));
  end;
end;

function TFIBCustomDataSet.UpdateStatus: TUpdateStatus;
begin
  if Active then
    result := PRecordData(GetActiveBuf)^.rdUpdateStatus
  else
    result := usUnmodified;
end;

function TFIBCustomDataSet.IsSequenced: Boolean;
begin
  Result := Assigned( FQSelect ) and FQSelect.EOF;
end;

function TFIBCustomDataSet.AdjustPosition(FCache: PChar; Offset: DWORD;
  Origin: Integer): Integer;
var
  OldCacheSize: Integer;
begin
  if (FCache = FBufferCache) then begin
    case Origin of
      FILE_BEGIN:    FBPos := Offset;
      FILE_CURRENT:  FBPos := FBPos + Offset;
      FILE_END:      FBPos := DWORD(FBEnd) + Offset;
    end;
    OldCacheSize := FCacheSize;
    while (FBPos >= DWORD(FCacheSize)) do
      Inc(FCacheSize, FBufferChunkSize);
    if FCacheSize > OldCacheSize then
      FIBAlloc(FBufferCache, FCacheSize, FCacheSize);
    result := FBPos;
  end else begin
    case Origin of
      FILE_BEGIN:    FOBPos := Offset;
      FILE_CURRENT:  FOBPos := FOBPos + Offset;
      FILE_END:      FOBPos := DWORD(FOBEnd) + Offset;
    end;
    OldCacheSize := FOldCacheSize;
    while (FBPos >= DWORD(FOldCacheSize)) do
      Inc(FOldCacheSize, FBufferChunkSize);
    if FOldCacheSize > OldCacheSize then
      FIBAlloc(FOldBufferCache, FOldCacheSize, FOldCacheSize);
    result := FOBPos;
  end;
end;

procedure TFIBCustomDataSet.ReadCache(FCache: PChar; Offset: DWORD; Origin: Integer;
  Buffer: PChar);
var
  pCache: PChar;
  bOld: Boolean;
begin
  bOld := (FCache = FOldBufferCache);
  pCache := PChar(AdjustPosition(FCache, Offset, Origin));
  if not bOld then
    pCache := FBufferCache + Integer(pCache)
  else
    pCache := FOldBufferCache + Integer(pCache);
  Move(pCache^, Buffer^, DWORD(FRecordBufferSize));
  AdjustPosition(FCache, FRecordBufferSize, FILE_CURRENT);
end;

procedure TFIBCustomDataSet.WriteCache(FCache: PChar; Offset: DWORD; Origin: Integer;
  Buffer: PChar);
var
  pCache: PChar;
  bOld: Boolean;
  dwEnd: DWORD;
begin
  bOld := (FCache = FOldBufferCache);
  pCache := PChar(AdjustPosition(FCache, Offset, Origin));
  if not bOld then
    pCache := FBufferCache + Integer(pCache)
  else
    pCache := FOldBufferCache + Integer(pCache);
  Move(Buffer^, pCache^, FRecordBufferSize);
  dwEnd := AdjustPosition(FCache, FRecordBufferSize, FILE_CURRENT);
  if not bOld then begin
    if (dwEnd > FBEnd) then
      FBEnd := dwEnd;
  end else begin
    if (dwEnd > FOBEnd) then
      FOBEnd := dwEnd;
  end;
end;

procedure TFIBCustomDataSet.ReadRecordCache(RecordNumber: Integer; Buffer: PChar;
  ReadOldBuffer: Boolean);
begin
  if FUniDirectional then
    RecordNumber := RecordNumber mod UniCache;
  if (ReadOldBuffer) then begin
    ReadRecordCache(RecordNumber, Buffer, False);
    if (PRecordData(Buffer)^.rdSavedOffset <> $FFFFFFFF) then
      ReadCache(
                FOldBufferCache,
                PRecordData(Buffer)^.rdSavedOffset,
                FILE_BEGIN,
                Buffer
               )
  end else
    ReadCache(
              FBufferCache,
              RecordNumber * FRecordBufferSize,
              FILE_BEGIN,
              Buffer
             );
end;

procedure TFIBCustomDataSet.WriteRecordCache(RecordNumber: Integer; Buffer: PChar);
begin
  if RecordNumber >= 0 then begin
    if FUniDirectional then
      RecordNumber := RecordNumber mod UniCache;
    WriteCache(
               FBufferCache,
               RecordNumber * FRecordBufferSize,
               FILE_BEGIN,
               Buffer
              );
  end;
end;

//
function TFIBCustomDataSet.GetNewBuffer:PChar;
begin
 Result:=AllocRecordBuffer;
 ReadRecordCache(PRecordData(GetActiveBuf)^.rdRecordNumber, Result, False);
end;

function TFIBCustomDataSet.GetOldBuffer:PChar;
begin
 Result:=AllocRecordBuffer;
 ReadRecordCache(PRecordData(GetActiveBuf)^.rdRecordNumber, Result, true);
end;

function   TFIBCustomDataSet.IsFiltered:boolean;
begin
 Result:=Filtered
end;
//
procedure TFIBCustomDataSet.SynchronizeBuffers(const ExecEvent,ToFibCach:boolean);
var i,rn:integer;
    FIBBuff,DSBuff:Pchar;
begin
 FIBBuff:=AllocRecordBuffer;
 try
  for i:=0 to Pred(BufferCount) do begin
   DSBuff:=Buffers[i];
   rn:=PRecordData(DSBuff)^.rdRecordNumber;
   if rn>=0 then begin
    ReadRecordCache(rn, FIBBuff, False);
    if ToFibCach then   begin
     Move(DSBuff^,FIBBuff^,FRecordBufferSize);
     WriteRecordCache(rn, FIBBuff);
    end
    else
     Move(FIBBuff^,DSBuff^,FRecordBufferSize)
   end;
  end;
 finally
  if ExecEvent then   DataEvent(deDataSetChange,0);
  FreeRecordBuffer(FIBBuff)
 end
end;

procedure TFIBCustomDataSet.SwapRecords(Recno1,Recno2:integer);
var  r1, r2: Pchar;
begin
    if Recno1=Recno2 then Exit;
    if (Recno1<0) or (Recno2<0) then Exit;
    if (Recno1>FRecordCount) or (Recno2>FRecordCount) then Exit;
    r1 := AllocRecordBuffer;
    r2 := AllocRecordBuffer;
    try
     ReadRecordCache(Recno1-1, r1, False);
     ReadRecordCache(Recno2-1, r2, False);
     PRecordData(r1)^.rdRecordNumber := Recno2-1;
     PRecordData(r2)^.rdRecordNumber := Recno1-1;
     WriteRecordCache(Recno1-1, PChar(r2));
     WriteRecordCache(Recno2-1, PChar(r1));
     SynchronizeBuffers(true,false);     
   finally
    FreeRecordBuffer(r1);
    FreeRecordBuffer(r2);
   end;
end;

procedure TFIBCustomDataSet.MoveRecord(OldRecno,NewRecno:integer);
var Buff,BuffRec:PChar;
    i:integer;
    frI,toI:integer;
    Sign:ShortInt;
begin
  if NewRecno<1 then NewRecno:=1;
  if NewRecno>FRecordCount then begin
    FetchAll;
    if NewRecno>FRecordCount then
     NewRecno:=FRecordCount;
  end;
  Dec(OldRecno);Dec(NewRecno);
  if OldRecno=NewRecno then Exit;
  if OldRecno>NewRecno then begin
   frI:=Pred(OldRecno);
   toI:=Pred(NewRecno);
   Sign:=1;
  end
  else begin
   frI:=Succ(OldRecno);
   toI:=Succ(NewRecno);
   Sign:=-1;
  end;
  Buff   :=AllocRecordBuffer;
  BuffRec   :=AllocRecordBuffer;
  ReadRecordCache(OldRecno, BuffRec, False);
 try
  i:=frI;
  while i<>toI do      begin
   ReadRecordCache(i, Buff, False);
   PRecordData(Buff)^.rdRecordNumber :=
    PRecordData(Buff)^.rdRecordNumber+Sign;
   WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
   i:=i-Sign
  end;
  PRecordData(BuffRec)^.rdRecordNumber :=NewRecno;
  WriteRecordCache(PRecordData(BuffRec)^.rdRecordNumber, BuffRec);
  SynchronizeBuffers(true,false);
 finally
   FreeRecordBuffer(Buff);
   FreeRecordBuffer(BuffRec);
 end;
end;

(*
 * TDataset overrides
 *)
function TFIBCustomDataSet.AllocRecordBuffer: PChar;
begin
  result := nil;
  FIBAlloc(result, FRecordBufferSize, FRecordBufferSize);
  Move(FModelBuffer^, result^, FRecordBufferSize);
end;

function TFIBCustomDataSet.CreateBlobStream(Field: TField; Mode: TBlobStreamMode): TStream;
var
  pb: PBlobDataArray;
  fs: TFIBBlobStream;
  Buff: PChar;
  bTr, bDB: Boolean;
begin
// Result:=inherited CreateBlobStream(Field, Mode)

  if   vTypeDispositionField in [dfNormal] then
  case State of // Replace Source
   dsFilter  : begin
                 Buff:=AllocRecordBuffer;
                 ReadRecordCache(FCurrentRecord, Buff, False);
               end;

   dsNewValue:  Buff := GetNewBuffer;
   dsOldValue:  Buff := GetOldBuffer;
  else
   Buff := GetActiveBuf;
  end
  else
  case vTypeDispositionField of
   dfRRecNumber,dfRWRecNumber:
               begin
                   Buff :=AllocRecordBuffer;
                   if State<>dsOldValue then
                    ReadRecordCache(vInspectRecno, Buff, False)
                   else
                    ReadRecordCache(vInspectRecno, Buff, True);
                end;
  end;
  try
   pb := PBlobDataArray(Buff + FBlobCacheOffset);
   if pb^[Field.Offset] = nil then begin
    CheckInsertMode(Buff);
    pb^[Field.Offset] := TFIBBlobStream.Create;
    fs := pb^[Field.Offset];
    FBlobStreamList.Add(Pointer(fs));
    fs.Mode := bmReadWrite;
    fs.Database := Database;
    fs.Transaction := Transaction;
    fs.BlobID :=
      PISC_QUAD(@Buff[PRecordData(Buff)^.rdFields[Field.FieldNo].fdDataOfs])^;
    if (CachedUpdates) then begin
      bTr := not Transaction.InTransaction;
      bDB := not Database.Connected;
      if bDB then
        Database.Open;
      if bTr then
        Transaction.StartTransaction;
      fs.Seek(0, soFromBeginning);
      if bTr then
        Transaction.Commit;
      if bDB then
        Database.Close;
    end;
    WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Pointer(Buff));
   end else
    fs := pb^[Field.Offset];
   result := TFIBDSBlobStream.Create(Field, fs, Mode);
  finally
   if (State in [dsFilter,dsOldValue,dsNewValue]) or (vTypeDispositionField<>dfNormal)
    then  FreeRecordBuffer(Buff);
  end;
end;

function TFIBCustomDataSet.CompareBookmarks(Bookmark1, Bookmark2: TBookmark): Integer;
const
  CMPLess = -1;
  CMPEql  =  0;
  CMPGtr  =  1;
  RetCodes: array[Boolean, Boolean] of ShortInt = ((2, CMPLess),
                                                   (CMPGtr, CMPEql));
begin
  result := RetCodes[Bookmark1 = nil, Bookmark2 = nil];

  if Result = 2 then  // Both bookmarks are initialized
  begin
    if PInteger(Bookmark1)^ < PInteger(Bookmark2)^ then Result := CMPLess
    else
    if PInteger(Bookmark1)^ > PInteger(Bookmark2)^ then Result := CMPGtr
    else Result := CMPEql;
  end;
end;

procedure TFIBCustomDataSet.DoBeforeCancel;
begin
  inherited;
end;

procedure TFIBCustomDataSet.DoBeforeDelete;
var
  Buff: PRecordData;
begin
  if  not CanDelete then Abort;
  Buff := PRecordData(GetActiveBuf);
  if Buff^.rdCachedUpdateStatus in [cusUnmodified] then
    SaveOldBuffer(PChar(Buff));
  inherited;
end;

procedure TFIBCustomDataSet.DoBeforeEdit;
var
  Buff: PRecordData;
  i:integer;
begin
  if not CanEdit then Abort;
  Buff := PRecordData(GetActiveBuf);
  if Buff^.rdCachedUpdateStatus in [cusUnmodified, cusInserted] then
    SaveOldBuffer(PChar(Buff));
  CopyRecordBuffer(GetActiveBuf, FOldBuffer);
  inherited;
  {$IFDEF SUPPORT_ARRAY_FIELD}
   for i:=0 to Pred(FieldCount) do begin
    if Fields[i] is TFIBArrayField then
     TFIBArrayField(Fields[i]).SaveOldBuffer
   end;
  {$ENDIF}

end;

procedure TFIBCustomDataSet.DoBeforePost; //override;
begin
 inherited;
end;

procedure TFIBCustomDataSet.DoBeforeInsert;
begin
  if not CanInsert then Abort;
  inherited;
end;

procedure TFIBCustomDataSet.DoAfterScroll;
begin
 if vDisableScrollCount=0 then  inherited;
end;

procedure TFIBCustomDataSet.DoBeforeScroll;
begin
 if vDisableScrollCount=0 then  inherited;
end;

procedure TFIBCustomDataSet.DisableScrollEvents;
begin
 Inc(vDisableScrollCount)
end;

procedure TFIBCustomDataSet.EnableScrollEvents;
begin
 if vDisableScrollCount>0 then
  Dec(vDisableScrollCount)
end;

procedure  TFIBCustomDataSet.DoAfterInsert;
var i:integer;
begin
  {$IFDEF SUPPORT_ARRAY_FIELD}
   for i:=0 to Pred(FieldCount) do begin
    if Fields[i] is TFIBArrayField then
     TFIBArrayField(Fields[i]).SaveOldBuffer
   end;
  {$ENDIF}
  inherited;

end;

procedure TFIBCustomDataSet.DoOnPostError(DataSet: TDataSet; E: EDatabaseError;
 var Action: TDataAction
);
var     i:integer;
begin
   {$IFDEF SUPPORT_ARRAY_FIELD}
  for i:=0 to Pred(FieldCount) do begin
    if Fields[i] is TFIBArrayField then
     TFIBArrayField(Fields[i]).RestoreOldBuffer
  end;
  {$ENDIF}
end;

procedure TFIBCustomDataSet.FetchAll;
var
  CurBookmark: String;
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState: Integer;
 {$ENDIF} 
begin
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
 {$ENDIF}
  try
    if FQSelect.EOF or not FQSelect.Open then
      exit;
    DisableControls;
    try
      CurBookmark := Bookmark;
      Last;
      Bookmark := CurBookmark;
    finally
      EnableControls;
    end;
  finally
   {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := iCurScreenState;
   {$ENDIF} 
  end;
end;

(*
 * Free up the record buffer allocated by TDataset
 *)
procedure TFIBCustomDataSet.FreeRecordBuffer(var Buffer: PChar);
begin
  FIBAlloc(Buffer, 0, 0);
end;

procedure TFIBCustomDataSet.GetBookmarkData(Buffer: PChar; Data: Pointer);
begin
  Move(PRecordData(Buffer)^.rdRecordNumber, Data^, BookmarkSize);
end;

function TFIBCustomDataSet.GetBookmarkFlag(Buffer: PChar): TBookmarkFlag;
begin
  result := PRecordData(Buffer)^.rdBookmarkFlag;
end;

function TFIBCustomDataSet.GetCanModify: Boolean;
begin
  result := (FQInsert.SQL.Text <> '') or (FQUpdate.SQL.Text <> '') or
            (FQDelete.SQL.Text <> '');
end;

function TFIBCustomDataSet.GetCurrentRecord(Buffer: PChar): Boolean;
begin
  if not IsEmpty and (GetBookmarkFlag(ActiveBuffer) = bfCurrent) then begin
    UpdateCursorPos;
    ReadRecordCache(PRecordData(ActiveBuffer)^.rdRecordNumber, Buffer, False);
    result := True;
  end else
    result := False;
end;

function TFIBCustomDataSet.GetDataSource: TDataSource;
begin
  if FSourceLink = nil then
    result := nil
  else
    result := FSourceLink.DataSource;
end;

function TFIBCustomDataSet.GetFieldClass(FieldType: TFieldType): TFieldClass;
begin
  Result:=nil;
  if (FieldType<=ftTypedBinary) then
   Result := DefaultFieldClasses[FieldType]
  else
  {$IFDEF SUPPORT_ARRAY_FIELD}
   if FieldType=ftBytes then Result :=  TFIBArrayField
  {$ENDIF}
end;

{$IFNDEF VER100}
function TFIBCustomDataSet.GetFieldData(FieldNo: Integer; Buffer: Pointer): Boolean;
begin
  result := GetFieldData(Fields[FieldNo-1], buffer);
end;
{$ENDIF}
{
function TFIBCustomDataSet.GetFieldData(Field: TField; Buffer: Pointer): Boolean;
var
  Buff, Data: PChar;
  CurrentRecord: PRecordData;
begin
  result := False;
  Buff := GetActiveBuf;
  if (Buff = nil) or (not IsVisible(Buff)) then exit;
  (* The intention here is to stuff the buffer with the data for the *)
  (* referenced field for the current record.                        *)
  CurrentRecord := PRecordData(Buff);
  if (Field.FieldNo > 0) and
     (Field.FieldNo <= CurrentRecord^.rdFieldCount) then begin
    if (IsVisible(Buff)) then
    begin
      result := not CurrentRecord^.rdFields[Field.FieldNo].fdIsNull;
      if result and (Buffer <> nil) then
        with CurrentRecord^.rdFields[Field.FieldNo] do begin
          Data := Buff + CurrentRecord^.rdFields[Field.FieldNo].fdDataOfs;
          if (fdDataType = SQL_VARYING) or (fdDataType = SQL_TEXT) then begin
            Move(Data^, Buffer^, fdDataLength);
            PChar(Buffer)[fdDataLength] := #0;
          end else
            Move(Data^, Buffer^, Field.DataSize);
        end;
    end
  end else if (Field.FieldNo < 0) then begin
    Inc(Buff, FRecordSize + Field.Offset);
    result := Boolean(Buff[0]);
    if result and (Buffer <> nil) then
      Move(Buff[1], Buffer^, Field.DataSize);
  end;
end;
}

function  TFIBCustomDataSet.RecordFieldValue(Field:TField;RecNumber:integer):Variant;
begin
 Result:=false;
 if //(State<>dsBrowse) or
   (RecNumber>Pred(FRecordCount)) then Exit;
 vInspectRecno:=RecNumber;
 try
  vTypeDispositionField:=dfRRecNumber;
  if not Field.isBlob then  Result:=Field.Value
  else Result:=Field.Value;
 finally
  vTypeDispositionField:=dfNormal
 end;
end;

function TFIBCustomDataSet.GetFieldData(Field: TField; Buffer: Pointer): Boolean;//override;
var
  Buff, Data: PChar;
  CurrentRecord: PRecordData;
begin
//  Hide TFIBDataSet.GetFieldData
  result := False;
  if   vTypeDispositionField in [dfNormal] then
  case State of
   dsFilter  : begin
                 Buff:=AllocRecordBuffer;
                 ReadRecordCache(FCurrentRecord, Buff, False);
               end;

   dsNewValue:  Buff := GetNewBuffer;
   dsOldValue:  Buff := GetOldBuffer;
  else
   Buff := GetActiveBuf;
  end
  else
  case vTypeDispositionField of
   dfRRecNumber,dfRWRecNumber:
               begin
                   Buff :=AllocRecordBuffer;
                   if State<>dsOldValue then
                    ReadRecordCache(vInspectRecno, Buff, False)
                   else
                    ReadRecordCache(vInspectRecno, Buff, True);
                end;
  end;
try
  if (Buff = nil) then Exit
  else begin
   if not (State in [dsOldValue]) then
    if (not IsVisibleStat(Buff) ) then exit;
   if IsFiltered and (vFilteredRecCounts=0) and  not (State in [dsFilter,dsInsert])
   then exit;
  end;
// End Replace

  (* The intention here is to stuff the buffer with the data for the *)
  (* referenced field for the current record.                        *)
  CurrentRecord := PRecordData(Buff);
  if (Field.FieldNo > 0) and
     (Field.FieldNo <= CurrentRecord^.rdFieldCount) and
     not ((State in [dsBrowse]) and (CurrentRecord^.rdRecordNumber >= FRecordCount)) then begin
    result := not CurrentRecord^.rdFields[Field.FieldNo].fdIsNull;
    if result and (Buffer <> nil) then
      with CurrentRecord^.rdFields[Field.FieldNo] do begin
        Data := Buff + CurrentRecord^.rdFields[Field.FieldNo].fdDataOfs;
        if (fdDataType = SQL_VARYING) or (fdDataType = SQL_TEXT) then begin
          Move(Data^, Buffer^, fdDataLength);
          PChar(Buffer)[fdDataLength] := #0;
          if poTrimCharFields in FOptions then
           PChar(Buffer)[Length(TrimRight(string(PChar(Buffer))))]:=#0;
        end else
         if ((FCurrentRecord>-1) and (FCurrentRecord<RecordCount))
          or not (State in [dsFilter]) then  
          Move(Data^, Buffer^, Field.DataSize);
      end;
  end else if (Field.FieldNo < 0) then begin
   // Calculated Fields
    Inc(Buff, FRecordSize + Field.Offset);
    result := Boolean(Buff[0]);
    if result and (Buffer <> nil) then
      Move(Buff[1], Buffer^, Field.DataSize);
    Dec(Buff, FRecordSize + Field.Offset); //Restore Calc buffer for FreeRecordBuffer
  end;
finally
 if (State in [dsFilter,dsOldValue,dsNewValue]) or
    (vTypeDispositionField<>dfNormal)
  then  FreeRecordBuffer(Buff);
end;
end;

(*
 * GetRecNo and SetRecNo both operate off of 1-based indexes as
 * opposed to 0-based indexes.
 * This is because we want LastRecordNumber/RecordCount = 1
 *)
function TFIBCustomDataSet.GetRecNo: Integer;
begin
  if GetActiveBuf = nil then
    result := 0
  else
    result := PRecordData(GetActiveBuf)^.rdRecordNumber + 1;
end;

{$WARNINGS OFF}
function TFIBCustomDataSet.GetRecord(Buffer: PChar; GetMode: TGetMode;
  DoCheck: Boolean): TGetResult;
var  Action:TDataAction;
begin
 Action:=daAbort;
 try
  result := grError;
  case GetMode of
    gmCurrent: begin
      if (FCurrentRecord >= 0) then begin
        if FCurrentRecord < FRecordCount then
          ReadRecordCache(FCurrentRecord, Buffer, False)
        else begin
          while (not FQSelect.EOF) and
                (FCurrentRecord >= FRecordCount)  and              
                (FQSelect.Next <> nil)
                 do begin
            FetchCurrentRecordToBuffer(FQSelect, FRecordCount, Buffer);
            Inc(FRecordCount);
          end;
          FCurrentRecord := FRecordCount - 1;
          if (FCurrentRecord >= 0) then
            ReadRecordCache(FCurrentRecord, Buffer, False);
        end;
        result := grOk;
      end else
        result := grBOF;
    end;
    gmNext: begin
      result := grOk;
      if FCurrentRecord = FRecordCount then
        result := grEOF
      else if FCurrentRecord = FRecordCount - 1 then begin
        if (not FQSelect.EOF) then begin
          FQSelect.Next;
          Inc(FCurrentRecord);
        end;
        if (FQSelect.EOF) then begin
          result := grEOF;
        end else begin
          FetchCurrentRecordToBuffer(FQSelect, FCurrentRecord, Buffer);
          Inc(FRecordCount);          
        end;
      end else if (FCurrentRecord < FRecordCount) then begin
        Inc(FCurrentRecord);
        ReadRecordCache(FCurrentRecord, Buffer, False);
      end;
    end;
    else (* gmPrior *) begin
      //CheckNotUniDirectional;
      if (FCurrentRecord = 0) then begin
        Dec(FCurrentRecord);
        result := grBOF;
      end else if (FCurrentRecord > 0) and
                  (FCurrentRecord <= FRecordCount) then begin
        Dec(FCurrentRecord);
        ReadRecordCache(FCurrentRecord, Buffer, False);
        result := grOk;
      end else if (FCurrentRecord = -1) then
        result := grBOF;
    end;
  end;
  if result = grOk then
    result := AdjustCurrentRecord(Buffer, GetMode);
  if result = grOk then with PRecordData(Buffer)^ do begin
    rdBookmarkFlag := bfCurrent;
    GetCalcFields(Buffer);
  end else if (result = grEOF) then begin
    CopyRecordBuffer(FModelBuffer, Buffer);
    PRecordData(Buffer)^.rdBookmarkFlag := bfEOF;
  end else if (result = grBOF) then begin
    CopyRecordBuffer(FModelBuffer, Buffer);
    PRecordData(Buffer)^.rdBookmarkFlag := bfBOF;
  end else if (result = grError) then begin
    CopyRecordBuffer(FModelBuffer, Buffer);
    PRecordData(Buffer)^.rdBookmarkFlag := bfEOF;
  end;
 except
   On E: EDatabaseError do
   if Assigned(FOnGetRecordError) then FOnGetRecordError(Self,E,Action)
 end;
end;

{$WARNINGS ON}
function TFIBCustomDataSet.GetRecordCount: Integer;
begin
  result := FRecordCount - FDeletedRecords;
end;

function TFIBCustomDataSet.GetRecordSize: Word;
begin
  result := FRecordBufferSize;
end;

procedure TFIBCustomDataSet.InternalAddRecord(Buffer: Pointer; Append: Boolean);
begin
  (* Cannot insert a record without a FQInsert query existing. *)
  if CanInsert then begin
    (*
     * When adding records, we *always* append.
     * Insertion is just too costly.
     *)
     // This comment is obsolute
   if Append then begin
    InternalLast;
    Inc(FCurrentRecord);
   end;
    with PRecordData(Buffer)^ do begin
      rdRecordNumber := FCurrentRecord;
      rdUpdateStatus := usInserted;
      rdCachedUpdateStatus := cusInserted;
    end;
    if not CachedUpdates then
      InternalPostRecord(FQInsert, Buffer)
    else
      WriteRecordCache(FCurrentRecord, Buffer);
    Inc(FRecordCount);
    InternalSetToRecord(Buffer);
  end else
    FIBError(feCannotInsert, [nil]);
end;

procedure TFIBCustomDataSet.InternalCancel;
var
  Buff: PChar;
  CurRec: Integer;
begin
  inherited;
  Buff := GetActiveBuf;
  if Buff <> nil then begin
    CurRec := FCurrentRecord;
    CheckInsertMode(Buff);
    if (State = dsEdit) then begin
      CopyRecordBuffer(FOldBuffer, Buff);
      WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
    end else begin  // insert mode...
      CopyRecordBuffer(FModelBuffer, Buff);
      PRecordData(Buff)^.rdUpdateStatus := usDeleted;
      PRecordData(Buff)^.rdCachedUpdateStatus := cusUnmodified;
      PRecordData(Buff)^.rdBookmarkFlag := bfEOF;
      // WriteRecordCache(FCurrentRecord, Buff);
      // Dec(FCurrentRecord);
      FCurrentRecord := CurRec;      
    end;
  end;
  UpdateBlobInfo(Buff,false);  
end;

procedure TFIBCustomDataSet.InternalClose;
begin
  FQSelect.Close;
  ClearMemoryCache;
  FreeRecordBuffer(FModelBuffer);
  FreeRecordBuffer(FOldBuffer);
  FCurrentRecord := -1;
  FOpen := False;
  FRecordCount := 0;
  FDeletedRecords := 0;
  FRecordSize := 0;

  FBPos := 0;
  FOBPos := 0;
  FCacheSize := 0;
  FOldCacheSize := 0;
  FBEnd := 0;
  FOBEnd := 0;
  FIBAlloc(FBufferCache, 0, 0);
  FIBAlloc(FOldBufferCache, 0, 0);

  BindFields(False);                    // Unbind the fields
  if DefaultFields then DestroyFields;  // Destroy the fields
end;

procedure TFIBCustomDataSet.InternalDelete;
var
  Buff: PChar;
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState: Integer;
 {$ENDIF} 
begin
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
 {$ENDIF}
  try
    Buff := GetActiveBuf;
    (* Cannot delete a record without a FQDelete query existing. *)
    if CanDelete then begin
      if not CachedUpdates then
        InternalDeleteRecord(FQDelete, Buff)
      else begin
        with PRecordData(Buff)^ do begin
          if rdCachedUpdateStatus = cusInserted then
            rdCachedUpdateStatus := cusUninserted
          else begin
            rdUpdateStatus := usDeleted;
            rdCachedUpdateStatus := cusDeleted;
          end;
        end;
        WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
      end;
      Inc(FDeletedRecords);
      FUpdatesPending := True;
    end else
      FIBError(feCannotDelete, [nil]);
  finally
   {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := iCurScreenState;
   {$ENDIF} 
  end;
end;

procedure TFIBCustomDataSet.InternalFirst;
begin
  FCurrentRecord := -1;
end;

procedure TFIBCustomDataSet.InternalGotoBookmark(Bookmark: Pointer);
begin
  FCurrentRecord := PInteger(Bookmark)^;
end;

procedure TFIBCustomDataSet.InternalHandleException;
begin
 {$IFNDEF FOR_CONSOLE}
  Application.HandleException(Self)
 {$ENDIF} 
end;


procedure TFIBCustomDataSet.InternalInitFieldDefs; 
var
  DataType: TFieldType;
  Size: Word;
  i, FieldNo: Integer;
  Name   : String;
  isSmallInt,isBCD:boolean;
{$IFDEF VER110}
  FieldDef: TFieldDef;
{$ENDIF}
begin
  (* Destroy any previously existing information... *)
  if not Prepared then begin
    Prepare;    Exit;
  end;
 {$IFDEF VER130}
  FieldDefs.BeginUpdate;
 {$ENDIF} 
  try
   FieldDefs.Clear;
   for i := 0 to FQSelect.Current.Count - 1 do
    with FQSelect.Current[i].Data^ do begin
      (* Get the field name *)
      SetString(Name, aliasname, aliasname_length);
      Size   := 0;
      case sqltype and not 1 of
        // All VARCHAR's must be converted to strings before recording
        // their values
        SQL_VARYING, SQL_TEXT: begin
          Size := sqllen;
          DataType := ftString;
        end;
        // All Doubles/Floats should be cast to doubles.
        //
        SQL_DOUBLE, SQL_FLOAT:
        begin
         {$IFDEF VER130}
          isBCD:=false;
         {$ELSE }
         isBCD:=(sqlscale>-5) and (sqlscale<0)
          {$IFDEF AddedFieldTypes_OnlyForNewFields}
          and ((FindField(Name)=nil) or (FindField(Name) is TBCDField))
          {$ENDIF}
         {$ENDIF}
         ;
         if isBCD  then begin
           Size:=-sqlscale;
           DataType := ftBCD;
         end
         else begin
          DataType := ftFloat;
         end;
        end;

        // SQL_LONG = 4 bytes
        SQL_SHORT, SQL_LONG: if (sqlscale < 0) then
            DataType := ftFloat
          else begin
          //Added Sources
           isSmallInt:=(sqlLen<>4)
         {$IFDEF AddedFieldTypes_OnlyForNewFields}
             and ((FindField(Name)=nil) or (FindField(Name) is TSmallIntField))
         {$ENDIF}
           ;
           if isSmallInt then
             DataType := ftSmallInt
           else
             DataType := ftInteger;
          end;
         SQL_INT64:
         begin
            if (sqlscale = 0) then
            DataType := ftBCD
            {$IFDEF VER130}
//              DataType := ftLargeInt
            {$ELSE}
//              DataType := ftBCD
            {$ENDIF}
            else if (sqlscale >= (-4)) then
            begin
              DataType := ftBCD;
            end
            else
              DataType := ftFloat;
         end;


// SQL_DATE = 8 bytes
        SQL_TIMESTAMP: DataType := ftDateTime;
        SQL_TYPE_TIME: DataType := ftTime;
        SQL_TYPE_DATE: DataType := ftDate;
// SQL_BLOB = variable
        SQL_BLOB: begin
          Size := SizeOf(TISC_QUAD);
          if sqlsubtype = 1 then
             DataType := ftMemo
          else
             DataType := ftBlob;
        end;
        // SQL_ARRAY = variable

        SQL_ARRAY: begin
          Size := SizeOf(TISC_QUAD);
          {$IFDEF VER130}
//          DataType :=ftArray
            DataType := ftBytes;
          {$ELSE}
           DataType := ftBytes;
          {$ENDIF}
        end;
        else
          DataType := ftUnknown;
      end;
      FieldNo := i + 1;
      if DataType <> ftUnknown then begin
        (*
         * C++-Builder has a different constructor for TFieldDef than
         * Delphi does. This is kinda annoying...
         * Anyways, I believe the currently discussed C++-Builder uses
         * the compiler define VER110, soo...
         *)
{$IFDEF VER110}
        if DataType <> ftUnknown then begin
          FieldDef := TFieldDef.Create( FieldDefs );
          FieldDef.Name := String( Name );
          FieldDef.DataType := DataType;
          FieldDef.Size := Size;
          FieldDef.Required := False;
          FieldDef.FieldNo := FieldNo;
          FieldDef.InternalCalcField := False;
        end;
{$ELSE}
        with TFieldDef.Create(FieldDefs, Name,
                   DataType, Size, False, FieldNo) do
          InternalCalcField := False;
{$ENDIF}
      end;
  end;
  finally
 {$IFDEF VER130}
   FieldDefs.EndUpdate;
 {$ENDIF}

  end
end;



procedure TFIBCustomDataSet.InternalInitRecord(Buffer: PChar);
begin
  CopyRecordBuffer(FModelBuffer, Buffer);
end;

procedure TFIBCustomDataSet.DoOnSelectFetch
 (RecordNumber:integer;   var StopFetching:boolean);
begin
 if Assigned(FBeforeFetchRecord) then
  FBeforeFetchRecord(QSelect,RecordNumber,StopFetching);
end;

procedure TFIBCustomDataSet.InternalLast;
var
  Buffer: PChar;
begin
  if (FQSelect.EOF) then
    FCurrentRecord := FRecordCount
  else begin
    Buffer := AllocRecordBuffer;
    try
      while (FQSelect.Next <> nil)  do begin
       FetchCurrentRecordToBuffer(FQSelect, FRecordCount, Buffer);
       Inc(FRecordCount);
      end;
    except
    end;
    FCurrentRecord := FRecordCount;
    FreeRecordBuffer(Buffer);
  end;
end;

procedure TFIBCustomDataSet.InternalOpen;
var
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState,
 {$ENDIF} 
  i: Integer;
  cur_param: TFIBXSQLVAR;
  cur_field: TField;
  s: TStream;
begin
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
 {$ENDIF} 
  try
    if FQSelect.SQL.Text = '' then
      FIBError(feEmptyQuery, [nil]);
    if not FPrepared then
      Prepare;
    if (Params.Count > 0) and (DataSource <> nil) then begin
      for i := 0 to Params.Count - 1 do begin
        cur_field := DataSource.DataSet.FindField(Params[i].Name);
        cur_param := Params[i];
        if (cur_field <> nil) then begin
          if (cur_field.IsNull) then
            cur_param.IsNull := True
          else case cur_field.DataType of
            ftString:
              cur_param.AsString := cur_field.AsString;
            ftSmallint, ftInteger, ftWord:
              cur_param.AsLong := cur_field.AsInteger;
            ftFloat, ftCurrency:
              cur_param.AsDouble := cur_field.AsFloat;
            ftBCD:
             cur_param.AsCurrency := cur_field.AsCurrency;  
            ftDate:
              cur_param.AsDate := cur_field.AsDateTime;
            ftDateTime:
              cur_param.AsDateTime := cur_field.AsDateTime;
            ftTime:
              cur_param.AsTime := cur_field.AsDateTime;
            ftBlob: begin
              s := nil;
              try
                s := DataSource.DataSet.
                       CreateBlobStream(cur_field, bmRead);
                cur_param.LoadFromStream(s);
              finally
                s.free;
              end;
            end;
            else
              FIBError(feNotSupported, [nil]);
          end;
        end;
      end;
    end;
    if FQSelect.SQLType = SQLSelect then begin
      if DefaultFields then CreateFields;
      BindFields(True);
      FCurrentRecord := -1;
      FQSelect.ExecQuery;
      FOpen := FQSelect.Open;
      (*
       * Initialize offsets, buffer sizes, etc...
       * 1. Initially FRecordSize is just the "RecordDataLength".
       * 2. Allocate a "model" buffer and do a dummy fetch
       * 3. After the dummy fetch, FRecordSize will be appropriately
       *    adjusted to reflect the additional "weight" of the field
       *    data.
       * 4. Set up the FCalcFieldsOffset, FBlobCacheOffset and FRecordBufferSize.
       * 5. Re-allocate the model buffer, accounting for the new
       *    FRecordBufferSize.
       * 6. Finally, calls to AllocRecordBuffer will work!.
       *)
      FRecordSize := RecordDataLength(FQSelect.Current.Count);      // (1)
      FIBAlloc(FModelBuffer, 0, FRecordSize);                 // (2)
      FetchCurrentRecordToBuffer(FQSelect, -1, FModelBuffer);
                                                                    // (3)
      FCalcFieldsOffset := FRecordSize;                             // (4)
      FBlobCacheOffset := FCalcFieldsOffset + CalcFieldsSize;
      FRecordBufferSize := (FBlobCacheOffset + (BlobFieldCount * SizeOf(TFIBBlobStream)));

      if UniDirectional then
        FBufferChunkSize := FRecordBufferSize * UniCache
      else
        FBufferChunkSize := FRecordBufferSize * BufferChunks;
      FIBAlloc(FBufferCache, FBufferChunkSize, FBufferChunkSize);
      FIBAlloc(FOldBufferCache, FBufferChunkSize, FBufferChunkSize);
      FBPos := 0;
      FOBPos := 0;
      FBEnd := 0;
      FOBEnd := 0;
      FCacheSize := FBufferChunkSize;
      FOldCacheSize := FBufferChunkSize;

      FIBAlloc(FModelBuffer, RecordDataLength(FQSelect.Current.Count),
                             FRecordBufferSize);

      FOldBuffer := AllocRecordBuffer;
    end else
      FQSelect.ExecQuery;
  finally
   {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := iCurScreenState;
   {$ENDIF} 
  end;
end;

procedure TFIBCustomDataSet.InternalPost;
var
  Qry: TFIBQuery;
  Buff: PChar;
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState: Integer;
 {$ENDIF} 
  bInserting: Boolean;
begin
 {$IFNDEF FOR_CONSOLE}
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
 {$ENDIF}
  try
    (* Cannot update a record without a FQUpdate/FQInsert query existing. *)
    Buff := GetActiveBuf;
    CheckEditState;
    CheckInsertMode(Buff);
    if (State = dsInsert) then begin
      bInserting := True;
      Qry := FQInsert;
      PRecordData(Buff)^.rdUpdateStatus := usInserted;
      PRecordData(Buff)^.rdCachedUpdateStatus := cusInserted;
      WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
      FCurrentRecord := PRecordData(Buff)^.rdRecordNumber;
    end else begin
      bInserting := False;
      Qry := FQUpdate;
      if PRecordData(Buff)^.rdCachedUpdateStatus = cusUnmodified then begin
        PRecordData(Buff)^.rdUpdateStatus := usModified;
        PRecordData(Buff)^.rdCachedUpdateStatus := cusModified;
      end else if PRecordData(Buff)^.
                    rdCachedUpdateStatus = cusUninserted then begin
        PRecordData(Buff)^.rdCachedUpdateStatus := cusInserted;
        Dec(FDeletedRecords);
      end;
    end;
    if (not CachedUpdates) then
      InternalPostRecord(Qry, Buff)
    else begin
      WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
      FUpdatesPending := True;
    end;
    if bInserting then
      Inc(FRecordCount);
  finally
   {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := iCurScreenState;
   {$ENDIF} 
  end;
end;

procedure TFIBCustomDataSet.InternalRefresh;
begin
  inherited;
  InternalRefreshRow(FQRefresh,GetActiveBuf);
end;

procedure TFIBCustomDataSet.InternalSetToRecord(Buffer: PChar);
begin
  InternalGotoBookmark(@(PRecordData(Buffer)^.rdRecordNumber));
end;

function TFIBCustomDataSet.IsCursorOpen: Boolean;
begin
  result := FOpen;
end;

function TFIBCustomDataSet.ExtLocate(const KeyFields: String; const KeyValues: Variant;
      Options: TExtLocateOptions): Boolean; //override;
var
  CurBookmark: String;
begin
  DisableControls;
  DoBeforeScroll;  
  DisableScrollEvents;
  Result := false;
  try
    CurBookmark := Bookmark;
    First;
     Result := InternalLocate(KeyFields, KeyValues,
      Options
     );
    if not result then
      Bookmark := CurBookmark;
  finally
    if Result then DoAfterScroll;
    EnableControls;
    EnableScrollEvents;
  end;
end;

function TFIBCustomDataSet.Locate(const KeyFields: string;
  const KeyValues: Variant; Options: TLocateOptions): Boolean;
var
  CurBookmark: String;
begin
  DisableControls;
  DoBeforeScroll;
  DisableScrollEvents;
  Result := false;
  try
    CurBookmark := Bookmark;
    First;
    result := InternalLocate(KeyFields, KeyValues,
     TExtLocateOptions(Options)
    );
    if not result then
      Bookmark := CurBookmark;
  finally
    if Result then DoAfterScroll;  
    EnableControls;
    EnableScrollEvents;    
  end;
end;


function TFIBCustomDataSet.Lookup(const KeyFields: string; const KeyValues: Variant;
  const ResultFields: string): Variant;
var
  CurBookmark: String;
begin
  DisableControls;
  CurBookmark := Bookmark;
  DisableScrollEvents;
  try
    First;
     if InternalLocate(KeyFields, KeyValues, []) then begin
      if (ResultFields <> '') then
        result := FieldValues[ResultFields]
      else
        result := Null;
    end else
      result := Null;
  finally
    EnableScrollEvents;
    Bookmark := CurBookmark;
    EnableControls;
  end;
end;

procedure TFIBCustomDataSet.SetBookmarkData(Buffer: PChar; Data: Pointer);
begin
  if Data<>nil then
   PRecordData(Buffer)^.rdRecordNumber := PInteger(Data)^;
end;

{$IFDEF VER120}
procedure TFIBCustomDataSet.SetBlockReadSize(Value: Integer);
begin
  if Value <> BlockReadSize then begin
    inherited;
    Next; // first record show twice prevent see TDataPacketWriter.WriteDataSet
  end;
end;
{$ENDIF}

procedure TFIBCustomDataSet.SetBookmarkFlag(Buffer: PChar; Value: TBookmarkFlag);
begin
  PRecordData(Buffer)^.rdBookmarkFlag := Value;
end;

procedure TFIBCustomDataSet.SetCachedUpdates(Value: Boolean);
begin
  if not Value and FCachedUpdates then
    CancelUpdates;
  FCachedUpdates := Value;
end;

procedure TFIBCustomDataSet.SetDataSource(Value: TDataSource);
begin
  if IsLinkedTo(Value) then FIBError(feCircularReference, [nil]);
  if FSourceLink <> nil then
    FSourceLink.DataSource := Value;
end;

procedure TFIBCustomDataSet.SetFieldData(Field: TField; Buffer: Pointer);
var
  Buff, TmpBuff: PChar;
begin

 if (Field.FieldNo >-1) or
   not (vTypeDispositionField in [dfRWRecNumber,dfWRecNumber])
 then begin
  Buff := GetActiveBuf;
  if Field.FieldNo < 0 then begin
    TmpBuff := Buff + FRecordSize + Field.Offset;
    Boolean(TmpBuff[0]) := LongBool(Buffer);
    if Boolean(TmpBuff[0]) then
      Move(Buffer^, TmpBuff[1], Field.DataSize);
    WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
  end else begin
    CheckEditState;
    with PRecordData(Buff)^ do begin
      (* If inserting, make sure certain settings are established. *)
      CheckInsertMode(Buff);
      if (Field.FieldNo > 0) and (Field.FieldNo <= rdFieldCount) then begin
        Field.Validate(Buffer);
        if (Buffer = nil) or
           (Field is TFIBStringField) and (PChar(Buffer)[0] = #0) then
          rdFields[Field.FieldNo].fdIsNull := True
        else begin
          Move(Buffer^, Buff[rdFields[Field.FieldNo].fdDataOfs],
                 rdFields[Field.FieldNo].fdDataSize);
          if (rdFields[Field.FieldNo].fdDataType = SQL_TEXT) or
             (rdFields[Field.FieldNo].fdDataType = SQL_VARYING) then
            rdFields[Field.FieldNo].fdDataLength := StrLen(PChar(Buffer));
          rdFields[Field.FieldNo].fdIsNull := False;
          if rdUpdateStatus = usUnmodified then begin
            if State = dsInsert then
              rdUpdateStatus := usInserted
            else
              rdUpdateStatus := usModified;
          end;
          WriteRecordCache(rdRecordNumber, Buff);
          SetModified(True);
        end;
      end;
    end;
  end;
  if not (State in [dsCalcFields, dsFilter, dsNewValue]) then
    DataEvent(deFieldChange, Longint(Field));
 end
 else
   try
    // for RefreshClientFields
    Buff:=AllocRecordBuffer;
    ReadRecordCache(vInspectRecno, Buff, False);
    TmpBuff := Buff + FRecordSize + Field.Offset;
    if LongBool(Buffer) then begin
     Move(Buffer^, TmpBuff[1], Field.DataSize);
     WriteRecordCache(vInspectRecno, Buff);
    end;
   finally
    FreeRecordBuffer(Buff)
   end;
end;

procedure TFIBCustomDataSet.SetRecNo(Value: Integer);
begin
  CheckBrowseMode;
  if (Value < 1) then
    Value := 1
  else if Value > FRecordCount then begin
    InternalLast;
    Value := Min(FRecordCount, Value);
  end;
  if (Value <> RecNo) then begin
    DoBeforeScroll;
    FCurrentRecord := Value - 1;
    Resync([]);
    DoAfterScroll;
  end;
end;

{$IFDEF VER100}

procedure TFIBCustomDataSet.Translate(Src, Dest: PChar; ToOem: Boolean);
begin
  if Src <> nil then
    StrCopy(Dest, Src)
end;
{$ELSE}
 function TFIBCustomDataSet.Translate(Src, Dest: PChar; ToOem: Boolean): Integer;
 begin
  if Src <> nil then
  begin
    StrCopy(Dest, Src);
    Result := StrLen(Dest);
  end else
    Result := 0;
 end;
{$ENDIF}


// Array support

{$IFDEF SUPPORT_ARRAY_FIELD}
function  TFIBCustomDataSet.ArrayFieldValue(Field:TField):Variant;
var qf:TFIBXSQLVAR;
begin
 Result:=false;
 if not Assigned(Field) then Exit;
 qf:=QSelect.FieldByName[Field.FieldName];
 if qf.FIBArray=nil then Exit;
 Result:=qf.FIBArray.GetFieldArrayValues(Field,
         DBHandle,TRHandle
       )
end;

procedure TFIBCustomDataSet.SetArrayValue(Field:TField;Value:Variant);
var qf:TFIBXSQLVAR;
begin
 if not (State in [dsEdit,dsInsert]) then Exit;
 if not Assigned(Field) then Exit;
 if VarIsEmpty( Value ) or VarIsNull( Value ) then begin
   Field.Clear;
   Exit;
 end;
 qf:=QSelect.FieldByName[Field.FieldName];
 if qf.FIBArray=nil then Exit;
 qf.FIBArray.SetFieldArrayValue(Value,Field,
         DBHandle,TRHandle
 )
end;

function TFIBCustomDataSet.GetElementFromValue( Field:TField;
          Indexes:array of integer):Variant;
var qf:TFIBXSQLVAR;
begin
 if not Assigned(Field) then Exit;
 qf:=QSelect.FieldByName[Field.FieldName];
 if qf.FIBArray=nil then Exit;
 Result:=
  qf.FIBArray.GetElementFromField(Field, Indexes,DBHandle,TRHandle );
end;

procedure TFIBCustomDataSet.SetArrayElementValue(Field:TField;Value:Variant;
     Indexes:array of integer
);
var qf:TFIBXSQLVAR;
begin
 if not (State in [dsEdit,dsInsert]) then Exit;
 if not Assigned(Field) then Exit;
 qf:=QSelect.FieldByName[Field.FieldName];
 if qf.FIBArray=nil then Exit;
 qf.FIBArray.PutElementToField(Field,Value,
        Indexes,
        DBHandle,TRHandle
 );
end;
{$ENDIF}

(*
 * Support routines
 *)

function RecordDataLength(n: Integer): Long;
begin
  result := SizeOf(TRecordData) + ((n - 1) * SizeOf(TFieldData));
end;

(*
 * Filter a record into a dataset from another. This allows you to essentially
 * "add" a record to the dataset without causing an insert to occur.
 *
 * "Add" the current record in FromDS to ToDS.
 *   ftByField: Use the FromFieldList and ToFieldList to copy records
 *              to ToDS.
 *              (Assert that Length(FromFieldList) = Length(ToFieldList)).
 *   ftCopy: Assume that the result set of FromDS and ToDS are identical.
 *           That is, the column list in FromDS is identical to that of ToDS.
 *
 *   FilterIn does *not* copy over calculated or lookup fields, unless you
 *   are filtering in by ftCopy.
 *)

procedure FilterIn(FromDS, ToDS: TFIBCustomDataSet; How: TFIBFilterType;
  FromFieldList, ToFieldList: array of Integer;
  RefreshTo: Boolean);
var
  BufferTo, BufferFrom: PRecordData;
  i, cFrom, cTo: Integer;
  ColFrom, ColTo: PChar;
begin
  FromDS.CheckActive;
  ToDS.CheckActive;
  if (FromDS.RecordCount = 0) then
    exit;
  BufferTo := nil;
  BufferFrom := PRecordData(FromDS.GetActiveBuf);
  try
    BufferTo := PRecordData(ToDS.AllocRecordBuffer);
    //FromDS.ReadRecordCache(FromDS.FCurrentRecord, PChar(BufferFrom), False);
    if (How = ftByField) then begin
      if SizeOf(FromFieldList) <> SizeOf(ToFieldList) then
        FIBError(feColumnListsDontMatch, [nil]);
      for i := 0 to SizeOf(FromFieldList) - 1 do
        if (FromFieldList[i] >= 0) and (ToFieldList[i] >= 0) then begin
          cFrom := FromFieldList[i] + 1;
          cTo := ToFieldList[i] + 1;
          if ((BufferFrom^.rdFields[cFrom].fdDataType =
               BufferTo^.rdFields[cTo].fdDataType) and
              (BufferFrom^.rdFields[cFrom].fdDataSize =
               BufferTo^.rdFields[cTo].fdDataSize)) then begin
            ColFrom := PChar(BufferFrom) +
                       BufferFrom^.rdFields[cFrom].fdDataOfs;
            ColTo := PChar(BufferTo) +
                       BufferTo^.rdFields[cTo].fdDataOfs;
            Move(ColFrom^, ColTo^, BufferFrom^.rdFields[cTo].fdDataSize);
          end else
            FIBError(feColumnTypesDontMatch, [cFrom, cTo]);
        end;
    end else begin
      if (BufferFrom^.rdFieldCount <> BufferTo^.rdFieldCount) or
         (FromDS.FRecordSize <> ToDS.FRecordSize) then
        FIBError(feColumnListsDontMatch, [nil]);
      Move(BufferFrom^, BufferTo^, ToDS.FRecordSize);
    end;
    BufferTo^.rdBookmarkFlag := bfCurrent;
    BufferTo^.rdRecordNumber := ToDS.FRecordCount;
    BufferTo^.rdUpdateStatus := usUnmodified;
    BufferTo^.rdCachedUpdateStatus := cusUnmodified;
    BufferTo^.rdSavedOffset := $FFFFFFFF;
    ToDS.WriteRecordCache(ToDS.FRecordCount, PChar(BufferTo));
    ToDS.FCurrentRecord := ToDS.FRecordCount;
    Inc(ToDs.FRecordCount);
    if (RefreshTo and (ToDS.Database <> nil) and
        ToDS.Database.Connected and (ToDS.Transaction <> nil) and
        ToDS.Transaction.Active) then
      ToDS.Refresh;
    ToDS.Last;
  finally
    ToDS.FreeRecordBuffer(PChar(BufferTo));
  end;
end;

(*
 * This is analogous to FilterIn. It allows you to remove a record from a
 * DataSet without deleting it.
 *
 * "Remove" the current record from FromDS. (It marks the record deleted
 * without causing a post.
 *)
procedure FilterOut(FromDS: TFIBCustomDataSet);
var
  BufferFrom: PRecordData;
begin
  FromDS.CheckActive;
  FromDS.DisableControls;
  try
    BufferFrom := PRecordData(FromDS.GetActiveBuf);
    BufferFrom^.rdCachedUpdateStatus := cusUnmodified;
    BufferFrom^.rdUpdateStatus := usDeleted;
    Inc(FromDS.FDeletedRecords);
    FromDS.WriteRecordCache(BufferFrom^.rdRecordNumber, PChar(BufferFrom));
    FromDS.Resync([]);
  finally
    FromDS.EnableControls;
  end;
end;

procedure ClearRecordCache(FromDS: TFIBCustomDataSet);
begin
  FromDS.CheckActive;
  FromDS.DisableControls;
  try
    with FromDS do begin
      FCurrentRecord := -1;
      FRecordCount := 0;
      FDeletedRecords := 0;

      FBPos := 0;
      FOBPos := 0;
      FCacheSize := 0;
      FOldCacheSize := 0;
      FBEnd := 0;
      FOBEnd := 0;
      FIBAlloc(FBufferCache, 0, FBufferChunkSize);
      FIBAlloc(FOldBufferCache, 0, FBufferChunkSize);
      Resync([]);
    end;
  finally
    FromDS.EnableControls;
  end;
end;

(*
 * Do a quick sort on the current result set in the data set.
 * If bFetchAll, then ensure that all records are fetched before doing
 * the sort.
 * Fields is a list of the fields to sort on.
 * Ordering is a list of booleans specifying DESC or ASC (False, True)
 *
 * Based on randomized quick sort from
 *)




procedure Sort(DataSet: TFIBCustomDataSet; Fields: array of const;
  Ordering: array of Boolean);
var
  r1, r2: Pointer;
  a, v: Variant;
  FieldCount, i: Integer;
//      DataSet
  Options:TFIBUpdateRecordTypes;
  ActiveRecNo:Integer;
 {$IFDEF CAN_DYN_ARRAY}
  arFields:array of TField;
 {$ENDIF}
// *******
  function EmptyVar(v: Variant): Boolean;
  begin
    result := VarIsNull(v) or VarIsEmpty(v);
  end;

  // <0 early, 0 equal, >0 later.
  function Compare(x, y: Variant): Integer;
  var
    i: Integer;
  begin
    i := 0;
    result := 0;
    while (i < FieldCount) do begin
      if EmptyVar(x[i]) or (x[i] < y[i]) then begin
        if (High(Ordering)<=i) and not Ordering[I] then result := 1 else
result := -1;
        break;
      end else
      if EmptyVar(y[i]) or (x[i] > y[i]) then begin
        if (High(Ordering)<=i) and not Ordering[I] then result := -1 else
result := 1;
        break;
      end;
      Inc(i);
    end;
  end;

  procedure Swap(p, q: Integer);
  var
    t: Variant;
  begin
    if p=q then Exit;
    if (Compare(a[p],a[q] )=0)   then Exit;
    if ActiveRecNo=p then
      ActiveRecNo:=q
    else if ActiveRecNo=q then
      ActiveRecNo:=p;
    t := a[p];
    a[p] := a[q];
    a[q] := t;
    DataSet.ReadRecordCache(p, r1, False);
    DataSet.ReadRecordCache(q, r2, False);
    PRecordData(r1)^.rdRecordNumber := q;
    PRecordData(r2)^.rdRecordNumber := p;
    DataSet.WriteRecordCache(p, PChar(r2));
    DataSet.WriteRecordCache(q, PChar(r1));
  end;

  function RandomizedPartition(p, r: Integer): Integer;
  var
    x: Variant;
    z:integer;
  begin
    result := 0;
    z:=RandomInteger(p, r);

    Swap(p, z);
    x := a[p];
    Inc(r); Dec(p);
    while True do begin
      repeat Dec(r) until  (r=-1) or (Compare(a[r], x) <= 0) ;
      repeat Inc(p) until (Compare(a[p], x) >= 0) or (p=DataSet.FRecordCount-1);
      if p < r then begin
         Swap(p, r)
      end
      else
      begin
        Result := r;
        break;
      end
    end;
  end;

  procedure RandomizedQuickSort(p, r: Integer);
  var
    q: Integer;
  begin

    if (p < r) then begin
      q := RandomizedPartition(p, r);
      if (q<>r) then
       RandomizedQuickSort(p, q);
      if (q+1<>p) then
       RandomizedQuickSort(q + 1, r);
    end;
  end;
begin
  // Load up the array a


  DataSet.CheckActive;
  {$IFNDEF FOR_CONSOLE}
  Screen.Cursor := FIBHourglassCursor;
  {$ENDIF}
  r1 := nil;
  r2 := nil;
  ActiveRecNo:=DataSet.RecNo-1;
  DataSet.DisableControls;
  DataSet.FetchAll;
  //     DataSet
  if DataSet.FRecordCount<2 then
  begin
    DataSet.EnableControls;
    {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := crDefault;
    {$ENDIF}
    Exit;
  end;
  //      DataSet  
  Options:=DataSet.UpdateRecordTypes;
  DataSet.UpdateRecordTypes:=[cusUnModified,cusModified,cusInserted,
                                cusUnInserted,cusDeleted];
  FieldCount := High(Fields) - Low(Fields) + 1;
  {$IFDEF CAN_DYN_ARRAY}
   SetLength(arFields,FieldCount);
   for i := Low(Fields) to High(Fields) do
       case Fields[i].VType of
        vtInteger: arFields[i]:=DataSet.Fields[Fields[i].VInteger];
        vtObject :
        begin
          if not(Fields[i].vObject is TField) or
             (TField(Fields[i].vObject).DataSet<>DataSet)
          then raise Exception.Create(SCantSort+IntToStr(i)+']');
         arFields[i]:=TField(Fields[i].VObject);
        end;
        vtAnsiString :
        begin
          if DataSet.FindField(string(Fields[i].vString))=nil
          then raise
           Exception.Create(SCantSort+IntToStr(i)+']');
          arFields[i]:=
           DataSet.FieldByName(string(Fields[i].vString))
        end;
       else
   raise
     Exception.Create(SCantSort+IntToStr(i)+']');
   end;
  {$ENDIF}
  try
    r1 := DataSet.AllocRecordBuffer;
    r2 := DataSet.AllocRecordBuffer;
    a := VarArrayCreate([0, DataSet.FRecordCount-1], varVariant);
    DataSet.First;

    while not DataSet.Eof do begin
      v := VarArrayCreate([0, FieldCount-1], varVariant);
  {$IFDEF CAN_DYN_ARRAY}
     for i := Low(Fields) to High(Fields) do
       v[i ]:=arFields[i].Value;
  {$ELSE}
      for i := Low(Fields) to High(Fields) do
       case Fields[i].VType of
        vtInteger: v[i - Low(Fields)]:=
                     DataSet.Fields[Fields[i].VInteger].Value;
        vtObject :
        begin
          if not(Fields[i].vObject is TField) or
             (TField(Fields[i].vObject).DataSet<>DataSet)
          then raise Exception.Create(SCantSort+IntToStr(i)+']');
         v[i - Low(Fields)] :=TField(Fields[i].VObject).Value;
        end;
        vtAnsiString :
        begin
          if DataSet.FindField(string(Fields[i].vString))=nil
          then raise
           Exception.Create(SCantSort+IntToStr(i)+']');
          v[i - Low(Fields)]:=
           DataSet.FieldByName(string(Fields[i].vString)).Value
        end;
       else
       raise
           Exception.Create(SCantSort+IntToStr(i)+']');
       end;
   {$ENDIF}
      a[DataSet.RecNo-1] := v;
      DataSet.Next;
    end;
    RandomizedQuickSort(0, DataSet.RecNo-1);
  finally
    DataSet.FreeRecordBuffer(PChar(r1));
    DataSet.FreeRecordBuffer(PChar(r2));
    DataSet.UpdateRecordTypes:=Options;
    DataSet.RecNo:=ActiveRecNo+1;
 {$IFNDEF FOR_CONSOLE}
    Screen.Cursor := crDefault;
 {$ENDIF}
    DataSet.EnableControls;
  end;
end;



(* TFIBDSBlobStream *)
constructor TFIBDSBlobStream.Create(AField: TField; ABlobStream: TFIBBlobStream;
                                    Mode: TBlobStreamMode);
begin
  FField := AField;
  FBlobStream := ABlobStream;
  FBlobStream.Seek(0, soFromBeginning);
  if (Mode = bmWrite) then
    FBlobStream.Truncate;
end;

function TFIBDSBlobStream.Read(var Buffer; Count: Longint): Longint;
begin
  result := FBlobStream.Read(Buffer, Count);
end;

function TFIBDSBlobStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  result := FBlobStream.Seek(Offset, Origin);
end;

procedure TFIBDSBlobStream.SetSize(NewSize: Longint);
begin
  FBlobStream.SetSize(NewSize);
end;

function TFIBDSBlobStream.Write(const Buffer; Count: Longint): Longint;
begin
  if not (FField.DataSet.State in [dsEdit, dsInsert]) then
    FIBError(feNotEditing, [nil]);
  TFIBDataSet(FField.DataSet).RecordModified(True);
  result := FBlobStream.Write(Buffer, Count);
  TFIBCustomDataSet(FField.DataSet).DataEvent(deFieldChange, Longint(FField));
end;



end.




