{********************************************************}
{                                                        }
{       Borland Deplphi                                  }
{       InterBase EventAlerter components                }
{       Copyright (c) 1995,98 Inprise Corporation        }
{                                                        }
{       Written by:                                      }
{         James Thorpe                                   }
{         CSA Australasia                                }
{         Compuserve: 100035,2064                        }
{         Internet:   csa@csaa.com.au                    }
{                                                        }
{       Adapted for FreeIBComponents                     }
{         12 Nov 1998 by Belokon Andre                   }
{         SoftHouseLabs@usa.net                          }
{                                                        }
{ Added   Oleg Kukarthev update                          }
{         04.1999 by 04.1999 by Serge Buzadzhy           }
{     email:  serge_buzadzhy@mail.ru,                    }
{             FidoNet: 2:467/44.37                       }
{                                                        }
{********************************************************}

unit pFIBEvent;

interface
{$I FIBPlus.INC}

uses
  SysUtils, Windows, Messages, Classes,
  DB, ibase,IB_Intf, ib_externals, FIB,  FIBDatabase
  , Forms
  ;

const
  MaxEvents = 15;
  EventLength = 64;

type

  TEventAlert = procedure( Sender: TObject; EventName: string; EventCount: longint;
                           var CancelAlerts: Boolean) of object;

  TEventBuffer = array[ 0..MaxEvents-1, 0..EventLength-1] of char;

  TpFIBEventer = class(TComponent)
  private
    FDatabase: TFIBDatabase;
    FEvents: TStrings;
    FOnEventAlert: TEventAlert;
    FQueued: Boolean;
    FRegistered: Boolean;
    Buffer: TEventBuffer;
    Changing: Boolean;
    EventThreadHandle : THandle; //Added Source
    FWindowHandle: HWND; //Added Source
    CS: TRTLCriticalSection;
    EventBuffer: PChar;
    EventBufferLen: integer;
    EventID: Long;
    ProcessingEvents: Boolean;
    RegisteredState: Boolean;
    ResultBuffer: PChar;
    procedure DoQueueEvents;
    procedure EventChange( sender: TObject);
    procedure ValidateDatabase(Database: TFIBDatabase);
    procedure WndProc(var Msg: TMessage);
  protected
    procedure HandleEvent;
    procedure Loaded; override;
    procedure Notification( AComponent: TComponent; Operation: TOperation); override;
    procedure SetEvents( value: TStrings);
    procedure SetDatabase( value: TFIBDatabase);
    procedure SetRegistered( value: boolean);
  public
    constructor Create( AOwner: TComponent); override;
    destructor Destroy; override;
    procedure CancelEvents;
    procedure QueueEvents;
    procedure RegisterEvents;
    procedure UnRegisterEvents;
    property  Queued: Boolean read FQueued;
  published
    property Database: TFIBDatabase read FDatabase write SetDatabase;
    property Events: TStrings read FEvents write SetEvents;
    property Registered: Boolean read FRegistered write SetRegistered;
    property OnEventAlert: TEventAlert read FOnEventAlert write FOnEventAlert;
  end;

  EFIBEAlertError = class( Exception);

implementation

resourcestring
  SNoEventsRegistered  = 'You must register events before queueing them';
  SInvalidCancellation = 'You cannot call CancelEvents from within an OnEventAlert handler';
  SInvalidEvent        = 'Invalid blank event added to EventAlerter events list';
  SInvalidQueueing     = 'You cannot call QueueEvents from within an OnEventAlert handler';
  SInvalidRegistration = 'You cannot Register or Unregister events from within an OnEventAlert handler';  SMaximumEvents       = 'You can only register 15 events per EventAlerter';

// TpFIBEventer

procedure HandleEvent( param: integer); stdcall;
begin
  // don't let exceptions propogate out of thread
  try
    TpFIBEventer( param).HandleEvent;
  except
    Application.HandleException( nil);
  end;
end;

procedure IBEventCallback( IBEventAlerter: TpFIBEventer; BufferLength: short; UpdatedBuffer: PChar);
cdecl;
begin
 // Added   Oleg Kukarthev update
    if Assigned(IBEventAlerter) and (BufferLength > 0) then
    with IBEventAlerter do begin
        if Changing then begin
          if EventThreadHandle <> 0 then
            CloseHandle(EventThreadHandle);
          DuplicateHandle(GetCurrentProcess, GetCurrentThread, GetCurrentProcess,
            @EventThreadHandle, SYNCHRONIZE or 1{THREAD_TERMINATE}, False, 0);
        end; { if Changing }
        try
          EnterCriticalSection(CS);
          Move(UpdatedBuffer^, ResultBuffer^, BufferLength);
        finally
          LeaveCriticalSection(CS);
        end;
        PostMessage(FWindowHandle, WM_User + 1, 0, 0);
    end;
end;

constructor TpFIBEventer.Create( AOwner: TComponent);
begin
  inherited Create( AOwner);
  FWindowHandle := AllocateHWnd(WndProc);
  InitializeCriticalSection( CS);
  FEvents := TStringList.Create;
  with TStringList( FEvents) do
  begin
    OnChange := EventChange;
    Duplicates := dupIgnore;
  end;
end;

destructor TpFIBEventer.Destroy;
begin
  UnregisterEvents;
  if EventThreadHandle <> 0 then begin
 //  Added   Oleg Kukarthev update
   WaitForSingleObject(EventThreadHandle, 10000{Infinite});
   CloseHandle(EventThreadHandle);
   DeleteCriticalSection(CS);
  end;

  SetDatabase( nil);
  FEvents.Free;
  DeallocateHWnd(FWindowHandle);
  inherited Destroy;
end;

procedure TpFIBEventer.ValidateDatabase(Database: TFIBDatabase);
begin
  if not Assigned(Database) or not Database.Connected then
    FIBError(feDatabaseClosed, [nil]);
end;

procedure TpFIBEventer.CancelEvents;
var
  dbHandle: TISC_DB_HANDLE;
begin
  if ProcessingEvents then
    raise EFIBEAlertError.Create( SInvalidCancellation);
  if FQueued then
  begin
    try
      // wait for event handler to finish before cancelling events
      EnterCriticalSection( CS);
      ValidateDatabase( Database);
      FQueued := false;
      Changing := true;
      dbHandle := Database.Handle;
      if Isc_Cancel_Events( StatusVector, @dbHandle, @EventID) <> 0 then
        IBError(Self);
    finally
      LeaveCriticalSection( CS);
    end;
  end;
end;

procedure TpFIBEventer.DoQueueEvents;
var
  callback: pointer;
  dbHandle: Tisc_db_handle;
begin
  ValidateDatabase( DataBase);
  callback := @IBEventCallback;
  dbHandle := Database.Handle;
  if Isc_Que_Events( StatusVector, @dbHandle, @EventID, EventBufferLen,
        EventBuffer, Tisc_callback(callback), PVoid(self)) <> 0 then
    IBError(Self);
  FQueued := true;
end;

procedure TpFIBEventer.EventChange( sender: TObject);
begin
  // check for blank event
  if TStringList(Events).IndexOf( '') <> -1 then
    raise EFIBEAlertError.Create( SInvalidEvent);
  // check for too many events
  if Events.Count > MaxEvents then
  begin
    TStringList(Events).OnChange := nil;
    Events.Delete( MaxEvents);
    TStringList(Events).OnChange := EventChange;
    raise EFIBEAlertError.Create( SMaximumEvents);
  end;
  if Registered then RegisterEvents;
end;

procedure TpFIBEventer.HandleEvent;
var
  CancelAlerts: Boolean;
  i: integer;
  status: PISC_STATUS;
begin
  try
    // prevent modification of vital data structures while handling events
    EnterCriticalSection( CS);
    ProcessingEvents := true;
    status:=StatusVector;
    Isc_Event_Counts( status, EventBufferLen, EventBuffer, ResultBuffer);
    CancelAlerts := false;
    if assigned(FOnEventAlert) and not Changing then
    begin
      for i := 0 to Events.Count-1 do
      begin
        try
          if (status^ <> 0) and not CancelAlerts then
            FOnEventAlert( self, Events[Events.Count-i-1], status^, CancelAlerts);
        except
          Application.HandleException( nil);
        end;
        Inc(status);
      end;
    end;
    Changing := false;
    if not CancelAlerts and FQueued then DoQueueEvents;
  finally
    ProcessingEvents := false;
    LeaveCriticalSection( CS);
  end;
end;

procedure TpFIBEventer.Loaded;
begin
  inherited Loaded;
  try
    if RegisteredState then RegisterEvents;
  except
    if csDesigning in ComponentState then
      Application.HandleException( self)
    else raise;
  end;
end;

procedure TpFIBEventer.Notification( AComponent: TComponent;
                                        Operation: TOperation);
begin
  inherited Notification( AComponent, Operation);
  if (Operation = opRemove) and (AComponent = FDatabase) then
  begin
    UnregisterEvents;
    FDatabase := nil;
  end;
end;

procedure TpFIBEventer.QueueEvents;
begin
  if not FRegistered then
    raise EFIBEAlertError.Create( SNoEventsRegistered);
  if ProcessingEvents then
    raise EFIBEAlertError.Create( SInvalidQueueing);
  if not FQueued then
  begin
    try
      // wait until current event handler is finished before queuing events
      EnterCriticalSection( CS);
      DoQueueEvents;
      Changing := true;
    finally
      LeaveCriticalSection( CS);
    end;
  end;
end;

procedure TpFIBEventer.RegisterEvents;
var
  i: integer;
  bufptr: pointer;
  eventbufptr: pointer;
  resultbufptr: pointer;
  buflen: integer;
begin
  ValidateDatabase( Database);
  if csDesigning in ComponentState then FRegistered := true
  else begin
    UnregisterEvents;
    if Events.Count = 0 then exit;
    for i := 0 to Events.Count-1 do
      StrPCopy( @Buffer[i][0], Events[i]);
    i := Events.Count;
    bufptr := @buffer[0];
    eventbufptr :=  @EventBuffer;
    resultBufPtr := @ResultBuffer;
    asm
      mov ecx, dword ptr [i]
      mov eax, dword ptr [bufptr]
      @@1:
      push eax
      add  eax, EventLength
      loop @@1
      push dword ptr [i]
      push dword ptr [resultBufPtr]
      push dword ptr [eventBufPtr]
      call [Isc_Event_Block]
      mov  dword ptr [bufLen], eax
      mov eax, dword ptr [i]
      shl eax, 2
      add eax, 12
      add esp, eax
    end;
    EventBufferlen := Buflen;
    FRegistered := true;
    QueueEvents;
  end;
end;

procedure TpFIBEventer.SetEvents( value: TStrings);
begin
  FEvents.Assign( value);
end;

procedure TpFIBEventer.SetDatabase( value: TFIBDatabase);
begin
  if value <> FDatabase then
  begin
    UnregisterEvents;
    if assigned( value) and value.Connected then ValidateDatabase( value);
    FDatabase := value;
  end;
end;

procedure TpFIBEventer.SetRegistered( value: Boolean);
begin
  if (csReading in ComponentState) then
    RegisteredState := value
  else if FRegistered <> value then
    if value then RegisterEvents else UnregisterEvents;
end;

procedure TpFIBEventer.UnregisterEvents;
begin
  if ProcessingEvents then
    raise EFIBEAlertError.Create( SInvalidRegistration);
  if csDesigning in ComponentState then
    FRegistered := false
  else if not (csLoading in ComponentState) then
  begin
    CancelEvents;
    if FRegistered then
    begin
      Isc_Free( EventBuffer);
      EventBuffer := nil;
      Isc_Free( ResultBuffer);
      ResultBuffer := nil;
    end;
    FRegistered := false;
  end;
end;


procedure TpFIBEventer.WndProc(var Msg: TMessage);
begin
  with Msg do begin
    if Msg = WM_User + 1 then begin
      try
        HandleEvent;
      except
        Application.HandleException(Self);
      end;
      Result := 1;
    end
    {else Result := 0;}
    else if Msg = WM_ActivateApp then Result := 0
    else Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
  end;
end;

end.
