{************************************************************************}
{ FIBPlus - component library  for direct access  to Interbase  databases}
{    FIBPlus is based in part on the product                             }
{    Free IB Components, written by Gregory H. Deatz for                 }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.                     }
{                                         Contact:       gdeatz@hlmdd.com}
{    Copyright (c) 1998-2001 Serge Buzadzhy                              }
{                                         Contact: serge_buzadzhy@mail.ru}
{  Please see the file FIBLicense.txt for full license information.      }
{************************************************************************}


Unit pFIBDataSet;

interface
{$I FIBPlus.INC}

uses
  Windows, Messages, SysUtils, Classes,
  Controls, Forms, Dialogs,
  DB,ibase,IB_Intf, ib_externals,FIB,FIBMiscellaneous,  pFIBDataBase,
  FIBDataBase,FIBDataSet,  FIBQuery,
  pFIBQuery,DSContainer,{$IFDEF FOR_ALL} pFIBProps {$ELSE}sbFIBProps {$ENDIF};

type



  TpSQLKind      =(skModify, skInsert, skDelete,skRefresh);
  TUpdateKinds  = set of TUpdateKind;


  TLockStatus = (lsSuccess,lsDeadLock,lsNotExist,lsMultiply,lsUnknownError);
  TLockErrorEvent=
   procedure(DataSet: TDataSet;LockError:TLockStatus;
    var ErrorMessage :string ;var Action: TDataAction) of object;

  TFakeRefreshKind=(frkEdit,frkInsert);

  TpFIBDataSet = class(TFIBDataSet)
  private
   FDataSet_ID:integer;
   FDefaultFormats:TFormatFields;
   FOnLockError   :TLockErrorEvent;
   FAllowedUpdateKinds:TUpdateKinds;
   vInsertRecno:integer;
   vSaveRefreshSQLTxt:string;
   vUserOnPostError:TDataSetErrorEvent;
   vUserOnDeleteError:TDataSetErrorEvent;
   vInSort        :boolean;
   vInsertRecordRunning:boolean;  //  InsertRecord running
// lists of UpdateObjects
   vUpdates:TList;
   vDeletes:TList;
   vInserts:TList;
//

   vIntQuery :TFIBQuery;
//
   FContainer:TDataSetsContainer;
   FReceiveEvents:Tstrings;
   FOnUserEvent:TUserEvent;

   FAutoUpdateOptions:TAutoUpdateOptions;
   FHasUncommitedChanges:boolean;

   FAllRecordCount:integer;
   procedure SetReceiveEvents(Value:Tstrings);
   procedure SetContainer(Value:TDataSetsContainer);
   //Property access procedures
   function  GetDeleteSQL: TStrings;
   procedure SetDeleteSQL(Value: TStrings);
   function  GetUpdateSQL: TStrings;
   procedure SetUpdateSQL(Value: TStrings);
   function  GetInsertSQL:TStrings;
   procedure SetInsertSQL(Value: TStrings);
   function  GetRefreshSQL:TStrings;
   procedure SetRefreshSQL(Value: TStrings);
   procedure SetRefreshSQLTxt(Value: String);
   function  GetSelectSQL:TStrings;
   procedure SetSelectSQL(Value: TStrings);
   function  GetDefaultFields:boolean;

   // UpdateObjects support
   function  ListForUO(KindUpdate:TUpdateKind):TList;
   procedure SynchroOrdersUO(List:TList);

   //
   procedure PrepareQuery(KindQuery:TUpdateKind);
  protected
    procedure SetPrepareOptions(Value:TpPrepareOptions); override;
    procedure DoOnPostError(DataSet: TDataSet; E: EDatabaseError; var Action: TDataAction); override;
    procedure DoOnDeleteError(DataSet: TDataSet; E: EDatabaseError; var Action: TDataAction);


    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    function GetCanModify: Boolean; override;

    function  IsVisible(Buffer: PChar): Boolean; override;
    procedure InternalPostRecord(Qry: TFIBQuery; Buff: Pointer);override;
    procedure InternalDeleteRecord(Qry: TFIBQuery; Buff: Pointer);override;

    procedure  AddedFilterRecord(DataSet: TDataSet; var Accept: Boolean); virtual; //abstract

    procedure  SetFiltered(Value: Boolean); override;
    procedure  DoBeforeOpen;    override;
    procedure  DoAfterOpen;     override;
    procedure  DoAfterClose;    override;
    procedure  DoBeforeInsert;  override;
    procedure  DoAfterInsert;   override;
    procedure  DoBeforeEdit;    override;
    procedure  DoBeforePost;    override;
    procedure  DoAfterPost;     override;
    procedure  DoBeforeCancel;  override;
    procedure  DoAfterCancel;   override;
    procedure  DoBeforeDelete;  override;
    procedure  DoOnNewRecord;   override;
    procedure  DoAfterDelete;   override;
    procedure  DoAfterEdit;     override;
    procedure  DoAfterScroll;   override;
    procedure  DoBeforeClose;   override;
    procedure  DoBeforeScroll;  override;
    procedure  DoOnCalcFields;  override;
    function   GetRecordCount: Integer; override;
    procedure  UpdateFieldsProps; virtual;
   {$IFNDEF VER130}
     function BCDToCurr(BCD: Pointer; var Curr: Currency): Boolean; override;
     function CurrToBCD(const Curr: Currency; BCD: Pointer; Precision,
      Decimals: Integer): Boolean; override;
   {$ENDIF}
    procedure StartTransaction;

    procedure DoTransactionEnding(Sender:TObject);
    procedure DoTransactionEnded(Sender:TObject);
    procedure ClearModifFlags(Kind:byte);
    procedure CloseProtect;
    function  RaiseLockError(LockError:TLockStatus;ExceptMessage:string):TDataAction;    
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Prepare;override;
    function  CanEdit  : Boolean; override;
    function  CanInsert: Boolean; override;
    function  CanDelete: Boolean; override;
    function  IsSequenced: Boolean; override;        // Scroll bar

    function  ExistActiveUO(KindUpdate:TUpdateKind):boolean;
     // Exist active Update objects

    function  AddUpdateObject(Value:TpFIBUpdateObject):integer;
    procedure RemoveUpdateObject(Value:TpFIBUpdateObject);

    procedure RefreshFilters;
    procedure RefreshClientFields;



    property  DefaultFields:boolean read GetDefaultFields ;
    function  ParamByName(const ParamName:string): TFIBXSQLVAR;
    function  RecordStatus(RecNumber:integer):TUpdateStatus;
    procedure CloneRecord(SrcRecord:integer; IgnoreFields:array of const);
    procedure CloneCurRecord(IgnoreFields:array of const);
    // Cached Routine
    procedure CommitUpdToCach; // Clear CU buffers
    procedure ApplyUpdToBase;  // Send Updates to Base
    procedure CancelUpdates; // hide ancestor method
    function  LockRecord(RaiseErr:boolean):TLockStatus;
    function  FilterRecno:integer;
    function  FN(const FieldName:string):TField; //FindField
    procedure DoSort(Fields: array of const; Ordering: array of Boolean);
    function  ParamNameCount(const ParamName:string):integer;

    procedure ExecUpdateObjects(KindUpdate:TUpdateKind;Buff: Pointer;
     aExecuteOrder:TFIBOrderExecUO);

    procedure DoUserEvent(Sender:Tobject; const UDE:string;var Info:string); dynamic;
    procedure OpenWP(ParamValues: array of Variant);

    function  GenerateSQLText
     (const TableName,KeyFieldNames:string;SK:TpSQLKind):string;
    function  GenerateSQLTextWA
     (const TableName:string;SK:TpSQLKind):string; // Where All
    procedure GenerateSQLs;
    
//AutoUpdate operations
    function  KeyField:TField;
    function  SqlTextGenID:string;
    procedure IncGenerator;

    procedure InsertRecord(const Values: array of const); //hide Ancestor method
    function  PrimaryKeyFields(const TableName:string):string;
    function  AllFields:string;

    procedure FakeRefresh(FromDataSet:TDataSet;Kind:TFakeRefreshKind
     ;FieldMap:Tstrings
    );
    procedure FakeRefreshByArrMap(
     FromDataSet:TDataSet;Kind:TFakeRefreshKind;
     const SourceFields,DestFields:array of String 
    );

    property HasUncommitedChanges:boolean read FHasUncommitedChanges
                                          write FHasUncommitedChanges;
    property  AllRecordCount:integer read FAllRecordCount;
  published
// "overrided" :) properties
    property DeleteSQL: TStrings read GetDeleteSQL write  SetDeleteSQL;
    property UpdateSQL: TStrings read GetUpdateSQL write  SetUpdateSQL;
    property InsertSQL: TStrings read GetInsertSQL write  SetInsertSQL;
    property RefreshSQL:TStrings read GetRefreshSQL write SetRefreshSQL;
    property SelectSQL: TStrings read GetSelectSQL write  SetSelectSQL;
//Added properties
    property  Filtered;
    property  OnFilterRecord;


    property  DefaultFormats:TFormatFields read FDefaultFormats write FDefaultFormats;
    property OnPostError:TDataSetErrorEvent read vUserOnPostError   write vUserOnPostError;
    property OnDeleteError:TDataSetErrorEvent read vUserOnDeleteError   write vUserOnDeleteError;
    property OnLockError:TLockErrorEvent read FOnLockError write FOnLockError;
    property AllowedUpdateKinds:TUpdateKinds  read FAllowedUpdateKinds write FAllowedUpdateKinds
     default [ukModify,ukInsert,ukDelete];
    property OnUserEvent:TUserEvent   read FOnUserEvent write FOnUserEvent;
    property Container:TDataSetsContainer read  FContainer Write SetContainer;
    property ReceiveEvents:Tstrings read  FReceiveEvents write    SetReceiveEvents;
    property AutoUpdateOptions:TAutoUpdateOptions  read FAutoUpdateOptions write FAutoUpdateOptions;
    property DataSet_ID:integer read FDataSet_ID write FDataSet_ID default 0;
  end;


function FieldInArray(Field:TField; Arr: array of const):boolean;



implementation

 uses FIBSQLMonitor,StrUtil,SqlTxtRtns,FIBConsts,pFIBDataInfo,CommonIB;

function UseFormat(const sFormat:string):string;
var pD:integer;
begin
 pD:=Pos('.',sFormat);
 if pD=0 then Result:=sFormat
 else
  Result:=Copy(sFormat,1,Pred(pd))+'.'
end;

//
constructor TpFIBDataSet.Create(AOwner: TComponent);
begin
 inherited Create(AOwner);
 FAllowedUpdateKinds:=[ukModify,ukInsert,ukDelete];
 FDefaultFormats:=TFormatFields.Create;
 vInSort  :=false;
 inherited OnPostError  :=DoOnPostError;
 inherited OnDeleteError:=DoOnDeleteError;

 vUpdates:=TList.Create;
 vDeletes:=TList.Create;
 vInserts:=TList.Create;




 FReceiveEvents     :=TstringList.Create;
 FAutoUpdateOptions :=TAutoUpdateOptions.Create(Self);
 vIntQuery          :=TFIBQuery.Create(Self);
 vInsertRecordRunning    :=false;

 {$IFDEF VER130}
  SparseArrays :=true;
 {$ENDIF}

 FBase.OnTransactionEnding := DoTransactionEnding;
 FBase.OnTransactionEnded  := DoTransactionEnded;
 FHasUncommitedChanges     := false;
 vInFakeRefresh            := false;
 FDataSet_ID               := 0;

 FAllRecordCount:=0;
end;

destructor TpFIBDataSet.Destroy;
begin
 FDefaultFormats.Free; vUpdates.Free; vDeletes.Free;
 vInserts.Free;        FReceiveEvents.Free;
 FAutoUpdateOptions.Free;
 inherited Destroy;
end;

procedure TpFIBDataSet.Notification(AComponent: TComponent; Operation: TOperation); //override;
begin
 inherited Notification(AComponent,Operation);
 case Operation of
  opRemove:  if (AComponent=FContainer)  then FContainer:=nil;
 end;
end;

procedure TpFIBDataSet.DoTransactionEnding(Sender:TObject);
begin
 FHasUncommitedChanges:=FHasUncommitedChanges and
  (Transaction.State in [tsDoRollBack,tsDoRollBackRetaining]);
 inherited DoTransactionEnding(Sender);
end;

procedure TpFIBDataSet.DoTransactionEnded(Sender:TObject);
begin
 inherited DoTransactionEnded(Sender);
 if Transaction.State in  [tsDoCommitRetaining,tsDoRollBackRetaining]
 then
  if  (poProtectedEdit in Options) and not CachedUpdates then
    CloseProtect;  
end;

function TpFIBDataSet.GetCanModify: Boolean; //override;
begin
 result := CanEdit or CanInsert or  CanDelete
end;

function TpFIBDataSet.CanEdit: Boolean; //override;
begin
 Result:=((inherited CanEdit or ExistActiveUO(ukModify) or
          (CachedUpdates and Assigned(OnUpdateRecord))
          )
          and  (ukModify in FAllowedUpdateKinds)
         )
 or vInFakeRefresh
end;

function TpFIBDataSet.CanInsert: Boolean; //override;
begin
 Result:=((inherited CanInsert or ExistActiveUO(ukInsert) or
          (CachedUpdates and Assigned(OnUpdateRecord))
          )
          and (ukInsert in FAllowedUpdateKinds)
         )
  or vInFakeRefresh
end;

function TpFIBDataSet.CanDelete: Boolean; //override;
begin
 Result:=(inherited CanDelete or ExistActiveUO(ukDelete) or
          (CachedUpdates and Assigned(OnUpdateRecord))
          )
   and (ukDelete in FAllowedUpdateKinds)
end;


function  TpFIBDataSet.IsSequenced: Boolean; //override;        // Scroll bar
begin
 Result:=inherited IsSequenced;
 if not Result and not Filtered then
  Result:=FAllRecordCount<>0
end;

procedure TpFIBDataSet.InsertRecord(const Values: array of const); //hide
begin
 vInsertRecordRunning:=true;
 try
  inherited InsertRecord(Values)
 finally
  vInsertRecordRunning:=false
 end
end;

// UpdateObjects support
function TpFIBDataSet.ListForUO(KindUpdate:TUpdateKind):TList;
begin
 Result:=nil;
 case KindUpdate of
  ukModify: Result:=vUpdates;
  ukInsert: Result:=vInserts;
  ukDelete: Result:=vDeletes;
 end;
end;

function  TpFIBDataSet.ExistActiveUO(KindUpdate:TUpdateKind):boolean;
var List:TList;
    i,lc:integer;

begin
  List:=ListForUO(KindUpdate);
  Result:=false;
  if List=nil then Exit;
  lc:=Pred(List.Count) ;
  for i:=0 to lc do begin
    Result:=TpFIBUpdateObject(List[i]).Active;
    if Result then Exit;
  end; 
end;

procedure TpFIBDataSet.SynchroOrdersUO(List:TList);
var i,lc:integer;
begin
 lc:=Pred(List.Count);
 with List do
 for i:=0 to lc do begin
   TpFIBUpdateObject(List[i]).ChangeOrderInList(i);
 end;
end;

function   TpFIBDataSet.AddUpdateObject(Value:TpFIBUpdateObject):integer;
var List:TList;
    OldPos:integer;
begin
 Result:=-1;
 if Value=nil then Exit;
 List:=ListForUO(Value.KindUpdate);
 if List=nil then Exit;
 with List do begin
  OldPos:=IndexOf(Value);
  if OldPos=-1 then begin
   if Value.OrderInList<Count then
    Insert(Value.OrderInList,Value)
   else
    Add(Value);
  end
  else begin
   if Value.OrderInList<Count then
     Move(OldPos,Value.OrderInList)
   else
     Move(OldPos,Pred(Count))
  end;
 end;
 SynchroOrdersUO(List);
 Result:= List.IndexOf(Value)
end;

procedure  TpFIBDataSet.RemoveUpdateObject(Value:TpFIBUpdateObject);
var List:TList;
begin
 if Value=nil then Exit;
 List:=ListForUO(Value.KindUpdate);
 if List<>nil then begin
  List.Remove(Value);
  SynchroOrdersUO(List);
 end;
end;

/// Execute UpdateObjects

{$WARNINGS OFF}
procedure TpFIBDataSet.ExecUpdateObjects(KindUpdate:TUpdateKind;Buff: Pointer;
 aExecuteOrder:TFIBOrderExecUO
);
var List:TList;
    i:integer;
    ms:TMemoryStream;
begin
 List:=ListForUO(KindUpdate);
 if List.Count>0 then try
  ms:=TMemoryStream.Create;
  UpdateBlobInfo(Buff,true);
  for i:=0 to Pred(List.Count) do
   with TpFIBUpdateObject(List[i]) do
   if Active and (ExecuteOrder=aExecuteOrder) and (Trim(SQL.Text)<>'') then begin
     if not Prepared then Prepare;
     SetQueryParams(TpFIBUpdateObject(List[i]),Buff);
     ExecQuery;
   end;
  finally
   ms.Free
  end;

end;
{$WARNINGS ON}
///
procedure TpFIBDataSet.PrepareQuery(KindQuery:TUpdateKind);
var Qry:TFIBQuery;
begin
 AutoStartUpdateTransaction;
 case KindQuery of
  ukModify: Qry:=QUpdate;
  ukInsert: Qry:=QInsert;
 else
//  ukDelete:
  Qry:=QDelete;
 end;
 try
  if (Qry.SQL.Text <> '') and (not Qry.Prepared)
  then
   Qry.Prepare;
 except
  on E: Exception do
   if (E is EFIBInterbaseError) and (EFIBInterbaseError(E).sqlcode = -551) and
    not (csDesigning in ComponentState) then
    FAllowedUpdateKinds:=FAllowedUpdateKinds -[KindQuery]
   else  Raise;
 end;
end;

procedure TpFIBDataSet.Prepare;//override;
var
  iCurScreenState: Integer;
begin
  iCurScreenState := Screen.Cursor;
  Screen.Cursor := FIBHourglassCursor;
  try
    FBase.CheckDatabase;
    FBase.CheckTransaction;
    if FQSelect.SQL.Text <> '' then begin
      if not FQSelect.Prepared then  FQSelect.Prepare;
      if csDesigning in ComponentState then begin
       PrepareQuery(ukModify);
       PrepareQuery(ukInsert);
       PrepareQuery(ukDelete);
      end; 
      FPrepared := True;
      InternalInitFieldDefs;
    end else
      FIBError(feEmptyQuery, [nil]);
  finally
    Screen.Cursor := iCurScreenState;
  end;
end;

function   TpFIBDataSet.GetRecordCount: Integer; //override;
begin
  Result:=inherited GetRecordCount;
  if Result<FAllRecordCount then Result:=FAllRecordCount
end;

procedure TpFIBDataSet.InternalPostRecord(Qry: TFIBQuery; Buff: Pointer);//override;
begin
  AutoStartUpdateTransaction;
  if vInFakeRefresh then Exit;
  UpdateBlobInfo(Buff,true);
  if Qry=QInsert then
   ExecUpdateObjects(ukInsert,Buff,oeBeforeDefault)
  else
   ExecUpdateObjects(ukModify,Buff,oeBeforeDefault);

  if Qry.SQL.Text<>'' then begin
   SetQueryParams(Qry, Buff);
   if Qry.Open then Qry.Close; 
   Qry.ExecQuery;
   if  Qry.SQLType in  [SQLSelect,SQLExecProcedure] then begin
    InternalRefreshRow(Qry,Buff);
   end;
  end;
  if Qry=QInsert then
   ExecUpdateObjects(ukInsert,Buff,oeAfterDefault)
  else
   ExecUpdateObjects(ukModify,Buff,oeAfterDefault);

  PRecordData(Buff)^.rdUpdateStatus := usUnmodified;
  PRecordData(Buff)^.rdCachedUpdateStatus := cusUnmodified;
  SetModified(False);
  WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);

  if not FCachedUpdates then
   AutoCommitUpdateTransaction;
  if (FQRefresh.SQL.Text <> '') and (poRefreshAfterPost in Options) then
  begin
   {$IFDEF VER130}
    DoBeforeRefresh;
   {$ENDIF}
    InternalRefreshRow(FQRefresh,Buff);
   {$IFDEF VER130}
    DoAfterRefresh;
   {$ENDIF}
  end;
  FHasUncommitedChanges:=Transaction.State=tsActive;
end;

procedure TpFIBDataSet.InternalDeleteRecord(Qry: TFIBQuery; Buff: Pointer);//override;
begin
  AutoStartUpdateTransaction;
  ExecUpdateObjects(ukDelete,Buff,oeBeforeDefault);
  if Qry.SQL.Text<>'' then begin
   SetQueryParams(Qry, Buff);
   Qry.ExecQuery;
  end;
  ExecUpdateObjects(ukDelete,Buff,oeAfterDefault);
  with PRecordData(Buff)^ do begin
    rdUpdateStatus := usDeleted;
    rdCachedUpdateStatus := cusUnmodified;
  end;
  WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
  if not FCachedUpdates then
   AutoCommitUpdateTransaction;
  FHasUncommitedChanges:=Transaction.State=tsActive;
end;



procedure TpFIBDataSet.DoOnDeleteError
 (DataSet: TDataSet; E: EDatabaseError; var Action: TDataAction);
begin
 if FContainer<>nil then FContainer.DataSetError(DataSet,deOnDeleteError,E,Action);
 if Assigned(vUserOnDeleteError) then vUserOnDeleteError(DataSet,E,Action);
end;

procedure TpFIBDataSet.DoOnPostError
 (DataSet: TDataSet; E: EDatabaseError; var Action: TDataAction);
var Buff:PChar;
    i:integer;
begin
 inherited;
 if not (poRefreshAfterPost in Options) then begin
   SetRefreshSQLTxt(vSaveRefreshSQLTxt);
 end;
 if FContainer<>nil then FContainer.DataSetError(DataSet,deOnPostError,E,Action);
 if Assigned(vUserOnPostError) then vUserOnPostError(DataSet,E,Action);
 if (Action=daRetry) or (poAppendMode in Options) or (State<>dsInsert) or
  (vInsertRecno=-1) then Exit;

//Error in Insert Mode

  Buff   :=AllocRecordBuffer;
  try
  with PRecordData(GetActiveBuf)^ do
    begin
      for i:=Succ(vInsertRecno) to FRecordCount do      begin
        ReadRecordCache(i, Buff, False);
        PRecordData(Buff)^.rdRecordNumber := PRecordData(Buff)^.rdRecordNumber-1;
        WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
      end;
      ReadRecordCache(FRecordCount, Buff, False);
      rdRecordNumber := FRecordCount;
      FCurrentRecord := FRecordCount;
      CopyRecordBuffer(GetActiveBuf,Buff);
    end
  finally
   FreeRecordBuffer(Buff);
  end;
end;

procedure  TpFIBDataSet.DoBeforePost; //override;
var ActiveBuff,Buff:PChar;
    i:integer;
begin
  if (State=dsInsert) and
   (FAutoUpdateOptions.GenBeforePost) then IncGenerator;
  if (State=dsInsert) then
  if not (poAppendMode in Options) then begin
   Buff      :=AllocRecordBuffer;
   ActiveBuff:=GetActiveBuf;
   try
   with PRecordData(ActiveBuff)^ do
     if (Modified) then
     if (vInsertRecno<>-1 )  then // vInsertRecno=-1 - Is Append
     begin
       //Insert Mode
       for i:=Pred(FRecordCount) downto vInsertRecno do      begin
         ReadRecordCache(i, Buff, False);
         PRecordData(Buff)^.rdRecordNumber :=
          PRecordData(Buff)^.rdRecordNumber+1;
         WriteRecordCache(PRecordData(Buff)^.rdRecordNumber, Buff);
       end;
       rdRecordNumber := vInsertRecno;
       FCurrentRecord := vInsertRecno;
       WriteRecordCache(rdRecordNumber, ActiveBuff);
     end
   finally
    FreeRecordBuffer(Buff);
   end;
  end;
  if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforePost);

  inherited;

  if not (poRefreshAfterPost in Options) then begin
   SetRefreshSQLTxt('');
  end;
end;


procedure  TpFIBDataSet.DoAfterPost; //override;
var ActBuff:PRecordData;
begin
  if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterPost);
  inherited;
  if not (poRefreshAfterPost in Options) then begin
   SetRefreshSQLTxt(vSaveRefreshSQLTxt);
  end;
  if  (poProtectedEdit in Options) and not CachedUpdates then begin
   ActBuff:=PRecordData(GetActiveBuf);
   if ActBuff<>nil then begin
    with ActBuff^ do
     rdUpdateStatus:=usModified; //   CommitRetaining
    SynchronizeBuffers(false,true)
   end
  end;
  if AutoCommit and CachedUpdates then begin
   ApplyUpdToBase;
   AutoCommitUpdateTransaction;
   CommitUpdToCach
  end;
  if (dcForceMasterRefresh in FDetailConditions) then  RefreshMasterDS;
end;



procedure  TpFIBDataSet.DoBeforeCancel;//override;
begin
 inherited;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeCancel); 
 if (State <> dsInsert) then vInsertRecno:=-1;
end;

procedure  TpFIBDataSet.DoAfterCancel; //override;
begin
 if (vInsertRecno<>-1) then SetRecNo(vInsertRecno+1);
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterCancel); 
 inherited;
end;

procedure  TpFIBDataSet.DoBeforeDelete;//override;
begin
 if  not CanDelete then Abort;
 if not CachedUpdates then
  PrepareQuery(ukDelete);
 if  not CanDelete then Abort;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeDelete);
 inherited;
end;

procedure  TpFIBDataSet.DoBeforeInsert; //override;
begin
 if not CanInsert then Abort;
 if not CachedUpdates then
  PrepareQuery(ukInsert);
 if not CanInsert then Abort;
 vInsertRecno:=PRecordData(GetActiveBuf)^.rdRecordNumber;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeInsert);
 if Assigned(BeforeInsert) then BeforeInsert(Self);
end;

procedure  TpFIBDataSet.DoAfterInsert;
var 
    Buffer:PChar;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterInsert);
 inherited;
 Buffer:=GetActiveBuf;
 if ((PRecordData(Buffer)^.rdRecordNumber=-1)
  and not vInsertRecordRunning)
  or (PRecordData(Buffer)^.rdBookmarkFlag=bfEof)
 then vInsertRecno:=-1;
end;

procedure  TpFIBDataSet.DoAfterEdit;      //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterEdit);
 inherited
end;

procedure TpFIBDataSet.DoAfterDelete; //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterDelete);
 if AutoCommit and CachedUpdates then begin
   ApplyUpdToBase;
   AutoCommitUpdateTransaction;
   CommitUpdToCach
 end;
 inherited
end;

procedure  TpFIBDataSet.DoAfterScroll;   //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterScroll);
 inherited
end;

procedure TpFIBDataSet.DoBeforeClose; //override;
begin
 if DisableCOCount>0 then Exit;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeClose);
 inherited
end;

procedure TpFIBDataSet.DoOnCalcFields;   //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deOnCalcFields);
 inherited
end;

procedure TpFIBDataSet.DoBeforeScroll;  //override;
begin
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeScroll);
 inherited
end;

procedure  TpFIBDataSet.DoOnNewRecord; //override;
var i:integer;
    de:string;
    vDifferenceTime:double;
begin
 if (not FAutoUpdateOptions.GenBeforePost) then IncGenerator;
 if DataBase is TpFIBDataBase then
  vDifferenceTime:=TpFIBDataBase(DataBase).DifferenceTime
 else
  vDifferenceTime:=0;
 for i:=0 to Pred(FieldCount) do
  with Fields[i] do  begin
   if (Fields[i]=KeyField) and
    AutoUpdateOptions.SelectGenId then Continue;
   de:=UpperCase(Trim(DefaultExpression));
   if de<>'' then
    if (de='NOW') or (de='CURRENT_TIME') or
     ((de='CURRENT_TIMESTAMP'))
     then
     asDateTime:=Now-vDifferenceTime
    else
    if (de='TODAY') or (de='CURRENT_DATE') then
     asDateTime:=Trunc(Now-vDifferenceTime)
    else 
    if (de='TOMORROW') then
     asDateTime:=Trunc(Now-vDifferenceTime)+1
    else 
    if (de='YESTERDAY') then
     asDateTime:=Trunc(Now-vDifferenceTime)-1
    else
    if de<>'NULL' then
     asString:=DefaultExpression;
  end;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deOnNewRecord);
 inherited DoOnNewRecord
end;

function
 TpFIBDataSet.RaiseLockError(LockError:TLockStatus;ExceptMessage:string):TDataAction;
begin
 Result:=daFail;
 if Assigned(FOnLockError) then FOnLockError(Self,LockError,ExceptMessage,Result);
 case Result of
  daFail : raise Exception.Create(ExceptMessage);
  daAbort: Abort
 end
end;

procedure TpFIBDataSet.DoBeforeEdit; //override;
begin
  if not CanEdit then Abort;
  if not CachedUpdates then
   PrepareQuery(ukModify);   
  if  (poProtectedEdit in Options) then
   if UpdateStatus = usUnModified then begin
    InternalRefresh;
    LockRecord(true);
   end;
  if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeEdit);
  inherited;
end;


procedure TpFIBDataSet.DoBeforeOpen;
begin
 if DisableCOCount>0 then Exit;
 StartTransaction;
 if (psApplyRepositary in PrepareOptions) and
   (DataSet_ID>0) then
 ListDataSetInfo.LoadDataSetInfo(Self);
 if FContainer<>nil then FContainer.DataSetEvent(Self,deBeforeOpen);
 inherited DoBeforeOpen;
 if psAskRecordCount in PrepareOptions then begin
  FAllRecordCount:=RecordCountInQuery(QSelect)
 end
 else
  FAllRecordCount:=0;

end;


procedure  TpFIBDataSet.DoAfterClose;// override;
begin
 if DisableCOCount>0 then Exit;
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterClose);
 inherited DoAfterClose;
end;

function  TpFIBDataSet.GetSelectSQL:TStrings;
begin
 Result:=inherited SelectSQL
end;

procedure TpFIBDataSet.SetSelectSQL(Value: TStrings);
begin
 if Active and (csDesigning in ComponentState) then Close;
 inherited SelectSQL:=Value
end;

function TpFIBDataSet.GetDeleteSQL: TStrings;
begin
 Result:=inherited DeleteSQL
end;

procedure TpFIBDataSet.SetDeleteSQL(Value: TStrings);
begin
 if not (poAllowChangeSqls in FOptions) then
  inherited DeleteSQL:=Value
 else
  try
   QDelete.OnSQLChanging:=nil;
   QDelete.SQL.Assign(Value);
  finally
   QDelete.OnSQLChanging:=SQLChanging
  end;
end;

function TpFIBDataSet.GetUpdateSQL: TStrings;
begin
 Result:=inherited UpdateSQL
end;

procedure TpFIBDataSet.SetUpdateSQL(Value: TStrings);
begin
 if not (poAllowChangeSqls in FOptions) then
  inherited UpdateSQL:=Value
 else
  try
   QUpdate.OnSQLChanging:=nil;
   QUpdate.SQL.Assign(Value);
  finally
   QUpdate.OnSQLChanging:=SQLChanging
  end;
end;

function TpFIBDataSet.GetInsertSQL: TStrings;
begin
 Result:=inherited InsertSQL
end;


procedure TpFIBDataSet.SetInsertSQL(Value: TStrings);
begin
 if not (poAllowChangeSqls in FOptions) then
  inherited InsertSQL:=Value
 else
  try
   QInsert.OnSQLChanging:=nil;
   QInsert.SQL.Assign(Value);
  finally
   QInsert.OnSQLChanging:=SQLChanging
  end;
end;

function TpFIBDataSet.GetRefreshSQL: TStrings;
begin
 Result:=inherited RefreshSQL
end;

procedure TpFIBDataSet.SetRefreshSQL(Value: TStrings);
begin
 if not (poAllowChangeSqls in FOptions) then
  inherited RefreshSQL:=Value
 else
  try
   QRefresh.OnSQLChanging:=nil;
   QRefresh.SQL.Assign(Value);
  finally
   QRefresh.OnSQLChanging:=SQLChanging
  end;
end;

procedure TpFIBDataSet.SetRefreshSQLTxt(Value: String);
begin
  try
   vSaveRefreshSQLTxt:=QRefresh.SQL.Text;
   QRefresh.OnSQLChanging:=nil;          
   QRefresh.SQL.Text:=Value;
  finally
   QRefresh.OnSQLChanging:=SQLChanging
  end;                     
end;

procedure TpFIBDataSet.SetPrepareOptions(Value:TpPrepareOptions);
var NeedUpdateFieldsProps:boolean;
begin
 NeedUpdateFieldsProps:=(psApplyRepositary in Value-FPrepareOptions);
 FPrepareOptions:=Value;
 if NeedUpdateFieldsProps then UpdateFieldsProps
end;

function  TpFIBDataSet.GetDefaultFields:boolean;
begin
   Result:= not Active and (FieldCount=0);
   if not Result then Result:=inherited DefaultFields
end;


procedure TpFIBDataSet.StartTransaction;
begin
 if DataBase=nil then
  if Transaction<>nil then  DataBase:=Transaction.DefaultDataBase;
 if Transaction=nil then
  if DataBase<>nil then  Transaction:=DataBase.DefaultTransaction;
 if poStartTransaction in Options then
  if (Transaction<>nil) and not Transaction.Active  then  Transaction.StartTransaction;
end;

//
function  TpFIBDataSet.ParamNameCount(const ParamName:string):integer;
var i:integer;
begin
 Result:=0;
 for i := 0 to Params.Count  do
  if Params[i].Name=ParamName then Inc(Result)
end;

function  TpFIBDataSet.ParamByName(const ParamName:string): TFIBXSQLVAR;
begin
  StartTransaction;
  Result:=Params.ByName[ParamName]
end;

procedure TpFIBDataSet.CancelUpdates;
var i   :integer;
begin
// Cancel Updates
   if not FUpdatesPending then Exit;
   for i:=0 to Pred(FRecordCount) do InternalRevertRecord(i);
   FUpdatesPending:=false;
   RefreshFilters
end;

procedure TpFIBDataSet.ClearModifFlags(Kind:byte);
var i   :integer;
    Buff:Pchar;
begin
// Commit Updates
 Buff:=AllocRecordBuffer;
 try
   for i:=0 to Pred(FRecordCount) do begin
    ReadRecordCache(i, Buff, False);
    with PRecordData(Buff)^ do begin
     case Kind of
      0: if rdCachedUpdateStatus =cusUnmodified then Continue;
      1: if (rdUpdateStatus in [usUnModified,usDeleted]) then Continue;
     end; //case
     if not((Kind=0) and (rdCachedUpdateStatus=cusDeleted)) then
      rdUpdateStatus:=usUnModified;
     rdCachedUpdateStatus:=cusUnmodified;      
     WriteRecordCache(i, Buff);
     SaveOldBuffer(Buff)
    end;
   end;
   FUpdatesPending:=false;
 finally
  FreeRecordBuffer(Buff);
 end;
 SynchronizeBuffers(false,false)
end;

procedure TpFIBDataSet.CloseProtect;
begin
 if Active then ClearModifFlags(1)
end;

//
procedure TpFIBDataSet.CommitUpdToCach;
begin
 SynchronizeBuffers(false,true);
 ClearModifFlags(0)
end;

procedure TpFIBDataSet.ApplyUpdToBase;
var i   :integer;
    Buff:Pchar;
    UpdateKind: TUpdateKind;
    UpdateAction: TFIBUpdateAction;
    cus:TCachedUpdateStatus;
    us :TUpdateStatus;
begin
 if State in [dsEdit, dsInsert] then  Post;
 StartTransaction;
 Buff:=AllocRecordBuffer;
 try
   for i:=0 to Pred(FRecordCount) do begin
    ReadRecordCache(i, Buff, False);
    with PRecordData(Buff)^ do begin
     if rdCachedUpdateStatus=cusUnmodified then Continue;
     us :=rdUpdateStatus;
     cus:=rdCachedUpdateStatus;
     FUpdatesPending:=true;
     case rdCachedUpdateStatus of
        cusModified:
          UpdateKind := ukModify;
        cusInserted:
          UpdateKind := ukInsert;
        else
          UpdateKind := ukDelete;
     end;
     try
       vTypeDispositionField:=dfRRecNumber;
       vInspectRecno     :=i;
       if (Assigned(FOnUpdateRecord)) then begin
        UpdateAction := uaFail;
        FOnUpdateRecord(Self, UpdateKind, UpdateAction);
       end else
        UpdateAction := uaApply;
       vTypeDispositionField:=dfNormal;
      Except
       on E: EFIBError do begin
            UpdateAction := uaFail;
            if Assigned(FOnUpdateError) then
              FOnUpdateError(Self, E, UpdateKind, UpdateAction);
            case UpdateAction of
              uaFail: raise;
              uaAbort: raise EAbort(E.Message);
       end;
        case UpdateAction of
        uaFail:
          FIBError(feUserAbort, [nil]);
        uaAbort:
          raise EAbort.Create('User abort');
        end;
       end;
      end;
      while (UpdateAction in [uaApply, uaRetry]) do begin
        try
          case rdCachedUpdateStatus of
            cusModified:
             if CanEdit then
               InternalPostRecord(FQUpdate, Buff);
            cusInserted:
             if CanInsert then
              InternalPostRecord(FQInsert, Buff);
            cusDeleted:
             if CanDelete then
              InternalDeleteRecord(FQDelete, Buff);
          end;
//        Restore CU status
          rdUpdateStatus:=us;
          rdCachedUpdateStatus:=cus;
          WriteRecordCache(i, Buff);
{         ^^^^^^^^^^         :)}
          UpdateAction := uaApplied;
        except
          (*
           * If there is an exception, then allow the user
           * to intervene (decide what to do about it).
           *)
          on E: EFIBError do begin
            UpdateAction := uaFail;
            if Assigned(FOnUpdateError) then
              FOnUpdateError(Self, E, UpdateKind, UpdateAction);
            case UpdateAction of
              uaFail: raise;
              uaAbort: raise EAbort(E.Message);
            end;
          end;
        end;
      end;
    end;
   end;
 finally
  FreeRecordBuffer(Buff);
  vTypeDispositionField:=dfNormal;
 end;
// SynhronizeBuffers
end;




function  TpFIBDataSet.RecordStatus(RecNumber:integer):TUpdateStatus;
var Buff:Pchar;
begin
 Buff :=AllocRecordBuffer;
 try
  ReadRecordCache(RecNumber, Buff, False);
  Result:=PRecordData(Buff)^.rdUpdateStatus;
 finally
  FreeRecordBuffer(Buff);
 end
end;


function TpFIBDataSet.FilterRecno:integer;
begin
 if State<>dsFilter then Result:=Recno
 else Result:=FCurrentRecord
end;


procedure TpFIBDataSet.UpdateFieldsProps;
var i:integer;
    scale:Short;
    vFiAlias,vFi:TpFIBFieldInfo;
    RelTable,RelField:string;

begin

// Format Fields routine
   for i:=0 to Pred(FieldCount) do begin
    if poAutoFormatFields in FOptions then begin
     if (Fields[i] is TDateTimeField)   then begin
      if (Fields[i] is TDateField)   then begin
       with TDateField(Fields[i]) do
       if DisplayFormat='' then DisplayFormat:=FDefaultFormats.DisplayFormatDate
      end
      else
      if (Fields[i] is TTimeField)   then begin
       with TTimeField(Fields[i]) do
       if DisplayFormat='' then DisplayFormat:=FDefaultFormats.DisplayFormatTime
      end
      else
      with TDateTimeField(Fields[i]) do     begin
       if DisplayFormat='' then DisplayFormat:=FDefaultFormats.DateTimeDisplayFormat;
      end
     end
     else // end DateTime
     if Fields[i] is TNumericField then
     with TNumericField(Fields[i]) do     begin
      scale:=GetFieldScale(TNumericField(Fields[i]));
      if scale>0 then   if Fields[i] is TBCDField then
        scale:=-TBCDField(Fields[i]).Size;
      if scale<0 then begin //=
      if DisplayFormat='' then
       DisplayFormat:=UseFormat(FDefaultFormats.NumericDisplayFormat)
                      +MakeStr('0',-scale);
      if EditFormat=''    then
       EditFormat   :=UseFormat(FDefaultFormats.NumericEditFormat)
                      +MakeStr('0',-scale);
      end;
     end; // end TNumericField
    end; //end Format

    if PrepareOptions<>[] then begin

     RelTable:='ALIAS';
     RelField:=Fields[i].FieldName;
     vFiAlias:=
      ListTableInfo.GetFieldInfo(DataBase,
       RelTable,RelField
      );
      RelTable:= GetRelationTableName(Fields[i]);
      RelField:= GetRelationFieldName(Fields[i]);
      if pfSetReadOnlyFields in PrepareOptions then begin
       Fields[i].ReadOnly:=Fields[i].ReadOnly or
         ((RelTable<>QInsert.ModifyTable)
         and (RelTable<>QUpdate.ModifyTable)
         )
      end;
      if ((RelField='') or (RelTable='')) and
       (vFiAlias=nil)
      then Continue;
      vFi:=
       ListTableInfo.GetFieldInfo(DataBase,
       RelTable,RelField
      );

      if (vFi=nil) then vFi:=vFiAlias;
      if (vFi=nil) then Continue;
                             
{     if pfImportDefaultValues in PrepareOptions then
       Fields[i].DefaultExpression:=vFi.DefaultValue;}

       if pfImportDefaultValues in PrepareOptions then
        // Be sure to handle '1.0' with different DecSep!
        if (vFi.DefaultValue='') then
         Fields[i].DefaultExpression:=''
        else
        if (Fields[i] is TNumericField) then begin
           if (DecimalSeparator <> '.') then
            Fields[i].DefaultExpression:=ReplaceStr(vFi.DefaultValue,'.', DecimalSeparator)
           else
            Fields[i].DefaultExpression:=vFi.DefaultValue
        end
        else
        if (Fields[i] is TDateTimeField) then begin
         if (Fields[i] is TDateField) then
          Fields[i].DefaultExpression :=ToClientDateFmt(vFi.DefaultValue,1)
         else
         if (Fields[i] is TTimeField) then
          Fields[i].DefaultExpression :=ToClientDateFmt(vFi.DefaultValue,2)
         else
          Fields[i].DefaultExpression :=ToClientDateFmt(vFi.DefaultValue,0)
        end
        else
          Fields[i].DefaultExpression:=vFi.DefaultValue;


     if (vFiAlias<>nil) then vFi:=vFiAlias;

     if pfSetReadOnlyFields in PrepareOptions then begin
      Fields[i].ReadOnly:= vFi.IsComputed or Fields[i].ReadOnly;
     end;


     if pfSetRequiredFields in PrepareOptions then
      Fields[i].Required:=(vFi.DefaultValue='') and
       not vFi.IsTriggered and
       not  QSelect.FieldByName[Fields[i].FieldName].IsNullable;

     if psApplyRepositary in PrepareOptions then
     // Info from FIB$FIELDS_INFO
      if vFi.WithAdditionalInfo then begin
       if vFi.DisplayLabel<>'' then Fields[i].DisplayLabel:=vFi.DisplayLabel;
       Fields[i].Visible:=vFi.Visible;
       if (vFi.DisplayFormat<>'') then
        if (Fields[i] is TNumericField)  then
          TNumericField(Fields[i]).DisplayFormat:=vFi.DisplayFormat
        else
        if (Fields[i] is TDateTimeField)  then
          TDateTimeField(Fields[i]).DisplayFormat:=vFi.DisplayFormat;
       if (vFi.EditFormat<>'') then
        if (Fields[i] is TNumericField)  then
          TNumericField(Fields[i]).EditFormat:=vFi.EditFormat;
      end;
    end;
   end; //end for

end;


procedure  TpFIBDataSet.DoAfterOpen;
begin
 FHasUncommitedChanges:=false;
 if vSelectSQLTextChanged then
  if FAutoUpdateOptions.AutoReWriteSqls then  GenerateSQLs;
 UpdateFieldsProps;
 FUpdatesPending:=false;
 if psGetOrderInfo in PrepareOptions then
   PrepareAdditionalInfo;
 //
 if FContainer<>nil then FContainer.DataSetEvent(Self,deAfterOpen);

 inherited;
end;

// Filter works


procedure TpFIBDataSet.AddedFilterRecord(DataSet: TDataSet; var Accept: Boolean);
begin
 // abstract method
end;

function TpFIBDataSet.IsVisible(Buffer: PChar): Boolean;
begin
 Result:=vInSort  ;
 if Result then exit;
 Result:= inherited IsVisible(Buffer);
 if IsFiltered and Result then try
  SetState(dsFilter);
  Result:= FCurrentRecord>-1;
  if Result and Filtered then
   if Assigned(OnFilterRecord) then OnFilterRecord(Self,Result);
  if Result then AddedFilterRecord(Self,Result);
  if Result then Inc(vFilteredRecCounts);
 finally
  SetState(dsBrowse);
 end
end;

procedure TpFIBDataSet.RefreshClientFields;
var i:integer;
    FIBBuff:Pchar;
begin
 FIBBuff:=AllocRecordBuffer;
 try
  vTypeDispositionField:=dfRWRecNumber;
  for i:=0 to Pred(FRecordCount) do begin
   ReadRecordCache(i, FIBBuff, False);
   vInspectRecno:=i;
   CalculateFields(FIBBuff);
  end;
  SynchronizeBuffers(true,false);
 finally
  vTypeDispositionField:=dfNormal;
  FreeRecordBuffer(FIBBuff)
 end
end;

procedure TpFIBDataSet.RefreshFilters;
var
    NeedSavePlace:boolean;
    OldRecNo:integer;
    BeforeScrollEvent,AfterScrollEvent:TDataSetNotifyEvent;
begin
  if csDestroying in ComponentState then Exit;
  if Active then begin
    CheckBrowseMode;
    NeedSavePlace:=IsVisible(ActiveBuffer);
    vFilteredRecCounts:=0;
    OldRecNo:=RecNo;
    DisableControls;
    BeforeScrollEvent:=BeforeScroll;
    AfterScrollEvent :=AfterScroll;
    BeforeScroll:=nil;
    AfterScroll :=nil;
    if not NeedSavePlace and Assigned(BeforeScrollEvent) then  BeforeScrollEvent(Self);
    try
      First;
      if NeedSavePlace then   RecNo:=OldRecNo;
    finally
      BeforeScroll:=BeforeScrollEvent;
      AfterScroll :=AfterScrollEvent;
      if Assigned(AfterScrollEvent) then  AfterScrollEvent(Self);
      EnableControls;
    end
  end
end;

procedure TpFIBDataSet.SetFiltered(Value: Boolean);
begin
 inherited SetFiltered(Value);
 RefreshFilters
end;

function  TpFIBDataSet.LockRecord(RaiseErr:boolean):TLockStatus;
var ExceptMessage:string;
    Retry:boolean;
    tf:Tfield;
    i:integer;
begin
 if Trim(QUpdate.SQL.Text)='' then
  raise Exception.Create('Can''t lock record. Update query is empty');
 SetQueryParams(QUpdate,GetActiveBuf);
 if CachedUpdates then
  for i:=0 to Pred(QUpdate.Params.Count) do begin
   tf:=  FindField(QUpdate.Params[i].Name);
   if (tf<>nil) and not tf.IsBlob then
   if tf.isNull then
    QUpdate.Params[i].IsNull:=true
   else
   if tf is TDateTimeField then
    QUpdate.Params[i].AsDateTime:=tf.OldValue
   else
    QUpdate.Params[i].Value:=tf.OldValue
  end;
 Result:=lsUnknownError;
 Retry:=true;
 while Retry do begin
  try
   QUpdate.ExecQuery;
   if QUpdate.RowsAffected=1 then  Result:=lsSuccess
   else
   if QUpdate.RowsAffected=0 then  Result:=lsNotExist
   else
   Result:=lsMultiply;
  except
   On E: EFIBError do
   if E.SQLCode=-913 then Result:=lsDeadLock
   else
    Result:=lsUnknownError
  end;
  if (Result<>lsSuccess) and RaiseErr then begin
   case Result of
     lsDeadLock: ExceptMessage   :=SEdDeadLockMess;
     lsNotExist: ExceptMessage   :=SEdNotExistRecord;
     lsMultiply: ExceptMessage   :=SEdMultiplyRecord;
     lsUnknownError:ExceptMessage:=SEdUnknownError;
   end;
   ExceptMessage:=SEdErrorPrefix+ExceptMessage;
   Retry:=RaiseLockError(Result,ExceptMessage)=daRetry
  end
  else
    Retry:=false
 end;
end;
//



// BCD routine
{$IFNDEF VER130}
type
   TBCDArr= array[0..7] of Byte;
   PBCDArr=^TBCDArr;

function TpFIBDataSet.BCDToCurr(BCD: Pointer; var Curr: Currency): Boolean;//override;
begin
   try
    Curr:=Double(TBCDArr(BCD^));
    Result :=true;
   except
    Result :=false;
   end
end;

function TpFIBDataSet.CurrToBCD(const Curr: Currency; BCD: Pointer; Precision,
      Decimals: Integer): Boolean; //override;
var Dest:Double;
    i,st:integer;
begin
    try
      // Round rtns
     st:=1;
     for i:=1 to Decimals do st:=10*st;
     Dest:=System.Int(Curr)+Round(Frac(Curr)*st)/st;
//
     Move(Dest, PBCDArr(BCD)^, 8);
     Result :=true;
   except
    Result :=false;
   end
end;
{$ENDIF}





procedure TpFIBDataSet.DoSort(Fields: array of const; Ordering: array of Boolean);
begin
 try
  vInSort:=true;
  Sort(Self,Fields,Ordering);
 finally
  vInSort:=false;
  RefreshFilters
 end;
end;

//  other routine stuff

procedure TpFIBDataSet.OpenWP(ParamValues: array of Variant);
begin
 if High(ParamValues)>-1 then
  FQSelect.SetParamValues(ParamValues);
 Open
end;

// Clone Records routine
function FieldInArray(Field:TField; Arr: array of const):boolean;
var i:integer;
begin
  Result:=false;
  for i:=Low(Arr) to High(Arr) do
   with Arr[i] do begin
     case VType of
      vtAnsiString: Result:=UpperCase(Field.FieldName)=UpperCase(string(VAnsiString));
      vtObject:
       if VObject is TStrings then
        Result:= TStrings(VObject).IndexOf(UpperCase(Field.FieldName))<>-1
       else
       if VObject is TField then Result:=VObject=Field;
     end;
     if Result then Exit
   end;
end;

procedure TpFIBDataSet.CloneRecord(SrcRecord:integer; IgnoreFields:array of const);
var i:integer;
begin
 if (SrcRecord>=(FRecordCount-FDeletedRecords)) or (SrcRecord<0) then Exit;
 if State<>dsInsert then Insert;
 for i:=0 to Pred(FieldCount) do
   if (Fields[i].FieldKind in [fkData]) and not Fields[i].IsBlob
    and not FieldInArray(Fields[i],IgnoreFields) then
     Fields[i].Value:=RecordFieldValue(Fields[i],SrcRecord);
end;

procedure TpFIBDataSet.CloneCurRecord(IgnoreFields:array of const );
begin
 CloneRecord(Recno-1, IgnoreFields)
end;

// Wrappers

function  TpFIBDataSet.FN(const FieldName:string):TField; //FindField
begin
  Result:=FindField(FieldName)
end;

// Containers stuff

procedure TpFIBDataSet.SetReceiveEvents(Value:Tstrings);
begin
 FReceiveEvents.Assign(Value)
end;

procedure TpFIBDataSet.SetContainer(Value:TDataSetsContainer);
begin
  if FContainer=Value then Exit;
  if FContainer<>nil then FContainer.RemoveDataSet(Self);
  FContainer:=Value;
  if FContainer<>nil then begin
   FContainer.FreeNotification(Self);
   FContainer.AddDataSet(Self);
  end;
end;

procedure TpFIBDataSet.DoUserEvent(Sender:Tobject; const UDE:string;var Info:string);
begin
 if UpperCase(UDE)='CLOSE' then  Close else
 if UpperCase(UDE)='OPEN' then   Open  else
 if FContainer<>nil then FContainer.UserEvent(Sender,Self,UDE,Info);
 if Assigned(FOnUserEvent) then FOnUserEvent(Sender,Self,UDE,Info)
end;

function  TpFIBDataSet.GenerateSQLText
     (const TableName,KeyFieldNames:string;SK:TpSQLKind):string;
 //    Updates
const
  Indent='     ';
var i:integer;
    pWhereClause,InsValuesStr:string;
    AcceptCount:integer;
    RealKeyFieldName,RealFieldName:string;
    FieldTableName :string;
    KeyFieldList:TList;
    FormatTableName:string;
    RelTableName   :string;
    vpFIBTableInfo:TpFIBTableInfo;
    vFi:TpFIBFieldInfo;
procedure  GetFieldList(List: TList; const FieldNames: string);
var
  Pos: Integer;
  Field: TField;
begin
  Pos := 1;
  while Pos <= Length(FieldNames) do
  begin
    Field := FindField(ExtractFieldName(FieldNames, Pos));
    if Field<>nil then List.Add(Field);
  end;
end;

begin
KeyFieldList:=TList.Create;
try
 GetFieldList(KeyFieldList, KeyFieldNames);
 if KeyFieldList.Count=0 then Exit;
 vpFIBTableInfo:=ListTableInfo.GetTableInfo(Database,TableName);
 if vpFIBTableInfo=nil then Exit;
 // Validate KeyFields
 pWhereClause:='';
 for i:=Pred(KeyFieldList.Count) downto 0 do begin
  RelTableName:=GetRelationTableName(TField(KeyFieldList[i]));
  FormatTableName:=FormatIdentifier(Database.SQLDialect,TableName);

  if RelTableName<>FormatTableName then begin
    KeyFieldList.Delete(i);
    Continue
  end;
  RealKeyFieldName:=GetRelationFieldName(TField(KeyFieldList[i]));
  vFi:=vpFIBTableInfo.FieldInfo(RealKeyFieldName);
  if (vFi=nil) or not vFi.CanIncToWhereClause then
    KeyFieldList.Delete(i)
  else
   pWhereClause:=iifStr(pWhereClause='','',' and ')+
    RealKeyFieldName+'=?'+
    FormatIdentifier(
     Database.SQLDialect,'OLD_'+TField(KeyFieldList[i]).FieldName
    );
 end;
 if KeyFieldList.Count=0 then Exit;
 case SK of
  skModify: Result:='Update '+FormatTableName+' Set';
  skInsert: begin
             Result:='Insert into '+FormatTableName+'(';
             InsValuesStr:='values (';
            end;
  skDelete: begin
             Result:='Delete from '+FormatTableName+CLRF+
                     'where '+pWhereClause
             ;
             Exit;
            end;
  skRefresh:begin
             Result:=
              AddToWhereClause(SelectSQL.text,pWhereClause);
             Exit;
            end;
 end;

 AcceptCount:=0;
 for i:=0 to FieldCount-1 do begin
   if (Fields[i].FieldKind<>fkData) then Continue;
   FieldTableName:=
    FormatIdentifier(Database.SQLDialect,
      GetRelationTableName(Fields[i])
    );
   if FieldTableName<>FormatTableName then
     Continue
   else
     Inc(AcceptCount) ;
   RealFieldName:=
    FormatIdentifier(Database.SQLDialect,
      GetRelationFieldName(Fields[i])
    );

   vFi:=vpFIBTableInfo.FieldInfo(RealFieldName);
   if (vFi=nil) or (vFi.IsComputed )    then Continue;

   Result       :=Result+iifStr(AcceptCount=1,'',',')+CLRF;
   InsValuesStr :=InsValuesStr+iifStr(AcceptCount=1,'',',')+CLRF;
   case SK of
    skModify:Result:=Result+Indent+RealFieldName+'=?'+
              FormatIdentifier(Database.SQLDialect,'NEW_'+Fields[i].FieldName);
    skInsert:begin
              Result      :=Result+Indent+RealFieldName;
              InsValuesStr:=InsValuesStr+Indent+'?'+
               FormatIdentifier(Database.SQLDialect,
                'NEW_'+Fields[i].FieldName
              );
             end;
   end;
 end; // end for

 case SK of
    skModify:Result:=Result+CLRF+'where '+pWhereClause;
    skInsert:begin
              Result:=Result+CLRF+')'+CLRF+InsValuesStr+CLRF+')'
             end;
 end;
finally
 KeyFieldList.Free
end;
end;

function  TpFIBDataSet.GenerateSQLTextWA
     (const TableName:string;SK:TpSQLKind):string; // Where All
begin
 Result:=GenerateSQLText(TableName,AllFields,SK)
end;


//AutoUpdate operations
function  TpFIBDataSet.KeyField:TField;
begin
 with FAutoUpdateOptions do Result:=FindField(KeyFields)
end;

function  TpFIBDataSet.SqlTextGenID:string;
begin
 with FAutoUpdateOptions do
  Result:='SELECT GEN_ID('+
   FormatIdentifier(Database.SQLDialect, GeneratorName)+
    ',1) FROM  RDB$DATABASE';
end;

procedure TpFIBDataSet.IncGenerator;
var kf:TField;
begin
  with FAutoUpdateOptions do
   if not SelectGenID then Exit;
   kf:=KeyField;
   if (kf=nil) or not (kf is TNumericField) then Exit;
   if kf.asInteger<=0 then
   with vIntQuery do begin
    Close;
    if DataBase    <>Self.DataBase then DataBase        :=Self.DataBase;
    if Transaction <>Self.Transaction  then Transaction :=Self.Transaction;
    if (SQL.Count=0) or (SQL[0]<>SqlTextGenID) then
    begin
     SQL.Clear;     SQl.Add(SqlTextGenID);
    end;
    if not Prepared then  Prepare;
     ExecQuery;
     if (kf is TFIBBCDField) and (TFIBBCDField(kf).Scale = 0) then
      TFIBBCDField(kf).asComp:=Fields[0].AsComp 
     else
      kf.asInteger:=Fields[0].Value;
   end;
end;

function  TpFIBDataSet.AllFields:string;
var
    i:integer;
begin
 Result:='';
 for i:=0  to Pred(FieldCount) do
  Result:=Result+iifStr(i>0,';','')+Fields[i].FieldName;
end;

function  TpFIBDataSet.PrimaryKeyFields(const TableName:string):string;
var PrimKeyFields:string;
    i,wc:integer;
    tf:TField;
begin
  Result:='';
  PrimKeyFields:=
   Trim(ListTableInfo.
    GetTableInfo(Database,TableName).PrimaryKeyFields
   );
  if PrimKeyFields='' then Exit;
  wc:=WordCount(PrimKeyFields,[';']);
  for i:=1 to wc do begin
    tf:=
     FieldByOrigin(TableName+'.'+
      ExtractWord(i,PrimKeyFields,[';'])
     );
    if tf=nil then begin
     Result:='';     Exit;
    end;
    Result:=Result+iifStr(i>1,';','')+tf.FieldName;
  end;
end;

procedure TpFIBDataSet.GenerateSQLs;
begin
 if FieldCount=0 then FieldDefs.Update;
 QDelete .OnSQLChanging:=nil;
 QInsert .OnSQLChanging:=nil;
 QUpdate .OnSQLChanging:=nil;
 QRefresh.OnSQLChanging:=nil;
 with FAutoUpdateOptions do try
  if Trim(KeyFields)='' then
   KeyFields:=PrimaryKeyFields(UpdateTableName);
   if KeyFields='' then   KeyFields:=AllFields;
  if KeyFields='' then Exit;
  if AutoUpdateOptions.CanChangeSQLs or (Trim(DeleteSQL.Text)='') then
   DeleteSQL.Text :=GenerateSQLText(UpdateTableName,KeyFields,skDelete);
  if AutoUpdateOptions.CanChangeSQLs or (Trim(UpdateSQL.Text)='') then
   UpdateSQL.Text :=GenerateSQLText(UpdateTableName,KeyFields,skModify);
  if AutoUpdateOptions.CanChangeSQLs or (Trim(InsertSQL.Text)='') then
   InsertSQL.Text :=GenerateSQLText(UpdateTableName,KeyFields,skInsert);
  if AutoUpdateOptions.CanChangeSQLs or (Trim(RefreshSQL.Text)='') then
   RefreshSQL.Text:=GenerateSQLText(UpdateTableName,KeyFields,skRefresh);
 finally
  QDelete .OnSQLChanging:=SQLChanging;
  QInsert .OnSQLChanging:=SQLChanging;
  QUpdate .OnSQLChanging:=SQLChanging;
  QRefresh.OnSQLChanging:=SQLChanging;
 end;
end;



procedure TpFIBDataSet.FakeRefresh(FromDataSet:TDataSet;Kind:TFakeRefreshKind
 ;FieldMap:Tstrings
);
var i:integer;
    fn1:Tfield;
    Buff: PChar;
    sfn:String;
    ForcedEdit:boolean;
    IsReadOnlyField:boolean;
begin
 vInFakeRefresh:=not (State in [dsInsert,dsEdit]);
 try
  ForcedEdit:=not (State in [dsInsert,dsEdit]) ;
  if ForcedEdit then
   if Kind= frkInsert then Insert else Edit;
  for i:=0 to Pred(FromDataSet.FieldCount) do begin
   if (FieldMap<>nil) then begin
    sfn:=FieldMap.Values[FromDataSet.Fields[i].FieldName];
    if sfn='' then  sfn:=FromDataSet.Fields[i].FieldName;
   end
   else
    sfn:=FromDataSet.Fields[i].FieldName;
   fn1:=FN(sfn);
   if fn1<>nil then begin
    IsReadOnlyField:=fn1.ReadOnly;
    fn1.ReadOnly:=false;
    try
     fn1.Value:=FromDataSet.Fields[i].Value;
    finally
     fn1.ReadOnly:=IsReadOnlyField;
    end; 
   end;
  end;
  if ForcedEdit then Post;
  Buff := GetActiveBuf;
  with PRecordData(Buff)^ do begin
     rdCachedUpdateStatus:=cusUnmodified;
     rdUpdateStatus := usUnmodified;
     WriteRecordCache(rdRecordNumber, Buff);
     SaveOldBuffer(Buff)
  end;
 finally
  vInFakeRefresh:=false
 end
end;

procedure TpFIBDataSet.FakeRefreshByArrMap(
           FromDataSet:TDataSet;Kind:TFakeRefreshKind;
          const SourceFields,DestFields:array of String
 );
var ts:TStrings;
    i,m:integer;

begin
 ts:=TStringList.Create;
 m:=High(SourceFields);
 if High(DestFields)>m then m:=High(DestFields);
 with ts do try
  for i:=0 to m do  Values[SourceFields[i]]:=DestFields[i];
  FakeRefresh(FromDataSet,Kind, ts);
 finally
  Free
 end;
end;


end.


