{************************************************************************}
{ FIBPlus - component library  for direct access  to Interbase  databases}
{    FIBPlus is based in part on the product                             }
{    Free IB Components, written by Gregory H. Deatz for                 }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.                     }
{                                         Contact:       gdeatz@hlmdd.com}
{    Copyright (c) 1998-2001 Serge Buzadzhy                              }
{                                         Contact: serge_buzadzhy@mail.ru}
{  Please see the file FIBLicense.txt for full license information.      }
{************************************************************************}

unit FIBDatabase;


interface
{$I FIBPlus.INC}
uses
  Windows, SysUtils, Classes,
  Forms, ExtCtrls, DBLogDlg,//IS GUI
  ibase,IB_Intf, ib_externals, DB,pFIBProps,
  FIB;

type
  TFIBDatabase = class;
  TFIBTransaction = class;
  TFIBBase = class;
  TDesignDBOption=(ddoIsDefaultDatabase,ddoStoreConnected);
  TDesignDBOptions = set of TDesignDBOption;


  (* TFIBDatabase *)

  TFIBDatabase = class(TComponent)
  protected
    FCanTimeout         : Boolean;                      // Can the timer actually timeout the connection?
    FDatasets           : TList;                        // TFIBDataSets attached.
    FTransactions       : TList;                        // TFIBTransactions attached.
    FDBName             : String;                       // DB's name
    FDBParams           : TStrings;                     // "Pretty" Parameters to database
    FDBParamsChanged    : Boolean;                      // Flag to determine if DPB must be regenerated
    FDPB                : PChar;                        // Parameters to DB as passed to IB.
    FDPBLength          : Short;                        // Length of parameter buffer
    FHandle             : TISC_DB_HANDLE;               // DB's handle
    FHandleIsShared     : Boolean;                      // Is the handle shared with another DB?
    FUseLoginPrompt     : Boolean;                      // Show a default login prompt?
    FOnConnect          : TNotifyEvent;                 // Upon successful connection...
    FOnTimeout          : TNotifyEvent;                 // Upon timing out...
    FDefaultTransaction : TFIBTransaction;              // Many transaction components can be specified, but this is the primary, or default one to use.
    FStreamedConnected  : Boolean;                      // Used for delaying the opening of the database.
    FTimer              : TTimer;                       // The timer ID.
    FUserNames          : TStringList;                  // For use only with GetUserNames
    FBackoutCount: TStringList;                         // isc_info_backout_count
    FDeleteCount: TStringList;                          // isc_info_delete_count
    FExpungeCount: TStringList;                         // isc_info_expunge_count
    FInsertCount: TStringList;                          // isc_info_insert_count
    FPurgeCount: TStringList;                           // isc_info_purge_count
    FReadIdxCount: TStringList;                         // isc_info_read_idx_count
    FReadSeqCount: TStringList;                         // isc_info_read_seq_count
    FUpdateCount: TStringList;                          // isc_info_update_count
//IB6
    FSQLDialect:integer;
    FUpperOldNames      : boolean; // compatibility with IB4..5 field names conventions
    FIBLoaded: Boolean;
    FConnectParams:TConnectParams;
    FDesignDBOptions:TDesignDBOptions;
    FBeforeDisconnect:TNotifyEvent;
    FAfterDisconnect:TNotifyEvent;
    FDifferenceTime : double;
//    FStoreConnected :boolean;
    FSynchronizeTime:boolean;
    FDefDataSetPrepareOptions:TpPrepareOptions;
    vInternalTransaction:TFIBTransaction;
    procedure DBParamsChange(Sender: TObject);
    procedure DBParamsChanging(Sender: TObject);
    function GetConnected: Boolean;                     // Is DB connected?
    function GetDatabaseName: String;
    function GetDataSet(Index: Integer): TFIBBase;      // Get the indexed Dataset.
    function GetDataSetCount: Integer;                  // Get the number of Datasets connected.
    function GetDBParamByDPB(const Idx: Integer): String;
    function GetTimeout: Integer;                       // Get the timeout
    function GetTransaction(Index: Integer): TFIBTransaction;
    function GetTransactionCount: Integer;
    function Login: Boolean;                            // Show login prompt
    procedure SetConnected(Value: Boolean);
    procedure SetDatabaseName(const Value: string);
    procedure SetDBParamByDPB(const Idx: Integer; Value: String);
    procedure SetDBParamByName(const ParName,Value: String);
    procedure SetDBParams(Value: TStrings);
    procedure SetDefaultTransaction(Value: TFIBTransaction);
    procedure SetHandle(Value: TISC_DB_HANDLE);
    procedure SetTimeout(Value: Integer);
    procedure TimeoutConnection(Sender: TObject);
    (* Database Info procedures -- Advanced stuff (translated from isc_database_info) *)
    function GetAllocation: Long; // isc_info_allocation
    function GetBaseLevel: Long; // isc_info_base_level
    function GetDBFileName: String; // isc_info_db_id
    function GetDBSiteName: String; // isc_info_db_id
    function GetDBImplementationNo: Long; // isc_info_implementation
    function GetDBImplementationClass: Long;
    function GetNoReserve: Long; // isc_info_no_reserve
    function GetODSMinorVersion: Long; // isc_info_ods_minor_version
    function GetODSMajorVersion: Long; // isc_info_ods_version
    function GetPageSize: Long;            // isc_info_page_size
    function GetVersion: String; // isc_info_info_version
    function GetCurrentMemory: Long; // isc_info_current_memory
    function GetForcedWrites: Long; // isc_info_forced_writes
    function GetMaxMemory: Long; // isc_info_max_memory
    function GetNumBuffers: Long; // isc_info_num_buffers
    function GetSweepInterval: Long; // isc_info_sweep_interval
    function GetUserNames: TStringList; // isc_info_user_names
    function GetFetches: Long; // isc_info_fetches
    function GetMarks: Long; // isc_info_marks
    function GetReads: Long; // isc_info_reads
    function GetWrites: Long; // isc_info_writes
    function GetBackoutCount: TStringList; // isc_info_backout_count
    function GetDeleteCount: TStringList; // isc_info_delete_count
    function GetExpungeCount: TStringList; // isc_info_expunge_count
    function GetInsertCount: TStringList; // isc_info_insert_count
    function GetPurgeCount: TStringList; // isc_info_purge_count
    function GetReadIdxCount: TStringList; // isc_info_read_idx_count
    function GetReadSeqCount: TStringList; // isc_info_read_seq_count
    function GetUpdateCount: TStringList; // isc_info_update_count
    function GetOperationCounts(DBInfoCommand: Integer; var FOperation: TStringList): TStringList;
    function GetLogFile: Long; // isc_info_log_file
    function GetCurLogFileName: String; // isc_info_cur_logfile_name
    function GetCurLogPartitionOffset: Long; // isc_info_cur_log_part_offset
    function GetNumWALBuffers: Long; // isc_info_num_wal_buffers
    function GetWALBufferSize: Long; // isc_info_wal_buffer_size
    function GetWALCheckpointLength: Long; // isc_info_wal_ckpt_length
    function GetWALCurCheckpointInterval: Long; // isc_info_wal_cur_ckpt_interval
    function GetWALPrvCheckpointFilename: String; // isc_info_wal_prv_ckpt_fname
    function GetWALPrvCheckpointPartOffset: Long; // isc_info_wal_prv_ckpt_poffset
    function GetWALGroupCommitWaitUSecs: Long; // isc_info_wal_grpc_wait_usecs
    function GetWALNumIO: Long; // isc_info_wal_num_id
    function GetWALAverageIOSize: Long; // isc_info_wal_avg_io_size
    function GetWALNumCommits: Long; // isc_info_wal_num_commits
    function GetWALAverageGroupCommitSize: Long; // isc_info_wal_avg_grpc_size
    function GetLongDBInfo(DBInfoCommand: Integer): Long;
    function GetProtectLongDBInfo(DBInfoCommand: Integer;var Success:boolean): Long;
    function GetStringDBInfo(DBInfoCommand: Integer): String;
    //IB6
    function  GetDBSQLDialect:Word;
    procedure SetSQLDialect(const Value: Integer);
    procedure SetDesignDBOptions(Value:TDesignDBOptions);
    function  GetReadOnly: Long;
    function  GetStoreConnected:boolean;
  protected
    procedure Loaded; override;
    procedure InternalClose(Force: Boolean); virtual ; //Added virtual
    procedure RemoveDataSet(Idx: Integer);
    procedure RemoveDataSets;
    procedure RemoveTransaction(Idx: Integer);
    procedure RemoveTransactions;
    function  AddDataSet(ds: TFIBBase): Integer;
    function  AddTransaction(TR: TFIBTransaction): Integer;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function   Call(ErrCode: ISC_STATUS; RaiseError: Boolean): ISC_STATUS;


    procedure CheckActive;                              // Raise error if DB is inactive
    procedure CheckInactive;                            // Raise error if DB is active
    procedure CheckDatabaseName;                        // Raise error if DBName is empty
    procedure Close;
    procedure CreateDatabase;
    property  DBParamByDPB[const Idx: Integer]: String read GetDBParamByDPB
                                                      write SetDBParamByDPB;
    procedure DropDatabase;
    function  FindTransaction(TR: TFIBTransaction): Integer;
    procedure ForceClose;
    function  IndexOfDBConst(st: String): Integer;       // Get the index of a given constant in DBParams
    procedure Open; virtual; //Added virtual

    function  TestConnected: Boolean;
    procedure ApplyUpdates(const DataSets: array of TDataSet);
    (* Properties    *)

    property CanTimeout: Boolean read FCanTimeout write FCanTimeout;
    property DataSetCount: Integer read GetDataSetCount;
    property DataSets[Index: Integer]: TFIBBase read GetDataSet;
    property Handle: TISC_DB_HANDLE read FHandle write SetHandle;
    property HandleIsShared: Boolean read FHandleIsShared;
    property TransactionCount: Integer read GetTransactionCount;
    property Transactions[Index: Integer]: TFIBTransaction read GetTransaction;
    (* Database Info properties -- Advanced stuff (translated from isc_database_info) *)
    property Allocation: Long read GetAllocation; // isc_info_allocation
    property BaseLevel: Long read GetBaseLevel; // isc_info_base_level
    property DBFileName: String read GetDBFileName; // isc_info_db_id
    property DBSiteName: String read GetDBSiteName;
    property DBImplementationNo: Long read GetDBImplementationNo; // isc_info_implementation
    property DBImplementationClass: Long read GetDBImplementationClass;
    property NoReserve: Long read GetNoReserve; // isc_info_no_reserve
    property ODSMinorVersion: Long read GetODSMinorVersion; // isc_info_ods_minor_version
    property ODSMajorVersion: Long read GetODSMajorVersion; // isc_info_ods_version
    property PageSize: Long read GetPageSize;            // isc_info_page_size
    property Version: String read GetVersion; // isc_info_info_version
    property CurrentMemory: Long read GetCurrentMemory; // isc_info_current_memory
    property ForcedWrites: Long read GetForcedWrites; // isc_info_forced_writes
    property MaxMemory: Long read GetMaxMemory; // isc_info_max_memory
    property NumBuffers: Long read GetNumBuffers; // isc_info_num_buffers
    property SweepInterval: Long read GetSweepInterval; // isc_info_sweep_interval
    property UserNames: TStringList read GetUserNames; // isc_info_user_names
    property Fetches: Long read GetFetches; // isc_info_fetches
    property Marks: Long read GetMarks; // isc_info_marks
    property Reads: Long read GetReads; // isc_info_reads
    property Writes: Long read GetWrites; // isc_info_writes
    property BackoutCount: TStringList read GetBackoutCount; // isc_info_backout_count
    property DeleteCount: TStringList read GetDeleteCount; // isc_info_delete_count
    property ExpungeCount: TStringList read GetExpungeCount; // isc_info_expunge_count
    property InsertCount: TStringList read GetInsertCount; // isc_info_insert_count
    property PurgeCount: TStringList read GetPurgeCount; // isc_info_purge_count
    property ReadIdxCount: TStringList read GetReadIdxCount; // isc_info_read_idx_count
    property ReadSeqCount: TStringList read GetReadSeqCount; // isc_info_read_seq_count
    property UpdateCount: TStringList read GetUpdateCount; // isc_info_update_count
    property LogFile: Long read GetLogFile; // isc_info_log_file
    property CurLogFileName: String read GetCurLogFileName; // isc_info_cur_logfile_name
    property CurLogPartitionOffset: Long read GetCurLogPartitionOffset; // isc_info_cur_log_part_offset
    property NumWALBuffers: Long read GetNumWALBuffers; // isc_info_num_wal_buffers
    property WALBufferSize: Long read GetWALBufferSize; // isc_info_wal_buffer_size
    property WALCheckpointLength: Long read GetWALCheckpointLength; // isc_info_wal_ckpt_length
    property WALCurCheckpointInterval: Long read GetWALCurCheckpointInterval; // isc_info_wal_cur_ckpt_interval
    property WALPrvCheckpointFilename: String read GetWALPrvCheckpointFilename; // isc_info_wal_prv_ckpt_fname
    property WALPrvCheckpointPartOffset: Long read GetWALPrvCheckpointPartOffset; // isc_info_wal_prv_ckpt_poffset
    property WALGroupCommitWaitUSecs: Long read GetWALGroupCommitWaitUSecs; // isc_info_wal_grpc_wait_usecs
    property WALNumIO: Long read GetWALNumIO; // isc_info_wal_num_id
    property WALAverageIOSize: Long read GetWALAverageIOSize; // isc_info_wal_avg_io_size
    property WALNumCommits: Long read GetWALNumCommits; // isc_info_wal_num_commits
    property WALAverageGroupCommitSize: Long read GetWALAverageGroupCommitSize; // isc_info_wal_avg_grpc_size
//IB6
    property DBSQLDialect:Word read GetDBSQLDialect;
    property ReadOnly:Long        read GetReadOnly ;
//IBX    
    property DatabaseName: string read GetDatabaseName write SetDatabaseName;

    property DifferenceTime : double read FDifferenceTime ;
  public
    procedure StartTransaction;
    procedure CloseDataSets;
    procedure Commit;
    procedure Rollback;
    procedure CommitRetaining;
    procedure RollbackRetaining;
    {$IFDEF D45}
    function Gen_Id(const GeneratorName: string; Step: Int64): Int64;
    {$ELSE}
    function Gen_Id(const GeneratorName: string; Step: Comp): Comp;
    {$ENDIF}
    function Execute(const SQL: string): boolean;
    function QueryValue(const SQL: string;FieldNo:integer):Variant;
    property StoreConnected :boolean read GetStoreConnected;    
  published
    property Connected: Boolean read GetConnected write SetConnected stored GetStoreConnected;
    property DBName: string read GetDatabaseName write SetDatabaseName;
    property DBParams: TStrings read FDBParams write SetDBParams;
    property DefaultTransaction: TFIBTransaction read FDefaultTransaction
                                                 write SetDefaultTransaction;
    //IB6
    property SQLDialect : Integer read FSQLDialect write SetSQLDialect;

    property Timeout: Integer read GetTimeout write SetTimeout;
    property UseLoginPrompt: Boolean read FUseLoginPrompt write FUseLoginPrompt default False;
    // Events
    property OnConnect: TNotifyEvent read FOnConnect write FOnConnect;
    //future? property OnGetDBName: TNotifyEvent read FOnGetDBName write FOnGetDBName;
    property OnTimeout: TNotifyEvent read FOnTimeout write FOnTimeout;
    property UpperOldNames: boolean read FUpperOldNames write FUpperOldNames
     default false; // compatibility with IB4..5 field names conventions

    property BeforeDisconnect:TNotifyEvent read FBeforeDisconnect write FBeforeDisconnect;
    property AfterDisconnect :TNotifyEvent read FAfterDisconnect  write FAfterDisconnect ;
    property ConnectParams  :TConnectParams read FConnectParams write FConnectParams stored false;
    property SynchronizeTime:boolean read FSynchronizeTime write FSynchronizeTime default true;

    property DesignDBOptions:TDesignDBOptions read FDesignDBOptions write SetDesignDBOptions default
     [ddoStoreConnected]
    ;
    property
     DataSetsPrepareOptions:TpPrepareOptions read FDefDataSetPrepareOptions
       write  FDefDataSetPrepareOptions
    {$IFNDEF BOOLEAN_FIELD_SUPPORT}
       default  [pfImportDefaultValues,psGetOrderInfo]
    {$ELSE}
       default  [pfImportDefaultValues,psGetOrderInfo,psUseBooleanField]
    {$ENDIF}

     ;
  end;

  (* TFIBTransaction *)

  TTransactionAction  =
   (TARollback, TARollbackRetaining,TACommit, TACommitRetaining);
  TTransactionState   =
  (tsActive,tsClosed,tsDoRollback,tsDoRollbackRetaining,tsDoCommit, tsDoCommitRetaining);

  TFIBTransaction = class(TComponent)           
  protected
    FCanTimeout         : Boolean;                      // Can the transaction timeout now?
    FDatabases          : TList;                        // TDatabases in transaction.
    FDataSets           : TList;                        // TFIBDataSets attached.
    FDefaultDatabase    : TFIBDatabase;                 // just like DefaultTransaction in FIBDatabase
    FHandle             : TISC_TR_HANDLE;               // TR's handle
    FHandleIsShared     : Boolean;
    FOnTimeout          : TNotifyEvent;                 // When the transaction times out...
    FStreamedActive     : Boolean;
    FTPB                : PChar;                        // Parameters to TR as passed to IB.
    FTPBLength          : Short;                        // Length of parameter buffer
    FTimer              : TTimer;                       // Timer for timing out transactions.
    FTimeoutAction      : TTransactionAction;           // Rollback or commit at end of timeout?
    FTRParams           : TStrings;                     // Parameters to Transactions.
    FTRParamsChanged    : Boolean;
    FState              : TTransactionState;
    function DoStoreActive:boolean;
    procedure EndTransaction(Action: TTransactionAction; Force: Boolean); virtual;
    // End the transaction using specified method...
    function GetDatabase(Index: Integer): TFIBDatabase; // Get the indexed database
    function GetDatabaseCount: Integer;                 // Get the number of databases in transaction
    function GetDataSet(Index: Integer): TFIBBase; // Get the indexed Dataset.
    function GetDataSetCount: Integer;                  // Get the number of Datasets connected.
    function GetInTransaction: Boolean;                 // Is there an active trans?
    function GetTimeout: Integer;
    procedure Loaded; override;
    procedure SetActive(Value: Boolean);
    procedure SetDefaultDatabase(Value: TFIBDatabase);
    procedure SetHandle(Value: TISC_TR_HANDLE);
    procedure SetTimeout(Value: Integer);               // Set the timeout
    procedure SetTRParams(Value: TStrings);
    procedure TimeoutTransaction(Sender: TObject);
    procedure TRParamsChange(Sender: TObject);
    procedure TRParamsChanging(Sender: TObject);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function AddDatabase(db: TFIBDatabase): Integer;
    function AddDataSet(ds: TFIBBase): Integer;
    procedure OnDatabaseDisconnecting(DB: TFIBDatabase);    
    function Call(ErrCode: ISC_STATUS; RaiseError: Boolean): ISC_STATUS;
    procedure CheckDatabasesInList;        // Raise error if no databases in list.
    procedure CheckInTransaction;          // Raise error if not in transaction
    procedure CheckNotInTransaction;       // Raise error if in transaction
    procedure Commit;           virtual;
    procedure CommitRetaining;  virtual;
    function FindDatabase(db: TFIBDatabase): Integer;
    procedure RemoveDatabase(Idx: Integer);
    procedure RemoveDatabases;
    procedure RemoveDataSet(Idx: Integer);
    procedure RemoveDataSets;
    procedure Rollback;         virtual;
    procedure RollbackRetaining;virtual;
    procedure StartTransaction;virtual; //Added virtual
    property DatabaseCount: Integer read GetDatabaseCount;
    property Databases[Index: Integer]: TFIBDatabase read GetDatabase;
    property DataSetCount: Integer read GetDataSetCount;
    property DataSets[Index: Integer]: TFIBBase read GetDataSet;
    property Handle: TISC_TR_HANDLE read FHandle write SetHandle;
    property HandleIsShared: Boolean read FHandleIsShared;
    property InTransaction: Boolean read GetInTransaction;
    property TPB: PChar read FTPB;
    property TPBLength: Short read FTPBLength;
    property State: TTransactionState read FState;
  published
    property Active: Boolean read GetInTransaction write SetActive stored DoStoreActive;
    property DefaultDatabase: TFIBDatabase read FDefaultDatabase
                                           write SetDefaultDatabase;
    property Timeout: Integer read GetTimeout write SetTimeout default 0;
    property TimeoutAction: TTransactionAction read FTimeoutAction write FTimeoutAction;
    property TRParams: TStrings read FTRParams write SetTRParams;
    // Events
    property OnTimeout: TNotifyEvent read FOnTimeout write FOnTimeout;

  end;

  (* TFIBBase *)
  (* Virtually all components in FIB are "descendents" of TFIBBase. *)
  (* It is to more easily manage the database and transaction *)
  (* connections. *)
  TFIBBase = class(TObject)
  protected
    FDatabase: TFIBDatabase;
    FIndexInDatabase: Integer;
    FTransaction: TFIBTransaction;
    FIndexInTransaction: Integer;
    FOwner: TObject;
    procedure FOnDatabaseConnecting; virtual;
    procedure FOnDatabaseConnected; virtual;

    procedure FOnDatabaseDisconnecting; virtual;
    procedure FOnDatabaseDisconnected; virtual;

    procedure FOnTransactionStarting; virtual;
    procedure FOnTransactionStarted; virtual;

    procedure FOnDatabaseFree; virtual;
    procedure FOnTransactionEnding; virtual;
    procedure FOnTransactionEnded; virtual;
    procedure FOnTransactionFree; virtual;
    function GetDBHandle: PISC_DB_HANDLE; virtual;
    function GetTRHandle: PISC_TR_HANDLE; virtual;
    procedure SetDatabase(Value: TFIBDatabase); virtual;
    procedure SetTransaction(Value: TFIBTransaction); virtual;
  public
    constructor Create(AOwner: TObject);
    destructor Destroy; override;
    procedure CheckDatabase; virtual;
    procedure CheckTransaction; virtual;
  public // properties
    OnDatabaseConnecting: TNotifyEvent;
    OnDatabaseConnected: TNotifyEvent;

    OnDatabaseDisconnecting: TNotifyEvent;
    OnDatabaseDisconnected: TNotifyEvent;
    OnDatabaseFree: TNotifyEvent;
    OnTransactionEnding: TNotifyEvent;
    OnTransactionEnded: TNotifyEvent;
    OnTransactionStarting: TNotifyEvent;
    OnTransactionStarted: TNotifyEvent;

    OnTransactionFree: TNotifyEvent;
    property Database: TFIBDatabase read FDatabase
                                    write SetDatabase;
    property DBHandle: PISC_DB_HANDLE read GetDBHandle;
    property Owner: TObject read FOwner;
    property TRHandle: PISC_TR_HANDLE read GetTRHandle;
    property Transaction: TFIBTransaction read FTransaction
                                          write SetTransaction;
  end;


var DefDataBase:TFIBDatabase;

implementation

uses
  FIBSQLMonitor,pFIBDataInfo,FIBDataSet,pFIBDataSet,FIBQuery,StrUtil;


var DatabaseList:TList;

  
(* TFIBDatabase *)

constructor TFIBDatabase.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FIBLoaded:=false;
  CheckIBLoaded;
  FIBLoaded:=true;
  FDataSets                            := TList.Create;
  FTransactions                        := TList.Create;
  FDBName                              := '';
  FDBParams                            := TStringList.Create;
  FDBParamsChanged                     := True;
  TStringList(FDBParams).OnChange      := DBParamsChange;
  TStringList(FDBParams).OnChanging    := DBParamsChanging;
  FDPB                                 := nil;
  FHandle                              := nil;
  FUserNames                           := nil;
  FBackoutCount                        := nil;
  FDeleteCount                         := nil;
  FExpungeCount                        := nil;
  FInsertCount                         := nil;
  FPurgeCount                          := nil;
  FReadIdxCount                        := nil;
  FReadSeqCount                        := nil;
  FUpdateCount                         := nil;
  FDesignDBOptions                     := [ddoStoreConnected];
  FTimer                               := TTimer.Create(Self);
  FTimer.Enabled                       := False;
  FTimer.Interval                      := 0;
  FTimer.OnTimer                       := TimeoutConnection;

  FSQLDialect                          :=1;
  FConnectParams  :=TConnectParams.Create(Self);
  FDifferenceTime                      := 0;
  FSynchronizeTime:=true;

//  if DefDataBase=nil then DefDataBase:=Self;
  vInternalTransaction:=TFIBTransaction.Create(Self);
  vInternalTransaction.DefaultDataBase:=Self;

 {$IFNDEF BOOLEAN_FIELD_SUPPORT}
  FDefDataSetPrepareOptions    :=
   [pfImportDefaultValues,psGetOrderInfo];
 {$ELSE}
   FDefDataSetPrepareOptions  :=
    [pfImportDefaultValues,psGetOrderInfo,psUseBooleanField];
 {$ENDIF}


  with vInternalTransaction.TRParams do begin
   Text:='write'+#13#10+
         'nowait'+#13#10+
         'rec_version'+#13#10+
         'read_committed'+#13#10;
  end;
  DatabaseList.Add(Self);
  
end;

destructor TFIBDatabase.Destroy;
var
  i: Integer;
begin
  DatabaseList.Remove(Self);
  if vInternalTransaction.Active then vInternalTransaction.Commit;
  vInternalTransaction.Free; vInternalTransaction:=nil;
  if DefDataBase=Self then
    DefDataBase:=nil;
  if FIBLoaded then begin
   Timeout := 0;
   if FHandle <> nil then ForceClose;
   // Tell Dataset's we're being freed.
   for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
    DataSets[i].FOnDatabaseFree;
   // Make sure that all DataSets are removed from the list.
   RemoveDataSets;
   // Make sure all transactions are removed from the list.
   // As they are removed, they will remove the db's entry in each
   // respective transaction.
   RemoveTransactions;
   FIBAlloc(FDPB, 0, 0);
   FDBParams.Free;
   FDataSets.Free;
   FUserNames.Free;
   FTransactions.Free;
   FBackoutCount.Free;
   FDeleteCount.Free;
   FExpungeCount.Free;
   FInsertCount.Free;
   FPurgeCount.Free;
   FReadIdxCount.Free;
   FReadSeqCount.Free;
   FUpdateCount.Free;
   FConnectParams.Free;
  end; 
  inherited Destroy;
end;

procedure TFIBDatabase.SetDesignDBOptions(Value:TDesignDBOptions);
begin
 FDesignDBOptions:=Value;
 if ddoIsDefaultDatabase in FDesignDBOptions then begin
   if (DefDataBase<>nil) and (DefDataBase<>Self) then
   with DefDataBase do
    FDesignDBOptions:=FDesignDBOptions-[ddoIsDefaultDatabase];
   DefDataBase:=Self
 end
 else
 if DefDataBase=Self then DefDataBase:=nil
end;

function  TFIBDatabase.GetStoreConnected:boolean;
begin
  Result:=Connected and
   (ddoStoreConnected in FDesignDBOptions)
end;

function TFIBDatabase.Call(ErrCode: ISC_STATUS;
  RaiseError: Boolean): ISC_STATUS;
begin
  result := ErrCode;
  FCanTimeout := False;
  if RaiseError and (ErrCode > 0) then
    IBError(Self);
end;

procedure TFIBDatabase.CheckActive;
var OldUseLoginPrompt:boolean;
begin
  if FStreamedConnected and (not Connected) then
    Loaded;
  if FHandle = nil then
   if not (csDesigning in ComponentState) then
    FIBError(feDatabaseClosed, [nil])
   else 
     if not Connected then begin
      OldUseLoginPrompt:=FUseLoginPrompt;
      FUseLoginPrompt  :=(ConnectParams.Password='');
      try
       Connected:=true;
      except
       if not FUseLoginPrompt then
       try
        FUseLoginPrompt  :=true;
        Connected:=true;
       except
       end;
      end;
      FUseLoginPrompt:=OldUseLoginPrompt;
    end;

end;

procedure TFIBDatabase.CheckInactive;
begin
  if FHandle <> nil then
    FIBError(feDatabaseOpen, [nil]);
end;

procedure TFIBDatabase.CheckDatabaseName;
begin
  if (FDBName = '') then
    FIBError(feDatabaseNameMissing, [nil]);
end;

function TFIBDatabase.AddDataSet(ds: TFIBBase): Integer;
begin
  result := 0;
  while (result < FDataSets.Count) and (FDataSets[result] <> nil)
     and  (FDataSets[result] <> ds) //    AddedSource

  do
    Inc(result);
  if (result = FDataSets.Count) then
    FDataSets.Add(ds)
  else
    FDataSets[result] := ds;
end;

function TFIBDatabase.AddTransaction(TR: TFIBTransaction): Integer;
begin
  result := 0;
  while (result < FTransactions.Count) and (FTransactions[result] <> nil)
     and (FTransactions[result] <> TR) //    AddedSource
  do
    Inc(result);
  if (result = FTransactions.Count) then
    FTransactions.Add(TR)
  else
    FTransactions[result] := TR;
end;

procedure TFIBDatabase.Close;
begin
  InternalClose(False);
end;

procedure TFIBDatabase.CreateDatabase;
var
  tr_handle: TISC_TR_HANDLE;
begin
  // Create database interprets the DBParams string list
  // as mere text. It makes it extremely simple to do this way.
  CheckInactive; // Make sure the database ain't connected.
  tr_handle := nil;
  Call(
    isc_dsql_execute_immediate(StatusVector, @FHandle, @tr_handle, 0,
                               PChar('CREATE DATABASE ''' + FDBName + ''' ' +
                               DBParams.Text), SQLDialect, nil),
    True);
end;

procedure TFIBDatabase.DropDatabase;
begin
  CheckActive;
  Call(isc_drop_database(StatusVector, @FHandle), True);
end;

// Set up the FDPBBuffer correctly.
procedure TFIBDatabase.DBParamsChange(Sender: TObject);
begin
  FDBParamsChanged := True;
end;

procedure TFIBDatabase.DBParamsChanging(Sender: TObject);
begin
  CheckInactive;
end;

function TFIBDatabase.FindTransaction(TR: TFIBTransaction): Integer;
var
  i: Integer;
begin
  result := -1;
  for i := 0 to FTransactions.Count - 1 do
    if TR = Transactions[i] then begin
      result := i;
      break;
    end;
end;

procedure TFIBDatabase.ForceClose;
begin
  InternalClose(True);
end;

function TFIBDatabase.GetConnected: Boolean;
begin
  result := FHandle <> nil;
end;

function TFIBDatabase.GetDatabaseName: String;
begin
  result := FDBName;
end;

function TFIBDatabase.GetDataSet(Index: Integer): TFIBBase;
begin
  result := FDataSets[Index];
end;

function TFIBDatabase.GetDataSetCount: Integer;
var
  i: Integer;
begin
  result := 0;
  for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
    Inc(result);
end;

function TFIBDatabase.GetDBParamByDPB(const Idx: Integer): String;
var
  ConstIdx, EqualsIdx: Integer;
begin
  if (Idx > 0) and (Idx <= isc_dpb_last_dpb_constant) then begin
    ConstIdx := IndexOfDBConst(DPBConstantNames[Idx]);
    if ConstIdx = -1 then
      result := ''
    else begin
      result := DBParams[ConstIdx];
      EqualsIdx := Pos('=', result);
      if EqualsIdx = 0 then
        result := ''
      else
        result := Copy(result, EqualsIdx + 1, Length(result));
    end;
  end else
    result := '';
end;

function TFIBDatabase.GetTimeout: Integer;
begin
   Result := FTimer.Interval;
end;

function TFIBDatabase.GetTransaction(Index: Integer): TFIBTransaction;
begin
  result := FTransactions[Index];
end;

function TFIBDatabase.GetTransactionCount: Integer;
var
  i: Integer;
begin
  result := 0;
  for i := 0 to FTransactions.Count - 1 do if FTransactions[i] <> nil then
    Inc(result);
end;

function TFIBDatabase.IndexOfDBConst(st: String): Integer;
var
  i, pos_of_str: Integer;
begin
  result := -1;
  for i := 0 to DBParams.Count - 1 do begin
    pos_of_str := PosCI(st, DBParams[i]);
    if (pos_of_str = 1) or (pos_of_str = Length(DPBPrefix) + 1) then begin
      result := i;
      break;
    end;
  end;
end;

(*
 * InternalClose -
 *  Close the database connection and all other
 *  attached components that the database
 *  connection is terminating.
 *)
procedure TFIBDatabase.InternalClose(Force: Boolean);
var
  i: Integer;
begin
  (*
   * Check that the database connection is active.
   *)
  CheckActive;
  if Assigned(FBeforeDisconnect) then FBeforeDisconnect(Self); 
  (*
   * Tell all connected transactions that we're disconnecting.
   * This is so transactions can commit/rollback, accordingly
   *)
  for i := 0 to FTransactions.Count - 1 do begin
    try
      if FTransactions[i] <> nil then
        Transactions[i].OnDatabaseDisconnecting(Self);
    except
      if not Force then
        raise;
    end;
  end;
  (*
   * Tell all attached components (TFIBBase's) that we're
   * disconnecting
   *)
  for i := 0 to FDataSets.Count - 1 do begin
    try
      if FDataSets[i] <> nil then
        DataSets[i].FOnDatabaseDisconnecting;
    except
      if not Force then
        raise;
    end;
  end;
  (*
   * Disconnect..., and if the force parameter is true, guarantee that
   * the handle is reset to nil.
   *)
  if (not HandleIsShared) and
     (Call(isc_detach_database(StatusVector, @FHandle), False) > 0) and
     (not Force) then
    IbError(Self)
  else begin
    FHandle := nil;
    FHandleIsShared := False;
  end;
  if MonitorHook<>nil then
   MonitorHook.DBDisconnect(Self);

  (*
   * Tell all attached components (TFIBBase's) that we have
   * disconnected.
   *)
  for i := 0 to FDataSets.Count - 1 do
    if FDataSets[i] <> nil then
      DataSets[i].FOnDatabaseDisconnected;
  if Assigned(FAfterDisconnect) then FAfterDisconnect(Self);      
end;


procedure TFIBDatabase.Loaded;
begin
  inherited Loaded;
  try
    if FStreamedConnected and (not Connected) then begin
      Open;
      if (FDefaultTransaction <> nil) and
         (FDefaultTransaction.FStreamedActive) and
         (not FDefaultTransaction.InTransaction) then
        FDefaultTransaction.StartTransaction;
      FStreamedConnected := False;
    end;
  except

    if csDesigning in ComponentState then
      Application.HandleException(Self)
    else
      raise;
  end;
end;

function TFIBDatabase.Login: Boolean;
var
  IndexOfUser, IndexOfPassword: Integer;
  Username, Password: String;
begin
  IndexOfUser := IndexOfDBConst(DPBConstantNames[isc_dpb_user_name]);
  if IndexOfUser <> -1 then
    Username := Copy(DBParams[IndexOfUser],
                                       Pos('=', DBParams[IndexOfUser]) + 1,
                                       Length(DBParams[IndexOfUser]));
  IndexOfPassword := IndexOfDBConst(DPBConstantNames[isc_dpb_password]);
  if IndexOfPassword <> -1 then
    Password := Copy(DBParams[IndexOfPassword],
                                       Pos('=', DBParams[IndexOfPassword]) + 1,
                                       Length(DBParams[IndexOfPassword]));

  Result := LoginDialogEx(DBName, Username, Password, False);
  if result then begin
    IndexOfPassword := IndexOfDBConst(DPBConstantNames[isc_dpb_password]);
    if IndexOfUser = -1 then
      DBParams.Add(DPBConstantNames[isc_dpb_user_name] + '=' + Username)
    else
      DBParams[IndexOfUser] := DPBConstantNames[isc_dpb_user_name] +
                               '=' + Username;
    if IndexOfPassword = -1 then
      DBParams.Add(DPBConstantNames[isc_dpb_password] + '=' + Password)
    else
      DBParams[IndexOfPassword] := DPBConstantNames[isc_dpb_password] +
                                   '=' + Password;
  end;
end;

(*
 * Open -
 *  Open a database connection.
 *)
procedure TFIBDatabase.Open;
var
  DPB: String;
  isc_res:ISC_STATUS;
  vD     :Variant;
  i: Integer;  
begin
  (*
   * Check that the database is *not* active, and that it
   * has a database name
   *)
  CheckInactive;
  CheckDatabaseName;
  (*
   * Use the canned login prompt if requested.
   *)
  for i := 0 to FDataSets.Count - 1 do
  begin
     if FDataSets[i] <> nil then
       DataSets[i].FOnDatabaseConnecting;
  end;

  if FUseLoginPrompt and not Login then
    FIBError(feOperationCancelled, [nil]);
  (*
   * Generate a new DPB if necessary
   *)
  if FDBParamsChanged then begin
    FDBParamsChanged := False;
    GenerateDPB(FDBParams, DPB, FDPBLength);
    FIBAlloc(FDPB, 0, FDPBLength);
    Move(DPB[1], FDPB[0], FDPBLength);
  end;
  (*
   * Attach to the database, and raise an IB error if
   * the statement doesn't execute correctly.
   *)
  isc_res:=Call(isc_attach_database(StatusVector, Length(FDBName),
                         PChar(FDBName), @FHandle,
                         FDPBLength, FDPB), False);
  if isc_res > 0 then begin
    FHandle := nil;
    IbError(Self);
  end;
  DPB:=FDPB;
  for i := 0 to FDataSets.Count - 1 do
  begin
   if FDataSets[i] <> nil then
      DataSets[i].FOnDatabaseConnected;
  end;

  if Assigned(FOnConnect) then
    FOnConnect(Self);
  if MonitorHook<>nil then
   MonitorHook.DBConnect(Self);
  FDifferenceTime:=0;
  if FSynchronizeTime and not (csDesigning in ComponentState) then begin
    if FSQLDialect<2 then
     vD:=QueryValue('SELECT CAST(''NOW'' AS DATE) from RDB$DATABASE',0)
    else
     vD:=QueryValue('SELECT CAST(''NOW'' AS TIMESTAMP) from RDB$DATABASE',0)
    ;
    if VarType(vD)<>varBoolean then
     FDifferenceTime:=Now-vD
  end;
end;

procedure TFIBDatabase.RemoveDataSet(Idx: Integer);
var
  ds: TFIBBase;
begin
  if (Idx >= 0) and (FDataSets[Idx] <> nil) then begin
    ds := DataSets[Idx];
    FDataSets[Idx] := nil;
    ds.Database := nil;
    // FDataSets.Pack;
  end;
end;

procedure TFIBDatabase.RemoveDataSets;
var
  i: Integer;
begin
  for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
    RemoveDataSet(i);
end;

procedure TFIBDatabase.RemoveTransaction(Idx: Integer);
var
  TR: TFIBTransaction;
begin
  if ((Idx >= 0) and (FTransactions[Idx] <> nil)) then begin
    TR := Transactions[Idx];
    FTransactions[Idx] := nil;
    TR.RemoveDatabase(TR.FindDatabase(Self));
    if TR = FDefaultTransaction then
      FDefaultTransaction := nil;
  end;
end;

procedure TFIBDatabase.RemoveTransactions;
var
  i: Integer;
begin
  for i := 0 to FTransactions.Count - 1 do if FTransactions[i] <> nil then
    RemoveTransaction(i);
end;

procedure TFIBDatabase.SetConnected(Value: Boolean);
begin
  if csReading in ComponentState then
    FStreamedConnected := Value
  else if Value then
    Open
  else
    Close;
end;

procedure TFIBDatabase.SetDatabaseName(const Value: string);
begin
  if FDBName <> Value then begin
    CheckInactive;
    FDBName := Value;
    ListTableInfo.ClearForDataBase(Self)
  end;
end;

procedure TFIBDatabase.SetDBParamByName(const ParName,Value: String);
var
  ConstIdx: Integer;
begin
  ConstIdx := IndexOfDBConst(ParName);
  if (Value = '') then begin
    if ConstIdx <> -1 then
      DBParams.Delete(ConstIdx);
  end else begin
    if (ConstIdx = -1) then
      DBParams.Add(ParName + '=' + Value)
    else
      DBParams[ConstIdx] := ParName + '=' + Value;
  end;
end;

procedure TFIBDatabase.SetDBParamByDPB(const Idx: Integer; Value: String);
begin
  SetDBParamByName(DPBConstantNames[Idx],Value);
end;

procedure TFIBDatabase.SetDBParams(Value: TStrings);
begin
  FDBParams.Assign(Value);
end;

procedure TFIBDatabase.SetDefaultTransaction(Value: TFIBTransaction);
begin
  if Assigned(Value) and (FDefaultTransaction <> Value) then begin
    Value.AddDatabase(Self);
    AddTransaction(Value);
  end;
  FDefaultTransaction := Value;
end;

procedure TFIBDatabase.SetHandle(Value: TISC_DB_HANDLE);
begin
  if HandleIsShared then
    Close
  else
    CheckInactive;
  FHandle := Value;
  FHandleIsShared := (Value <> nil);
end;

procedure TFIBDatabase.SetTimeout(Value: Integer);
begin
  if Value < 0 then
    FIBError(feTimeoutNegative, [nil])
  else if (Value = 0) then begin
    FTimer.Enabled := False;
    FTimer.Interval := 0;
  end else if (Value > 0) then begin
    FTimer.Interval := Value;
    if not (csDesigning in ComponentState) then
      FTimer.Enabled := True;
  end;
end;

function TFIBDatabase.TestConnected: Boolean;
begin
  result := Connected;
  if result then begin
    try
      if BaseLevel = 0 then ; // who cares
    except
      ForceClose;
      result := False;
    end;
  end;
end;

procedure TFIBDatabase.TimeoutConnection(Sender: TObject);
begin
  if Connected then begin
    if FCanTimeout then begin
      ForceClose;
      if Assigned(FOnTimeout) then
        FOnTimeout(Self);
    end else begin
      FCanTimeout := True;
    end;
  end;
end;

(* Database Info procedures -- Advanced stuff (translated from isc_database_info) *)
function TFIBDatabase.GetAllocation: Long;
begin
  result := GetLongDBInfo(isc_info_allocation);
end;

function TFIBDatabase.GetBaseLevel: Long;
var
  local_buffer: array[0..FIBLocalBufferLength - 1] of Char;
  DBInfoCommand: Char;
begin
  DBInfoCommand := Char(isc_info_base_level);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                         FIBLocalBufferLength, local_buffer), True);
  result := isc_vax_integer(@local_buffer[4], 1);
end;

function TFIBDatabase.GetDBFileName: String;
var
  local_buffer: array[0..FIBLocalBufferLength - 1] of Char;
  DBInfoCommand: Char;
begin
  DBInfoCommand := Char(isc_info_db_id);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                         FIBLocalBufferLength, local_buffer), True);
  local_buffer[5 + Int(local_buffer[4])] := #0;
  result := String(PChar(@local_buffer[5]));
end;

function TFIBDatabase.GetDBSiteName: String;
var
  local_buffer: array[0..FIBBigLocalBufferLength - 1] of Char;
  p: PChar;
  DBInfoCommand: Char;
begin
  DBInfoCommand := Char(isc_info_db_id);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                        FIBLocalBufferLength, local_buffer), True);
  p := @local_buffer[5 + Int(local_buffer[4])]; // DBSiteName Length
  p := p + Int(p^) + 1;                         // End of DBSiteName
  p^ := #0;                                     // Null it.
  result := String(PChar(@local_buffer[6 + Int(local_buffer[4])]));
end;

function TFIBDatabase.GetDBImplementationNo: Long;
var
  local_buffer: array[0..FIBLocalBufferLength - 1] of Char;
  DBInfoCommand: Char;
begin
  DBInfoCommand := Char(isc_info_implementation);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                        FIBLocalBufferLength, local_buffer), True);
  result := isc_vax_integer(@local_buffer[3], 1);
end;

function TFIBDatabase.GetDBImplementationClass: Long;
var
  local_buffer: array[0..FIBLocalBufferLength - 1] of Char;
  DBInfoCommand: Char;
begin
  DBInfoCommand := Char(isc_info_implementation);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                         FIBLocalBufferLength, local_buffer), True);
  result := isc_vax_integer(@local_buffer[4], 1);
end;

function TFIBDatabase.GetNoReserve: Long;
begin
  result := GetLongDBInfo(isc_info_no_reserve);
end;

function TFIBDatabase.GetODSMinorVersion: Long;
begin
  result := GetLongDBInfo(isc_info_ods_minor_version);
end;

function TFIBDatabase.GetODSMajorVersion: Long;
begin
  result := GetLongDBInfo(isc_info_ods_version);
end;

function TFIBDatabase.GetPageSize: Long;
begin
  result := GetLongDBInfo(isc_info_page_size);
end;

function TFIBDatabase.GetVersion: String;
var
  local_buffer: array[0..FIBBigLocalBufferLength - 1] of Char;
  DBInfoCommand: Char;
begin
  DBInfoCommand := Char(isc_info_version);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                        FIBBigLocalBufferLength, local_buffer), True);
  local_buffer[5 + Int(local_buffer[4])] := #0;
  result := String(PChar(@local_buffer[5]));
end;

function TFIBDatabase.GetCurrentMemory: Long;
begin
  result := GetLongDBInfo(isc_info_current_memory);
end;

function TFIBDatabase.GetForcedWrites: Long;
begin
  result := GetLongDBInfo(isc_info_forced_writes);
end;

function TFIBDatabase.GetMaxMemory: Long;
begin
  result := GetLongDBInfo(isc_info_max_memory);
end;

function TFIBDatabase.GetNumBuffers: Long;
begin
  result := GetLongDBInfo(isc_info_num_buffers);
end;

function TFIBDatabase.GetSweepInterval: Long; // isc_info_sweep_interval
begin
  result := GetLongDBInfo(isc_info_sweep_interval);
end;

function TFIBDatabase.GetUserNames: TStringList;
var
  local_buffer: array[0..FIBHugeLocalBufferLength - 1] of Char;
  temp_buffer: array[0..FIBLocalBufferLength - 2] of Char;
  DBInfoCommand: Char;
  i, user_length: Integer;
begin
  if FUserNames = nil then FUserNames := TStringList.Create;
  result := FUserNames;
  DBInfoCommand := Char(isc_info_user_names);
  Call(isc_database_info(StatusVector, @FHandle, 1, @DBInfoCommand,
                        FIBHugeLocalBufferLength, local_buffer), True);
  FUserNames.Clear;
  i := 0;
  while local_buffer[i] = Char(isc_info_user_names) do
  begin
    Inc(i, 3); // skip "isc_info_user_names byte" & two unknown bytes of structure (see below)
    user_length := Long(local_buffer[i]);
    Inc(i,1);
    Move(local_buffer[i], temp_buffer[0], user_length);
    Inc(i, user_length);
    temp_buffer[user_length] := #0;
    FUserNames.Add(String(temp_buffer));
  end;
end;

function TFIBDatabase.GetFetches: Long;
begin
  result := GetLongDBInfo(isc_info_fetches);
end;

function TFIBDatabase.GetReadOnly: Long;
begin
  result := GetLongDBInfo(isc_info_db_read_only);
end;

function TFIBDatabase.GetMarks: Long;
begin
  result := GetLongDBInfo(isc_info_marks);
end;

function TFIBDatabase.GetReads: Long;
begin
  result := GetLongDBInfo(isc_info_reads);
end;

function TFIBDatabase.GetWrites: Long;
begin
  result := GetLongDBInfo(isc_info_writes);
end;

function TFIBDatabase.GetOperationCounts(DBInfoCommand: Integer; var FOperation: TStringList): TStringList;
var
  local_buffer: array[0..FIBHugeLocalBufferLength - 1] of Char;
  _DBInfoCommand: Char;
  i, qtd_tables, id_table, qtd_operations: Integer;
begin
  if FOperation = nil then FOperation := TStringList.Create;
  result := FOperation;
  _DBInfoCommand := Char(DBInfoCommand);
  Call(isc_database_info(StatusVector, @FHandle, 1, @_DBInfoCommand,
                         FIBHugeLocalBufferLength, local_buffer), True);
  FOperation.Clear;
  // 1. 1 byte specifying the item type requested (e.g., isc_info_insert_count).
  // 2. 2 bytes telling how many bytes compose the subsequent value pairs.
  // 3. A pair of values for each table in the database on wich the requested
  //    type of operation has occurred since the database was last attached.
  // Each pair consists of:
  // 1. 2 bytes specifying the table ID.
  // 2. 4 bytes listing the number of operations (e.g., inserts) done on that table.
  qtd_tables := trunc(isc_vax_integer(@local_buffer[1],2)/6);
  for i := 0 to qtd_tables - 1 do
  begin
    id_table := isc_vax_integer(@local_buffer[3+(i*6)],2);
    qtd_operations := isc_vax_integer(@local_buffer[5+(i*6)],4);
    FOperation.Add(IntToStr(id_table)+'='+IntToStr(qtd_operations));
  end;
end;

function TFIBDatabase.GetBackoutCount: TStringList;
begin
  result := GetOperationCounts(isc_info_backout_count,FBackoutCount);
end;

function TFIBDatabase.GetDeleteCount: TStringList;
begin
  result := GetOperationCounts(isc_info_delete_count,FDeleteCount);
end;

function TFIBDatabase.GetExpungeCount: TStringList;
begin
  result := GetOperationCounts(isc_info_expunge_count,FExpungeCount);
end;

function TFIBDatabase.GetInsertCount: TStringList;
begin
  result := GetOperationCounts(isc_info_insert_count,FInsertCount);
end;

function TFIBDatabase.GetPurgeCount: TStringList;
begin
  result := GetOperationCounts(isc_info_purge_count,FPurgeCount);
end;

function TFIBDatabase.GetReadIdxCount: TStringList;
begin
  result := GetOperationCounts(isc_info_read_idx_count,FReadIdxCount);
end;

function TFIBDatabase.GetReadSeqCount: TStringList;
begin
  result := GetOperationCounts(isc_info_read_seq_count,FReadSeqCount);
end;

function TFIBDatabase.GetUpdateCount: TStringList;
begin
  result := GetOperationCounts(isc_info_update_count,FUpdateCount);
end;

function TFIBDatabase.GetLogFile: Long;
begin
  result := GetLongDBInfo(isc_info_logfile);
end;

function TFIBDatabase.GetCurLogFileName: String;
begin
  result := GetStringDBInfo(isc_info_cur_logfile_name);
end;

function TFIBDatabase.GetCurLogPartitionOffset: Long;
begin
  result := GetLongDBInfo(isc_info_cur_log_part_offset);
end;

function TFIBDatabase.GetNumWALBuffers: Long;
begin
  result := GetLongDBInfo(isc_info_num_wal_buffers);
end;

function TFIBDatabase.GetWALBufferSize: Long;
begin
  result := GetLongDBInfo(isc_info_wal_buffer_size);
end;

function TFIBDatabase.GetWALCheckpointLength: Long;
begin
  result := GetLongDBInfo(isc_info_wal_ckpt_length);
end;

function TFIBDatabase.GetWALCurCheckpointInterval: Long;
begin
  result := GetLongDBInfo(isc_info_wal_cur_ckpt_interval);
end;

function TFIBDatabase.GetWALPrvCheckpointFilename: String;
begin
  result := GetStringDBInfo(isc_info_wal_prv_ckpt_fname);
end;

function TFIBDatabase.GetWALPrvCheckpointPartOffset: Long;
begin
  result := GetLongDBInfo(isc_info_wal_prv_ckpt_poffset);
end;

function TFIBDatabase.GetWALGroupCommitWaitUSecs: Long;
begin
  result := GetLongDBInfo(isc_info_wal_grpc_wait_usecs);
end;

function TFIBDatabase.GetWALNumIO: Long;
begin
  result := GetLongDBInfo(isc_info_wal_num_io);
end;

function TFIBDatabase.GetWALAverageIOSize: Long;
begin
  result := GetLongDBInfo(isc_info_wal_avg_io_size);
end;

function TFIBDatabase.GetWALNumCommits: Long;
begin
  result := GetLongDBInfo(isc_info_wal_num_commits);
end;

function TFIBDatabase.GetWALAverageGroupCommitSize: Long;
begin
  result := GetLongDBInfo(isc_info_wal_avg_grpc_size);
end;

function TFIBDatabase.GetProtectLongDBInfo
 (DBInfoCommand: Integer;var Success:boolean): Long;
var
  local_buffer: array[0..FIBLocalBufferLength - 1] of Char;
  length: Integer;
  _DBInfoCommand: Char;
begin
  _DBInfoCommand := Char(DBInfoCommand);
  Call(isc_database_info(StatusVector, @FHandle, 1, @_DBInfoCommand,
                         FIBLocalBufferLength, local_buffer), True);
  Success:=local_buffer[0] = _DBInfoCommand;
  length := isc_vax_integer(@local_buffer[1], 2);
  result := isc_vax_integer(@local_buffer[3], length);
end;

function TFIBDatabase.GetLongDBInfo(DBInfoCommand: Integer): Long;
var Success:boolean;
begin
  result := GetProtectLongDBInfo(DBInfoCommand,Success)
end;

function TFIBDatabase.GetStringDBInfo(DBInfoCommand: Integer): String;
var
  local_buffer: array[0..FIBBigLocalBufferLength - 1] of Char;
  _DBInfoCommand: Char;
begin
  _DBInfoCommand := Char(DBInfoCommand);
  Call(isc_database_info(StatusVector, @FHandle, 1, @_DBInfoCommand,
                         FIBBigLocalBufferLength, local_buffer), True);
  local_buffer[4 + Int(local_buffer[3])] := #0;
  result := String(PChar(@local_buffer[4]));
end;
//Wrappers DBParams

//IB6
function TFIBDatabase.GetDBSQLDialect:Word;
var Success:boolean;
begin
 Result:=GetProtectLongDBInfo(isc_info_db_SQL_dialect,Success);
 if not Success then Result:=1
end;


procedure TFIBDatabase.SetSQLDialect(const Value: Integer);
begin
  if (Value < 1) then FIBError(feSQLDialectInvalid, [nil]);
  if ((FHandle = nil) or (Value <= DBSQLDialect))  then
   FSQLDialect := Value
  else
   FIBError(feSQLDialectInvalid, [nil])
end;

procedure TFIBDatabase.ApplyUpdates(const DataSets: array of TDataSet);
var I: Integer;
    DS: TFIBCustomDataSet;
    TR: TFIBTransaction;
begin
  TR := nil;
  for I := 0 to High(DataSets) do begin
    DS := TFIBCustomDataSet(DataSets[I]);
    if DS.Database <> Self then FIBError(feUpdateWrongDB, [nil]);
    if TR = nil then TR := DS.Transaction;
    if (DS.Transaction <> TR) or (TR = nil) then
      FIBError(feUpdateWrongTR, [nil]);
  end;
  TR.CheckInTransaction;
  for I := 0 to High(DataSets) do
    if DataSets[I] is TpFIBDataSet then
      TpFIBDataSet(DataSets[I]).ApplyUpdToBase;
  TR.CommitRetaining;
  for I := 0 to High(DataSets) do
    if DataSets[I] is TpFIBDataSet then
      TpFIBDataSet(DataSets[I]).CommitUpdToCach;
end;


procedure TFIBDatabase.StartTransaction;
begin
  if DefaultTransaction = nil then
    FIBError(feTransactionNotAssigned, [nil]);
  DefaultTransaction.StartTransaction;
end;

procedure TFIBDatabase.CloseDataSets;
var Index: Integer;
begin
  for Index := 0 to pred(DataSetCount) do
    if (DataSets[Index].Owner is TFIBDataSet) then
      TFIBDataSet(DataSets[Index].Owner).Close;
end;

procedure TFIBDatabase.Commit;
begin
  if DefaultTransaction = nil then
    FIBError(feTransactionNotAssigned, [nil]);
  DefaultTransaction.Commit;
end;

procedure TFIBDatabase.Rollback;
begin
  if DefaultTransaction = nil then
    FIBError(feTransactionNotAssigned, [nil]);
  DefaultTransaction.Rollback;
end;

procedure TFIBDatabase.CommitRetaining;
begin
  if DefaultTransaction = nil then
    FIBError(feTransactionNotAssigned, [nil]);
  DefaultTransaction.CommitRetaining;
end;

procedure TFIBDatabase.RollbackRetaining;
begin
  if DefaultTransaction = nil then
    FIBError(feTransactionNotAssigned, [nil]);
  DefaultTransaction.RollbackRetaining;
end;

function TFIBDatabase.QueryValue(const SQL: string;FieldNo:integer):Variant;
var Query: TFIBQuery;
begin
  Result := false;
  Query := TFIBQuery.Create(nil);
  Query.SQL.Text := SQL;
  Query.Database := Self;
  Query.Transaction := vInternalTransaction;
  try
   try
    if not Query.Transaction.Active then
     Query.Transaction.StartTransaction;
    Query.ExecQuery;
    if (Query.Current.Count>0)and (FieldNo>=0) then
     Result := Query.Fields[FieldNo].Value;
   except
    Query.Transaction.RollBack;
   end
  finally
    Query.Transaction.Commit;
    Query.Free;
  end;
end;


function TFIBDatabase.Execute(const SQL: string): boolean;
begin
  Result := false;
  try
   QueryValue(SQL,-1);
   Result := true;
  except
  end
end;

{$IFDEF D45}
function TFIBDatabase.Gen_Id(const GeneratorName: string; Step: Int64): Int64;
{$ELSE}
function TFIBDatabase.Gen_Id(const GeneratorName: string; Step: Comp): Comp;
{$ENDIF}
var Query: TFIBQuery;
begin
  Result := 0;

  Query := TFIBQuery.Create(nil);
  Query.SQL.Text := 'select gen_id(' +
    FormatIdentifier(SQLDialect, GeneratorName) + ', ' +
      IntToStr(
{$IFDEF D45}
       Step
{$ELSE}
       trunc(Step)
{$ENDIF}
       ) +
    ') from RDB$DATABASE';
  Query.Database    := Self;
  Query.Transaction := vInternalTransaction;
  if not Query.Transaction.Active then
   Query.Transaction.StartTransaction;
  try
    Query.ExecQuery;
{$IFDEF D45}
    Result := Query.Fields[0].AsInt64;
{$ELSE}
    Result := Query.Fields[0].AsComp;
{$ENDIF}
  except
  end;
  Query.Transaction.Commit;
  Query.Free;
end;



(* TFIBTransaction *)

constructor TFIBTransaction.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDatabases                           := TList.Create;
  FDataSets                            := TList.Create;
  FHandle                              := nil;
  FTPB                                 := nil;
  FTPBLength                           := 0;
  FTRParams                            := TStringList.Create;
  FTRParamsChanged                     := True;
  TStringList(FTRParams).OnChange      := TRParamsChange;
  TStringList(FTRParams).OnChanging    := TRParamsChanging;
  FTimer                               := TTimer.Create(Self);
  FTimer.Enabled                       := False;
  FTimer.Interval                      := 0;
  FTimer.OnTimer                       := TimeoutTransaction;
  FState                               := tsClosed;
  if (csDesigning in ComponentState)
  and ((Owner<>nil) and  not (csLoading in Owner.ComponentState))
  then
  DefaultDataBase                      :=DefDataBase;
end;

destructor TFIBTransaction.Destroy;
var
  i: Integer;
begin
  if InTransaction then
    EndTransaction(FTimeoutAction, True);
  for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
    DataSets[i].FOnTransactionFree;
  RemoveDataSets;
  RemoveDatabases;
  FIBAlloc(FTPB, 0, 0);
  FTRParams.Free;
  FDataSets.Free;
  FDatabases.Free;
  inherited Destroy;
end;

function TFIBTransaction.DoStoreActive:boolean;
begin
 Result:=Active and DefaultDataBase.StoreConnected
end;

function TFIBTransaction.Call(ErrCode: ISC_STATUS;
  RaiseError: Boolean): ISC_STATUS;
var
  i: Integer;
begin
  result := ErrCode;
  for i := 0 to FDatabases.Count - 1 do if FDatabases[i] <> nil then
    Databases[i].FCanTimeout := False;
  FCanTimeout := False;
  if RaiseError and (result > 0) then
    IbError(Self);
end;

procedure TFIBTransaction.CheckDatabasesInList;
begin
  if GetDatabaseCount = 0 then
    FIBError(feNoDatabasesInTransaction, [nil]);
end;

procedure TFIBTransaction.CheckInTransaction;
begin
  if FStreamedActive and (not InTransaction) then
    Loaded;
  if (FHandle = nil)  then
   if not (csDesigning in ComponentState)  then
    FIBError(feNotInTransaction, [nil])
   else
    StartTransaction;
end;

procedure TFIBTransaction.CheckNotInTransaction;
begin
  if (FHandle <> nil) then
    FIBError(feInTransaction, [nil]);
end;

function TFIBTransaction.AddDatabase(db: TFIBDatabase): Integer;
var
  i: Integer;
  nil_found: Boolean;
begin
  i := FindDatabase(db);
  if i <> -1 then begin
    result := i;
    exit;
  end;
  nil_found := False;
  i := 0;
  while (not nil_found) and (i < FDatabases.Count) do begin
    nil_found := (FDatabases[i] = nil);
    if (not nil_found) then Inc(i);
  end;
  if (nil_found) then begin
    FDatabases[i] := db;
    result := i;
  end else begin
    result := FDatabases.Count;
    FDatabases.Add(db);
  end;
end;

function TFIBTransaction.AddDataSet(ds: TFIBBase): Integer;
begin
  result := 0;
  while (result < FDataSets.Count) and (FDataSets[result] <> nil) do
    Inc(result);
  if (result = FDataSets.Count) then
    FDataSets.Add(ds)
  else
    FDataSets[result] := ds;
end;

procedure TFIBTransaction.Commit;
begin
  EndTransaction(TACommit, False);
end;

procedure TFIBTransaction.CommitRetaining;
begin
  EndTransaction(TACommitRetaining, False);
end;

(*
 * EndTransaction -
 *  End a transaction ...
 *)
procedure TFIBTransaction.EndTransaction(Action: TTransactionAction;
  Force: Boolean);
var
  status: ISC_STATUS;
  i: Integer;
begin
  CheckInTransaction;
  case Action of
    TARollback, TACommit: begin
      if (HandleIsShared) and
         (Action <> FTimeoutAction) and
         (not Force) then
        FIBError(feCantEndSharedTransaction, [nil]);
      if Action=TARollback then
       FState:=tsDoRollback
      else
       FState:=tsDoCommit;
      for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
        DataSets[i].FOnTransactionEnding;
      if HandleIsShared then begin
        FHandle := nil;
        FHandleIsShared := False;
        status := 0;
      end else if (Action = TARollback) then
        status := Call(isc_rollback_transaction(StatusVector, @FHandle), False)
      else
        status := Call(isc_commit_transaction(StatusVector, @FHandle), False);
      if ((Force) and (status > 0)) then
        status := Call(isc_rollback_transaction(StatusVector, @FHandle), False);
      if Force then
        FHandle := nil
      else if (status > 0) then
        IBError(Self);
      for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
        DataSets[i].FOnTransactionEnded;
    end;
    TACommitRetaining:
    begin
     FState:=tsDoCommitRetaining;
     for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
        DataSets[i].FOnTransactionEnding;
      Call(isc_commit_retaining(StatusVector, @FHandle), True);
      for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
        DataSets[i].FOnTransactionEnded;
    end;
    TARollbackRetaining: begin
      FState:=tsDoRollbackRetaining;
      for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
        DataSets[i].FOnTransactionEnding;
      Call(isc_rollback_retaining(StatusVector, @FHandle), True);
      for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
        DataSets[i].FOnTransactionEnded;
    end;
  end;
  case Action of
    TACommit:
    if MonitorHook<>nil then
      MonitorHook.TRCommit(Self);
    TARollback:
    if MonitorHook<>nil then
      MonitorHook.TRRollback(Self);
    TACommitRetaining:
    if MonitorHook<>nil then
      MonitorHook.TRCommitRetaining(Self);
    TARollbackRetaining:
    if MonitorHook<>nil then
      MonitorHook.TRRollbackRetaining(Self);
  end;
  if InTransaction then
   FState:=tsActive
  else
   FState:=tsClosed
end;

function TFIBTransaction.GetDatabase(Index: Integer): TFIBDatabase;
begin
  result := FDatabases[Index];
end;

function TFIBTransaction.GetDatabaseCount: Integer;
var
  i, Cnt: Integer;
begin
  result := 0;
  Cnt := FDatabases.Count - 1;
  for i := 0 to Cnt do if FDatabases[i] <> nil then
    Inc(result);
end;

function TFIBTransaction.GetDataSet(Index: Integer): TFIBBase;
begin
  result := FDataSets[Index];
end;

function TFIBTransaction.GetDataSetCount: Integer;
var
  i, Cnt: Integer;
begin
  result := 0;
  Cnt := FDataSets.Count - 1;
  for i := 0 to Cnt do if FDataSets[i] <> nil then
    Inc(result);
end;

function TFIBTransaction.GetInTransaction: Boolean;
begin
  result := (FHandle <> nil);
end;

function TFIBTransaction.FindDatabase(db: TFIBDatabase): Integer;
var
  i: Integer;
begin
  result := -1;
  for i := 0 to FDatabases.Count - 1 do
    if db = TFIBDatabase(FDatabases[i]) then begin
      result := i;
      break;
    end;
end;

function TFIBTransaction.GetTimeout: Integer;
begin
   result := FTimer.Interval;
end;

procedure TFIBTransaction.Loaded;
begin
  inherited Loaded;
  try
   if FStreamedActive and (not InTransaction) then StartTransaction;
  except
    if csDesigning in ComponentState then
      Application.HandleException(Self)
    else
      raise;
  end;
end;

procedure TFIBTransaction.OnDatabaseDisconnecting(DB: TFIBDatabase);
begin
  if InTransaction then
    // Force the transaction to end
  if FTimeoutAction= TACommitRetaining then
    EndTransaction(TACommit, True)
  else
    EndTransaction(FTimeoutAction, True);
end;

procedure TFIBTransaction.RemoveDatabase(Idx: Integer);
var
  DB: TFIBDatabase;
begin
  if ((Idx >= 0) and (FDatabases[Idx] <> nil)) then begin
    DB := Databases[Idx];
    FDatabases[Idx] := nil;
    DB.RemoveTransaction(DB.FindTransaction(Self));
    if DB = FDefaultDatabase then 
      FDefaultDatabase := nil;

  end;
end;

procedure TFIBTransaction.RemoveDatabases;
var
  i: Integer;
begin
  for i := 0 to FDatabases.Count - 1 do if FDatabases[i] <> nil then
    RemoveDatabase(i);
end;

procedure TFIBTransaction.RemoveDataSet(Idx: Integer);
var
  ds: TFIBBase;
begin
  if ((Idx >= 0) and (FDataSets[Idx] <> nil)) then begin
    ds := DataSets[Idx];
    FDataSets[Idx] := nil;
    ds.Transaction := nil;
  end;
end;

procedure TFIBTransaction.RemoveDataSets;
var
  i: Integer;
begin
  for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
    RemoveDataSet(i);
end;

procedure TFIBTransaction.Rollback;
begin
  EndTransaction(TARollback, False);
end;

procedure TFIBTransaction.RollbackRetaining;
begin
   EndTransaction(TARollbackRetaining, False);
end;

procedure TFIBTransaction.SetActive(Value: Boolean);
begin
  if csReading in ComponentState then
    FStreamedActive := Value
  else if Value and not InTransaction then
    StartTransaction
  else if not Value and InTransaction then
    Rollback;
end;

procedure TFIBTransaction.SetDefaultDatabase(Value: TFIBDatabase);
begin
  if Assigned(Value) and (Value <> FDefaultDatabase) then begin
    Value.AddTransaction(Self);
    AddDatabase(Value);
  end;
  FDefaultDatabase := Value;
end;

procedure TFIBTransaction.SetHandle(Value: TISC_TR_HANDLE);
begin
  if (HandleIsShared) then
    EndTransaction(TimeoutAction, True)
  else
    CheckNotInTransaction;
  FHandle := Value;
  FHandleIsShared := (Value <> nil);
end;

procedure TFIBTransaction.SetTimeout(Value: Integer);
begin
  if Value < 0 then
    FIBError(feTimeoutNegative, [nil])
  else if (Value = 0) then begin
    FTimer.Enabled := False;
    FTimer.Interval := 0;
  end else if (Value > 0) then begin
    FTimer.Interval := Value;
    if not (csDesigning in ComponentState) then
      FTimer.Enabled := True;
  end;
end;

procedure TFIBTransaction.SetTRParams(Value: TStrings);
begin
  FTRParams.Assign(Value);
end;

(*
 * StartTransaction -
 *  Start a transaction....
 *)
procedure TFIBTransaction.StartTransaction;
var
  pteb: PISC_TEB_ARRAY;
  TPB: String;
  i: Integer;
begin
  (*
   * Check that we're not already in a transaction.
   * Check that there is at least one database in the list.
   *)
  CheckNotInTransaction;
  CheckDatabasesInList;
  (*
   * Make sure that a current TPB is generated.
   *)
  if FTRParamsChanged then begin
    FTRParamsChanged := False;
    GenerateTPB(FTRParams, TPB, FTPBLength);
    FIBAlloc(FTPB, 0, FTPBLength);
    Move(TPB[1], FTPB[0], FTPBLength);
  end;
  (*
   * Set up a database record for each database in the list.
   * Ordinarily, each database could have its own set of
   * transaction options; however, FIB makes a transaction have
   * one set of transaction options..
   * Is this a limitation? Certainly. It this bad?
   * I don't think so...
   *)
  pteb := nil;
  FIBAlloc(pteb, 0, DatabaseCount * SizeOf(TISC_TEB));
  try
    for i := 0 to DatabaseCount - 1 do if Databases[i] <> nil then begin
      Databases[i].CheckActive; //Added Source  // Alexander Samusenko 2:452/999.8

      pteb^[i].db_handle := @(Databases[i].Handle);
      pteb^[i].tpb_length := FTPBLength;
      pteb^[i].tpb_address := FTPB;
    end;
    for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
     DataSets[i].FOnTransactionStarting;

    (*
     * Finally, start the transaction
     *)
    if (Call(isc_start_multiple(StatusVector, @FHandle,
                               DatabaseCount, PISC_TEB(pteb)), False) > 0)

      {and(csLoading in ComponentState)} //Added Source Alexander Samusenko

    then begin
      FHandle := nil;
      IbError(Self);
    end;
    for i := 0 to FDataSets.Count - 1 do if FDataSets[i] <> nil then
      DataSets[i].FOnTransactionStarted;

    if MonitorHook<>nil then
     MonitorHook.TRStart(Self);
  finally
    FIBAlloc(pteb, 0, 0);
  end;
  FState:=tsActive
end;

procedure TFIBTransaction.TimeoutTransaction(Sender: TObject);
begin
  if InTransaction then begin
    if FCanTimeout then begin
      EndTransaction(FTimeoutAction, True);
      if Assigned(FOnTimeout) then
        FOnTimeout(Self);
    end else begin
      FCanTimeout := True;
    end;
  end;
end;

procedure TFIBTransaction.TRParamsChange(Sender: TObject);
begin
  FTRParamsChanged := True;
end;

procedure TFIBTransaction.TRParamsChanging(Sender: TObject);
begin
  CheckNotInTransaction;
end;

(* TFIBBase *)
constructor TFIBBase.Create(AOwner: TObject);
begin
  FOwner := AOwner;
end;

destructor TFIBBase.Destroy;
begin
  SetDatabase(nil);
  SetTransaction(nil);
  inherited;
end;

procedure TFIBBase.CheckDatabase;
begin
  if (FDatabase = nil) then FIBError(feDatabaseNotAssigned, [nil]);
  FDatabase.CheckActive
end;

procedure TFIBBase.CheckTransaction;
begin
  if FTransaction = nil then FIBError(feTransactionNotAssigned, [nil]);
  FTransaction.CheckInTransaction;
end;

function TFIBBase.GetDBHandle: PISC_DB_HANDLE;
begin
  CheckDatabase;
  result := @FDatabase.Handle;
end;

function TFIBBase.GetTRHandle: PISC_TR_HANDLE;
begin
  CheckTransaction;
  result := @FTransaction.Handle;
end;

 procedure TFIBBase.FOnDatabaseConnecting;
 begin
   if Assigned(OnDatabaseConnecting) then
     OnDatabaseConnecting(Self);
 end;

 procedure TFIBBase.FOnDatabaseConnected;
 begin
   if Assigned(OnDatabaseConnected) then
     OnDatabaseConnected(Self);
 end;


procedure TFIBBase.FOnDatabaseDisconnecting;
begin
  if Assigned(OnDatabaseDisconnecting) then
    OnDatabaseDisconnecting(Self);
end;

procedure TFIBBase.FOnDatabaseDisconnected;
begin
  if Assigned(OnDatabaseDisconnected) then
    OnDatabaseDisconnected(Self);
end;

procedure TFIBBase.FOnDatabaseFree;
begin
  if Assigned(OnDatabaseFree) then
    OnDatabaseFree(Self);
  // Ensure that FDatabase is nil
  SetDatabase(nil);
  SetTransaction(nil);
end;


procedure TFIBBase.FOnTransactionStarting;
begin
  if Assigned(OnTransactionStarting) then
    OnTransactionStarting(Self);
end;

procedure TFIBBase.FOnTransactionStarted;
begin
  if Assigned(OnTransactionStarted) then
    OnTransactionStarted(Self);
end;

procedure TFIBBase.FOnTransactionEnding;
begin
  if Assigned(OnTransactionEnding) then
    OnTransactionEnding(Self);
end;

procedure TFIBBase.FOnTransactionEnded;
begin
  if Assigned(OnTransactionEnded) then
    OnTransactionEnded(Self);
end;

procedure TFIBBase.FOnTransactionFree;
begin
  if Assigned(OnTransactionFree) then
    OnTransactionFree(Self);
  // Ensure that FTransaction is nil
  FTransaction := nil;
end;

procedure TFIBBase.SetDatabase(Value: TFIBDatabase);
begin
  if (FDatabase <> nil) then
    FDatabase.RemoveDataSet(FIndexInDatabase);
  FDatabase := Value;
  if (FDatabase <> nil) then begin
    FIndexInDatabase := FDatabase.AddDataSet(Self);
    if (FTransaction = nil) then
      Transaction := FDatabase.DefaultTransaction;
  end;
end;

procedure TFIBBase.SetTransaction(Value: TFIBTransaction);
begin
  if (FTransaction <> nil) then
    FTransaction.RemoveDataSet(FIndexInTransaction);
  FTransaction := Value;
  if (FTransaction <> nil) then
    FIndexInTransaction := FTransaction.AddDataSet(Self);
end;

initialization
 DefDataBase :=nil;
 DatabaseList:=TList.Create;
finalization
 DatabaseList.Free
end.
