unit pFIBMigrator;

interface
uses Classes, DB, pFIBDataSet, FIBDataSet, pFIBQuery, FIBDatabase,
     pFIBDatabase, FIBQuery, Forms, SysUtils;

type

  PDataSetPair = ^TDataSetPair;
  TDataSetPair = record
    OldDataSet,
    NewDataSet: TDataSet;
  end;

  TDataSetList = class(TList)
  private
    function GetDataSetPairs(Index: Integer): PDataSetPair;
    procedure SetDataSetPairs(Index: Integer; Item: PDataSetPair);
    function GetOldDataSets(Index: Integer): TDataSet;
    procedure SetOldDataSets(Index: Integer; Item: TDataSet);
    function GetNewDataSets(Index: Integer): TDataSet;
    procedure SetNewDataSets(Index: Integer; Item: TDataSet);
  public
    function GetPaired(aDataSet: TDataSet): TDataSet;
    function IndexOfOldDataSet(aDataSet: TDataSet): Integer;
    procedure DeletePair(Index: Integer);
    procedure ClearAll;
    destructor Destroy; override;
    property Pairs[Index: Integer]: PDataSetPair read GetDataSetPairs write SetDataSetPairs;
    property OldDataSets[Index: Integer]: TDataSet read GetOldDataSets write SetOldDataSets;
    property NewDataSets[Index: Integer]: TDataSet read GetNewDataSets write SetNewDataSets;
  end;

  TCustomFIBPlusMigrator = class(TComponent)
  private
    FFIBDatabase: TFIBDatabase;
    FDataSets: TDataSetList;
    FFields: TList;
    function GetDatabase: TFIBDatabase;
    procedure SetFIBDatabase(Value: TFIBDatabase);
  protected

    { Common methods }
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure MigrateDataSets;
    procedure MigrateQueries;
    procedure MigrateDataSources;
    procedure UpdateDesignInfo(MustDie, NewComponent: TComponent);
    procedure AfterCreateFIBComponent(MustDie: TComponent; FIBCmp: TComponent); virtual;
    procedure AssignEvents(aDataSet: TpFIBDataSet; MustDie: TDataSet);
    procedure AssignFields(OldField, NewField: TField);
    procedure NewDataSetPair(aDataSet: TpFIBDataSet; MustDie: TDataSet);
    procedure MoveFields(aDataSet: TpFIBDataSet; MustDie: TDataSet);
    procedure AssignLookupFields(MustDie: TDataSet);
    procedure CreateFIBDataSet(aDataSet: TpFIBDataSet; MustDie: TDataSet); virtual;

    { Methods for ansectors }
    procedure MigrateTransactions; virtual; abstract;
    function ReplaceField(aDataSet: TpFIBDataSet; aField: TField): boolean; virtual; abstract;
    procedure CheckDataSet(MustDie: TDataSet); virtual; abstract;
    procedure CheckQuery(aComponent: TComponent); virtual; abstract;
    function CheckForFree(aComponent: TComponent): boolean; virtual; abstract;
    function IsDataSetNeeded(MustDie: TDataSet): boolean; virtual; abstract;

    procedure SetStoredProcedure(vQuery: TpFIBQuery; MustDie: TDataSet); virtual; abstract;
    function GetNeededTransaction(MustDie: TDataSet): TpFIBTransaction; virtual; abstract;
    procedure GetNeededSQLs(aDataSet: TpFIBDataSet; MustDie: TDataSet); virtual; abstract;
    procedure GetCachedUpdates(aDataSet: TpFIBDataSet; MustDie: TDataSet); virtual; abstract;
    procedure GetNeededUpdateObject(aDataSet: TpFIBDataSet; MustDie: TDataSet); virtual; abstract;
  public
    procedure Migrate; virtual;
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
  published
    property FIBDatabase: TFIBDatabase read GetDatabase write SetFIBDatabase;
  end;

procedure Register;

implementation
uses DsgnIntf;

{$R MIGRATE2FIBPLUS.DCR}
type

  TFIBPlusMigratorEditor = class(TComponentEditor)
  public
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): string; override;
    function GetVerbCount: Integer; override;
  end;

  function TFIBPlusMigratorEditor.GetVerbCount: Integer;
  begin
    Result := 1;
  end;

  function TFIBPlusMigratorEditor.GetVerb(Index: Integer): string;
  begin
    Result := 'Migrate to FIBPlus';
  end;

  procedure TFIBPlusMigratorEditor.ExecuteVerb(Index: Integer);
  begin
    TCustomFIBPlusMigrator(Component).Migrate;
  end;

procedure Register;
begin
  RegisterComponentEditor(TCustomFIBPlusMigrator, TFIBPlusMigratorEditor);
end;

{ TDataSetList }
function TDataSetList.GetDataSetPairs(Index: Integer): PDataSetPair;
begin
  Result := PDataSetPair(Items[Index]);
end;

function TDataSetList.GetNewDataSets(Index: Integer): TDataSet;
begin
  Result := Pairs[Index].NewDataSet;
end;

function TDataSetList.GetOldDataSets(Index: Integer): TDataSet;
begin
  Result := Pairs[Index].OldDataSet;
end;

procedure TDataSetList.SetDataSetPairs(Index: Integer; Item: PDataSetPair);
begin
  Items[Index] := Item;
end;

procedure TDataSetList.SetNewDataSets(Index: Integer; Item: TDataSet);
begin
  Pairs[Index].NewDataSet := Item;
end;

procedure TDataSetList.SetOldDataSets(Index: Integer; Item: TDataSet);
begin
  Pairs[Index].OldDataSet := Item;
end;

function TDataSetList.GetPaired(aDataSet: TDataSet): TDataSet;
var Index: Integer;
begin
  Result := nil;
  for Index := 0 to pred(Count) do
    if OldDataSets[Index] = aDataSet then begin
      Result := NewDataSets[Index];
      exit;
    end;
end;

function TDataSetList.IndexOfOldDataSet(aDataSet: TDataSet): Integer;
var Index: Integer;
begin
  Result := -1;
  for Index := 0 to pred(Count) do
    if OldDataSets[Index] = aDataSet then begin
      Result := Index;
      exit;
    end;
end;

procedure TDataSetList.ClearAll;
var Index: Integer;
begin
  for Index := pred(Count) downto 0 do DeletePair(Index);
end;

procedure TDataSetList.DeletePair(Index: Integer);
begin
  FreeMem(PDataSetPair(Items[Index]));
  Delete(Index);
end;

destructor TDataSetList.Destroy;
begin
  ClearAll;
  inherited;
end;

{ TCustomFIBPlusMigrator }
function TCustomFIBPlusMigrator.GetDatabase: TFIBDatabase;
begin
  Result := FFIBDatabase;
end;

procedure TCustomFIBPlusMigrator.SetFIBDatabase(Value: TFIBDatabase);
begin
  FFIBDatabase := Value;
  if Value <> nil then Value.FreeNotification(Self)
end;


procedure TCustomFIBPlusMigrator.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (aComponent = FFIBDatabase) then
     FFIBDatabase := nil;
end;

procedure TCustomFIBPlusMigrator.AssignEvents(aDataSet: TpFIBDataSet;
  MustDie: TDataSet);
begin
  with MustDie do begin
    aDataSet.BeforeOpen     := BeforeOpen;
    aDataSet.AfterOpen      := AfterOpen ;
    aDataSet.BeforeClose    := BeforeClose;
    aDataSet.AfterClose     := AfterClose;
    aDataSet.BeforeInsert   := BeforeInsert;
    aDataSet.AfterInsert    := AfterInsert;
    aDataSet.BeforeEdit     := BeforeEdit;
    aDataSet.AfterEdit      := AfterEdit;
    aDataSet.BeforePost     := BeforePost;
    aDataSet.AfterPost      := AfterPost;
    aDataSet.BeforeCancel   := BeforeCancel;
    aDataSet.AfterCancel    := AfterCancel;
    aDataSet.BeforeDelete   := BeforeDelete;
    aDataSet.AfterDelete    := AfterDelete;
    aDataSet.BeforeScroll   := BeforeScroll;
    aDataSet.AfterScroll    := AfterScroll;
    aDataSet.OnCalcFields   := OnCalcFields;
    aDataSet.OnDeleteError  := OnDeleteError;
    aDataSet.OnEditError    := OnEditError;
    aDataSet.OnFilterRecord := OnFilterRecord;
    aDataSet.OnNewRecord    := OnNewRecord;
    aDataSet.OnPostError    := OnPostError;
 // Clear
    BeforeOpen     := nil;
    AfterOpen      := nil;
    BeforeClose    := nil;
    AfterClose     := nil;
    BeforeInsert   := nil;
    AfterInsert    := nil;
    BeforeEdit     := nil;
    AfterEdit      := nil;
    BeforePost     := nil;
    AfterPost      := nil;
    BeforeCancel   := nil;
    AfterCancel    := nil;
    BeforeDelete   := nil;
    AfterDelete    := nil;
    BeforeScroll   := nil;
    AfterScroll    := nil;
    OnCalcFields   := nil;
    OnDeleteError  := nil;
    OnEditError    := nil;
    OnFilterRecord := nil;
    OnNewRecord    := nil;
    OnPostError    := nil;
  end;
end;

procedure TCustomFIBPlusMigrator.AssignFields(OldField, NewField: TField);
var s: string;
begin
  with OldField do begin
    // Common properties
    NewField.FieldName    := FieldName;
    NewField.DisplayLabel := DisplayLabel;
    NewField.FieldKind    := FieldKind;
    NewField.EditMask     := EditMask;
    NewField.Alignment    := Alignment;
    NewField.DefaultExpression := DefaultExpression;
    NewField.DisplayWidth := DisplayWidth;
    NewField.Visible      := Visible;

    // Lookup properties
    NewField.KeyFields := KeyFields;
    NewField.LookupCache := LookupCache;
    NewField.LookupDataSet := FDataSets.GetPaired(LookupDataSet);
    NewField.LookupKeyFields := LookupKeyFields;
    NewField.LookupResultField := LookupResultField;

    // Events
    NewField.OnChange     := OnChange;
    NewField.OnGetText    := OnGetText;
    NewField.OnSetText    := OnSetText;
    NewField.OnValidate   := OnValidate;

    s := Name;
    Name := 'Die_' + Name;
    NewField.Name := s;
  end
end;

procedure TCustomFIBPlusMigrator.MoveFields(aDataSet: TpFIBDataSet;
  MustDie: TDataSet);
begin
  if  MustDie.DefaultFields then Exit;

  with MustDie do
    while FieldCount > 0 do
      if not ReplaceField(aDataSet, Fields[0]) then
         Fields[0].DataSet := aDataSet;
end;

procedure  TCustomFIBPlusMigrator.AfterCreateFIBComponent(MustDie: TComponent;
  FIBCmp: TComponent);
begin
 // for override
end;

procedure  TCustomFIBPlusMigrator.UpdateDesignInfo(MustDie, NewComponent: TComponent);
begin
  NewComponent.DesignInfo := MustDie.DesignInfo;
  MustDie.Owner.RemoveComponent(NewComponent);
  MustDie.Owner.InsertComponent(NewComponent);
end;

procedure  TCustomFIBPlusMigrator.CreateFIBDataSet(aDataSet: TpFIBDataSet; MustDie: TDataSet);
var Index, NamePos: integer;
    rName1, rName2: string;
    vQuery: TpFIBQuery;
    vFIBTransaction: TpFIBTransaction;

begin
  vFIBTransaction := GetNeededTransaction(MustDie);
  MustDie.Close;
  if IsDataSetNeeded(MustDie) then begin
    UpdateDesignInfo(MustDie, aDataSet);
    aDataSet.DataBase := FFIBDatabase;

    if vFIBTransaction <> nil then
      aDataSet.Transaction := vFIBTransaction;

    GetNeededSQLs(aDataSet, MustDie);

    AssignEvents(aDataSet, MustDie);
    MoveFields  (aDataSet, MustDie);

    aDataSet.DataSource := MustDie.DataSource;

    GetCachedUpdates(aDataSet, MustDie);
    MustDie.Name := 'Die_' + MustDie.Name;
    aDataSet.Name := Copy(MustDie.Name, 5, length(MustDie.Name));

    for Index := 0 to pred(aDataSet.FieldCount) do begin
      rName1 := aDataSet.Fields[Index].Name;
      NamePos := Pos(aDataSet.Name, rName1);
      if NamePos = 1 then begin
        rName2 := Copy(rName1, length(aDataSet.Name) + 1, length(rName1));
        NamePos := Pos(aDataSet.Name, rName2);
        if NamePos = 1 then aDataSet.Fields[Index].Name := rName2;
      end;
    end;

    GetNeededUpdateObject(aDataSet, MustDie);
    AfterCreateFIBComponent(MustDie, aDataSet);
  end // IsDataset End
  else begin
    aDataSet.Free;
    Index := FDataSets.IndexOfOldDataSet(MustDie);
    if Index <> -1 then
      FDataSets.Pairs[Index].NewDataSet := nil;

    MustDie.Close;
    vQuery := TpFIBQuery.Create(Owner);
    UpdateDesignInfo(MustDie, vQuery);
    vQuery.DataBase := FFIBDatabase ;

    if vFIBTransaction <> nil then
      vQuery.Transaction := vFIBTransaction;

    // MustDie.Name := 'Die_' + Name;
    vQuery.Name := Copy(MustDie.Name, 5, length(vQuery.Name));
    SetStoredProcedure(vQuery, MustDie);

    AfterCreateFIBComponent(MustDie, vQuery);
  end;
end;

procedure  TCustomFIBPlusMigrator.MigrateDataSources;
var FormIndex, CompIndex: Integer;
    DataSource: TDataSource;
    DataSet: TDataSet;
begin
  for FormIndex := 0 to Pred(Screen.FormCount) do
    with Screen.Forms[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
        if (Components[CompIndex] is TDataSource) then begin
          DataSource := (Components[CompIndex] as TDataSource);
          DataSet := nil;
          if FDataSets.IndexOfOldDataSet(DataSource.DataSet) <> -1 then
            DataSet := FDataSets.GetPaired(DataSource.DataSet);
          if DataSet <> nil then
            DataSource.DataSet := DataSet;
        end;
    end;

  for FormIndex := 0 to Pred(Screen.DataModuleCount) do
    with Screen.DataModules[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
        if (Components[CompIndex] is TDataSource) then begin
          DataSource := (Components[CompIndex] as TDataSource);
          DataSet := nil;
          if FDataSets.IndexOfOldDataSet(DataSource.DataSet) <> -1 then
            DataSet := FDataSets.GetPaired(DataSource.DataSet);
          if DataSet <> nil then
            DataSource.DataSet := DataSet;
        end;
    end;
end;

procedure  TCustomFIBPlusMigrator.Migrate;
var FormIndex, CompIndex: Integer;
begin
  if not Assigned(FIBDatabase) then
    raise Exception.Create('Database not assigned');

  MigrateTransactions;
  MigrateDataSets;
  MigrateDataSources;

  for FormIndex := 0 to Pred(Screen.DataModuleCount) do
    with Screen.DataModules[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      CompIndex := 0;
      while CompIndex < ComponentCount do
        if not CheckForFree(Components[CompIndex]) then Inc(CompIndex)
        else begin
           if (Components[CompIndex] is TDataSet) then
              AssignLookupFields(Components[CompIndex] as TDataSet);
           Components[CompIndex].Free;
        end;
    end;

  for FormIndex := 0 to Pred(Screen.FormCount) do
    with Screen.Forms[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      CompIndex := 0;
      while CompIndex < ComponentCount do
        if not CheckForFree(Components[CompIndex]) then Inc(CompIndex)
        else begin
           if (Components[CompIndex] is TDataSet) then
              AssignLookupFields(Components[CompIndex] as TDataSet);
           Components[CompIndex].Free;
        end;
    end;
end;

procedure TCustomFIBPlusMigrator.MigrateDataSets;
var FormIndex, CompIndex: Integer;
begin
  {   .   , 
       .  ,
      
  }
  for FormIndex := 0 to Pred(Screen.FormCount) do
    with Screen.Forms[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
       if (CompIndex <= pred(ComponentCount)) and
          (Components[CompIndex] is TDataSet) then begin
         CheckDataSet(Components[CompIndex] as TDataSet);
       end;
    end;

  for FormIndex := 0 to Pred(Screen.DataModuleCount) do
    with Screen.DataModules[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
       if (CompIndex <= pred(ComponentCount)) and
          (Components[CompIndex] is TDataSet) then
         CheckDataSet(Components[CompIndex] as TDataSet);
    end;

  for CompIndex := 0 to pred(FDataSets.Count) do
    CreateFIBDataSet(TpFIBDataSet(FDataSets.NewDataSets[CompIndex]),
      FDataSets.OldDataSets[CompIndex]);
end;

procedure TCustomFIBPlusMigrator.MigrateQueries;
var FormIndex, CompIndex: Integer;
begin
  for FormIndex := 0 to Pred(Screen.FormCount) do
    with Screen.Forms[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
        if (CompIndex <= pred(ComponentCount)) then
          CheckQuery(Components[CompIndex]);
    end;

  for FormIndex := 0 to Pred(Screen.DataModuleCount) do
    with Screen.DataModules[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
        if (CompIndex <= pred(ComponentCount)) then
          CheckQuery(Components[CompIndex]);
    end;
end;

constructor TCustomFIBPlusMigrator.Create(aOwner: TComponent);
begin
  inherited;
  FDataSets := TDataSetList.Create;
  FFields := TList.Create;
end;

destructor TCustomFIBPlusMigrator.Destroy;
begin
  FDataSets.Free;
  FFields.Free;
  inherited;
end;

procedure TCustomFIBPlusMigrator.NewDataSetPair(aDataSet: TpFIBDataSet;
  MustDie: TDataSet);
var Pair: PDataSetPair;
begin
  New(Pair);
  Pair.OldDataSet := MustDie;
  Pair.NewDataSet := aDataSet;
  FDataSets.Add(Pair);
end;

procedure TCustomFIBPlusMigrator.AssignLookupFields(MustDie: TDataSet);
var FormIndex, CompIndex: Integer;
    LinkField: TField;
begin
  for FormIndex := 0 to Pred(Screen.FormCount) do
    with Screen.Forms[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
        if (Components[CompIndex] is TField) then begin
          LinkField := (Components[CompIndex] as TField);
          if (LinkField.LookupDataSet = MustDie) then
             LinkField.LookupDataSet := FDataSets.GetPaired(MustDie);
        end;
    end;

  for FormIndex := 0 to Pred(Screen.DataModuleCount) do
    with Screen.DataModules[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for CompIndex := 0 to Pred(ComponentCount) do
        if (Components[CompIndex] is TField) then begin
          LinkField := (Components[CompIndex] as TField);
          if (LinkField.LookupDataSet = MustDie) then
             LinkField.LookupDataSet := FDataSets.GetPaired(MustDie);
        end;
    end;
end;

end.
