unit IBX2FIBPlus;

interface
uses Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
     Dialogs, DB, pFIBDataSet, FIBDataSet, pFIBQuery, IBTable,
     IBCustomDataSet, IBUpdateSQL, IBStoredProc, pFIBDatabase,
     IBDatabase, FIBDatabase, IBSQL, FIBQuery, pFIBMigrator, FIB,
     IBQuery;

type

  TIBX2FIBPlus = class(TCustomFIBPlusMigrator)
  private
    {FMasterFieldsList: TStringList;
    FDetailFieldsList: TStringList;}
  protected
    { Common methods }
    procedure MigrateTransactions; override;

    { Added for IBX }
    procedure CreateFIBTrans(MustDie: TIBTransaction);
    procedure CreateFIBQuery(MustDie: TIBSQL);
    {procedure ExtractDetailFields(MustDie: TIBTable);
    procedure GenerateIBTableSQL(MustDie: TIBTable; aDataSet: TpFIBDataSet);}

    { Overrided methods }
    function ReplaceField(aDataSet: TpFIBDataSet; aField: TField): boolean; override;
    procedure CheckDataSet(MustDie: TDataSet); override;
    procedure CheckQuery(aComponent: TComponent); override;
    function CheckForFree(aComponent: TComponent): boolean; override;
    function IsDataSetNeeded(MustDie: TDataSet): boolean; override;

    procedure SetStoredProcedure(vQuery: TpFIBQuery; MustDie: TDataSet); override;
    function GetNeededTransaction(MustDie: TDataSet): TpFIBTransaction; override;
    procedure GetNeededSQLs(aDataSet: TpFIBDataSet; MustDie: TDataSet); override;
    procedure GetCachedUpdates(aDataSet: TpFIBDataSet; MustDie: TDataSet); override;
    procedure GetNeededUpdateObject(aDataSet: TpFIBDataSet; MustDie: TDataSet); override;
  public
    {constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;}
  end;

procedure Register;

implementation

type
  THackIBDataSet = class(TIBCustomDataSet);

procedure Register;
begin
  RegisterComponents('FIBPlus', [TIBX2FIBPlus]);
end;

{ TIbx2FIBPlus }
function TIBX2FIBPlus.ReplaceField(aDataSet: TpFIBDataSet; aField: TField): boolean;
// var NewField: TField;
begin
  Result := false;
  {
  Result := (aField is TIBStringField) or
            (aField is TIBBCDField);
  if not Result then exit;

  NewField := nil;
  
  if aField is TIBStringField then
    NewField := TFIBStringField.Create(aDataSet.Owner);

  if aField is TIBBCDField then
    NewField := TFIBBCDField.Create(aDataSet.Owner);

  NewField.FieldName := aField.FieldName;
  NewField.DataSet := aDataSet;
  AssignFields(aField, NewField);
  aField.Free;
  }
end;

procedure  TIbx2FIBPlus.CreateFibTrans(MustDie: TIBTransaction);
var FIBTr: TpFIBTransaction;
    rName: string;
begin
  with MustDie do begin
    FIBTr := TpFIBTransaction.Create(Owner);
    FIBTr.DesignInfo := DesignInfo;
    MustDie.Owner.RemoveComponent(FIBTr);
    MustDie.Owner.InsertComponent(FIBTr);

    FIBTr.TPBMode := tpbDefault;
    FIBTr.TRParams := Params;
    FIBTr.DefaultDatabase := FIBDatabase;
    FIBTr.Active := Active;
    rName := Name;
    Name := 'Die_' + Name;
    FIBTr.Name := rName;
    FIBTr.OnTimeout := OnIdleTimer;
    case DefaultAction of
     IBDataBase.taCommit            : FIBTr.TimeOutAction := taCommit;
     IBDataBase.taCommitRetaining   : FIBTr.TimeOutAction := taCommitRetaining;
     IBDataBase.taRollBack          : FIBTr.TimeOutAction := taRollBack;
     IBDataBase.taRollBackRetaining : FIBTr.TimeOutAction := taRollBackRetaining;
    end;
  end;
end;

procedure TIBX2FIBPlus.CreateFIBQuery(MustDie: TIBSQL);
var vQuery: TpFIBQuery;
    vFIBTransaction: TComponent;
begin
  with MustDie do begin
    if Transaction <> nil then
      vFIBTransaction := Transaction.Owner.FindComponent(
         Copy(Transaction.Name, 5, length(Transaction.Name)))
    else
      vFIBTransaction := nil;

    MustDie.Close;
    vQuery := TpFIBQuery.Create(Owner);
    vQuery.DesignInfo := DesignInfo;
    MustDie.Owner.RemoveComponent(vQuery);
    MustDie.Owner.InsertComponent(vQuery);
    vQuery.DataBase := FIBDatabase;
    if vFIBTransaction <> nil then
     vQuery.Transaction := TFIBTransaction(vFIBTransaction);

    MustDie.Name := 'Die_' + Name;
    vQuery.Name := Copy(MustDie.Name, 5, length(vQuery.Name));
    with MustDie do begin
      vQuery.GoToFirstRecordOnExecute := GoToFirstRecordOnExecute;
      vQuery.ParamCheck := ParamCheck;
      vQuery.SQL.Assign(SQL);
      vQuery.OnSQLChanging := OnSQLChanging;
      OnSQLChanging := nil;
    end;
    AfterCreateFIBComponent(MustDie, vQuery);
  end;
end;

function TIbx2FIBPlus.GetNeededTransaction(MustDie: TDataSet): TpFIBTransaction;
begin
  Result := nil;
  with (MustDie as TIBCustomDataSet) do
    if Transaction <> nil then
      Result := TpFIBTransaction(Transaction.Owner.FindComponent(
         Copy(Transaction.Name, 5, length(Transaction.Name))));
end;

procedure TIbx2FIBPlus.GetNeededSQLs(aDataSet: TpFIBDataSet; MustDie: TDataSet);
begin

  if MustDie is TIBDataSet then begin
    aDataSet.SelectSQL  := TIBDataSet(MustDie).SelectSQL;
    aDataSet.InsertSQL  := TIBDataSet(MustDie).InsertSQL;
    aDataSet.UpdateSQL  := TIBDataSet(MustDie).ModifySQL;
    aDataSet.DeleteSQL  := TIBDataSet(MustDie).DeleteSQL;
    aDataSet.RefreshSQL := TIBDataSet(MustDie).RefreshSQL;
  end;

  if MustDie is TIBTable then begin
    aDataSet.SelectSQL.Text := 'select * from ' +
       TIBTable(MustDie).TableName;
    with aDataSet.AutoUpdateOptions do begin
      AutoReWriteSqls := True;
      UpdateTableName := TIBTable(MustDie).TableName;
      KeyFields := TIBTable(MustDie).IndexFieldNames;
    end;
  end;

  if MustDie is TIBQuery then
    aDataSet.SelectSQL := TIBQuery(MustDie).SQL;
end;

procedure TIbx2FIBPlus.GetCachedUpdates(aDataSet: TpFIBDataSet; MustDie: TDataSet);
begin
  aDataSet.CachedUpdates := THackIBDataSet(MustDie).CachedUpdates;
end;

procedure  TIbx2FIBPlus.GetNeededUpdateObject(aDataSet: TpFIBDataSet; MustDie: TDataSet);
begin
  with (MustDie as TIBCustomDataSet) do
    if Assigned(UpdateObject) and (UpdateObject is TIBUpdateSQL) then
      with (UpdateObject as TIBUpdateSQL) do begin
        aDataSet.UpdateSQL := ModifySQL;
        aDataSet.InsertSQL := InsertSQL;
        aDataSet.DeleteSQL := DeleteSQL;
      end;
end;

procedure  TIbx2FIBPlus.SetStoredProcedure(vQuery: TpFIBQuery; MustDie: TDataSet);
var SpPars: string;
    Index: Integer;
begin
  with (MustDie as TIBStoredProc) do begin
    vQuery.SQL.Text := 'EXECUTE PROCEDURE ' + StoredProcName;
    if ParamCount = 0 then Prepare;
    if ParamCount > 0 then begin
      SpPars := '';
      for Index := 0 to Pred(ParamCount) do
        if Params[Index].ParamType = ptInput then
           SpPars := SpPars + '?' + Params[Index].Name + ',';
      if SpPars <> '' then SpPars := '(' +
          Copy(SpPars, 1, Length(SpPars) - 1) + ')';
      vQuery.SQL.Text := vQuery.SQL.Text + SpPars;
    end;
    MustDie.Name := 'Die_' + MustDie.Name;
    vQuery.Name := Copy(MustDie.Name, 5, length(MustDie.Name));
  end;
end;

function TIbx2FIBPlus.CheckForFree(aComponent: TComponent): boolean;
begin
  Result := (aComponent is TIBCustomDataSet) or
            (aComponent is TIBUpdateSQL) or
            (aComponent is TIBTransaction) or
            (aComponent is TIBSQL);
end;

procedure TIBX2FIBPlus.CheckQuery(aComponent: TComponent);
begin
  if (aComponent is TIBSQL) then CreateFIBQuery(aComponent as TIBSQL);
end;

procedure TIBX2FIBPlus.MigrateTransactions;
var FormIndex, Index: Integer;
begin
  for FormIndex := 0 to Pred(Screen.FormCount) do
    with Screen.Forms[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for Index := 0 to Pred(ComponentCount) do
        if Components[Index] is TIBTransaction then
          CreateFibTrans(Components[Index] as TIBTransaction);
    end;

  for FormIndex := 0 to Pred(Screen.DataModuleCount) do
    with Screen.DataModules[FormIndex] do begin
      if not (csDesigning in ComponentState) then Continue;
      for Index := 0 to Pred(ComponentCount)  do
        if Components[Index] is TIBTransaction then
          CreateFibTrans(Components[Index] as TIBTransaction);
    end;
end;

procedure TIBX2FIBPlus.CheckDataSet(MustDie: TDataSet);
var aDataSet: TpFIBDataSet;
begin
  if (MustDie is TIBCustomDataSet) then begin
    // Add dataset pair to list
    aDataSet := TpFIBDataSet.Create(MustDie.Owner);
    NewDataSetPair(aDataSet, MustDie);
  end;
end;

function TIBX2FIBPlus.IsDataSetNeeded(MustDie: TDataSet): boolean;
begin
  Result := not (MustDie is TIBStoredProc);
end;

{
procedure TIBX2FIBPlus.ExtractDetailFields(MustDie: TIBTable);
var I: Integer;
    DetailFieldNames: String;
begin
  FMasterFieldsList.Clear;
  FDetailFieldsList.Clear;
  i := 1;
  while i <= Length(MustDie.MasterFields) do
    FMasterFieldsList.Add(ExtractFieldName(MustDie.MasterFields, i));
  i := 1;
  if MustDie.IndexFieldNames = '' then
    MustDie.GetIndexNames(FDetailFieldsList)
  else begin
    DetailFieldNames := MustDie.IndexFieldNames;
    while i <= Length(DetailFieldNames) do
      FDetailFieldsList.Add(ExtractFieldName(DetailFieldNames, i));
  end;
end;

procedure TIBX2FIBPlus.GenerateIBTableSQL(MustDie: TIBTable; aDataSet: TpFIBDataSet);
var I: Integer;
begin
  if (MustDie.MasterSource <> nil) and
     (MustDie.MasterSource.DataSet <> nil) and
     (MustDie.MasterFields <> '') then begin
     aDataSet.SelectSQL.Text := aDataSet.SelectSQL.Text + ' WHERE ';
    ExtractDetailFields(MustDie);
    if FDetailFieldsList.Count < FMasterFieldsList.Count then
      FIBError(feUnknownError, [nil]);
    for I := 0 to pred(FMasterFieldsList.Count) do begin
      if I > 0 then
        aDataSet.SelectSQL.Text := aDataSet.SelectSQL.Text + 'AND ';
      aDataSet.SelectSQL.Text := aDataSet.SelectSQL.Text +
        '''' + FDetailFieldsList.Strings[I] + '''' +
        ' = :' +
        '''' + FMasterFieldsList.Strings[I] + '''';
    end;
  end;
end;

constructor TIBX2FIBPlus.Create(aOwner: TComponent);
begin
  inherited;
  FMasterFieldsList := TStringList.Create;
  FDetailFieldsList := TStringList.Create;
end;

destructor TIBX2FIBPlus.Destroy;
begin
  FMasterFieldsList.Free;
  FDetailFieldsList.Free;
  inherited;
end;
}

end.


