{***************************************************************
 *
 * Unit Name : ExportData
 * Purpose   : Export Data Component
 * Author    : Eugen Mihailescu
 * Company   : EM Quicksoft Romania SRL
 * Copyright : 1998,2002  All rights reserved.
 * Web Site  : http://www.em-quicksoft.com
 * Support   : support@em-quicksoft.com
 * History   : 03/19/2000
 *
 * Disclaimer: The objectives of this program is only educational.
 *             You MUST NOT use it without getting any material benefit on it
 *             You have the right to share this code with others without removing
 *             the copyright and/or disclaimer notice.
 *             You are allowed to include its code in your application
 *             only if it was designed for non commercial purpose,
 *             otherwise we claim $10 for each copy of your
 *             application which contains this code inside.
 ****************************************************************}

Unit ExportData;

Interface

Uses WinProcs,
  Shellapi,
  fProgress,
  DsgnIntf,
  DDEMan,
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Forms,
  Dialogs,
  Db,
  DBTables;

Resourcestring
  FILTER_TYPES      = 'Comma delimited file (*.csv)|*.csv|Text file (*.txt)|*.txt|Any file (*.*)|*.*';
  DEFAULT_FILTER    = 'csv';
  DEF_CAPTION       = 'Export data';

Type
  TDBExportOption = (eoAutoOpen, eoCanCancel, eoShowProgress);
  TDBExportOptions = Set Of TDBExportOption;
  TDBExportHeader = (ehNone, ehFieldName, ehDisplayName);
Const
  DefaultOptions    = [eoAutoOpen, eoShowProgress];

Type
  TExcelData = Class(TComponent)
  private
    FMacro: String;
    FMacroPath: String;
    FDDE: TDdeClientConv;
    FConnected: Boolean;
    FExeName: String;
    FDecimals: Integer;
    FOnClose: TNotifyEvent;
    FOnOpen: TNotifyEvent;
    FBatch: Boolean;
    FMin: Integer;
    FMax: Integer;
    FFirstRow: Integer;
    FFirstCol: Integer;
    FLastCol: Integer;
    FLines: TStrings;                   { using TStringList }
    FCells: TStrings;                   { using TStringList }
    FLastTime: TDateTime;
    FCounter: Integer;
    FLimit: Integer;
    Procedure SetExeName(Const Value: String);
    Procedure SetConnect(Const Value: Boolean);
    Procedure SetMin(Const Value: Integer);
    Procedure SetMax(Const Value: Integer);
    Function GetSelection: String;
    Function GetReady: Boolean;
  protected
    Procedure DoRect(Top, Left, Bottom, Right: Integer;
      Data: TStrings; Request: Boolean);
    Procedure CheckConnection; virtual;
    Procedure LinkSystem;
    Procedure OpenLink(Sender: TObject);
    Procedure ShutDown(Sender: TObject);
    Procedure LocateExcel; virtual;
    Procedure CheckLimit; virtual;
  public
    Constructor Create(AOwner: TComponent); override;
    Destructor Destroy; override;
    Procedure Connect;
    Procedure Disconnect;
    Procedure Wait;
    Procedure ProcessMessages; virtual;
    Function Request(Const Item: String): String;
    Procedure Exec(Const Cmd: String);
    Procedure Run(Const Mn: String);
    Procedure Select(Row, col: Integer);
    Procedure PutStr(Row, col: Integer; Const S: String);
    Procedure PutExt(Row, col: Integer; E: Extended); virtual;
    Procedure PutInt(Row, col: Integer; I: Longint); virtual;
    Procedure PutDay(Row, col: Integer; D: TDateTime); virtual;
    Procedure BatchStart(FirstRow, FirstCol: Integer);
    Procedure BatchCancel;
    Procedure BatchSend;
    Procedure GetBooks(Books: TStrings);
    Procedure GetSheets(Const Book: String; Sheets: TStrings);
    Procedure GetRange(R: TRect; Lines: TStrings);
    Function GetCell(Row, col: Integer): String;
    Procedure OpenMacroFile(Const Fn: String; hide: Boolean);
    Procedure CloseMacroFile;
    Property DDE: TDdeCLientConv read FDDE;
    Property Connected: Boolean read FConnected write SetConnect;
    Property Ready: Boolean read GetReady;
    Property Selection: String read GetSelection;
    Property Lines: TStrings read FLines;
    Property FirstRow: Integer read FFirstRow;
    Property FirstCol: Integer read FFirstCol;
    Property LastCol: Integer read FLastCol write FLastCol;
    Property BatchOn: Boolean read FBatch;
  published
    Property ExeName: String read FExeName write SetExeName;
    Property ExecLimit: Integer read FLimit write FLimit;
    Property Decimals: Integer read FDecimals write FDecimals;
    Property BatchMin: Integer read FMin write SetMin;
    Property BatchMax: Integer read FMax write SetMax;
    Property OnClose: TNotifyEvent read FOnClose write FOnClose;
    Property OnOpen: TNotifyEvent read FOnOpen write FOnOpen;
  End;

  TDBExport = Class(TComponent)
  private
    FDataSource: TDataSource;
    FFieldDelimiter: char;
    FFileName: TFileName;
    FCaption: TCaption;
    FHeader: TDBExportHeader;
    FOptions: TDBExportOptions;
    FOnExecute: TNotifyEvent;
    FOnInitialize: TNotifyEvent;
    FExecuting: Boolean;

  protected
    Procedure SetDataSource(Value: TDataSource);
    Procedure Notification(AComponent: TComponent; Operation: TOperation); override;

    Function StoredCaption: Boolean;
    Function GetFileName: String;
    Function GetHeader: String;
    Function GetRecord: String;

    Function FilteredData(Const Data: String): String;
    Function GetSeparator: char;

  public
    Constructor Create(AOwner: TComponent); override;

    Function Execute: Boolean;
    Property FieldDelimiter: char read FFieldDelimiter;

  published
    Property DataSource: TDataSource read FDataSource write SetDataSource;
    Property FileName: TFileName read FFileName write FFileName;
    Property Caption: TCaption read FCaption write FCaption stored StoredCaption;
    Property Header: TDBExportHeader read FHeader write FHeader default ehNone;
    Property Options: TDBExportOptions read FOptions write FOptions default DefaultOptions;

    Property OnExecute: TNotifyEvent read FOnExecute write FOnExecute;
    Property OnInitialize: TNotifyEvent read FOnInitialize write FOnInitialize;
  End;

  TExportDataEditor = Class(TComponentEditor)
    Function GetVerbCount: Integer; override;
    Function GetVerb(Index: Integer): String; override;
    Procedure ExecuteVerb(Index: Integer); override;
  End;

  TDEFileNameProperty = Class(TStringProperty)
  public
    Function GetAttributes: TPropertyAttributes; override;
    Procedure Edit; override;
  End;

Type
  TScopeType = (stEntireScope, stVisibleRecords);
  TLanguage = (lgRomanian, lgEnglish, lgCustom);

Type
  TExportDataFormat = (dfASCII, dfCSV, dfDBase, dfParadox, dfFoxPro, dfExcel, dfClipboard, dfNone);
  TExportFile = Record
    FormatType: TExportDataFormat;
    FileName: String;
    Selection: Integer;
    Pages: Integer;
    txtFormat: Integer;
  End;

  TExportDataFile = Class(TComponent)
  private
    FDataSet: TDataSet;
    fTable: TTable;
    fExcel: TExcelData;
    fCSV: TDBExport;
    fDataFormat: TExportDataFormat;
    fScope: TScopeType;
    FLanguage: TLanguage;
    fFileName: TFileName;
    fAbortOnError: Boolean;
  protected
    Procedure SetDataSet(Value: TDataSet);
    Procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    Procedure ExportTableAs(ExportType: TExportFile);
    Procedure ExportDB(ExportType: TExportFile);
    Procedure ExportXLS(ExportType: TExportFile);
    Procedure CopyDB(Source: TDataSet; Destination: TTable);
  public
    Constructor Create(Owner: TComponent); override;
    Destructor Destroy; override;
    Function Execute: Boolean;
  published
    Property DataSet: TDataSet read FDataSet write SetDataSet;
    Property ExportFormat: TExportDataFormat read fDataFormat write fDataFormat default dfASCII;
    Property Scope: TScopeType read fScope write fScope default stVisibleRecords;
    Property Language: TLanguage read FLanguage write FLanguage default lgRomanian;
    Property ExportAs: TFileName read fFileName write fFileName;
    Property AbortOnError: Boolean read fAbortOnError write fAbortOnError;
  End;

Procedure Register;

Const                                   { English messages }
  msgNoExcel        = 'Excel cannot be lunched';
  msgNoConnect      = 'Excel not connected';
  msgNoLink         = 'Excel cannot linked';
  msgNoRespond      = 'Excel not responding';
  msgNoReply        = '*** No Reply ***';
  msgNotAccepted    = '" not accepted by Excel';
  msgNoMacro        = 'Macro is not opened';
  msgNoTable        = 'Table is not opened';

Implementation
Uses ExportAs;

//******************* TExportData.SetDataSource *************************

Procedure TDBExport.SetDataSource;
Begin
  If Value <> FDataSource Then
    Begin
      FDataSource := Value;
      If Assigned(FDataSource) Then
        FDataSource.FreeNotification(Self);
    End;
End;
//******************* TExportData.GetFileName *************************

Function TDBExport.GetFileName;
Var
  FSaveDlg          : TSaveDialog;
Begin
  FSaveDlg := TSaveDialog.Create(Nil);
  Try
    FSaveDlg.DefaultExt := DEFAULT_FILTER;
    FSaveDlg.FileName := FFileName;
    FSaveDlg.Filter := FILTER_TYPES;
    If FSaveDlg.Execute Then
      Result := FSaveDlg.FileName
    Else
      abort;
  Finally
    FSaveDlg.Free;
  End;
End;
//******************* TDBExport.Notification *************************

Procedure TDBExport.Notification;
Begin
  Inherited Notification(AComponent, Operation);
  If (Operation = opRemove) And (AComponent = FDataSource) Then
    FDataSource := Nil;
End;
//******************* TDBExport.FilteredData *************************

Function TDBExport.FilteredData;
Begin
  If Assigned(StrScan(PChar(Data), FFieldDelimiter)) Then
    Result := AnsiQuotedStr(Data, '"')
  Else
    Result := Data;
End;
//******************* TDBExport.GetHeader *************************

Function TDBExport.GetHeader;
Var
  FieldIdx          : Integer;
Begin
  Result := '';
  With FDataSource.DataSet Do
    For FieldIdx := 0 To FieldCount - 1 Do
      If Fields[FieldIdx].Visible And Not Fields[FieldIdx].IsBlob Then
        Case FHeader Of
          ehFieldName: Result := Result + Fields[FieldIdx].FieldName + FFieldDelimiter;
          ehDisplayName: Result := Result + FilteredData(Fields[FieldIdx].DisplayName) + FFieldDelimiter;
        End;
  SetLength(Result, Length(Result) - 1);
End;
//******************* TDBExport.GetRecord *************************

Function TDBExport.GetRecord;
Var
  FieldIdx          : Integer;
Begin
  Result := '';
  With FDataSource.DataSet Do
    For FieldIdx := 0 To FieldCount - 1 Do
      If Fields[FieldIdx].Visible And Not Fields[FieldIdx].IsBlob Then
        If (Fields[FieldIdx].DataType = ftString) Then
          Result := Result + FilteredData(Fields[FieldIdx].AsString) + FFieldDelimiter
        Else
          Result := Result + Fields[FieldIdx].AsString + FFieldDelimiter;
  SetLength(Result, Length(Result) - 1);
End;
//******************* TDBExport.StoredCaption *************************

Function TDBExport.StoredCaption;
Begin
  Result := FCaption <> DEF_CAPTION;
End;
//******************* TDBExport.GetSeparator *************************

Function TDBExport.GetSeparator;
Var
  Size              : Integer;
  Sep               : PChar;
Begin
  // Allocate seperator size.
  Size := GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SLIST, Nil, 0);
  Sep := StrAlloc(Size + 1);
  Try
    GetLocaleInfo(LOCALE_USER_DEFAULT, LOCALE_SLIST, Sep, Size);
    Result := Sep^;
  Finally
    StrDispose(Sep);
  End;
End;

//******************* TDBExport.Execute *************************

Function TDBExport.Execute;
Var
  FContents         : TStrings;
  FOpened           : Boolean;
  Progress          : TProgressForm;
Begin
  If FExecuting Then abort;
  Result := False;
  Progress := Nil;
  FExecuting := True;
  Try
    If Assigned(FOnExecute) Then
      FOnExecute(Self);
    If Assigned(FDataSource) And Assigned(FDataSource.DataSet) Then
      Begin
        If Trim(FFileName) = '' Then FFileName := GetFileName;
        If eoShowProgress In FOptions Then Progress := TProgressForm.Create(Nil);
        Try
          If eoShowProgress In FOptions Then
            Begin
              Progress.StatusMessage := 'Initializing...';
              Progress.CanCancel := eoCanCancel In FOptions;
              Progress.Caption := FCaption;
              Progress.Show;
              Progress.Refresh;
            End;
          FDataSource.DataSet.DisableControls;
          If Assigned(FOnInitialize) Then
            FOnInitialize(Self);
          Try
            FOpened := (eoAutoOpen In FOptions) And (Not FDataSource.DataSet.Active);
            FContents := TStringList.Create;
            Try
              If FOpened Then FDataSource.DataSet.Open;
              Try
                If (FHeader <> ehNone) Then FContents.Add(GetHeader);
                If (eoShowProgress In FOptions) Then
                  Begin
                    Progress.MaxValue := FDataSource.DataSet.RecordCount;
                    Progress.StatusMessage := 'Exporting...';
                  End;
                FDataSource.DataSet.First;
                While Not FDataSource.DataSet.Eof Do
                  Begin
                    FContents.Add(GetRecord);
                    FDataSource.DataSet.Next;
                    If eoShowProgress In FOptions Then
                      Begin
                        Progress.ProgressBy(1);
                        If (eoCanCancel In FOptions) Then Application.ProcessMessages;
                        If Progress.ModalResult <> mrNone Then abort;
                      End;
                  End;
              Finally
                If FOpened Then FDataSource.DataSet.Close;
              End;                      // DataSet.Open
              If eoShowProgress In FOptions Then Progress.StatusMessage := 'Saving...';
              FContents.SaveToFile(FFileName);
              Result := True;
            Finally
              FContents.Free;
            End;                        // FContents.Create
          Finally
            FDataSource.DataSet.EnableControls;
          End;                          // DataSet.DisableControls
        Finally
          If eoShowProgress In FOptions Then Progress.Release;
        End;                            // TProgressForm.Create
      End;
  Finally
    FExecuting := False;
  End;                                  // Executing
End;

//******************* TDBExport.Create *************************

Constructor TDBExport.Create;
Begin
  Inherited Create(AOwner);
  FFieldDelimiter := GetSeparator;
  FOptions := DefaultOptions;
  FCaption := DEF_CAPTION;
  FHeader := ehNone;
End;

//******************* TExportDataEditor.GetVerbCount *************************

Function TExportDataEditor.GetVerbCount;
Begin
  Result := 1;
End;

//******************* TExportDataEditor.GetVerb *************************

Function TExportDataEditor.GetVerb;
Begin
  Case (Index) Of
    0: Result := '&Export dataset';
  End;                                  //case
End;

//******************* TExportDataEditor.ExecuteVerb *************************

Procedure TExportDataEditor.ExecuteVerb;
Begin
  Case (Index) Of
    0: (Component As TDBExport).Execute;
  End;                                  //case
End;

//******************* TDEFilenameProperty.Edit *************************

Procedure TDEFilenameProperty.Edit;
Begin
  With (GetComponent(0) As TDBExport) Do
    Try
      FileName := GetFileName;
    Except
      On EAbort Do ;
    End;
End;
//******************* TDEFilenameProperty.GetAttributes *************************

Function TDEFilenameProperty.GetAttributes: TPropertyAttributes;
Begin
  Result := [paDialog, paRevertable];
End;

//******************* TExcelData.Create *************************

Constructor TExcelData.Create(AOwner: TComponent);
Begin
  Inherited Create(AOwner);
  If Not (csDesigning In ComponentState) Then
    Begin
      FDDE := TDdeClientConv.Create(Nil);
      FDDE.ConnectMode := ddeManual;
      FDDE.OnOpen := OpenLink;
      FDDE.OnClose := ShutDown;
    End;
  SetExeName('Excel');
  FLastTime := Now;
  FLimit := 99;
  FDecimals := 2;
  FBatch := False;
  FMin := 200;
  FMax := 250;
End;
//******************* TExcelData.Destroy *************************

Destructor TExcelData.Destroy;
Begin
  If Not (csDesigning In ComponentState) Then FDDE.Free;
  If FLines <> Nil Then FLines.Free;
  If FCells <> Nil Then FCells.Free;
  Inherited Destroy;
End;
//******************* TExcelData.SetExeName *************************

Procedure TExcelData.SetExeName(Const Value: String);
Begin
  Disconnect;
  FExeName := ChangeFileExt(Value, '');
  If Not (csDesigning In ComponentState) Then
    FDDE.ServiceApplication := FExeName;
End;
//******************* TExcelData.SetConnect *************************

Procedure TExcelData.SetConnect(Const Value: Boolean);
Begin
  If FConnected = Value Then Exit;
  If Value Then
    Connect
  Else
    Disconnect;
End;
//******************* TExcelData.SetMin *************************

Procedure TExcelData.SetMin(Const Value: Integer);
Begin
  If Value > FMax Then
    FMin := FMax
  Else
    FMin := Value;
End;
//******************* TExcelData.SetMax *************************

Procedure TExcelData.SetMax(Const Value: Integer);
Begin
  If Value < FMin Then
    FMax := FMin
  Else
    FMax := Value;
End;
//******************* TExcelData.GetSelection *************************

Function TExcelData.GetSelection: String;
Var
  I                 : Integer;
Begin
  Result := Request('Selection');
  I := Pos('''', Result);
  While I > 0 Do
    Begin
      Delete(Result, I, 1);
      I := Pos('''', Result);
    End
End;
//******************* TExcelData.GetReady *************************

Function TExcelData.GetReady: Boolean;
Begin
  Result := 'Ready' = Request('Status');
End;
//******************* TExcelData.DoRect *************************

Procedure TExcelData.DoRect(Top, Left, Bottom, Right: Integer;
  Data: TStrings; Request: Boolean);
Var
  I                 : Integer;
  Sel, Item         : String;
  RowMark,
    ColMark         : char;
  Reply             : PChar;

  Procedure Synchronize;
  Begin
    ProcessMessages;
    Reply := FDDE.RequestData(Copy(Item, 1, Pos(':', Item) - 1));
    StrDispose(Reply);                  { Just to wait for Excel }
  End;

Begin
  Select(1, 1);
  Sel := Selection;
  I := Pos('!', Sel);
  If I = 0 Then Raise Exception.Create(msgNoTable);
  RowMark := Sel[I + 1];                { Some nationalized version }
  ColMark := Sel[I + 3];                {  using other then R and C }
  FDDE.OnOpen := Nil;
  FDDE.OnClose := Nil;                  { Disable event handlers }
  Try
    FDDE.SetLink('Excel', Copy(Sel, 1, I - 1)); { Topic = Sheet name }
    If Not FDDE.OpenLink Then
      Raise Exception.Create(msgNoLink);
    Item := Format('%s%d%s%d:%s%d%s%d', [RowMark, Top, ColMark, Left,
      RowMark, Bottom, ColMark, Right]);
    ProcessMessages;
    If Request Then
      Begin
        Reply := FDDE.RequestData(Item);
        If Reply <> Nil Then Data.SetText(Reply);
        StrDispose(Reply);
      End
    Else
      If FDDE.PokeDataLines(Item, Data) Then
      Synchronize
    Else
      If FDDE.PokeDataLines(Item, Data) Then
      Synchronize                       { Sometimes first call fails }
    Else
      Raise Exception.Create('"Block ' + Item + msgNotAccepted);
  Finally
    LinkSystem;
    If Not FDDE.OpenLink
      And Assigned(FOnClose) Then FOnClose(Self);
    FDDE.OnOpen := OpenLink;
    FDDE.OnClose := ShutDown;           { Enable event handlers }
  End;
End;
//******************* TExcelData.LinkSystem *************************

Procedure TExcelData.LinkSystem;
Begin
  FDDE.SetLink('Excel', 'System');
End;
//******************* TExcelData.CheckConnection *************************

Procedure TExcelData.CheckConnection;
Begin
  If Connected Then
    ProcessMessages
  Else
    Raise Exception.Create(msgNoConnect);
End;
//******************* TExcelData.OpenLink *************************

Procedure TExcelData.OpenLink(Sender: TObject);
Begin
  FConnected := True;
  If Assigned(FOnOpen) Then FOnOpen(Self);
End;
//******************* TExcelData.ShutDown *************************

Procedure TExcelData.ShutDown(Sender: TObject);
Begin
  FConnected := False;
  If Assigned(FOnClose) Then FOnClose(Self);
End;
//******************* TExcelData.LocateExcel *************************

Procedure TExcelData.LocateExcel;
{$IFDEF WIN32}
Const
  BuffSize          = 511;
  {$ELSE}

  Procedure ConvertLongFn(Var Fn: String);
  Var
    Version         : Word;
    Pfn             : Pointer;
  Begin
    If GetWinFlags And $4000 <> 0 Then Exit; { WinNT+ }
    Version := LOWORD(GetVersion);
    If Hi(Version) >= 95 Then           { Win95+ }
      Begin
        { Convert 8.3 path from Long File Names under Win95
          based on the idea of Mieczyslaw Dyla  malwa@elb.pl }
        StrPCopy(@Fn, Fn);
        Pfn := Addr(Fn);
        Asm
        push ds
        mov ax, 7160h
        mov cl, 1
        mov ch, 0
        les di,[DWORD PTR Fn]
        lds si,[Pfn];
        int 21h
        pop ds
        End;
        Fn := StrPas(@Fn)
      End
  End;

Const
  BuffSize          = 255;
  {$ENDIF}
  ExcelExe          = 'EXCEL.EXE';
Var
  Buff              : Array[0..BuffSize] Of char;
  Fn                : String;
  Len               : Longint;
Begin
  Len := BuffSize;
  StrPCopy(Buff, '.XLS');
  If (RegQueryValue(HKEY_CLASSES_ROOT, Buff, Buff, Len)
    = ERROR_SUCCESS) And (StrScan(Buff, 'E') <> Nil) Then
    Begin
      StrCat(Buff, '\Shell\Open\Command');
      Len := BuffSize;
      If RegQueryValue(HKEY_CLASSES_ROOT, Buff, Buff, Len)
        = ERROR_SUCCESS Then
        Begin
          Fn := StrPas(StrUpper(Buff));
          Len := Pos(ExcelExe, Fn);
          Delete(Fn, Len + Length(ExcelExe), BuffSize);
          If Buff[0] = '"' Then Delete(Fn, 1, 1);
          {$IFNDEF WIN32}
          ConvertLongFn(Fn);            { Thanks to Mieczyslaw Dyla }
          {$ENDIF}
          If fileexists(Fn) Then
            ExeName := Fn
        End
    End
End;
//******************* TExcelData.CheckLimit *************************

Procedure TExcelData.CheckLimit;
Var
  Work              : Extended;
Begin
  Inc(FCounter);
  If FCounter > FLimit Then
    Begin
      FCounter := 0;
      Work := FLastTime;
      FLastTime := Now;
      Work := FLastTime - Work;
      If Work < 2E-5 Then Wait;         { 1.728 sec }
    End;
End;
//******************* TExcelData.Connect *************************

Procedure TExcelData.Connect;
Begin
  If FConnected Then Exit;
  LinkSystem;
  If FDDE.OpenLink Then Exit;
  LocateExcel;
  If FDDE.OpenLink Then Exit;           { Try again }
  If FDDE.OpenLink Then Exit;           { Once more }
  Raise Exception.Create(msgNoExcel + #13 + ExeName);
End;
//******************* TExcelData.Disconnect *************************

Procedure TExcelData.Disconnect;
Begin
  If FConnected Then FDDE.CloseLink;
End;
//******************* TExcelData.Wait *************************

Procedure TExcelData.Wait;
Const
  TryCount          = 64;
Var
  I                 : Integer;
Begin
  I := 0;
  Repeat
    ProcessMessages;
    If Ready Then Break;                { Waiting for Excel }
    Inc(I);
  Until I = TryCount;
  If I = TryCount Then
    Raise Exception.Create(msgNoRespond);
End;
//******************* TExcelData.ProcessMessages *************************

Procedure TExcelData.ProcessMessages;
Begin
  Application.ProcessMessages;
End;
//******************* TExcelData.Request *************************

Function TExcelData.Request(Const Item: String): String;
Var
  Reply             : PChar;
Begin
  CheckConnection;
  Reply := FDDE.RequestData(Item);
  If Reply = Nil Then
    Result := msgNoReply
  Else
    Result := StrPas(Reply);
  StrDispose(Reply);
End;
//******************* TExcelData.Exec *************************

Procedure TExcelData.Exec(Const Cmd: String);
Var
  A                 : Array[0..555] Of char;
Begin
  CheckConnection;
  CheckLimit;
  StrPCopy(A, Cmd);
  If FDDE.ExecuteMacro(A, False) Then
    ProcessMessages
  Else
    Begin
      Wait;
      If FDDE.ExecuteMacro(A, True) Then
        While FDDE.WaitStat Do
          ProcessMessages
      Else
        Raise Exception.Create('"' + Cmd + msgNotAccepted);
    End
End;
//******************* TExcelData.Run *************************

Procedure TExcelData.Run(Const Mn: String);
Begin
  If FMacro = '' Then
    Raise Exception.Create(msgNoMacro);
  Exec('[RUN("' + FMacro + '!' + Mn + '";FALSE)]');
End;
//******************* TExcelData.Select *************************

Procedure TExcelData.Select(Row, col: Integer);
Begin
  Exec(Format('[SELECT("R%dC%d")]', [Row, col]));
End;
//******************* TExcelData.PutStr *************************

Procedure TExcelData.PutStr(Row, col: Integer; Const S: String);

  Procedure SendMin;
  Var
    I               : Integer;
  Begin
    FCells.Clear;
    For I := 0 To FMin - 1 Do
      Begin
        FCells.Add(FLines[0]);          { FCells as work space }
        FLines.Delete(0);
      End;
    DoRect(FFirstRow, FFirstCol, FFirstRow + FMin - 1, FLastCol,
      FCells, False);
    Inc(FFirstRow, FMin);
  End;

  Procedure DoBatch;
  Var
    I, j, Index     : Integer;
    Line            : String;
  Begin
    Index := Row - FFirstRow;           { Index to modify }
    If Index >= Lines.Count Then
      For I := Lines.Count To Index Do  { Expand if needed }
        Lines.Add('');
    If Lines.Count > FMax Then          { Send if needed }
      Begin
        SendMin;
        Index := Row - FFirstRow;       { Recalc Index }
      End;
    If col > FLastCol Then FLastCol := col; { Adjust to max }
    Line := Lines[Index];
    FCells.Clear;                       { Empty FCells }
    j := 1;
    For I := 1 To Length(Line) Do       { Line disasseble }
      If Line[I] = #9 Then
        Begin
          FCells.Add(Copy(Line, j, I - j));
          j := I + 1;
        End;
    FCells.Add(Copy(Line, j, Length(Line) + 1 - j));
    If FCells.Count < col - FFirstCol + 1 Then
      For I := FCells.Count To col - FFirstCol Do { Expand if needed }
        FCells.Add('');
    FCells[col - FFirstCol] := S;       { Replace cell }
    Line := FCells[0];
    For I := 1 To FCells.Count - 1 Do   { Line reasseble }
      Line := Line + #9 + FCells[I];
    Lines[Index] := Line;               { Replace line }
  End;

Begin                                   { TExcelData.PutStr }
  If BatchOn And (col >= FFirstCol) And (Row >= FFirstRow) Then
    DoBatch
  Else
    Exec(Format('[FORMULA("%s","R%dC%d")]', [S, Row, col]));
End;
//******************* TExcelData.PutExt *************************

Procedure TExcelData.PutExt(Row, col: Integer; E: Extended);
Begin
  PutStr(Row, col, Format('%0.*f', [Decimals, E]));
End;
//******************* TExcelData.PutInt *************************

Procedure TExcelData.PutInt(Row, col: Integer; I: Longint);
Begin
  PutStr(Row, col, IntToStr(I));
End;
//******************* TExcelData.PutDay *************************

Procedure TExcelData.PutDay(Row, col: Integer; D: TDateTime);
Begin
  PutStr(Row, col, DateToStr(D));
End;
//******************* TExcelData.BatchStart *************************

Procedure TExcelData.BatchStart(FirstRow, FirstCol: Integer);
Begin
  If FLines = Nil Then
    FLines := TStringList.Create
  Else
    FLines.Clear;
  If FCells = Nil Then
    FCells := TStringList.Create
  Else
    FCells.Clear;
  FFirstRow := FirstRow;
  FFirstCol := FirstCol;
  FLastCol := FirstCol;
  FBatch := True;
End;
//******************* TExcelData.BatchCancel *************************

Procedure TExcelData.BatchCancel;
Begin
  If FLines <> Nil Then FLines.Free;
  If FCells <> Nil Then FCells.Free;
  FLines := Nil;
  FCells := Nil;
  FBatch := False;
End;
//******************* TExcelData.BatchSend *************************

Procedure TExcelData.BatchSend;
Begin
  If (FLines <> Nil) And (FLines.Count > 0) Then
    DoRect(FFirstRow, FFirstCol, FFirstRow + FLines.Count - 1,
      FLastCol, FLines, False);
  BatchCancel
End;
//******************* TExcelData.GetBooks *************************

Procedure TExcelData.GetBooks(Books: TStrings);
Var
  Reply, p          : PChar;
  B                 : Array[0..80] Of char;
  Name              : String;
  Len               : Integer;
Begin
  CheckConnection;
  Reply := FDDE.RequestData('Topics');
  p := Reply;
  If Reply <> Nil Then
    Repeat
      p := StrScan(p, '[');
      If p <> Nil Then
        Begin
          Inc(p);
          Len := StrScan(p, ']') - p;
          Name := StrPas(StrLCopy(B, p, Len));
          If (Name <> ':') And (Books.IndexOf(Name) < 0) Then
            Books.Add(Name);
        End;
    Until p = Nil;
  StrDispose(Reply);
End;
//******************* TExcelData.GetSheets *************************

Procedure TExcelData.GetSheets(Const Book: String; Sheets: TStrings);
Var
  Reply, p, Tab, F  : PChar;
  Sheet             : String;
  Len, n            : Integer;
  B, U              : Array[0..80] Of char;
Begin
  CheckConnection;
  Reply := FDDE.RequestData('Topics');
  StrUpper(StrPCopy(B, Book));
  p := Reply;
  If Reply <> Nil Then
    Repeat
      Tab := StrScan(p, #9);
      If Tab = Nil Then
        Len := StrLen(p)
      Else
        Len := Tab - p;
      StrUpper(StrLCopy(U, p, Len));
      F := StrPos(U, B);
      If F <> Nil Then
        Begin
          n := Length(Book) + F - U + 1;
          Inc(p, n);
          Dec(Len, n);
          Sheet := StrPas(StrLCopy(U, p, Len));
          Sheets.Add(Sheet);
        End;
      p := Tab + 1
    Until Tab = Nil;
  StrDispose(Reply);
End;
//******************* TExcelData.GetRange *************************

Procedure TExcelData.GetRange(R: TRect; Lines: TStrings);
Begin
  DoRect(R.Top, R.Left, R.Bottom, R.Right, Lines, True);
End;
//******************* TExcelData.GetCell *************************

Function TExcelData.GetCell(Row, col: Integer): String;
Var
  Data              : TStringList;
Begin
  Result := msgNoReply;
  Data := TStringList.Create;
  Try
    DoRect(Row, col, Row, col, Data, True);
    If Data.Count = 1 Then Result := Data[0];
  Finally
    Data.Free
  End;
End;
//******************* TExcelData.OpenMacroFile *************************

Procedure TExcelData.OpenMacroFile(Const Fn: String; hide: Boolean);
Begin
  If FMacroPath = Fn Then Exit;
  CloseMacroFile;
  Exec('[OPEN("' + Fn + '")]');
  If hide Then Exec('[HIDE()]');
  FMacroPath := Fn;
  FMacro := ExtractFileName(Fn);
End;
//******************* TExcelData.CloseMacroFile *************************

Procedure TExcelData.CloseMacroFile;
Begin
  If FMacro <> '' Then
    Try
      Exec('[UNHIDE("' + FMacro + '")]');
      Exec('[ACTIVATE("' + FMacro + '")]');
      Exec('[CLOSE(FALSE)]');
    Finally
      FMacro := '';
      FMacroPath := '';
    End;
End;

//******************* TExportData.Create *************************

Constructor TExportDataFile.Create(Owner: TComponent);
Begin
  {  fTable := TTable.Create(Owner);
    fExcel := TExcelData.Create(Owner);
    fCSV := TDBExport.Create(Owner);}
  Inherited Create(Owner);
End;
//******************* TExportDataFile.Destroy *************************

Destructor TExportDataFile.Destroy;
Begin
  {  fTable.Free;
    fExcel.Free;
    fCSV.Free;}
  Inherited Destroy;
End;
//******************* TExportDataFile.SetDataSet *************************

Procedure TExportDataFile.SetDataSet;
Begin
  If Value <> FDataSet Then
    Begin
      FDataSet := Value;
      If Assigned(FDataSet) Then
        FDataSet.FreeNotification(Self);
    End;
End;
//******************* TExportDataFile.Notification *************************

Procedure TExportDataFile.Notification(AComponent: TComponent; Operation: TOperation);
Begin
  Inherited Notification(AComponent, Operation);
  If (Operation = opRemove) And (AComponent = FDataSet) Then
    FDataSet := Nil;
End;
//******************* TExportDataFile.Execute *************************

Function TExportDataFile.Execute: Boolean;
Var
  ExportFrm         : TExportFrm;
  R                 : TExportFile;
Begin
  Result := False;
  ExportFrm := TExportFrm.Create(Self);
  Try
    R := ExportFrm.ExportDataAs(fFileName);
    fFileName := R.FileName;
    ExportTableAs(R);
  Finally
    ExportFrm.Free;
  End;
End;
//******************* TExportDataFile.ExportTableAs *************************

Procedure TExportDataFile.ExportTableAs(ExportType: TExportFile);
Begin
  Case ExportType.FormatType Of
    dfParadox, dfDBase, dfFoxPro: ExportDB(ExportType);
    dfExcel: ExportXLS(ExportType);
  End;
  If (exporttype.FormatType = dfCSV) And (exporttype.txtFormat = 1) Then
    Begin
      fCSV := TDBExport.Create(Self);
      Try
        fCSV.FileName := exporttype.FileName;
        fCSV.Execute;
      Finally
        fCSV.Free;
      End;
    End;
  If (exporttype.FormatType = dfCSV) And (exporttype.txtFormat = 0) Then
    ExportDB(ExportType);
End;
//******************* TExportDataFile.ExportDB *************************

Procedure TExportDataFile.ExportDB(ExportType: TExportFile);
Var
  I                 : Integer;
  B                 : tdbdataset;
Begin
  fTable := TTable.Create(Self);
  Try
    If fileexists(fFileName) Then
      If MessageDlg('File already on the disk. Do you wish to overwrite it?',
        mtConfirmation, [mbYes, mbNo], 0) = mrNo Then
        Exit;
    Case ExportType.FormatType Of
      dfParadox: fTable.TableType := ttParadox;
      dfDBase: fTable.TableType := ttDBASE;
      dfASCII, dfCSV: fTable.TableType := ttASCII;
      dfFoxPro: fTable.TableType := ttFoxPro;
    End;
    fTable.TableName := fFileName;
    fTable.FieldDefs.Clear;
    For I := 0 To FDataSet.FieldCount - 1 Do
      fTable.FieldDefs.Add(FDataSet.FieldDefs.Items[I].Name, FDataSet.FieldDefs.Items[I].DataType, FDataSet.FieldDefs.Items[I].Size, False);
    fTable.CreateTable;

    CopyDB(FDataSet, fTable);
  Finally
    fTable.Free;
  End;
End;
//******************* TExportDataFile.ExportXLS *************************

Procedure TExportDataFile.ExportXLS(ExportType: TExportFile);
Var
  I, j              : Integer;
  B                 : TBookmark;
Begin
  fExcel := TExcelData.Create(Self);
  Try
    B := FDataSet.GetBookmark;
    Try
      fExcel.Connect;
      fExcel.Exec('[CLOSE(FALSE)]');
      fExcel.Exec('[NEW(1)]');
      fExcel.Exec('[SAVE.AS("' + fFileName + '";1;"";FALSE;"";TRUE)]'
        );
      fExcel.Exec('[WORKBOOK.SELECT("Sheet1")]');
      FDataSet.First;
      For I := 1 To FDataSet.RecordCount Do
        Begin
          For j := 0 To FDataSet.FieldDefs.Count - 1 Do
            Try
              Case FDataSet.FieldDefs.Items[j].DataType Of
                ftDateTime: fExcel.PutDay(I, j + 1, FDataSet.Fields[j].AsDateTime);
                ftString: fExcel.PutStr(I, j + 1, FDataSet.Fields[j].AsString);
                ftWord, ftInteger: fExcel.PutInt(I, j + 1, FDataSet.Fields[j].AsInteger);
                ftFloat: fExcel.PutExt(I, j + 1, FDataSet.Fields[j].asFloat);
                Else
                  fExcel.PutStr(I, j + 1, FDataSet.Fields[j].AsString);
              End;
            Except
            End;
          FDataSet.Next;
          Application.ProcessMessages;
        End;
      fExcel.Exec('[COLUMN.WIDTH(0,"C1:C' + IntToStr(FDataSet.FieldCount - 1) + '",FALSE,3)]');
      fExcel.Exec('[SELECT("R1C1:R' + IntToStr(FDataSet.RecordCount - 1) + 'C' + IntToStr(FDataSet.FieldCount - 1) + '")]');
      fExcel.Exec('[FORMAT.NUMBER("#.##0,00")]');
      fExcel.Exec('[ALIGNMENT(4)]');
      fExcel.Disconnect;
    Except
      On E: Exception Do
        MessageDlg('An error occurs during export process. Process aborted.'#13#10'System message: ' + E.Message, mtError, [mbOK], 0);
    End;
    FDataSet.GotoBookmark(B);
    FDataSet.FreeBookmark(B);
  Finally
    fExcel.Free;
  End;
End;
//******************* TExportDataFileCopyDB *************************

Procedure TExportDataFile.CopyDB(Source: TDataSet; Destination: TTable);
Var
  I                 : Integer;
  B                 : TBookmark;
Begin
  B := Source.GetBookmark;
  Destination.Active := True;
  Source.First;
  While Not Source.Eof Do
    Begin
      Destination.Append;
      Destination.Edit;
      For I := 0 To Source.FieldDefs.Count - 1 Do
        Destination.FieldByName(Source.FieldDefs.Items[I].Name).AsVariant := Source.FieldByName(Source.FieldDefs.Items[I].Name).AsVariant;
      Destination.Post;
      Source.Next;
    End;
  Destination.Active := False;
  Source.GotoBookmark(B);
  Source.FreeBookmark(B);
End;

Procedure Register;
Begin
  RegisterComponents('EM-Quicksoft', [TExportDataFile]);
  RegisterComponentEditor(TDBExport, TExportDataEditor);
  RegisterPropertyEditor(TypeInfo(String), TDBExport, 'FileName', TDEFilenameProperty);
End;

End.

