unit Coders;

interface

Uses Sysutils, Classes, Windows;

Type
    TArray  = array[0..0] of Byte;
    PArray = ^TArray;
    TCodeType = (ctDefault, ctHex, ctPrintable, ctLatin, ctBase64, ctUUE, ctXX);

    TDefaultCoder = Class( TObject )
    private
           FOrgString,
           FCodedString,

           FCodeTable : String;
           FDecode    : Array[0..255] of byte;
           FFillChar  : Char;
           FCodeType  : TCodeType;

           Procedure SetCodeTable( Value : String );
    Protected
           Function  GetCodeChar( By : Byte ) : Char; virtual;
           Function  GetByte( ch : Char ) : Byte; virtual;
           Function  CodeString( value : String ) : String; virtual;
           Function  DeCodeString( value : String ) : String; virtual;
           Procedure SetOrgString( Value : String );
           Procedure SetCodeString(value : String );
    Public
          Constructor Create;
          Destructor  Destroy; override;
          Property    OrgString   : String Read ForgString Write SetOrgString;
          Property    CodedString : String Read FCodedString Write SetCodeString;
          Property    CodeTable   : String Read FCodedString Write SetCodeTable;
          Property    FillChar    : Char Read  FFillChar Write FFillChar;
          Property    CodeTyoe    : TCodeType Read FCodeType;
    End;

    ThexCoder = Class( TDefaultCoder )
    Public
          Constructor Create;
    Protected
           Function  CodeString( value : String ) : String; override;
           Function  DeCodeString( value : String ) : String; override;
    End;

    TQuotedPrintableCoder = Class( TDefaultCoder )
    Public
          Constructor Create;
    Protected
           Function  CodeString( value : String ) : String; override;
           Function  DeCodeString( value : String ) : String; override;
    End;

    TLatinCoder = Class( TDefaultCoder )
    Public
          Constructor Create;
    Protected
           Function  CodeString( value : String ) : String; override;
           Function  DeCodeString( value : String ) : String; override;
    End;


    TBase64Coder = Class( TDefaultCoder )
    Protected
           Function  CodeString( value : String ) : String; override;
           Function  DeCodeString( value : String ) : String; override;
    Public
          Constructor Create;
    End;

    TUUeCoder = Class( TBase64Coder )
    Public
          Constructor Create;
    End;

    TXXCoder = Class( TBase64Coder )
    Public
          Constructor Create;
    End;


    TMainCoder = Class( TObject )
    Private
           Fcoder : TDefaultCoder;
    Protected
             Function GetCoderType : TCodeType;
             Function GetOrgStr : String;
             Function GetCodedStr: String;
             Procedure SetOrgString( Value : String );
             Procedure SetCodedString( Value : String );
    Public
          Constructor Create( WhatCoder : TCodeType );
          Property    CodeType : TCodeType Read GetCoderType;
          Property    OrgString : String Read GetOrgStr Write SetOrgString;
          Property    CodedString : String Read GetCodedStr Write SetCodedString;
    End;


implementation

Procedure TmainCoder.SetCodedString;
Begin
     case Fcoder.CodeTyoe of
          ctBase64:       (Fcoder as TBase64Coder).CodedString := Value;
          ctHex:          (Fcoder as ThexCoder).CodedString := Value;
          ctLatin:        (Fcoder as TLatinCoder).CodedString := Value;
          ctPrintable:    (Fcoder as TQuotedPrintableCoder).CodedString := Value;
          ctUUE:          (Fcoder as TUUeCoder).CodedString := Value;
          ctXX:           (Fcoder as TXXCoder).CodedString := Value;

          else            (Fcoder as TDefaultCoder).CodedString := Value;
     End;
End;

Procedure TmainCoder.SetOrgString;
Begin
     case Fcoder.CodeTyoe of
          ctBase64:       (Fcoder as TBase64Coder).OrgString := Value;
          ctHex:          (Fcoder as ThexCoder).OrgString := Value;
          ctLatin:        (Fcoder as TLatinCoder).OrgString := Value;
          ctPrintable:    (Fcoder as TQuotedPrintableCoder).OrgString := Value;
          ctUUE:          (Fcoder as TUUeCoder).OrgString := Value;
          ctXX:           (Fcoder as TXXCoder).OrgString := Value;

          else            (Fcoder as TDefaultCoder).OrgString := Value;
     End;
End;


Function TmainCoder.GetCodedStr;
Begin
     Result := Fcoder.CodedString;
End;

Function TmainCoder.GetOrgStr;
Begin
     Result := Fcoder.OrgString;
End;

Function TmainCoder.GetCoderType;
Begin
     Result := Fcoder.CodeTyoe;
End;

Constructor TMainCoder.Create;
Begin
     case WhatCoder of
          ctBase64:     Fcoder := TBase64Coder.Create;
          ctHex:        Fcoder := ThexCoder.Create;
          ctPrintable:  Fcoder := TQuotedPrintableCoder.Create;
          ctLatin:      Fcoder := TLatinCoder.Create;
          ctUUE:        Fcoder := TUUeCoder.Create;
          ctXX:         Fcoder := TXXCoder.Create;

          else          Fcoder := TDefaultCoder.Create;
     End;
End;

Type
    TCoderRec = Record
      Case integer of
           0: (Bytes : array[0..4] of Byte);
           1: (Pack  : Cardinal);
           2: (Chars : Array[0..4] of Char );
    End;

Constructor TXXCoder.Create;
Begin
     inherited;
     CodeTable := '+-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
     FFillChar := '~';
     FCodeType := ctXX;
End;

Constructor TUUeCoder.Create;
Begin
     inherited;
     CodeTable := '`!"#$%&''()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_';
     FFillChar := '~';
     FCodeType := ctUUE;
End;

Function TBase64Coder.CodeString;
Var InRec, OutRec : TCoderRec;
    s             : String;
    i, Len        : Integer;
    b             : Byte;
Begin
     s := '';
     while value > '' Do
     Begin
          System.FillChar(InRec,sizeof(TCoderRec),0);
          System.FillChar(OutRec,sizeof(TCoderRec),0);
          if Length(value) >= 3 Then
          Begin
               StrPCopy(InRec.Chars,copy(value,1,3));
               Delete(value,1,3);
               Len := 3;
          End Else
          Begin
               if length(value) = 2 Then
               Begin
                    StrPCopy(InRec.Chars,copy(value,1,2));
                    Len := 2;
               End Else
               Begin
                    InRec.Chars[0] := Value[1];
                    Len := 1;
               End;
               value := '';
          End;
          i := 0;

          OutRec.Bytes[0] := (InRec.Bytes[0] shr 2) and 63;
          OutRec.Bytes[1] := ((InRec.Bytes[0] shl 4) or (InRec.Bytes[1] shr 4)) and 63;
          OutRec.Bytes[2] := ((InRec.Bytes[1] shl 2) or (InRec.Bytes[2] shr 6)) and 63;
          OutRec.Bytes[3] := InRec.Bytes[2] and 63;

          for i := 0 to 3 do OutRec.Chars[i] := GetCodeChar(Succ(OutRec.Bytes[i]));

          Case Len Of
               1:  Begin OutRec.Chars[2] := FFillChar; OutRec.Chars[3] := FFillChar; End;
               2:  OutRec.Chars[3] := FFillChar;
          End;

          s := s + StrPas(OutRec.Chars);
     End;
     Result := s;
End;

Function TBase64Coder.DeCodeString;
Var InRec, OutRec : TCoderRec;
    s             : String;
    i, Len, k     : Integer;
    b             : Byte;
Begin
     Result := '';
     while (Length(value) mod 4) > 0 Do
     Begin
          value := value + FFillChar;
     End;
     s := '';
     while value > '' Do
     Begin
          System.FillChar(InRec,sizeof(TCoderRec),0);
          System.FillChar(OutRec,sizeof(TCoderRec),0);
          StrPCopy(InRec.Chars,copy(value,1,4));
          Delete(value,1,4);

          Len := 0;
          for i := 0 To 3 Do
          Begin
               if InRec.Chars[i] = FFillChar Then Break;
               OutRec.Bytes[i] := GetByte(inRec.Chars[i]);
               Inc(Len);
          End;

          k := ((OutRec.bytes[0] shl 6) or (OutRec.bytes[1])) shl 12;
          case len of
               3:  k := k or (OutRec.bytes[2] shl 6);
               4:  k := k or (OutRec.bytes[2] shl 6) or OutRec.bytes[3];
          End;
          OutRec.Pack := k;

          case Pred(Len) of
               1:        Begin
                              OutRec.Bytes[0] := OutRec.bytes[2];
                              OutRec.Bytes[1] := 0;
                         End;

               2:        Begin
                              OutRec.Bytes[0] := OutRec.bytes[2];
                              OutRec.Bytes[1] := OutRec.bytes[1];
                              OutRec.Bytes[2] := 0;
                         End;

               else      Begin
                              b := OutRec.bytes[0];
                              OutRec.Bytes[0] := OutRec.bytes[2];
                              OutRec.Bytes[2] := b;
                              OutRec.Bytes[3] := 0;
                         End;
          End;
          s := s + StrPas(OutRec.Chars);
     End;
     Result := s;
End;


Constructor TBase64Coder.Create;
Var s : String;
    c : Char;
Begin
     inherited;
     s := '';
     For c := 'A' to 'Z' Do s := s + c;
     For c := 'a' to 'z' Do s := s + c;
     For c := '0' to '9' Do s := s + c;
     CodeTable := s + '+/';
     FFillChar := '=';
     FCodeType := ctBase64;
End;

Constructor TQuotedPrintableCoder.Create;
Begin
     inherited;
     FCodeType := ctPrintable;
End;

Function TQuotedPrintableCoder.CodeString;
var  s : String;

     Function HexOut( Ch : Char ) : String;
     Begin
          Result := '=' + IntToHex(Ord(ch),2);
     End;

Begin
     s := '';
     while value > '' Do
     Begin
          if value[1] in [#33..#60,#62..#162] then s := s + value[1]
                                              else s := s + HexOut(value[1]);
          Delete(value,1,1);
     End;
     Result := s;
End;

Function TQuotedPrintableCoder.DeCodeString;
Var s, x : String;
Begin
     s := '';
     while value > '' Do
     Begin
          if value[1] = '=' Then
          Begin
               x := Copy(value,1,3);
               x[1] := '$';
               s := s + Chr(StrtoIntDef(x,0));
               Delete(value,1,3);
          End Else
          Begin
               s := s + value[1];
               Delete(Value,1,1);
          End;
     End;
     Result := s;
End;

Constructor ThexCoder.Create;
Begin
     inherited;
     FCodeType := ctHex;
End;

Function ThexCoder.DeCodeString;
Var Ts : TStringList;
    s, x : String;
Begin
     s := '';
     ts := TStringList.Create;
     ts.Text := value;
     while ts.Count > 0 Do
     Begin
          x := trim(Ts[0]);
          Ts.Delete(0);
          while x > '' Do
          Begin
               if Length(x) >= 2 Then
               Begin
                    s := s + Chr(StrToIntDef('$'+copy(x,1,2),0));
                    delete(x,1,2);
               End Else
               Begin
                    s := s + Chr(StrToIntDef('$'+x,0));
                    x := '';
               End;
          End;
     End;
     Ts.Free;
     Result := s;
End;

Function ThexCoder.CodeString;
var  Ts : TStringList;
     s, x: String;
     i   : Integer;
Begin
     ts := TStringList.Create;
     while value > '' Do
     Begin
          if Length(value) > 16 Then
          Begin
               s := copy(value,1,16);
               Delete(value,1,16);
          End Else
          Begin
               s := value;
               value := '';
          End;
          x := '';
          for i := 1 To Length(s) Do x := x + IntToHex(Ord(s[i]),2);
          Ts.Add(x);
     End;
     Result := Ts.Text;
     Ts.Free;
End;

Constructor TDefaultCoder.Create;
Var i : Integer;
Begin
     inherited;
     FCodeType := ctDefault;
     for i := 0 to 255 do FDecode[i] := i;
End;

Destructor TDefaultCoder.Destroy;
Begin
     inherited;
End;

Procedure TDefaultCoder.SetCodeTable;
Var i     : Integer;
    b     : Byte;
    a     : Byte;
Begin
     System.FillChar(FDecode,256,255);
     FCodeTable := Value;
     if FCodeTable > '' Then
     Begin
          for i := 1 To Length(FCodeTable) Do
          Begin
               a := Ord(FCodeTable[i]);
               b := Pred(i);
               FDecode[a] := b;
          End;
     End;
End;

Function TDefaultCoder.GetByte;
begin
     Result := FDecode[Ord(Ch)];
End;

Function TDefaultCoder.GetCodeChar;
Begin
     if (by >= 0) and (by <= Length(FCodeTable)) Then Result := FCodeTable[by]
                                                 Else Result := Chr(by);
End;

Function TDefaultCoder.CodeString;
var i : Integer;
    s : String;
Begin
     s := '';
     for i := 1 to length(value) Do s := s + GetCodeChar(Ord(Value[i]));
     Result := s;
End;

Function TDefaultCoder.DeCodeString;
var i   : Integer;
    s   : String;
Begin
     s := '';
     for i := 1 to Length(value) Do
     Begin
          s := s + chr(GetByte(value[i]));
     End;
     Result := s;
End;

Procedure TDefaultCoder.SetOrgString;
Begin
     FOrgString := Value;
     FCodedString := CodeString(FOrgString);
End;

Procedure TDefaultCoder.SetCodeString;
Begin
     FCodedString := value;
     FOrgString := DeCodeString(FCodedString);
End;

const Entities:array [1..101,1..2] of string=(
('&quot;',  '&#34;'),
('&amp;',   '&#38;'),
('&prime;', '&#39;'),
('&lt;',    '&#60;'),
('&gt;',    '&#62;'),
('&nbsp;',  '&#160;'),
('&iexcl;', '&#161;'),
('&cent;',  '&#162;'),
('&pound;', '&#163;'),
('&curren;','&#164;'),
('&yen;',   '&#165;'),
('&brvbar;','&#166;'),
('&sect;',  '&#167;'),
('&uml;',   '&#168;'),
('&copy;',  '&#169;'),
('&ordf;',  '&#170;'),
('&laquo;', '&#171;'),
('&not;',   '&#172;'),
('&shy;',   '&#173;'),
('&reg;',   '&#174;'),
('&macr;',  '&#175;'),
('&deg;',   '&#176;'),
('&plusmn;','&#177;'),
('&sup2;',  '&#178;'),
('&sup3;',  '&#179;'),
('&acute;', '&#180;'),
('&micro;', '&#181;'),
('&para;',  '&#182;'),
('&middot;','&#183;'),
('&cedil;', '&#184;'),
('&sup1;',  '&#185;'),
('&ordm;',  '&#186;'),
('&raquo;', '&#187;'),
('&frac14;','&#188;'),
('&frac12;','&#189;'),
('&frac34;','&#190;'),
('&iquest;','&#191;'),
('&Agrave;','&#192;'),
('&Aacute;','&#193;'),
('&Acirc;', '&#194;'),
('&Atilde;','&#195;'),
('&Auml;',  '&#196;'),
('&Aring;', '&#197;'),
('&AElig;', '&#198;'),
('&Ccedil;','&#199;'),
('&Egrave;','&#200;'),
('&Eacute;','&#201;'),
('&Ecirc;', '&#202;'),
('&Euml;',  '&#203;'),
('&Igrave;','&#204;'),
('&Iacute;','&#205;'),
('&Icirc;', '&#206;'),
('&Iuml;',  '&#207;'),
('&ETH;',   '&#208;'),
('&Ntilde;','&#209;'),
('&Ograve;','&#210;'),
('&Oacute;','&#211;'),
('&Ocirc;', '&#212;'),
('&Otilde;','&#213;'),
('&Ouml;',  '&#214;'),
('&times;', '&#215;'),
('&Oslash;','&#216;'),
('&Ugrave;','&#217;'),
('&Uacute;','&#218;'),
('&Ucirc;', '&#219;'),
('&Uuml;',  '&#220;'),
('&Yacute;','&#221;'),
('&THORN;', '&#222;'),
('&szlig;', '&#223;'),
('&agrave;','&#224;'),
('&aacute;','&#225;'),
('&acirc;', '&#226;'),
('&atilde;','&#227;'),
('&auml;',  '&#228;'),
('&aring;', '&#229;'),
('&aelig;', '&#230;'),
('&ccedil;','&#231;'),
('&egrave;','&#232;'),
('&eacute;','&#233;'),
('&ecirc;', '&#234;'),
('&euml;',  '&#235;'),
('&igrave;','&#236;'),
('&iacute;','&#237;'),
('&icirc;', '&#238;'),
('&iuml;',  '&#239;'),
('&eth;',   '&#240;'),
('&ntilde;','&#241;'),
('&ograve;','&#242;'),
('&oacute;','&#243;'),
('&ocirc;', '&#244;'),
('&otilde;','&#245;'),
('&ouml;',  '&#246;'),
('&divide;','&#247;'),
('&oslash;','&#248;'),
('&ugrave;','&#249;'),
('&uacute;','&#250;'),
('&ucirc;', '&#251;'),
('&uuml;',  '&#252;'),
('&yacute;','&#253;'),
('&thorn;', '&#254;'),
('&yuml;',  '&#255;'));



// charset with iso-8559-1 (Western Latin-1)
const CharSet:array [0..255] of char=(
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
' ',
'!',
'"',
'#',
'$',
'%',
'&',
#39,
'(',
')',
'*',
'+',
',',
'-',
'.',
'/',
'0',
'1',
'2',
'3',
'4',
'5',
'6',
'7',
'8',
'9',
':',
';',
'<',
'=',
'>',
'?',
'@',
'A',
'B',
'C',
'D',
'E',
'F',
'G',
'H',
'I',
'J',
'K',
'L',
'M',
'N',
'O',
'P',
'Q',
'R',
'S',
'T',
'U',
'V',
'W',
'X',
'Y',
'Z',
'[',
'\',
']',
'^',
'_',
'`',
'a',
'b',
'c',
'd',
'e',
'f',
'g',
'h',
'i',
'j',
'k',
'l',
'm',
'n',
'o',
'p',
'q',
'r',
's',
't',
'u',
'v',
'w',
'x',
'y',
'z',
'{',
'|',
'}',
'~',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',

// charset iso-8559-1 (Western Latin-1)
' ',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'',
'');

Constructor TLatinCoder.Create;
Begin
     FCodeType := ctLatin;
End;

Function TLatinCoder.CodeString;
Var s : String;
         Function CharToLatin(Value : Char) : String;
         Var
         v, i   : Integer;
         s      : String;
         Begin
              Result := Value;
              v := Ord(Value);
              if (v >= 34) and (v <= 255) Then
              Begin
                   For i := 1 to 101 Do
                   Begin
                        s := Entities[i,2];
                        Delete(s,length(s),1);
                        Delete(s,1,2);
                        if v = StrToIntDef(trim(s),0) Then
                        Begin
                             Result := Entities[i,1];
                             break;
                        End;
                   End;
              End;
         End;
Begin
     s := '';
     while Length(value) > 0 Do
     Begin
          s := s + CharToLatin(value[1]);
          Delete(value,1,1);
     End;
     Result := s;
End;

Function TLatinCoder.DeCodeString;
var
   start, i     : Integer;
   s, entity, begl, endl, x  : string;
   ch                        : char;

         Function LatinToChar( Value : String ) : Char;
         Var entity : String;
         cpos   : integer;

                Function GetChar : Char;
                Begin
                     delete(entity,1,2);
                     if entity[1] in ['x','X'] then entity[1] := '$';
                     cpos   := StrToIntDef(entity,32);
                     Result := CharSet[cpos];
                End;

                Function GetLatin : Char;
                var i : Integer;
                Begin
                     entity := Trim(entity)+';';
                     Result := #0;
                     for i := 1 to 100 Do
                     Begin
                          if Entities[i,1] = entity then
                          begin
                               entity := Entities[i,2];
                               delete(entity,length(entity),1);
                               result := GetChar;
                               break;
                          end;
                     End;
                End;

                Begin
                     entity := Value;
                     if entity[2] = '#' Then Result := GetChar
                                        else Result := GetLatin;
                End;

Begin
     repeat
           s := LowerCase(Value);
           start := pos('&',s);
           if start > 0 Then
           Begin
                begl   := copy(Value,1, Pred(start));
                entity := '&';
                x      := #0;
                delete(value, start, 1);
                while ( Length(Value) >= start ) do
                begin
                     ch := value[start];
                     x  := x + ch;
                     if ch = ';' then ch := ' ';
                     entity := entity + ch;
                     Delete(Value, start, 1);
                     if ch = ' ' then break;
                end;
                delete(Value, 1, length(begl));
                endl := Value;
                ch := LatinToChar(entity);
                if ch > #0 Then Value := begl + ch + endl
                           else Value := begl + x + endl;
           End;
     until start = 0;

     for i := 1 to Length(Value) Do
         if value[i] = #0 Then Value[i] := '&';

     result := value;

End;

end.
