unit FileSplitter;

{------------------------------------------------------------------------------}
{  #7 Components - TFileSplitter v1.03                                         }
{                                                                              }
{  Check http://sedlan.tripod.com for new, updated versions                    }
{------------------------------------------------------------------------------}
{******************************************************************************}
{                                                                              }
{       Component description:                                                 }
{                                                                              }
{       TFileSplitter is an invisible component with a simple task to          }
{  split a file into multiple files of certain size. The most common need      }
{  for this is when you want to put your (big) file on floppy disks, for       }
{  backup or any other purposes. This component also includes method for       }
{  reversing the process - the unsplitting. All existing files will be         }
{  overwritten without warning during the process of splitting/unsplitting.    }
{       When splitting, if the SplitFileName property is empty, a file(s)      }
{  with the same name as FileName will be created, with extension(s) as        }
{  follows: '.SPL', '.001', '.002', etc.                                       }
{       When unsplitting, if the FileName property is empty, a file with       }
{  the same name as SplitFileName will be created, with extension '.XXX'.      }
{                                                                              }
{******************************************************************************}
{                                                                              }
{       Properties:                                                            }
{       -----------                                                            }
{       BufferSize        - Size of buffer used for reading input data and     }
{                           writing to output file(s).                         }
{       FileName          - The name of a file to be split, or the name of a   }
{                           resulting file in case of unsplitting.             }
{       ReduceFirstSizeBy - Size of the first split file will be reduced by    }
{                           this number of bytes                               }
{       Size              - Maximum size (in bytes) of the resulting           }
{                           (splitted) file(s). Not needed in case of          }
{                           unsplitting.                                       }
{       SplitFileName     - The name of a resulting file, or the name of the   }
{                           first file in case of unsplitting.                 }
{                                                                              }
{       Events:                                                                }
{       -------                                                                }
{       OnNeedDisk        - If assigned, this event is fired before the        }
{                           creation of each splitted file. Also, this event   }
{                           is fired before appending each splitted file to    }
{                           the original file, in case of unsplitting.         }
{       OnProgress        - If assigned, this event gets called every time     }
{                           the buffer is written to the resulting file. This  }
{                           applies to both splitting and unsplitting.         }
{                                                                              }
{       Methods:                                                               }
{       --------                                                               }
{       Split             - This method executes the splitting.                }
{       UnSplit           - This method does the unsplitting.                  }
{                                                                              }
{******************************************************************************}
{                                                                              }
{       Known bugs:                                                            }
{                                                                              }
{       There are no known bugs at this time.                                  }
{                                                                              }
{******************************************************************************}
{                                                                              }
{       Version history:                                                       }
{                                                                              }
{       March 08, 2000       - v1.00       - Initial release                   }
{       March 20, 2000       - v1.01       - OnNeedDisk was called one time    }
{                                            AFTER the splitting. Corrected    }
{                                            now.                              }
{                                          - OnProgress event created.         }
{                                          - Signature is placed on the        }
{                                            first split file. This helps      }
{                                            to count files during             }
{                                            unsplitting.                      }
{                                          - OnNeedDisk now has one more       }
{                                            parameter - NumOfDisks. This      }
{                                            parameter is zero for the         }
{                                            first call when unsplitting.      }
{       March 22, 2000       - v1.02       - Boy, was this buggy...            }
{       May   07, 2000       - v1.03       - Property ReduceFirstSizeBy added. }
{                                            This property can be used to      }
{                                            make the first split file smaller }
{                                            by a number of bytes. Helpful if  }
{                                            you are developing some kind of a }
{                                            setup program. Added upon request }
{                                            by Mr. Florian Haag.              }
{                                                                              }
{******************************************************************************}
{                                                                              }
{          Copyright (c) 2000 by Jovan Sedlan. All rights reserved.            }
{                                                                              }
{  Copyright:                                                                  }
{                                                                              }
{  TFileSplitter (hereafter "component") source code, and any other source     }
{  code inside this archive is copyrighted by Jovan Sedlan (hereafter          }
{  "author"), and shall remain the exclusive property of the author.           }
{                                                                              }
{  Distribution Rights:                                                        }
{                                                                              }
{  You are granted a non-exlusive, royalty-free right to produce and           }
{  distribute compiled binary files (executables, DLLs, etc.) that are         }
{  built with this component's source code unless specifically stated          }
{  otherwise.                                                                  }
{  You are further granted permission to redistribute the component source     }
{  code in source code form, provided that the original archive as found on    }
{  our web site (http://sedlan.tripod.com) is distributed unmodified.          }
{  For example, if you create a descendant of TFileSplitter, you must          }
{  include in  the distribution package the FileSpl.zip file in the exact      }
{  form that you downloaded it from http://sedlan.tripod.com                   }
{                                                                              }
{  Restrictions:                                                               }
{                                                                              }
{  Without the express written consent of the author, you may not:             }
{   * Distribute modified versions of any DFS source code by itself. You       }
{     must include the original archive as you found it on the web.            }
{   * Sell or lease any portion of DFS source code. You are, of course,        }
{     free to sell any of your own original code that works with, enhances,    }
{     etc. this component's source code.                                       }
{   * Distribute component's source code for profit.                           }
{                                                                              }
{  Warranty:                                                                   }
{                                                                              }
{  There is absolutely no warranty of any kind whatsoever with this            }
{  component's source code (hereafter "software"). The software is provided    }
{  to you "AS-IS", and all risks and losses associated with it's use are       }
{  assumed by you. In no event shall the author of the softare, Jovan          }
{  Sedlan, be held accountable for any damages or losses that may occur        }
{  from use or misuse of the software.                                         }
{                                                                              }
{  Support:                                                                    }
{                                                                              }
{  Support is provided via email only. The source code for this software is    }
{  provided free of charge. As such, I can not guarantee any support           }
{  whatsoever. While I do try to answer all questions that I receive, and      }
{  address all problems that are reported to me, you must understand that I    }
{  simply cannot guarantee that this will always be so.                        }
{                                                                              }
{------------------------------------------------------------------------------}
{  The latest version of my components are always available on the web at:     }
{    http://sedlan.tripod.com                                                  }
{------------------------------------------------------------------------------}
{  Date last modified:  May 07, 2000.                                          }
{------------------------------------------------------------------------------}
{******************************************************************************}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DsgnIntf;

type
  TSignature = record
    Copyright: array [1..29] of Char;
    NumberOfFiles: Word;
  end;
  
  EFileError = class(Exception);

  TNeedDiskEvent = procedure(Sender: TObject; DiskID, NumOfDisks: Word; var Continue: Boolean) of object;
  TProgressEvent = procedure(Sender: TObject; PercentDone: Word) of object;

  TFSFilenameProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TFSSplitFilenameProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TFileSplitter = class(TComponent)
  private
    FFileName: TFileName;
    FSize: LongInt;
    FSplitFileName: TFileName;
    FOutDifferent: Boolean;
    FFile: TFileStream;
    FSplitFile: TFileStream;
    FBuffer: Pointer;
    FBufferSize: Integer;
    FReduceFirstSizeBy: LongInt;
    Signature: TSignature;

    FOnNeedDisk: TNeedDiskEvent;
    FOnProgress: TProgressEvent;

    procedure SetFileName(Value: TFileName);
    procedure SetSize(Value: LongInt);
    procedure SetReduceFirstSizeBy(Value: LongInt);
    procedure SetBufferSize(Value: Integer);
    procedure SetSplitFileName(Value: TFileName);
  protected
  public
    constructor Create(AOwner: TComponent); override;
    procedure Split;
    procedure UnSplit;
  published
    property FileName: TFileName read FFileName write SetFileName;
    property SplitFileName: TFileName read FSplitFileName write SetSplitFileName;
    property Size: LongInt read FSize write SetSize;
    property BufferSize: Integer read FBufferSize write SetBufferSize;
    property ReduceFirstSizeBy: LongInt read FReduceFirstSizeBy write SetReduceFirstSizeBy;

    property OnNeedDisk: TNeedDiskEvent read FOnNeedDisk write FOnNeedDisk;
    property OnProgress: TProgressEvent read FOnProgress write FOnProgress;
  end;

procedure Register;

implementation

{ TFSFilenameProperty }

procedure TFSFilenameProperty.Edit;
var
  FSFileOpen: TOpenDialog;
  
begin
  FSFileOpen := TOpenDialog.Create(Application);
  FSFileOpen.Filename := GetValue;
  FSFileOpen.Filter := 'All files (*.*)|*.*';
  FSFileOpen.Options := FSFileOpen.Options + [ofShowHelp, ofPathMustExist];
  FSFileOpen.Title := 'Select a file to split';
  try
    if FSFileOpen.Execute then
      SetValue(FSFileOpen.Filename);
  finally
    FSFileOpen.Free;
  end;
end;

function TFSFilenameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paRevertable];
end;

{ TFSSplitFilenameProperty }

procedure TFSSplitFilenameProperty.Edit;
var
  FSFileSave: TSaveDialog;

begin
  FSFileSave := TSaveDialog.Create(Application);
  FSFileSave.Filename := GetValue;
  FSFileSave.Filter := 'Split files (*.spl)|*.SPL';
  FSFileSave.Options := FSFileSave.Options + [ofShowHelp, ofPathMustExist];
  FSFileSave.Title := 'Select the first split file (.SPL)';
  try
    if FSFileSave.Execute then
      SetValue(FSFileSave.Filename);
  finally
    FSFileSave.Free;
  end;
end;

function TFSSplitFilenameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paRevertable];
end;

{ TFileSplitter }

{ Sets the name of the file to split }
procedure TFileSplitter.SetFileName(Value: TFileName);
begin
  if Value <> FFileName then
  begin
    FFileName := Value;
    if not FOutDifferent then
      FSplitFileName := ChangeFileExt(Value, '.SPL');
  end;
end;

{ Sets the name of the first split file - all other split files will
  have the same name, with extensions .001, .002, .003 etc. }
procedure TFileSplitter.SetSplitFileName(Value: TFileName);
begin
  if Value <> FSplitFileName then
  begin
    FSplitFileName := Value;
    FOutDifferent := True;
  end;
end;

{ Sets the maximum size of split files }
procedure TFileSplitter.SetSize(Value: LongInt);
begin
  if Value <> FSize then
    FSize := Value;
end;

{ Sets the maximum size of the first split file }
procedure TFileSplitter.SetReduceFirstSizeBy(Value: LongInt);
begin
  if Value <> FReduceFirstSizeBy then
    if (FSize - SizeOf(Signature) - Value) > 0 then
      FReduceFirstSizeBy := Value;
end;

{ Sets the buffer size }
procedure TFileSplitter.SetBufferSize(Value: Integer);
begin
  if Value <> FBufferSize then
    FBufferSize := Value;
end;

{ Component constructor... }
constructor TFileSplitter.Create(AOwner: TComponent);
begin
  inherited;
  FFileName := '';
  FSplitFileName := FFileName;
  FOutDifferent := False;
  FSize := 1457664;
  FBufferSize := 1457664;
  FReduceFirstSizeBy := 0;
  Signature.Copyright := 'TFileSplitter by Jovan Sedlan';
  Signature.NumberOfFiles := 0;
end;

{ EraseFile deletes a file on disk }
procedure EraseFile(FN: TFileName);
var
  f: File;

begin
  AssignFile(f, FN);
  Erase(f);
end;

{ StrZero returns a string representation of a Number,
  Len characters long, with leading zeroes }
function StrZero(Number: LongInt; Len: Byte): String;
var
  Temp: String;

begin
  Temp := Trim(IntToStr(Number));
  Result := StringOfChar('0', Len - Length(Temp)) + Temp;
end;

{ Split method does the actual work - splits a file into multiple files }
procedure TFileSplitter.Split;
var
  i: Integer;
  Continue: Boolean;
  FileCount, PercentDone: Integer;
  sFileName: TFileName;
  BytesToRead, BytesWritten, BytesRead, BytesStored, LastSplitFileSize: LongInt;
  TheSize: LongInt;

begin
  if FileExists(FFileName) then
  begin
    { Make sure the filename is correct }
    if FSplitFileName = '' then
      FSplitFileName := ChangeFileExt(FFileName, '.SPL');

    { Prepare for reading from file }
    FFile := TFileStream.Create(FFileName, fmOpenRead);
    GetMem(FBuffer, FBufferSize);

    { Calculate the number of resulting files }
    FileCount := (FFile.Size + SizeOf(Signature)) div FSize;
    LastSplitFileSize := (FFile.Size + SizeOf(Signature)) mod FSize;
    if LastSplitFileSize <> 0 then
      Inc(FileCount);
    if (LastSplitFileSize + FReduceFirstSizeBy) > FSize then
      Inc(FileCount);
    Signature.NumberOfFiles := FileCount;

    try
      { Write resulting file(s) }
      i := 1;
      Continue := True;
      sFileName := FSplitFileName;
      while (i <= FileCount) do
      begin
        if Assigned(FOnNeedDisk) then
          FOnNeedDisk(Self, i, Signature.NumberOfFiles, Continue);

        if Continue then
        begin
          { Delete the resulting file if it exists }
          if FileExists(sFileName) then
            EraseFile(sFileName);
          FSplitFile := TFileStream.Create(sFileName, fmCreate);

          BytesWritten := 0;
          if i = 1 then
          begin
            { Put the signature in the first split file }
            FSplitFile.Write(Signature, SizeOf(Signature));
            { Write one file }
            BytesToRead := FBufferSize;
            if BytesToRead > FSize - FReduceFirstSizeBy then
              BytesToRead := FSize - FReduceFirstSizeBy;

            BytesToRead := BytesToRead - SizeOf(Signature);
            BytesWritten := SizeOf(Signature);

            { Real size of the split file }
            TheSize := FSize - FReduceFirstSizeBy;
          end
          else
          begin
            { Write one file }
            BytesToRead := FBufferSize;
            if BytesToRead > FSize then
              BytesToRead := FSize;

            { Real size of the split file }
            TheSize := FSize;
          end;

          while BytesWritten < TheSize do
          begin
            if BytesToRead > TheSize - BytesWritten then
              BytesToRead := TheSize - BytesWritten;
            BytesRead := FFile.Read(FBuffer^, BytesToRead);
            BytesStored := FSplitFile.Write(FBuffer^, BytesRead);
            Inc(BytesWritten, BytesStored);

            { Call OnProgress event }
            if Assigned(FOnProgress) then
            begin
              if i = FileCount then
                PercentDone := (BytesWritten * 100) div LastSplitFileSize
              else
                PercentDone := (BytesWritten * 100) div TheSize;
              FOnProgress(Self, PercentDone);
            end;

            if FFile.Position = FFile.Size then
              Break;
          end;
          FSplitFile.Free;
          sFileName := ChangeFileExt(sFileName, '.' + StrZero(i, 3));
          Inc(i);
        end;
      end;
    finally
      { Clean up }
      FreeMem(FBuffer);
      FFile.Free;
    end;
  end
  else
    raise EFileError.Create('Input file ''' + FFileName + ''' does not exist.');
end;

{ Method UnSplit creates the original file from the split files. Name and
  extension of the original file must be set in FileName property }
procedure TFileSplitter.UnSplit;
var
  i: Integer;
  Continue: Boolean;
  sFileName: TFileName;
  BytesRead, TotalRead: LongInt;

begin
  if FileExists(FSplitFileName) then
  begin
    { Make sure the filename is correct }
    if FFileName = '' then
      FFileName := ChangeFileExt(FSplitFileName, '.XXX');

    { Delete the resulting file if it exists }
    if FileExists(FFileName) then
      EraseFile(FFileName);

    { Prepare for writing to file }
    FFile := TFileStream.Create(FFileName, fmCreate);
    GetMem(FBuffer, FBufferSize);

    try
      i := 0;
      sFileName := FSplitFileName;

      { Read files one by one and write to resulting file }
      Continue := True;
      Signature.NumberOfFiles := 0;
      while Continue do
      begin
        if Assigned(FOnNeedDisk) then
          FOnNeedDisk(Self, i + 1, Signature.NumberOfFiles, Continue);

        if Continue then
        begin
          if FileExists(sFileName) then
          begin
            try
              FSplitFile := TFileStream.Create(sFileName, fmOpenRead);

              { Read the signature from the first split file }
              if i = 0 then
                FSplitFile.Read(Signature, SizeOf(Signature));

              TotalRead := 0;
              while FSplitFile.Size <> FSplitFile.Position do
              begin
                BytesRead := FSplitFile.Read(FBuffer^, FBufferSize);
                FFile.Write(FBuffer^, BytesRead);
                TotalRead := TotalRead + BytesRead;

                { Call OnProgress event }
                if Assigned(FOnProgress) then
                  FOnProgress(Self, (TotalRead * 100) div FSplitFile.Size);
              end;
            finally
              FSplitFile.Free;
            end;
            Inc(i);
            sFileName := ChangeFileExt(sFileName, '.' + StrZero(i, 3));
          end
          else
            Continue := False;
        end;
      end;
    finally
      FreeMem(FBuffer);
      FFile.Free;
    end;
  end
  else
    raise EFileError.Create('Input file ''' + FSplitFileName + ''' does not exist.');
end;

{ Register component and property editors }
procedure Register;
begin
  RegisterComponents('SEDLAN A.D.', [TFileSplitter]);
  RegisterPropertyEditor(TypeInfo(TFileName), TFileSplitter, 'FileName', TFSFileNameProperty);
  RegisterPropertyEditor(TypeInfo(TFileName), TFileSplitter, 'SplitFileName', TFSSplitFileNameProperty);
end;

end.

