unit Injected;

interface

uses
  Windows, Classes, Graphics, Controls, Forms, ExtCtrls, SyncObjs,
  Menus, ImgList, RegValues, LightTimer, TrayAreaInjector_TLB;

type
  TProcessSlot = record
    ProcessName: string;
    ID: DWord;
    WindowClass: string;
    Window: THandle;
    InjectedWindow: THandle;
  end;
  TRegConfig = class(TRegValues)
  public
    Top,Left,Height,Width: Integer;
    WindowClass: string;
    InjectionCheckInterval: Integer;
    ShowExit: Boolean;
    FontName: string;
    FontSize: Integer;
    FontBold: Boolean;
    procedure Load;
    procedure Store;
    constructor Create(AOwner: TComponent); override;
  end;
  TInfoPanel = class(TPanel)
  private
    FImage: TImage;
    OWidth: Integer;
    procedure MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer); reintroduce;
    procedure MouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer); reintroduce;
    procedure MouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer); reintroduce;
  public
    property Image: TImage read FImage;
    constructor Create(AOwner: TComponent; AParent: TWinControl); reintroduce;
  end;
  TFormInjected = class;
  TPanelInjected = class(TPanel, IInterface, ITrayArea)
  private
    FormInjected: TFormInjected;
    Timer: TLightTimer;
    PanelList: TStringList;
    MinHeight: Integer;
    MinWidth: Integer;
    ProcessName,WindowClass: string;
    ProcessID: Integer;
    ListKey: string;
    procedure Reorder;
    procedure Resize; reintroduce;
    procedure OnTimer(Sender: TObject);
  private
    function Get_PanelText(Handle: Integer): WideString; safecall;
    procedure Set_PanelText(Handle: Integer; const Value: WideString); safecall;
    function Get_PanelImageIndex(Handle: Integer): Integer; safecall;
    procedure Set_PanelImageIndex(Handle: Integer; Value: Integer); safecall;
  public
    property PanelText[Handle: Integer]: WideString read Get_PanelText write Set_PanelText;
    property PanelImageIndex[Handle: Integer]: Integer read Get_PanelImageIndex write Set_PanelImageIndex;
    function AddPanel(const ID: WideString; const AText: WideString; AImageIndex: Integer; AWidth: Integer): Integer; safecall;
    procedure RemovePanel(Handle: Integer); safecall;
    function AddPicture(Picture: OleVariant): Integer; safecall;
    procedure AttachTrayArea(ProcessID: Integer; const WindowClass: WideString); safecall;
  protected
    FRefCount: Integer;
    function QueryInterface(const IID: TGUID; out Obj): HResult; override; stdcall;
    function _AddRef: Integer; stdcall;
    function _Release: Integer; stdcall;
  public
    property RefCount: Integer read FRefCount;
    constructor Create(AOwner: TComponent; const Key: string); reintroduce;
    destructor Destroy; override;
    class function Instanciate(ProcessID: Integer): ITrayArea;
  end;
  TFormInjected = class(TForm)
    PopupMenu: TPopupMenu;
    ItemTerminate: TMenuItem;
    ImageList: TImageList;
    procedure ItemTerminateClick(Sender: TObject);
    procedure FormMainResize(Sender: TObject);
  private
    FPanelInjected: TPanelInjected;
    FConfig: TRegConfig;
    ProcessSlot: TProcessSlot;
    RMouse: record
      Move: Boolean;
      ResizeWidth: Boolean;
      ResizeHeight: Boolean;
      DownPos: TPoint;
      Pos: TPoint;
    end;
    class function AttachProcess(var ProcessSlot: TProcessSlot): Boolean;
    procedure SetPanelInjected(const Value: TPanelInjected);
    procedure MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer); reintroduce;
    procedure MouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer); reintroduce;
    procedure MouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer); reintroduce;
  public
    property PanelInjected: TPanelInjected read FPanelInjected write SetPanelInjected;
    property Config: TRegConfig read FConfig;
    destructor Destroy; override;
    function CheckInjection: Boolean;
    class function CreateInjectedForm(var ProcessName: string; var ProcessID: Integer; WindowClass: string): TFormInjected;
  end;

implementation

{$R *.dfm}

uses
  ComServ, tlhelp32, Math, SysUtils, Types, Variants;

type
  ETargetWindow = class(Exception);
  ENoProcessInfo = class(Exception);

var
  PanelsMutex: TCriticalSection;
  PanelInjectedList: TStringList;

{ TRegConfig }

constructor TRegConfig.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  AutoLoad := false;
  AutoSave := false;
end;

procedure TRegConfig.Load;
var
  S: string;

  function ParseStr(var 	ParseBuffer: string;
                    const TargetKey: Char): string;
  var
    I: Integer;
  begin
    I := Pos(TargetKey, ParseBuffer);
    If I = 0 Then Begin
      Result := ParseBuffer;
      ParseBuffer := '';
    end
    else begin
      Result := Copy(ParseBuffer,1,I - 1);
      Delete(ParseBuffer,1,I);
    end;
  end;

begin
  Items.Add;
  Items[0].DefValue := '0,0,100,40';
  Items[0].ValueName := 'AreaRect';
  Items[0].ValueType := vtString;
  Items[0].AutoStore := True;
  Items.Add;
  Items[1].DefValue := '10000';
  Items[1].ValueName := 'InjectionCheckInterval';
  Items[1].ValueType := vtInteger;
  Items[1].AutoStore := True;
  Items.Add;
  Items[2].DefValue := '0';
  Items[2].ValueName := 'ShowExit';
  Items[2].ValueType := vtBoolean;
  Items[2].AutoStore := True;
  Items.Add;
  Items[3].DefValue := 'MS Sans Serif';
  Items[3].ValueName := 'FontName';
  Items[3].ValueType := vtString;
  Items[3].AutoStore := True;
  Items.Add;
  Items[4].DefValue := '8';
  Items[4].ValueName := 'FontSize';
  Items[4].ValueType := vtInteger;
  Items[4].AutoStore := True;
  Items.Add;
  Items[5].DefValue := '0';
  Items[5].ValueName := 'FontBold';
  Items[5].ValueType := vtBoolean;
  Items[5].AutoStore := True;
  Items.Add;
  Items[6].DefValue := 'Shell_TrayWnd';
  Items[6].ValueName := 'WindowClass';
  Items[6].ValueType := vtString;
  Items[6].AutoStore := True;
  inherited LoadValues;
  InjectionCheckInterval := Item['InjectionCheckInterval'];
  ShowExit := Item['ShowExit'];
  FontName := Item['FontName'];
  FontSize := Item['FontSize'];
  FontBold := Item['FontBold'];
  S := Item['AreaRect'];
  try
    Left := StrToInt(ParseStr(S,','));
    Top := StrToInt(ParseStr(S,','));
    Width := StrToInt(ParseStr(S,','));
    Height := StrToInt(ParseStr(S,','));
  except
    ;
  end;
  WindowClass := Item['WindowClass'];
end;

procedure TRegConfig.Store;
var
  S: string;
begin
  Item['InjectionCheckInterval'] := InjectionCheckInterval;
  Item['ShowExit'] := ShowExit;
  Item['FontName'] := FontName;
  Item['FontSize'] := FontSize;
  Item['FontBold'] := FontBold;
  S := Format('%d,%d,%d,%d',[Left,Top,Width,Height]);
  Item['AreaRect'] := S;
  Item['WindowClass'] := WindowClass;
  inherited SaveValues;
end;

{ TInfoPanel }

procedure TInfoPanel.MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned((Parent as TPanel).OnMouseDown) then
    (Parent as TPanel).OnMouseDown(Sender,Button,Shift,X + Left,Y + Top);
end;

procedure TInfoPanel.MouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer);
begin
  if Assigned((Parent as TPanel).OnMouseMove) then
    (Parent as TPanel).OnMouseMove(Sender,Shift,X + Left ,Y + Top);
end;

procedure TInfoPanel.MouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned((Parent as TPanel).OnMouseUp) then
    (Parent as TPanel).OnMouseUp(Sender,Button,Shift,X + Left,Y + Top);
end;

constructor TInfoPanel.Create(AOwner: TComponent; AParent: TWinControl);
begin
  inherited Create(AOwner);
  Alignment := taLeftJustify;
  BevelInner := bvNone;
  BevelOuter := bvRaised;
  ParentFont := true;
  Parent := AParent;
  Height := 18;
  OnMouseDown := MouseDown;
  OnMouseUp := MouseUp;
  OnMouseMove := MouseMove;
  FImage := TImage.Create(Self);
  FImage.Align := alRight;
  FImage.Width := 16;
  FImage.Parent := Self;
  FImage.Center := true;
  FImage.Transparent := true;
  FImage.Tag := -1;
end;

{ TPanelInjected }

procedure TPanelInjected.Resize;
var
  I: Integer;
  InfoPanel: TInfoPanel;
begin
  Font.Name := FormInjected.Config.FontName;
  Font.Size := FormInjected.Config.FontSize;
  if FormInjected.Config.FontBold then
    Font.Style := Font.Style + [fsBold]
  else
    Font.Style := Font.Style - [fsBold];
  Canvas.Font := Font;
  for I := 0 to Pred(PanelList.Count) do begin
    if Assigned(PanelList.Objects[I]) then begin
      InfoPanel := (PanelList.Objects[I] as TInfoPanel);
      if InfoPanel.OWidth = 0 then begin
        InfoPanel.Width := Canvas.TextWidth(InfoPanel.Caption) + 20;
        InfoPanel.OWidth := InfoPanel.Width;
      end
      else if InfoPanel.OWidth < 0 then
        InfoPanel.OWidth := InfoPanel.Width;
      if InfoPanel.Image.Tag >= 0 then try
        FormInjected.ImageList.GetBitmap(InfoPanel.Image.Tag,InfoPanel.Image.Picture.Bitmap);
      except
        ;
      end;
    end;
  end;
end;

procedure TPanelInjected.Reorder;
var
  I,T,L: Integer;
  InfoPanel,PreviousInfoPanel: TInfoPanel;
begin
  L := 1;
  T := 1;
  PreviousInfoPanel := nil;
  MinHeight := 0;
  MinWidth := 0;
  for I := 0 to Pred(PanelList.Count) do begin
    if Assigned(PanelList.Objects[I]) then begin
      InfoPanel := (PanelList.Objects[I] as TInfoPanel);
      InfoPanel.Width := InfoPanel.OWidth;
      if (InfoPanel.Width + 2) > MinWidth then
        MinWidth := InfoPanel.Width + 2;
      if MinHeight = 0 then
        MinHeight := InfoPanel.Height + 2;
      if ((L + InfoPanel.Width + 1) <= ClientWidth) or not Assigned(PreviousInfoPanel) then begin
        InfoPanel.Left := L;
        L := L + InfoPanel.Width;
        InfoPanel.Top := T;
      end
      else begin
        L := 1;
        T := T + InfoPanel.Height;
        InfoPanel.Left := L;
        L := L + InfoPanel.Width;
        InfoPanel.Top := T;
        if Assigned(PreviousInfoPanel) then
          PreviousInfoPanel.Width := ClientWidth - PreviousInfoPanel.Left - 1;
        if (T + InfoPanel.Height + 1) > ClientHeight then begin
          TForm(Parent).ClientHeight := T + InfoPanel.Height + 1;
          ClientHeight := TForm(Parent).ClientHeight;
          MinHeight := Height;
        end;
      end;
      PreviousInfoPanel := PanelList.Objects[I] as TInfoPanel;
    end;
  end;
end;

function TPanelInjected.AddPanel(const ID: WideString; const AText: WideString; AImageIndex: Integer; AWidth: Integer): Integer;

 	function GetDarkerColor(BaseColor: TColor;
  								        Delta: Integer): TColor;
	begin
  	Result := RGB(Max(GetRValue(ColorToRGB(BaseColor)) - Delta, 0),
								  Max(GetGValue(ColorToRGB(BaseColor)) - Delta, 0),
								  Max(GetBValue(ColorToRGB(BaseColor)) - Delta, 0));
                  	Color := ColorToRGB(Color);
	end;

begin
  PanelsMutex.Enter;
  try
    Result := PanelList.IndexOf(ID);
    if Result < 0 then begin
      Result := PanelList.AddObject(ID,TInfoPanel.Create(nil,Self));
      with PanelList.Objects[Result] as TInfoPanel do begin
        Color := GetDarkerColor(Color,-16);
        if AWidth > 0 then begin
          Width := AWidth;
          OWidth := AWidth;
        end
        else if AWidth = 0 then begin
          Width := 18;
          OWidth := 0;
        end
        else begin
          Width := 18;
          OWidth := -1;
        end;
        Set_PanelText(Result,AText);
        Set_PanelImageIndex(Result,AImageIndex);
      end;
      if Assigned(Parent) then begin
        Resize;
        Reorder;
      end;
    end;
  finally
    PanelsMutex.Leave;
  end;
end;

procedure TPanelInjected.RemovePanel(Handle: Integer);
begin
  PanelsMutex.Enter;
  try
    PanelList.Objects[Handle].Free;
    PanelList.Objects[Handle] := nil;
    PanelList.Strings[Handle] := '';
    if Assigned(Parent) then
      Reorder;
  finally
    PanelsMutex.Leave;
  end;
end;

function TPanelInjected.Get_PanelText(Handle: Integer): WideString;
begin
  PanelsMutex.Enter;
  try
    Result := (PanelList.Objects[Handle] as TInfoPanel).Caption;
  finally
    PanelsMutex.Leave;
  end;
end;

procedure TPanelInjected.Set_PanelText(Handle: Integer; const Value: WideString);
begin
  PanelsMutex.Enter;
  try
    (PanelList.Objects[Handle] as TInfoPanel).Caption := string(Value);
  finally
    PanelsMutex.Leave;
  end;
end;

function TPanelInjected.Get_PanelImageIndex(Handle: Integer): Integer;
begin
  PanelsMutex.Enter;
  try
    Result := (PanelList.Objects[Handle] as TInfoPanel).Image.Tag;
  finally
    PanelsMutex.Leave;
  end;
end;

procedure TPanelInjected.Set_PanelImageIndex(Handle: Integer; Value: Integer);
begin
  PanelsMutex.Enter;
  try
    if Assigned(FormInjected) then begin
      if Value >= 0 then begin
        with (PanelList.Objects[Handle] as TInfoPanel) do begin
          Image.Tag := Value;
          Image.Picture.Metafile.Clear;
          FormInjected.ImageList.GetBitmap(Value,Image.Picture.Bitmap);
        end;
      end
      else with (PanelList.Objects[Handle] as TInfoPanel) do begin
        Image.Tag := Value;
        Image.Picture.Metafile.Clear;
      end;
    end
    else
      (PanelList.Objects[Handle] as TInfoPanel).Image.Tag := Value;
  finally
    PanelsMutex.Leave;
  end;
end;

function TPanelInjected.AddPicture(Picture: OleVariant): Integer;
type
  PPixelArray = ^TPixelArray;
  TPixelArray = array[0..0] of LongWord;
var
  PictureBitmap: TBitmap;
  ScanLine: PPixelArray;
  PictureLine: Variant;
  PictureLineData: PPixelArray;
  Row,Col: Integer;
begin
  PanelsMutex.Enter;
  if Assigned(FormInjected) then begin
    PictureBitmap := TBitmap.Create;
    try
      PictureBitmap.PixelFormat := pf32bit;
      PictureBitmap.Height := Succ(VarArrayHighBound(Picture,1));
      PictureLine := Picture[0];
      PictureBitmap.Width := Succ(VarArrayHighBound(PictureLine,1));
      for Row := 0 to Pred(PictureBitmap.Height) do begin
        ScanLine := PictureBitmap.ScanLine[Row];
        PictureLine := Picture[Row];
        PictureLineData := VarArrayLock(PictureLine);
        try
          for Col := 0 to Pred(PictureBitmap.Width) do
            ScanLine^[Col] := PictureLineData^[Col];
        finally
          VarArrayUnlock(PictureLine);
        end;
      end;
      Result := FormInjected.ImageList.Add(PictureBitmap,nil);
    finally
      PictureBitmap.Free;
      PanelsMutex.Leave;
    end;
  end
  else begin     
    Result := -1;
    PanelsMutex.Leave;
  end;
end;

procedure TPanelInjected.AttachTrayArea(ProcessID: Integer; const WindowClass: WideString);
begin
  PanelsMutex.Enter;
  try
    if not Assigned(FormInjected) then begin
      Self.ProcessName := '';
      Self.ProcessID := ProcessID;
      Self.WindowClass := WindowClass;
      FormInjected := TFormInjected.CreateInjectedForm(Self.ProcessName,Self.ProcessID,Self.WindowClass);
      if Assigned(FormInjected) then begin
        FormInjected.PanelInjected := Self;
        FormInjected.Show;
        Timer.Interval := FormInjected.Config.InjectionCheckInterval;
        Timer.Enabled := true;
      end
      else begin
        Timer.Interval := 1000;
        Timer.Enabled := true;
      end;
    end;
  finally
    PanelsMutex.Leave;
  end;
end;

procedure TPanelInjected.OnTimer(Sender: TObject);
begin
  PanelsMutex.Enter;
  try
    if Timer.Enabled then begin
      if not Assigned(FormInjected) then begin
        FormInjected := TFormInjected.CreateInjectedForm(Self.ProcessName,Self.ProcessID,Self.WindowClass);
        if Assigned(FormInjected) then begin
          FormInjected.PanelInjected := Self;
          FormInjected.Show;
          Timer.Interval := FormInjected.Config.InjectionCheckInterval;
        end;
      end
      else begin
        if not FormInjected.CheckInjection then begin
          while RefCount > 0 do
            _Release;
          if PanelInjectedList.Count = 0 then begin
            ComServer.Free;
            ComServer := nil;
            Application.Terminate;
          end;
        end;
      end;
    end;
  finally
    PanelsMutex.Leave;
  end;
end;

function TPanelInjected.QueryInterface(const IID: TGUID; out Obj): HResult;
begin
  if GetInterface(IID, Obj) then
    Result := 0
  else
    Result := E_NOINTERFACE;
end;

function TPanelInjected._AddRef: Integer;
begin
  Result := InterlockedIncrement(FRefCount);
end;

function TPanelInjected._Release: Integer;
begin
  Result := InterlockedDecrement(FRefCount);
  if Result = 0 then
    Destroy;
end;

constructor TPanelInjected.Create(AOwner: TComponent; const Key: string);
begin
  inherited Create(AOwner);
  ListKey := Key;
  Timer := TLightTimer.Create(0,OnTimer,[]);
  PanelList := TStringList.Create;
  Caption := '';
  BevelInner := bvNone;
  BevelOuter := bvLowered;
end;

destructor TPanelInjected.Destroy;
var
  I: Integer;
begin
  PanelsMutex.Enter;
  try
    Timer.Enabled := false;
    Timer.Free;
    for I := 0 to Pred(PanelList.Count) do
      PanelList.Objects[I].Free;
    PanelList.Free;
    I := PanelInjectedList.IndexOf(ListKey);
    PanelInjectedList.Delete(I);
    if Assigned(FormInjected) then begin
      FormInjected.PanelInjected := nil;
      FormInjected.Free;
      FormInjected := nil;
    end;
  finally
    PanelsMutex.Leave;
  end;
  inherited Destroy;
end;

class function TPanelInjected.Instanciate(ProcessID: Integer): ITrayArea;
var
  I: Integer;
  ListKey: string;
  Panel: TPanelInjected;
begin
  ListKey := IntToStr(ProcessID);
  PanelsMutex.Enter;
  try
    I := PanelInjectedList.IndexOf(ListKey);
    if I >= 0 then begin
      Panel := PanelInjectedList.Objects[I] as TPanelInjected;
      Panel.QueryInterface(IID_ITrayArea,Result);
    end
    else begin
      Panel := TPanelInjected.Create(nil,ListKey);
      PanelInjectedList.AddObject(ListKey,Panel);
      Panel.QueryInterface(IID_ITrayArea,Result);
    end;
  finally
    PanelsMutex.Leave;
  end;
end;

{ TFormInjected }

procedure TFormInjected.MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

  function IsStartMove: Boolean;
  begin
    Result := (Button = mbLeft) and (X < (Width div 2));
  end;

  function IsStartResizeWidth: Boolean;
  begin
    Result := (Button = mbLeft) and (X > (Width div 2)) and (Y < (Height div 2));
  end;

  function IsStartResizeHeight: Boolean;
  begin
    Result := (Button = mbLeft) and (X > (Width div 2)) and (Y > (Height div 2));
  end;

begin
  if IsStartMove then begin
    RMouse.Move := true;
    RMouse.DownPos.X := Left - Mouse.CursorPos.X;
    RMouse.DownPos.Y := Top - Mouse.CursorPos.Y;
  end
  else if IsStartResizeWidth then begin
    RMouse.ResizeWidth := true;
    RMouse.DownPos.X := Width - Mouse.CursorPos.X;
  end
  else if IsStartResizeHeight then begin
    RMouse.ResizeHeight := true;
    RMouse.DownPos.Y := Height - Mouse.CursorPos.Y;
  end;
end;

procedure TFormInjected.MouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  NewHeight,NewWidth: Integer;

  function IsStartMove: Boolean;
  begin
    Result := (X < (Width div 2));
  end;

  function IsStartResizeWidth: Boolean;
  begin
    Result := (X > (Width div 2)) and (Y < (Height div 2));
  end;

  function IsStartResizeHeight: Boolean;
  begin
    Result := (X > (Width div 2)) and (Y > (Height div 2));
  end;

begin
  if RMouse.Move then begin
    Left := RMouse.DownPos.X + Mouse.CursorPos.X;
    Top := RMouse.DownPos.Y + Mouse.CursorPos.Y;
  end
  else if RMouse.ResizeWidth then begin
    NewWidth := RMouse.DownPos.X + Mouse.CursorPos.X;
    if NewWidth >= PanelInjected.MinWidth then
      Width := NewWidth;
  end
  else if RMouse.ResizeHeight then begin
    NewHeight := RMouse.DownPos.Y + Mouse.CursorPos.Y;
    if NewHeight >= PanelInjected.MinHeight then
      Height := NewHeight;
  end
  else begin
    if IsStartMove then
      TWinControl(Sender).Cursor := crSizeAll
    else if IsStartResizeWidth then
      TWinControl(Sender).Cursor := crSizeWE
    else if IsStartResizeHeight then
      TWinControl(Sender).Cursor := crSizeNS
    else
      TWinControl(Sender).Cursor := crDefault;
  end;
end;

procedure TFormInjected.MouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  RMouse.Move := false;
  RMouse.ResizeWidth := false;
  RMouse.ResizeHeight := false;
end;

procedure TFormInjected.ItemTerminateClick(Sender: TObject);
begin
  while Assigned(PanelInjected) do
    if PanelInjected.RefCount > 0 then
      PanelInjected._Release;
  if PanelInjectedList.Count = 0 then begin
    ComServer.Free;
    ComServer := nil;
    Application.Terminate;
  end;
end;

procedure TFormInjected.FormMainResize(Sender: TObject);
begin
  if Assigned(FPanelInjected) then
    FPanelInjected.Reorder;
end;

procedure TFormInjected.SetPanelInjected(const Value: TPanelInjected);
begin
  if Assigned(FPanelInjected) then begin
    FPanelInjected.OnMouseDown := nil;
    FPanelInjected.OnMouseMove := nil;
    FPanelInjected.OnMouseUp := nil;
    FPanelInjected.PopupMenu := nil;
    FPanelInjected.Parent := nil;
    FPanelInjected := nil;
  end;
  if Assigned(Value) then begin
    FPanelInjected := Value;
    FPanelInjected.Parent := Self;
    FPanelInjected.Align := alClient;
    if FConfig.ShowExit then
      FPanelInjected.PopupMenu := PopupMenu;
    FPanelInjected.Resize;
    FPanelInjected.Reorder;
    FPanelInjected.OnMouseDown := MouseDown;
    FPanelInjected.OnMouseMove := MouseMove;
    FPanelInjected.OnMouseUp := MouseUp;
  end;
end;

destructor TFormInjected.Destroy;
begin
  FConfig.Top := Top;
  FConfig.Left := Left;
  FConfig.Height := Height;
  FConfig.Width := Width;
  FConfig.Store;
  FConfig.Free;
  inherited Destroy;
end;

function EnumWindowsCallback(WindowHandle: THandle; Par: LParam): LRESULT; stdcall;
var
  ClassName: string;
  ProcessSlot: ^TProcessSlot;
  ProcessID: DWord;
begin
  ProcessSlot := Pointer(Par);
  SetLength(ClassName,128);
  GetClassName(WindowHandle,PChar(ClassName),128 + 1);
  SetLength(ClassName,Min(StrLen(PChar(ClassName)),Length(ProcessSlot^.WindowClass)));
  GetWindowThreadProcessId(WindowHandle,ProcessID);
  if (ProcessID = ProcessSlot^.ID) and (Pos(ClassName,ProcessSlot^.WindowClass) > 0) then
    ProcessSlot^.Window := WindowHandle;
  Result := -1;
end;

class function TFormInjected.AttachProcess(var ProcessSlot: TProcessSlot): Boolean;
var
  Snap: THandle;
  PE32: TPROCESSENTRY32;
begin
  Result := false;
  Snap := CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if Snap <> 0 then try
    PE32.dwSize := SizeOf(TPROCESSENTRY32);
    if ProcessSlot.ProcessName <> '' then begin
      if Process32First(Snap,PE32) then begin
        if Pos(string(PE32.szExeFile),ProcessSlot.ProcessName) > 0 then
          Result := true
        else while Process32Next(Snap,PE32) do
          if Pos(string(PE32.szExeFile),ProcessSlot.ProcessName) > 0 then begin
            Result := true;
            Break;
          end;
      end;
    end
    else begin
      if Process32First(Snap,PE32) then begin
        if ProcessSlot.ID = PE32.th32ProcessID then
          Result := true
        else while Process32Next(Snap,PE32) do
          if ProcessSlot.ID = PE32.th32ProcessID then begin
            Result := true;
            Break;
          end;
      end;
    end;
  finally
    CloseHandle(Snap);
  end;
  if Result then begin
    ProcessSlot.ID := PE32.th32ProcessID;
    ProcessSlot.ProcessName := string(PE32.szExeFile);
  end;
end;

function TFormInjected.CheckInjection: Boolean;
begin
  ProcessSlot.Window := 0;
  EnumWindows(@EnumWindowsCallback,LParam(@ProcessSlot));
  Result := (ProcessSlot.Window <> 0);
end;

class function TFormInjected.CreateInjectedForm(var ProcessName: string; var ProcessID: Integer; WindowClass: string): TFormInjected;
var
  ProcessSlot: TProcessSlot;
  Config: TRegConfig;
begin
  Result := nil;
  Config := nil;
  ProcessSlot.ProcessName := ProcessName;
  ProcessSlot.ID := ProcessID;
  ProcessSlot.WindowClass := WindowClass;
  ProcessSlot.Window := 0;
  ProcessSlot.InjectedWindow := 0;
  if AttachProcess(ProcessSlot) then begin
    ProcessName := ProcessSlot.ProcessName;
    ProcessID := ProcessSlot.ID;
    Config := TRegConfig.Create(nil);
    Config.Key := 'SOFTWARE\TrayAreaInjector\' + ProcessName;
    Config.Load;
    EnumWindows(@EnumWindowsCallback,LParam(@ProcessSlot));
    if ProcessSlot.Window <> 0 then begin
      if Config.WindowClass = '' then
        ProcessSlot.InjectedWindow := ProcessSlot.Window
      else
        ProcessSlot.InjectedWindow := FindWindow(PChar(Config.WindowClass),nil);
      if ProcessSlot.InjectedWindow <> 0 then begin
        Result := TFormInjected.CreateParented(ProcessSlot.InjectedWindow);
        Result.FConfig := Config;
        Result.Top := Config.Top;
        Result.Left := Config.Left;
        Result.Height := Config.Height;
        Result.Width := Config.Width;
        Result.ProcessSlot := ProcessSlot;
      end;
    end;
  end;
  if not Assigned(Result) then
    Config.Free;
end;

initialization

  PanelsMutex := TCriticalSection.Create;
  PanelInjectedList := TStringList.Create;

finalization

  PanelInjectedList.Free;
  PanelsMutex.Free;

end.
