unit Visibles.BorderExtender;

interface

uses
  System.Drawing,
  System.Windows.Forms,
  System.ComponentModel,
  System.Collections;

type
  TBorderExtender = class;
  [ToolboxBitmapAttribute(TypeOf(TBorderExtender),'TBorderExtender.bmp')]
  [ProvideProperty('Border3DStyle', TypeOf(&Control))]
  TBorderExtender = class(&Component, IExtenderProvider)
  private
    FBorder3DStyle: HashTable;
    Parents: HashTable;
    procedure PaintHandler(Sender: TObject; e: PaintEventArgs);
    procedure ParentPaintHandler(Sender: TObject; e: PaintEventArgs);
    procedure ParentResizeHandler(Sender: TObject; e: EventArgs);
    procedure ResizeHandler(Sender: TObject; e: EventArgs);
    procedure DockChangedHandler(Sender: TObject; e: EventArgs);
    procedure ParentChangedHandler(Sender: TObject; e: EventArgs);
  strict protected
    procedure Dispose(disposing: Boolean); override;
  public
    constructor Create;
    function TweakColor(Color: System.Drawing.Color; Delta: Integer): System.Drawing.Color; overload;
    function TweakColor(Color: System.Drawing.Color; RDelta,GDelta,BDelta: Integer): System.Drawing.Color; overload;
    function CanExtend(Extendee: TObject): Boolean;
    [Description('Controls what type of border is designed aroud the control (requires BorderStyle at None)'), Category('Appearance')]
    function GetBorder3DStyle(Ctl: TObject): System.Windows.Forms.Border3DStyle;
    procedure SetBorder3DStyle(Ctl: TObject; Value: System.Windows.Forms.Border3DStyle);
  end;

{$R 'Visibles.TBorderExtender.bmp'}

implementation

constructor TBorderExtender.Create;
begin
  inherited;
  FBorder3DStyle := HashTable.Create;
  Parents := HashTable.Create;
end;

procedure TBorderExtender.Dispose(Disposing: Boolean);
var
  Extendee: &Control;
begin
  if Disposing then begin
    with FBorder3DStyle.Keys.GetEnumerator do while MoveNext do begin
      Extendee := Current as &Control;
      Exclude(Extendee.Paint, PaintHandler);
      Exclude(Extendee.Resize, ResizeHandler);
      Exclude(Extendee.ParentChanged, ParentChangedHandler);
      Exclude(Extendee.DockChanged, DockChangedHandler);
      if Assigned(Extendee.Parent) then begin
        Exclude(Extendee.Parent.Paint, ParentPaintHandler);
        Exclude(Extendee.Parent.Resize, ParentResizeHandler);
      end;
    end;
  end;
  inherited;
end;

function TBorderExtender.TweakColor(Color: System.Drawing.Color; Delta: Integer): System.Drawing.Color;
begin
  Result := System.Drawing.Color.FromARGB(Color.A,Math.Min(Color.R + Delta,Color.R.MaxValue),
                                                  Math.Min(Color.G + Delta,Color.G.MaxValue),
                                                  Math.Min(Color.B + Delta,Color.B.MaxValue));
end;


function TBorderExtender.TweakColor(Color: System.Drawing.Color; RDelta, GDelta, BDelta: Integer): System.Drawing.Color;
begin
  Result := System.Drawing.Color.FromARGB(Color.A,Math.Min(Color.R + RDelta,Color.R.MaxValue),
                                                  Math.Min(Color.G + GDelta,Color.G.MaxValue),
                                                  Math.Min(Color.B + BDelta,Color.B.MaxValue));
end;

function TBorderExtender.CanExtend(Extendee: TObject): Boolean;
begin
  Result := (Extendee is Control) and not (Extendee is &Form) and not (Extendee is TBorderExtender);
end;

procedure TBorderExtender.PaintHandler(Sender: TObject; e: PaintEventArgs);
var
  Extendee: &Control;
  Style: System.Windows.Forms.Border3DStyle;
begin
  Extendee := Sender as &Control;
  Style := GetBorder3DStyle(Extendee);
  if Style <> System.Windows.Forms.Border3DStyle.Adjust then
    ControlPaint.DrawBorder3D(e.Graphics,Extendee.ClientRectangle,Style);
end;

procedure TBorderExtender.ParentChangedHandler(Sender: TObject; e: EventArgs);
var
  Extendee,ExtendeeParent: Control;
begin
  Extendee := Sender as &Control;
  ExtendeeParent := Parents[Extendee] as &Control;
  if Assigned(ExtendeeParent) then begin
    Parents[Extendee] := nil;
    Exclude(ExtendeeParent.Paint, ParentPaintHandler);
    Exclude(ExtendeeParent.Resize, ParentResizeHandler);
    if (ExtendeeParent is &ScrollableControl) and
       (Extendee.Dock <> System.Windows.Forms.DockStyle.None) then with ExtendeeParent as &ScrollableControl do begin
      DockPadding.All := 0;
      Invalidate;
    end;
  end;
  if Assigned(Extendee.Parent) then begin
    Parents[Extendee] := Extendee.Parent;
    if (Extendee.Parent is &ScrollableControl) and
       (Extendee.Dock <> System.Windows.Forms.DockStyle.None) then with Extendee.Parent as &ScrollableControl do begin
      case GetBorder3DStyle(Extendee) of
        System.Windows.Forms.Border3DStyle.Adjust: DockPadding.All := 0;
        System.Windows.Forms.Border3DStyle.RaisedInner,
        System.Windows.Forms.Border3DStyle.RaisedOuter,
        System.Windows.Forms.Border3DStyle.SunkenInner,
        System.Windows.Forms.Border3DStyle.SunkenOuter,
        System.Windows.Forms.Border3DStyle.Flat: DockPadding.All := 1;
        System.Windows.Forms.Border3DStyle.Bump,
        System.Windows.Forms.Border3DStyle.Etched,
        System.Windows.Forms.Border3DStyle.Raised,
        System.Windows.Forms.Border3DStyle.Sunken: DockPadding.All := 2;
      end;
      Invalidate;
    end;
    Include(Extendee.Parent.Paint, ParentPaintHandler);
    Include(Extendee.Parent.Resize, ParentResizeHandler);
  end;
  inherited;
end;

procedure TBorderExtender.DockChangedHandler(Sender: TObject; e: EventArgs);
begin
  SetBorder3DStyle(Sender,GetBorder3DStyle(Sender));
end;

procedure TBorderExtender.ParentPaintHandler(Sender: TObject; e: PaintEventArgs);
var
  Extendee: &Control;
  ExtendeeParent: &Control;
  BorderRectangle: &Rectangle;
  Style: System.Windows.Forms.Border3DStyle;
  I: Integer;
begin
  ExtendeeParent := Sender as &Control;
  for I := 0 to Pred(ExtendeeParent.Controls.Count) do begin
    Extendee := ExtendeeParent.Controls.Item[I];
    if Assigned(Extendee) and Assigned(Parents[Extendee]) then begin
      Style := GetBorder3DStyle(Extendee);
      if Style <> System.Windows.Forms.Border3DStyle.Adjust then begin
        BorderRectangle := ExtendeeParent.RectangleToClient(Extendee.RectangleToScreen(Extendee.DisplayRectangle));
        case Style of
          System.Windows.Forms.Border3DStyle.Adjust: ;
          System.Windows.Forms.Border3DStyle.RaisedInner,
          System.Windows.Forms.Border3DStyle.RaisedOuter,
          System.Windows.Forms.Border3DStyle.SunkenInner,
          System.Windows.Forms.Border3DStyle.SunkenOuter,
          System.Windows.Forms.Border3DStyle.Flat: BorderRectangle := &Rectangle.Inflate(BorderRectangle,1,1);
          System.Windows.Forms.Border3DStyle.Bump,
          System.Windows.Forms.Border3DStyle.Etched,
          System.Windows.Forms.Border3DStyle.Raised,
          System.Windows.Forms.Border3DStyle.Sunken: BorderRectangle := &Rectangle.Inflate(BorderRectangle,2,2);
        end;
        ControlPaint.DrawBorder3D(e.Graphics,BorderRectangle,Style);
      end;
    end;
  end;
end;

procedure TBorderExtender.ParentResizeHandler(Sender: TObject; e: EventArgs);
begin
  (Sender as &Control).Invalidate;
end;

procedure TBorderExtender.ResizeHandler(Sender: TObject; e: EventArgs);
begin
  (Sender as &Control).Invalidate;
end;

function TBorderExtender.GetBorder3DStyle(Ctl: TObject): System.Windows.Forms.Border3DStyle;
var
  Style: TObject;
begin
  Style := FBorder3DStyle[Ctl as &Control];
  if Assigned(Style) then
    Result := System.Windows.Forms.Border3DStyle(Style)
  else
    Result := System.Windows.Forms.Border3DStyle.Adjust;
end;

procedure TBorderExtender.SetBorder3DStyle(Ctl: TObject; Value: System.Windows.Forms.Border3DStyle);
var
  Extendee: Control;
begin
  Extendee := Ctl as &Control;
  if Value = System.Windows.Forms.Border3DStyle.Adjust then begin
    if Extendee is &Panel then begin
      Exclude(Extendee.Paint, PaintHandler);
      Exclude(Extendee.Resize, ResizeHandler);
      with Extendee as &Panel do
        DockPadding.All := 0;
    end
    else begin
      Exclude(Extendee.ParentChanged, ParentChangedHandler);
      Exclude(Extendee.DockChanged, DockChangedHandler);
      if Assigned(Extendee.Parent) then begin
        Exclude(Extendee.Parent.Paint, ParentPaintHandler);
        Exclude(Extendee.Parent.Resize, ParentResizeHandler);
        if (Extendee.Parent is &ScrollableControl) and
           (Extendee.Dock <> System.Windows.Forms.DockStyle.None) then with (Extendee.Parent as &ScrollableControl) do begin
          DockPadding.All := 0;
          Invalidate;
        end;
      end;
    end;
    FBorder3DStyle.Remove(Extendee);
    Parents.Remove(Extendee);
  end
  else begin
    FBorder3DStyle[Extendee] := Value;
    if Extendee is &Panel then begin
      Parents[Extendee] := nil;
      Include(Extendee.Paint, PaintHandler);
      Include(Extendee.Resize, ResizeHandler);
      with Extendee as &Panel do case Value of
        System.Windows.Forms.Border3DStyle.Adjust: DockPadding.All := 0;
        System.Windows.Forms.Border3DStyle.RaisedInner,
        System.Windows.Forms.Border3DStyle.RaisedOuter,
        System.Windows.Forms.Border3DStyle.SunkenInner,
        System.Windows.Forms.Border3DStyle.SunkenOuter,
        System.Windows.Forms.Border3DStyle.Flat: DockPadding.All := 1;
        System.Windows.Forms.Border3DStyle.Bump,
        System.Windows.Forms.Border3DStyle.Etched,
        System.Windows.Forms.Border3DStyle.Raised,
        System.Windows.Forms.Border3DStyle.Sunken: DockPadding.All := 2;
      end;
    end
    else begin
      Parents[Extendee] := Extendee.Parent;
      Include(Extendee.ParentChanged, ParentChangedHandler);
      if Extendee.DesignMode then
        Include(Extendee.DockChanged, DockChangedHandler);
      if Assigned(Extendee.Parent) then begin
        if (Extendee.Parent is &ScrollableControl) and
           (Extendee.Dock <> System.Windows.Forms.DockStyle.None) then with Extendee.Parent as &ScrollableControl do begin
          case Value of
            System.Windows.Forms.Border3DStyle.Adjust: DockPadding.All := 0;
            System.Windows.Forms.Border3DStyle.RaisedInner,
            System.Windows.Forms.Border3DStyle.RaisedOuter,
            System.Windows.Forms.Border3DStyle.SunkenInner,
            System.Windows.Forms.Border3DStyle.SunkenOuter,
            System.Windows.Forms.Border3DStyle.Flat: DockPadding.All := 1;
            System.Windows.Forms.Border3DStyle.Bump,
            System.Windows.Forms.Border3DStyle.Etched,
            System.Windows.Forms.Border3DStyle.Raised,
            System.Windows.Forms.Border3DStyle.Sunken: DockPadding.All := 2;
          end;
          Invalidate;
        end;
        Include(Extendee.Parent.Paint, ParentPaintHandler);
        Include(Extendee.Parent.Resize, ParentResizeHandler);
      end;
    end;
  end;
  Extendee.Invalidate;
end;

end.
